﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App;
using App.ConfigData;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace Viewer
{
    /// <summary>
    /// モデル用メッセージです。
    /// </summary>
    public abstract class ModelMessage : BaseMessage
    {
        /// <summary>どのプロパティが編集されたかのフラグ。</summary>
        protected uint _modifiedProperties;
        /// <summary>プロパティの中での要素編集フラグ(RGBAのRとか)。</summary>
        protected uint _modifiedPropertyElements;
        /// <summary>メッセージカテゴリー</summary>
        protected override MessageCategory _messageCategory { get { return MessageCategory.Model; } }
        //---------------------------------------------------------------------
        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(GuiObject model)
        {
            Viewer.Select.Send(model);
        }

        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(GuiObjectGroup targets)
        {
#if true
            Viewer.Select.Send(targets, GuiObjectID.Model);
#endif
        }

        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(ArrayList targets)
        {
            Viewer.Select.Send(targets, GuiObjectID.Model);
        }

    }

    /// <summary>
    /// アニメーションバインド
    /// </summary>
    public sealed class EditModelLayout : ModelMessage
    {
        readonly Model _target;
        readonly bool								isBind_;
        readonly NintendoWare.G3d.Edit.Math.Vector3	scale_;
        readonly NintendoWare.G3d.Edit.Math.Vector3	rotate_;
        readonly NintendoWare.G3d.Edit.Math.Vector3	translate_;
        readonly bool initialSend_;
        readonly bool sendIdentity_;

        /// <summary>コンストラクタ</summary>
        private EditModelLayout(
            Model target,
            bool isBind,
            NintendoWare.G3d.Edit.Math.Vector3 scale,
            NintendoWare.G3d.Edit.Math.Vector3 rotate,
            NintendoWare.G3d.Edit.Math.Vector3 translate,
            bool initialSend,
            bool sendIdentity)
        {
            _target		= target;
            isBind_		= isBind;
            scale_		= scale;
            rotate_		= rotate;
            translate_	= translate;

            // このメッセージは間引き対象
            IsCompressible = true;

            initialSend_ = initialSend;
            sendIdentity_ = sendIdentity;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            if (!_target.IsAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("EditModelLayout Execute not send Message : target({0}) is not attatched. ", _target.FileName);
                return;
            }

            // アタッチされたときはレイアウトを設定しない
            if (_target.IsSendAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("EditModelLayout Execute not send Message : target({0}) is attatched. ", _target.FileName);
                _target.SetLayoutActionOnLoad = null;
                return;
            }

            DebugConsole.WriteLine("G3dHioLibProxy.Hio.EditModelLayout-----------------------------------------------------------");

            if ((!initialSend_ || _target.SetLayoutActionOnLoad == null) &&
                (sendIdentity_ ||
                 scale_.X != 1 || scale_.Y != 1 || scale_.Z != 1 ||
                 rotate_.X != 0 || rotate_.Y != 0 || rotate_.Z != 0 ||
                 translate_.X != 0 || translate_.Y != 0 || translate_.Z != 0))
            {
                G3dHioLibProxy.Hio.EditModelLayout(
                    isBind_,
                    scale_,
                    rotate_,
                    translate_
                );
                _target.SetLayoutActionOnLoad = null;
            }
            else
            {
                // 最初の配置情報の送信まえに既に受信していたら、何もしない。
                if (_target.SetLayoutActionOnLoad != null)
                {
                    // デバッグメッセージ
                    DebugConsole.WriteLine("EditModelLayout Execute not send Message : target({0}) has receivedModelLayoutOnLoad. ", _target.FileName);
                    TheApp.MainFrame.BeginInvoke(_target.SetLayoutActionOnLoad);
                    _target.SetLayoutActionOnLoad = null;
                    return;
                }
            }
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            var editModelLayout = msg as EditModelLayout;
            if (editModelLayout == null)
            {
                return false;
            }
            return
                (_target      == editModelLayout._target     ) &&
                (isBind_      == editModelLayout.isBind_     ) &&
                (initialSend_ == editModelLayout.initialSend_) &&
                (sendIdentity_ == editModelLayout.sendIdentity_);
        }

        /// <summary>送る</summary>
        public static void Send(
            Model target,
            bool isBind,
            NintendoWare.G3d.Edit.Math.Vector3 scale,
            NintendoWare.G3d.Edit.Math.Vector3 rotate,
            NintendoWare.G3d.Edit.Math.Vector3 translate,
            bool afterModelLoad,
            bool sendIdentity=false)
        {
            Select(target);

            (new EditModelLayout(target, isBind, scale, rotate, translate, afterModelLoad, sendIdentity)).Push();
        }
    }

    public sealed class QueryModelLayoutMessage : ModelMessage
    {
        /// <summary>コンストラクタ</summary>
        private QueryModelLayoutMessage()
        {
        }

        bool isBind;
        Model _target;

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                if (_target.IsAttached && (_target.IsSendAttached == false))
                {
                    DebugConsole.WriteLine("G3dHioLibProxy.Hio.QueryModelLayout");
                    G3dHioLibProxy.Hio.QueryModelLayout(isBind);
                }
                else
                {
                    _target.WaitingQueryModelLayout = false;
                }
            }
            else
            {
                _target.WaitingQueryModelLayout = false;
            }
        }

        public static void Send(Model model)
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                Select(model);
                model.WaitingQueryModelLayout = true;
                (new QueryModelLayoutMessage()
                {
                    isBind = model.PreviewInfo.IsBind,
                    _target = model,
                }).Push();
            }
        }
    }

    /// <summary>
    /// アニメーションバインド
    /// </summary>
    public sealed class EditBoneBind : ModelMessage
    {
        readonly Model _target;
        readonly Model _targetModel;
        readonly int _boneIndex;

        /// <summary>コンストラクタ</summary>
        private EditBoneBind(Model target, Model targetModel, int boneIndex)
        {
            _target = target;
            _targetModel = targetModel;
            _boneIndex = boneIndex;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            if(!_target.IsAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("EditBoneBind Execute not send Message : this model({0}) is not attatched. ", _target.FileName);
                return;
            }

            // 選択を外すときは、nullになるので・・・
            if(_targetModel != null && !_targetModel.IsAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("EditBoneBind Execute not send Message : target model({0}) is not attatched. ", _targetModel.FileName);
                return;
            }

            // アタッチされているときは編集しない
            if (_target.IsSendAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("EditModelLayout Execute not send Message : target({0}) is attatched. ", _target.FileName);
                return;
            }
            DebugConsole.WriteLine("EditBoneBind : {0}, {1}-------------------------------------------", _targetModel, _boneIndex);
            G3dHioLibProxy.Hio.EditBoneBind(_targetModel, _boneIndex);
        }

        /// <summary>送る</summary>
        public static void Send(Model target, Model targetModel, int boneIndex)
        {
            Select(target);

            (new EditBoneBind(target, targetModel, boneIndex)).Push();
        }
    }

    public sealed class LoadOptimizedShaderArchive : ModelMessage
    {
        readonly Model _target;
        readonly List<Model.ShaderOptimizeData> ShaderOptimizeData;
        readonly List<ShaderDefinition> shaderDefinitions;
        readonly List<string> shadingModels;
        readonly string modelName;
        readonly string dumpShaderSourcePath;

        // モデルのバイナリコンバート用データ
        private readonly List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> _modelDataList = new List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>>();
        private readonly bool optimizeShader;

        private LoadOptimizedShaderArchive(Model target)
        {
            _target = target;

            // 参照ファイル(テクスチャ)を取得する。
            var docs = GetDocuments(_target).Where(x => !(x is AnimationDocument));
            foreach (var doc in docs)
            {
                // スレッドセーフにするために、IntermediateFileDocument, nw4f_3difType, List<G3dStream>を保持しておく。
                var data =
                    new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>(
                        (IntermediateFileDocument)doc, CopyNW4F3DIF((IntermediateFileDocument)doc), GetStream((IntermediateFileDocument)doc), doc.Name
                        );
                _modelDataList.Add(data);
            }
            // マテリアル参照を解決する
            if (target.Materials.Any(x => x.ParentMaterials.Any()))
            {
                var resolvedModel = _modelDataList[0].Item2.Item as modelType;
                Debug.Assert(resolvedModel != null, "resolvedModel != null");
                Debug.Assert(target.Materials.ToArray().Length == resolvedModel.material_array.length);
                Viewer.LoadOrReloadModel.ResolveParentMaterials(target, resolvedModel);
            }

            // PreBinalize command
            if (Viewer.Manager.Instance.IsConnected && ApplicationConfig.FileIo.PreBinarizeCommand.HasCommand)
            {
                var newlist = PrePostIO.ExecutePreBinarize(_modelDataList);
                if (newlist != null && newlist.Any())
                {
                    _modelDataList = newlist;
                }
            }

            ShaderOptimizeData = HioUtility.CreateShaderOptimizeData(target, out shaderDefinitions, out shadingModels, true);

            modelName = target.Name;
            dumpShaderSourcePath = ApplicationConfig.UserSetting.IO.DumpShaderSource ? TemporaryFileUtility.ShaderFolderPath : null;
            optimizeShader = DocumentManager.OptimizeShader;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (!_target.IsAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("LoadOptimizedShaderArchive Execute not send Message : this model({0}) is not attatched. ", _target.FileName);
                return;
            }

            using (var block = new LoadProgressDialog.DialogBlock(App.res.Strings.Viewer_Loading, () => FileMessage.DisconnectOnMainThreadAsync(), true))
            {
                bool binarizeError;
                HioUtility.LoadOptimizedShader(
                    _target,
                    ShaderOptimizeData,
                    _modelDataList,
                    out binarizeError,
                    false,
                    shaderDefinitions,
                    shadingModels,
                    block,
                    modelName,
                    dumpShaderSourcePath,
                    optimizeShader,
                    false);
            }
        }

        public static void Send(Model model)
        {
            (new LoadOptimizedShaderArchive(model)).Push();
        }

        public static void OnShaderDefinitionChanged(ShaderDefinition shaderDefinition, string[] modifiedShadingModels)
        {
            if (DocumentManager.OptimizeShader)
            {
                foreach (var model in DocumentManager.Models)
                {
                    if (model.Materials.Any(x => x.MaterialShaderAssign.ShaderDefinitionFileName == shaderDefinition.Name))
                    {
                        Send(model);
                    }
                }
            }
            else if (modifiedShadingModels != null)
            {
                foreach (var model in DocumentManager.Models)
                {
                    if (model.Materials.Any(x => x.MaterialShaderAssign.ShaderDefinitionFileName == shaderDefinition.Name &&
                        modifiedShadingModels.Contains(x.MaterialShaderAssign.ShaderName)))
                    {
                        Send(model);
                    }
                }
            }
        }
    }

    public sealed class EditSetShapeLodLevel : ModelMessage
    {
        /// <summary>コンストラクタ</summary>
        private EditSetShapeLodLevel(Model model, int selectedIndex)
        {
            _target = model;
            _index = selectedIndex;
        }

        readonly Model _target;
        readonly int _index;

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            if (!_target.IsAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("EditSetShapeLodLevel Execute not send Message : target({0}) is not attatched. ", _target.FileName);
                return;
            }
            DebugConsole.WriteLine("G3dHioLibProxy.Hio.EditSetShapeLodLevel-----------------------------------------------------------");
            G3dHioLibProxy.Hio.EditSetShapeLodLevel(_index);
        }

        public static void Send(Model model, int selectedIndex)
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                Select(model);
                (new EditSetShapeLodLevel(model, selectedIndex)).Push();
            }
        }
    }

    public sealed class EditResetShapeLodLevel : ModelMessage
    {
        /// <summary>コンストラクタ</summary>
        private EditResetShapeLodLevel(Model model)
        {
            _target = model;
        }

        readonly Model _target;

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            if (!_target.IsAttached)
            {
                // デバッグメッセージ
                DebugConsole.WriteLine("EditResetShapeLodLevel Execute not send Message : target({0}) is not attatched. ", _target.FileName);
                return;
            }
            DebugConsole.WriteLine("G3dHioLibProxy.Hio.EditResetShapeLodLevel-----------------------------------------------------------");
            G3dHioLibProxy.Hio.EditResetShapeLodLevel();
        }

        public static void Send(Model model)
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                Select(model);
                (new EditResetShapeLodLevel(model)).Push();
            }
        }
    }
}
