﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections;
using System.Collections.Generic;
using App;
using App.Data;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace Viewer
{
    /// <summary>
    /// ボーン用メッセージです。
    /// </summary>
    public abstract class BoneMessage : BaseMessage
    {
        /// <summary>どのプロパティが編集されたかのフラグ。</summary>
        protected uint _modifiedProperties;
        /// <summary>プロパティの中での要素編集フラグ(RGBAのRとか)。</summary>
        protected uint _modifiedPropertyElements;
        /// <summary>メッセージカテゴリー</summary>
        protected override MessageCategory _messageCategory { get { return MessageCategory.Bone; } }
        //---------------------------------------------------------------------
        /// <summary>
        /// 選択。		/// <summary>対象ボーンのリスト。</summary>
        protected ArrayList _targetBoneList;
        /// <summary>実行</summary>
        protected abstract void ExecuteInternal();

        /// </summary>
        protected static void Select(GuiObject bone)
        {
            Viewer.Select.Send(bone);
        }

        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(GuiObjectGroup targets)
        {
#if true
            Viewer.Select.Send(targets, GuiObjectID.Bone);
#endif
        }

        /// <summary>
        /// 選択。
        /// </summary>
        protected static void Select(ArrayList targets)
        {
            Viewer.Select.Send(targets, GuiObjectID.Bone);
        }

        protected static ArrayList GetTargetList(GuiObjectGroup targets)
        {
            ArrayList targetList = new ArrayList();
            foreach (var target in targets.GetObjects(GuiObjectID.Bone))
            {
                targetList.Add(target);
            }
            return targetList;
        }

        /// <summary>
        /// 対象ボーンを持つモデルがアタッチされていれば、送信可。
        /// </summary>
        protected bool CanSend()
        {
            if (_targetBoneList == null)
            {
                //// デバッグメッセージ
                //DebugConsole.WriteLine("BoneMessage Execute not send Message : Execute() is not impliment. ");
                return false;
            }

            foreach (var bone in _targetBoneList)
            {
                Bone targetBone = bone as Bone;
                if (targetBone != null)
                {
                    if (targetBone.Owner.IsAttached)
                    {
                        return true;
                    }
                }
            }

            // デバッグメッセージ
            DebugConsole.WriteLine("BoneMessage Execute not send Message : Owner model is not attatched. ");
            return false;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                return;
            }

            // モデルファイルがアタッチされていない場合は送信しない。
            if (!CanSend())
            {
                return;
            }

            // 実際の実行
            ExecuteInternal();
        }

    }

    /// <summary>
    /// ボーンの表示/非表示を設定します。
    /// </summary>
    public sealed class SetBoneVisibility : BoneMessage
    {
        private readonly bool _data;

        /// <summary>コンストラクタ</summary>
        private SetBoneVisibility(bool src, ArrayList targetBoneList)
        {
            _data = src;
            _targetBoneList = targetBoneList;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditBoneVisibility(_data);

            DebugConsole.WriteLine("EditBoneVisibility Execute  : ({0})", _data);
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, bool src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetBoneVisibility(src, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, bool src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetBoneVisibility(src, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetBoneVisibility((bool)sameData, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (bool)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }

    /// <summary>
    /// ボーンのビルボードの設定を行います。
    /// </summary>
    public sealed class SetBoneBillBoard : BoneMessage
    {
        private readonly bone_billboardType _billBoardMode;

        static readonly Dictionary<bone_billboardType, NintendoWare.G3d.Edit.BillboardKind>	billboardDict_ =
            new Dictionary<bone_billboardType, NintendoWare.G3d.Edit.BillboardKind>()
            {
                {bone_billboardType.none,				NintendoWare.G3d.Edit.BillboardKind.None},
                {bone_billboardType.world_viewvector,				NintendoWare.G3d.Edit.BillboardKind.World},
                {bone_billboardType.world_viewpoint,	NintendoWare.G3d.Edit.BillboardKind.WorldViewpoint},
                {bone_billboardType.screen_viewvector,				NintendoWare.G3d.Edit.BillboardKind.Screen},
                {bone_billboardType.screen_viewpoint,	NintendoWare.G3d.Edit.BillboardKind.ScreenViewpoint},
                {bone_billboardType.yaxis_viewvector,				NintendoWare.G3d.Edit.BillboardKind.YAxis},
                {bone_billboardType.yaxis_viewpoint,	NintendoWare.G3d.Edit.BillboardKind.YAxisViewpoint}
            };

        /// <summary>コンストラクタ</summary>
        private SetBoneBillBoard(bone_billboardType src, uint modifiedProperties, uint modifiedPropertyElements, ArrayList targetBoneList)
        {
            _billBoardMode = src;
            _modifiedProperties = modifiedProperties;
            _modifiedPropertyElements = modifiedPropertyElements;
            _targetBoneList = targetBoneList;
        }

        /// <summary>実行</summary>
        protected override void ExecuteInternal()
        {
            HioUtility.EditBoneBillboard(billboardDict_[_billBoardMode]);

            DebugConsole.WriteLine("EditBoneBillboard Execute  : ({0})", billboardDict_[_billBoardMode]);
        }

        /// <summary>送る</summary>
        public static void Send(GuiObject target, bone_billboardType src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(target);

            ArrayList targetList = new ArrayList();
            targetList.Add(target);
            (new SetBoneBillBoard(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(GuiObjectGroup targets, bone_billboardType src, uint modifiedProperties, uint modifiedPropertyElements)
        {
            Select(targets);

            ArrayList targetList = GetTargetList(targets);
            (new SetBoneBillBoard(src, modifiedProperties, modifiedPropertyElements, targetList)).Push();
        }

        /// <summary>送る</summary>
        public static void Send(ArrayList targets, object[] _data, uint modifiedProperties, uint modifiedPropertyElements)
        {
            object sameData = CheckSameData(targets, _data);
            if (sameData != null)
            {
                Select(targets);

                (new SetBoneBillBoard((bone_billboardType)sameData, modifiedProperties, modifiedPropertyElements, targets)).Push();
            }
            else
            {
                for (int i = 0; i < _data.Length; i++)
                {
                    Send((GuiObject)targets[i], (bone_billboardType)_data[i], modifiedProperties, modifiedPropertyElements);
                }
            }
        }
    }
}
