﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App;
using App.Data;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace Viewer
{
    /// <summary>
    /// バイナリデータを読み込みます。
    /// </summary>
    public abstract class AnimationMessage : BaseMessage
    {
        private static readonly object syncObject_ = new object();

        private static float currentFrame;
        public static float CurrentFrame
        {
            get
            {
                lock (syncObject_)
                {
                    return currentFrame;
                }
            }
            set
            {
                lock (syncObject_)
                {
                    currentFrame = value;
                }
            }
        }

        public static bool IsPlaying { get; set; }

        /// <summary>どのプロパティが編集されたかのフラグ。</summary>
        protected uint _modifiedProperties;
        /// <summary>プロパティの中での要素編集フラグ(RGBAのRとか)。</summary>
        protected uint _modifiedPropertyElements;
        /// <summary>メッセージカテゴリー</summary>
        protected override MessageCategory _messageCategory { get { return MessageCategory.Animation; } }

        // MaterialVisibilityAnimation, BoneVisibilityAnimation がキーフレームを持っているか？
        // それ以外のアニメーションの場合は true を返します。
        public static bool CanBind(AnimationDocument anim)
        {
            switch (anim.ObjectID)
            {
                // ビジビリティアニメーションのみキーがないものはバインドしない
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1118
                // #5
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1301

                case GuiObjectID.MaterialVisibilityAnimation:
                    return (anim as MaterialVisibilityAnimation).MaterialVisibilityMatAnims.Any(x => IsConstOrCurve(x));

                case GuiObjectID.BoneVisibilityAnimation:
                    return (anim as BoneVisibilityAnimation).BoneVisibilityBoneAnims.Any(x => IsConstOrCurve(x) && x.binarize_visibility);

                case GuiObjectID.TexturePatternAnimation:
                    return (anim as TexturePatternAnimation).TexPatternMatAnims.Any(x => x.PatternAnimTargets.Any(y => IsConstOrCurve(y)));

                case GuiObjectID.SkeletalAnimation:
                    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1465
                    return true;

                case GuiObjectID.ShapeAnimation:
                    return (anim as ShapeAnimation).VertexShapeAnims.Any(x => x.ShapeAnims.Any(y => IsConstOrCurve(y)));

                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                    return (anim as ShaderParameterAnimation).ShaderParamAnims.Any(x => x.ParamAnims.Any(y => y.ParamAnimTargets.Any(z => IsConstOrCurve(z))));
                case GuiObjectID.MaterialAnimation:
                    // TODO: TexSrt 部分は？
                    return (anim as MaterialAnimation).AnimTargets.Any(x => IsConstOrCurve(x));
                case GuiObjectID.SceneAnimation:
                case GuiObjectID.CameraAnimation:
                case GuiObjectID.LightAnimation:
                case GuiObjectID.FogAnimation:
                    return true;
                default:
                    Debug.Assert(false);
                    return false;
            }
        }

        // コンスタントあるいはカーブかどうか?
        private static bool IsConstOrCurve(AnimTarget animTarget)
        {
            CurveExportType exportType = animTarget.ExportType;
            return exportType == CurveExportType.Constant || exportType == CurveExportType.Curve;
        }
    }

    /// <summary>
    /// アニメーションバインド
    /// </summary>
    public sealed class BindAnimations : AnimationMessage
    {
        private readonly Model							target_;
        private readonly Tuple<string, AnimationDocument>[]	animations_;
        private readonly Dictionary<AnimationDocument, float> pausingFrame = new Dictionary<AnimationDocument,float>();
        /// <summary>コンストラクタ</summary>
        private BindAnimations(Model target, IEnumerable<AnimationDocument> animations)
        {
            target_ = target;
            animations_ = animations.Where(x => CanBind(x)).Select(x => new Tuple<string, AnimationDocument>(x.FileName, x)).ToArray();
            foreach (var anim in animations)
            {
                double frame;
                if (anim.PauseFrames.TryGetValue(new KeyValuePair<object, string>(target_.ModelId, target.PreviewAnimSet.Name), out frame))
                {
                    pausingFrame[anim] = (float)frame;
                }
            }
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                if (!target_.IsAttached)
                {
                    // モデルがアタッチされていなかったら、BindAnimation()は送らない。
                    DebugConsole.WriteLine("G3dHioLibProxy.Hio.BindAnimations(); target model is not attached. ModelFile = {0}", target_.FileName);
                    return;
                }

                if (!target_.HioLoaded)
                {
                    // モデルがアタッチされていなかったら、BindAnimation()は送らない。
                    DebugConsole.WriteLine("G3dHioLibProxy.Hio.BindAnimations(); target model is not loaded. ModelFile = {0}", target_.FileName);
                    return;
                }

                DebugConsole.WriteLine("G3dHioLibProxy.Hio.BindAnimations(); ---------------------------------------------{0}", animations_.Count());

#if true
                HioUtility.UpdateAnimationBinds(target_, animations_, pausingFrame, true);
#else
                //var f = Comparer<Document>.Default;
                //target_.PreviewingAnimations.Where(x => x.HioLoaded).OrderBy(x => x);

                var unbindAnimations = target_.PreviewingAnimations.Where(x => x.Item2.HioLoaded).ToArray();

                if (unbindAnimations.SequenceEqual(animations_))
                {
                    target_.PreviewingAnimations.Clear();
                    target_.PreviewingAnimations.AddRange(animations_);
                    DebugConsole.WriteLine("BindAnimations Skipped c");
                    return;
                }

                foreach (var animation in unbindAnimations)
                {
                    DebugConsole.WriteLine("unbind animation name = {0}", animation.Item1);
                }

                G3dHioLibProxy.Hio.UnbindAnimations(target_, unbindAnimations.Select(x => x.Item2).ToArray());

                foreach (var animation in animations_)
                {
                    DebugConsole.WriteLine("bind animation name = {0}", animation.Item1);
                }

                G3dHioLibProxy.Hio.BindAnimations(target_, animations_.Select(x => x.Item2).ToArray());

                // バインドしたアニメーションをモデルに保存する。
                target_.PreviewingAnimations.Clear();
                target_.PreviewingAnimations.AddRange(animations_);

                // フレームを設定する
                if (AnimationMessage.IsPlaying)
                {
                    G3dHioLibProxy.Hio.PlayFrameCtrl(AnimationMessage.CurrentFrame);
                }
                else
                {
                    G3dHioLibProxy.Hio.StopFrameCtrl(AnimationMessage.CurrentFrame);
                }
#endif


            }
        }

        /// <summary>送る</summary>
        public static void Send(Model target, IEnumerable<AnimationDocument> animation)
        {
            (new BindAnimations(target, animation)).Push();
            Connecter.PulseMessageThread();
        }
    }

    /// <summary>
    /// リターゲットホストモデルの設定
    /// </summary>
    public class SetRetargetHostModel : AnimationMessage
    {
        AnimationDocument target;
        Model model;
        private SetRetargetHostModel(AnimationDocument animation)
        {
            target = animation;
            model = animation.GetRetargetHost();
        }

        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                HioUtility.SetRetargetHostModel(target, model);
            }
        }

        public static void Send(AnimationDocument animation)
        {
            (new SetRetargetHostModel(animation)).Push();
        }
    }

    /// <summary>
    /// アニメーションバインド
    /// </summary>
    public sealed class BindSceneAnimations : AnimationMessage
    {
        /// <summary>コンストラクタ</summary>
        private BindSceneAnimations(IEnumerable<SceneAnimation> animations)
        {
            sceneAnimations = animations.Select(x => new Tuple<string, SceneAnimation>(x.FileName, x)).ToArray();
        }

        private readonly Tuple<string, SceneAnimation>[] sceneAnimations;

        /// <summary>実行</summary>
        public override void Execute()
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                var unbindAnimations = DocumentManager.PreviewingSceneAnimations.Where(x => x.Item2.HioLoaded).ToArray();

                if (unbindAnimations.SequenceEqual(sceneAnimations))
                {
                    DocumentManager.PreviewingSceneAnimations.Clear();
                    DocumentManager.PreviewingSceneAnimations.AddRange(sceneAnimations);
                    DebugConsole.WriteLine("BindScneAnimations Skipped");
                    return;
                }

                foreach (var animation in unbindAnimations)
                {
                    DebugConsole.WriteLine("unbind animation name = {0}", animation.Item1);
                }
                G3dHioLibProxy.Hio.UnbindSceneAnimations(unbindAnimations.Select(x => x.Item2).ToArray());

                DebugConsole.WriteLine("G3dHioLibProxy.Hio.BindSceneAnimations(); ---------------------------------------------{0}", sceneAnimations.Count());
                foreach (var animation in sceneAnimations)
                {
                    DebugConsole.WriteLine("bind animation name = {0}", animation.Item1);
                }

                if (sceneAnimations.Any())
                {
                    G3dHioLibProxy.Hio.BindSceneAnimations(sceneAnimations.Select(x => x.Item2).ToArray());

                    // フレームを設定する
                    if (AnimationMessage.IsPlaying)
                    {
                        G3dHioLibProxy.Hio.PlayFrameCtrl(AnimationMessage.CurrentFrame);
                    }
                    else
                    {
                        G3dHioLibProxy.Hio.StopFrameCtrl(AnimationMessage.CurrentFrame);
                    }
                }

                // バインドしたアニメーションを保存する。
                DocumentManager.PreviewingSceneAnimations.Clear();
                DocumentManager.PreviewingSceneAnimations.AddRange(sceneAnimations);
            }
        }

        /// <summary>送る</summary>
        public static void Send(IEnumerable<SceneAnimation> animations)
        {
            (new BindSceneAnimations(animations)).Push();
            Connecter.PulseMessageThread();
        }
    }

    /// <summary>
    ///
    /// </summary>
    public sealed class EditCurve : AnimationMessage
    {
        private readonly GuiObjectID						targetGuiObjectID_;
        public  uint							parentIndex_;
        public  uint							curveIndex_;
        public  uint							curveComponentIndex_;
        public  bool							isRotate_;
        public  bool							quantized_;
        private readonly AnimTarget						animTarget_;
        public  int								enabledCurveIndex_;
        public  GuiObject						document_;
        public  NintendoWare.G3d.Edit.ValueType valueType_;
        public CurveExportType exportType_;

        private EditCurve(
            GuiObject						animation,
            uint							parentIndex,
            uint							curveIndex,
            uint							curveComponentIndex,
            bool							isRotate,
            AnimTarget						animTarget,
            int								curveIndexInBinary,
            GuiObjectID						id,
            NintendoWare.G3d.Edit.ValueType	valueType,
            bool							quantized,
            CurveExportType exportType
        )
        {
            document_ = animation;
            targetGuiObjectID_		= id;
            parentIndex_			= parentIndex;
            curveIndex_				= curveIndex;
            curveComponentIndex_	= curveComponentIndex;
            isRotate_				= isRotate;
            animTarget_				= ObjectUtility.Clone(animTarget);
            valueType_ = valueType;
            quantized_ = quantized;
            enabledCurveIndex_ = curveIndexInBinary;
            exportType_ = exportType;

            // このメッセージは間引き対象
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            EditCurve target = msg as EditCurve;
            if (target == null)
            {
                return false;
            }

            if (target.document_ == document_ &&
                target.parentIndex_ == parentIndex_ &&
                target.curveIndex_ == curveIndex_ &&
                target.curveComponentIndex_ == curveComponentIndex_)
            {
                return true;
            }

            return false;
        }

        /// <summary>実行</summary>
        public override void Execute()
        {
            Debug.Assert(document_ is Document);

            if (animTarget_.IsFileOutputable == false)
            {
                return;
            }

            switch(targetGuiObjectID_)
            {
                case GuiObjectID.MaterialAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditShaderParamCurve / materialNameIndex:{0}, paramAnimIndex:{1}, valueType:{2}",
    parentIndex_, curveIndex_, valueType_);
                    G3dHioLibProxy.Hio.EditMaterialCurve(parentIndex_, curveComponentIndex_, (int)curveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditShaderParamCurve / materialNameIndex:{0}, paramAnimIndex:{1}, componentIndex:{2}, curveIndex:{3}, valueType:{4}",
                        parentIndex_, curveIndex_, curveComponentIndex_, enabledCurveIndex_, valueType_);
                    G3dHioLibProxy.Hio.EditShaderParamCurve(parentIndex_, curveIndex_, curveComponentIndex_, enabledCurveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                case GuiObjectID.TexturePatternAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditTexPatternCurve / materialNameIndex:{0}, curveIndex:{1}, valueType:{2}",
                        parentIndex_, curveIndex_, valueType_);
                    G3dHioLibProxy.Hio.EditTexPatternCurve(parentIndex_, (int)curveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                case GuiObjectID.MaterialVisibilityAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditMatVisibilityCurve / animIndex:{0}, curveIndex:{1}, valueType:{2}",
                        parentIndex_, enabledCurveIndex_, valueType_);
                    G3dHioLibProxy.Hio.EditMatVisibilityCurve(parentIndex_, enabledCurveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                case GuiObjectID.BoneVisibilityAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditBoneVisibilityCurve / animIndex:{0}, curveIndex:{1}, valueType:{2}",
                        parentIndex_, enabledCurveIndex_, valueType_);
                    G3dHioLibProxy.Hio.EditBoneVisibilityCurve(parentIndex_, enabledCurveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                case GuiObjectID.ShapeAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditShapeCurve / targetIndex:{0}, curveIndex:{1}",
                        parentIndex_, curveIndex_);
                    G3dHioLibProxy.Hio.EditShapeCurve(parentIndex_, (int)curveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                case GuiObjectID.CameraAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditCameraAnimCurve / cameraIndex:{0}, targetIndex:{1}, curveIndex:{2}",
                        parentIndex_, curveComponentIndex_, curveIndex_);
                    G3dHioLibProxy.Hio.EditCameraAnimCurve(parentIndex_, (int)curveComponentIndex_, (int)curveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                case GuiObjectID.LightAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditLightAnimCurve / lightIndex:{0}, targetIndex:{1}, curveIndex:{2}",
                        parentIndex_, curveComponentIndex_,  curveIndex_);
                    G3dHioLibProxy.Hio.EditLightAnimCurve(parentIndex_, (int)curveComponentIndex_, (int)curveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                case GuiObjectID.FogAnimation:
                {
                    DebugConsole.WriteLine("Hio.EditFogAnimCurve / fogIndex:{0}, targetIndex:{1}, curveIndex:{2}",
                         parentIndex_, curveComponentIndex_, curveIndex_);
                    G3dHioLibProxy.Hio.EditFogAnimCurve(parentIndex_, (int)curveComponentIndex_, (int)curveIndex_, new HioCurve(animTarget_, isRotate_, valueType_, quantized_, exportType_));
                    break;
                }

                default:
                {
                    DebugConsole.WriteLine("EditCurve.Execute() -- 未実装");
                    break;
                }
            }
        }

        /// <summary>送る</summary>
        public static void Send(AnimationDocument animation, uint parentIndex, uint curveIndex, uint curveComponentIndex, bool isRotate, AnimTarget animTarget, int curveIndexInBinary, GuiObjectID id, App.Data.PrimitiveTypeKind valueType, bool quantized, CurveExportType exportType)
        {
            Viewer.TransactionMessage.Send(true);
            Viewer.Select.Send(animation);
            (new EditCurve(animation, parentIndex, curveIndex, curveComponentIndex, isRotate, animTarget, curveIndexInBinary, id, ToValueType(valueType), quantized, exportType)).Push();
            Viewer.TransactionMessage.Send(false);
        }

        public static NintendoWare.G3d.Edit.ValueType ToValueType(App.Data.PrimitiveTypeKind type)
        {
            switch (type)
            {
                case PrimitiveTypeKind.Float:
                    return NintendoWare.G3d.Edit.ValueType.Float;
                case PrimitiveTypeKind.Int:
                case PrimitiveTypeKind.Uint:
                    return NintendoWare.G3d.Edit.ValueType.Int;
                case PrimitiveTypeKind.Bool:
                    return NintendoWare.G3d.Edit.ValueType.Bool;
                default:
                    throw new NotImplementedException();
            }
        }
    }

    public sealed class AnimationPauseMessage : AnimationMessage
    {
        public GuiObject target;
        public Model model;
        private readonly float frame;
        private readonly bool pause;
        private AnimationPauseMessage(GuiObject animation, bool pause, float frame, Model model)
        {
            target = animation;
            this.frame = frame;
            this.pause = pause;
            this.model = model;
            IsCompressible = true;
        }

        public override bool IsSameTarget(BaseMessage msg)
        {
            var message = msg as AnimationPauseMessage;
            return message != null && message.target == target && message.model == model;
        }

        public override void Execute()
        {
            Debug.Assert(target is IPause);
            Debug.Assert(target is NintendoWare.G3d.Edit.IEditTarget);
            DebugConsole.WriteLine("EditAnimationPause: {0}, {1}, {2}", (target is Document)? ((Document)target).FileName: target.Name, frame, pause);
            if (((NintendoWare.G3d.Edit.IEditTarget)target).IsAttached && model.IsAttached)
            {
                G3dHioLibProxy.Hio.EditAnimationPause((NintendoWare.G3d.Edit.IEditModelTarget)model, (NintendoWare.G3d.Edit.IEditTarget)target, frame, pause);
            }
            if (pause == false)
            {
                // フレームを設定する
                if (AnimationMessage.IsPlaying)
                {
                    G3dHioLibProxy.Hio.PlayFrameCtrl(AnimationMessage.CurrentFrame);
                }
                else
                {
                    G3dHioLibProxy.Hio.StopFrameCtrl(AnimationMessage.CurrentFrame);
                }
            }
        }

        public static void Send(GuiObject animation, bool pause, float frame, Model model)
        {
            (new AnimationPauseMessage(animation, pause, frame, model)).Push();
        }
    }
}
