﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using App;
using App.ConfigData;
using App.Controls;
using App.Command;
using App.Data;
using App.Utility;
using App.Win32;
using App.res;
using AppConfig;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace Viewer
{
    /// <summary>
    /// ビューアのメッセージ用接続。
    /// </summary>
    public sealed class Connecter
    {
        private readonly Manager manager_ = null;
        private Thread messageThread_ = null;
        private Thread pollingThread_ = null;
        private Thread pollPingThread_ = null;

        // 終了処理
        private volatile bool finalize_ = false;

        // メッセージがある時のインターバル
        // この変数には直接アクセスしてはならない。
        // IntervalExistMessage を使うこと。
        private long intervalExistMessage_ = 50;

        /// <summary>
        /// メッセージがある時のインターバルをスレッドセーフで取得もしくは設定する。
        /// インターバルは他の状態に依存しない独立した変数なので不要なメモリバリアは行っていない。アトミック操作のみ行われる。
        /// </summary>
        public int IntervalExistMessage
        {
            get
            {
                return (int)System.Threading.Interlocked.Read(ref intervalExistMessage_);
            }
            set
            {
                System.Threading.Interlocked.Exchange(ref intervalExistMessage_, value);
            }
        }

        private readonly object FileLoadedSyncRoot = new object();
        private event Action FileLoaded2 = null;
        private readonly object FileReloadedSyncRoot = new object();
        private event Action FileReloaded2 = null;

        private const int intervalPollingThread_	= 30;
        private const int intervalPollPingThread_	= 1000;

        private readonly object PollPingLockObject_ = new object();
        public object PollPingLockObject{ get{ return PollPingLockObject_; } }

        /// <summary>
        /// 接続しているかどうか？
        /// </summary>
        public bool IsConnected{ get { return G3dHioLibProxy.Hio.IsConnected; } }

        /// <summary>
        /// オブジェクトの選択が変更された時のイベント
        /// </summary>
        private void GuiObjectManager_SelectedChanged(object sender, EventArgs e)
        {
            Viewer.ViewerUtility.SendSelectedObjects((GuiObjectGroup)sender, App.ConfigData.ApplicationConfig.Setting.Preview.IsPreviewCameraFocus);
        }

        private object eventHandlerLock = new object();
        public void AddFileReloadedAction(Action action)
        {
            lock(eventHandlerLock)
            {
                FileReloaded2 += action;
            }
        }

        public void RemoveFileReloadedAction(Action action)
        {
            lock(eventHandlerLock)
            {
                FileReloaded2 -= action;
            }
        }

        public void AddFileLoadedAction(Action action)
        {
            lock (eventHandlerLock)
            {
                FileLoaded2 += action;
            }
        }

        public void RemoveFileLoadedAction(Action action)
        {
            lock (eventHandlerLock)
            {
                FileLoaded2 -= action;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Connecter(Manager manager)
        {
            manager_ = manager;

            G3dHioLibProxy.Hio.RecvInfo.AttachModelReceived += OnAttachCommandReceived;
            G3dHioLibProxy.Hio.RecvInfo.DetachModelReceived += OnDetachCommandReceived;
            G3dHioLibProxy.Hio.RecvInfo.FileLoaded += OnFileLoaded;
            G3dHioLibProxy.Hio.RecvInfo.ModelFileLoaded += OnModelFileLoaded;
            G3dHioLibProxy.Hio.RecvInfo.FileReloaded += OnFileReloaded;
            G3dHioLibProxy.Hio.RecvInfo.RenderInfoReceived += OnRenderInfoReceived;
            G3dHioLibProxy.Hio.RecvInfo.ModelLayoutReceived += OnRecvInfo_ModelLayoutReceived;
            G3dHioLibProxy.Hio.RecvInfo.AttachShaderArchiveReceived += OnRecvInfo_AttachShaderArchiveReceived;
            G3dHioLibProxy.Hio.RecvInfo.DetachShaderArchiveReceived += OnRecvInfo_DetachShaderArchiveReceived;
            G3dHioLibProxy.Hio.RecvInfo.UpdatedShaderProgramReceived += OnRecvInfo_UpdateShaderProgramReceived;
            G3dHioLibProxy.Hio.RecvInfo.AbnormalPacketReceived += OnRecvInfo_AbnormalPacketReceived;
            G3dHioLibProxy.Hio.RecvInfo.IncorrectVersionReceived += OnRecvInfo_IncorrectVersionReceived;
            G3dHioLibProxy.Hio.RecvInfo.SelectionTargetReceived += OnRecvInfo_SelectionTargetReceived;
            G3dHioLibProxy.Hio.RecvInfo.CodePageUpdateReceived += OnRecvInfo_CodePageUpdateReceived;
            G3dHioLibProxy.Hio.RecvInfo.RuntimeErrorOccured += RecvInfo_RuntimeErrorOccured;
            G3dHioLibProxy.Hio.RecvInfo.BeginFreezeReceived += OnBeginFreezeReceived;
            G3dHioLibProxy.Hio.RecvInfo.EndFreezeReceived += OnEndFreezeReceived;
            G3dHioLibProxy.Hio.RecvInfo.PlayFrameCtrlReceived += OnRecvInfo_PlayFrameCtrlReceived;
            G3dHioLibProxy.Hio.RecvInfo.StopFrameCtrlReceived += OnRecvInfo_StopFrameCtrlReceived;
            G3dHioLibProxy.Hio.RecvInfo.SendFrameReceived += OnRecvInfo_SendFrameReceived;
            G3dHioLibProxy.Hio.RecvInfo.SendFrameStepReceived += OnRecvInfo_SendFrameStepReceived;
            G3dHioLibProxy.Hio.RecvInfo.SendModelNextAnimReceived += OnRecvInfo_SendModelNextAnimReceived;
            G3dHioLibProxy.Hio.RecvInfo.SendModelPrevAnimReceived += OnRecvInfo_SendModelPrevAnimReceived;
            G3dHioLibProxy.Hio.RecvInfo.ShowMessageRequested += RecvInfo_ShowMessageRequested;

            {
                pollPingThread_ = new Thread(new ThreadStart(PollPingThreadMain))
                {
                    IsBackground = true,
                    Priority = ThreadPriority.AboveNormal,
                    CurrentCulture = Thread.CurrentThread.CurrentCulture,
                    CurrentUICulture = Thread.CurrentThread.CurrentUICulture,
                };
                pollPingThread_.Start();
            }
        }

        private ManualResetEventSlim disconnectResetEvent = new ManualResetEventSlim();
        private ManualResetEventSlim actionResetEvent = new ManualResetEventSlim();
        private void InvokeOnUiThread(Action action, bool wait = true, bool waitCanFocus = false)
        {
            if (wait)
            {
                actionResetEvent.Reset();
            }
            var connectionId = ConnectionId;
            var invoker = new MethodInvoker(
                    () =>
                    {
                        try
                        {
                            if (!disconnectResetEvent.IsSet && connectionId == ConnectionId)
                            {
                                action();

                                // action 実行中に切断処理が走っていた場合 action の影響を消す必要がある
                                if (IsDisconnected)
                                {
                                    ResetStatus();
                                }
                            }
                        }
                        finally
                        {
                            if (wait)
                            {
                                actionResetEvent.Set();
                            }
                        }
                    });
            if (waitCanFocus)
            {
                TheApp.MainFrame.InvokeAfterCanFocus(invoker);
            }
            else
            {
                TheApp.MainFrame.BeginInvoke(invoker);
            }

            if (wait)
            {
                WaitHandle.WaitAny(new[] { disconnectResetEvent.WaitHandle, actionResetEvent.WaitHandle });
            }
        }

        public void PrepareConnect()
        {
            disconnectResetEvent.Reset();
        }

        public void PrepareDisconnect()
        {
            disconnectResetEvent.Set();
        }

        /// <summary>
        /// ランタイムからのメッセージの表示
        /// </summary>
        private void RecvInfo_ShowMessageRequested(object sender, NintendoWare.G3d.Edit.ShowMessageRequestedArgs e)
        {
            InvokeOnUiThread(
                wait:false,
                action:(Action)(()=>
            {
                string messageFormat = Strings.Viewer_MessageFromViewer;
                switch (e.MessageType)
                {
                    case NintendoWare.G3d.Edit.MessageType.RuntimeInfo:
                    case NintendoWare.G3d.Edit.MessageType.RuntimeWarning:
                        messageFormat = Strings.Viewer_RuntimeMessageFromViewer;
                        break;
                    case NintendoWare.G3d.Edit.MessageType.RuntimeError:
                        {
                            // ランタイムエラーの場合は切断
                            Viewer.Manager.Instance.Close();
                            messageFormat = Strings.Viewer_RuntimeError + "\n{0}";
                        }
                        break;
                    default:
                        break;
                }

                switch (e.MessageDestination)
                {
                    case NintendoWare.G3d.Edit.MessageDestination.Dialog:
                        switch (e.MessageType)
                        {
                            case NintendoWare.G3d.Edit.MessageType.UserError:
                            case NintendoWare.G3d.Edit.MessageType.RuntimeError:
                                App.AppContext.NotificationHandler.MessageBoxError(messageFormat, e.Message);
                                break;
                            case NintendoWare.G3d.Edit.MessageType.UserInfo:
                            case NintendoWare.G3d.Edit.MessageType.RuntimeInfo:
                                App.AppContext.NotificationHandler.MessageBoxInformation(messageFormat, e.Message);
                                break;
                            case NintendoWare.G3d.Edit.MessageType.UserWarning:
                            case NintendoWare.G3d.Edit.MessageType.RuntimeWarning:
                                App.AppContext.NotificationHandler.MessageBoxWarning(messageFormat, e.Message);
                                break;
                        }
                        break;
                    case NintendoWare.G3d.Edit.MessageDestination.Log:
                        {
                            MessageLog.LogType type = MessageLog.LogType.Information;
                            switch (e.MessageType)
                            {
                                case NintendoWare.G3d.Edit.MessageType.UserError:
                                case NintendoWare.G3d.Edit.MessageType.RuntimeError:
                                    type = MessageLog.LogType.Error;
                                    break;
                                case NintendoWare.G3d.Edit.MessageType.UserInfo:
                                case NintendoWare.G3d.Edit.MessageType.RuntimeInfo:
                                    type = MessageLog.LogType.Information;
                                    break;
                                case NintendoWare.G3d.Edit.MessageType.UserWarning:
                                case NintendoWare.G3d.Edit.MessageType.RuntimeWarning:
                                    type = MessageLog.LogType.Warning;
                                    break;
                            }
                            App.AppContext.NotificationHandler.WriteMessageLog(type, string.Format(messageFormat, e.Message));
                            break;
                        }
                }
            }));
        }

        /// <summary>
        /// ランタイムでのエラー通知
        /// </summary>
        private void RecvInfo_RuntimeErrorOccured(object sender, NintendoWare.G3d.Edit.RuntimeErrorOccuredEventArgs e)
        {
            InvokeOnUiThread(wait:false,
                action:(Action)(() =>
            {
                // 切断
                Viewer.Manager.Instance.Close();

                // エラー通知
                string errorMessage = Strings.Viewer_RuntimeError + "\n";
                switch (e.RuntimeError)
                {
                    case NintendoWare.G3d.Edit.RuntimeError.NoError:
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.InvalidMaterialCount:
                        errorMessage += Strings.Viewer_InvalidMaterialCount + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.AttachCanceled:
                        errorMessage += Strings.Viewer_AttachCanceled + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.LoadFileFailed:
                        errorMessage += Strings.Viewer_LoadFileFailed + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.InsufficientMemory:
                        errorMessage += Strings.Viewer_InsufficientMemory + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.BindAnimFailed:
                        errorMessage += Strings.Viewer_BindAnimFailed + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.RetargetHostModelNotFound:
                        errorMessage += Strings.Viewer_RetargetHostModelNotFound + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.DuplicateModelObjKey:
                        errorMessage += Strings.Viewer_DuplicateModelObjKey + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.InvalidShaderDetected:
                        errorMessage += Strings.Viewer_InvalidShaderDetected + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.InvalidModelAttached:
                        errorMessage += Strings.Viewer_InvalidModelAttached + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.TargetModelNotFound:
                        errorMessage += Strings.Viewer_TargetModelNotFound + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.OpenFileFailed:
                        errorMessage += Strings.Viewer_OpenFileFailed + "\n";
                        break;
                    case NintendoWare.G3d.Edit.RuntimeError.UnknownError:
                        errorMessage += Strings.Viewer_UnknownError + "\n";
                        break;

                }

                errorMessage += Strings.Viewer_Disconnected;
                UIMessageBox.Error(errorMessage);
            }));

        }

        public void Destroy()
        {
            if (pollPingThread_ != null)
            {
                pollPingThread_.Abort();
                // Abort 後に必要な処理がないので Join しない
                pollPingThread_ = null;
            }
        }

        public bool ConnectMessageReceiver(TeamConfig.PlatformPreset platform, ConfigCommon.HioTarget target)
        {
            // すぐに接続すると接続に失敗する
            var interval = DateTime.Now - lastDisconnectTime;
            if (interval.TotalMilliseconds < 100)
            {
                Thread.Sleep(Math.Max(0, 100 - (int)interval.TotalMilliseconds));
            }
            if (ConnectHio(platform, target) == false)
            {
                //Debug.Assert(false);
                DisconnectHio();
                return false;
            }

            return true;
        }

        private void DisconnectMessageReceiver()
        {
            DisconnectHio();
            lastDisconnectTime = DateTime.Now;
        }

        public void AddSelectedEvent()
        {
            // 選択オブジェクト変更イベントを登録する
            App.AppContext.SelectedTargetChanged += GuiObjectManager_SelectedChanged;
        }

        public void RemoveSelectedEvent()
        {
            // 選択オブジェクト変更イベントを抜く
            App.AppContext.SelectedTargetChanged -= GuiObjectManager_SelectedChanged;
        }

        private static readonly object connectionIdLock = new object();
        private static int connectionId = 0;
        public static int ConnectionId
        {
            get
            {
                lock (connectionIdLock)
                {
                    return connectionId;
                }
            }
            set
            {
                lock (connectionIdLock)
                {
                    connectionId = value;
                }
            }
        }


        /// <summary>
        /// 接続する
        /// </summary>
        public void Connect(TeamConfig.PlatformPreset platform, ConfigCommon.HioTarget target)
        {
            if (ConnectMessageReceiver(platform, target) == false)
            {
                DebugConsole.WriteLine("connect failed.");
                return;
            }

            ConnectionId++;
            IsDisconnected = false;
            finalize_ = false;
            PrepareConnect();

            // スレッド開始
            {
                messageThread_ = new Thread(new ThreadStart(MessageThreadMain))
                {
                    IsBackground = true,
                    Priority = ThreadPriority.AboveNormal,
                    CurrentCulture = Thread.CurrentThread.CurrentCulture,
                    CurrentUICulture = Thread.CurrentThread.CurrentUICulture,
                };
                messageThread_.Start();
            }
            {
                pollingThread_ = new Thread(new ThreadStart(PollThreadMain))
                {
                    IsBackground = true,
                    Priority = ThreadPriority.AboveNormal,
                    CurrentCulture = Thread.CurrentThread.CurrentCulture,
                    CurrentUICulture = Thread.CurrentThread.CurrentUICulture,
                };
                pollingThread_.Start();
            }

            Thread.Sleep(0);

            // 選択オブジェクト変更イベントを登録する
            AddSelectedEvent();
        }

        public static object AbortLock = new object();

        /// <summary>
        /// 閉じる。
        /// </summary>
        public void Close()
        {
            // 選択オブジェクト変更イベントを抜く
            RemoveSelectedEvent();

            if (manager_.IsExistMessages)
            {
                // メッセージが残っていたら、削除する。
                manager_.ClearMessages();
            }

            finalize_ = true;

            // 切断
            DisconnectMessageReceiver();

            // スレッドの停止を待つ
            {
                if (messageThread_ != null)
                {
                    lock (AbortLock)
                    {
                        messageThread_.Abort();
                    }
                    bool joined = messageThread_.Join(1000);
                    //					Debug.Assert(joined);

                    if (joined == false)
                    {
                        DebugConsole.WriteLine("FALSE : messageThread_.Join(1000)");
                    }

                    messageThread_ = null;
                }
                if (pollingThread_ != null)
                {
                    lock (AbortLock)
                    {
                        pollingThread_.Abort();
                    }
                    bool joined = pollingThread_.Join(1000);
                    //					Debug.Assert(joined);

                    if (joined == false)
                    {
                        DebugConsole.WriteLine("FALSE : pollingThread_.Join(1000)");
                    }

                    pollingThread_ = null;
                }
            }

            ResetStatus();
            IsDisconnected = true;
        }

        // ResetStatus も含めて切断処理が完了したか
        public bool IsDisconnected = true;

        public void ResetStatus()
        {
            lock (ModelFileLoadedActions)
            {
                ModelFileLoadedActions.Clear();
            }

            // 通信が切れたので、ドキュメントのg3d::edit との通信情報をリセット
            foreach (var document in DocumentManager.Documents)
            {
                // TODO: タイミングの確認
                document.ResetStatus();
                document.UnloadedFromHio();
            }

            DocumentManager.PreviewingSceneAnimations.Clear();
            lock (DocumentManager.PreviewingModels)
            {
                DocumentManager.PreviewingModels.Clear();
            }

            lock (DocumentManager.PreviewingTextures)
            {
                foreach (var texture in DocumentManager.PreviewingTextures.Keys)
                {
                    texture.ResetStatus();
                    texture.UnloadedFromHio();
                }
                DocumentManager.PreviewingTextures.Clear();
            }

            lock (DocumentManager.TextureBindings)
            {
                DocumentManager.TextureBindings.Clear();
                DocumentManager.BindingTextures.Clear();
            }

            SuspendUnloadTexture.Reset();
            SetFrameCount.lastFrameCount = -1;
            Select._currentType = GuiObjectID.DummyObject;
            QueryRenderInfoQueue.Clear();

            // pollingThread 用の初期化
            IsPollingFreezing = false;
            RenderInfoPackFromHio.Clear();

            lock (BeforeLoadModelActions)
            {
                BeforeLoadModelActions.Clear();
            }
        }

        private static readonly object MessageThreadMainMonitorObject = new object();

        public static void PulseMessageThread()
        {
            if (TheApp.MainFrame != null)
            {
                TheApp.MainFrame.UpdateCurrentFrame();
            }
            lock (Connecter.MessageThreadMainMonitorObject)
            {
                System.Threading.Monitor.Pulse(Connecter.MessageThreadMainMonitorObject);
            }
        }

        DateTime lastDisconnectTime = new DateTime();
        //---------------------------------------------------------------------
        /// <summary>
        /// 実行。
        /// </summary>
        private void MessageThreadMain()
        {
            try
            {
                // 切断後すぐは不安定なので待つ
                var interval = DateTime.Now - lastDisconnectTime;
                if (interval.TotalMilliseconds < 1000)
                {
                    Thread.Sleep(1000 - Math.Max((int)interval.TotalMilliseconds, 0));
                }
                DebugConsole.WriteLine(interval.TotalMilliseconds.ToString());
                while (true)
                {
                    //Thread.Sleep(IntervalExistMessage);
                    lock (MessageThreadMainMonitorObject)
                    {
                        Monitor.Wait(MessageThreadMainMonitorObject, IntervalExistMessage);
                    }

                    // 切断されていたら、抜ける。
                    if (IsConnected == false)
                    {
                        if (finalize_ == false)
                        {
                            if (TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                            {
                                TheApp.MainFrame.BeginInvoke(
                                    new MethodInvoker(
                                        () => TheApp.MainFrame.ConnectToHio(false))
                                );

                                break;
                            }
                        }
                        return;
                    }
                    MessageThreadProcess();
                }
            }
            catch (ThreadAbortException)
            {
                // Abortに対しては特に何も行わない
                return;
            }
            catch (Exception exception)
            {
                if (finalize_) { return; }
                string message = Strings.Viewer_ExceptionMsg + "\n\n";
                message += exception.Message;

#if DEBUG
                message += "\n" + exception.StackTrace;
#endif

                DebugConsole.WriteLine(message);
                UIMessageBox.Error(message);
                if (!finalize_ && TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    TheApp.MainFrame.BeginInvoke(
                        new MethodInvoker(
                            () =>
                            {
                                // 例外で切断されたときは自動接続を解除する
                                G3dHioLibProxy.Hio.IsAutoConnection = false;
                                TheApp.MainFrame.ConnectToHio(false);
                            })
                    );
                }
            }
            finally
            {
                TransactionMessage.Initialize();
                lastDisconnectTime = DateTime.Now;
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// スレッド処理
        /// </summary>
        private void MessageThreadProcess()
        {
            // メッセージキューに溜まっているメッセージを処理します
            if(manager_.IsExistMessages)
            {
                try
                {
                    manager_.IsExecuting = true;

                    foreach (var message in manager_.PopMessages())
                    {
                        // 切断されていたら、抜ける。
                        if (IsConnected == false)
                        {
                            return;
                        }

                        DebugConsole.WriteLine("■message.Execute() : {0}", message.GetType());

                        message.Execute();
                        lock (sync_)
                        {
                            Monitor.Pulse(sync_);
                        }
                    }
                }
                finally
                {
                    manager_.IsExecuting = false;
                }
            }
        }

        // プラットフォーム
        // 例: "Cafe"
        public static TeamConfig.PlatformPreset Platform;

        private bool ConnectHio(TeamConfig.PlatformPreset platform, ConfigCommon.HioTarget target)
        {
            Platform = platform;
            G3dHioLibProxy.ChangeCommDevice(target, platform);

            if (G3dHioLibProxy.Hio.IsConnected == false)
            {
                return G3dHioLibProxy.Connect();
            }

            return false;
        }

        // 必要かどうか不明だけど Poll 中にDisconnect しないようにしておく
        private static object disconnectLock = new object();
        private void DisconnectHio()
        {
            PrepareDisconnect();
            lock (disconnectLock)
            {
                G3dHioLibProxy.Hio.Disconnect();
            }
        }

        private void Poll()
        {
            lock (TransactionMessage.sync)
            {
                lock (disconnectLock)
                {
                    if (IsConnected)
                    {
                        G3dHioLibProxy.Hio.Poll();
                    }
                }
            }
        }

        private static readonly object sync_ = new object();
        private bool IsPollingFreezing = false;
        private void PollThreadMain()
        {
            try
            {
                while (IsConnected)
                {
                    lock (sync_)
                    {
                        Monitor.Wait(sync_, intervalPollingThread_);
                    }

                    Poll();
                }
            }
            catch(ThreadAbortException)
            {
                // Abortに対しては特に何も行わない
                return;
            }
            catch(Exception exception)
            {
                if (finalize_) { return; }
                string message = Strings.Viewer_ExceptionMsg + "\n\n";
                message += exception.Message;

#if DEBUG
                message += "\n" + exception.StackTrace;
#endif

                DebugConsole.WriteLine(message);
                UIMessageBox.Error(message);
                if (!finalize_ && TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    TheApp.MainFrame.BeginInvoke(
                        new MethodInvoker(
                            () => TheApp.MainFrame.ConnectToHio(false))
                    );
                }
            }
        }

        private void OnEndFreezeReceived(object sender, NintendoWare.G3d.Edit.EndFreezeReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.ResetEndFreezeReceived");
            IsPollingFreezing = false;

            // RenderInfo 用の処理
            EndFreezeReceivedRenderInfo();
        }

        private void OnBeginFreezeReceived(object sender, NintendoWare.G3d.Edit.BeginFreezeReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.ResetBeginFreezeReceived");
            IsPollingFreezing = true;
        }


        public static readonly object SyncPollPing = new object();
        private void PollPingThreadMain()
        {
            try
            {
                while(true)
                {
                    // intervalPollPingThread_ 毎または Monitor.Pulse されるたびに処理する
                    lock (SyncPollPing)
                    {
                        Monitor.Wait(SyncPollPing, intervalPollPingThread_);
                    }

                    lock(PollPingLockObject)
                    {
                        G3dHioLibProxy.Hio.PollPing();
                    }

                    PollPingThreadMain_AutoConnection();
                }
            }
            catch(ThreadAbortException)
            {
                // Abortに対しては特に何も行わない
                return;
            }
            catch(Exception exception)
            {
                if (finalize_) { return; }
                string message = Strings.Viewer_ExceptionMsg + "\n\n";
                message += exception.Message;

#if DEBUG
                message += "\n" + exception.StackTrace;
#endif

                DebugConsole.WriteLine(message);
                UIMessageBox.Error(message);
                if (!finalize_ && TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    TheApp.MainFrame.BeginInvoke(
                        new MethodInvoker(
                            () => TheApp.MainFrame.ConnectToHio(false))
                    );
                }
            }
        }

        public static Dictionary<string, Action<Model>> BeforeLoadModelActions = new Dictionary<string, Action<Model>>();

        private void OnAttachCommandReceived(object sender, NintendoWare.G3d.Edit.AttachModelReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsAttachModelReceived");
            DebugConsole.WriteLine("OnAttachCommandRecieved");

            Document target = null;

            uint resFileKey	= 0;
            uint resModelKey = 0;
            var modelObjKey	= arg.ModelObjKey;
            var recvInfoAttachName = arg.AttachName;
            var recvInfoAttachPathName = arg.AttachPath;

            {
                // モデルロード
                Action<string[]> loadModel = (fileNames) =>
                {
                    var modelName = Path.GetFileNameWithoutExtension(fileNames[0]);
                    lock (BeforeLoadModelActions)
                    {
                        BeforeLoadModelActions[modelName] =
                        model =>
                        {
                            model.ModelObjKey = modelObjKey;
                            model.IsSendAttached = true;
                            model.IsAttached = true;
                        };
                    }

                    {
                        DocumentManager.LoadFromFile(fileNames, showErrorDialog:false);
                        target = DocumentManager.Models.FirstOrDefault(x => x.Name == modelName);
                        if (target == null)
                        {
                            lock (BeforeLoadModelActions)
                            {
                                BeforeLoadModelActions.Remove(modelName);
                            }
                        }
                    }
                };

                Action<string, Model> reopenModel = (fileName, oldModel) =>
                {
                    lock (BeforeLoadModelActions)
                    {
                        BeforeLoadModelActions[oldModel.Name] =
                            model =>
                            {
                                model.ModelObjKey = modelObjKey;
                                model.IsSendAttached = true;
                                model.IsAttached = true;
                            };
                    }

                    {
                        var commandSet = new EditCommandSet();
                        DocumentManager.LoadFromFile(Enumerable.Repeat(fileName,1), showErrorDialog: false, commandSet:commandSet);
                        var newModel = DocumentManager.Models.FirstOrDefault(x => x.Name == oldModel.Name);
                        if (newModel != null)
                        {
                            target = newModel;
                            commandSet.Add(
                                DocumentManager.CreateAnimationSetEditCommand(newModel, oldModel.AnimationSets.ToArray()).Execute());
                            commandSet.Add(DocumentManager.CreateAnimationsEditCommand(newModel.DefaultAnimationSet, oldModel.DefaultAnimationSet.Animations.ToArray(), false).Execute());
                            commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(newModel.AllAnimations.Distinct()).Execute());
                            newModel.PreviewAnimSet = newModel.AnimationSetsWithDefault.FirstOrDefault(x => x.Name == (oldModel.PreviewAnimSet != null ? oldModel.PreviewAnimSet.Name : null));
                        }
                        else
                        {
                            BeforeLoadModelActions.Remove(oldModel.Name);
                        }

                        TheApp.CommandManager.Add(commandSet);

                        /*
                        // エラーや通信切断が起きた場合は resetEvent2.Set が呼ばれないので定期的に条件確認
                        while (manager_.IsExistMessages || manager_.IsExecuting)
                        {
                            if (resetEvent2.Wait(200))
                            {
                                break;
                            }
                        }*/
                    }
                };

                // 既存モデルをアタッチ
                Action attachExistringModel = () =>
                {
                    Debug.Assert(target is Model);
                    var model = target as Model;

                    // 表示状態にする
                    model.PreviewInfo.Visible = true;
                    // プロジェクトの変更マーク更新用
                    DocumentManager.ProjectDocument.SetMaybeModified();

                    List<Tuple<string, AnimationDocument>> previewAnimations;
                    if (model.PreviewAnimSet != null)
                    {
                        var key = new KeyValuePair<object, string>(model.ModelId, model.PreviewAnimSet.Name);
                        previewAnimations = DocumentManager.GetAnimations(model.PreviewAnimSet.Animations).Where(x => !x.Pause.InvisibleBinds.Contains(key)).Select(x => new Tuple<string, AnimationDocument>(x.FileName, x)).ToList();
                    }
                    else
                    {
                        previewAnimations = new List<Tuple<string, AnimationDocument>>();
                    }

                    var modelDataList = new List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>>();
                    {
                        var docs = BaseMessage.GetDocuments(model).Where(x => !(x is AnimationDocument));
                        foreach (var doc in docs)
                        {
                            var data =
                                new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>(
                                    (IntermediateFileDocument)doc,
                                    BaseMessage.CopyNW4F3DIF((IntermediateFileDocument)doc),
                                    BaseMessage.GetStream((IntermediateFileDocument)doc),
                                    doc.Name
                                );
                            modelDataList.Add(data);
                        }

                        if (model.Materials.Any(x => x.ParentMaterials.Any()))
                        {
                            var resolvedModel = modelDataList[0].Item2.Item as modelType;
                            Debug.Assert(resolvedModel != null, "resolvedModel != null");
                            Debug.Assert(model.Materials.Count == resolvedModel.material_array.length);
                            Viewer.LoadOrReloadModel.ResolveParentMaterials(model, resolvedModel);
                        }
                    }

                    List<ShaderDefinition> shaderDefinitions;
                    List<string> shadingModels;
                    var ShaderOptimizeData = HioUtility.CreateShaderOptimizeData(model, out shaderDefinitions, out shadingModels);
                    var modelName = model.Name;
                    var dumpShaderSourcePath = ApplicationConfig.UserSetting.IO.DumpShaderSource ? TemporaryFileUtility.ShaderFolderPath : null;
                    var optimizeShader = DocumentManager.OptimizeShader;
                    Viewer.Close.Send(
                        model,
                        () =>
                        {
                            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1391#19
                            // todo:無限ループになる可能性がある。
                            while (model.IsAttached)
                            {
                                DebugConsole.WriteLine("sync model.IsAttached:false");
                                Thread.Sleep(30);
                                Poll();
                            }

                            model.ResetStatus();

                            model.ResFileKey		= resFileKey;
                            model.ResModelKey		= resModelKey;
                            model.ModelObjKey		= modelObjKey;

                            model.IsSendAttached	= true;
                            model.IsAttached		= true;
                            model.HioLoaded = true;

                            using (var block = new LoadProgressDialog.DialogBlock(Strings.Viewer_Loading, () => FileMessage.DisconnectOnMainThreadAsync(), true))
                            {
                                // モデルバイナリ生成
                                string tmp = block.Message;
                                block.Message = string.Format(Strings.Viewer_ConvertingModel, modelName);
                                uint alignmentSize;
                                var bfresFileName = HioUtility.CreateModelBinaryFile(modelDataList, true, out alignmentSize);
                                block.Message = tmp;

                                Debug.Assert(bfresFileName != null);

                                var textures = new List<Texture>();
                                if (Connecter.Platform.UseNw)
                                {
                                    foreach (var item in modelDataList)
                                    {
                                        if (item.Item1.ObjectID == GuiObjectID.Texture)
                                        {
                                            model.loadedTextures.Add(item.Item4, new Tuple<byte[], List<G3dStreamCachedComparer>>(
                                                ObjectUtility.ToBytes(item.Item2.Item),
                                                item.Item3.Select(x => new G3dStreamCachedComparer(x)).ToList()));
                                        }
                                    }
                                }
                                else
                                {
                                    // テクスチャを送る
                                    foreach (var item in modelDataList)
                                    {
                                        if (item.Item1.ObjectID == GuiObjectID.Texture)
                                        {
                                            HioUtility.LoadOrReloadOrSkipTexture((Texture)item.Item1, item.Item2, item.Item3, item.Item4, block);
                                            HioUtility.AddTextureBinding((Texture)item.Item1, model);
                                            textures.Add((Texture)item.Item1);
                                        }
                                    }
                                }

                                {
                                    G3dHioLibProxy.Hio.SendAttachModel(model, new NintendoWare.G3d.Edit.FileData() { FileName = bfresFileName, Alignment = alignmentSize });

                                    if (!Connecter.Platform.UseNw)
                                    {
                                        // TODO: 必要なら更新があったときだけ送る
                                        DebugConsole.WriteLine("HIO.SetTextureBindings --------------- " + target.Name + " " + textures.Count());
                                        G3dHioLibProxy.Hio.SetTextureBindingsForModel(target as NintendoWare.G3d.Edit.IEditModelTarget, textures.OfType<NintendoWare.G3d.Edit.IEditTextureTarget>().ToArray());
                                    }

                                    if (ShaderOptimizeData.Any())
                                    {
                                        foreach (var material in ShaderOptimizeData.SelectMany(x => x.materials))
                                        {
                                            material.forceOptimize = material.optimize;
                                        }

                                        // シェーダー転送
                                        bool binarizeError;
                                        HioUtility.LoadOptimizedShader(model, ShaderOptimizeData, modelDataList, out binarizeError, true, shaderDefinitions, shadingModels, block, modelName, dumpShaderSourcePath, optimizeShader, false);
                                    }
                                    lock (DocumentManager.PreviewingModels)
                                    {
                                        DocumentManager.PreviewingModels.Add(model);
                                    }
                                }

#if false
                                G3dHioLibProxy.Hio.BindAnimations(model, previewAnimations.Select(x => x.Item2).ToArray());
                                model.PreviewingAnimations.AddRange(previewAnimations);
#endif
                            }
                        }
                    );

                    // バインド
                    Viewer.BindAnimations.Send(model, previewAnimations.Select(x => x.Item2));

                    // バインド関係が更新されることがあるので再転送
                    foreach (var model1 in DocumentManager.Models)
                    {
                        if (model1 != model && model1.PreviewInfo.BindModelName == model.Name)
                        {
                            model1.SendEditBoneBind();
                            model1.SendEditModelLayout(false, sendChildren: true);
                        }
                    }
                };

                InvokeOnUiThread(wait:false, waitCanFocus:true,
                    action: () =>
                        {
                            // 切断されていたら、無視
                            if (IsConnected == false)
                            {
                                return;
                            }

                            // 既にアタッチされていれば要求を無視
                            if (DocumentManager.Models.Any(x => x.IsAttached && x.ModelObjKey == modelObjKey))
                            {
                                G3dHioLibProxy.CancelAttachModel(modelObjKey);
                                return;
                            }

                            // ・AttachPathName が指定されていない時には必ずダイアログからfmd を選ぶようにして下さい。
                            var attachPathName = recvInfoAttachPathName;
                            if (PathUtility.NormalizePathString(System.Environment.ExpandEnvironmentVariables(recvInfoAttachPathName), ref attachPathName) == false)
                            {
                                attachPathName = PathUtility.ToWindowsPathSeparator(attachPathName);
                            }

                            // resetEvent0
                            {
                                if ((string.IsNullOrEmpty(recvInfoAttachPathName) == false) &&
                                    (ApplicationConfig.FileIo.GetExistsAttachFullPathName(attachPathName) != null) &&
                                    (DocumentManager.Models.Any(x => x.Name == recvInfoAttachName)))
                                {
                                    // 既に読み込まれているモデルにアタッチする
                                    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1391

                                    var fullPath = Path.GetFullPath(ApplicationConfig.FileIo.GetExistsAttachFullPathName(attachPathName));
                                    target = DocumentManager.Models.FirstOrDefault(x => x.Name == recvInfoAttachName);

                                    var targetFilePath = target.FilePath;
                                    if (PathUtility.NormalizePathString(target.FilePath, ref targetFilePath) == false)
                                    {
                                        targetFilePath = PathUtility.ToWindowsPathSeparator(targetFilePath);
                                    }

                                    if (!string.Equals(targetFilePath, fullPath, StringComparison.OrdinalIgnoreCase))
                                    {
                                        // AttachPathName と 3DEditor に読み込まれている fmd の名前が
                                        // 不一致の場合にはその旨（パスが不一致なので 3DEditor に読み込まれている
                                        // fmd を閉じてやり直してもらう）をダイアログ表示して
                                        // アタッチをキャンセルしてください。

                                        UIMessageBox.Warning(
                                            string.Format(
                                                Strings.Viewer_PathIsMismatch,
                                                target.FilePath,
                                                recvInfoAttachPathName));

                                        G3dHioLibProxy.CancelAttachModel(modelObjKey);
                                    }
                                    else
                                    {
                                        if (target.IsModifiedObject || target.ContentsModified)
                                        {
                                            // すでに読み込まれているモデルに変更がある場合にはその状態でアタッチするか、
                                            // モデルを開きなおすか、アタッチをキャンセルするかをダイアログで確認します。
                                            // 開きなおす場合には保存するかも確認します。

                                            Utility.WakeupWindow(TheApp.MainFrame.Handle, false);

                                            using (
                                                var dialog = new OnAttachCommandReceivedDialog_Modified(recvInfoAttachName))
                                            {
                                                dialog.ShowDialog();

                                                switch (dialog.AttachMode)
                                                {
                                                        // 変更済みの既存モデルをアタッチ
                                                    case OnAttachCommandReceivedDialog_Modified.Mode.ModifiedModel:
                                                        {
                                                            attachExistringModel();
                                                            break;
                                                        }

                                                        // モデルを開き直してアタッチ
                                                    case OnAttachCommandReceivedDialog_Modified.Mode.ReloadedModel:
                                                        {
                                                            if (DocumentManager.RemoveDocumentWithChildren(target, null, false, false))
                                                            {
                                                                reopenModel(target.FilePath, (Model)target);
                                                            }
                                                            else
                                                            {
                                                                target = null;
                                                            }

                                                            break;
                                                        }

                                                        // アタッチをキャンセル
                                                    case OnAttachCommandReceivedDialog_Modified.Mode.Cancel:
                                                        {
                                                            G3dHioLibProxy.CancelAttachModel(modelObjKey);
                                                            break;
                                                        }
                                                }
                                            }

                                        }
                                        else
                                        {
                                            // モデルに変更がない場合には
                                            // 既存モデルにアタッチするか、アタッチをキャンセルするかを確認します。

                                            Utility.WakeupWindow(TheApp.MainFrame.Handle, false);

                                            using (
                                                var dialog =
                                                    new OnAttachCommandReceivedDialog_NonModified(recvInfoAttachName))
                                            {
                                                dialog.ShowDialog();

                                                switch (dialog.AttachMode)
                                                {
                                                        // 既存モデルをアタッチ
                                                    case OnAttachCommandReceivedDialog_NonModified.Mode.ExistringModel:
                                                        {
                                                            attachExistringModel();
                                                            break;
                                                        }

                                                        // アタッチをキャンセル
                                                    case OnAttachCommandReceivedDialog_NonModified.Mode.Cancel:
                                                        {
                                                            G3dHioLibProxy.CancelAttachModel(modelObjKey);
                                                            break;
                                                        }
                                                }
                                            }
                                        }
                                    }
                                }
                                else
                                {
                                    string[] fileNames = null;
                                    {
                                        if (string.IsNullOrEmpty(recvInfoAttachPathName) == false)
                                        {
                                            var fullPath = ApplicationConfig.FileIo.GetExistsAttachFullPathName(attachPathName);
                                            if (fullPath != null)
                                            {
                                                fileNames = new []{fullPath};
                                            }
                                            else
                                            {
                                                // 環境変数
                                                var envVars = StringUtility.MakeEnvironmentVariablesFromPathString(recvInfoAttachPathName).ToArray();
                                                if (envVars.Any() == false)
                                                {
                                                    App.Controls.UIMessageBox.Warning(string.Format(Strings.IO_Load_FileNotExistsWithFilename, recvInfoAttachPathName));
                                                }
                                                else
                                                {
                                                    var temp = string.Empty;

                                                    foreach(var envVar in envVars)
                                                    {
                                                        var v = System.Environment.GetEnvironmentVariable(envVar) ?? Strings.Connecter_UndefinedEnvironmentVariable;
                                                        temp += "%" + envVar + "% : " + v + "\n";
                                                    }

                                                    App.Controls.UIMessageBox.Warning(string.Format(Strings.IO_Load_FileNotExistsWithFilename, recvInfoAttachPathName) + "\n\n--\n" + Strings.Connecter_EnvironmentVariable + "\n" + temp);
                                                }
                                            }
                                        }
                                    }

                                    Utility.WakeupWindow(TheApp.MainFrame.Handle, false);

                                    var dialogTitle = string.Format(Strings.Viewer_FileAttachDialogTitle, recvInfoAttachName);
                                    var isDialogOk  = (fileNames == null) && DialogUtility.ExecuteOpenFileDialog(out fileNames, GuiObjectID.Model, false, dialogTitle);

                                    if ((fileNames != null) && (fileNames.Length == 0))
                                    {
                                        fileNames = null;
                                    }

                                    if ((fileNames != null) || isDialogOk)
                                    {
                                        var fmdFilePath	= fileNames[0];
                                        var fmdFileName = Path.GetFileNameWithoutExtension(fmdFilePath);

                                        var loadedFmd = DocumentManager.Models.FirstOrDefault(x => x.Name.Equals(fmdFileName, StringComparison.OrdinalIgnoreCase));

                                        if ((loadedFmd != null) &&
                                            string.Equals(fmdFileName ,loadedFmd.Name, StringComparison.OrdinalIgnoreCase) &&
                                            !string.Equals(fmdFilePath, loadedFmd.FilePath, StringComparison.OrdinalIgnoreCase))
                                        {
                                            // また、AttachPathName が指定されていない時には必ずダイアログから
                                            // fmd を選ぶようにして下さい。
                                            // （AttachName と fmd の名前が必ず一致してるとは限らないため。）
                                            // このときも上記と同様に、ダイアログ指定の fmd と 3DEditor に
                                            // 読み込まれている fmd の名前が一致、パス不一致の場合には
                                            // ダイアログを表示してアタッチをキャンセルしてください。

                                            UIMessageBox.Warning(
                                                string.Format(
                                                    Strings.Viewer_PathIsMismatch,
                                                    loadedFmd.FilePath,
                                                    fmdFilePath
                                                )
                                            );

                                            target = null;
                                        }
                                        else
                                        {
                                            if (loadedFmd == null)
                                            {
                                                loadModel(fileNames);
                                            }
                                            else
                                            {
                                                // AttachPathName が使われていないときにダイアログからアタッチモデルを
                                                // 選択した後、既に 3DEditor に読み込まれているモデルが編集中であったら
                                                // 編集中のモデルを使うか開きなおすかをダイアログで確認してください。
                                                // 開きなおす場合は保存するかも確認してください。

                                                if (loadedFmd.IsModifiedObject || loadedFmd.ContentsModified)
                                                {
                                                    Utility.WakeupWindow(TheApp.MainFrame.Handle, false);

                                                    using (var dialog = new OnAttachCommandReceivedDialog_Modified(recvInfoAttachName))
                                                    {
                                                        dialog.ShowDialog();

                                                        switch(dialog.AttachMode)
                                                        {
                                                            // 変更済みの既存モデルをアタッチ
                                                            case OnAttachCommandReceivedDialog_Modified.Mode.ModifiedModel:
                                                            {
                                                                target = loadedFmd;
                                                                attachExistringModel();
                                                                break;
                                                            }

                                                            // モデルを開き直してアタッチ
                                                            case OnAttachCommandReceivedDialog_Modified.Mode.ReloadedModel:
                                                            {
                                                                // 閉じて開く
                                                                if (DocumentManager.RemoveDocumentWithChildren(loadedFmd, null, false, false))
                                                                {
                                                                    reopenModel(loadedFmd.FilePath, loadedFmd);
                                                                }
                                                                else
                                                                {
                                                                    target = null;
                                                                }
                                                                break;
                                                            }

                                                            // アタッチをキャンセル
                                                            case OnAttachCommandReceivedDialog_Modified.Mode.Cancel:
                                                            {
                                                                target = null;
                                                                break;
                                                            }
                                                        }
                                                    }
                                                }
                                                else
                                                {
                                                    target = loadedFmd;
                                                    attachExistringModel();
                                                }
                                            }
                                        }
                                    }

                                    if (target == null)
                                    {
                                        G3dHioLibProxy.CancelAttachModel(modelObjKey);
                                    }
                                }
                            }


                            if (target != null)
                            {
                                // ビューを更新する
                                App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
                            }
                        }
                    );
            }
        }

        private void OnDetachCommandReceived(object sender, NintendoWare.G3d.Edit.DetachModelReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsDetachModelReceived");
            Document target = null;

            var resetEvent = new ManualResetEventSlim();
            {
                InvokeOnUiThread(waitCanFocus:true,
                        action:() =>
                        {
                            // 切断されていたら、無視
                            if (IsConnected == false)
                            {
                                resetEvent.Set();
                                return;
                            }

                            target = DocumentManager.EditModelTargets.FirstOrDefault(x => ((NintendoWare.G3d.Edit.IEditModelTarget)x).ModelObjKey == arg.ModelObjKey);
                            if (target != null)
                            {
                                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=802
                                // １、Hio.SendAttachModel()　で送られたものはUnloadModel を実行しない。
                                // ２、Hio.SendAttachModel()　で送られていないものはUnloadModel を実行。

                                Debug.Assert(target is NintendoWare.G3d.Edit.IEditModelTarget);
                                var editModelTarget = (NintendoWare.G3d.Edit.IEditModelTarget)target;

                                Debug.Assert(target is Model);
                                var model = target as Model;

                                if (model.IsSendAttached)
                                {
                                    editModelTarget.ResetStatus();
                                    target.UnloadedFromHio();
                                    model.IsSendAttached_BeforeResetStatus = true;
                                }

                                // デタッチされたものは非表示に変える
                                Viewer.Close.Send(model);

                                // Visible の変更はコマンドの対象ではない
                                model.PreviewInfo.Visible = false;

                                // プロジェクトの変更マーク更新用
                                DocumentManager.ProjectDocument.SetMaybeModified();
                                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
                            }
                        }
                    );
            }
        }

        // ロード待ち状態のイベント
        public static Dictionary<uint, NintendoWare.G3d.Edit.IEditTarget> WaitingLoadFiles
            = new Dictionary<uint, NintendoWare.G3d.Edit.IEditTarget>();

        private void OnFileLoaded(object sender, NintendoWare.G3d.Edit.FileLoadedEventArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsFileLoaded");
            lock (FileLoadedSyncRoot)
            {
                NintendoWare.G3d.Edit.IEditTarget target = null;

                {
                    InvokeOnUiThread(
                            () =>
                            {
                                lock (Connecter.WaitingLoadFiles)
                                {
                                    Connecter.WaitingLoadFiles.TryGetValue(arg.ToolKey, out target);
                                }

                                if (target is Document)
                                {
                                    DebugConsole.WriteLine("Key: " + target.Key);
                                    ((Document)target).ResFileKey = arg.ResFileKey;
                                    ((Document)target).IsAttached = true;
                                }
                                else
                                {
                                    DebugConsole.WriteLine("ToolKey: " + arg.ToolKey);

                                    // Load 後に送信スレッドが停止した可能性がある。
                                    //Debug.Assert(false);
                                }
                            }
                        );
                }
            }

            lock (eventHandlerLock)
            {
                if (FileLoaded2 != null)
                {
                    FileLoaded2();
                }
            }
        }

        // ロード待ち状態のイベント
        public static Dictionary<uint, Tuple<Model, ManualResetEventSlim>> WaitingModels
            = new Dictionary<uint, Tuple<Model, ManualResetEventSlim>>();

        private void OnModelFileLoaded(object sender, NintendoWare.G3d.Edit.ModelFileLoadedEventArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsModelFileLoaded");
            DebugConsole.WriteLine("OnModelFileLoaded() - start");
            Document target = null;

            var resetEvent = new ManualResetEventSlim();
            {
                InvokeOnUiThread(
                        () =>
                        {
                            Tuple<Model, ManualResetEventSlim> item;
                            lock (Connecter.WaitingModels)
                            {
                                if (Connecter.WaitingModels.TryGetValue(arg.ToolKey, out item))
                                {
                                    target = item.Item1;
                                }
                            }

                            if (target != null)
                            {
                                Debug.Assert(target is NintendoWare.G3d.Edit.IEditModelTarget);

                                if (target is Model)
                                {
                                    var model = target as Model;

                                    model.ResFileKey	= arg.ResFileKey;
                                    model.ResModelKey =   arg.ResModelKey;
                                    model.ModelObjKey =   arg.ModelObjKey;
                                    model.IsAttached	= true;

                                    lock (ModelFileLoadedActions)
                                    {
                                        Action<Model> action;
                                        if (ModelFileLoadedActions.TryGetValue(model.ModelObjKey, out action))
                                        {
                                            model.SetLayoutActionOnLoad = () => action(model);
                                            ModelFileLoadedActions.Remove(model.ModelObjKey);
                                        }
                                    }
                                }
                                else
                                {
                                    Debug.Assert(false);	// 未実装
                                }

                                lock (Connecter.WaitingModels)
                                {
                                    // 例外が起きて Dispose されているとよくないので念のために取り直す
                                    if (Connecter.WaitingModels.TryGetValue(arg.ToolKey, out item))
                                    {
                                        item.Item2.Set();
                                    }
                                    else
                                    {
                                        Debug.Assert(false);
                                    }
                                }
                            }
                        }
                    );
            }
            DebugConsole.WriteLine("OnModelFileLoaded() - end");
            //LoadProgressDialog.Hide();
        }

        public static Dictionary<uint, NintendoWare.G3d.Edit.IEditTarget> WaitingReloads
    = new Dictionary<uint, NintendoWare.G3d.Edit.IEditTarget>();
        private void OnFileReloaded(object sender, NintendoWare.G3d.Edit.FileReloadedEventArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsFileReloaded");

            // 注意:
            // FileReloaded は、AddFileReloadedAction(), RemoveFileReloadedAction() で
            // 操作される可能性があるので lock する

            lock (FileReloadedSyncRoot)
            {
                NintendoWare.G3d.Edit.IEditTarget target = null;
                {

                    InvokeOnUiThread(
                            () =>
                            {
                                lock (WaitingReloads)
                                {
                                    Connecter.WaitingReloads.TryGetValue(arg.ResFileKey, out target);
                                }

                                if (target is AnimationDocument)
                                {
                                    ((AnimationDocument)target).ResFileKey = arg.NewResFileKey;
                                }
                                else
                                {

                                }
                            }
                        );
                }

                //LoadProgressDialog.Hide();
            }

            lock (eventHandlerLock)
            {
                if (FileReloaded2 != null)
                {
                    FileReloaded2();
                }
            }
        }

        private readonly List<NintendoWare.G3d.Edit.RenderInfoPack> RenderInfoPackFromHio = new List<NintendoWare.G3d.Edit.RenderInfoPack>();
        private void OnRenderInfoReceived(object sender, NintendoWare.G3d.Edit.RenderInfoReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsRenderInfoReceived");
            DebugConsole.WriteLine("OnRenderInfoReceived");
            RenderInfoPackFromHio.Add(CopyRenderInfoPack(arg.SentRenderInfoPack));
            if (!IsPollingFreezing)
            {
                SetRenderInfo(RenderInfoPackFromHio);
            }
        }

        private void SetRenderInfo(List<NintendoWare.G3d.Edit.RenderInfoPack> renderInfoPacks)
        {
            if (renderInfoPacks.Any())
            {
                InvokeOnUiThread(
                        () =>
                            {
                                using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                                {
                                    Viewer.TransactionMessage.Send(true, overInterval: true, waitRuntimeState: true);
                                    var targetMaterials = new List<Material>();
                                    foreach (var renderInfoPack in renderInfoPacks)
                                    {
                                        DebugConsole.WriteLine("OnRenderInfoReceived : Invoked");
                                        var target = DocumentManager.Models.FirstOrDefault(x => x.ModelObjKey == renderInfoPack.ModelObjKey);
                                        if (target != null)
                                        {
                                            Debug.Assert(renderInfoPack.MaterialIndex < target.Materials.Count());
                                            var targetMaterial = target.Materials[renderInfoPack.MaterialIndex];

                                            targetMaterial.SetRenderInfoPackFromHio(new RenderInfoPackFromHio(renderInfoPack,
                                                                                                              targetMaterial.MaterialShaderAssign
                                                                                                                            .ShadingModel));

                                            // ビューを更新する
                                            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(targetMaterial, null));
                                        }
                                    }

                                    QueryRenderInfoQueue.Send();
                                    Viewer.TransactionMessage.Send(false, overInterval: true, waitRuntimeState: true);
                                }
                            }
                        );
                renderInfoPacks.Clear();
            }
        }

        public NintendoWare.G3d.Edit.RenderInfoPack CopyRenderInfoPack(NintendoWare.G3d.Edit.RenderInfoPack pack)
        {
            var newPack = new NintendoWare.G3d.Edit.RenderInfoPack()
                {
                    ModelObjKey = pack.ModelObjKey,
                    MaterialIndex = pack.MaterialIndex,
                };
            foreach (var item in pack.Items)
            {
                newPack.Items.Add(item);
            }

            return newPack;
        }

        private void EndFreezeReceivedRenderInfo()
        {
            SetRenderInfo(RenderInfoPackFromHio);
        }

        /// <summary>
        /// モデルがロードされたときに実行するアクションを登録 key:ModelObjKey
        /// </summary>
        public static Dictionary<uint, Action<Model>> ModelFileLoadedActions = new Dictionary<uint, Action<Model>>();

        private void OnRecvInfo_ModelLayoutReceived(object sender, NintendoWare.G3d.Edit.ModelLayoutReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsModelLayoutReceived");
            {
                InvokeOnUiThread(
                        () =>
                        {
                            var target = (Model)DocumentManager.EditModelTargets.FirstOrDefault(x => ((Model)x).ModelObjKey == arg.ModelObjKey);
                            if (target != null)
                            {
                                target.PreviewInfo.Scale.Set(
                                    arg.Scale.X,
                                    arg.Scale.Y,
                                    arg.Scale.Z
                                );

                                target.PreviewInfo.Rotate.Set(
                                    MathUtility.ToDegree(arg.Rotate.X),
                                    MathUtility.ToDegree(arg.Rotate.Y),
                                    MathUtility.ToDegree(arg.Rotate.Z)
                                );

                                target.PreviewInfo.Translate.Set(
                                    arg.Translate.X,
                                    arg.Translate.Y,
                                    arg.Translate.Z
                                );

                                DocumentManager.ProjectDocument.SetMaybeModified();
                                DebugConsole.WriteLine("OnRecvInfo_ModelLayoutReceived");
                                // ビューを更新する
                                App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
                            }
                            else
                            {
                                var sx=arg.Scale.X;
                                var sy=arg.Scale.Y;
                                var sz=arg.Scale.Z;
                                var rx=MathUtility.ToDegree(arg.Rotate.X);
                                var ry=MathUtility.ToDegree(arg.Rotate.Y);
                                var rz = MathUtility.ToDegree(arg.Rotate.Z);
                                var tx=arg.Translate.X;
                                var ty=arg.Translate.Y;
                                var tz = arg.Translate.Z;

                                lock (ModelFileLoadedActions)
                                {

                                    ModelFileLoadedActions[arg.ModelObjKey] = (Action<Model>)(model =>
                                        {
                                            //model.ReceivedModelLayoutOnLoad = true; // 同期をとっているので送信スレッドとのコンフリクトはないはず。
                                            model.PreviewInfo.Scale.Set(
                                                arg.Scale.X,
                                                arg.Scale.Y,
                                                arg.Scale.Z
                                            );

                                            model.PreviewInfo.Rotate.Set(
                                                MathUtility.ToDegree(arg.Rotate.X),
                                                MathUtility.ToDegree(arg.Rotate.Y),
                                                MathUtility.ToDegree(arg.Rotate.Z)
                                            );

                                            model.PreviewInfo.Translate.Set(
                                                arg.Translate.X,
                                                arg.Translate.Y,
                                                arg.Translate.Z
                                            );

                                            DocumentManager.ProjectDocument.SetMaybeModified();
                                            DebugConsole.WriteLine("OnRecvInfo_ModelLayoutReceived");
                                            // ビューを更新する
                                            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(model, null));

                                            model.WaitingQueryModelLayout = false;
                                        });
                                }
                            }

                            if (target != null)
                            {
                                target.WaitingQueryModelLayout = false;
                            }
                        }
                    );
            }
        }

        private void OnRecvInfo_AttachShaderArchiveReceived(object sender, NintendoWare.G3d.Edit.AttachShaderArchiveReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsAttachShaderArchiveReceived");

            {
                InvokeOnUiThread(
                        () =>
                        {
                            // 切断されていたら、無視
                            if (IsConnected == false)
                            {
                                return;
                            }

                            // 既にアタッチされていれば要求を無視
                            if (DocumentManager.ShaderDefinitions.Any(x => x.IsAttached && x.ShaderArchiveKey == arg.ShaderArchiveKey))
                            {
                                G3dHioLibProxy.CancelAttachShaderArchive(arg.ShaderArchiveKey);
                                return;
                            }

                            var attachPathName = arg.AttachPath;
                            if (PathUtility.NormalizePathString(System.Environment.ExpandEnvironmentVariables(arg.AttachPath), ref attachPathName) == false)
                            {
                                attachPathName = PathUtility.ToWindowsPathSeparator(attachPathName);
                            }

                            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1389
                            // ・AttachPath が指定されていない時には必ずダイアログからfsd を選ぶようにして下さい。
                            // ・既に読み込まれているシェーダーにアタッチする
                            if ((string.IsNullOrEmpty(arg.AttachPath) == false) &&
                                (ApplicationConfig.FileIo.GetExistsAttachFullPathName(attachPathName) != null) &&
                                DocumentManager.ShaderDefinitions.Any(x => x.Name == arg.AttachName))
                            {
                                var fullPath = Path.GetFullPath(ApplicationConfig.FileIo.GetExistsAttachFullPathName(attachPathName));
                                var target = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == arg.AttachName);

                                var targetFilePath = target.FilePath;
                                if (PathUtility.NormalizePathString(target.FilePath, ref targetFilePath) == false)
                                {
                                    targetFilePath = PathUtility.ToWindowsPathSeparator(targetFilePath);
                                }

                                if (!string.Equals(targetFilePath, fullPath, StringComparison.OrdinalIgnoreCase))
                                {
                                    // AttachPath と 3DEditor に読み込まれている fsd の名前が
                                    // 不一致の場合にはその旨（パスが不一致なので 3DEditor に読み込まれている
                                    // fsd を閉じてやり直してもらう）をダイアログ表示して
                                    // アタッチをキャンセルしてください。

                                    UIMessageBox.Warning(
                                        string.Format(
                                            Strings.Viewer_PathIsMismatch,
                                            target.FilePath,
                                            arg.AttachPath
                                        )
                                    );

                                    G3dHioLibProxy.CancelAttachShaderArchive(arg.ShaderArchiveKey);
                                }
                                else
                                {
                                    lock (target)
                                    {
                                        if (!Viewer.Manager.IsDeviceTargeted && !target.CompileTested)
                                        {
                                            // コンパイルテストを行う
                                            byte[] result;
                                            if (ShaderDifinitionUtility.RebuildShaderDefinition(target, out result, true, Connecter.Platform) == null)
                                            {
                                                target.FailedToBinarize = true;
                                            }

                                            target.CompileTested = true;
                                        }

                                        if (!target.FailedToBinarize)
                                        {
                                            target.IsAttached = true;

                                            // キーを保存
                                            target.ShaderArchiveKey = arg.ShaderArchiveKey;
                                            target.updatedAttachedShadingModel.Clear();

                                            // HIOに送る
                                            G3dHioLibProxy.Hio.SendAttachShaderArchive(target);

                                            // GLバイナリ生成オプションを取得
                                            target.LastIsAttachShaderArchiveBinary = arg.IsAttachShaderArchiveBinary;

                                            if (target.ModifiedNotMaterialShadingModelIndices.Any())
                                            {
                                                G3dHioLibProxy.Hio.EditShadingModels(target, target.ModifiedNotMaterialShadingModelIndices.OrderBy(x => x).ToArray());
                                            }

                                            if (target.ModifiedNotMaterialShadingModelIndices.Any())
                                            {
                                                Viewer.LoadOptimizedShaderArchive.OnShaderDefinitionChanged(target,
                                                    target.ModifiedNotMaterialShadingModelIndices.OrderBy(x => x).Select(x => target.Definitions[x].Name).ToArray());
                                            }
                                        }
                                        else
                                        {
                                            UIMessageBox.Warning(string.Format(Strings.Viewer_CancelAttachForBinarizeError, target.Name));

                                            G3dHioLibProxy.CancelAttachShaderArchive(arg.ShaderArchiveKey);
                                        }
                                    }

                                    // ビューを更新する
                                    App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
                                }
                            }
                            else
                            {
                                string[] fileNames = null;
                                {
                                    if (string.IsNullOrEmpty(arg.AttachPath) == false)
                                    {
                                        var fullPath = ApplicationConfig.FileIo.GetExistsAttachFullPathName(attachPathName);
                                        if (fullPath != null)
                                        {
                                            fileNames = new []{fullPath};
                                        }
                                        else
                                        {
                                            // AttachPath が見つからない場合は警告ダイアログを出す。

                                            // 環境変数
                                            var envVars = StringUtility.MakeEnvironmentVariablesFromPathString(arg.AttachPath).ToArray();
                                            if (envVars.Any() == false)
                                            {
                                                App.Controls.UIMessageBox.Warning(string.Format(Strings.IO_Load_FileNotExistsWithFilename, arg.AttachPath));
                                            }
                                            else
                                            {
                                                var temp = string.Empty;

                                                foreach(var envVar in envVars)
                                                {
                                                    var v = System.Environment.GetEnvironmentVariable(envVar) ?? Strings.Connecter_UndefinedEnvironmentVariable;
                                                    temp += "%" + envVar + "% : " + v + "\n";
                                                }

                                                App.Controls.UIMessageBox.Warning(string.Format(Strings.IO_Load_FileNotExistsWithFilename, arg.AttachPath) + "\n\n--\n" + Strings.Connecter_EnvironmentVariable + "\n" + temp);
                                            }
                                        }
                                    }
                                }

                                Utility.WakeupWindow(TheApp.MainFrame.Handle, false);

                                var dialogTitle = string.Format(Strings.Viewer_FileAttachDialogTitle, arg.AttachName);
                                var isDialogOk  = (fileNames == null) && DialogUtility.ExecuteOpenFileDialog(out fileNames, GuiObjectID.ShaderDefinition, false, dialogTitle);

                                if ((fileNames != null) && (fileNames.Length == 0))
                                {
                                    fileNames = null;
                                }

                                if ((fileNames != null) || isDialogOk)
                                {
                                    var fsdFilePath	= fileNames[0];
                                    var fsdFileName = Path.GetFileNameWithoutExtension(fsdFilePath);

                                    var loadedFsd = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.FileName == Path.GetFileName(fsdFilePath));

                                    if ((loadedFsd != null) &&
                                        string.Equals(fsdFileName, loadedFsd.Name, StringComparison.OrdinalIgnoreCase) &&
                                        !string.Equals(fsdFilePath, loadedFsd.FilePath, StringComparison.OrdinalIgnoreCase))
                                    {
                                        // また、AttachPath が指定されていない時には必ずダイアログから
                                        // fsd を選ぶようにして下さい。
                                        // （AttachName と fsd の名前が必ず一致してるとは限らないため。）
                                        // このときも上記と同様に、ダイアログ指定の fsd と 3DEditor に
                                        // 読み込まれている fsd の名前が一致、パス不一致の場合には
                                        // ダイアログを表示してアタッチをキャンセルしてください。

                                        UIMessageBox.Warning(
                                            string.Format(
                                                Strings.Viewer_PathIsMismatch,
                                                loadedFsd.FilePath,
                                                fsdFilePath
                                            )
                                        );

                                        G3dHioLibProxy.CancelAttachShaderArchive(arg.ShaderArchiveKey);
                                    }
                                    else
                                    {
                                        if (loadedFsd == null)
                                        {
                                            // ファイルを読み込む
                                            DocumentManager.LoadFromFile(fileNames);
                                        }

                                        var target = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.FileName == Path.GetFileName(fsdFilePath));
                                        if (target == null)
                                        {
                                            G3dHioLibProxy.CancelAttachShaderArchive(arg.ShaderArchiveKey);
                                        }
                                        else
                                        {
                                            lock (target)
                                            {
                                                if (!Viewer.Manager.IsDeviceTargeted && !target.CompileTested)
                                                {
                                                    // コンパイルテストを行う
                                                    byte[] result;
                                                    if (ShaderDifinitionUtility.RebuildShaderDefinition(target, out result, true, Connecter.Platform) == null)
                                                    {
                                                        target.FailedToBinarize = true;
                                                        if (target.IsAttached)
                                                        {
                                                            // 既にアタッチされている!？
                                                            Debug.Assert(false);

                                                            //HioUtility.UnloadShaderArchive(target, false);
                                                        }
                                                    }

                                                    target.CompileTested = true;
                                                }

                                                if (!target.FailedToBinarize)
                                                {
                                                    target.IsAttached = true;

                                                    // キーを保存
                                                    target.ShaderArchiveKey = arg.ShaderArchiveKey;
                                                    target.updatedAttachedShadingModel.Clear();

                                                    // HIOに送る
                                                    G3dHioLibProxy.Hio.SendAttachShaderArchive(target);

                                                    // GLバイナリ生成オプションを取得
                                                    target.LastIsAttachShaderArchiveBinary = arg.IsAttachShaderArchiveBinary;

                                                    if (target.ModifiedNotMaterialShadingModelIndices.Any())
                                                    {
                                                        G3dHioLibProxy.Hio.EditShadingModels(target, target.ModifiedNotMaterialShadingModelIndices.OrderBy(x => x).ToArray());
                                                    }

                                                    if (target.ModifiedNotMaterialShadingModelIndices.Any())
                                                    {
                                                        Viewer.LoadOptimizedShaderArchive.OnShaderDefinitionChanged(target,
                                                            target.ModifiedNotMaterialShadingModelIndices.OrderBy(x => x).Select(x => target.Definitions[x].Name).ToArray());
                                                    }
                                                }
                                                else
                                                {
                                                    UIMessageBox.Warning(string.Format(Strings.Viewer_CancelAttachForBinarizeError, target.Name));

                                                    G3dHioLibProxy.CancelAttachShaderArchive(arg.ShaderArchiveKey);
                                                }
                                            }
                                            // ビューを更新する
                                            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
                                        }
                                    }
                                }
                                else
                                {
                                    G3dHioLibProxy.CancelAttachShaderArchive(arg.ShaderArchiveKey);
                                }
                            }
                        }
                    );
            }
        }

        private void OnRecvInfo_DetachShaderArchiveReceived(object sender, NintendoWare.G3d.Edit.DetachShaderArchiveReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsDetachShaderArchiveReceived");

            {
                InvokeOnUiThread(waitCanFocus:true,
                    action:() =>
                        {
                            // 切断されていたら、無視
                            if (IsConnected == false)
                            {
                                return;
                            }

                            var target =
                                DocumentManager.ShaderDefinitions.FirstOrDefault(
                                    x => ((NintendoWare.G3d.Edit.IEditShaderArchiveTarget)x).ShaderArchiveKey == arg.ShaderArchiveKey
                                );

                            if (target != null)
                            {
                                var shaderArchiveTarget = target as NintendoWare.G3d.Edit.IEditShaderArchiveTarget;

                                shaderArchiveTarget.ResetStatus();
                                target.UnloadedFromHio();

                                // スタートアップで開いたファイルは閉じない
                                if (target.OpenedFromStartUp == false)
                                {
                                    bool cancel = false;
                                    // shaderDefinitionの場合、保存はしないが、閉じるかどうか？の確認はする。
                                    if (target.IsModifiedObject)
                                    {
                                        Utility.WakeupWindow(TheApp.MainFrame.Handle, false);

                                        UIMessageBox msgBox = new UIMessageBox(Strings.IO_CloseModifiedShaderDifinitionFile,
                                                                                UIMessageBoxButtons.YesNo,
                                                                                MessageBoxIcon.Information,
                                                                                MessageBoxDefaultButton.Button1,
                                                                                false);
                                        if (msgBox.ShowDialog() != System.Windows.Forms.DialogResult.Yes)
                                        {
                                            cancel = true;
                                        }
                                    }

                                    // ファイルを閉じる
                                    if (!cancel)
                                    {
                                        using (var block = new DocumentsCloseDialog.CustomizeBlock(isDisableCancelButton: true))
                                        {
                                            DocumentManager.RemoveDocumentWithChildren(target, null, false, false);
                                        }
                                    }
                                }

                                if (target.ModifiedNotMaterialShadingModelIndices.Any())
                                {
                                    Viewer.LoadOptimizedShaderArchive.OnShaderDefinitionChanged(target,
                                        target.ModifiedNotMaterialShadingModelIndices.OrderBy(x => x).Select(x => target.Definitions[x].Name).ToArray());
                                }

                                // シェーダーページ更新のために通知
                                App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
                            }
                        }
                    );
            }
        }

        private void OnRecvInfo_UpdateShaderProgramReceived(object sender, NintendoWare.G3d.Edit.UpdatedShaderProgramReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsUpdateShaderProgramReceived");
            {
                InvokeOnUiThread(
                        () =>
                        {
                            // 動作
                            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1007

                            var target = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.ShaderArchiveKey == arg.ShaderArchiveKey);
                            if (target != null)
                            {
                                var data = ObjectUtility.Clone(target.Data);
                                data.shading_model_array = Enumerable.Repeat(data.shading_model_array.shading_model[arg.ShadingModelIndex], 1)
                                    .ToG3dArrayElement<shading_modelType, shading_model_arrayType>();

                                string tempDirPath = ShaderDifinitionUtility.GetShaderDefinitionTemporaryPath(target);
                                string fsdbFilePath = tempDirPath + "\\" + target.Name + "_" + arg.ShadingModelIndex.ToString() + ".fsdb";
                                string shadingModelName = data.shading_model_array.GetItems().First().name;

                                // choice を書き換える
                                {
                                    ShaderDefinition shaderDefinition = new ShaderDefinition(data, target.BinaryStreams);
                                    shaderDefinition.Name = Path.GetFileNameWithoutExtension(fsdbFilePath);
                                    shaderDefinition.FileDotExt = Path.GetExtension(fsdbFilePath);

                                    Debug.Assert(shaderDefinition != null);
                                    Debug.Assert(shaderDefinition.Definitions.Count == 1);			// 1つしか無いはず

                                    var definition = shaderDefinition.Definitions[0];

                                    if (definition.Data.option_var_array != null && arg.ShaderCompileInfoArray.Count != definition.Data.option_var_array.Count)
                                    {
                                        DebugConsole.WriteLine("Options in received ShaderCompileInfoArray:");
                                        for (int index = 0; index < arg.ShaderCompileInfoArray.Count; ++index)
                                        {
                                            NintendoWare.G3d.Edit.IShaderCompileInfo info = arg.ShaderCompileInfoArray[index];
                                            DebugConsole.WriteLine(
                                                string.Format("[{0}] Option = {1}, Choice = {2}", index, info.Option, info.Choice));
                                        }

                                        DebugConsole.WriteLine("Options in fsd:");
                                        for (int index = 0; index < definition.Data.option_var_array.Count; ++index)
                                        {
                                            option_varType option = definition.Data.option_var_array.Items[index];
                                            DebugConsole.WriteLine(
                                                string.Format("[{0}] Option = {1}, Choice = {2}", index, option.id, option.choice));
                                        }

                                        UIMessageBox.Warning(Strings.Viewer_ShaderCompileInfoArrayCount_Illegal
                                            + string.Format("(ResShaderArchive {0}) != (fsd {1})\n", arg.ShaderCompileInfoArray.Count, definition.Data.option_var_array.Count));

                                        // 失敗したのでコンバートさせないようにするため null にしておく
                                        fsdbFilePath = null;
                                    }
                                    else
                                    {
                                        foreach (var info in arg.ShaderCompileInfoArray)
                                        {
                                            var option = definition.Data.option_var_array.option_var.FirstOrDefault(x => x.id == info.Option);
                                            if (option != null)
                                            {
                                                option.choice = info.Choice;
                                                option.@default = info.Choice;
                                            }
                                        }

                                        // 書き換えた状態をファイルに書き出す
                                        (new DocumentSaver()).WriteDocument(shaderDefinition, fsdbFilePath, false);
                                    }
                                }

                                // fsd作成までは成功している
                                if (fsdbFilePath != null)
                                {
                                    // シェーダーをバイナリコンバートする
                                    var bfshaFilePath = TemporaryFileUtility.MakeTemporaryFileName(".bfsha");
                                    bool forPC = !Viewer.Manager.IsDeviceTargeted;
                                    lock (target)
                                    {
                                        ShdrcvtrManager.TargetType targetType = ShdrcvtrManager.TargetType.Cafe;
                                        bool gl_binary = false;
                                        bool gl_source = false;
                                        if (forPC)
                                        {
                                            targetType = ShdrcvtrManager.TargetType.PC;
                                            if (target.LastIsAttachShaderArchiveBinary)
                                            {
                                                gl_binary = true;
                                            }
                                            else
                                            {
                                                gl_source = true;
                                            }
                                        }

                                        bool binaryConverted = !target.FailedToBinarize &&
                                            ShdrcvtrManager.ConvertShaderToBinaryForViewer(fsdbFilePath, bfshaFilePath, gl_source, gl_binary, targetType, Connecter.Platform, G3dHioLibProxy.Hio.GetRuntimeAddressType() == NintendoWare.G3d.Edit.PlatformAddressType.Adress32,
                                                null);
                                        if (!binaryConverted)
                                        {
                                            target.FailedToBinarize = true;

                                            // Unload は行わず無視する
                                            //HioUtility.UnloadShaderArchive(target, false);

                                            // バイナリコンバート失敗
                                            //UIMessageBox.Error(App.res.Strings.ShaderConvertError_Binarize, target.FileName);

                                            // ビューを更新する
                                            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
                                        }
                                        else
                                        {

                                            DebugConsole.WriteLine("ReloadShaderProgram");
                                            bool ok = G3dHioLibProxy.Hio.ReloadShaderProgram(
                                                target,
                                                arg.ShadingModelIndex,
                                                arg.ShaderProgramIndex,
                                                bfshaFilePath
                                            );
                                            Debug.Assert(ok);

                                            target.updatedAttachedShadingModel.Add(shadingModelName);

                                            // ビューを更新する
                                            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
                                        }
                                    }
                                }
                                else
                                {
                                }
                            }
                        }
                    );
            }
        }

        private void OnRecvInfo_AbnormalPacketReceived(object sender, NintendoWare.G3d.Edit.AbnormalPacketReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsAbnormalPacketReceived");

            InvokeOnUiThread(
                        () =>
                        {
                            Viewer.Manager.Instance.Close();

                            App.Controls.UIMessageBox.Warning(App.res.Strings.Viewer_AbnormalPacketReceived);
                        }
                    );
        }

        private void OnRecvInfo_IncorrectVersionReceived(object sender, NintendoWare.G3d.Edit.IncorrectVersionReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆arg.IsIncorrectVersionReceived");
            InvokeOnUiThread(
                    () =>
                    {
                        var runtimeVersion = string.Format("{0}.{1}.{2}.{3}",
                            arg.MajorVersion,
                            arg.MinorVersion,
                            arg.MicroVersion,
                            arg.BugFixVersion);
                        var editorVersion = string.Format("{0}.{1}.{2}.{3}",
                            G3dHioLibProxy.Hio.MajorVersion,
                            G3dHioLibProxy.Hio.MinorVersion,
                            G3dHioLibProxy.Hio.MicroVersion,
                            G3dHioLibProxy.Hio.BugFixVersion);
                        Viewer.Manager.Instance.Close();
                        App.Controls.UIMessageBox.Warning(Strings.Viewer_IncorrectVersionReceived, runtimeVersion, editorVersion);
                    }
                );
        }

        private void OnRecvInfo_SelectionTargetReceived(object sender, NintendoWare.G3d.Edit.SelectionTargetReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsSelectionTargetReceived");

                InvokeOnUiThread(
                        () =>
                        {
                            int materialSelectionInfoCount = 0;
                            int clearMaterialSelectionInfoCount = 0;

                            // モーダルダイアログが開かれているときは無視する
                            if (!TheApp.MainFrame.CanFocus)
                            {
                                // 無視した場合はログに出す
                                MessageLog.Write(MessageLog.LogType.Information, Strings.Viewer_SelectionTargetIgnored);
                            }
                            else if (ControlUtility.IsAnyCapturing())
                            {
                                // キャプチャ中も無視
                                MessageLog.Write(MessageLog.LogType.Information, Strings.Viewer_SelectionTargetIgnoredMouseCapture);
                            }
                            else
                            {
                                foreach (var info in arg.SelectionTargetInfoArray)
                                {
                                    // マテリアル選択の場合
                                    {
                                        var selInfo = info as NintendoWare.G3d.Edit.MaterialSelectionInfo;
                                        if (selInfo != null)
                                        {
                                            OnRecvInfo_SelectionTargetReceivedInternal_MaterialSelectionInfo(selInfo, materialSelectionInfoCount);
                                            ++materialSelectionInfoCount;
                                            continue;
                                        }
                                    }

                                    // マテリアル選択解除の場合
                                    {
                                        var selInfo = info as NintendoWare.G3d.Edit.ClearMaterialSelectionInfo;
                                        if (selInfo != null)
                                        {
                                            OnRecvInfo_SelectionTargetReceivedInternal_ClearMaterialSelectionInfo();
                                            ++clearMaterialSelectionInfoCount;
                                            continue;
                                        }
                                    }
                                }
                            }
                        }
                    );
        }

        private void OnRecvInfo_SelectionTargetReceivedInternal_MaterialSelectionInfo(NintendoWare.G3d.Edit.MaterialSelectionInfo info, int count)
        {
            // 指定モデルを探す
            var target = DocumentManager.Models.FirstOrDefault(x => x.ModelObjKey == info.ModelObjKey);
            if (target != null)
            {
                var selected =
                    (count == 0) ?
                        new GuiObjectGroup() :
                        new GuiObjectGroup(App.AppContext.SelectedTarget);

                foreach(var index in info.MaterialIndices)
                {
                    if (selected.Contains(target.Materials[index]) == false)
                    {
                        selected.Add(target.Materials[index], false);
                    }
                }

                if (!selected.IsEmpty && selected.Active == null)
                {
                    selected.Active = selected.Objects[0];
                }

                App.AppContext.SelectedTarget.Set(selected.Objects, selected.Active);

                // 選択アイテムが見える位置までスクロールする
                var list = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.MaterialList) as App.ObjectView.List.ObjectListView;
                Debug.Assert(list != null);
                list.EnsureSelectedItemVisible();

                // ダイアログを表示する
                App.PropertyEdit.ObjectPropertyDialog.ShowPropertyDialog();
            }
        }

        private void OnRecvInfo_SelectionTargetReceivedInternal_ClearMaterialSelectionInfo()
        {
            var selected = new GuiObjectGroup(App.AppContext.SelectedTarget);

            // マテリアルを削除する
            selected.Remove(GuiObjectID.Material);

            if (!selected.IsEmpty && selected.Active == null)
            {
                selected.Active = selected.Objects[0];
            }

            App.AppContext.SelectedTarget.Set(selected.Objects, selected.Active);
        }

        private void OnRecvInfo_CodePageUpdateReceived(object sender, NintendoWare.G3d.Edit.CodePageUpdateReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsCodePageUpdateReceived");

            {
                InvokeOnUiThread(
                        () =>
                        {
                            G3dHioLibProxy.CodePage = arg.CodePage;
                        }
                    );
            }
        }

        private void OnRecvInfo_PlayFrameCtrlReceived(object sender, NintendoWare.G3d.Edit.PlayFrameCtrlReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsPlayFrameCtrlReceived");
            InvokeOnUiThread(
                () =>
                {
                    using (var block = new App.PlayControl.NoSendPlayFrameCtrlBlock())
                    {
                        DebugConsole.WriteLine("G3dHioLibProxy.Hio.RecvInfo.Frame" + arg.Frame);
                        TheApp.MainFrame.PreviewCurrentFrame = arg.Frame;
                        TheApp.MainFrame.PlayAnimation();
                    }
                });
        }

        private void OnRecvInfo_StopFrameCtrlReceived(object sender, NintendoWare.G3d.Edit.StopFrameCtrlReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsStopFrameCtrlReceived");
            InvokeOnUiThread(
                () =>
                {
                    using (var block = new App.PlayControl.NoSendPlayFrameCtrlBlock())
                    {
                        DebugConsole.WriteLine("G3dHioLibProxy.Hio.RecvInfo.Frame" + arg.Frame);
                        TheApp.MainFrame.PauseAnimation();
                        TheApp.MainFrame.PreviewCurrentFrame = arg.Frame;
                    }
                });
        }

        private void OnRecvInfo_SendFrameReceived(object sender, NintendoWare.G3d.Edit.SendFrameReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsSendFrameReceived");
            InvokeOnUiThread(
                () =>
                {
                    using (var block = new App.PlayControl.NoSendPlayFrameCtrlBlock())
                    {
                        // StopFrameCtrlReceived と同じ処理
                        DebugConsole.WriteLine("G3dHioLibProxy.Hio.RecvInfo.Frame" + arg.Frame);
                        TheApp.MainFrame.PauseAnimation();
                        TheApp.MainFrame.PreviewCurrentFrame = arg.Frame;
                    }
                });
        }

        private void OnRecvInfo_SendFrameStepReceived(object sender, NintendoWare.G3d.Edit.SendFrameStepReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsSendFrameStepReceived");
            InvokeOnUiThread(
                () =>
                {
                    using (var block = new App.PlayControl.NoSendPlayFrameCtrlBlock())
                    {
                        DebugConsole.WriteLine("G3dHioLibProxy.Hio.RecvInfo.FrameStep" + arg.FrameStep);
                        TheApp.MainFrame.SetFrameScale(arg.FrameStep);
                    }
                });
        }

        private void OnRecvInfo_SendModelNextAnimReceived(object sender, NintendoWare.G3d.Edit.SendModelNextAnimReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsSendModelNextAnimReceived");
            InvokeOnUiThread(
                () =>
                {
                    var model = DocumentManager.Models.FirstOrDefault(x => x.ModelObjKey == arg.ModelObjKey);
                    if (model != null)
                    {
                        var orderedAnimationSet = model.AnimationSets.OrderBy(x => x.Name).ToArray();
                        var index = Array.IndexOf(orderedAnimationSet, model.PreviewAnimSet);
                        if (index + 1 < orderedAnimationSet.Length)
                        {
                            model.PreviewAnimSet = orderedAnimationSet[index + 1];
                        }
                        else
                        {
                            model.PreviewAnimSet = model.DefaultAnimationSet;
                        }
                    }
                });
        }

        private void OnRecvInfo_SendModelPrevAnimReceived(object sender, NintendoWare.G3d.Edit.SendModelPrevAnimReceivedArgs arg)
        {
            DebugConsole.WriteLine("◆G3dHioLibProxy.Hio.RecvInfo.IsSendModelPrevAnimReceived");
            InvokeOnUiThread(
                () =>
                {
                    var model = DocumentManager.Models.FirstOrDefault(x => x.ModelObjKey == arg.ModelObjKey);
                    if (model != null)
                    {
                        var reverseOrderedAnimationSet = model.AnimationSets.OrderBy(x => x.Name).Reverse().ToArray();
                        var index = Array.IndexOf(reverseOrderedAnimationSet, model.PreviewAnimSet);
                        if (index + 1 < reverseOrderedAnimationSet.Length)
                        {
                            model.PreviewAnimSet = reverseOrderedAnimationSet[index + 1];
                        }
                        else
                        {
                            model.PreviewAnimSet = model.DefaultAnimationSet;
                        }
                    }
                });
        }

        private void PollPingThreadMain_AutoConnection()
        {
            // 自動接続が有効なとき
            if (G3dHioLibProxy.Hio.IsAutoConnection)
            {
                // 接続処理を行う
                if (G3dHioLibProxy.Hio.IsPingReceived && (G3dHioLibProxy.Hio.IsConnected == false))
                {
                    var resetEvent = new ManualResetEventSlim();
                    {
                        TheApp.MainFrame.BeginInvoke(
                            new MethodInvoker(
                                () =>
                                {
                                    // モーダルダイアログが出ているときやドラッグ中は処理しない
                                    if (TheApp.MainFrame.CanFocus && !ControlUtility.IsAnyCapturing())
                                    {
                                        System.Collections.Generic.List<TexturePatternAnimation> list = DocumentManager.GetInvalidTexPatAnimation();

                                        if (list.Any())
                                        {
                                            string FileName = "";
                                            foreach (var anim in list)
                                            {
                                                FileName += string.Format("{0} ", anim.FileName);
                                            }

                                            App.Controls.UIMessageBox.Warning(App.res.Strings.TexturePatternAnimation_OutOfRange_AutoConnectOff, FileName);
                                            G3dHioLibProxy.Hio.IsAutoConnection = false;
                                        }
                                        else
                                        {
                                            TheApp.MainFrame.ConnectToHio(true, () => G3dHioLibProxy.Hio.IsAutoConnection = false);
                                        }
                                    }

                                    resetEvent.Set();
                                }
                            )
                        );
                    }
                    resetEvent.Wait();
                }

                // 切断処理を行う
                else
                if ((G3dHioLibProxy.Hio.IsPingReceived == false) && G3dHioLibProxy.Hio.IsConnected)
                {
                    var resetEvent = new ManualResetEventSlim();
                    {
                        TheApp.MainFrame.BeginInvoke(
                            new MethodInvoker(
                                () =>
                                {
                                    TheApp.MainFrame.ConnectToHio(false);
                                    resetEvent.Set();
                                }
                            )
                        );
                    }
                    resetEvent.Wait();
                }
            }
        }
    }
}
