﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    using System.Threading;

    using nw.g3d.iflib;

    public static class TemporaryFileUtility
    {
        private static TemporoaryFileManager TempFolder;
        private static TemporoaryFileManager ViewerFolder;
        private static TemporoaryFileManager ShaderFolder;

        private static string tempRoot;

        // 複数起動時にテンポラリフォルダ操作の競合を防ぐ
        private static Mutex mutex = new Mutex(false, "Nintendo_3dEditor_TemporaryFile_Mutex");

        internal static void Destroy()
        {
            try
            {
                mutex.WaitOne();
                TempFolder.Destroy(false);
                ViewerFolder.Destroy(false);
                ShaderFolder.Destroy(true);
                // Nintendo_3dEditor_Workフォルダが空のときは削除
                var entries = Directory.EnumerateFileSystemEntries(tempRoot);
                if (!entries.Any())
                {
                    try
                    {
                        Directory.Delete(tempRoot);
                    }
                    catch (Exception)
                    {
                    }
                }
            }
            finally
            {
                mutex.ReleaseMutex();
            }
        }

        internal static void Initialize()
        {
            try
            {
                mutex.WaitOne();
                tempRoot = Path.Combine(Path.GetTempPath(), "Nintendo_3dEditor_Work");
                Directory.CreateDirectory(tempRoot);
                TempFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"Temp"));
                ViewerFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"Viewer"));
                ShaderFolder = new TemporoaryFileManager(Path.Combine(tempRoot, @"Shader"));
                TempFolder.Initialize();
                ViewerFolder.Initialize();
                ShaderFolder.Initialize();
            }
            finally
            {
                mutex.ReleaseMutex();
            }
        }

        internal static string ShaderFolderPath
        {
            get
            {
                return ShaderFolder.FolderPath;
            }
        }

        internal static string MakeTemporaryFileName(string ext)
        {
            return TempFolder.MakeTemporaryFileName(ext);
        }

        internal static DisposableFileName MakeDisposableFileName(string ext)
        {
            return new DisposableFileName(MakeTemporaryFileName(ext));
        }

        internal static DisposableDirectoryName MakeDisposableDirectoryName()
        {
            return new DisposableDirectoryName(MakeTempDirectory());
        }

        internal static string MakeViewerTempFileName(string ext)
        {
            return ViewerFolder.MakeTemporaryFileName(ext);
        }

        internal static string MakeViewerTempDirectory()
        {
            return ViewerFolder.MakeTempDirectory();
        }

        internal static string MakeShaderTempFileName(string ext)
        {
            return ShaderFolder.MakeTemporaryFileName(ext);
        }

        internal static string MakeTempDirectory()
        {
            return TempFolder.MakeTempDirectory();
        }
    }

    public sealed class DisposableFileName : IDisposable
    {
        public DisposableFileName(string path)
        {
            Path = path;
        }

        public string Path { get; private set; }

        public void Dispose()
        {
            try
            {
                if (File.Exists(Path))
                {
                    File.Delete(Path);
                }
            }
            catch
            {
                Debug.Assert(false);
            }
        }
    }

    public sealed class DisposableDirectoryName : IDisposable
    {
        public DisposableDirectoryName(string path)
        {
            Path = path;
        }

        public string Path { get; private set; }

        public void Dispose()
        {
            try
            {
                if (Directory.Exists(Path))
                {
                    Directory.Delete(Path, true);
                }
            }
            catch
            {
                Debug.Assert(false);
            }
        }
    }
    public class TemporoaryFileManager
    {
        private static readonly object syncObject_ = new object();

        private readonly string tempRootDir_;
        private string tempDir_;
        private int    nameCounter_;

        private FileStream lockFile_ = null;

        private const string LockFileName = "lock";

        public TemporoaryFileManager(string rootPath)
        {
            tempRootDir_ = rootPath;
        }

        public void Initialize()
        {
            Destroy(false, false);
            using (var process = Process.GetCurrentProcess())
            {
                tempDir_ = Path.Combine(tempRootDir_, process.Id.ToString());
//			    try
                {
                    Directory.CreateDirectory(tempDir_);
                }
//			    catch (Exception)
                {
                }
                try
                {
                    lockFile_ = File.Open(Path.Combine(tempDir_, LockFileName), FileMode.OpenOrCreate, FileAccess.Read, FileShare.None);
                }
                catch
                {
                }
            }
        }

        /// <summary>
        /// 指定したディレクトリがlockファイルでロックされてなければ削除する。
        /// </summary>
        /// <param name="dir"></param>
        /// <param name="flag"></param>
        private static void DeleteUnlockedDirectory(string dir, bool flag)
        {
            var lockpath = Path.Combine(dir, LockFileName);
            if (File.Exists(lockpath))
            {
                try
                {
                    var templock = File.Open(lockpath, FileMode.Open, FileAccess.Read, FileShare.None);
                    templock.Close();
                }
                catch (Exception)
                {
                    return;
                }
            }
            Directory.Delete(dir, flag);
        }

        public void Destroy(bool deleteNonDigit, bool deleteSelf = true)
        {
            if (lockFile_ != null)
            {
                lockFile_.Close();
            }

            if (!Directory.Exists(this.tempRootDir_))
            {
                return;
            }

            foreach (var dir in Directory.EnumerateDirectories(this.tempRootDir_))
            {
                try
                {
                    if (deleteNonDigit)
                    {
                        DeleteUnlockedDirectory(dir, true);
                    }
                    else
                    {
                        string name = Path.GetFileName(dir);
                        if (name.All(x => char.IsDigit(x)))
                        {
                            DeleteUnlockedDirectory(dir, true);
                        }
                    }
                }
                catch
                {


                }
            }

            // フォルダが空のとき削除
            if (deleteSelf && !Directory.EnumerateFileSystemEntries(this.tempRootDir_).Any())
            {
                try
                {
                    Directory.Delete(this.tempRootDir_);
                }
                catch (Exception)
                {
                }
            }
        }

        public string FolderPath
        {
            get
            {
                return tempDir_;
            }
        }

        // テンポラリファイル名を作る
        public string MakeTemporaryFileName(string ext)
        {
            using(var process = Process.GetCurrentProcess())
            {
                lock (syncObject_)
                {
                    ++nameCounter_;

                    return
                        Path.Combine(
                            tempDir_,
                            nameCounter_.ToString() + (ext ?? string.Empty)
                        );
                }
            }
        }

        public string MakeTempDirectory()
        {
            using (var process = Process.GetCurrentProcess())
            {
                lock (syncObject_)
                {
                    ++nameCounter_;

                    var directory = Path.Combine(tempDir_, nameCounter_.ToString());
                    Directory.CreateDirectory(
                        directory);
                    return directory;
                }
            }
        }
    }

    public sealed class G3dStreamCache : IDisposable
    {
        public G3dStreamCache(List<G3dStream> g3dstreams)
        {
            // ストリームデータ列の準備
            streamCount = g3dstreams.Count;
            byte[][] streams = new byte[streamCount][];
            streamSizes = new int[streamCount];

            G3dParallel.For(0, streamCount, delegate(int i)
            {
                using (var mem = new MemoryStream())
                using (BinaryWriter wt = new BinaryWriter(mem))
                {
                    g3dstreams[i].Write(wt);
                    streams[i] = mem.ToArray();
                    streamSizes[i] = streams[i].Length;
                }
            });

            path = TemporaryFileUtility.MakeTemporaryFileName(null);
            fileStream = new FileStream(path, FileMode.Create, FileAccess.ReadWrite);

            for (int i = 0; i < streamCount; i++)
            {
                //writer.Write(streams[i], 0, streams[i].Length);
                //streamSizes[i] = streams[i].Length;
                fileStream.Write(streams[i], 0, streams[i].Length);
            }
            fileStream.Flush();
        }

        public List<G3dStream> GetStreams()
        {
            fileStream.Seek(0, SeekOrigin.Begin);
            byte[][] streams = new byte[streamCount][];
            for (int i = 0; i < streamCount; i++)
            {
                streams[i] = new byte[streamSizes[i]];
                fileStream.Read(streams[i], 0, streams[i].Length);
            }

            var g3dStreams = Enumerable.Range(0, streamCount).Select(x => new G3dStream()).ToList();
            G3dParallel.For(0, streamCount, delegate(int i)
            {
                using (BinaryReader streamReader = new BinaryReader(new MemoryStream(streams[i])))
                {
                    g3dStreams[i].Read(streamReader);
                }
            });

            return g3dStreams;
        }
        private readonly int[] streamSizes;
        private readonly int streamCount;
        private readonly FileStream fileStream;
        private readonly string path;

        public void Dispose()
        {
            fileStream.Close();
            if (File.Exists(path))
            {
                File.Delete(path);
            }
            else
            {
                Debug.Assert(false);
            }
        }
    }
}
