﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using App.Data;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using App.ConfigData;

namespace App.Utility
{
    public static class ShaderDifinitionUtility
    {
        public static readonly string TheIncludeFolder = "include";

        static public string RebuildShaderDefinition(ShaderDefinition shaderDef, out byte[] outputFsd, bool compile_test, TeamConfig.PlatformPreset platform)
        {
            DisposableDirectoryName disposableDirectory = null;
            try
            {
                string tempDirPath;
                if (compile_test)
                {
                    // compile_test のときのみフォルダを削除する
                    disposableDirectory = TemporaryFileUtility.MakeDisposableDirectoryName();
                    tempDirPath = disposableDirectory.Path;
                    OutputGLSL(shaderDef, tempDirPath);
                }
                else
                {
                    tempDirPath = GetShaderDefinitionTemporaryPath(shaderDef);
                }
                string fscaFilePath = tempDirPath + "\\" + shaderDef.Name + ".fsca";
                string fsdbFilePath = tempDirPath + "\\" + shaderDef.Name + ".fsdb";

                outputFsd = null;
                // ShaderDefinition から、ShaderConfigファイルを生成する。
                nw4f_3difType fscType = CreateShaderConfigFromShaderDifinition(shaderDef);
                if (fscType == null)
                {
                    return null;
                }

                // ファイルイメージに変換する。
                byte[] fileImage = CreateBinaryData(shaderDef.Name + ".fsca", fscType);



                // シェーダーコンバーターでfscからfsdファイルを生成する。
                if (!ShdrcvtrManager.ConvertFscToFsd(
                    fscData:fileImage,
                    fscFilePath:fscaFilePath,
                    fsdFilePath:fsdbFilePath,
                    log:true,
                    output:false,
                    outputFsd:out outputFsd,
                    compile_test:compile_test,
                    specificShadingModel:false,
                    platform:platform))
                {
                    // エラーの場合どうしましょ？
                    DebugConsole.WriteLine("ConvertFscToFsd failed: ShaderDefinition.name(" + shaderDef.Name + ")");
                    return null;
                }

                // fsd にコンバイナー用オプション定義を埋め込む。
                var hasCombiner = (shaderDef?.Data?.shading_model_array != null) && shaderDef.Data.shading_model_array.shading_model
                    .Where(x => x.option_var_array != null)
                    .SelectMany(x => x.option_var_array.option_var)
                    .Any(x => x.ui_item?.value == ui_item_valueType.combiner);
                if (hasCombiner)
                {
                    try
                    {
                        // 1. outputFsd をファイルに保存 (ShdrcvtrManager.ConvertFscToFsd() の output が false なので)
                        // 2. 保存したファイルにコンバイナー用オプション定義を埋め込む (定義埋め込みなので ecmb ファイルは指定しない)
                        // 3. 埋め込んだファイルイメージを取得する
                        File.WriteAllBytes(fsdbFilePath, outputFsd);
                        if (!CombinerShaderConverterManager.ConvertShader(fsdbFilePath, Enumerable.Empty<string>(), true))
                        {
                            throw new Exception();
                        }
                        outputFsd = File.ReadAllBytes(fsdbFilePath);
                    }
                    catch
                    {
                        // エラーの場合どうしましょ？
                        DebugConsole.WriteLine("ConvertShader failed: ShaderDefinition.name(" + shaderDef.Name + ")");
                        return null;
                    }
                }

                return fsdbFilePath;
            }
            finally
            {
                if (disposableDirectory != null)
                {
                    disposableDirectory.Dispose();
                }
            }
        }

        static public List<string> OutputGLSLFileFromFsd(ShaderDefinition shaderDef)
        {

            if (shaderDef.GlslFilePathList != null &&
                shaderDef.GlslFilePathList.Any())
            {
                return shaderDef.GlslFilePathList;
            }

            // テンポラリファイルディレクトリ生成
            string tempDirPath = GetShaderDefinitionTemporaryPath(shaderDef);

            List<string> glslFilePathList = OutputGLSL(shaderDef, tempDirPath);

            using (var stopWatch = new DebugStopWatch("SetSources"))
            {
                SetSources(shaderDef);
            }

            // シェーディングモデルごとの fsdb を生成
            shaderDef.GlslFilePathList = glslFilePathList;
            return glslFilePathList;
        }

        static public void RefreshGLSLFileFromFsd(ShaderDefinition shaderDef)
        {
            shaderDef.DisableSourceWatcher();
            string tempDirPath = GetShaderDefinitionTemporaryPath(shaderDef);
            shaderDef.GlslFilePathList = OutputGLSL(shaderDef, tempDirPath);
        }

        static private List<string> OutputGLSL(ShaderDefinition shaderDef, string tempDirPath)
        {
            List<string> glslFilePathList = new List<string>();
            Directory.CreateDirectory(tempDirPath);

            foreach (var shaderSrc in shaderDef.Data.shader_src_array.shader_src)
            {
                string glslFilePath = string.Format("{0}\\{1}\\{2}", tempDirPath, TheIncludeFolder, shaderSrc.path);

                // インクルードファイルのディレクトリを作成
                Directory.CreateDirectory(Path.GetDirectoryName(glslFilePath));

                glslFilePathList.Add(glslFilePath);
                try
                {
                    using (StreamWriter writer = new StreamWriter(glslFilePath, false, Encoding.GetEncoding(shaderDef.Data.shader_definition_info.code_page)))
                    //					using (StreamWriter writer = new StreamWriter(glslFilePath, false))
                    {
                        writer.Write(shaderDef.BinaryStreams[shaderSrc.stream_index].StringData);
                    }
                }
                catch (Exception)
                {
                    DebugConsole.WriteLine("OutputGLSLFileFromFsd glsl File output failed.");
                }
            }

            return glslFilePathList;
        }

        static private void SetSources(ShaderDefinition shaderDef)
        {
            foreach (var shadingModel in shaderDef.Data.shading_model_array.GetItems())
            {
                using (var stopWatch = new DebugStopWatch("SetSources " + shadingModel.name))
                {
                    var fsc = CreateShaderConfig(shadingModel, shaderDef.Data.shader_src_array, shaderDef.Data.shader_definition_info.code_page, shaderDef.Data.force_include_array);
                    byte[] fileImage = ShaderDifinitionUtility.CreateBinaryData(shadingModel.name + ".fsca", fsc);
                    string tempDirPath = ShaderDifinitionUtility.GetShaderDefinitionTemporaryPath(shaderDef);
                    string fscaFilePath = tempDirPath + "\\" + shadingModel.name + ".fsca";
                    string fsdbFilePath = tempDirPath + "\\" + shadingModel.name + ".fsdb";

                    byte[] outputFsd;
                    // シェーダーコンバーターでfscからfsdファイルを生成する。
                    // compile_test が false のときは platform は Nn Nw の区別だけ。Nn と Nw の切り替えはサポートできていないので、細かいことは気にしなくてよい
                    if (!ShdrcvtrManager.ConvertFscToFsd(
                        fscData:fileImage,
                        fscFilePath:fscaFilePath,
                        fsdFilePath:fsdbFilePath,
                        log:false,
                        output:false,
                        outputFsd:out outputFsd,
                        compile_test:false,
                        specificShadingModel:true,
                        platform:App.AppContext.SelectedPlatformPreset))
                    {
                        // エラーの場合どうしましょ？
                        DebugConsole.WriteLine("SetSource failed: shading_model.name(" + shadingModel.name + ")");
                        continue;
                    }

                    try
                    {
                        byte[] textImage;
                        byte[] binaryImage;
                        IfBinaryReadUtility.Separate(outputFsd, out textImage, out binaryImage);
                        bool updated;
                        nw4f_3difType nw4f_3dif = IfTextReadUtility.Read(textImage, null, out updated);

                        shaderDef.SourceFileNames.Add(shadingModel.name, ((shader_definitionType)nw4f_3dif.Item).shader_src_array.GetItems().Select(x => Path.GetFileName(x.path)).ToArray());
                    }
                    catch
                    {
                        // 何もしない
                        DebugConsole.WriteLine("SetSource failed: shading_model.name(" + shadingModel.name + ")");
                    }
                }
            }
        }

        // ShaderDefinitionファイルをテンポラリ保存する際のパス
        static public string GetShaderDefinitionTemporaryPath(ShaderDefinition shaderDef)
        {
            return Path.Combine(App.Utility.TemporaryFileUtility.ShaderFolderPath, shaderDef.Name);
        }

        static public void CreateGLSLFilePathList(ShaderDefinition shaderDef)
        {
            if (shaderDef.GlslFilePathList != null &&
                shaderDef.GlslFilePathList.Any())
            {
                return;
            }

            List<string> glslFilePathList = new List<string>();
            List<string> includePathList = new List<string>();

            // テンポラリファイルディレクトリ生成
            string tempDirPath = GetShaderDefinitionTemporaryPath(shaderDef);
            Directory.CreateDirectory(tempDirPath);

            foreach (var shaderSrc in shaderDef.Data.shader_src_array.shader_src)
            {
                string glslFilePath = string.Format("{0}\\{1}\\{2}", tempDirPath, TheIncludeFolder, shaderSrc.path);
                glslFilePathList.Add(glslFilePath);
            }

            shaderDef.GlslFilePathList = glslFilePathList;
        }

        static private nw4f_3difType CreateShaderConfigFromShaderDifinition(ShaderDefinition shaderDef)
        {
            if(shaderDef.Data == null ||
               shaderDef.Data.shading_model_array == null ||
               shaderDef.Data.shader_src_array == null)
            {
                return null;
            }

            nw4f_3difType nw4f_3dif_ = new nw4f_3difType();
            shader_configType shaderConfig = new shader_configType();

            // <shader_config_info>
            shader_config_infoType shaderConfigInfo = new shader_config_infoType();
            shaderConfigInfo.code_page = shaderDef.Data.shader_definition_info.code_page; //
//			shaderConfigInfo.code_page = 65001; //
            shaderConfig.shader_config_info = shaderConfigInfo;

            // <force_include_file>
            shaderConfig.force_include_file_array = (from force_include in shaderDef.Data.force_include_array.GetItems()
                                                     let src = shaderDef.Data.shader_src_array.shader_src[force_include.src_index]
                                                     select new force_include_fileType() { path = src.path })
                                                     .ToG3dArrayElement<force_include_fileType, force_include_file_arrayType>();

            // <include_path_array>
            shaderConfig.include_path_array = new include_path_arrayType()
            {
                include_path = new include_pathType[]
                {
                    new include_pathType()
                    {
                        path = TheIncludeFolder,
                    }
                }
            };

            // <shader_array>
            shader_arrayType shaderArray = new shader_arrayType();
            List<shaderType> shaderList = new List<shaderType>();

            foreach (var shadingModel in shaderDef.Data.shading_model_array.shading_model)
            {
                shaderType shader = new shaderType();
                shader.name = shadingModel.name;
                shader.material_shader = shadingModel.material_shader;
                shaderList.Add(shader);

                // <vertex_shader>
                if (shadingModel.vertex_stage == null)
                {
                    shader.vertex_shader = null;
                }
                else
                {
                    vertex_shaderType vertexShader = new vertex_shaderType();
                    vertexShader.path = shaderDef.Data.shader_src_array.shader_src[shadingModel.vertex_stage.src_index].path;
                    shader.vertex_shader = vertexShader;
                }

                // <geometry_shader>
                if (shadingModel.geometry_stage == null)
                {
                    shader.geometry_shader = null;
                }
                else
                {
                    geometry_shaderType geometryShader = new geometry_shaderType();
                    geometryShader.path = shaderDef.Data.shader_src_array.shader_src[shadingModel.geometry_stage.src_index].path;
                    shader.geometry_shader = geometryShader;
                }

                // <fragment_shader>
                if (shadingModel.fragment_stage == null)
                {
                    shader.fragment_shader = null;
                }
                else
                {
                    fragment_shaderType fragmentShader = new fragment_shaderType();
                    fragmentShader.path = shaderDef.Data.shader_src_array.shader_src[shadingModel.fragment_stage.src_index].path;
                    shader.fragment_shader = fragmentShader;
                }

                // <compute_shader>
                if (shadingModel.compute_stage == null)
                {
                    shader.compute_shader = null;
                }
                else
                {
                    compute_shaderType computeShader = new compute_shaderType();
                    computeShader.path = shaderDef.Data.shader_src_array.shader_src[shadingModel.compute_stage.src_index].path;
                    shader.compute_shader = computeShader;
                }

                // <macro_array>
                if (shadingModel.macro_array == null)
                {
                    shader.macro_array = null;
                }
                else
                {
                    macro_arrayType macroArray = new macro_arrayType();
                    List<macroType> macros = new List<macroType>();
                    foreach (var macrotype in shadingModel.macro_array.macro)
                    {
                        macroType macro = new macroType();
                        macro.name = macrotype.name;
                        macro.value = macrotype.value;
                        macros.Add(macro);
                    }
                    macroArray.length = macros.Count();
                    macroArray.macro = macros.ToArray();
                    shader.macro_array = macroArray;
                }

                // <variation_array>
                variation_arrayType varArray = new variation_arrayType();
                List<variationType> variations = new List<variationType>();
                if (shadingModel.option_var_array != null)
                {
                    foreach (var optionVar in shadingModel.option_var_array.option_var)
                    {
                        variationType variation = new variationType();
                        variation.id = optionVar.id;
                        variation.choice = optionVar.choice;
                        variation.@default = optionVar.@default;
                        variation.branch = optionVar.branch;
                        variations.Add(variation);
                    }
                    varArray.length = variations.Count();
                    varArray.variation = variations.ToArray();
                    shader.variation_array = varArray;
                }
                else
                {
                    shader.variation_array = null;
                }
            }

            shaderArray.shader = shaderList.ToArray();
            shaderConfig.shader_array = shaderArray;

            nw4f_3dif_.Item = shaderConfig;
            nw4f_3dif_.file_info = null;

            return nw4f_3dif_;
        }

        static public nw4f_3difType CreateShaderConfig(shading_modelType shadingModel,
            shader_src_arrayType shaderSrcArray,
            int codepage,
            force_include_arrayType forceIncludeArray)
        {
            return CreateShaderConfig(Enumerable.Repeat(shadingModel, 1), shaderSrcArray, codepage, forceIncludeArray);
        }

        static public nw4f_3difType CreateShaderConfig(IEnumerable<shading_modelType> shadingModels,
            shader_src_arrayType shaderSrcArray,
            int codepage,
            force_include_arrayType forceIncludeArray)
        {
            nw4f_3difType nw4f_3dif_ = new nw4f_3difType();

            shader_configType shaderConfig = new shader_configType();

            // <shader_config_info>
            shader_config_infoType shaderConfigInfo = new shader_config_infoType();
            shaderConfigInfo.code_page = codepage; //
            shaderConfig.shader_config_info = shaderConfigInfo;

            // <force_include_file>
            shaderConfig.force_include_file_array = (from force_include in forceIncludeArray.GetItems()
                                                     let src = shaderSrcArray.shader_src[force_include.src_index]
                                                     select new force_include_fileType() { path = src.path })
                                                     .ToG3dArrayElement<force_include_fileType, force_include_file_arrayType>();

            // <include_path_array>
            shaderConfig.include_path_array = new include_path_arrayType()
            {
                include_path = new include_pathType[]
                {
                    new include_pathType()
                    {
                        path = TheIncludeFolder,
                    }
                }
            };

            // <shader_array>
            shader_arrayType shaderArray = new shader_arrayType();
            List<shaderType> shaderList = new List<shaderType>();
            foreach (var shadingModel in shadingModels)
            {
                var shader = new shaderType();
                shader.name = shadingModel.name;
                shader.material_shader = shadingModel.material_shader;
                shaderList.Add(shader);

                // <vertex_shader>
                if (shadingModel.vertex_stage == null)
                {
                    shader.vertex_shader = null;
                }
                else
                {
                    shader.vertex_shader = new vertex_shaderType
                    {
                        path = shaderSrcArray.shader_src[shadingModel.vertex_stage.src_index].path
                    };
                }

                // <geometry_shader>
                if (shadingModel.geometry_stage == null)
                {
                    shader.geometry_shader = null;
                }
                else
                {
                    shader.geometry_shader = new geometry_shaderType
                    {
                        path = shaderSrcArray.shader_src[shadingModel.geometry_stage.src_index].path
                    };
                }

                // <fragment_shader>
                if (shadingModel.fragment_stage == null)
                {
                    shader.fragment_shader = null;
                }
                else
                {
                    shader.fragment_shader = new fragment_shaderType
                    {
                        path = shaderSrcArray.shader_src[shadingModel.fragment_stage.src_index].path
                    };
                }

                // <compute_shader>
                if (shadingModel.compute_stage == null)
                {
                    shader.compute_shader = null;
                }
                else
                {
                    shader.compute_shader = new compute_shaderType
                    {
                        path = shaderSrcArray.shader_src[shadingModel.compute_stage.src_index].path
                    };
                }


                // <macro_array>
                if (shadingModel.macro_array == null)
                {
                    shader.macro_array = null;
                }
                else
                {
                    macro_arrayType macroArray = new macro_arrayType();
                    List<macroType> macros = new List<macroType>();
                    foreach (var macrotype in shadingModel.macro_array.macro)
                    {
                        macroType macro = new macroType();
                        macro.name = macrotype.name;
                        macro.value = macrotype.value;
                        macros.Add(macro);
                    }
                    macroArray.length = macros.Count();
                    macroArray.macro = macros.ToArray();
                    shader.macro_array = macroArray;
                }

                // <variation_array>
                variation_arrayType varArray = new variation_arrayType();
                List<variationType> variations = new List<variationType>();
                if (shadingModel.option_var_array != null)
                {
                    foreach (var optionVar in shadingModel.option_var_array.option_var)
                    {
                        variationType variation = new variationType();
                        variation.id = optionVar.id;
                        variation.choice = optionVar.choice;
                        variation.@default = optionVar.@default;
                        variation.branch = optionVar.branch;
                        variations.Add(variation);
                    }
                    varArray.length = variations.Count();
                    varArray.variation = variations.ToArray();
                    shader.variation_array = varArray;
                }
                else
                {
                    shader.variation_array = null;
                }
            }

            shaderArray.shader = shaderList.ToArray();
            shaderConfig.shader_array = shaderArray;


            nw4f_3dif_.Item = shaderConfig;
            nw4f_3dif_.file_info = null;

            return nw4f_3dif_;
        }

        /// <summary>
        /// ドキュメント(モデル、テクスチャ、アニメーション等)からnw4f_3difとBinaryStreamをバイナリデータとして出力する。
        /// </summary>
        public static byte[] CreateBinaryData(string fileName, nw4f_3difType fscType)
        {
            byte[] fileImage;

            using (var watch = new DebugStopWatch("Create_nw4f_3difType " + fileName))
            {
                List<G3dStream> streams = new List<G3dStream>();
                // バイナリデータに変換する。
                lock (Viewer.Connecter.AbortLock)
                {
                    fileImage = IfBinaryFormatter.FormatStream(fscType, streams, null);
                }
            }

            return fileImage;
        }
    }
}
