﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using App.Command;
using App.Data;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    /// <summary>
    /// マテリアルのシェーダー割り当ての編集用のクラス
    /// </summary>
    public static class ShaderAssignUtility
    {
        public enum ErrorType
        {
            None,
            Unimportant,
            Critical,
            NotMaterialShader,
        };

        public static bool IsMaterialShader(Material material)
        {
            var definition = material.MaterialShaderAssign.ShadingModel;
            return definition == null || definition.IsMaterialShader();
        }
        public static bool IsConsistentWithDefinition(Material material)
        {
            var definition = material.MaterialShaderAssign.ShadingModel;
            var error = ShaderAssignUtility.IsConsistentWithDefinition(material, definition, false);
            return error== ErrorType.None || error == ErrorType.NotMaterialShader;
        }

        public static ErrorType IsConsistentWithDefinitionDetail(Material material)
        {
            var definition = material.MaterialShaderAssign.ShadingModel;
            return ShaderAssignUtility.IsConsistentWithDefinition(material, definition, true);
        }

        /// <summary>
        /// シェーディングモデルに対応しているか?
        /// </summary>
        public static ErrorType IsConsistentWithDefinition(Material material, shading_modelType definition, bool detailCheck)
        {
            if (definition == null)
            {
                return ErrorType.None;
            }

            if (!definition.IsMaterialShader())
            {
                // マテリアル用以外に対してはチェックや修正を行わない
                return ErrorType.NotMaterialShader;
            }

            bool ok = true;

            // リビジョンの比較
            if (IfShaderAssignUtility.CompareRevision(material.MaterialShaderAssign.Revision, definition) != IfShaderAssignUtility.CompareResult.Ok)
            {
                return ErrorType.Critical;
            }

            // オプションの比較
            switch (IfShaderOptionUtility.Compare(material.MaterialShaderAssign.ShaderOptions, definition))
            {
                case IfShaderAssignUtility.CompareResult.NotSameOrder:
                    ok = false;
                    break;
                case IfShaderAssignUtility.CompareResult.Conflict:
                    return ErrorType.Critical;
            }

            // サンプラ割り当ての比較
            switch (IfSamplerAssignUtility.Compare(material.MaterialShaderAssign.SamplerAssigns, definition, material.sampler_array))
            {
                case IfShaderAssignUtility.CompareResult.NotSameOrder:
                case IfShaderAssignUtility.CompareResult.NotFullyAssigned:
                    ok = false;
                    break;
                case IfShaderAssignUtility.CompareResult.Conflict:
                    return ErrorType.Critical;
            }

            // ユニフォームの比較
            switch (IfShaderParameterUtility.Compare(material.MaterialShaderAssign.ShaderParams, definition))
            {
                case IfShaderAssignUtility.CompareResult.NotSameOrder:
                    ok = false;
                    break;
                case IfShaderAssignUtility.CompareResult.Conflict:
                    return ErrorType.Critical;
            }

            foreach (var model in material.Referrers)
            {
                // 頂点属性の比較
                switch (IfAttributeAssignUtility.Compare(
                    material.MaterialShaderAssign.AttribAssigns,
                    model.Data,
                    material.Name,
                    definition))
                {
                    case IfShaderAssignUtility.CompareResult.NotSameOrder:
                    case IfShaderAssignUtility.CompareResult.NotFullyAssigned:
                        ok = false;
                        break;
                    case IfShaderAssignUtility.CompareResult.Conflict:
                        return ErrorType.Critical;
                }

                // 頂点バッファの比較
                switch (IfVertexBufferUtility.Compare(model.Data, material.Name, material.MaterialShaderAssign.AttribAssigns, definition))
                {
                    case IfShaderAssignUtility.CompareResult.Conflict:
                        return ErrorType.Critical;
                }
            }

            // 描画情報の比較
            IEnumerable<render_info_slotType> render_infos = definition.RenderInfoSlots();
            switch (IfRenderInfoUtility.Compare(material.MaterialShaderAssign.To_render_info_array(material, true).GetItems().ToArray(), definition))
            {
                case IfShaderAssignUtility.CompareResult.NotSameOrder:
                    ok = false;
                    break;
                case IfShaderAssignUtility.CompareResult.Conflict:
                    return ErrorType.Critical;
            }

            var RenderInfos = material.MaterialShaderAssign.RenderInfos;
            if (render_infos.Count() != RenderInfos.Count)
            {
                return ErrorType.Critical;
            }

            if (RenderInfos.Zip(render_infos, (x, y) => x.name == y.name && x.type == y.Type() && x.values.Count <= y.count).Any(x => !x))
            {
                if (detailCheck &&
                    RenderInfos.OrderBy(x => x.name).Zip(render_infos.OrderBy(x => x.name), (x, y) => x.name == y.name && x.type == y.Type() && x.values.Count(z => !string.IsNullOrWhiteSpace(z)) <= y.count).All(x => x))
                {
                    ok = false;
                }
                else
                {
                    return ErrorType.Critical;
                }
            }

            // マテリアル参照情報の比較
            switch (IfApplyBaseMaterialUtility.Compare(material.MaterialReference, definition))
            {
                case IfShaderAssignUtility.CompareResult.NotSameOrder:
                    ok = false;
                    break;
                case IfShaderAssignUtility.CompareResult.Conflict:
                    return ErrorType.Critical;
            }

            return ok ? ErrorType.None : ErrorType.Unimportant;
        }

        public static string InconsistentMessage(Model model, Material material, out int count, IfShaderAssignUtility.InconsistentMessageType type)
        {
            var builder = new StringBuilder();
            var shading_model = material.MaterialShaderAssign.ShadingModel;
            material.Data.shader_assign = material.MaterialShaderAssign.To_shader_assign(material, true);
            material.Data.sampler_array = material.sampler_array.sampler.ToG3dArrayElement<samplerType, sampler_arrayType>();
            var messages = IfShaderAssignUtility.UpdateMessages(model.Data, material.Data, shading_model, true, type);
            count = messages.Count;
            foreach (var message in messages)
            {
                builder.AppendLine(message);
            }

            return builder.ToString();
        }

        /// <summary>
        /// エラー情報
        /// </summary>
        public class MaterialAnimationError
        {
            public IHasShaderParameterAnimation anim;
            public IShaderParamMatAnim matAnim;
            public List<AnimationError> errors;
            public ShaderDefinition shaderDefinition;
            public shading_modelType shading_model;
            public bool canFix;
        }

        /// <summary>
        /// エラー情報
        /// </summary>
        public class AnimationError
        {
            public ShaderParameterAnimation.ParamAnim paramAnim;
            public uniform_varType uniform;
            public int[] table;
            public string message;
        }

        /// <summary>
        /// アニメーションがシェーディングモデルと一致するか
        /// </summary>
        public static IEnumerable<MaterialAnimationError> CheckAnimation(IHasShaderParameterAnimation anim)
        {
            var triples = (from model in DocumentManager.Models
                             where model.AllAnimations.Contains((AnimationDocument)anim)
                             from material in model.Materials
                             let shadingModel = material.MaterialShaderAssign.ShadingModel
                             where shadingModel != null
                             // マテリアル用でないシェーディングモデルが割り当てられている場合は無視
                             where shadingModel.IsMaterialShader()
                             let shaderDefinition = material.MaterialShaderAssign.ShaderDefinition
                             select new { material, shaderDefinition, shadingModel }).ToArray();

            foreach (var matAnim in anim.IShaderParamMatAnims)
            {
                var triple = triples.FirstOrDefault(x => x.material.Name == matAnim.mat_name);
                if (triple != null)
                {
                    List<AnimationError> errors = new List<AnimationError>();
                    foreach (var paramAnim in matAnim.ParamAnims)
                    {
                        // チェック用に param_animType を構築
                        var paramanim = new param_animType();
                        var targetTypeList = new List<param_anim_targetType>();

                        foreach (var target in paramAnim.ParamAnimTargets)
                        {
                            // キーフレームが無いものは削除するように修正します。
                            if (target.KeyFrames.Any() == false)
                            {
                                continue;
                            }

                            G3dStream stream;
                            var targetType = target.ConvertTo_param_anim_targetType(out stream);
                            targetTypeList.Add(targetType);
                        }
                        // TextureSRTの場合、Modeのみのカーブの場合は、保存しない。
                        if ((paramAnim.IsTexSRT))
                        {
                            if (targetTypeList.Count == 1)
                            {
                                var animTarget = targetTypeList[0];
                                if (animTarget.component_index == 0)
                                {
                                    continue;
                                }
                            }
                        }

                        if (targetTypeList.Any())
                        {
                            paramanim.id = paramAnim.id;
                            paramanim.type = paramAnim.type;
                            paramanim.param_anim_target = targetTypeList.ToArray();
                            uniform_varType uniform;
                            int[] table;
                            string message;
                            if (!IfAnimationAssignUtility.IsConsistentWithShadingModel(paramanim, triple.shadingModel, out uniform, out table, out message, IfShaderAssignUtility.InconsistentMessageType.Plan))
                            {
                                errors.Add(new AnimationError()
                                {
                                    paramAnim = paramAnim,
                                    uniform = uniform,
                                    table = table,
                                    message = message,
                                });
                            }
                        }
                    }

                    if (errors.Any())
                    {
                        yield return new MaterialAnimationError()
                        {
                            anim = anim,
                            matAnim = matAnim,
                            errors = errors,
                            shaderDefinition = triple.shaderDefinition,
                            shading_model = triple.shadingModel,
                            canFix = errors.Any(x => x.table != null),
                        };
                    }
                }
            }
        }

        #region コマンド
        public static EditCommandSet ExecuteFixParameters(Material[] materials, bool updateModified = true, bool useHint = false, bool reload = true)
        {
            Debug.Assert(materials.All(x => !IsConsistentWithDefinition(x)));
            DebugConsole.WriteLine("ExecuteFixParameters");

            EditCommandSet commandSet2 = new EditCommandSet();
            EditCommandSet commandSet = new EditCommandSet();
            Viewer.ViewerDrawSuppressBlock.DisposedDelegate dispose;
            if (!reload)
            {
                dispose = Viewer.ViewerDrawSuppressBlock.DiscardMessages;
            }
            else
            {
                dispose = Viewer.ViewerDrawSuppressBlock.DiscardAllMessages;
            }

            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                var models = materials.SelectMany(x => x.Referrers).Distinct().ToArray();
                using (var viewerSuppressBlock = new Viewer.ViewerDrawSuppressBlock(dispose))
                {
                    var targets = new GuiObjectGroup();
                    var revisions = new List<int>();
                    var options = new List<List<shader_optionType>>();
                    var samplers = new List<List<sampler_assignType>>();
                    var parameters = new List<List<shader_paramType>>();
                    var attribs = new List<List<attrib_assignType>>();
                    var renderInfos = new List<List<RenderInfo>>();
                    var vertexBuffers = new List<List<VtxBufferArrayEditData>>();
                    var combinerOptionInfos = new List<nw3de_CombinerOptionInfo>();
                    var materialReferences = new List<nw3de_MaterialReference>();

                    Action<modelType, Material> update = (model, material) =>
                    {
                        if (!targets.Contains(material))
                        {
                            targets.Add(material);
                        }
                        material.Data.shader_assign = material.MaterialShaderAssign.To_shader_assign(material, true);
                        material.Data.sampler_array = material.sampler_array.sampler.Any() ? ObjectUtility.Clone(material.sampler_array) : null;

                        Tuple<int, vtx_buffer_arrayType>[] oldBuffers;
                        if (model != null)
                        {
                            oldBuffers = (from shape in model.shape_array.GetItems().Where(x => x.shape_info.mat_name == material.Name)
                                          let index = shape.shape_info.vertex_index
                                          select new Tuple<int, vtx_buffer_arrayType>(index, model.vertex_array.vertex[index].vtx_buffer_array)).ToArray();
                        }
                        else
                        {
                            oldBuffers = new Tuple<int, vtx_buffer_arrayType>[] { };
                        }

                        // 書き換えられるかもしれないのでクローン
                        foreach (var item in oldBuffers)
                        {
                            model.vertex_array.vertex[item.Item1].vtx_buffer_array = ObjectUtility.Clone(item.Item2);
                        }

                        // model が null でも IfShaderAssignUtility.UpdateShaderAssign() での更新が必要。
                        IfShaderAssignUtility.UpdateShaderAssign(
                            model,
                            material.Data,
                            material.MaterialShaderAssign.ShaderDefinitionFileName,
                            material.MaterialShaderAssign.ShadingModel,
                            useHint,
                            allowEmptyString: true
                        );

                        // 不要なマテリアル参照情報の削除
                        var materialReference = ObjectUtility.Clone(material.MaterialReference);
                        IfApplyBaseMaterialUtility.UpdateMaterialReference(materialReference, material.MaterialShaderAssign.ShadingModel);

                        var shader_assign = material.Data.shader_assign;
                        if (shader_assign != null)
                        {
                            revisions.Add(shader_assign.revision);
                            options.Add(shader_assign.ShaderOptions().ToList());
                            samplers.Add(shader_assign.SamplerAssigns().ToList());
                            parameters.Add(shader_assign.ShaderParams().ToList());
                            attribs.Add(shader_assign.AttribAssigns().ToList());
                            renderInfos.Add(ConvertToRenderInfo(shader_assign.RenderInfos(), material.MaterialShaderAssign.RenderInfos).ToList());
                        }
                        else
                        {
                            revisions.Add(0);
                            options.Add(new List<shader_optionType>());
                            samplers.Add(new List<sampler_assignType>());
                            parameters.Add(new List<shader_paramType>());
                            attribs.Add(new List<attrib_assignType>());
                            renderInfos.Add(new List<RenderInfo>());
                        }

                        combinerOptionInfos.Add(ObjectUtility.Clone(material.CombinerOptionInfo));

                        materialReferences.Add(materialReference);

                        if (model != null)
                        {
                            vertexBuffers.Add(oldBuffers.Select(x =>
                                new VtxBufferArrayEditData()
                                {
                                    vertex = model.vertex_array.vertex[x.Item1],
                                    vtx_buffer_array = model.vertex_array.vertex[x.Item1].vtx_buffer_array
                                }).ToList());
                        }
                        else
                        {
                            vertexBuffers.Add(oldBuffers.Select(x => new VtxBufferArrayEditData()).ToList());
                        }

                        // コマンド実行前に一旦元に戻す
                        foreach (var item in oldBuffers)
                        {
                            model.vertex_array.vertex[item.Item1].vtx_buffer_array = item.Item2;
                        }
                    };

                    foreach (var material in materials)
                    {
                        if (material.Referrers.Any())
                        {
                            // fmd のマテリアルに含まれるモデルや fmt を参照しているモデルを更新。
                            foreach (var model in material.Referrers.Select(x => x.Data))
                            {
                                update(model, material);
                            }
                        }
                        else if (material.OwnerDocument is SeparateMaterial)
                        {
                            // どこからも参照されていない fmt のマテリアルを更新。
                            update(null, material);
                        }
                    }

                    commandSet.Add(ShaderAssignUtility.CreateRevisionEditCommand(targets, revisions, updateModified).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateShaderOptionsEditCommand(targets, options, updateModified).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateSamplerAssignsEditCommand(targets, samplers, updateModified, false).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateShaderParamsEditCommand(targets, parameters, updateModified).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateAttribAssignsEditCommand(targets, attribs, updateModified, false).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateRenderInfoEditCommand(targets, renderInfos, updateModified, false).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateVtxBufferEditCommand(targets, vertexBuffers, updateModified).Execute());
                    commandSet.Add(Material.CreateEditCommand_CombinerOption(targets, combinerOptionInfos, updateModified, false).Execute());
                    commandSet.Add(Material.CreateEditCommand_MaterialReference(targets, materialReferences).Execute());

                    // コマンドセットにモデル再転送のデリゲートを設定する。
                    commandSet.SetViewerDrawSuppressBlockDelegate(dispose);
                }
                commandSet.Reverse();
                commandSet2.Add(commandSet);
                if (reload)
                {
                    EventHandler onPostEdit = (s, e) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }
                    };
                    onPostEdit(null, null);
                    commandSet2.OnPostEdit += onPostEdit;
                }
                return commandSet2;
            }
        }

        /// <summary>
        /// アニメーション修正コマンドの作成
        /// ランタイムへの通知は行わない
        /// </summary>
        public static EditCommandSet CreateFixShaderParamAnim(IEnumerable<MaterialAnimationError> errors)
        {
            var documents = errors.Select(x => x.anim).Distinct().ToArray();
            var commandSet = new EditCommandSet();
            foreach (var error in errors)
            {
                commandSet.Add(CreateFixShaderParamAnim(error.anim, error));
            }

            commandSet.OnPostEdit += (s, e) =>
                {
                    foreach (var document in documents)
                    {
                        document.UpdateIsModifiedAnimTargetAll();
                    }
                };
            return commandSet;
        }

        private class ParamAnimSwapData
        {
            public ShaderParameterAnimation.ParamAnim paramAnim;
            public shader_param_typeType type;
            public List<ShaderParameterAnimation.ParamAnimTarget> ParamAnimTargets;
        }

        private static EditCommand CreateFixShaderParamAnim(IHasShaderParameterAnimation animation, MaterialAnimationError errors)
        {
            var swapData = new List<ParamAnimSwapData>();
            foreach (var error in errors.errors.Where(x => x.table != null))
            {
                var TempParamAnim = new ShaderParameterAnimation.ParamAnim(error.paramAnim.id, error.uniform.type);
                var ParamAnimTargets = TempParamAnim.ParamAnimTargets;
                for (int i = 0; i < error.table.Length; i++)
                {
                    var index = error.table[i];
                    if (index != -1)
                    {
                        var target = error.paramAnim.ParamAnimTargets.FirstOrDefault(x => x.component_index == index);
                        if (target != null)
                        {
                            ParamAnimTargets[i] = ObjectUtility.Clone(target);
                            ParamAnimTargets[i].component_index = i;
                        }
                    }
                }
                swapData.Add(new ParamAnimSwapData()
                {
                    paramAnim = error.paramAnim,
                    type = error.uniform.type,
                    ParamAnimTargets = ParamAnimTargets,
                });
            }

            return new GeneralGroupReferenceEditCommand<object>(
                new GuiObjectGroup((Document)animation),
                ((Document)animation).ObjectID,
                Enumerable.Repeat(new object(), 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    foreach (var item in swapData)
                    {
                        var type = item.paramAnim.type;
                        item.paramAnim.type = item.type;
                        item.type = type;
                        var targets = item.paramAnim.ParamAnimTargets;
                        item.paramAnim.ParamAnimTargets = item.ParamAnimTargets;
                        item.ParamAnimTargets = targets;
                    }
                });
        }

        /// <summary>
        /// 中間ファイルに出力されなかった頂点属性割り当てやサンプラ割り当てを未指定の要素として挿入する
        /// </summary>
        public static EditCommandSet AddNotAssignedItems(Material[] materials)
        {
            DebugConsole.WriteLine("AddNotAssignedItems");

            EditCommandSet commandSet = new EditCommandSet();
            EditCommandSet commandSet2 = new EditCommandSet();

            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                // 中間ファイルにすると違いがないので送信しない。
                using (var viewerSuppressBlock = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardMessages))
                {
                    var targets = new GuiObjectGroup();
                    var samplers = new List<List<sampler_assignType>>();
                    var attribs = new List<List<attrib_assignType>>();
                    foreach (var material in materials)
                    {
                        targets.Add(material);
                        var createdAttribs = material.Referrers.Any() ? IfAttributeAssignUtility.CreateAttribAssigns(
                            material.MaterialShaderAssign.ShadingModel,
                            material.Referrers.First().Data,
                            material.Name,
                            material.MaterialShaderAssign.AttribAssigns,
                            false,
                            true).ToList() : new List<attrib_assignType>();

                        var newAttribs = new List<attrib_assignType>();

                        // 新しいのとマージ
                        foreach (var created in createdAttribs)
                        {
                            var old = material.MaterialShaderAssign.AttribAssigns.FirstOrDefault(x => x.id == created.id);
                            if (old != null)
                            {
                                newAttribs.Add(old);
                            }
                            else
                            {
                                newAttribs.Add(created);
                            }
                        }

                        // 残りの古い部分を追加
                        foreach (var old in material.MaterialShaderAssign.AttribAssigns)
                        {
                            if (!createdAttribs.Any(x => x.id == old.id))
                            {
                                newAttribs.Add(old);
                            }
                        }
                        attribs.Add(newAttribs);

                        var createdSamplers = IfSamplerAssignUtility.CreateSamplerAssigns(
                            material.MaterialShaderAssign.ShadingModel,
                            material.sampler_array.GetItems(),
                            material.MaterialShaderAssign.SamplerAssigns,
                            false,
                            true).ToList();

                        var newSamplers = new List<sampler_assignType>();

                        // 新しいのとマージ
                        foreach (var created in createdSamplers)
                        {
                            var old = material.MaterialShaderAssign.SamplerAssigns.FirstOrDefault(x => x.id == created.id);
                            if (old != null)
                            {
                                newSamplers.Add(old);
                            }
                            else
                            {
                                newSamplers.Add(created);
                            }
                        }

                        // 残りの古い部分を追加
                        foreach (var old in material.MaterialShaderAssign.SamplerAssigns)
                        {
                            if (!createdSamplers.Any(x => x.id == old.id))
                            {
                                newSamplers.Add(old);
                            }
                        }

                        samplers.Add(newSamplers);
                    }

                    commandSet.Add(ShaderAssignUtility.CreateSamplerAssignsEditCommand(targets, samplers, false).Execute());
                    commandSet.Add(ShaderAssignUtility.CreateAttribAssignsEditCommand(targets, attribs, false).Execute());

                    // コマンドセットにモデル再転送のデリゲートを設定する。
                    commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardMessages);
                }
                commandSet.Reverse();
                commandSet2.Add(commandSet);

                return commandSet2;
            }
        }

        public static IEnumerable<RenderInfo> ConvertToRenderInfo(IEnumerable<render_infoType> source, IEnumerable<RenderInfo> oldRenderInfo)
        {
            foreach (var item in source)
            {
                var old = oldRenderInfo.FirstOrDefault(x => x.name == item.name && x.type == item.type);
                var values = G3dDataParser.Tokenize(item.Value ?? "").ToList();
                yield return new RenderInfo()
                {
                    name = item.name,
                    type = item.type,
                    internalValues = values,
                    recieved = old != null ? old.recieved : false,
                    HioCount = old != null ? old.HioCount : -1,
                    IsDefaultEmpty = old == null,
                };
            }
        }

        public static GroupEditCommand CreateEditCommand_ShaderAssign(GuiObjectGroup targets, IEnumerable<MaterialShaderAssign> ShaderAssigns, bool sendViewer)
        {
            Debug.Assert(targets.GetObjects(GuiObjectID.Material).Count == ShaderAssigns.Count());
            return new GeneralGroupReferenceEditCommand<MaterialShaderAssign>(
                targets,
                GuiObjectID.Material,
                ShaderAssigns,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    Debug.Assert(target is Material);
                    var material = target as Material;
                    swap = material.MaterialShaderAssign;
                    material.MaterialShaderAssign = (MaterialShaderAssign)data;
                    //material.RenderInfoQueried = false;
                },
                createEventArgsDelegate: (x, y) => new App.PropertyEdit.DocumentPropertyChangedShaderArgs(x, y),
                postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (sendViewer && editTargets.Count > 0)
                    {
                        var models = editTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                        if (models.Any())
                        {
                            foreach (var model in models)
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                            }
                        }
                    }
                }
            );
        }

        public static EditCommand CreateRevisionEditCommand(
            GuiObjectGroup targets,
            List<int> revisions,
            bool updateModified)
        {
            return new GeneralGroupReferenceEditCommand<object>(
                targets,
                GuiObjectID.Material,
                revisions.Select(x => (object)x),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    swap = material.MaterialShaderAssign.Revision;
                    material.MaterialShaderAssign.Revision = (int)data;
                },
                updateModified : updateModified
                );
        }

        public static EditCommand CreateShaderOptionsEditCommand(
            GuiObjectGroup targets,
            IEnumerable<List<shader_optionType>> shaderOptions,
            bool updateModified = true)
        {
            return new GeneralGroupReferenceEditCommand<List<shader_optionType>>(
                    targets,
                    GuiObjectID.Material,
                    shaderOptions,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (Material)target;
                        swap = material.MaterialShaderAssign.ShaderOptions;
                        material.MaterialShaderAssign.ShaderOptions = (List<shader_optionType>)data;
                    },
                    createEventArgsDelegate: (x, y) => new App.PropertyEdit.DocumentPropertyChangedShaderArgs(x, y),
                    postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                    {
                        // CreateShaderOptionsEditCommand, CreateSamplerAssignsEditCommand, CreateShaderParamsEditCommand
                        // は、今のところ3つセットで呼ばれているので、CreateSamplerAssignsEditCommandの中で、
                        // モデルの再転送を行っています。
                    },
                    updateModified : updateModified
                    );
        }

        public static EditCommand CreateSamplerAssignsEditCommand(
            GuiObjectGroup targets,
            IEnumerable<List<sampler_assignType>> sampler_assigns,
            bool updateModified = true,
            bool reload = true
            )
        {
            return new GeneralGroupReferenceEditCommand<List<sampler_assignType>>(
                targets,
                GuiObjectID.Material,
                sampler_assigns,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    swap = material.MaterialShaderAssign.SamplerAssigns;
                    material.MaterialShaderAssign.SamplerAssigns = (List<sampler_assignType>)data;
                },
                createEventArgsDelegate: (x, y) => new App.PropertyEdit.DocumentPropertyChangedShaderArgs(x, y),
                postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (editTargets.Count > 0 && reload)
                    {
                        var models = editTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                        if (models.Any())
                        {
                            foreach (var model in models)
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                            }
                        }
                    }
                },
                updateModified: updateModified
            );
        }

        public static EditCommand CreateShaderParamsEditCommand(
            GuiObjectGroup targets,
            IEnumerable<List<shader_paramType>> shader_params,
            bool updateModified = true)
        {
            return new GeneralGroupReferenceEditCommand<List<shader_paramType>>(
                    targets,
                    GuiObjectID.Material,
                    shader_params,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (Material)target;
                        swap = material.MaterialShaderAssign.ShaderParams;
                        material.MaterialShaderAssign.ShaderParams = (List<shader_paramType>)data;
                    },
                    createEventArgsDelegate: (x, y) => new App.PropertyEdit.DocumentPropertyChangedShaderArgs(x, y),
                    postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                    {
                        // CreateShaderOptionsEditCommand, CreateSamplerAssignsEditCommand, CreateShaderParamsEditCommand
                        // は、今のところ3つセットで呼ばれているので、CreateSamplerAssignsEditCommandの中で、
                        // モデルの再転送を行っています。
                    },
                    updateModified: updateModified
                    );
        }

        public static EditCommand CreateAttribAssignsEditCommand(
            GuiObjectGroup targets,
            IEnumerable<List<attrib_assignType>> attrib_assigns,
            bool updateModified = true,
            bool reload = true)
        {
            return new GeneralGroupReferenceEditCommand<List<attrib_assignType>>(
                targets,
                GuiObjectID.Material,
                attrib_assigns,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    swap = material.MaterialShaderAssign.AttribAssigns;
                    material.MaterialShaderAssign.AttribAssigns = (List<attrib_assignType>)data;
                },
                createEventArgsDelegate: (x, y) => new App.PropertyEdit.DocumentPropertyChangedShaderArgs(x, y),
                postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (editTargets.Count > 0 && reload)
                    {
                        var models = editTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                        if (models.Any())
                        {
                            foreach (var model in models)
                            {
                                // ビューアへModelを送信
                                //Viewer.ViewerUtility.SendModel(model, true);
                            }
                        }
                    }
                },
                updateModified: updateModified
                );
        }

        public static EditCommand CreateRenderInfoEditCommand(
            GuiObjectGroup targets,
            IEnumerable<List<RenderInfo>> render_infos,
            bool updateModified = true,
            bool reload = true)
        {
            return new GeneralGroupReferenceEditCommand<List<RenderInfo>>(
                targets,
                GuiObjectID.Material,
                render_infos,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    swap = material.MaterialShaderAssign.RenderInfos;
                    material.MaterialShaderAssign.RenderInfos = (List<RenderInfo>)data;
                    //material.RenderInfoQueried = false;
                },
                postEditDelegate: delegate(System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (editTargets.Count > 0 && reload)
                    {
                        var models = editTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct().ToArray();
                        if (models.Any())
                        {
                            foreach (var model in models)
                            {
                                // ビューアへModelを送信
                                //Viewer.ViewerUtility.SendModel(model, true);
                            }
                        }
                    }
                },
                updateModified: updateModified
            );
        }

        private class VtxBufferArrayEditData
        {
            public vertexType vertex;
            public vtx_buffer_arrayType vtx_buffer_array;
        }

        private static EditCommand CreateVtxBufferEditCommand(GuiObjectGroup targets, List<List<VtxBufferArrayEditData>> bufferArrays, bool updateModified)
        {
            return new GeneralGroupReferenceEditCommand<List<VtxBufferArrayEditData>>(
                targets,
                GuiObjectID.Material,
                bufferArrays,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var buffers = (List<VtxBufferArrayEditData>)data;
                    foreach (var bufferArray in buffers)
                    {
                        var temp = bufferArray.vertex.vtx_buffer_array;
                        bufferArray.vertex.vtx_buffer_array = bufferArray.vtx_buffer_array;
                        bufferArray.vtx_buffer_array = temp;
                    }
                    swap = buffers;

                    // 他のコマンドと一緒にコマンドセット内で実行するのでビューアメッセージは省略
                },
                updateModified: updateModified);
        }

        public static EditCommand CreateVtxBufferEditCommand(GuiObjectGroup targets, bool updateModified = true)
        {
            var materials = targets.GetObjects(GuiObjectID.Material).OfType<Material>();

            var bufferArrays = (from material in materials
                                let attrib_assings = material.MaterialShaderAssign != null ? material.MaterialShaderAssign.AttribAssigns : Enumerable.Empty<attrib_assignType>()
                                let shadingModel = material.MaterialShaderAssign != null ? material.MaterialShaderAssign.ShadingModel : null
                                select (from tuple in material.Referrers.Any() ? IfVertexBufferUtility.CreateVtxBufferArray(material.Referrers.First().Data, material.Name, attrib_assings, shadingModel) : Enumerable.Empty<Tuple<vertexType, vtx_buffer_arrayType>>()
                                        select new VtxBufferArrayEditData()
                                        {
                                            vertex = tuple.Item1,
                                            vtx_buffer_array = tuple.Item2
                                        }).ToList()).ToList();

            return CreateVtxBufferEditCommand(targets, bufferArrays, updateModified);
        }
        #endregion

        #region デフォルト設定
        public static string ApplyOriginalMaterial(uniform_varType uniform_var, original_materialType original, out string hint)
        {
            switch (uniform_var.type)
            {
                case shader_param_typeType.float3:
                case shader_param_typeType.float4:
                    if (original.original_color_array != null)
                    {
                        foreach (var color in original.original_color_array.original_color)
                        {
                            if (IfShaderParameterUtility.IsOriginalColorApplicable(uniform_var.id, uniform_var.Item().ToString(), uniform_var.type.ToString(), uniform_var.hint, color.hint))
                            {
                                if (uniform_var.type == shader_param_typeType.float3)
                                {
                                    hint = color.hint;
                                    return IfUtility.MakeArrayString(color.color);
                                }
                                else
                                {
                                    hint = color.hint;
                                    return IfUtility.MakeArrayString(color.color.Concat(Enumerable.Repeat<float>(1.0f, 1)));
                                }
                            }
                        }
                    }
                    break;
                case shader_param_typeType.srt2d:
                    if (original.original_texsrt_array != null)
                    {
                        foreach (var texsrt in original.original_texsrt_array.original_texsrt)
                        {
                            if (IfShaderParameterUtility.IsOriginalSrtApplicable(uniform_var.id, uniform_var.type.ToString(), uniform_var.hint, texsrt.hint, texsrt.mode.ToString()))
                            {
                                hint = texsrt.hint;
                                return IfUtility.MakeArrayString(
                                    new float[] {
                                        texsrt.scale[0], texsrt.scale[1],
                                        texsrt.rotate,
                                        texsrt.translate[0], texsrt.translate[1]
                                    });
                            }
                        }
                    }
                    break;
                case shader_param_typeType.texsrt:
                    if (original.original_texsrt_array != null)
                    {
                        foreach (var texsrt in original.original_texsrt_array.original_texsrt)
                        {
                            if (IfShaderParameterUtility.IsOriginalSrtApplicable(uniform_var.id, uniform_var.type.ToString(), uniform_var.hint, texsrt.hint, texsrt.mode.ToString()))
                            {
                                hint = texsrt.hint;
                                return IfUtility.MakeArrayString(
                                    new float[] {
                                        (texsrt.mode == original_texsrt_modeType.maya) ? 0.0f :
                                        (texsrt.mode == original_texsrt_modeType.Item3dsmax) ? 1.0f :
                                        (texsrt.mode == original_texsrt_modeType.softimage) ? 2.0f : 0.0f,
                                        texsrt.scale[0], texsrt.scale[1],
                                        texsrt.rotate,
                                        texsrt.translate[0], texsrt.translate[1]
                                    });
                            }
                        }
                    }
                    break;
            }

            hint = null;
            return null;
        }

        public static render_infoType To_render_info(RenderInfo renderInfo, string[] values)
        {
            var builder = new StringBuilder();
            int count = 0;
            foreach (var value in values)
            {
                if (string.IsNullOrWhiteSpace(value))
                {
                    continue;
                }
                if (value.Length > 0)
                {
                    builder.Append(" ");
                }
                builder.Append(value);
                count++;
            }
            return new render_infoType()
            {
                name = renderInfo.name,
                count = count,
                Value = builder.Length > 0 ? builder.ToString() : null,
                type = renderInfo.type,
            };
        }

        #endregion
    }
}
