﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    public static class PathUtility
    {
        /// <remarks>
        /// ディレクトリのパスの末尾には必ずパス区切り記号(\ か /)を付けること
        /// </remarks>
        static public string MakeRelativePath(string basePath, string filePath)
        {
            Func<string, string> correctPath = (path) =>
            {
                // Uri に渡すパスに .. や . が含まれていても正常に相対パス化できが、連続するディレクトリセパレータの解釈が Windows のものとは異なる。
                // Win32.NativeMethods.PathCanonicalize() では連続したディレクトリセパレータの正規化は行えないので手動で正規化を行う。

                // ディレクトリセパレータを DirectorySeparatorChar に統一する。
                path = path.Replace(System.IO.Path.AltDirectorySeparatorChar, System.IO.Path.DirectorySeparatorChar);

                // UNC パスでない場合は、連続するディレクトリセパレータをひとつにまとめる。
                var pathUri = new Uri(path);
                if (!pathUri.IsUnc)
                {
                    // 二連続以上のディレクトリセパレータをひとつにするために置換ではなくループで処理する必要がある。
                    var target = new string(Enumerable.Repeat(System.IO.Path.DirectorySeparatorChar, 2).ToArray());
                    for (var i = path.IndexOf(target); i != -1; i = path.IndexOf(target, i))
                    {
                        path = path.Remove(i, 1);
                    }
                }

                return path;
            };

            // basePath を整理。
            basePath = correctPath(basePath);

            // filePath を整理。
            filePath = correctPath(filePath);

            var baseUri = new Uri(basePath);
            var fileUri = new Uri(filePath);

            // 相対パスを取得します。
            var relativeUri = baseUri.MakeRelativeUri(fileUri);
            // 相対パスを取得できない場合は元ファイルパスを返す
            if (relativeUri.IsAbsoluteUri)
            {
                return filePath;
            }
            var relativePathEscaped = relativeUri.ToString();

            // Uri クラス が +  エンコードしてくれない文字があるので、自分で置換します。
            relativePathEscaped = relativePathEscaped.Replace("+", "%2b");

            // エスケープされていない文字列に変換します
            string resultPath = System.Web.HttpUtility.UrlDecode(relativePathEscaped);

            return resultPath;
        }

        // 中間ファイル文字列から、アスキー・バイナリ部分を取り除いたファイル名を作ります。
        // 例) abc.fmda => abc.fmd
        //     def.ftx => def.ftx        アスキー・バイナリ部分がなければ加工なし
        //     fspa => fsp               拡張子だけでもOK
        //     .fspa => .fsp             拡張子だけでもOK
        static public string MakeNoFormatIfName(string src)
        {
            const int asciiBinarySignLength = 1;
            Debug.Assert(G3dPath.Text.Length == asciiBinarySignLength);
            Debug.Assert(G3dPath.Binary.Length == asciiBinarySignLength);
            src = src.ToLower();

            if (!src.EndsWith("tga") && (src.EndsWith(G3dPath.Text) || src.EndsWith(G3dPath.Binary)))
            {
                src = src.Substring(0, src.Length - asciiBinarySignLength);
            }

            return src;
        }

        static public string ToIfBinaryName(string src)
        {
            return MakeNoFormatIfName(src) + G3dPath.Binary;
        }

        static public string ToIfAsciiName(string src)
        {
            return MakeNoFormatIfName(src) + G3dPath.Text;
        }

        // ファイルダイアログのフィルタ文字列を生成します。(exts は. なしの拡張子)
        public static string MakeFileter(string description, IEnumerable<string> exts)
        {
            var builder = new StringBuilder();
            foreach (var ext in exts)
            {
                builder.Append(builder.Length == 0 ? "*." : ";*.");
                builder.Append(ext);
            }

            string extensions = builder.ToString();
            return description + " (" + extensions + ")|" + extensions;
        }

        // 正規化したパス文字列を得る
        // 成功したかどうかを返す。
        //	成功した場合は、dst に正規化したパスを格納する。
        //	失敗したら、dst の操作は行わない。
        public static bool NormalizePathString(string src, ref string dst)
        {
            try
            {
                if (Path.IsPathRooted(src))
                {
                    // ドキュメントに記述は見当たらないが、
                    // GetFullPath の段階で、/ は \ に変換されるので、
                    // ToWindowsPathSeparater は無くてもいいはず
                    dst = ToWindowsPathSeparator(Path.GetFullPath(src));
                    return true;
                }
                else
                {
                    return false;
                }
            }
            catch
            {
                return false;
            }
        }

        // '/'区切りのパス文字列から'\'区切りのパス文字列を作ります。
        public static string ToWindowsPathSeparator(string src)
        {
            return src.Replace('/', '\\');
        }

        // 名前は完全一位、拡張子は大文字小文字の違いを除いて一致
        public static bool ExistsFileName(string path, out string correctPath)
        {
            try
            {
                var info = new FileInfo(path);
                if (info.Exists)
                {
                    var fileName = Path.GetFileNameWithoutExtension(info.Name);

                    // 大文字小文字を無視して同じファイル名 (拡張子を含む) のファイルを列挙。
                    // DirectoryInfo.GetFiles() の searchPattern 引数が大文字小文字をどう扱うかの仕様が見つからないので手動比較している。
                    foreach (var file in info.Directory.GetFiles().Where(x => string.Compare(info.Name, x.Name, true) == 0))
                    {
                        var fileName2 = Path.GetFileNameWithoutExtension(file.Name);

                        // 拡張子を除いた名前が完全一致するかどうか。
                        var perfectMatching = string.CompareOrdinal(fileName, fileName2) == 0;

                        // 完全一致かどうかに関わらずパスは整える。
                        correctPath = file.FullName;
                        return perfectMatching;
                    }
                }
            }
            catch
            { }

            correctPath = null;
            return false;
        }

        // 拡張子込みの名前が完全一位
        public static bool ExistsFileNameWithExtension(string path, out string fileName, out string correctName)
        {
            fileName = null;
            correctName = null;

            try
            {
                var info = new FileInfo(path);
                if (info.Exists)
                {
                    fileName = info.Name;

                    // 大文字小文字を無視して同じファイル名 (拡張子を含む) のファイルを列挙。
                    // DirectoryInfo.GetFiles() の searchPattern 引数が大文字小文字をどう扱うかの仕様が見つからないので手動比較している。
                    foreach (var file in info.Directory.GetFiles().Where(x => string.Compare(x.Name, info.Name, true) == 0))
                    {
                        // 拡張子込みの名前が完全一致するかどうか。
                        var perfectMatching = string.CompareOrdinal(file.Name, info.Name) == 0;

                        // 大文字小文字が異なる場合はファイル名を整える。
                        if (!perfectMatching)
                        {
                            correctName = file.Name;
                        }

                        return perfectMatching;
                    }
                }
            }
            catch
            { }

            return false;
        }

        // ワイルドカードを含んだパスを正規化する
        // ただしワイルドカードの後に ".." , "." が有る場合は正規化出来ない
        public static string GetFullPath(string path)
        {
            var fullpath = string.Empty;
            // パスルート
            var root = Path.GetPathRoot(path);
            if (root == null)
            {
                return path;
            }
            path = path.Replace("/", "\\");
            // ルート以外のパスをフォルダごとに分割
            var underroot = path.Substring(root.Length);
            var splited = underroot.Split(new[] { '\\' }, StringSplitOptions.RemoveEmptyEntries);
            // 分割したパスをワイルドカードを含むフォルダ、ファイルで前後に分ける
            // c:\AAA\...\AAA\BBB*\CCC*\*.fmdb のばあい、「c:\AAA\..\AAA」「BBB*\CCC*\*.fmdb」
            var isSearchFolder = path.Last() == '\\';
            var joined = string.Empty;
            var basePath = string.Empty;
            var wildCardPath = string.Empty;
            foreach (var entry in splited)
            {
                if (string.IsNullOrEmpty(basePath))
                {
                    if (entry.Contains('*') || entry.Contains('?'))
                    {
                        basePath = joined;
                        joined = entry;
                        if (splited.Last() == entry)
                        {
                            wildCardPath = joined;
                        }
                    }
                    else if (splited.Last() == entry)
                    {
                        joined = Path.Combine(joined, entry);
                        basePath = joined;
                        joined = string.Empty;
                    }
                    else
                    {
                        joined = Path.Combine(joined, entry);
                    }
                }
                else
                {
                    joined = Path.Combine(joined, entry);
                    if (splited.Last() == entry)
                    {
                        wildCardPath = joined;
                    }
                }
            }

            basePath = Path.GetFullPath(Path.Combine(root,basePath));
            fullpath = Path.Combine(basePath, wildCardPath);
            if (isSearchFolder && fullpath.Last() != '\\')
            {
                fullpath += '\\';
            }

            return fullpath;
        }
    }
}
