﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization.Formatters.Binary;
using System.Linq;
using System.Xml;
using System.Xml.Serialization;
using System.Collections.Generic;
using System.Linq.Expressions;
namespace App.Utility
{
    public static class ObjectUtility
    {
        /// <summary>
        /// 対象オブジェクトのディープコピークローンを取得します。
        /// </summary>
        /// <typeparam name="T">対象オブジェクトの型</typeparam>
        /// <param name="target">コピー対象オブジェクトを指定します。</param>
        /// <returns>ディープコピーのクローンを返します。</returns>
        /// <remarks>このメソッでディープコピーするには、対象クラスがシリアライズ可能である必要があります。</remarks>
        public static T Clone<T>(T target)
        {
            if (target == null)
            {
                return target;
            }
            else if (target is Nintendo.G3dTool.Entities.IToolDataObject)
            {
                // Nintendo.G3dTool.Entities.ToolDataObject とその継承元には [Serializable] が付いていないため BinaryFormatter を使えない。
                // 代わりに ToolDataObject.DeepCopy() を呼び出す。
                return (T)((Nintendo.G3dTool.Entities.IToolDataObject)target).DeepCopy();
            }
            else
            {
                using (var stream = new MemoryStream())
                {
                    // 対象オブジェクトをシリアライズ
                    BinaryFormatter formatter = new BinaryFormatter();
                    formatter.Serialize(stream, target);
                    stream.Position = 0;

                    // シリアライズデータをデシリアライズ
                    return (T)formatter.Deserialize(stream);
                }
            }
        }

        public static T CloneWithXmlSerializer<T>(T target)
        {
            if (target == null)
            {
                return target;
            }
            else
            {
#if true

                using (var stream = new MemoryStream())
                {
                    var serializer = new XmlSerializer(typeof(T));
                    var writerSettings = new XmlWriterSettings()
                    {
                        Indent = false,
                        CheckCharacters = false,
                        OmitXmlDeclaration = true,
                    };
                    using (var writer = XmlWriter.Create(stream, writerSettings))
                    {
                        serializer.Serialize(writer, target);
                    }
                    stream.Seek(0, SeekOrigin.Begin);
                    var readerSetting = new XmlReaderSettings()
                    {
                        CheckCharacters = false,
                    };
                    using (var reader = XmlReader.Create(stream, readerSetting))
                    {
                        return (T)serializer.Deserialize(reader);
                    }
                }
#else
                using (var stream = new MemoryStream())
                {
                    var serializer = new XmlSerializer(typeof(T));
                    serializer.Serialize(stream, target);
                    stream.Seek(0, SeekOrigin.Begin);
                    return (T)serializer.Deserialize(stream);
                }
#endif
            }
        }

        public static T[] MultipleClone<T>(T target, int length)
        {
            T[] clones = new T[length];
            for (int i = 0; i < length; i++)
            {
                clones[i] = Clone<T>(target);
            }

            return clones;
        }

        public static byte[] ToBytes<T>(T target)
        {
            if (target == null)
            {
                return new byte[0];
            }
            else
            {
                using (var stream = new MemoryStream())
                {
                    BinaryFormatter formatter = new BinaryFormatter();
                    formatter.Serialize(stream, target);
                    return stream.ToArray();
                }
            }
        }

        /// <summary>
        /// byte 列の高速比較
        /// </summary>
        public static unsafe bool IsSameBytes(byte[] a1, byte[] a2)
        {
            if (a1 == null || a2 == null || a1.Length != a2.Length)
            {
                return a1 == a2;
            }
            fixed (byte* p1 = a1, p2 = a2)
            {
                int len = a1.Length/8;
                long* l1 = (long*)p1;
                long* l2 = (long*)p2;

                // 8byte毎に比較
                for (int i=0;i<len;i++)
                {
                    if (l1[i] != l2[i])
                    {
                        return false;
                    }
                }

                // 残りを比較
                for (int i=len*8;i<a1.Length;i++)
                {
                    if (p1[i] != p2[i])
                    {
                        return false;
                    }
                }
                return true;
            }
        }
    }
}
