﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;

namespace App.Utility
{
    /// <summary>
    /// テクスチャ画像ビットマップ作成クラス。
    /// 実テクスチャデータからカラー/アルファそれぞれの Bitmap イメージを作成します。
    /// </summary>
    public static class ImageBitmapCreator
    {
        /// <summary>
        /// ミップマップ用 Bitmap 配列を１つの Bitmap に結合します。
        /// 下位レベルの画像は左下方向に向かって連続的に配置されます。
        /// 結合条件を満たさない場合は NULL が返ります。
        /// </summary>
        public static Bitmap CombineMipmapBitmaps(Bitmap[] bitmaps, Size mipmapMinSize, ConfigCommon.MipmapOrientationType mipmapOrientation)
        {
            //-----------------------------------------------------------------
            // 準備
            //-----------------------------------------------------------------
            // ２レベル以上必要
            if (bitmaps.Length == 1)
            {
                return (Bitmap)bitmaps[0].Clone();
            }

            // フォーマット確認
            var format = bitmaps[0].PixelFormat;
            switch (format)
            {
                // 対応するフォーマットのみ
                case PixelFormat.Format32bppArgb:
                    break;
                // 対応外のフォーマット
                default:
                    return null;
            }

            // サイズ確認
            var topLvSize = bitmaps[0].Size;
            var totalSize = topLvSize;
            var checkSize = topLvSize;
            for (int i = 1; i < bitmaps.Length; i++)
            {
                checkSize =
                    new Size(
                        Math.Max(mipmapMinSize.Width,  checkSize.Width  / 2),
                        Math.Max(mipmapMinSize.Height, checkSize.Height / 2)
                    );

                // 下位レベルがトップレベルと同じフォーマットか
                Debug.Assert(bitmaps[i].PixelFormat == format);

                if (mipmapOrientation == ConfigCommon.MipmapOrientationType.Horizontal)
                {
                    totalSize.Width += checkSize.Width;
                }
                else
                {
                    totalSize.Height += checkSize.Height;
                }
            }

            //-----------------------------------------------------------------
            // 作成
            //-----------------------------------------------------------------
            var bitmap = new Bitmap(totalSize.Width, totalSize.Height, PixelFormat.Format32bppArgb);
            using (var bp = Graphics.FromImage(bitmap))
            {
                bp.Clear(Color.FromArgb(0x00, 0x00, 0x00, 0x00));
            }

            CombineBitmaps_32bppArgb(bitmap, bitmaps, mipmapOrientation);
            return bitmap;
        }

        /// <summary>
        /// ビットマップ結合（32bppArgb）
        /// </summary>
        private static void CombineBitmaps_32bppArgb(Bitmap dstBitmap, Bitmap[] srcBitmaps, ConfigCommon.MipmapOrientationType mipmapOrientation)
        {
            int pixelOffset = 0;
            unsafe
            {
                // 先画像に
                using (var dstBp = new BitmapProcess(dstBitmap))
                {
                    byte* dstBmpPtr = (byte*)(void*)dstBp.BitmapData.Scan0;
                    int dstStride = dstBp.BitmapData.Stride;

                    // 元画像をレベル順にコピー
                    for (int i = 0; i < srcBitmaps.Length; i++)
                    {
                        var srcBitmap = srcBitmaps[i];

                        using (var srcBp = new BitmapProcess(srcBitmap))
                        {
                            byte* srcBmpPtr = (byte*)(void*)srcBp.BitmapData.Scan0;
                            int srcStride = srcBp.BitmapData.Stride;
                            int rasterSize = srcBitmap.Width * 4;

                            if (mipmapOrientation == ConfigCommon.MipmapOrientationType.Horizontal)
                            {
                                for (int y = 0; y < srcBitmap.Height; y++)
                                {
                                    using (var streamSrc = new UnmanagedMemoryStream(srcBmpPtr + srcStride * y,						rasterSize, rasterSize, FileAccess.Read ))
                                    using (var streamDst = new UnmanagedMemoryStream(dstBmpPtr + dstStride * y + (pixelOffset * 4),	rasterSize, rasterSize, FileAccess.Write))
                                    {
                                        streamSrc.CopyTo(streamDst);
                                    }
                                }

                                pixelOffset += srcBitmap.Width;
                            }
                            else
                            {
                                for (int y = 0; y < srcBitmap.Height; y++)
                                {
                                    using (var streamSrc = new UnmanagedMemoryStream(srcBmpPtr + srcStride * y,						rasterSize, rasterSize, FileAccess.Read ))
                                    using (var streamDst = new UnmanagedMemoryStream(dstBmpPtr + dstStride * (pixelOffset + y),		rasterSize, rasterSize, FileAccess.Write))
                                    {
                                        streamSrc.CopyTo(streamDst);
                                    }
                                }

                                pixelOffset += srcBitmap.Height;
                            }
                        }
                    }
                }
            }
        }
    }
}
