﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Drawing.Imaging;

namespace App.Utility
{
    /// <summary>
    /// カラーブレンダクラス。
    /// </summary>
    public static partial class ColorUtility
    {
        /// <summary>
        /// 半々の比率でブレンド。
        /// </summary>
        public static Color Blend(Color color1, Color color2)
        {
            return Blend(color1, color2, 0.5f);
        }

        /// <summary>
        /// 混合率を指定してブレンド。
        /// </summary>
        public static Color Blend(Color color1, Color color2, float ratio)
        {
            byte r = ToElement(ToRatio(color1.R) * (1.0f - ratio) + ToRatio(color2.R) * ratio);
            byte g = ToElement(ToRatio(color1.G) * (1.0f - ratio) + ToRatio(color2.G) * ratio);
            byte b = ToElement(ToRatio(color1.B) * (1.0f - ratio) + ToRatio(color2.B) * ratio);

            return Color.FromArgb(r, g, b);
        }

        /// <summary>
        /// アルファカラーのブレンド。
        /// </summary>
        public static Color AlphaBlend(Color color, Color alphaColor)
        {
            return Blend(color, alphaColor, ToRatio(alphaColor.A));
        }

        /// <summary>
        /// 行列を指定してブレンド。
        /// </summary>
        public static Color MatrixBlend(Color color, ColorMatrix cm)
        {
            float rf = ToRatio(color.R);
            float gf = ToRatio(color.G);
            float bf = ToRatio(color.B);
            float af = ToRatio(color.A);

            byte r = ToElement(rf * cm.Matrix00 + gf * cm.Matrix10 + bf * cm.Matrix20 + af * cm.Matrix30 + 1.0f * cm.Matrix40);
            byte g = ToElement(rf * cm.Matrix01 + gf * cm.Matrix11 + bf * cm.Matrix21 + af * cm.Matrix31 + 1.0f * cm.Matrix41);
            byte b = ToElement(rf * cm.Matrix02 + gf * cm.Matrix12 + bf * cm.Matrix22 + af * cm.Matrix32 + 1.0f * cm.Matrix42);
            byte a = ToElement(rf * cm.Matrix03 + gf * cm.Matrix13 + bf * cm.Matrix23 + af * cm.Matrix33 + 1.0f * cm.Matrix43);

            return Color.FromArgb(a, r, g, b);
        }

        /// <summary>
        /// 要素（0-255）を割合率（0.0-1.0）に変換。
        /// </summary>
        public static float ToRatio(byte element)
        {
            return (float)element / 255.0f;
        }

        /// <summary>
        /// 割合率（0.0-1.0）を要素（0-255）に変換。
        /// </summary>
        public static byte ToElement(float ratio)
        {
            if      (ratio < 0.0f) { return 0;                      }
            else if (ratio > 1.0f) { return 255;                    }
            else                   { return (byte)(ratio * 255.0f); }
        }

        /// <summary>
        /// 割合率（0.0-1.0）を要素（0-255）に変換するための必要最小限な精度に丸める。
        /// </summary>
        public static float Round(float ratio)
        {
            // 256 諧調には小数第 3 位までは必要。
            // ここでは小数第 4 位の切り上げが適切であり、四捨五入は不適切。
            return
                (ratio > 0.0f) ? (float)(Math.Ceiling(ratio * 1000.0) / 1000.0) :
                (ratio < 0.0f) ? (float)(Math.Floor(ratio * 1000.0) / 1000.0) :
                ratio;
        }

        /// <summary>
        /// ガンマ補正
        /// </summary>
        public static Color Pow(Color color, double gamma)
        {
            if (gamma == 1)
            {
                return color;
            }
            else if (NearyEqual(gamma, double_gamma22) || NearyEqual((float)gamma, float_gamma22))
            {
                // 補整値 2.2 では厳密な変換を行う。
                return ToSrgbFromLinear(color);
            }
            else
            {
                byte r = (byte)(Math.Pow(color.R / 255.0, gamma) * 255.0);
                byte g = (byte)(Math.Pow(color.G / 255.0, gamma) * 255.0);
                byte b = (byte)(Math.Pow(color.B / 255.0, gamma) * 255.0);
                return Color.FromArgb(color.A, r, g, b);
            }
        }

        /// <summary>
        /// sRGB カラースペースをリニアカラースペースに変換する。
        /// </summary>
        public static Color ToLienarFromSrgb(Color color)
        {
            var r = (byte)(ToLinearFromSrgb(color.R / 255.0) * 255);
            var g = (byte)(ToLinearFromSrgb(color.G / 255.0) * 255);
            var b = (byte)(ToLinearFromSrgb(color.B / 255.0) * 255);
            return Color.FromArgb(color.A, r, g, b);
        }

        /// <summary>
        /// sRGB カラースペースをリニアカラースペースに変換する。
        /// </summary>
        public static double ToLinearFromSrgb(double x)
        {
            // https://en.wikipedia.org/wiki/SRGB
            return (x <= 0.04045) ? (x / 12.92) : Math.Pow((x + 0.055) / (1.0 + 0.055), 2.4);
        }

        /// <summary>
        /// リニアカラースペースを sRGB カラースペースに変換する。
        /// </summary>
        public static Color ToSrgbFromLinear(Color color)
        {
            var r = (byte)(ToSrgbFromLinear(color.R / 255.0) * 255);
            var g = (byte)(ToSrgbFromLinear(color.G / 255.0) * 255);
            var b = (byte)(ToSrgbFromLinear(color.B / 255.0) * 255);
            return Color.FromArgb(color.A, r, g, b);
        }

        /// <summary>
        /// リニアカラースペースを sRGB カラースペースに変換する。
        /// </summary>
        public static double ToSrgbFromLinear(double x)
        {
            // https://en.wikipedia.org/wiki/SRGB
            return (x <= 0.0031308) ? (12.92 * x) : ((1.0 + 0.055) * Math.Pow(x, 1.0 / 2.4) - 0.055);
        }

        public static Color GammaCorrect22(Color src)
        {
            // 元々は gamma22_[] のテーブルで変換を行っていたが、テーブル値は Pow(linear, 1.0 / 2.2) で作成された値のようなので厳密な変換には適さない。
            // ToSrgbFromLinear() を使って厳密な変換を行う。
            // テーブルを Pow() から ToSrgbFromLinear() のものに書き換えてもよいかも。
            return ToSrgbFromLinear(src);
        }

        public static Color ToGray(Color color)
        {
            const float r = 0.298912F;
            const float g = 0.586611F;
            const float b = 0.114478F;

            var gray = (byte)Math.Min(255, r * color.R + g * color.G + b * color.B);

            return Color.FromArgb(color.A, gray, gray, gray);
        }

        private static bool NearyEqual(float a, float b)
        {
            return Math.Abs(a - b) < (float_epsilon * (float)Math.Pow(2.0f, Math.Log(Math.Max(Math.Abs(a), Math.Abs(b)), 2.0f)));
        }

        private static bool NearyEqual(double a, double b)
        {
            return Math.Abs(a - b) < (double_epsilon * Math.Pow(2.0, Math.Log(Math.Max(Math.Abs(a), Math.Abs(b)), 2.0)));
        }

        private static double CalcEpsilon(double x)
        {
            double e = x;
            while ((e / 2.0 + x) > x)
            {
                e /= 2.0;
            }
            return e;
        }

        private static float CalcEpsilon(float x)
        {
            float e = x;
            while ((e / 2.0f + x) > x)
            {
                e /= 2.0f;
            }
            return e;
        }

        const float float_gamma22 = 1.0f / 2.2f;
        static readonly float float_epsilon = CalcEpsilon(1.0f);

        const double double_gamma22 = 1.0 / 2.2;
        static readonly double double_epsilon = CalcEpsilon(1.0);
    }
}
