﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Linq;
using App.Command;
using App.Controls;
using App.Data;
using App.Utility;

namespace App.PropertyEdit
{
    partial class ObjectPropertyPanel
    {
        private ColorPickerAdapter symbolColorAdapter_ = null;
        private GuiObjectColorPickerTrigger guiObjectColorPickerTrigger_ = null;

        private class GuiObjectColorPickerTrigger : IColorPickerTrigger
        {
            public GuiObjectGroup GuiObjectGroup{ private get; set;}

            public RgbaColor Color
            {
                get
                {
                    RgbaColor? color = ((GuiObject)(GuiObjectGroup.Active)).EditColor;
                    return color.HasValue ? color.Value : RgbaColor.FromColor(System.Drawing.Color.White);
                }
            }

            /// <summary>
            /// 編集用カラーかどうか。
            /// </summary>
            public bool IsMarkColor
            {
                get { return true; }
            }

            public bool EnableAlpha
            {
                get
                {
                    return false;
                }
            }

            public bool IsDefaultLinear
            {
                get;
                set;
            }

            public bool ReadOnly
            {
                get
                {
                    return false;
                }
            }

            /// <summary>
            /// HDR の上限。
            /// </summary>
            public float HDRUpperBound
            {
                get
                {
                    return 1.0f;
                }
            }
        }

        private void InitializeEditColor()
        {
            guiObjectColorPickerTrigger_ = new GuiObjectColorPickerTrigger()
            {
                IsDefaultLinear = false,
            };
            symbolColorAdapter_ = new ColorPickerAdapter(guiObjectColorPickerTrigger_);
            symbolColorAdapter_.ColorEdit += Event_symbolColorAdapter_ColorEdit;
            symbolColorAdapter_.Nullify += Event_symbolColorAdapter_Nullify;
            ecmEditColorMark.Gamma = 1;
        }

        private void UpdateEditColor()
        {
            guiObjectColorPickerTrigger_.GuiObjectGroup = Targets;

            var target = Targets.Active;
            if (target != null)
            {

                ecmEditColorMark.Color = target.EditColor;
                ecmEditColorMark.Enabled = target.Editable;

                modifiedMarkEditColor.Visible = target.Editable && target.EditColor != target.savedEditColor;
                lblComment.IsModified = target.Editable && target.Comment != target.savedComment;
                lblLabel.IsModified = target.Editable && target.Label != target.savedLabel;
                tbxComment.Value = target.Comment;
                tbxComment.Enabled = target.Editable;
                tbxLabel.Value = target.Label;
                tbxLabel.Enabled = target.Editable;
            }
            else
            {
                ecmEditColorMark.Enabled = false;

                modifiedMarkEditColor.Visible = false;
                lblComment.IsModified = false;
                lblLabel.IsModified = false;
                tbxComment.Value = string.Empty;
                tbxComment.Enabled = false;
                tbxLabel.Value = string.Empty;
                tbxLabel.Enabled = false;
            }
        }

        private void Event_symbolColorAdapter_ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (e.EditFixed)
            {
                var editables = Targets.Objects.Where(x => x.Editable);
                TheApp.CommandManager.Execute(CreateEditColorEditCommand(new GuiObjectGroup(editables), RgbaColor.Round(e.Color)));
                ecmEditColorMark.Invalidate();
            }
        }

        private void Event_symbolColorAdapter_Nullify()
        {
            var editables = Targets.Objects.Where(x => x.Editable);
            TheApp.CommandManager.Execute(CreateEditColorEditCommand(new GuiObjectGroup(editables), null));
            ecmEditColorMark.Invalidate();
        }

        public static GroupEditCommand CreateEditColorEditCommand(GuiObjectGroup targets, RgbaColor? color)
        {
            return
                new GeneralGroupReferenceEditCommand</* Color? */object>(
                    targets,
                    null,
                    ObjectUtility.MultipleClone((object)color, targets.Objects.Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        swap = ObjectUtility.Clone(target.EditColor);
                        target.EditColor = (RgbaColor?)data;
                    }
                );
        }

        public static GroupEditCommand CreateEditCommentEditCommand(GuiObjectGroup targets, string comment)
        {
            return
                new GeneralGroupReferenceEditCommand<string>(
                    targets,
                    null,
                    ObjectUtility.MultipleClone(comment, targets.Objects.Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        swap = ObjectUtility.Clone(target.Comment);
                        target.Comment = (string)data;
                    }
                );
        }

        public static GroupEditCommand CreateEditLabelEditCommand(GuiObjectGroup targets, string label)
        {
            return
                new GeneralGroupReferenceEditCommand<string>(
                    targets,
                    null,
                    ObjectUtility.MultipleClone(label, targets.Objects.Count),
                    delegate (ref GuiObject target, ref object data, ref object swap)
                    {
                        swap = ObjectUtility.Clone(target.Label);
                        target.Label = (string)data;
                    }
                );
        }
    }
}
