﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public class AnimationCurveDrawInfo
    {
        public AnimationCurveDrawInfo(CurveViewPainter painter, PaintEventArgs eventArgs)
        {
            Painter			= painter;
            EventArgs		= eventArgs;
        }

        public CurveViewPainter	Painter{		get; set; }
        public PaintEventArgs	EventArgs{		get; set; }

        public bool				IsVisibleFrameValue{	get; set; }
        public bool				IsVisibleSlope{			get; set; }
        public bool				IsVisibleMinMax{		get; set; }
        public bool				IsVisibleCurrentValue{	get; set; }
        public bool IsVisibleCurveName { get; set; }

        public bool				IsSelected{	get; set; }
    }

    public class AnimationCurveSelectionInfo
    {
        public class PostEndSelectionResultType
        {
            public bool IsHit{ get; set; }
        }

        public AnimationCurveSelectionInfo()
        {
            PostEndSelectionResult = new PostEndSelectionResultType();
        }

        public double	MinFrame{			get; set; }
        public double	MaxFrame{			get; set; }
        public double	MinValue{			get; set; }
        public double	MaxValue{			get; set; }
        public int		SelectionLeft{		get; set; }
        public int		SelectionTop{		get; set; }
        public int		SelectionRight{		get; set; }
        public int		SelectionBottom{	get; set; }

        public CurveView        View    { get; set; }
        public CurveViewPainter Painter
        {
            get { return View.Painter; }
        }

        public bool				IsToggle{	get; set; }

        public PostEndSelectionResultType	PostEndSelectionResult{	get; set; }
    }

    public interface IAnimationCurve
    {
        List<KeyFrame>		KeyFrames{				get;		}
        InterpolationType	CurveInterpolationType{	get; set;	}
        PrimitiveTypeKind	CurvePrimitiveType{		get;		}
        string				ParentName{				get;		}
        string				Name{					get;		}
        string				FullPath{				get;		}
        int					ComponentIndex{			get;		}
        int ColorComponentIndex { get; }
        bool				IsRotate{				get;		}
        float?	MinClampValue{	get;	}
        float?	MaxClampValue{	get;	}
        float?	MinFitValue{	get;	}
        float?	MaxFitValue{	get;	}
        Color	CurveColor		{	get;	}

        bool 								IsEditable		{ get; set; }
        AnimationDocument.NonEditableKind	NonEditableKind	{ get; set; }
        object								NonEditableKindDisplayAux	{ get; set; }

        AnimTarget GetAnimTarget(GuiObject targetOwner);
        AnimTarget CreateAnimTarget(GuiObject targetOwner);
        AnimTarget CreateTemporaryTarget(GuiObject targetOwner);
        void SetAnimTarget(GuiObject targetOwner, AnimTarget target);
        float GetDefaultValue(GuiObject targetOwner);
        bool IsSame(IAnimationCurve animCurve);
        void UpdateIsModified(GuiObject targetOwner);
        bool IsColorCurve { get; }
    }

    // HIO用カーブ
    public class HioCurve : NintendoWare.G3d.Edit.ICurve<NintendoWare.G3d.Edit.ICurveValue>
    {
        //public static NintendoWare.G3d.Edit.Interpolatio
        public HioCurve(AnimTarget animTarget, bool isRotate, NintendoWare.G3d.Edit.ValueType valueType, bool quantized, CurveExportType exportType)
        {
            if (exportType == CurveExportType.Curve)
            {
                IsDegreeValue = isRotate;

                Values = animTarget.KeyFrames;
                InterpolationType = Convert(animTarget.CurveInterpolationType);
                PreWrap = Convert(animTarget.pre_wrap);
                PostWrap = Convert(animTarget.post_wrap);
                if (quantized)
                {
                    FrameType = Convert(animTarget.QuantizationInfo.frame_type);
                    KeyType = Convert(animTarget.QuantizationInfo.key_type);
                    Scale = animTarget.QuantizationInfo.scale;
                    Offset = animTarget.QuantizationInfo.offset;
                }
                else
                {
                    FrameType = NintendoWare.G3d.Edit.FrameType.Frame32;
                    KeyType = NintendoWare.G3d.Edit.KeyType.Key32;
                    Scale = 1;
                    Offset = 0;
                }
                IsBaked = animTarget.baked;
                ValueType = valueType;
            }
            else
            {
                Debug.Assert(exportType == CurveExportType.Constant);
                Debug.Assert(animTarget.KeyFrames.Count > 0);
                IsDegreeValue = isRotate;

                InterpolationType = NintendoWare.G3d.Edit.InterpolationType.Step;
                PreWrap = NintendoWare.G3d.Edit.WrapType.Clamp;
                PostWrap = NintendoWare.G3d.Edit.WrapType.Clamp;
                FrameType = NintendoWare.G3d.Edit.FrameType.None;
                KeyType = NintendoWare.G3d.Edit.KeyType.None;
                Scale = 1;
                Offset = 0;
                IsBaked = false;
                ValueType = valueType;
                var value = animTarget.KeyFrames[0].Value;
                DebugConsole.WriteLine(value.ToString());
                Values = new List<KeyFrame>()
                {
                    new KeyFrame() {
                        Frame = 0,
                        Value = value,
                        InSlope = 0,
                        OutSlope = 0,
                    },
                    new KeyFrame() {
                        Frame = 1,
                        Value = value,
                        InSlope = 0,
                        OutSlope = 0,
                    }
                };
            }
        }

        public NintendoWare.G3d.Edit.ICurveValue[] ValueArray
        {
            get
            {
                return Values.ToArray<NintendoWare.G3d.Edit.ICurveValue>();
            }
        }

        public IEnumerable<NintendoWare.G3d.Edit.ICurveValue> CurveValues
        {
            get
            {
                return Values;
            }
        }

        public IEnumerable<NintendoWare.G3d.Edit.ICurveValue>	Values				{ get; private set; }
        public NintendoWare.G3d.Edit.InterpolationType			InterpolationType	{ get; private set; }
        public NintendoWare.G3d.Edit.FrameType					FrameType			{ get; private set; }
        public NintendoWare.G3d.Edit.KeyType					KeyType				{ get; private set; }
        public NintendoWare.G3d.Edit.WrapType					PreWrap				{ get; private set; }
        public NintendoWare.G3d.Edit.WrapType					PostWrap			{ get; private set; }
        public NintendoWare.G3d.Edit.ValueType ValueType { get; private set; }
        public float											Scale				{ get; private set; }
        public float Offset { get; private set; }
        public bool IsBaked { get; private set; }
        public bool IsDegreeValue { get; private set; }

        private static NintendoWare.G3d.Edit.InterpolationType Convert(App.Data.InterpolationType v)
        {
            switch (v)
            {
                case App.Data.InterpolationType.Hermite:
                    return NintendoWare.G3d.Edit.InterpolationType.Hermite;
                case App.Data.InterpolationType.Linear:
                    return NintendoWare.G3d.Edit.InterpolationType.Linear;
                case App.Data.InterpolationType.Step:
                    return NintendoWare.G3d.Edit.InterpolationType.Step;
            }
            return NintendoWare.G3d.Edit.InterpolationType.Hermite;
        }
        // 型変換用
        private static NintendoWare.G3d.Edit.FrameType	Convert(curve_frame_typeType v)
        {
            Debug.Assert(frameTypeDict_.ContainsKey(v));
            return frameTypeDict_[v];
        }

        private static NintendoWare.G3d.Edit.KeyType	Convert(curve_key_typeType v)
        {
            Debug.Assert(keyTypeDict_.ContainsKey(v));
            return keyTypeDict_[v];
        }

        private static NintendoWare.G3d.Edit.WrapType	Convert(curve_wrapType v)
        {
            Debug.Assert(wrapTypeDict_.ContainsKey(v));
            return wrapTypeDict_[v];
        }

        private static readonly Dictionary<curve_frame_typeType, NintendoWare.G3d.Edit.FrameType>	frameTypeDict_	=
            new Dictionary<curve_frame_typeType, NintendoWare.G3d.Edit.FrameType>()
            {
                {curve_frame_typeType.none,		NintendoWare.G3d.Edit.FrameType.None},
                {curve_frame_typeType.frame32,	NintendoWare.G3d.Edit.FrameType.Frame32},
                {curve_frame_typeType.frame16,	NintendoWare.G3d.Edit.FrameType.Frame16},
                {curve_frame_typeType.frame8,	NintendoWare.G3d.Edit.FrameType.Frame8},
            };

        private static readonly Dictionary<curve_key_typeType, NintendoWare.G3d.Edit.KeyType>		keyTypeDict_	=
            new Dictionary<curve_key_typeType, NintendoWare.G3d.Edit.KeyType>()
            {
                {curve_key_typeType.none,	NintendoWare.G3d.Edit.KeyType.None},
                {curve_key_typeType.key32,	NintendoWare.G3d.Edit.KeyType.Key32},
                {curve_key_typeType.key16,	NintendoWare.G3d.Edit.KeyType.Key16},
                {curve_key_typeType.key8,	NintendoWare.G3d.Edit.KeyType.Key8},
            };

        private static readonly Dictionary<curve_wrapType, NintendoWare.G3d.Edit.WrapType>			wrapTypeDict_	=
            new Dictionary<curve_wrapType, NintendoWare.G3d.Edit.WrapType>()
            {
                {curve_wrapType.clamp,	NintendoWare.G3d.Edit.WrapType.Clamp},
                {curve_wrapType.repeat,	NintendoWare.G3d.Edit.WrapType.Repeat},
                {curve_wrapType.mirror,	NintendoWare.G3d.Edit.WrapType.Mirror},
                {curve_wrapType.relative_repeat, NintendoWare.G3d.Edit.WrapType.RelativeRepeat},
            };
    }

    // 拡張メソッド
    public static class AnimationCurveExtensions
    {
        public static bool HasKeyFrame(this IAnimationCurve curve)
        {
            return curve.KeyFrames.Any();
        }

        public static void SetCurvesSequentialIndex(this IAnimationCurve curve, int index = 0)
        {
            curve.KeyFrames.ForEach(x => x.Index = index ++);
        }

        public static void SortByFrame(this IAnimationCurve curve)
        {
            KeyFramesUtility.SortByFrame(curve.KeyFrames);
        }

        public static void SortByIndex(this IAnimationCurve curve)
        {
            curve.KeyFrames.Sort((x, y) => x.Index.CompareTo(y.Index));
        }

        public static bool IsSameCurve(this IAnimationCurve curve, IAnimationCurve target)
        {
            return
                (curve.ParentName		== target.ParentName) &&
                (curve.Name				== target.Name) &&
                (curve.ComponentIndex	== target.ComponentIndex);
        }

        public static float SnapValue(this IAnimationCurve curve, float src, double snap)
        {
            switch(curve.CurvePrimitiveType)
            {
                case PrimitiveTypeKind.Bool:		return (src >= 0.5f) ? 1.0f : 0.0f;
                case PrimitiveTypeKind.Int:			return (float)Math.Round(src, MidpointRounding.AwayFromZero);
                case PrimitiveTypeKind.Uint:        return (float)Math.Max(Math.Round(src, MidpointRounding.AwayFromZero), 0);
                case PrimitiveTypeKind.Float:		return snap != 0 ? (float)(Math.Round(src / snap, MidpointRounding.AwayFromZero) * snap): src;
                default:	Debug.Assert(false);	break;
            }

            return src;
        }

        public static float ClampValue(this IAnimationCurve curve, float src)
        {
            if (curve.MinClampValue != null)
            {
                src = Math.Max(src, (float)curve.MinClampValue);
            }

            if (curve.MaxClampValue != null)
            {
                src = Math.Min(src, (float)curve.MaxClampValue);
            }

            switch (curve.CurvePrimitiveType)
            {
                case PrimitiveTypeKind.Bool: return (src >= 0.5f) ? 1.0f : 0.0f;
                case PrimitiveTypeKind.Int: return (float)Math.Round(src, MidpointRounding.AwayFromZero);
                case PrimitiveTypeKind.Uint: return (float)Math.Max(Math.Round(src, MidpointRounding.AwayFromZero), 0);
            }
            return src;
        }

        public static float calcShiftFrame(float frame, List<KeyFrame> keys, curve_wrapType preWrap, curve_wrapType postWrap, out int shiftCount)
        {
            float targetFrame = frame;
            float startFrame = keys[0].Frame;
            float endFrame = keys[keys.Count - 1].Frame;
            float frameRange = keys[keys.Count - 1].Frame - keys[0].Frame;


            if (frameRange == 0.0f)
            {
                // 同じフレームに複数キーが打たれている
                // どんなwrapモードでもフレームは同じ
                shiftCount = 0;
                return targetFrame;
            }

            float dist = targetFrame - startFrame;
            shiftCount = (int)Math.Floor(dist / frameRange);

            // キーフレームを打ってある範囲のフレームが指定されていない場合に、範囲内のフレームにずらす。
            if ((targetFrame < startFrame))
            {
                // 一番小さいキーフレームより小さいframeの場合、
                // preWrapに応じて処理する。
                switch (preWrap)
                {
                    case curve_wrapType.clamp:
                        targetFrame = startFrame;
                        break;
                    case curve_wrapType.repeat:
                    case curve_wrapType.relative_repeat:
                        {
                            targetFrame -= shiftCount * frameRange;
                            break;
                        }
                    case curve_wrapType.mirror:
                        {
                            targetFrame -= shiftCount * frameRange;
                            if (shiftCount % 2 != 0)
                            {
                                targetFrame = endFrame - (targetFrame - startFrame);
                            }
                            break;
                        }
                }
            }
            else if ((targetFrame > endFrame))
            {
                // 一番大きいキーフレームより大きいframe
                // postWrapに応じて処理する。
                switch (postWrap)
                {
                    case curve_wrapType.clamp:
                        targetFrame = keys[keys.Count - 1].Frame;
                        break;
                    case curve_wrapType.repeat:
                    case curve_wrapType.relative_repeat:
                        {
                            targetFrame -= shiftCount * frameRange;
                            break;
                        }
                    case curve_wrapType.mirror:
                        {
                            targetFrame -= shiftCount * frameRange;
                            if (shiftCount % 2 != 0)
                            {
                                targetFrame = endFrame - (targetFrame - startFrame);
                            }
                            break;
                        }
                }
            }
            else
            {
                // 範囲内なのでずれはない
                shiftCount = 0;
            }

            return targetFrame;
        }

        public static float calcFrameValue(float frame, List<KeyFrame> keys, InterpolationType InterpolationType)
        {
            for (int i = 0; i < keys.Count - 1; ++i)
            {
                KeyFrame left = keys[i + 0];
                KeyFrame right = keys[i + 1];

                if ((frame >= left.Frame) &&
                    (frame < right.Frame))
                {
                    switch (InterpolationType)
                    {
                        case InterpolationType.Hermite:
                            {
                                return MathUtility.HermiteInterpolation(frame, left, right);
                            }

                        case InterpolationType.Linear:
                            {
                                return MathUtility.LinearInterpolation(frame, left, right);
                            }

                        case InterpolationType.Step:
                            {
                                return left.Value;
                            }

                        default:
                            {
                                Debug.Assert(false);
                                return 0.0f;
                            }
                    }
                }
            }

            // この判定を行わないと、Stepカーブの場合に上手くいかない。
            if (keys.Count > 0)
            {
                if (frame <= keys[0].Frame)
                {
                    return keys[0].Value;
                }

                if (frame >= keys[keys.Count - 1].Frame)
                {
                    return keys[keys.Count - 1].Value;
                }
            }

            // 指定フレームが範囲外の場合
            Debug.Assert(false);
            return 0.0f;
        }

        public static float calcFrameSlope(float frame, List<KeyFrame> keys, InterpolationType InterpolationType)
        {
            for (int i = 0; i < keys.Count - 1; ++i)
            {
                KeyFrame left = keys[i + 0];
                KeyFrame right = keys[i + 1];

                if ((frame >= left.Frame) &&
                    (frame <= right.Frame) &&
                    (left.Frame != right.Frame))
                {
                    switch (InterpolationType)
                    {
                        case InterpolationType.Hermite:
                            {
                                return MathUtility.HermiteSlope(frame, left, right);
                            }

                        case InterpolationType.Linear:
                            {
                                return (right.Value-left.Value)/(right.Frame-left.Frame);
                            }

                        case InterpolationType.Step:
                            {
                                return 0;
                            }

                        default:
                            {
                                Debug.Assert(false);
                                return 0.0f;
                            }
                    }
                }
            }

            return 0;
        }

        public static float GetValue(this IAnimationCurve curve, float frame, curve_wrapType preWrap, curve_wrapType postWrap)
        {
            return GetValue(curve.KeyFrames, curve.CurveInterpolationType, frame, preWrap, postWrap);
        }

        public static float GetValue(List<KeyFrame> keys, InterpolationType interpolationType, float frame, curve_wrapType preWrap, curve_wrapType postWrap)
        {
            // 念のため、ソートしておきます。
            KeyFramesUtility.SortByFrame(keys);

            // キーフレームが一つの場合は、コンスタントカーブなので、
            // どのフレームでも同じ値になります。
            if (keys.Count == 1)
            {
                return keys[0].Value;
            }

            // キーフレームが無い場合は、カーブが無いので、0.0fで。
            if (!keys.Any())
            {
                return 0.0f;
            }

            int shiftCount;

            // キーフレームを打ってある範囲のフレームが指定されていない場合に、範囲内のフレームにずらす。
            var targetFrame = calcShiftFrame(frame, keys, preWrap, postWrap, out shiftCount);

            // 指定フレームの値を取得する。
            float value = App.PropertyEdit.AnimationCurveExtensions.calcFrameValue(targetFrame, keys, interpolationType);

            if ((shiftCount < 0 && preWrap == curve_wrapType.relative_repeat) ||
                (shiftCount > 0 && postWrap == curve_wrapType.relative_repeat))
            {
                value += shiftCount * (keys[keys.Count - 1].Value - keys[0].Value);
            }

            return value;
        }

        public static void GetMinMax(List<KeyFrame> keys, InterpolationType interpolationType, float endFrame, curve_wrapType preWrap, curve_wrapType postWrap, out float min, out float max)
        {
            if (keys.Count == 0)
            {
                min = 0;
                max = 0;
            }

            // keys が [0,endFrame] に入っていない場合は正しくない
            min = keys[0].Value;
            max = keys[0].Value;
            for(int i=1; i<keys.Count; i++)
            {
                switch (interpolationType)
                {
                    case InterpolationType.Hermite:
                        float min1;
                        float max1;
                        MathUtility.HermiteMinMax(keys[i - 1], keys[i], out min1, out max1);
                        min = Math.Min(min1, min);
                        max = Math.Max(max1, max);
                        break;
                    case InterpolationType.Linear:
                    case InterpolationType.Step:
                        min = Math.Min(keys[i].Value, min);
                        max = Math.Max(keys[i].Value, max);
                        break;
                }
            }

            float keyRangeMin = min;
            float keyRangeMax = max;

            if (preWrap == curve_wrapType.relative_repeat && 0 < keys[0].Frame)
            {
                int shiftCount;
                var shiftedStartFrame = calcShiftFrame(0, keys, preWrap, postWrap, out shiftCount);
                var endKeyFrame = keys[keys.Count - 1].Frame;

                var diff = keys[keys.Count - 1].Value - keys[0].Value;
                if (shiftCount < 0)
                {
                    max = Math.Max((shiftCount + 1) * diff + keyRangeMax, max);
                    min = Math.Min((shiftCount + 1) * diff + keyRangeMin, min);
                }

                for (int i = 1; i < keys.Count; i++)
                {
                    if (keys[i].Frame < shiftedStartFrame)
                    {
                        continue;
                    }

                    float min1 = 0;
                    float max1 = 0;
                    switch (interpolationType)
                    {
                        case InterpolationType.Hermite:
                            MathUtility.HermiteMinMax(keys[i - 1], keys[i], out min1, out max1, shiftedStartFrame, endKeyFrame);
                            break;
                        case InterpolationType.Linear:
                            MathUtility.LinearMinMax(keys[i - 1], keys[i], out min1, out max1, shiftedStartFrame, endKeyFrame);
                            break;
                        case InterpolationType.Step:
                            MathUtility.StepMinMax(keys[i - 1], keys[i], out min1, out max1, shiftedStartFrame, endKeyFrame);
                            break;
                    }

                    min1 += shiftCount*diff;
                    max1 += shiftCount*diff;
                    min = Math.Min(min1, min);
                    max = Math.Max(max1, max);
                }
            }

            if (postWrap == curve_wrapType.relative_repeat && keys[keys.Count - 1].Frame < endFrame)
            {
                int shiftCount;
                var startFrame = keys[0].Frame;
                var shiftedEndFrame =calcShiftFrame(endFrame, keys, preWrap, postWrap, out shiftCount);

                var diff = keys[keys.Count - 1].Value - keys[0].Value;
                if (shiftCount > 0)
                {
                    max = Math.Max((shiftCount - 1) * diff + keyRangeMax, max);
                    min = Math.Min((shiftCount - 1) * diff + keyRangeMin, min);
                }

                for (int i = 1; i < keys.Count; i++)
                {
                    if (shiftedEndFrame<keys[i-1].Frame)
                    {
                        continue;
                    }

                    float min1 = 0;
                    float max1 = 0;
                    switch (interpolationType)
                    {
                        case InterpolationType.Hermite:
                            MathUtility.HermiteMinMax(keys[i - 1], keys[i], out min1, out max1, startFrame, shiftedEndFrame);
                            break;
                        case InterpolationType.Linear:
                            MathUtility.LinearMinMax(keys[i - 1], keys[i], out min1, out max1, startFrame, shiftedEndFrame);
                            break;
                        case InterpolationType.Step:
                            MathUtility.StepMinMax(keys[i - 1], keys[i], out min1, out max1, startFrame, shiftedEndFrame);
                            break;
                    }

                    min1 += shiftCount * diff;
                    max1 += shiftCount * diff;
                    min = Math.Min(min1, min);
                    max = Math.Max(max1, max);
                }
            }

        }

        public static float GetSlope(List<KeyFrame> keys, InterpolationType interpolationType, float frame, curve_wrapType preWrap, curve_wrapType postWrap)
        {
            // 念のため、ソートしておきます。
            KeyFramesUtility.SortByFrame(keys);

            // キーフレームが一つの場合は、コンスタントカーブなので、
            // どのフレームでも同じ値になります。
            if (keys.Count == 1)
            {
                return keys[0].Value;
            }

            // キーフレームが無い場合は、カーブが無いので、0.0fで。
            if (!keys.Any())
            {
                return 0.0f;
            }

            // post_wrap, pre_wrap が clamp なら水平
            if (frame < keys[0].Frame && preWrap == curve_wrapType.clamp)
            {
                return 0.0f;
            }
            else if (frame > keys[keys.Count - 1].Frame && postWrap == curve_wrapType.clamp)
            {
                return 0.0f;
            }

            int shiftCount;

            // キーフレームを打ってある範囲のフレームが指定されていない場合に、範囲内のフレームにずらす。
            var targetFrame = calcShiftFrame(frame, keys, preWrap, postWrap, out shiftCount);

            // 指定フレームの値を取得する。
            float value = App.PropertyEdit.AnimationCurveExtensions.calcFrameSlope(targetFrame, keys, interpolationType);

            if ((shiftCount < 0 && preWrap == curve_wrapType.mirror) ||
                (shiftCount > 0 && postWrap == curve_wrapType.mirror))
            {
                //value += shiftCount * (keys[keys.Count - 1].Value - keys[0].Value);
                if (shiftCount % 2 != 0)
                {
                    value = -value;
                }
            }

            return value;
        }

        public static void MakeHermiteSmoothSlope(this IAnimationCurve curve, float frame, curve_wrapType preWrap, curve_wrapType postWrap, out float inSlope, out float outSlope)
        {
            MakeHermiteSmoothSlope(curve.KeyFrames, curve.CurveInterpolationType, frame, preWrap, postWrap, out inSlope, out outSlope);
        }

        public static void MakeHermiteSmoothSlope(List<KeyFrame> keys, InterpolationType interpolation, float frame, curve_wrapType preWrap, curve_wrapType postWrap, out float inSlope, out float outSlope)
        {
            inSlope = 0.0f;
            outSlope = 0.0f;
            if (keys.Count < 2)
            {
                // 答えが求まらない・求めづらい
                return;
            }

#if true
            // 近似より精度がよい
            outSlope = inSlope = GetSlope(keys, interpolation, frame, preWrap, postWrap);
#else
            // 近似。本当は微分で求める？
            double delta = 0.0001;
            double inValue = GetValue(keys, interpolation, frame - (float)delta, preWrap, postWrap);
            double outValue = GetValue(keys, interpolation, frame + (float)delta, preWrap, postWrap);

            outSlope = inSlope = (float)(outValue - inValue) / ((frame + (float)delta) - (frame - (float)delta));//(float)((outValue - inValue) / (delta * 2));
#endif
        }

        public static KeyFrame CreateKey(this IAnimationCurve curve, float frame, float value, float inSlope, float outSlope, double snapFrame, double snapValue, bool clampValue,  bool clampFrame, float frameCount)
        {
            value = curve.SnapValue(value, snapValue);
            if (clampValue)
            {
                value = curve.ClampValue(value);
            }

            frame = snapFrame > 0 ? ((float)(Math.Round(frame / snapFrame) * snapFrame)) : frame;
            if (clampFrame)
            {
                frame = Math.Min(Math.Max(frame, 0.0f), frameCount);
            }

            return
                new KeyFrame()
                {
                    Frame = frame,
                    Value = value,
                    InSlope = inSlope,
                    OutSlope = outSlope,
                    Selected = true
                };
        }
    }

    public static class KeyFramesUtility
    {
        public static void SortByFrame(List<KeyFrame> keyFrames)
        {
            for (int i = 0; i < keyFrames.Count; i++)
            {
                keyFrames[i].SubIndex = i;
            }

            keyFrames.Sort(
                (x, y) =>
                    (x.Frame != y.Frame) ? x.Frame.CompareTo(y.Frame) :
                    (x.Index != y.Index) ? x.Index.CompareTo(y.Index) :
                                           x.SubIndex.CompareTo(y.SubIndex)
            );
        }
    }
}
