﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Command;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;

namespace App.PropertyEdit
{
    public partial class CurveViewState : InputControlBase
    {
        public event EventHandler	KeyParamChanged;

        /// <summary>
        /// ディスプレイ座標X軸方向のスクロールをScaleX で割った量。
        /// ディスプレイ座標でもカーブ平面座標でもない。
        /// </summary>
        public double ScrollX{		get{ return scrollX_; }
                                    set{ if (scrollX_ != value){	scrollX_ = value;	view_.Invalidate(); } }	}

        /// <summary>
        /// ディスプレイ座標Y軸方向のスクロールをScaleY で割った量。
        /// ディスプレイ座標でもカーブ平面座標でもない。
        /// </summary>
        public double ScrollY{		get{ return scrollY_; }
                                    set{ if (scrollY_ != value){	scrollY_ = value;	view_.Invalidate(); } }	}

        /// <summary>
        /// ディスプレイ上のX方向の1ピクセルのカーブ平面での量
        /// </summary>
        public double ScaleX{		get{ return scaleX_; }
                                    set{ value = CurveEditorUtility.ClampScaleX(value); if (scaleX_ != value){	scaleX_ = value;	view_.Invalidate(); } }	}

        /// <summary>
        /// ディスプレイ上のY方向の1ピクセルのカーブ平面での量
        /// </summary>
        public double ScaleY{		get{ return scaleY_; }
                                    set{ value = CurveEditorUtility.ClampScaleY(value); if (scaleY_ != value){	scaleY_ = value;	view_.Invalidate(); } }	}

        public bool IsSelection{	get{ return isSelection_; }
                                    private set{ if (isSelection_ != value){ isSelection_ = value; view_.Invalidate(); } }	}

        public double CurrentFrame
        {
            get{ return currentFrame_; }
            set
            {
                value += -0.5f;
                value = Math.Ceiling(value);

                if (currentFrame_ != value)
                {
                    currentFrame_ = value;
                    view_.Invalidate();
                }

                App.AppContext.PreviewCurrentFrame = CurrentFrame;
            }
        }

        public bool Pausing
        {
            get;
            private set;
        }

        public double PauseFrame
        {
            get;
            private set;
        }

        public void SetPause(bool pause, double frame)
        {
            frame = Math.Round(frame);
            bool invalidate = pause != Pausing || (pause && frame != PauseFrame);
            Pausing = pause;
            PauseFrame = frame;
            if (invalidate)
            {
                view_.Invalidate();
            }
        }

        private bool isScaleScroll_				= false;
        private bool isMove_					= false;
        private bool isColorKeyMove_			= false;	// カラーキードラッグで移動
        private bool isVisibleKeyMove_			= false;	// ビジブルキードラッグで移動
        private bool isTexturePatternKeyMove_	= false;	// テクスチャパターンキードラッグで移動

        private bool IsBarKeyMove { get { return isColorKeyMove_ || isVisibleKeyMove_ || isTexturePatternKeyMove_; } }	// バーキーで移動

        private bool isCurrentFrameDragging_ = false;
        private bool IsCurrentFrameDragging
        {
            get
            {
                return isCurrentFrameDragging_;
            }

            set
            {
                isCurrentFrameDragging_ = value;
                App.AppContext.IsEdittingCurrentFrame = isCurrentFrameDragging_;
            }
        }

        private bool IsPausingFrameDragging
        {
            get;
            set;
        }

        public int SelectionX { get {	return Math.Min(DragBegin.BeginX, DragMove.MoveX);	} }
        public int SelectionY { get {	return Math.Min(DragBegin.BeginY, DragMove.MoveY);	} }
        public int SelectionW { get {	return Math.Abs(DragBegin.BeginX - DragMove.MoveX);	} }
        public int SelectionH { get {	return Math.Abs(DragBegin.BeginY - DragMove.MoveY);	} }

        private bool	IsUseFrameSnap{			get { return view_.IsUseFrameSnap;		} }
        private bool	IsUseValueSnap{			get { return view_.IsUseValueSnap;		} }
        private double	FrameSnapFactor{		get { return view_.FrameSnapFactor;		} }
        private double	ValueSnapFactor{		get { return view_.ValueSnapFactor;		} }
        private bool	IsKeyAddMode{			get { return view_.IsKeyAddMode;		} }
        public bool		IsClampFrame{			get { return view_.IsClampFrame;		} }
        public bool		IsClampValue{			get { return view_.IsClampValue;		} }
        private bool	IsClampCurrentFrame{	get { return view_.IsClampCurrentFrame;	} }

        public List<IAnimationCurve>		SelectedCurves { get { return view_.SelectedCurves; } set { view_.SelectedCurves = value; } }
        public IEnumerable<IAnimationCurve>	EditableCurves { get { return view_.EditableCurves; } }
        public List<IAnimationCurve>		VisibledCurves { get { return view_.VisibledCurves; } }
        public List<IAnimationCurve>		MovableCurves {  get { return view_.MovableCurves; } }
        private EditModeType				EditMode {		 get { return view_.EditMode;	} }
        private bool _addOrInsertKeyFrameToEmptyCurve;

 		private readonly CurveView	view_;
        private double		scrollX_		= 0.0f;
        private double		scrollY_		= 0.0f;
        private double		scaleX_			= 1.0f;
        private double		scaleY_			= 1.0f;
        private bool		isSelection_	= false;
        private double		currentFrame_	= 0.0f;

        private double		dragBeginScrollX_;
        private double		dragBeginScrollY_;
        private double		dragBeginScaleX_;
        private double		dragBeginScaleY_;

        // ドラッグ前のキーの値
        private Dictionary<IAnimationCurve, List<KeyFrame>> PreservedKeyFrames =
            new Dictionary<IAnimationCurve, List<KeyFrame>>();

        public void PreserveKeyFrames(IAnimationCurve curve)
        {
            if (PreservedKeyFrames.ContainsKey(curve))
            {
                return;
            }

            PreservedKeyFrames.Add(curve, curve.KeyFrames.Select(x => x.Clone()).ToList());
        }

        public class AutoSplineSlopeInfo
        {
            public bool dragged { get; set; }
            public bool autoSlope { get; set; }
            public float inSlope { get; set; }
            public float outSlope { get; set; }
            public bool inSplineSlope { get; set; }
            public bool outSplineSlope { get; set; }
        }

        // スプラインスロープの自動計算を行うか
        private Dictionary<IAnimationCurve, AutoSplineSlopeInfo[]> AutoSplineCurveIndices =
            new Dictionary<IAnimationCurve, AutoSplineSlopeInfo[]>();

        public void SetAutoSplineCurveIndices(bool add = false)
        {
            Debug.Assert(AutoSplineCurveIndices.Count == 0);

            if (!view_.IsAutoSplineSlope)
            {
                return;
            }

            foreach (var group in DraggableObjects.Select(x => x.Interface).OfType<KeyFrameHolder>().GroupBy(x => x.Curve))
            {
                var curve = group.Key;
                var dragged = new bool[curve.KeyFrames.Count];
                bool anyAutoSlopeSpline = false;
                foreach (var holder in group)
                {
                    var keyFrame = holder.KeyFrame;
                    if (holder.AutoSlopeSpline)
                    {
                        anyAutoSlopeSpline = true;
                        dragged[keyFrame.Index] = true;
                    }
                }

                if (anyAutoSlopeSpline)
                {
                    curve.SortByFrame();
                    var info = Enumerable.Range(0, curve.KeyFrames.Count).Select(
                        x => new AutoSplineSlopeInfo { dragged = dragged[x] }).ToArray();
                    var addIndex = add ? curve.KeyFrames.FindIndex(x => dragged[x.Index]): -1;
                    for (int i=0;i<curve.KeyFrames.Count;i++)
                    {
                        int index = curve.KeyFrames[i].Index;
                        info[index].inSlope = curve.KeyFrames[i].InSlope;
                        info[index].outSlope = curve.KeyFrames[i].OutSlope;

                        bool inSpline;
                        bool outSpline;
                        IsSplineSlope(curve.KeyFrames, i, addIndex, out inSpline, out outSpline);
                        info[index].inSplineSlope = inSpline;
                        info[index].outSplineSlope = outSpline;
                    }
                    AutoSplineCurveIndices[curve] = info;
                }
            }

            //UpdateAutoSplineCurveIndices();
        }

        public void IsSplineSlope(List<KeyFrame> keys, int index, int addedIndex, out bool inSpline, out bool outSpline)
        {
            if (index == addedIndex)
            {
                inSpline = outSpline = true;
                return;
            }

            int previous = index - 1;
            if (previous == addedIndex)
            {
                previous = (addedIndex == 0) ? index : previous - 1;
            }
            previous = Math.Max(0, previous);

            int next = index + 1;
            if (next == addedIndex)
            {
                next = (addedIndex == keys.Count - 1) ? index : next + 1;
            }
            next = Math.Min(keys.Count - 1, next);
            float dx = keys[next].Frame - keys[previous].Frame;
            float dy = keys[next].Value - keys[previous].Value;
            if (dx < 0.0001)
            {
                inSpline = slopeNearlyEqual(keys[index].InSlope, 0);
                outSpline = slopeNearlyEqual(keys[index].OutSlope, 0);
            }
            else
            {
                float slope = dy/dx;
                inSpline = slopeNearlyEqual(keys[index].InSlope, slope);
                outSpline = slopeNearlyEqual(keys[index].OutSlope, slope);
            }
        }

        private bool slopeNearlyEqual(float a, float b)
        {
            // 絶対誤差または相対誤差が1e-5 以下
            return Math.Abs(a - b) <= (1e-5)*Math.Max(1.0f, Math.Max(Math.Abs(a), Math.Abs(b)));
        }

        public void UpdateAutoSplineCurveIndices()
        {
            // 傾きを計算する
            foreach (var item in AutoSplineCurveIndices)
            {
                var curve = item.Key;
                var info = item.Value;
                for (int i = 0; i < curve.KeyFrames.Count; i++)
                {
                    int index = curve.KeyFrames[i].Index;
                    if (!info[index].inSplineSlope && !info[index].outSplineSlope)
                    {
                        continue;
                    }

                    var slope = GetSplineSlope(curve.KeyFrames, i);

                    if (info[index].inSplineSlope)
                    {
                        curve.KeyFrames[i].InSlope = slope;
                    }

                    if (info[index].outSplineSlope)
                    {
                        curve.KeyFrames[i].OutSlope = slope;
                    }
                }
            }
        }

        public float GetSplineSlope(List<KeyFrame> keyFrames, int i)
        {
            var current = keyFrames[i];
            var previous = keyFrames[i - 1 >= 0 ? i - 1 : i];
            var next = keyFrames[i + 1 < keyFrames.Count ? i + 1 : i];

            if (next.Frame - previous.Frame < 0.0001)
            {
                return 0;
            }
            return (next.Value - previous.Value) / (next.Frame - previous.Frame);
        }
        public CurveViewState(CurveView view) :
            base(view)
        {
            view_ = view;

            //MouseDoubleClick	+= OnMouseDoubleClick;
            MouseDown			+= OnMouseDown;
            MouseMove			+= OnMouseMove;
            MouseDrag			+= OnMouseDrag;
            MouseUp				+= OnMouseUp;
            MouseWheel			+= OnMouseWheel;
            MouseLeave			+= OnMouseLeave;
            LostFocus += OnLostFocus;
        }

        private void EditValueOfKey(int mouseX, int mouseY)
        {
            var commandSet = new EditCommandSet();
            {
                var colorSet = GetColorKey(mouseX, mouseY);
                var visibleSet = GetVisibleKey(mouseX, mouseY);
                var texturePatternSet = GetTexturePatternKey(mouseX, mouseY);

                // カラーキーがつかめるかどうか
                if (colorSet != null)
                {
                    ColorPickerKeyMove(colorSet, true);
                }
                // ビジブルキーがつかめるかどうか
                else
                    if (visibleSet != null)
                    {
                        var curve = visibleSet.Curve;

                        curve.SetCurvesSequentialIndex();

                        var keyFrame = curve.KeyFrames.FirstOrDefault(x => x.Frame == visibleSet.Frame);
                        if (keyFrame != null)
                        {
                            commandSet.Add(CreateKeyFrameValueEditCommand(curve, keyFrame, (keyFrame.Value == 1.0f) ? 0.0f : 1.0f));
                        }
                    }
                    // テクスチャパターンキーがつかめるかどうか
                    else
                        if (texturePatternSet != null)
                        {
                            var curve = texturePatternSet.Curve;

                            curve.SetCurvesSequentialIndex();

                            var keyFrame = curve.KeyFrames.FirstOrDefault(x => x.Frame == texturePatternSet.Frame);
                            if (keyFrame != null)
                            {
                                Debug.Assert(view_.TargetGroup.Active is IHasTexturePatternAnimation);

                                var texPatternAnim = view_.TargetGroup.Active as IHasTexturePatternAnimation;

                                using (var dialog = new TexturePatternTextureSelectDialog(texPatternAnim))
                                {
                                    dialog.SelectedIndex = (int)keyFrame.Value;

                                    if (dialog.ShowDialog() == DialogResult.OK)
                                    {
                                        commandSet.Add(CreateKeyFrameValueEditCommand(curve, keyFrame, dialog.SelectedIndex));
                                    }
                                }
                            }
                        }
            }

            if (commandSet.CommandCount > 0)
            {
                commandSet.OnPostEdit += (ss, ee) => Viewer.HioUtility.FlushCurves(view_.TargetGroup, view_.TargetGuiObjectID);

                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private ICommand CreateKeyFrameValueEditCommand(IAnimationCurve curve, KeyFrame keyFrame, float value)
        {
            var newKeyFrames = ObjectUtility.Clone(curve.KeyFrames);
            var newKeyFrame = newKeyFrames[keyFrame.Index];

            newKeyFrame.Value = value;

            return
                AnimationCurveEditCommand.Create(
                    view_.TargetGroup,
                    view_.TargetGuiObjectID,
                    curve,
                    newKeyFrames
                );
        }

        private void OnMouseDown(MouseEventArgs e)
        {
            bool isColorKeyFrameAddable				= IsInHRulerColorBar(e.X, e.Y);
            bool isVisibleKeyFrameAddable           = IsInHRulerVisibleBar(e.X, e.Y);
            bool isTexturePatternKeyFrameAddable	= IsInHRulerTexturePatternBar(e.X, e.Y);
            bool isCurrentFrameDragging				= (isColorKeyFrameAddable == false) &&
                                                      (isVisibleKeyFrameAddable == false) &&
                                                      (isTexturePatternKeyFrameAddable == false) &&
                                                      IsInHRuler(e.X, e.Y);
            var isAllColorCurveEditable     = view_.SelectedColorCurves.Any()           && view_.SelectedColorCurves.All(x => x.IsEditable);
            var isAllVisibleCurveEditable   = view_.SelectedVisibleCurves.Any()         && view_.SelectedVisibleCurves.All(x => x.IsEditable);
            var isAllTexPatCurveEditable    = view_.SelectedTexturePatternCurves.Any()  && view_.SelectedTexturePatternCurves.All(x => x.IsEditable);
            _addOrInsertKeyFrameToEmptyCurve = false;

            // カラー追加
            if (isColorKeyFrameAddable && (e.Button == MouseButtons.Left) && isAllColorCurveEditable)
            {
                IsDragging = false;
                var frame = (float)view_.GetXInCurvePlane(e.X);
                ColorPickerKeyAdd(frame);
            }
            // ビジブル追加
            else if (isVisibleKeyFrameAddable && (e.Button == MouseButtons.Left) && isAllVisibleCurveEditable)
            {
                var frame = (float)view_.GetXInCurvePlane(e.X);

                Debug.Assert(view_.SelectedVisibleCurves.Any());
                view_.AddKey(view_.SelectedVisibleCurves.Where(x => x.IsEditable).ToList(), frame, 1.0f);
            }
            // テクスチャパターン追加
            else if (isTexturePatternKeyFrameAddable && (e.Button == MouseButtons.Left) && isAllTexPatCurveEditable)
            {
                if (EditableCurves.Any())
                {
                    var frame = (float)view_.GetXInCurvePlane(e.X);

                    var texPatternAnim = view_.TargetGroup.Active as IHasTexturePatternAnimation;

                    using (var dialog = new TexturePatternTextureSelectDialog(texPatternAnim))
                    {
                        dialog.SelectedIndex = (int)view_.GetXInCurvePlane(e.X);

                        if (dialog.ShowDialog() == DialogResult.OK)
                        {
                            Debug.Assert(view_.SelectedTexturePatternCurves.Any());
                            view_.AddKey(view_.SelectedTexturePatternCurves, frame, dialog.SelectedIndex);
                        }

                        IsDragging = false;
                    }
                }
            }
            // カーブエディタ上のクリック
            else
            {
                var colorSet = GetColorKey(e.X, e.Y);
                if (colorSet != null)
                {
                    // ダブルクリック中にカラーピッカーが閉じられるのを防ぐ
                    if (colorPickerAdapter != null && ColorPickerDialog.TimerStartedAdapter == colorPickerAdapter)
                    {
                        ColorPickerDialog.StopTimer();
                        ColorPickerKeyMove(colorSet, false);
                    }

                    BeginColorKeyMove(colorSet);
                    isColorKeyMove_ = true;

                    return;
                }

                var visibleSet = GetVisibleKey(e.X, e.Y);
                if (visibleSet != null)
                {
                    BeginVisibleKeyMove(visibleSet);
                    isVisibleKeyMove_ = true;

                    return;
                }

                var texturePatternSet = GetTexturePatternKey(e.X, e.Y);
                if (texturePatternSet != null)
                {
                    BeginTexturePatternKeyMove(texturePatternSet);
                    isTexturePatternKeyMove_ = true;

                    return;
                }

                {
                    // スクロールorスケール操作か？
                    var isScrollScale =
                        ((DragBegin.ModifierKeys & Keys.Alt) != 0) &&
                        ((e.Button == MouseButtons.Middle) || (e.Button == MouseButtons.Right));

                    if (IsKeyAddMode)
                    {
                        if (isCurrentFrameDragging)
                        {
                            if (Pausing)
                            {
                                IsPausingFrameDragging = true;
                            }
                            else
                            {
                                IsCurrentFrameDragging = true;
                            }
                            SetFrame(e.X, IsClampCurrentFrame);
                        }
                        else
                        if (isScrollScale)						// スクロール・スケール
                        {
                            isScaleScroll_ = true;
                            BeginScaleScroll(e);
                        }
                        else
                        if (e.Button == MouseButtons.Left)		// 選択
                        {
                            IsSelection = true;
                        }
                        else
                        if (e.Button == MouseButtons.Middle)
                        {
                            isMove_ = true;
                            BeginMove(e);
                        }
                    }
                    else
                    {
                        if (isCurrentFrameDragging)
                        {
                            if (Pausing)
                            {
                                IsPausingFrameDragging = true;
                            }
                            else
                            {
                                IsCurrentFrameDragging = true;
                            }
                            SetFrame(e.X, IsClampCurrentFrame);
                        }
                        else
                        if (isScrollScale)						// スクロール・スケール
                        {
                            isScaleScroll_ = true;
                            BeginScaleScroll(e);
                        }
                        else
                        if (e.Button == MouseButtons.Left)
                        {
                            // 選択中のキーフレーム・ハンドルを左ドラッグの場合は移動
                            if ((EditMode == EditModeType.Multi || EditMode == EditModeType.Near)
                                && IsSelectionClicked(e, 0, 0))
                            {
                                isMove_ = true;
                                BeginMove(e);
                            }
                            else // 選択
                            {
                                IsSelection = true;
                                MoveSelection(e, 0, 0);
                            }
                        }
                        else
                        // 移動
                        if (e.Button == MouseButtons.Middle)
                        {
                            isMove_ = true;
                            BeginMove(e);
                        }
                    }
                }
            }
        }

        // カラーキーがつかめるか？
        // つかめた場合はそのキーが影響するカラーセット
        // つかめない場合は null
        public CurveView.ColorSet GetColorKey(int x, int y)
        {
            bool hdr = view_.IsHDR();
            var keyTop = view_.Painter.HRulerBgTop - CurveEditorConst.ColorBarSize + CurveViewPainter.ColorKeyTopOffset;
            var keyBottom = keyTop + CurveViewPainter.ColorKeyImageHeight;

            if ((y >= keyTop   ) &&
                (y <= keyBottom))
            {
//				var curves = view_.SelectedColorCurves;
                var curves = view_.SelectedColorCurves.Where(c => c.IsEditable).ToList();

                var isAnyColorSet	= view_.IsAnyColorSet(curves);
                if (isAnyColorSet == false)
                {
                    var colorCurveGroup	= view_.Painter.MakeColorCurveGroup(true);
                    foreach(var colorSet in view_.MakeSameFrameColorSet(curves, view_.TargetGroup.Active, colorCurveGroup))
                    {
                        var keyX     = view_.Painter.MakeCurveViewScreenPosX(colorSet.Frame);
                        var keyLeft  = keyX - Math.Abs(CurveViewPainter.ColorKeyLeftOffset);
                        var keyRight = keyX + Math.Abs(CurveViewPainter.ColorKeyLeftOffset);

                        if ((x >= keyLeft ) &&
                            (x <= keyRight))
                        {
                            return colorSet;
                        }
                    }
                }
            }

            return null;
        }

        // ビジブルキーがつかめるか？
        private CurveView.VisibleSet GetVisibleKey(int x, int y)
        {
            const int notHitHeight = 2;
            var keyTop = view_.Painter.HRulerBgTop + CurveViewPainter.ColorKeyTopOffset - CurveEditorConst.ColorBarSize + notHitHeight;
            var keyBottom = keyTop + CurveViewPainter.ColorKeyImageHeight - notHitHeight;

            if ((y >= keyTop   ) &&
                (y <= keyBottom))
            {
//				var curves = view_.SelectedVisibleCurves;
                var curves = view_.SelectedVisibleCurves.Where(c => c.IsEditable).ToList();
                if (curves.Count == 1)
                {
                    var curve = curves.First();

                    foreach(var keyFrame in curve.KeyFrames)
                    {
                        var keyX     = view_.Painter.MakeCurveViewScreenPosX(keyFrame.Frame);
                        var keyLeft  = keyX - Math.Abs(CurveViewPainter.ColorKeyLeftOffset);
                        var keyRight = keyX + Math.Abs(CurveViewPainter.ColorKeyLeftOffset);

                        if ((x >= keyLeft ) &&
                            (x <= keyRight))
                        {
                            return
                                new CurveView.VisibleSet
                                {
                                    Frame		= keyFrame.Frame,
                                    IsVisible	= keyFrame.Value == 1.0f,
                                    Curve		= curve
                                };
                        }
                    }
                }
            }

            return null;
        }

        // テクスチャパターンキーがつかめるか？
        private CurveView.TexturePatternSet GetTexturePatternKey(int x, int y)
        {
            var keyTop = view_.Painter.HRulerBgTop + CurveViewPainter.ColorKeyTopOffset - CurveEditorConst.ColorBarSize;
            var keyBottom = keyTop + CurveViewPainter.ColorKeyImageHeight;

            if ((y >= keyTop   ) &&
                (y <= keyBottom))
            {
//				var curves = view_.SelectedTexturePatternCurves;
                var curves = view_.SelectedTexturePatternCurves.Where(c => c.IsEditable).ToList();
                if (curves.Count == 1)
                {
                    var curve = curves.First();

                    foreach(var keyFrame in curve.KeyFrames)
                    {
                        var keyX     = view_.Painter.MakeCurveViewScreenPosX(keyFrame.Frame);
                        var keyLeft  = keyX - Math.Abs(CurveViewPainter.ColorKeyLeftOffset);
                        var keyRight = keyX + Math.Abs(CurveViewPainter.ColorKeyLeftOffset);

                        if ((x >= keyLeft ) &&
                            (x <= keyRight))
                        {
                            return
                                new CurveView.TexturePatternSet
                                {
                                    Frame				= keyFrame.Frame,
                                    TexturePatternIndex	= (int)keyFrame.Value,
                                    Curve				= curve
                                };
                        }
                    }
                }
            }

            return null;
        }

        private bool IsInHRuler(int x, int y)
        {
            return
                (x >= view_.Painter.VRulerBgRight) && (x <= view_.Painter.HRulerBgRight ) &&
                (y >= view_.Painter.HRulerBgTop  ) && (y <= view_.Painter.HRulerBgBottom);
        }

        private bool IsInHRulerEditBar(int x, int y)
        {
            return
                (x >= view_.Painter.VRulerBgRight) && (x <= view_.Painter.HRulerBgRight ) &&
                (y <= view_.Painter.HRulerBgTop  ) && (y >= view_.Painter.HRulerBgTop - CurveEditorConst.ColorBarSize);
        }

        public bool IsInHRulerColorBar(int x, int y)
        {
            return IsInHRulerEditBar(x, y) && view_.IsSelectedColorCurve;
        }

        private bool IsInHRulerVisibleBar(int x, int y)
        {
            return IsInHRulerEditBar(x, y) && view_.IsSelectedVisibleCurve;
        }

        private bool IsInHRulerTexturePatternBar(int x, int y)
        {
            return IsInHRulerEditBar(x, y) && view_.IsSelectedTexturePatternCurve;
        }

        private void OnMouseMove(MouseEventArgs e)
        {
            if (DragMove.IsMoved)
            {
                UpdateMouseCursor(e.X, e.Y);

                ShowEditBarCurrentImage(e.X, e.Y);
            }
        }

        private void OnMouseDrag(MouseEventArgs e, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY)
        {
            view_.Invalidate();

            if (IsBarKeyMove)
            {
                MoveMove(e, moveDiffX, moveDiffY);
            }
            else
            if (IsKeyAddMode)
            {
                if (IsCurrentFrameDragging || IsPausingFrameDragging)
                {
                    SetFrame(e.X, IsClampCurrentFrame);
                }
                else
                if (isScaleScroll_)			// スクロール・スケール
                {
                    MoveScaleScroll(e, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY);
                }
                else if (isMove_)
                {
                    MoveMove(e, moveDiffX, moveDiffY);
                }
            }
            else
            {
                if (IsCurrentFrameDragging || IsPausingFrameDragging)
                {
                    SetFrame(e.X, IsClampCurrentFrame);
                }
                else
                if (isScaleScroll_)			// スクロール・スケール
                {
                    MoveScaleScroll(e, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY);
                }
                // 選択
                else
                if (IsSelection)
                {
                    MoveSelection(e, moveDiffX, moveDiffY);
                }
                else
                // 移動
                if (isMove_)
                {
                    MoveMove(e, moveDiffX, moveDiffY);
                }
            }
        }

        private void OnMouseUp(MouseEventArgs e)
        {
            view_.Cursor = Cursors.Default;
            view_.Invalidate();

            if (IsBarKeyMove)
            {
                isColorKeyMove_ = false;
                isVisibleKeyMove_ = false;
                isTexturePatternKeyMove_ = false;

                if (IsDragged)
                {
                    EndMove();
                    if (colorPickerAdapter != null &&
                        (colorPickerAdapter == ColorPickerDialog.Adapter || colorPickerAdapter == ColorPickerDialog.TimerStartedAdapter))
                    {
                        EditValueOfKey(e.X, e.Y);
                    }
                }
                else
                {
                    EditValueOfKey(DragBegin.BeginX, DragBegin.BeginY);
                }
            }
            else
            if (IsKeyAddMode)
            {
                if (IsCurrentFrameDragging)
                {
                    IsCurrentFrameDragging = false;
                }
                else if (IsPausingFrameDragging)
                {
                    IsPausingFrameDragging = false;
                }
                else if (isScaleScroll_) // スクロール・スケール
                {
                    isScaleScroll_ = false;
                }
                else if (IsSelection) // 選択
                {
                    IsSelection = false;
                    EndSelection();
                }
                else if (isMove_) // 編集
                {
                    isMove_ = false;

                    if (IsDragged)
                    {
                        EndMove();
                    }
                }
            }
            else
            {
                if (IsCurrentFrameDragging)
                {
                    IsCurrentFrameDragging = false;
                }
                else if (IsPausingFrameDragging)
                {
                    IsPausingFrameDragging = false;
                }
                else if (isScaleScroll_)	// スクロール・スケール
                {
                    isScaleScroll_ = false;
                }
                else if (IsSelection) // 選択
                {
                    IsSelection = false;
                    EndSelection();
                }
                else if (isMove_) // 編集
                {
                    isMove_ = false;

                    if (IsDragged)
                    {
                        EndMove();
                    }
                }
            }

            PreservedKeyFrames.Clear();
            AutoSplineCurveIndices.Clear();

            if (e.Button == MouseButtons.Right)
            {
                view_.ShowContextMenu(new ContextMenuPopupEventArgs(e.Location, false));
            }
        }

        private void OnLostFocus(EventArgs e)
        {

            if (colorPickerAdapter != null &&
                ColorPickerDialog.Adapter == colorPickerAdapter)
            {
                ColorPickerDialog.EndConnection(colorPickerAdapter);
            }

            view_.Cursor = Cursors.Default;
            view_.Invalidate();

            if (IsBarKeyMove)
            {
                isColorKeyMove_ = false;
                isVisibleKeyMove_ = false;
                isTexturePatternKeyMove_ = false;

                if (IsDragged)
                {
                    CancelMove();
                }
            }
            else
                if (IsKeyAddMode)
                {
                    if (IsCurrentFrameDragging)
                    {
                        IsCurrentFrameDragging = false;
                    }
                    else if (IsPausingFrameDragging)
                    {
                        IsPausingFrameDragging = false;
                    }
                    else if (isScaleScroll_) // スクロール・スケール
                    {
                        isScaleScroll_ = false;
                    }
                    else if (IsSelection) // 選択
                    {
                        IsSelection = false;
                        EndSelection();
                    }
                    else if (isMove_)
                    {
                        isMove_ = false;

                        if (IsDragged)
                        {
                            CancelMove();
                        }
                    }
                }
                else
                {
                    if (IsCurrentFrameDragging)
                    {
                        IsCurrentFrameDragging = false;
                    }
                    else if (IsPausingFrameDragging)
                    {
                        IsPausingFrameDragging = false;
                    }
                    else if (isScaleScroll_) // スクロール・スケール
                    {
                        isScaleScroll_ = false;
                    }
                    else if (IsSelection) // 選択
                    {
                        IsSelection = false;
                        EndSelection();
                    }
                    else if (isMove_) // 編集
                    {
                        isMove_ = false;

                        if (IsDragged)
                        {
                            CancelMove();
                        }
                    }
                }

            PreservedKeyFrames.Clear();
            AutoSplineCurveIndices.Clear();
        }

        private void OnMouseWheel(MouseEventArgs e)
        {
            int    centerScreenX = view_.CurveBgWidth  / 2;
            int    centerScreenY = view_.CurveBgHeight / 2;
            double beforeCurveX = view_.GetXInCurvePlane(centerScreenX);
            double beforeCurveY = view_.GetYInCurvePlane(centerScreenY);

            // マウスホイールの値を適当な値で制限する
            const int DeltaMax = 300;
            var delta = Math.Max(Math.Min(-e.Delta, DeltaMax), -DeltaMax);
            double moveDiffX = delta * CurveEditorConst.WheelSensitivity;
            double moveDiffY = delta * CurveEditorConst.WheelSensitivity;
            double speedX = ScaleX * CurveEditorConst.DragSensitivity;
            double speedY = ScaleY * CurveEditorConst.DragSensitivity;

            double newScaleX = CurveEditorUtility.ClampScaleX(ScaleX + moveDiffX * speedX);
            double newScaleY = CurveEditorUtility.ClampScaleY(ScaleY + moveDiffY * speedY);

            ScaleX = newScaleX;
            ScaleY = newScaleY;

            double moveBeforeScreenX;
            double moveBeforeScreenY;
            MakeCurveViewScreenPos((float)beforeCurveX, (float)beforeCurveY, out moveBeforeScreenX, out moveBeforeScreenY);

            // マウス位置をスケールの中心にする
            ScrollX += (moveBeforeScreenX - centerScreenX) / ScaleX;
            ScrollY += (moveBeforeScreenY - centerScreenY) / ScaleY;
        }

        private void OnMouseLeave(EventArgs e)
        {
            if (editBarCurrentImage_ != null)
            {
                editBarCurrentImage_.Hide();
            }
        }

        private void SetFrame(int mouseX, bool isClamp)
        {
            var frame = view_.GetXInCurvePlane(mouseX);

            if (isClamp)
            {
                frame = Math.Min(view_.FrameCount, Math.Max(frame, 0));
            }

            if (Pausing)
            {
                var iPause = view_.TargetGroup.Active as IPause;
                if (iPause != null)
                {
                    iPause.SetPause(true, frame);
                    App.AppContext.OnPauseChanged(false);
                }
            }
            else
            {
                CurrentFrame = frame;
            }
        }

        private void BeginScaleScroll(MouseEventArgs e)
        {
            dragBeginScrollX_	= scrollX_;
            dragBeginScrollY_	= scrollY_;
            dragBeginScaleX_	= scaleX_;
            dragBeginScaleY_	= scaleY_;

            // スクロール
            if (((DragBegin.ModifierKeys & Keys.Alt) != 0) && (e.Button == MouseButtons.Middle))
            {
                view_.Cursor	= ResCursors.ViewScroll;
            }

            // スケール
            else if ((DragBegin.ModifierKeys == Keys.Alt) && (e.Button == MouseButtons.Right))
            {
                view_.Cursor	= ResCursors.Magnify;
            }
        }

        private void BeginColorKeyMove(CurveView.ColorSet colorSet)
        {
            MovableCurves.ForEach(x => x.SetCurvesSequentialIndex());

            foreach(var curve in colorSet.Curves)
            {
                var keyFrame = curve.KeyFrames.FirstOrDefault(x => x.Frame == colorSet.Frame);

                if (keyFrame != null)
                {
                    PreserveKeyFrames(curve);
                    AddDragObject(
                        new KeyFrameHolder(keyFrame, this, curve, KeyFrameHolder.NearSelectType.Key, EditModeType.Multi)
                        {
                            EnableValueEdit = false
                        }
                    );
                }
            }

            SetAutoSplineCurveIndices();
        }

        private void BeginVisibleKeyMove(CurveView.VisibleSet visibleSet)
        {
            MovableCurves.ForEach(x => x.SetCurvesSequentialIndex());

            var keyFrame = visibleSet.Curve.KeyFrames.FirstOrDefault(x => x.Frame == visibleSet.Frame);
            if (keyFrame != null)
            {
                PreserveKeyFrames(visibleSet.Curve);
                AddDragObject(
                    new KeyFrameHolder(keyFrame, this, visibleSet.Curve, KeyFrameHolder.NearSelectType.Key, EditModeType.Multi)
                    {
                        EnableValueEdit = false
                    }
                );
            }

            SetAutoSplineCurveIndices();
        }

        private void BeginTexturePatternKeyMove(CurveView.TexturePatternSet texturePatternSet)
        {
            MovableCurves.ForEach(x => x.SetCurvesSequentialIndex());

            var keyFrame = texturePatternSet.Curve.KeyFrames.FirstOrDefault(x => x.Frame == texturePatternSet.Frame);
            if (keyFrame != null)
            {
                PreserveKeyFrames(texturePatternSet.Curve);
                AddDragObject(
                    new KeyFrameHolder(keyFrame, this, texturePatternSet.Curve, KeyFrameHolder.NearSelectType.Key, EditModeType.Multi)
                    {
                        EnableValueEdit = false
                    }
                );
            }

            SetAutoSplineCurveIndices();
        }

        private void BeginMove(MouseEventArgs e)
        {
            switch(EditMode)
            {
                case EditModeType.Multi:	BeginMove_Multi(e);	break;
                case EditModeType.Near:		BeginMove_Near(e);	break;
                case EditModeType.Scale:	BeginMove_Scale(e);	break;
                case EditModeType.Add: BeginMove_Add(e); break;
                case EditModeType.Insert: BeginMove_Insert(e); break;
                default:
                    Debug.Assert(false);
                    break;
            }
        }

        private void BeginMove_Multi(MouseEventArgs e)
        {
            foreach(var curve in MovableCurves)
            {
                curve.SetCurvesSequentialIndex();

                foreach(var key in curve.KeyFrames.Where(x => x.AnySelected))
                {
                    PreserveKeyFrames(curve);
                    AddDragObject(new KeyFrameHolder(key, this, curve, KeyFrameHolder.NearSelectType.NoNearSelect, EditMode));
                }
            }

            SetAutoSplineCurveIndices();
        }

        private void BeginMove_Near(MouseEventArgs e)
        {
            KeyFrame						nearKey				= null;
            IAnimationCurve					nearKeyVisibleCurve	= null;
            var	nearKeySelectType	= KeyFrameHolder.NearSelectType.NoNearSelect;
            {
                var minDistanceSq = double.MaxValue;
                var mouseX = (double)e.X;
                var mouseY = (double)e.Y;

                const double minimumNear = 16.0 * 16.0;			// 最低限の距離の二乗

                foreach(var curve in MovableCurves)
                {
                    curve.SetCurvesSequentialIndex();

                    // キーフレーム
                    {
                        foreach(KeyFrame key in curve.KeyFrames)
                        {
                            double keyX, keyY;
                            MakeCurveViewScreenPos(
                                key.Frame,
                                key.Value,
                                out keyX,
                                out keyY
                            );

                            // 近い距離のを保存する
                            {
                                // キー
                                {
                                    double distanceSq = (mouseX - keyX) * (mouseX - keyX) + (mouseY - keyY) * (mouseY - keyY);
                                    if ((distanceSq < minimumNear  ) &&
                                        (distanceSq < minDistanceSq))
                                    {
                                        if (distanceSq < minDistanceSq)
                                        {
                                            minDistanceSq = distanceSq;

                                            nearKey				= key;
                                            nearKeyVisibleCurve	= curve;
                                            nearKeySelectType	= KeyFrameHolder.NearSelectType.Key;
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // ハンドル
                    if(curve.CurveInterpolationType == InterpolationType.Hermite){

                        List<KeyFrame> keys = curve.KeyFrames;

                        for (int i = 0;i < keys.Count;++ i)
                        {
                            //var left    = (i > 0) ? keys[i - 1] : dummyKeyFrame;
                            var current = keys[i + 0];
                            //var right   = (i < keys.Count() - 1) ? keys[i + 1] : dummyKeyFrame;

                            double keyX, keyY;
                            MakeCurveViewScreenPos(
                                current.Frame,
                                current.Value,
                                out keyX,
                                out keyY
                            );

                            // アウトスロープ
                            if (current.AnySelected)
                            {
                                double handleX, handleY;
                                MakeCurveViewScreenHandleOffset(
                                    current.OutSlope,
                                    CurveEditorConst.HandleLineLengthScale,
                                    out handleX,
                                    out handleY
                                );

                                double x = keyX + handleX;
                                double y = keyY + handleY;

                                {
                                    double distanceSq = (mouseX - x) * (mouseX - x) + (mouseY - y) * (mouseY - y);
                                    if ((distanceSq < minimumNear  ) &&
                                        (distanceSq < minDistanceSq))
                                    {
                                        if (distanceSq < minDistanceSq)
                                        {
                                            minDistanceSq = distanceSq;

                                            nearKey				= current;
                                            nearKeyVisibleCurve	= curve;
                                            nearKeySelectType	= KeyFrameHolder.NearSelectType.OutHandle;
                                        }
                                    }
                                }
                            }

                            // インスロープ
                            if (current.AnySelected)
                            {
                                double handleX, handleY;
                                MakeCurveViewScreenHandleOffset(
                                    current.InSlope,
                                    CurveEditorConst.HandleLineLengthScale,
                                    out handleX,
                                    out handleY
                                );

                                double x = keyX - handleX;
                                double y = keyY - handleY;

                                {
                                    double distanceSq = (mouseX - x) * (mouseX - x) + (mouseY - y) * (mouseY - y);
                                    if ((distanceSq < minimumNear  ) &&
                                        (distanceSq < minDistanceSq))
                                    {
                                        if (distanceSq < minDistanceSq)
                                        {
                                            minDistanceSq = distanceSq;

                                            nearKey				= current;
                                            nearKeyVisibleCurve	= curve;
                                            nearKeySelectType	= KeyFrameHolder.NearSelectType.InHandle;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }

            if ((nearKey != null) && (nearKeyVisibleCurve != null))
            {
                PreserveKeyFrames(nearKeyVisibleCurve);
                AddDragObject(new KeyFrameHolder(nearKey, this, nearKeyVisibleCurve, nearKeySelectType, EditMode));
                SetAutoSplineCurveIndices();
            }
        }

        private void BeginMove_Add(MouseEventArgs e)
        {
            if (view_.ParentPanel.CheckEditRestriction() == false)
            {
                view_.Invalidate();
                return;
            }

            var inSlope = 0.0f;
            var outSlope = 0.0f;
            var frameSnapFactor = FrameSnapFactor;
            var valueSnapFactor = ValueSnapFactor;
            var mouseFrame = (float)view_.GetXInCurvePlane(e.X);

            // キースナップ有効時には倍数スナップを使わない。
            if (view_.IsSnapToKey)
            {
                frameSnapFactor = 0;
                valueSnapFactor = 0;
            }

            if (frameSnapFactor > 0)
            {
                mouseFrame = (float)(Math.Round(mouseFrame / frameSnapFactor, MidpointRounding.AwayFromZero) * frameSnapFactor);
            }

            var mouseValue = (float)view_.GetYInCurvePlane(e.Y);

            if (EditableCurves.Any())
            {
                foreach (var curve in MovableCurves)
                {
                    curve.KeyFrames.ForEach(x => x.Selected = false);
                }

                foreach (var curve in EditableCurves)
                {
                    _addOrInsertKeyFrameToEmptyCurve |= (curve.KeyFrames.Count == 0);
                    curve.SetCurvesSequentialIndex();
                    PreserveKeyFrames(curve);
                    var key = curve.CreateKey(mouseFrame, mouseValue, inSlope, outSlope, frameSnapFactor, valueSnapFactor, IsClampValue, IsClampFrame, view_.ParentPanel.FrameCount);
                    AddDragObject(new KeyFrameHolder(key, this, curve, KeyFrameHolder.NearSelectType.Key, EditMode));
                }

                SetAutoSplineCurveIndices(true);
                UpdateAutoSplineCurveIndices();
                foreach (var holder in DraggableObjects.Select(x => x.Interface).OfType<KeyFrameHolder>())
                {
                    holder.SetCurveIndex();
                }

                BeginInsertOrAdd();

                view_.Invalidate();
            }
        }

        private void BeginMove_Insert(MouseEventArgs e)
        {
            if (view_.ParentPanel.CheckEditRestriction() == false)
            {
                view_.Invalidate();
                return;
            }

            var inSlope = 0.0f;
            var outSlope = 0.0f;
            var frameSnapFactor = FrameSnapFactor;
            var valueSnapFactor = ValueSnapFactor;
            var mouseFrame = (float)view_.GetXInCurvePlane(e.X);

            // キースナップ有効時には倍数スナップを使わない。
            if (view_.IsSnapToKey)
            {
                frameSnapFactor = 0;
                valueSnapFactor = 0;
            }

            if (frameSnapFactor > 0)
            {
                mouseFrame = (float)(Math.Round(mouseFrame / frameSnapFactor, MidpointRounding.AwayFromZero) * frameSnapFactor);
            }

            if (EditableCurves.Any())
            {
                foreach (var curve in MovableCurves)
                {
                    curve.KeyFrames.ForEach(x => x.Selected = false);
                }

                foreach (var curve in EditableCurves)
                {
                    _addOrInsertKeyFrameToEmptyCurve |= (curve.KeyFrames.Count == 0);
                    curve.SetCurvesSequentialIndex();
                    PreserveKeyFrames(curve);
                    var curveAnimTarget = curve.GetAnimTarget(view_.TargetGroup.Active);
                    var mouseValue = (float)curve.GetValue(mouseFrame, curveAnimTarget.pre_wrap, curveAnimTarget.post_wrap);
                    var key = curve.CreateKey(mouseFrame, mouseValue, inSlope, outSlope, frameSnapFactor, valueSnapFactor, IsClampValue, IsClampFrame, view_.ParentPanel.FrameCount);
                    AddDragObject(new KeyFrameHolder(key, this, curve, KeyFrameHolder.NearSelectType.Key, EditMode));
                }

                // 挿入時はスプライン補正を行わない
                //SetAutoSplineCurveIndices();

                foreach (var holder in DraggableObjects.Select(x => x.Interface).OfType<KeyFrameHolder>())
                {
                    holder.SetCurveIndex();
                }

                BeginInsertOrAdd();

                // 挿入モード時は束縛しない
                ConstraintEnabled = false;

                view_.Invalidate();
            }
        }

        private void BeginMove_Scale(MouseEventArgs e)
        {
            foreach(var curve in MovableCurves)
            {
                curve.SetCurvesSequentialIndex();

                foreach(var key in curve.KeyFrames.Where(x => x.AnySelected))
                {
                    PreserveKeyFrames(curve);
                    AddDragObject(new KeyFrameHolder(key, this, curve, KeyFrameHolder.NearSelectType.NoNearSelect, EditMode));
                }
            }

            SetAutoSplineCurveIndices();
        }

        public void BeginInsertOrAdd()
        {
            // 追加挿入時はドラッグされたことにする
            IsDragged = true;

            MovableCurves.ForEach(x => x.SortByFrame());

            if (KeyParamChanged != null)
            {
                KeyParamChanged(this, EventArgs.Empty);
            }

            if (_addOrInsertKeyFrameToEmptyCurve)
            {
                Viewer.LoadOrReloadAnimation.Send((AnimationDocument)view_.TargetGroup.Active.OwnerDocument);
            }
            else
            {
                // HIOに渡す
                HioSendToEdittingCurve();
            }
        }

        public void MoveMove(MouseEventArgs e, int moveDiffX, int moveDiffY)
        {
            // TODO: Drag 対象だけでいいはず
            // 並びを元に戻す
            MovableCurves.ForEach(x => x.SortByFrame());

            // 傾きを更新する
            UpdateAutoSplineCurveIndices();

            if (KeyParamChanged != null)
            {
                KeyParamChanged(this, EventArgs.Empty);
            }

            // HIOに渡す
            HioSendToEdittingCurve();
        }

        private void MoveScaleScroll(MouseEventArgs e, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY)
        {
            // スクロール
            if (((DragBegin.ModifierKeys & Keys.Alt) != 0) && (e.Button == MouseButtons.Middle))
            {
                ScrollX = dragBeginScrollX_ + moveDiffX / ScaleX;
                ScrollY = dragBeginScrollY_ + moveDiffY / ScaleY;
            }

            // スケール
            else
            {
                if (e.Button == MouseButtons.Right)
                {
                    // Alt+右ボタン 左右ドラッグでビューの均等スケール、ホイール
                    if (DragBegin.ModifierKeys == Keys.Alt)
                    {
                        moveDiffY = - moveDiffX;
                    }
                    else
                    // Alt+Shift+右ボタン 左右/上下ドラッグ開始でビューのフレーム方向スケール
                    if ((DragBegin.ModifierKeys & (Keys.Alt | Keys.Shift)) == (Keys.Alt | Keys.Shift))
                    {
                        moveDiffX = moveClipDiffX;
                        moveDiffY = moveClipDiffY;
                        if (DragMove.IsClipX)
                        {
                            view_.Cursor = ResCursors.MagnifyVertical;
                        }
                        else if (DragMove.IsClipY)
                        {
                            view_.Cursor = ResCursors.MagnifyHorizontal;
                        }
                    }
                }

                int centerScreenX = view_.CurveBgWidth / 2;
                int centerScreenY = view_.CurveBgHeight / 2;
                double beforeCurveX = view_.GetXInCurvePlane(centerScreenX);
                double beforeCurveY = view_.GetYInCurvePlane(centerScreenY);

                // 天地反転
                moveDiffY = -moveDiffY;

                double speedX = dragBeginScaleX_ * CurveEditorConst.DragSensitivity;
                double speedY = dragBeginScaleY_ * CurveEditorConst.DragSensitivity;

                double newScaleX = CurveEditorUtility.ClampScaleX(dragBeginScaleX_ + moveDiffX * speedX);
                double newScaleY = CurveEditorUtility.ClampScaleY(dragBeginScaleY_ + moveDiffY * speedY);

                ScaleX = newScaleX;
                ScaleY = newScaleY;

                double moveBeforeScreenX;
                double moveBeforeScreenY;
                MakeCurveViewScreenPos((float)beforeCurveX, (float)beforeCurveY, out moveBeforeScreenX, out moveBeforeScreenY);

                // ドラッグ開始位置をスケールの中心にする
                ScrollX += (moveBeforeScreenX - centerScreenX) / ScaleX;
                ScrollY += (moveBeforeScreenY - centerScreenY) / ScaleY;
            }
        }

        private bool IsSelectionClicked(MouseEventArgs e, int moveDiffX, int moveDiffY)
        {
            return SelectedCurves.Any(x => IsSelectionClicked(x, GetAnimationCurveSelectionInfo()));
        }

        private void MoveSelection(MouseEventArgs e, int moveDiffX, int moveDiffY)
        {
            VisibledCurves.ForEach(x => MoveSelection(x, GetAnimationCurveSelectionInfo()));
        }

        private AnimationCurveSelectionInfo GetAnimationCurveSelectionInfo()
        {
            int selectionLeft;
            int selectionTop;
            int selectionRight;
            int selectionBottom;

            if (SelectionW < 8 && SelectionH < 8)
            {
                selectionLeft = SelectionX + SelectionW/2 - 4;
                selectionTop = SelectionY + SelectionH/2 - 4;
                selectionRight = SelectionX + SelectionW/2 + 4;
                selectionBottom = SelectionY + SelectionH/2 + 4;
            }
            else
            {
                selectionLeft = SelectionX;
                selectionTop = SelectionY;
                selectionRight = SelectionX + SelectionW;
                selectionBottom = SelectionY + SelectionH;
            }

            double tempFrame0 = view_.GetXInCurvePlane(selectionLeft);
            double tempFrame1 = view_.GetXInCurvePlane(selectionRight);
            double tempValue0 = view_.GetYInCurvePlane(selectionTop);
            double tempValue1 = view_.GetYInCurvePlane(selectionBottom);

            double minFrame = Math.Min(tempFrame0, tempFrame1);
            double maxFrame = Math.Max(tempFrame0, tempFrame1);
            double minValue = Math.Min(tempValue0, tempValue1);
            double maxValue = Math.Max(tempValue0, tempValue1);

            var info = new AnimationCurveSelectionInfo
            {
                MinFrame = minFrame,
                MaxFrame = maxFrame,
                MinValue = minValue,
                MaxValue = maxValue,
                SelectionLeft = selectionLeft,
                SelectionTop = selectionTop,
                SelectionRight = selectionRight,
                SelectionBottom = selectionBottom,
                //
                View = view_
            };
            return info;
        }

        private void MoveSelection(IAnimationCurve curve, AnimationCurveSelectionInfo info)
        {
            List<KeyFrame> keyFrames = curve.KeyFrames;

            keyFrames.ForEach(x => x.ResetAreaSelected());

            foreach (var key in keyFrames)
            {
                if (CurveEditorUtility.IsHit(key, info))
                {
                    key.AreaSelected = true;
                    continue;
                }

                if (key.AnySelected)
                {
                    {
                        float p0x, p0y, p1x, p1y;
                        Handle(key, true, out p0x, out p0y, out p1x, out p1y);
                        if (CurveEditorUtility.IsHitScreen(p0x, p0y, p1x, p1y, info))
                        {
                            key.InHandleAreaSelected = true;
                        }
                    }
                    {
                        float p0x, p0y, p1x, p1y;
                        Handle(key, false, out p0x, out p0y, out p1x, out p1y);
                        if (CurveEditorUtility.IsHitScreen(p0x, p0y, p1x, p1y, info))
                        {
                            key.OutHandleAreaSelected = true;
                        }
                    }
                }
            }
        }

        private bool IsSelectionClicked(IAnimationCurve curve, AnimationCurveSelectionInfo info)
        {
            var keyFrames = curve.KeyFrames;

            foreach (var key in keyFrames)
            {
                if (key.Selected && CurveEditorUtility.IsHit(key, info))
                {
                    return true;
                }

                if (key.AnySelected)
                {
                    float p0x, p0y, p1x, p1y;
                    Handle(key, true, out p0x, out p0y, out p1x, out p1y);
                    if (CurveEditorUtility.IsHitScreen(p0x, p0y, p1x, p1y, info))
                    {
                        return true;
                    }
                    Handle(key, false, out p0x, out p0y, out p1x, out p1y);
                    if (CurveEditorUtility.IsHitScreen(p0x, p0y, p1x, p1y, info))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private void EndSelection()
        {
            if (VisibledCurves == null || !VisibledCurves.Any())
            {
                return;
            }
            var singleSelection = false;
            int selectionX = SelectionX;
            int selectionY = SelectionY;
            int selectionW = SelectionW;
            int selectionH = SelectionH;

            // クリックでも選択したいため、あまり動いていなかったら補正する
            // また、キーを一つだけ選択する
            const int clicksize = 8;
            if (selectionW < clicksize && selectionH < clicksize)
            {
                selectionX = selectionX + selectionW / 2 - clicksize / 2;
                selectionY = selectionY + selectionH / 2 - clicksize / 2;
                selectionW = clicksize;
                selectionH = clicksize;
                singleSelection = true;
            }

            double tempFrame0 = view_.GetXInCurvePlane(selectionX);
            double tempFrame1 = view_.GetXInCurvePlane(selectionX + selectionW);
            double tempValue0 = view_.GetYInCurvePlane(selectionY);
            double tempValue1 = view_.GetYInCurvePlane(selectionY + selectionH);

            double minFrame = Math.Min(tempFrame0, tempFrame1);
            double maxFrame = Math.Max(tempFrame0, tempFrame1);
            double minValue = Math.Min(tempValue0, tempValue1);
            double maxValue = Math.Max(tempValue0, tempValue1);

            int selectionLeft = selectionX;
            int selectionTop = selectionY;
            int selectionRight = selectionX + selectionW;
            int selectionBottom = selectionY + selectionH;

            //bool isToggle = (DragBegin.ModifierKeys & (Keys.Shift | Keys.Control)) != 0;

            var info = new AnimationCurveSelectionInfo
            {
                MinFrame = minFrame,
                MaxFrame = maxFrame,
                MinValue = minValue,
                MaxValue = maxValue,
                SelectionLeft = selectionLeft,
                SelectionTop = selectionTop,
                SelectionRight = selectionRight,
                SelectionBottom = selectionBottom,
                //
                View = view_,
                //IsToggle = isToggle,
            };

            VisibledCurves.ForEach(x => x.KeyFrames.ForEach(y => { y.AreaSelected = false; y.InHandleAreaSelected = false; y.OutHandleAreaSelected = false; }));

            // ヒット時のセレクション状態
            bool toggle = (DragBegin.ModifierKeys & Keys.Shift) != 0;
            bool hitSelect = DragBegin.ModifierKeys != Keys.Control;

            // ヒットしなかったときにセレクションを解除するか
            bool failDesect = (DragBegin.ModifierKeys & (Keys.Shift | Keys.Control)) == 0;

            // ヒットしたか
            bool hit = false;
            bool isCurveHit = false;

            // コントロールポイントのヒット
            //var selectedKeysList = new List<Tuple<IAnimationCurve, List<KeyFrame>>>();
            var deselectKeys = new List<KeyFrame>();
            var reversedCurves = VisibledCurves.AsEnumerable().Reverse().ToList();
            foreach (var curve in reversedCurves)
            {
                foreach (var key in curve.KeyFrames)
                {
                    bool selected = key.Selected;
                    if (!(singleSelection && hit) &&
                        CurveEditorUtility.IsHit(key, info))
                    {
                        key.Selected = toggle ? !selected: hitSelect;
                        if (toggle)
                        {
                            key.OutHandleSelected = false;
                            key.InHandleSelected = false;
                        }
                        hit = true;
                    }
                    else if (failDesect && selected)
                    {
                        // 選択状態がハンドルヒットに使われるので、このタイミングではできない
                        deselectKeys.Add(key);
                    }
                }
            }

            if (hit)
            {
                if (failDesect)
                {
                    deselectKeys.ForEach(x => x.Selected = false);
                    reversedCurves.ForEach(x => x.KeyFrames.ForEach(y => { y.OutHandleSelected = false; y.InHandleSelected = false; }));
                }
            }
            else
            {
                // ハンドルのヒット
                foreach (var curve in reversedCurves)
                {
                    foreach (var key in curve.KeyFrames)
                    {
                        if (key.AnySelected)
                        {
                            if (singleSelection && hit)
                            {
                                if (failDesect)
                                {
                                    key.InHandleSelected  = false;
                                    key.OutHandleSelected = false;
                                }
                            }
                            else
                            {
                                float p0x, p0y, p1x, p1y;
                                Handle(key, true, out p0x, out p0y, out p1x, out p1y);
                                if (CurveEditorUtility.IsHitScreen(p0x, p0y, p1x, p1y, info))
                                {
                                    hit = true;
                                    if ((toggle && key.InHandleSelected) || !hitSelect)
                                    {
                                        key.InHandleSelected = false;
                                    }
                                    else
                                    {
                                        key.InHandleSelected = true;
                                        key.Selected = false;
                                    }
                                }
                                else if (failDesect)
                                {
                                    key.InHandleSelected = false;
                                }
//							}
//							{
//								float p0x, p0y, p1x, p1y;
                                Handle(key, false, out p0x, out p0y, out p1x, out p1y);
                                if (CurveEditorUtility.IsHitScreen(p0x, p0y, p1x, p1y, info))
                                {
                                    hit = true;
                                    if ((toggle && key.OutHandleSelected) || !hitSelect)
                                    {
                                        key.OutHandleSelected = false;
                                    }
                                    else
                                    {
                                        key.OutHandleSelected = true;
                                        key.Selected = false;
                                    }
                                }
                                else if (failDesect)
                                {
                                    key.OutHandleSelected = false;
                                }
                            }
                        }
                    }
                }

                if (failDesect)
                {
                    reversedCurves.ForEach(x => x.KeyFrames.ForEach(y => { y.Selected = false; }));
                }

                // カーブのヒット
                if (!hit &&
                    hitSelect &&
                    info.SelectionLeft != info.SelectionRight &&
                    info.SelectionBottom != info.SelectionTop)
                {
                    foreach (var curve in reversedCurves)
                    {
                        if (!(singleSelection && isCurveHit) &&
                            CurveEditorUtility.IsHit(curve, info))
                        {
                            isCurveHit = true;

                            if (toggle)
                            {
                                curve.KeyFrames.ForEach(x =>
                                {
                                    if (!x.Selected)
                                    {
                                        x.Selected = true;
                                        x.InHandleSelected = false;
                                        x.OutHandleSelected = false;
                                    }
                                    else
                                    {
                                        x.Selected = false;
                                    }
                                });
                            }
                            else
                            {
                                curve.KeyFrames.ForEach(x =>
                                {
                                    x.Selected = true;
                                    x.InHandleSelected = false;
                                    x.OutHandleSelected = false;
                                });
                            }
                        }
                    }
                }
            }

            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=780
            // #26
            // 選択カーブ全てがキーフレームを持っていなかったら選択操作はしない
            if (hit || isCurveHit || SelectedCurves.Any(x => x.KeyFrames.Any()))
            {
                // 編集対象の更新
                SelectedCurves = reversedCurves.Where(x => x.KeyFrames.Any(y => y.AnySelected)).ToList();
            }

            if (KeyParamChanged != null)
            {
                KeyParamChanged(this, EventArgs.Empty);
            }
        }

        public void Handle(KeyFrame key, bool inSlope, out float p0x, out float p0y, out float p1x, out float p1y)
        {
            double d0x, d0y, d1x, d1y;
            MakeCurveViewScreenPos(
                key.Frame,
                key.Value,
                out d0x,
                out d0y);

            p0x = (float)d0x;
            p0y = (float)d0y;

            MakeCurveViewScreenHandleOffset(
                inSlope? key.InSlope : key.OutSlope,
                CurveEditorConst.HandleLineLengthScale,
                out d1x,
                out d1y
            );

            if (inSlope)
            {
                p1x = (float)(d0x - d1x);
                p1y = (float)(d0y - d1y);
            }
            else
            {
                p1x = (float)(d0x + d1x);
                p1y = (float)(d0y + d1y);
            }
        }

        public bool PreEndSelection(IAnimationCurve curve, AnimationCurveSelectionInfo info)
        {
            bool isModified		= false;

            List<KeyFrame> keyFrames = curve.KeyFrames;

            // 選択範囲系は全てオフ
            keyFrames.ForEach(x => x.ResetAreaSelected());

            // ハンドル
            {
                var	outHandleSelects = new List<KeyFrame>();
                var	inHandleSelects  = new List<KeyFrame>();

                if (curve.CurveInterpolationType == InterpolationType.Hermite)
                {
                    if (keyFrames.Any())
                    {
                        for (int i = 0; i != keyFrames.Count - 1; ++i)
                        {
                            var left  = keyFrames[i + 0];
                            var right = keyFrames[i + 1];

                            if (left.AnySelected || right.AnySelected)
                            {
                                {
                                    double keyX, keyY;
                                    MakeCurveViewScreenPos(
                                        left.Frame,
                                        left.Value,
                                        out keyX,
                                        out keyY
                                    );

                                    double handleX, handleY;
                                    MakeCurveViewScreenHandleOffset(
                                        left.OutSlope,
                                        CurveEditorConst.HandleLineLengthScale,
                                        out handleX,
                                        out handleY
                                    );

                                    var screenX = (int)(keyX + handleX);
                                    var screenY = (int)(keyY + handleY);

                                    if ((screenX >= info.SelectionLeft) && (screenX <= info.SelectionRight ) &&
                                        (screenY >= info.SelectionTop ) && (screenY <= info.SelectionBottom))
                                    {
                                        outHandleSelects.Add(left);
                                    }
                                }

                                {
                                    double keyX, keyY;
                                    MakeCurveViewScreenPos(
                                        right.Frame,
                                        right.Value,
                                        out keyX,
                                        out keyY
                                    );

                                    double handleX, handleY;
                                    MakeCurveViewScreenHandleOffset(
                                        right.InSlope,
                                        CurveEditorConst.HandleLineLengthScale,
                                        out handleX,
                                        out handleY
                                    );

                                    var screenX = (int)(keyX - handleX);
                                    var screenY = (int)(keyY - handleY);

                                    if ((screenX >= info.SelectionLeft) && (screenX <= info.SelectionRight ) &&
                                        (screenY >= info.SelectionTop ) && (screenY <= info.SelectionBottom))
                                    {
                                        inHandleSelects.Add(right);
                                    }
                                }
                            }
                        }
                    }
                }

                {
                    keyFrames.ForEach(
                        x => {
                            x.OutHandleSelected	= false;
                            x.InHandleSelected	= false;
                        }
                    );

                    inHandleSelects.ForEach( x => x.InHandleSelected  = true);
                    outHandleSelects.ForEach(x => x.OutHandleSelected = true);

                    if (inHandleSelects.Any() || outHandleSelects.Any())
                    {
                        isModified = true;
                    }
                }
            }

            // キーフレーム
            {
                // すべて未選択にする
                if (info.IsToggle == false)
                {
                    keyFrames.ForEach(x => x.Selected = false);
                }

                foreach(var key in keyFrames)
                {
                    double frame = key.Frame;
                    double value = key.Value;

                    if ((frame >= info.MinFrame) && (frame <= info.MaxFrame) &&
                        (value >= info.MinValue) && (value <= info.MaxValue))
                    {
                        key.Selected = ! key.Selected;
                        isModified = true;
                    }
                }
            }

            return isModified;
        }



        public void PostEndSelection(IAnimationCurve curve, AnimationCurveSelectionInfo info)
        {
            var selectionW = info.SelectionRight  - info.SelectionLeft;
            var selectionH = info.SelectionBottom - info.SelectionTop;

            info.PostEndSelectionResult.IsHit =
                (selectionW > 0) &&
                (selectionH > 0) &&
                CurveEditorUtility.IsHit(curve, info);
        }

        public void EndMove()
        {

            if (view_.CheckEditRestriction() == false)
            {
                view_.Invalidate();
                return;
            }

            if (DraggableObjects.Any())
            {
                DebugConsole.WriteLine("EndMove");
                // 並びを元に戻す
                //MovableCurves.ForEach(x => x.SortByIndex());

                var commandSet = new EditCommandSet();
                {
                    commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                    foreach (var keyData in PreservedKeyFrames)
                    {
                        var curve = keyData.Key;
                        var editted = curve.KeyFrames.ToList();
                        // 一旦元に戻す
                        curve.KeyFrames.Clear();
                        curve.KeyFrames.AddRange(keyData.Value);

                        commandSet.Add(
                            AnimationCurveEditCommand.Create(
                                view_.TargetGroup,
                                view_.TargetGuiObjectID,
                                curve,
                                editted,
                                notReloadFirst: _addOrInsertKeyFrameToEmptyCurve));
                    }
                }

                if (commandSet.CommandCount > 0)
                {
                    commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(view_.TargetGroup, view_.TargetGuiObjectID);

                    TheApp.CommandManager.Add(commandSet.Execute());
                }
            }

            if (KeyParamChanged != null)
            {
                KeyParamChanged(this, EventArgs.Empty);
            }
        }

        public void CancelMove()
        {
            DebugConsole.WriteLine("CancelMove");
            // TODO: Drag 対象だけでいいはず
            // 並びを元に戻す
            //MovableCurves.ForEach(x => x.SortByFrame());

            // 元に戻す
            foreach (var keyData in PreservedKeyFrames)
            {
                var curve = keyData.Key;
                curve.KeyFrames.Clear();
                curve.KeyFrames.AddRange(keyData.Value);
            }

            if (view_.TargetGroup.Active == null)
            {
                // 編集対象がなくなったりしたらここでおしまい。
                return;
            }

            if (KeyParamChanged != null)
            {
                KeyParamChanged(this, EventArgs.Empty);
            }

            if (_addOrInsertKeyFrameToEmptyCurve)
            {
                Viewer.LoadOrReloadAnimation.Send((AnimationDocument)view_.TargetGroup.Active.OwnerDocument);
            }
            else
            {
                // HIOに渡す
                HioSendToEdittingCurve();
            }

            // ドラッグ中にカーブツリービューが更新される場合があるので、もとに戻す必要がある
            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(view_.TargetGroup.Active, null));
        }

        public void MakeCurveViewScreenPos(float srcX, float srcY, out double dstX, out double dstY)
        {
            view_.MakeCurveViewScreenPos(srcX, srcY, out dstX, out dstY);
        }

        public void MakeCurveViewScreenHandleOffset(double slope, double handleLineLength, out double handleX, out double handleY)
        {
            view_.MakeCurveViewScreenHandleOffset(slope, handleLineLength, out handleX, out handleY);
        }

        public double MakeSlope(double x, double y)
        {
            return view_.MakeSlope(x, y);
        }

        private double SnapFrame(double src)
        {
            return IsUseFrameSnap ? Math.Round(src / FrameSnapFactor, MidpointRounding.AwayFromZero) * FrameSnapFactor : src;
        }

        private double SnapValue(double src)
        {
            if (IsUseValueSnap)
            {
                double valueSnapFactor = ValueSnapFactor;
                return Math.Round(src / valueSnapFactor, MidpointRounding.AwayFromZero) * valueSnapFactor;
            }
            return src;
        }

        private double ClampFrame(double src)
        {
            return IsClampFrame ? Math.Min(Math.Max(src, 0.0), view_.FrameCount) : src;
        }

        private void UpdateMouseCursor(int x, int y)
        {
            var isAllColorCurveEditable = view_.SelectedColorCurves.Any() && view_.SelectedColorCurves.All(c => c.IsEditable);
            var isAllVisibleCurveEditable = view_.SelectedVisibleCurves.Any() && view_.SelectedVisibleCurves.All(c => c.IsEditable);
            var isAllTexPatCurveEditable = view_.SelectedTexturePatternCurves.Any() && view_.SelectedTexturePatternCurves.All(c => c.IsEditable);

            var isInHRulerColorBar			= IsInHRulerColorBar(x, y);
            var isInHRulerVisibleBar		= IsInHRulerVisibleBar(x, y);
            var isInHRulerTexturePatternBar	= IsInHRulerTexturePatternBar(x, y);
            var isInHRuler					= IsInHRuler(x, y);

            var isNotInHRulerColorBar			= ! isInHRulerColorBar;
            var isNotInHRulerVisibleBar			= ! isInHRulerVisibleBar;
            var isNotInHRulerTexturePatternBar	= ! isInHRulerTexturePatternBar;

            view_.Cursor =
                isInHRulerColorBar && isAllColorCurveEditable			? Cursors.Hand :
                isInHRulerVisibleBar && isAllVisibleCurveEditable       ? Cursors.Hand :
                isInHRulerTexturePatternBar && isAllTexPatCurveEditable	? Cursors.Hand :
                isInHRuler && isNotInHRulerColorBar && isNotInHRulerVisibleBar && isNotInHRulerTexturePatternBar ? Cursors.VSplit :
                                                                      Cursors.Default;
        }

        private TexturePreviewToolTip editBarCurrentImage_ = null;

        // テクスチャの画像を大きく表示する
        private void ShowEditBarCurrentImage(int x, int y)
        {
            if (IsInHRulerTexturePatternBar(x, y))
            {
                if (editBarCurrentImage_ == null)
                {
                    Form parentForm = null;

                    {
                        var parent = view_.Parent;

                        while(true)
                        {
                            parentForm = parent as Form;
                            if (parentForm != null)
                            {
                                break;
                            }

                            parent = parent.Parent;
                        }
                    }

                    editBarCurrentImage_ = new TexturePreviewToolTip
                    {
                        Owner = parentForm
                    };
                }

                var curves = view_.SelectedTexturePatternCurves;
                if (curves.Any(c => c.KeyFrames.Any()))
                {
                    var curve			= curves.First();
                    var frame			= (float)view_.GetXInCurvePlane(x);
                    var texPatternAnim	= view_.TargetGroup.Active as IHasTexturePatternAnimation;
                    //var info			= (IHasTexturePatternAnimationCurveTreeNodeInfo)curve;
                    var target			= curve.GetAnimTarget(view_.TargetGroup.Active);
                    var value			= curve.GetValue(frame, target.pre_wrap, target.post_wrap);

                    if (texPatternAnim != null)
                        editBarCurrentImage_.Texture = texPatternAnim.GetTextureFromPatternIndex((int)value);
                }
                else
                {
                    editBarCurrentImage_.Texture = null;
                }

                editBarCurrentImage_.Location = view_.PointToScreen(new Point(x + 1, y + 1));

                editBarCurrentImage_.Show();
            }
            else
            {
                if (editBarCurrentImage_ != null)
                {
                    editBarCurrentImage_.Hide();
                }
            }
        }

        // HIOに編集中のカーブを送る
        private void HioSendToEdittingCurve()
        {
            // ドラッグしているオブジェクトがなければ
            if (DraggableObjects.Any(x => ((KeyFrameHolder)x.Interface).sendCurve) == false)
            {
                return;
            }

            var animation = view_.TargetGroup.Active;
            var target = (AnimationDocument)animation.OwnerDocument;
            Viewer.TransactionMessage.Send(true);
            foreach (var draggableObject in DraggableObjects)
            {
                var holder = (KeyFrameHolder)draggableObject.Interface;
                if (holder.sendCurve)
                {
                    var curve = holder.Curve;
                    var animTarget = curve.GetAnimTarget(animation);
                    if (animTarget != null)
                    {
                        animTarget.IsFileOutputable = AnimationCurveEditCommand.MakeIsFileOutputable(animation, curve);
                    }

                    if (animTarget == null) continue;

                    var exportType = animTarget.ExportType;
                    switch (exportType)
                    {
                        case CurveExportType.Curve:
                        case CurveExportType.Constant:
                            Viewer.EditCurve.Send(
                                target,
                                (uint)holder.parentIndex,
                                (uint)holder.curveIndex,
                                (uint)holder.curveComponentIndex,
                                curve.IsRotate,
                                animTarget,
                                holder.curveIndexInBinary,
                                animation.ObjectID,
                                curve.CurvePrimitiveType,
                                false,
                                exportType);
                            break;
                    }
                }
            }
            Viewer.TransactionMessage.Send(false);
        }
    }
}
