﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Controls;
using App.Utility;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;

namespace App.PropertyEdit
{
    public partial class CurveViewState
    {
        private class DragKey
        {
            public IAnimationCurve AnimationCurve;
            public KeyFrame KeyFrame;
        }

        private List<DragKey> SetDragKeys(CurveView.ColorSet colorSet)
        {
            var editTarget = new List<DragKey>();
            foreach (var curve in colorSet.Curves)
            {
                curve.SetCurvesSequentialIndex();
                var keyFrame = curve.KeyFrames.FirstOrDefault(x => x.Frame == colorSet.Frame);
                if (keyFrame != null)
                {
                    PreserveKeyFrames(curve);
                    editTarget.Add(new DragKey
                    {
                        AnimationCurve = curve,
                        KeyFrame = keyFrame,
                    });
                    AddDragObject(new KeyFrameHolder(keyFrame, this, curve, KeyFrameHolder.NearSelectType.Key, EditModeType.Multi));
                }
            }

            SetAutoSplineCurveIndices();
            UpdateAutoSplineCurveIndices();
            return editTarget;
        }

        public ColorPickerAdapter colorPickerAdapter = null;

        public void ColorPickerKeyMove(CurveView.ColorSet colorSet, bool activate)
        {
            //bool hasA = colorSet.Curves.Any(x => x.ComponentIndex == 3);
            var alphaCurve = colorSet.Curves.FirstOrDefault(x => x.ComponentIndex == 3);
            var hasA = alphaCurve != null && alphaCurve.KeyFrames.Any(x => x.Frame == colorSet.Frame);
            List<DragKey> editTarget = null;

            var colorPickerTrigger = new ColorPickerTrigger();
            colorPickerTrigger.Color = colorSet.Color;
            colorPickerTrigger.EnableAlpha = hasA;
            colorPickerTrigger.IsDefaultLinear = ConfigData.ApplicationConfig.Color.GammaCorrection;
            colorPickerTrigger.HDRUpperBound = float.MaxValue;
            if (view_.ParentPanel.IsClampValue)
            {
                var curve = colorSet.Curves.First();
                if (curve.MaxClampValue.HasValue)
                {
                    colorPickerTrigger.HDRUpperBound = curve.MaxClampValue.Value;
                }
            }
            colorPickerAdapter = new ColorPickerAdapter(colorPickerTrigger);
            var lastRgba = new float[]{-1,-1,-1,-1};
            bool dragging = false;
            SuppressBlock block = null;
            var hasKey = new bool[4];
            foreach (var curve in colorSet.Curves)
            {
                if (curve.KeyFrames.Any(x => x.Frame == colorSet.Frame))
                {
                    hasKey[curve.ColorComponentIndex] = true;
                }
            }

            // コマンド実行等による変更を反映
            EventHandler formUpdated = (s, e) =>
                {
                    // ドラッグ中は無視
                    if (!dragging)
                    {
                        Debug.Assert(colorSet != null);
                        colorSet = view_.UpdateColorSet(colorSet, hasKey);
                        if (colorSet == null)
                        {
                            ColorPickerDialog.EndConnection(colorPickerAdapter);
                            return;
                        }
                        colorPickerTrigger.Color = colorSet.Color;
                        ColorPickerDialog.UpdateConnection(colorPickerAdapter);
                    }
                };

            view_.ParentPanel.FormUpdated += formUpdated;

            // ドラッグ中は Undo/Redo をさせない
            colorPickerAdapter.CanUndoRedo += () => !dragging;

            // 変更処理
            colorPickerAdapter.ColorEdit += (s, e) =>
            {
                if (!dragging)
                {
                    block = new SuppressBlock(App.AppContext.UpdateFromOutSideBlockCounter);

                    // ドラッグ開始
                    editTarget = SetDragKeys(colorSet);
                    dragging = true;
                }

                var rgba = new[]
                {
                    e.Color.R,
                    e.Color.G,
                    e.Color.B,
                    e.Color.A
                };

                foreach (var target in editTarget)
                {
                    target.KeyFrame.Value = rgba[target.AnimationCurve.ColorComponentIndex];
                }

                if (e.EditFixed)
                {
                    // ドラッグ終了
                    foreach (DragableObject o in DraggableObjects)
                    {
                        o.Interface.DragEnd(o.Detail, this);
                    }

                    EndMove();

                    PreservedKeyFrames.Clear();
                    AutoSplineCurveIndices.Clear();
                    DraggableObjects.Clear();

                    view_.Invalidate();

                    dragging = false;
                    block.Dispose();
                    block = null;
                }
                else
                {
                    // ドラッグ中
                    if (!lastRgba.SequenceEqual(rgba))
                    {
                        MoveMove(null, 0, 0);
                        view_.Invalidate();
                    }
                }

                lastRgba = rgba;
            };

            // 終了前処理
            colorPickerAdapter.BeforeDisconnect += (s, e) =>
            {
                view_.ParentPanel.FormUpdated -= formUpdated;
                if (dragging)
                {
                    foreach (DragableObject o in DraggableObjects)
                    {
                        o.Interface.DragCancel(o.Detail, this);
                    }

                    CancelMove();

                    PreservedKeyFrames.Clear();
                    AutoSplineCurveIndices.Clear();

                    DraggableObjects.Clear();
                    dragging = false;
                    block.Dispose();
                    block = null;
                }
            };

            // 接続開始
            ColorPickerDialog.StartConnection(colorPickerAdapter, activate);
        }

        public void ColorPickerKeyAdd(float frame)
        {
            float HDRFactor;
            bool hasA;

            bool isSnapToKey = view_.IsSnapToKey;

            if (IsUseFrameSnap && !isSnapToKey)
            {
                frame = (float)SnapFrame(frame);
            }

            // TODO: float のカラーで受け取る
            var color = view_.MakeColorFromColorCurve(frame, out HDRFactor, out hasA).ToColor();
            var lastRgba = new[]
            {
                (float)Math.Round(color.R * HDRFactor / 255.0f, 2),
                (float)Math.Round(color.G * HDRFactor / 255.0f, 2),
                (float)Math.Round(color.B * HDRFactor / 255.0f, 2),
                (float)Math.Round(color.A * HDRFactor / 255.0f, 2),
            };

            if (IsUseValueSnap && !isSnapToKey)
            {
                lastRgba[0] = (float)SnapValue(lastRgba[0]);
                lastRgba[1] = (float)SnapValue(lastRgba[1]);
                lastRgba[2] = (float)SnapValue(lastRgba[2]);
                lastRgba[3] = (float)SnapValue(lastRgba[3]);
            }

            var colorSet = new CurveView.ColorSet()
            {
                Color = new RgbaColor(lastRgba[0], lastRgba[1], lastRgba[2], lastRgba[3]),
                Curves = view_.SelectedColorCurves.ToList(),
                Frame = frame,
            };

            var editTarget = new List<DragKey>();

            // 追加
            foreach (var curve in view_.SelectedColorCurves)
            {
                _addOrInsertKeyFrameToEmptyCurve |= (curve.KeyFrames.Count == 0);
                curve.SetCurvesSequentialIndex();
                var keyFrame = new KeyFrame
                {
                    Frame = frame,
                    Value = lastRgba[curve.ColorComponentIndex],
                    Selected = true,
                };
                editTarget.Add(new DragKey
                {
                    AnimationCurve = curve,
                    KeyFrame = keyFrame,
                });
                PreserveKeyFrames(curve);
                AddDragObject(new KeyFrameHolder(keyFrame, this, curve, KeyFrameHolder.NearSelectType.Key, EditModeType.Add));
            }

            SetAutoSplineCurveIndices(true);
            UpdateAutoSplineCurveIndices();

            foreach (var holder in DraggableObjects.Select(x => x.Interface).OfType<KeyFrameHolder>())
            {
                holder.SetCurveIndex();
            }

            BeginInsertOrAdd();

            // 直ぐに追加の確定
            foreach (DragableObject o in DraggableObjects)
            {
                o.Interface.DragEnd(o.Detail, this);
            }

            EndMove();
            PreservedKeyFrames.Clear();
            AutoSplineCurveIndices.Clear();
            DraggableObjects.Clear();

            // 移動用のダイアログ表示
            colorSet = view_.UpdateColorSet(colorSet, null);
            if (colorSet != null)
            {
                ColorPickerKeyMove(colorSet, true);
            }
            else
            {
                Debug.Assert(false);
            }
        }

        public class ColorPickerTrigger : IColorPickerTrigger
        {
            /// <summary>
            /// カラー。
            /// </summary>
            public RgbaColor Color
            {
                get;
                set;
            }

            /// <summary>
            /// 編集用カラーかどうか。
            /// </summary>
            public bool IsMarkColor
            {
                get { return false; }
            }

            /// <summary>
            /// アルファ値有効フラグ。
            /// </summary>
            public bool EnableAlpha
            {
                get;
                set;
            }

            public bool IsDefaultLinear
            {
                get;
                set;
            }

            public bool ReadOnly
            {
                get;
                set;
            }

            /// <summary>
            /// HDR の上限。
            /// </summary>
            public float HDRUpperBound
            {
                get;
                set;
            }
        }

        public void InsertColorKey(IEnumerable<IAnimationCurve> ColorCurves, float frame, RgbaColor color)
        {
            var editTarget = new List<DragKey>();
            var values = new[] { color.R, color.G, color.B, color.A };
            // 追加
            foreach (var curve in ColorCurves)
            {
                if (ColorEditPanel.CopiedEditMode == ColorEditMode.RGBA ||
                    curve.ColorComponentIndex < 3)
                _addOrInsertKeyFrameToEmptyCurve |= (curve.KeyFrames.Count == 0);
                curve.SetCurvesSequentialIndex();
                var keyFrame = new KeyFrame
                {
                    Frame = frame,
                    Value = values[curve.ColorComponentIndex],
                    Selected = true,
                };
                editTarget.Add(new DragKey
                {
                    AnimationCurve = curve,
                    KeyFrame = keyFrame,
                });
                PreserveKeyFrames(curve);
                AddDragObject(new KeyFrameHolder(keyFrame, this, curve, KeyFrameHolder.NearSelectType.Key, EditModeType.Add));
            }

            SetAutoSplineCurveIndices(true);
            UpdateAutoSplineCurveIndices();

            foreach (var holder in DraggableObjects.Select(x => x.Interface).OfType<KeyFrameHolder>())
            {
                holder.SetCurveIndex();
            }

            BeginInsertOrAdd();

            // 直ぐに追加の確定
            foreach (DragableObject o in DraggableObjects)
            {
                o.Interface.DragEnd(o.Detail, this);
            }

            EndMove();
            PreservedKeyFrames.Clear();
            AutoSplineCurveIndices.Clear();
            DraggableObjects.Clear();
        }
    }
}
