﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Properties;
using App.Utility;

namespace App.PropertyEdit
{
    public class CurveTreeView : UITreeView
    {
        public List<CurveTreeNode> AllCurveNodes
        {
            get
            {
                var allNodes = new List<CurveTreeNode>();
                {
                    foreach (var node in Nodes.OfType<CurveTreeNode>())
                    {
                        allNodes.AddRange(TreeNodes(node, x => x.Tag != null));
                    }
                }
                return allNodes;
            }
        }

        public IEnumerable<CurveTreeNode> AllAnimatableNodes
        {
            get { return AllTreeNodes.Where(node => node.Info != null && node.Info.AnimationCurve != null); }
        }

        public static IEnumerable<Tuple<CurveTreeNode, string>> NodesAndFullPath(CurveTreeNode root, string path)
        {
            return Enumerable.Repeat(new Tuple<CurveTreeNode, string>(root, path), 1).Concat(
                root.Nodes.OfType<CurveTreeNode>().SelectMany(x => NodesAndFullPath(x, path + "\\" + x.id)));
        }

        public IEnumerable<CurveTreeNode> AllTreeNodes
        {
            get
            {
                return Nodes.OfType<CurveTreeNode>().SelectMany(node => TreeNodes(node, x => true));
            }
        }

        public IEnumerable<TreeNode> AllTreeNodesLeafFirst()
        {
            return Nodes.OfType<TreeNode>().SelectMany(x => AllTreeNodesLeafFirst(x));
        }

        public IEnumerable<TreeNode> AllTreeNodesLeafFirst(TreeNode node)
        {
            return node.Nodes.OfType<TreeNode>().SelectMany(x => AllTreeNodesLeafFirst(x)).Concat(Enumerable.Repeat(node, 1));
        }

        public IEnumerable<CurveTreeNode> TreeNodesLeafFirst(CurveTreeNode node, Predicate<CurveTreeNode> predicate)
        {
            var nodes = node.Nodes.OfType<CurveTreeNode>().SelectMany(x => TreeNodesLeafFirst(x, predicate));
            if (predicate(node))
            {
                return nodes.Concat(Enumerable.Repeat(node, 1));
            }

            return nodes;
        }

        public static IEnumerable<CurveTreeNode> TreeNodes(CurveTreeNode node, Predicate<CurveTreeNode> predicate)
        {
            var nodes = node.Nodes.OfType<CurveTreeNode>().SelectMany(x => TreeNodes(x, predicate));
            if (predicate(node))
            {
                return Enumerable.Repeat(node, 1).Concat(nodes);
            }

            return nodes;
        }

        private static readonly ImageList imageList = new ImageList()
        {
            ImageSize			= new Size(16, 16),
            ColorDepth			= ColorDepth.Depth32Bit,
            TransparentColor	= Color.Transparent
        };

        static CurveTreeView()
        {
            var selectStates = new Tuple<NodeSelectState, Bitmap> []
            {
                new Tuple<NodeSelectState, Bitmap>(NodeSelectState.Unselected , Resources.AnimationEditor_Unselected),
                new Tuple<NodeSelectState, Bitmap>(NodeSelectState.Selected, Resources.AnimationEditor_Selected),
                new Tuple<NodeSelectState, Bitmap>(NodeSelectState.SemiSelected, Resources.AnimationEditor_SemiSelected)
            };

            var hasCurves = new Tuple<HasCurveType, Bitmap>[]
            {
                new Tuple<HasCurveType, Bitmap>(HasCurveType.NoCurve, null),
                new Tuple<HasCurveType, Bitmap>(HasCurveType.SomeCurve, Resources.AnimationEditor_AnyCurve),
                new Tuple<HasCurveType, Bitmap>(HasCurveType.AllCurve, Resources.AnimationEditor_AllCurve)
            };

            const float r = 0.298912F;
            const float g = 0.586611F;
            const float b = 0.114478F;
            float[][] matrixElement =
                      {new float[]{r, r, r, 0, 0},
                       new float[]{g, g, g, 0, 0},
                       new float[]{b, b, b, 0, 0},
                       new float[]{0, 0, 0, 1, 0},
                       new float[]{0, 0, 0, 0, 1}};
            ColorMatrix matrix = new ColorMatrix(matrixElement);
            ImageAttributes attr = new ImageAttributes();
            attr.SetColorMatrix(matrix);

            foreach (Editable editable in Enum.GetValues(typeof(Editable)))
            {
                foreach (var curve in hasCurves)
                {
                    foreach (var state in selectStates)
                    {
                        Bitmap bitmap;
                        Graphics graphics;
                        // state
                        if (editable == Editable.NoAndSelected)
                        {
                            bitmap = new Bitmap(state.Item2.Width, state.Item2.Height);
                            graphics = Graphics.FromImage(bitmap);
                            GraphicsUtility.DrawGrayImage(
                                graphics,
                                state.Item2,
                                0, 0, state.Item2.Width, state.Item2.Height,
                                0, 0, state.Item2.Width, state.Item2.Height
                            );
                        }
                        else
                        {
                            bitmap = new Bitmap(state.Item2);
                            graphics = Graphics.FromImage(bitmap);
                        }

                        // curve
                        if (curve.Item2 != null)
                        {
                            if (editable == Editable.Yes)
                            {
                                graphics.DrawImage(curve.Item2, 0, 0);
                            }
                            else
                            {
                                GraphicsUtility.DrawGrayImage(
                                    graphics,
                                    curve.Item2,
                                    0, 0, curve.Item2.Width, curve.Item2.Height,
                                    0, 0, curve.Item2.Width, curve.Item2.Height
                                );
                            }
                        }

                        imageList.Images.Add(bitmap);
                    }
                }
            }
        }

        public void InitializeForm(CurveEditorPanel parent)
        {
        }

        public CurveTreeView()
        {
            ImageList = imageList;

            DrawMode = TreeViewDrawMode.OwnerDrawText;
        }

        public enum NodeSelectState
        {
            Unselected,
            Selected,
            SemiSelected
        }

        public enum HasCurveType
        {
            NoCurve,
            SomeCurve,
            AllCurve
        }

        public enum Editable
        {
            No,
            NoAndSelected,
            Yes,
        }

        private const int NodeSelectStatePlace = 1;
        private static readonly int HasCurvePlace = NodeSelectStatePlace * Enum.GetValues(typeof(NodeSelectState)).Length;
        private static readonly int EditablePlace = HasCurvePlace * Enum.GetValues(typeof(HasCurveType)).Length;
        public void SetNodeState(TreeNode target, NodeSelectState state, HasCurveType hasCurve, Editable editable)
        {
            int index = (int)state * NodeSelectStatePlace + (int)hasCurve*HasCurvePlace + (int)editable * EditablePlace;

            if (index != target.ImageIndex)
            {
                target.ImageIndex = index;
            }

            if (index != target.SelectedImageIndex)
            {
                target.SelectedImageIndex = index;
            }
        }

        public NodeSelectState GetNodeState(TreeNode target)
        {
            return (NodeSelectState)((target.ImageIndex % HasCurvePlace)/NodeSelectStatePlace);
        }

        public HasCurveType GetHasCurveType(TreeNode target)
        {
            return (HasCurveType)((target.ImageIndex % EditablePlace) / HasCurvePlace);
        }

        public Editable GetEditable(TreeNode target)
        {
            return (Editable)(target.ImageIndex / EditablePlace);
        }

        public void ResetShaderParameterNodeExpand()
        {
            using (var block = new UpdateBlock(this))
            {
                var topNode = TopNode;
                foreach (var node in AllTreeNodes.OfType<CurveTreeNode>())
                {
                    bool expand = true;
                    switch (node.Info.Category)
                    {
                        case CurveTreeNodeCategory.ParamAnim:
                        case CurveTreeNodeCategory.ShaderParamAnimTarget:
                        case CurveTreeNodeCategory.ShaderParamGroup:
                            expand = false;
                            break;
                    }

                    if (expand)
                    {
                        node.Expand();
                    }
                    else
                    {
                        node.Collapse();
                    }
                }
                TopNode = topNode;
            }
        }

        public void ResetNodeExpand()
        {
            using (var block = new UpdateBlock(this))
            {
                SetExpand(2);
            }
        }

        public void OpenNodeExpand()
        {
            using (var block = new UpdateBlock(this))
            {
                SetExpand(9);
            }
        }

        public void SetExpand(int level)
        {
            var topNode = TopNode;
            {
                foreach(TreeNode node in Nodes)
                {
                    SetExpand(node, level, 0);
                }
            }
            TopNode = topNode;
        }

        private void SetExpand(TreeNode node, int level, int currentLevel)
        {
            if (currentLevel < level)
            {
                node.Expand();

                foreach(TreeNode childNode in node.Nodes)
                {
                    SetExpand(childNode, level, currentLevel + 1);
                }
            }
            else
            {
                node.Collapse();

                foreach(TreeNode childNode in node.Nodes)
                {
                    SetExpand(childNode, level, currentLevel + 1);
                }
            }
        }

        public void VisibleAllNode()
        {
            using (var ub = new UpdateBlock(this))
            {
                SetCheck(true);
            }
        }

        public void InvisibleAllNode()
        {
            using (var ub = new UpdateBlock(this))
            {
                SetCheck(false);
            }
        }

        public void SetCheck(bool isChecked)
        {
            var topNode = TopNode;
            {
                foreach(TreeNode node in Nodes)
                {
                    SetCheck(node, isChecked);
                }
            }
            TopNode = topNode;
        }

        private void SetCheck(TreeNode node, bool isChecked)
        {
            node.Checked = isChecked;

            foreach(TreeNode childNode in node.Nodes)
            {
                SetCheck(childNode, isChecked);
            }
        }

        protected override void OnKeyPress(KeyPressEventArgs e)
        {
            e.Handled = true;
            base.OnKeyPress(e);
        }

        private static readonly Dictionary<string, int> stringWidths_ = new Dictionary<string, int>();

        protected override void OnDrawNode(DrawTreeNodeEventArgs e)
        {
            var anim = e.Node.Tag as IAnimationCurve;

            var graphics = e.Graphics;
            var nodeFont = (e.Node.NodeFont == null) ? Font : e.Node.NodeFont;
            var bounds   = e.Node.Bounds;

            // この状態は描画しない
            if ((bounds.X == 0) && (bounds.Y == 0))
            {
                return;
            }

            Debug.Assert(e.Node is CurveTreeNode);
            var node = e.Node as CurveTreeNode;

            graphics.FillRectangle(SystemBrushes.Window, bounds);

            int textOffset = 12;
            int boxSize = 7;
            var bgBounds = bounds;

            if (anim != null)
            {
                bgBounds.X += textOffset;

                int width = 0;
                if (stringWidths_.TryGetValue(e.Node.Text, out width) == false)
                {
                    width = TextRenderer.MeasureText(graphics, e.Node.Text, nodeFont).Width;
                    stringWidths_[e.Node.Text] = width;
                }

                bgBounds.Width = width;
            }

            if (Enabled)
            {
                if (e.Node.IsSelected)
                {
                    graphics.FillRectangle(SystemBrushes.Highlight, bgBounds);
                    TextRenderer.DrawText(graphics, e.Node.Text, nodeFont, new Point(bgBounds.X, bgBounds.Y), Color.White);
                }
                else
                {
                    TextRenderer.DrawText(graphics, e.Node.Text, nodeFont, new Point(bgBounds.X, bgBounds.Y), Color.Black);
                }

                // 色ボックス
                if (anim != null)
                {
                    var rect = new Rectangle(bounds.X, bounds.Y + (bounds.Height - boxSize) / 2, boxSize, boxSize);

                    using(var brush = new SolidBrush(anim.CurveColor))
                    {
                        graphics.FillRectangle(brush, rect);
                    }

                    graphics.DrawRectangle(Pens.Black, rect);
                }
            }
            else
            {
                TextRenderer.DrawText(graphics, e.Node.Text, nodeFont, new Point(bgBounds.X, bgBounds.Y), Color.Gray);
            }

            var x = bgBounds.Right;

            if (node.Info.IsModified)
            {
                var image = ModifiedMark.StarMark;
                graphics.DrawImage(image, new Point(x, (bgBounds.Top + bgBounds.Bottom - image.Height) / 2));

                x += image.Width;
            }

            // image
            if (node.Info.Image != null)
            {
                x += 2;
                graphics.DrawImage(node.Info.Image, new Point(x, (bgBounds.Top + bgBounds.Bottom - node.Info.Image.Height) / 2));
            }
        }

        private HintToolTip	toolhint_ = new HintToolTip();
        private string		oldHintString_ = null;

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            var pos = PointToClient(Cursor.Position);

            var hitInfo = HitTest(pos.X, pos.Y);

            var node = GetNodeAt(pos) as CurveTreeNode;
            if ((node != null) &&
                (node.Info.ImageToolTipString != null) &&
                (hitInfo.Location != TreeViewHitTestLocations.Indent))
            {
                if (oldHintString_ != node.Info.ImageToolTipString)
                {
                    toolhint_.Show(node.Info.ImageToolTipString, this, pos.X + 6, pos.Y + 6);
                    oldHintString_ = node.Info.ImageToolTipString;
                }
            }
            else
            {
                toolhint_.Hide(this);
                toolhint_.RemoveAll();

                oldHintString_ = null;
            }
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);
            oldHintString_ = null;
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            toolhint_.Dispose();
            toolhint_ = null;
        }
    }
}
