﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using App.Win32;
using App.res;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using System.Threading;
using System.Globalization;
using System.IO;

using App.FileView;

namespace App.PropertyEdit
{
    [ToolboxItem(true)]
    public partial class CurveEditorPanel : UIUserControl
    {
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public GuiObjectGroup			TargetGroup{		get; set; }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public GuiObjectID				TargetGuiObjectID{	get; private set; }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public CurveTreeView			CurveTreeView{		get; private set; }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public List<IAnimationCurve>	SelectedCurves
        {
            get {			return SelectedCurves_;	}
            set {
                SelectedCurves_ = value;
                EditableCurves = SelectedCurves.Where(x => x.IsEditable).ToList();
                if (ChangeSelectedCurves != null)
                {
                    ChangeSelectedCurves(this, EventArgs.Empty);
                }

                UpdateTreeViewSelected();

                var anyHermite = SelectedCurves.Any(x => x.IsEditable && x.CurveInterpolationType == InterpolationType.Hermite);
                SetSlopeControlState(anyHermite);
                bool isAnySelectedKeys = SelectedCurves.Any(x => x.KeyFrames.Any());
                keyMoveScaleUtility.SetScalingControlState(isAnySelectedKeys);
                tscFrame.Enabled = tscValue.Enabled = isAnySelectedKeys;

                EditToolStrip.EnableToolStripItems(IsEditableAnimation);
                PasteToolStrip.EnableToolStripItems(IsEditableAnimation);
                UpdateSnapToolbar();
            }
        }

        private bool IsEditableAnimation
        {
            get
            {
                return TargetGuiObjectID != GuiObjectID.SkeletalAnimation && TargetGuiObjectID != GuiObjectID.ShapeAnimation;
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public List<IAnimationCurve> EditableCurves
        {
            get;
            private set;
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public List<IAnimationCurve>	VisibledCurves
        {
            get {			return VisibledCurves_;	}
            set {
                VisibledCurves_ = value;
                MovableCurves = VisibledCurves.Where(x => x.IsEditable).ToList();
                if (ChangeVisibledCurves != null)
                {
                    ChangeVisibledCurves(this, EventArgs.Empty);
                }
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public List<IAnimationCurve> MovableCurves
        {
            get;
            private set;
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public IEnumerable<IAnimationCurve> AllCurves { get { return treeNodes_.Where(x => x.Tag is IAnimationCurve).Select(x => x.Tag as IAnimationCurve); } }

        // アップデート時の情報
        // アップデート時でなければnull
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public ObjectPropertyPage.UpdateFormInfo UpdateFormInfo{ get; private set; }

        private List<IAnimationCurve>	VisibledCurves_ = new List<IAnimationCurve>();
        private List<IAnimationCurve>	SelectedCurves_ = new List<IAnimationCurve>();

        public delegate int	GetFrameCountDelegate();
        public GetFrameCountDelegate	GetFrameCount;

        private static readonly CurveEditorCopyData copyData = new CurveEditorCopyData();

        public int		FrameCount{
            get{
                if (TheApp.IsInitializedMainFrame)
                {
                    Debug.Assert(GetFrameCount != null);
                    return GetFrameCount();
                }
                else
                {
                    return 0;
                }
            }
        }

        public bool IsKeyAddMode {			get { return (EditMode == EditModeType.Insert) || (EditMode == EditModeType.Add); } }

        public bool IsUseFrameSnap {		get { return tscFrameSnap.SelectedIndex > 0; } }
        public bool IsUseValueSnap {		get { return tscValueSnap.SelectedIndex > 0; } }

        public bool IsVisibleFrameValue { get; set; }//{	get { return visibleSettingPopup.cbxVisibleFrameValue.Checked;		} }
        public bool IsVisibleSlope { get; set; }//{ get { return visibleSettingPopup.cbxVisibleSlope.Checked; } }
        public bool IsVisibleMinMax { get; set; }//{ get { return visibleSettingPopup.cbxVisibleMinMax.Checked; } }
        public bool IsVisibleCurrentValue { get; set; }//{ get { return visibleSettingPopup.cbxVisibleCurrentValue.Checked; } }
        public bool IsVisibleCurveName { get; set; }//{ get { return visibleSettingPopup.cbxVisibleCurveName.Checked; } }
        public bool IsClampFrame{			get { return tsbClampFrame.Checked;	 			} }
        public bool IsAutoSplineSlope { get { return tsbAutoSplineSlope.Checked; } }
        public bool IsClampValue{
            get {
                return IsForcedToClampValue || isClampValue;
            }
        }
        public bool IsSnapToKey { get { return tsbSnapToKey.Checked; } }
        private bool isClampValue;

        public bool IsClampCurrentFrame{	get { return tsbClampCurrentFrame.Checked;	} }

        private bool IsForcedToClampValue
        {
            get
            {
                return TargetGroup.Active != null &&
                        TargetGroup.Active.ObjectID == GuiObjectID.TexturePatternAnimation &&
                        Viewer.Manager.Instance.IsConnected;
            }
        }

        public double FrameSnapFactor{		get { return IsUseFrameSnap ? double.Parse((string)tscFrameSnap.SelectedItem) : 0.0; } }
        public double ValueSnapFactor{		get { return IsUseValueSnap ? double.Parse((string)tscValueSnap.SelectedItem) : 0.0; } }

        // ツリービュー範囲選択用情報
        private TreeNode	oldSelectedNode_	= null;
        private TreeNode[]	treeNodes_			= new TreeNode[0];

        #region EditMode
        private static EditModeType editMode = EditModeType.Multi; // 初期値は複数移動

        /// <summary>
        /// 編集モードはカーブエディタに共通
        /// </summary>
        public static EditModeType EditMode
        {
            get
            {
                return editMode;
            }
            set
            {
                if (editMode != value)
                {
                    editMode = value;
                    if (EditModeChangedEvent != null)
                    {
                        EditModeChangedEvent();
                    }
                }
            }
        }

        public static event Action EditModeChangedEvent;

        private void UpdateEditModeButton()
        {
            using (var sb = new UIControlEventSuppressBlock())
            {
                tsbEditModeNear.Checked = false;
                tsbEditModeMulti.Checked = false;
                tsbEditModeScale.Checked = false;
                tsbEditModeInsert.Checked = false;
                tsbEditModeAdd.Checked = false;

                switch (editMode)
                {
                    case EditModeType.Near: tsbEditModeNear.Checked = true; break;
                    case EditModeType.Multi: tsbEditModeMulti.Checked = true; break;
                    case EditModeType.Scale: tsbEditModeScale.Checked = true; break;
                    case EditModeType.Insert: tsbEditModeInsert.Checked = true; break;
                    case EditModeType.Add: tsbEditModeAdd.Checked = true; break;
                }
            }
        }
        #endregion

        public event EventHandler ChangeSelectedCurves = null;
        public event EventHandler ChangeVisibledCurves = null;
        public event EventHandler UpdateTreeView = null;

        private readonly static List<double> snapFactors = new List<double>(new []{0.0, 0.001, 0.01, 0.1, 1.0, 10.0});

        private bool isConfigLoaded_ = false;

        public CurveEditorPanel()
        {
            InitializeComponent();

            // ボタンが押されたときの背景を青にする
            var renderer = new UIToolStripRenderer();
            ViewBottomToolStrip.Renderer = renderer;
            EditToolStrip.Renderer = renderer;
            PlayToolStrip.Renderer = renderer;
            PasteToolStrip.Renderer = renderer;
            SnapToolStrip.Renderer = renderer;

            tsbEditModeNear.Tag		= EditModeType.Near;
            tsbEditModeMulti.Tag	= EditModeType.Multi;
            tsbEditModeScale.Tag	= EditModeType.Scale;
            tsbEditModeInsert.Tag	= EditModeType.Insert;
            tsbEditModeAdd.Tag		= EditModeType.Add;
            UpdateEditModeButton();

            using (var frameBlock = new UpdateBlock(tscFrameSnap))
            using (var ValueBlock = new UpdateBlock(tscValueSnap))
            {
                foreach(double factor in snapFactors)
                {
                    tscFrameSnap.Items.Add((factor == 0.0) ? Strings.CurveEdit_NoSnap : factor.ToString());
                    tscValueSnap.Items.Add((factor == 0.0) ? Strings.CurveEdit_NoSnap : factor.ToString());
                }
            }

            tsbToHermiteKey.Tag	= InterpolationType.Hermite;
            tsbToLinearKey.Tag	= InterpolationType.Linear;
            tsbToStepKey.Tag	= InterpolationType.Step;

            tsbUnionTangent.Tag	= true;
            tsbBreakTangent.Tag	= false;

//			LoadConfig();

            EventHandler previewCurrentFrameChanged = (s, a) =>
            {
                CurveView.State.CurrentFrame = App.AppContext.PreviewCurrentFrame;
            };

            if (TheApp.IsInitializedMainFrame)
            {
                CurveView.State.CurrentFrame = App.AppContext.PreviewCurrentFrame;
            }

            EditModeChangedEvent += UpdateEditModeButton;
            App.AppContext.PauseChangedEvent += UpdatePause;
            App.AppContext.PreviewCurrentFrameChanged += previewCurrentFrameChanged;

            Disposed += delegate
            {
                EditModeChangedEvent -= UpdateEditModeButton;
                App.AppContext.PauseChangedEvent -= UpdatePause;
                App.AppContext.PreviewCurrentFrameChanged -= previewCurrentFrameChanged;
            };

            //CurveView.ContextMenuStrip = uiContextMenuStripCurveView;

/*
            tsbMoveScaleTool.Checked = KeyMoveScaleUtility.IsVisible;
            keyMoveScaleUtility.Display(KeyMoveScaleUtility.IsVisible);
*/
//            keyMoveScaleUtility.Visible = KeyMoveScaleUtility.IsVisible;
        }

        private void tsbVisibleSettingPopup_Click(object sender, EventArgs args)
        {
            var visibleSettingPopup = new VisibleSettingPopup();
            tsbVisibleSettingPopup.Enabled = false;
            visibleSettingPopup.FormClosed += (s, e) =>
            {
                tsbVisibleSettingPopup.Checked = false;
                tsbVisibleSettingPopup.Enabled = true;
            };
            visibleSettingPopup.cbxVisibleCurrentValue.Checked = IsVisibleCurrentValue;
            visibleSettingPopup.cbxVisibleSlope.Checked = IsVisibleSlope;
            visibleSettingPopup.cbxVisibleMinMax.Checked = IsVisibleMinMax;
            visibleSettingPopup.cbxVisibleFrameValue.Checked = IsVisibleFrameValue;
            visibleSettingPopup.cbxVisibleCurveName.Checked = IsVisibleCurveName;

            visibleSettingPopup.ValueChanged += (s, e) =>
            {
                IsVisibleCurrentValue = visibleSettingPopup.cbxVisibleCurrentValue.Checked;
                IsVisibleSlope = visibleSettingPopup.cbxVisibleSlope.Checked;
                IsVisibleMinMax = visibleSettingPopup.cbxVisibleMinMax.Checked;
                IsVisibleFrameValue = visibleSettingPopup.cbxVisibleFrameValue.Checked;
                IsVisibleCurveName = visibleSettingPopup.cbxVisibleCurveName.Checked;
                UpdateConfig();
                CurveView.Invalidate();
            };
            var parent = tsbVisibleSettingPopup.GetCurrentParent();
            var rect = parent.RectangleToScreen(tsbVisibleSettingPopup.Bounds);
            visibleSettingPopup.Location = new Point(rect.Right - visibleSettingPopup.Width, rect.Top - visibleSettingPopup.Height);
            visibleSettingPopup.Show(parent);
        }

        public void AddKey(List<IAnimationCurve> curves, Point pos, bool isForceInsertMode = false, float? frame = null, float? value = null)
        {
            if (CheckEditRestriction() == false)
            {
                CurveView.Invalidate();
                return;
            }

            var commandSet = new EditCommandSet();
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                foreach (var curve in curves)
                {
                    float inSlope	= 0.0f;
                    float outSlope	= 0.0f;
                    var frameSnapFactor = FrameSnapFactor;
                    var valueSnapFactor = ValueSnapFactor;
                    float mouseFrame = (float)CurveView.GetXInCurvePlane(pos.X);

                    // キースナップ有効時には倍数スナップを使わない。
                    if (IsSnapToKey)
                    {
                        frameSnapFactor = 0;
                        valueSnapFactor = 0;
                    }

                    if (frame != null)
                    {
                        mouseFrame = (float)frame;
                    }

                    // 値を計算する前にフレームをスナップ
                    if (frameSnapFactor > 0)
                    {
                        mouseFrame = (float)(Math.Round(mouseFrame / frameSnapFactor, MidpointRounding.AwayFromZero) * frameSnapFactor);
                    }

                    float mouseValue = (float)CurveView.GetYInCurvePlane(pos.Y);

                    if (isForceInsertMode || (EditMode == EditModeType.Insert))
                    {
                        if (curve.HasKeyFrame())
                        {
                            var curveAnimTarget = curve.GetAnimTarget(TargetGroup.Active);
                            mouseValue = (float)curve.GetValue(mouseFrame, curveAnimTarget.pre_wrap, curveAnimTarget.post_wrap);
                            curve.MakeHermiteSmoothSlope(mouseFrame, curveAnimTarget.pre_wrap, curveAnimTarget.post_wrap, out inSlope, out outSlope);
                        }
                        else
                        {
                            mouseValue = curve.GetDefaultValue(TargetGroup.Active);
                        }
                    }

                    if (value != null)
                    {
                        mouseValue = (float)value;
                    }

                    var keys = new List<KeyFrame>(curve.KeyFrames);
                    var key  = curve.CreateKey(mouseFrame, mouseValue, inSlope, outSlope, frameSnapFactor, valueSnapFactor, IsClampValue, IsClampFrame, FrameCount);
                    {
                        keys.Add(key);
                    }

                    commandSet.Add(
                        AnimationCurveEditCommand.Create(
                            TargetGroup,
                            TargetGuiObjectID,
                            curve,
                            keys
                        )
                    );

                    //SetKeyParam(key);
                };
            }

            if (commandSet.CommandCount > 0)
            {
                commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        public void AddKey(List<IAnimationCurve> curves, float frame, float value)
        {
            if (CheckEditRestriction() == false)
            {
                CurveView.Invalidate();
                return;
            }

            var commandSet = new EditCommandSet();
            {
                var frameSnapFactor = FrameSnapFactor;
                var valueSnapFactor = ValueSnapFactor;

                // キースナップ有効時には倍数スナップを使わない。
                if (IsSnapToKey)
                {
                    frameSnapFactor = 0;
                    valueSnapFactor = 0;
                }

                // 値を計算する前にフレームをスナップ
                if (frameSnapFactor > 0)
                {
                    frame = (float)(Math.Round(frame / frameSnapFactor, MidpointRounding.AwayFromZero) * frameSnapFactor);
                }

                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                foreach (var curve in curves)
                {
                    var keys = new List<KeyFrame>(curve.KeyFrames);
                    var key = curve.CreateKey(frame, value, 0.0f, 0.0f, frameSnapFactor, valueSnapFactor, IsClampValue, IsClampFrame, FrameCount);
                    {
                        keys.Add(key);
                    }

                    commandSet.Add(
                        AnimationCurveEditCommand.Create(
                            TargetGroup,
                            TargetGuiObjectID,
                            curve,
                            keys
                        )
                    );

                    //SetKeyParam(key);
                }
            }

            if (commandSet.CommandCount > 0)
            {
                commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        public void AddColorKey(List<IAnimationCurve> curves, float frame, Color color, float factor)
        {
            if (CheckEditRestriction() == false)
            {
                CurveView.Invalidate();
                return;
            }

            var commandSet = new EditCommandSet();
            {
                var rgba = (new RgbaColor(color, factor)).ToArray();

                var frameSnapFactor = FrameSnapFactor;
                var valueSnapFactor = ValueSnapFactor;

                // キースナップ有効時には倍数スナップを使わない。
                if (IsSnapToKey)
                {
                    frameSnapFactor = 0;
                    valueSnapFactor = 0;
                }

                // 値を計算する前にフレームをスナップ
                if (frameSnapFactor > 0)
                {
                    frame = (float)(Math.Round(frame / frameSnapFactor, MidpointRounding.AwayFromZero) * frameSnapFactor);
                }

                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                foreach (var curve in curves)
                {
                    List<KeyFrame> keys = null;

                    // TODO: まとめたい
                    if (curve is ShaderParameterAnimationCurveTreeNodeInfo)
                    {
                        var info = curve as ShaderParameterAnimationCurveTreeNodeInfo;

                        keys = new List<KeyFrame>(curve.KeyFrames);
                        var key = curve.CreateKey(frame, rgba[info.ComponentIndex], 0.0f, 0.0f, frameSnapFactor, valueSnapFactor, IsClampValue, IsClampFrame, FrameCount);
                        {
                            keys.Add(key);
                        }
                    }
                    else if (curve is ShaderParameterAnimationCurveTreeNodeInfo2)
                    {
                        var info = curve as ShaderParameterAnimationCurveTreeNodeInfo;

                        keys = new List<KeyFrame>(curve.KeyFrames);
                        var key = curve.CreateKey(frame, rgba[info.ComponentIndex], 0.0f, 0.0f, frameSnapFactor, valueSnapFactor, IsClampValue, IsClampFrame, FrameCount);
                        {
                            keys.Add(key);
                        }
                    }
                    else
                    {
                        Debug.Assert(false);
                    }

                    commandSet.Add(
                        AnimationCurveEditCommand.Create(
                            TargetGroup,
                            TargetGuiObjectID,
                            curve,
                            keys
                        )
                    );

                    //SetKeyParam(key);
                }
            }

            if (commandSet.CommandCount > 0)
            {
                commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        public void SetKeyParam()
        {
            KeyParams keyParams = new KeyParams();
            {
                foreach(var visibleCurve in VisibledCurves)
                {
                    foreach(var key in visibleCurve.KeyFrames)
                    {
                        if (key.AnySelected)
                        {
                            keyParams.SetKeyFrame(key, visibleCurve);
                        }
                    }
                }
            }

            SetKeyParam(keyParams);
        }

        /// <summary>
        /// 複数のキーフレームの共通部分を割り出す
        /// </summary>
        public class KeyParams
        {
            private bool empty = true;
            private int sameCount = 4;

            public bool SameFrames { get; private set; }
            public bool SameValues { get; private set; }
            public bool SameInSlopes { get; private set; }
            public bool SameOutSlopes { get; private set; }
            public bool SameIsUnionSlope { get; private set; }

            public float Frame { get; private set; }
            public float Value { get; private set; }
            public float InSlope { get; private set; }
            public float OutSlope { get; private set; }
            public bool IsUnionSlope { get; private set; }

            public void SetKeyFrame(KeyFrame targetKey, IAnimationCurve curve)
            {
                if (sameCount == 0)
                {
                    return;
                }

                if (empty)
                {
                    SameFrames = true;
                    SameValues = true;
                    if (curve.CurveInterpolationType != InterpolationType.Hermite)
                    {
                        SameInSlopes = false;
                        SameOutSlopes = false;
                        SameIsUnionSlope = false;
                        sameCount = 2;
                    }
                    else
                    {
                        SameInSlopes = true;
                        SameOutSlopes = true;
                        SameIsUnionSlope = true;
                    }

                    Frame = targetKey.Frame;
                    Value = targetKey.Value;
                    InSlope = targetKey.InSlope;
                    OutSlope = targetKey.OutSlope;
                    IsUnionSlope = targetKey.IsUnionSlope;
                    empty = false;
                }
                else
                {
                    if (SameFrames && Frame != targetKey.Frame)
                    {
                        SameFrames = false;
                        sameCount--;
                    }

                    if (SameValues && Value != targetKey.Value)
                    {
                        SameValues = false;
                        sameCount--;
                    }

                    if (SameInSlopes && (InSlope != targetKey.InSlope || curve.CurveInterpolationType != InterpolationType.Hermite))
                    {
                        SameInSlopes = false;
                        sameCount--;
                    }

                    if (SameOutSlopes && (OutSlope != targetKey.OutSlope || curve.CurveInterpolationType != InterpolationType.Hermite))
                    {
                        SameOutSlopes = false;
                        sameCount--;
                    }

                    if (SameIsUnionSlope && (IsUnionSlope != targetKey.IsUnionSlope || curve.CurveInterpolationType != InterpolationType.Hermite))
                    {
                        SameIsUnionSlope = false;
                        sameCount--;
                    }
                }
            }
        }

        /// <summary>
        /// ツールボックスのキー情報を更新
        /// </summary>
        public void SetKeyParam(KeyParams keyParams)
        {
            using (var suppressBlock = new UIControlEventSuppressBlock())
            {
                // tscFrame
                if (keyParams.SameFrames)
                {
                    tscFrame.Value = keyParams.Frame;
                }
                else
                {
                    tscFrame.ClearText();
                }

                // tscValue
                if (keyParams.SameValues)
                {
                    tscValue.Value = keyParams.Value;
                }
                else
                {
                    tscValue.ClearText();
                }

                // tscInSlope
                if (keyParams.SameInSlopes)
                {
                    tscInSlope.Value = keyParams.InSlope;
                }
                else
                {
                    tscInSlope.ClearText();
                }

                // tscOutSlope
                if (keyParams.SameOutSlopes)
                {
                    tscOutSlope.Value = keyParams.OutSlope;
                }
                else
                {
                    tscOutSlope.ClearText();
                }

                // tsbBreakTangent と tsbUnionTangent
                if (keyParams.SameIsUnionSlope)
                {
                    if (keyParams.IsUnionSlope)
                    {
                        tsbBreakTangent.Checked = false;
                        tsbUnionTangent.Checked = true;
                    }
                    else
                    {
                        tsbBreakTangent.Checked = true;
                        tsbUnionTangent.Checked = false;
                    }
                }
                else
                {
                    tsbBreakTangent.Checked = false;
                    tsbUnionTangent.Checked = false;
                }
            }
        }

        private void LoadConfig()
        {
            using (var suppressBlock = new UIControlEventSuppressBlock())
            {
                int frameSnapIndex = snapFactors.IndexOf(ApplicationConfig.Setting.CurveEditor.FrameSnapFactor);
                tscFrameSnap.SelectedIndex = (frameSnapIndex != -1) ? Math.Min(tscFrameSnap.Items.Count - 1, frameSnapIndex) : 0;

                int valueSnapindex = snapFactors.IndexOf(ApplicationConfig.Setting.CurveEditor.ValueSnapFactor);
                tscValueSnap.SelectedIndex = (valueSnapindex != -1) ? Math.Min(tscValueSnap.Items.Count - 1, valueSnapindex) : 0;

                tspPasteMethod.SelectedIndex	= (int)ApplicationConfig.Setting.CurveEditor.PasteMethod;
                tspPastePosition.SelectedIndex	= (int)ApplicationConfig.Setting.CurveEditor.PastePosition;

                IsVisibleFrameValue = ApplicationConfig.Setting.CurveEditor.IsVisibleFrameValue;
                IsVisibleSlope = ApplicationConfig.Setting.CurveEditor.IsVisibleSlope;
                IsVisibleMinMax = ApplicationConfig.Setting.CurveEditor.IsVisibleMinMax;
                IsVisibleCurrentValue = ApplicationConfig.Setting.CurveEditor.IsVisibleCurrentValue;
                IsVisibleCurveName = ApplicationConfig.Setting.CurveEditor.IsVisibleCurveName;

                tsbClampCurrentFrame.Checked	= ApplicationConfig.Setting.CurveEditor.IsClampCurrentFrame;
                isClampValue					= ApplicationConfig.Setting.CurveEditor.IsClampValue;
                tsbClampFrame.Checked			= ApplicationConfig.Setting.CurveEditor.IsClampFrame;
                tsbAutoSplineSlope.Checked = ApplicationConfig.Setting.CurveEditor.IsAutoSplineSlope;

                tsbSnapToKey.Checked            = ApplicationConfig.Setting.CurveEditor.IsSnapToOtherKey;
                UpdateToolBarLayout();
            }
        }

        private void UpdateToolBarLayout()
        {
            // ツールバー位置

            // Suspend するとずれる
            //TopStripPanel.SuspendLayout();
            var stripInfo = new[]{
                        new {Strip = EditToolStrip,		Location = ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Edit},
                        new {Strip = FitToolStrip,		Location = ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Fit},
                        new {Strip = KeyFrameToolStrip,	Location = ApplicationConfig.Setting.CurveEditor.ToolStripLocation.KeyFrame},
                        new {Strip = PlayToolStrip,		Location = ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Play},
                        new {Strip = PasteToolStrip, Location = ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Paste},
                        new {Strip = SnapToolStrip, Location = ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Snap},
                    };

            TopStripPanel.Controls.Clear();
            var rows = from strip in stripInfo
                       group strip by strip.Location.Location.Y;
            foreach (var row in rows.OrderBy(x => x.Key))
            {
                if (row.Any(x => x.Location.Collapsed))
                {
                    foreach (var s in row.OrderByDescending(x => x.Location.Location.X))
                    {
                        TopStripPanel.Join(s.Strip, s.Location.Location);
                    }
                }
                else
                {
                    foreach (var s in row.OrderBy(x => x.Location.Location.X))
                    {
                        TopStripPanel.Join(s.Strip, s.Location.Location);
                    }
                }
            }
            /*
            foreach (var s in stripInfo.OrderBy(x => x.Location.Y * 100000 + x.Location.X))
            {
                TopStripPanel.Join(s.Strip, s.Location);
            }*/

            //TopStripPanel.ResumeLayout();
            //TopStripPanel.PerformLayout();
            UpdateSnapToolbar();
        }

        public event EventHandler TreeViewAfterCheck = null;
        public event EventHandler UpdateSelectedCurve = null;

        private AnimationObjectPropertyPanel animationObjectPropertyPanel_;

        public void Initialize(AnimationObjectPropertyPanel animationObjectPropertyPanel, GuiObjectID targetGuiObjectID)
        {
            TargetGuiObjectID = targetGuiObjectID;

            CurveTreeView = animationObjectPropertyPanel.CurveTreeView;
            CurveTreeView.ContextMenuPopup	+= CurveTreeView_ContextMenuPopup;
            CurveTreeView.AfterCheck		+= CurveTreeView_AfterCheck;
            CurveTreeView.MouseDown 		+= CurveTreeView_MouseDown;
            CurveTreeView.DoubleClick		+= CurveTreeView_DoubleClick;
            CurveTreeView.InitializeForm(this);

            CurveView.ContextMenuPopup += CurveView_ContextMenuPopup;
            CurveView.InitializeForm(this);

            animationObjectPropertyPanel.AllShowNodeChanged += tsmAllShowNode_CheckedChanged;
            animationObjectPropertyPanel.SortByAlphabeticalChanged += tsmSortNode_CheckedChanged;
            animationObjectPropertyPanel.ShaderParamAnimGroupCondShowChanged	+= tsmShaderParamAnimGroupCondShow_CheckedChanged;
            animationObjectPropertyPanel.OnlyTargetShowNodeChanged				+= tsmOnlyTargetShowNode_CheckedChanged;
            animationObjectPropertyPanel.ShowKeyedMaterialChanged               += tsmShowKeyedMaterial_CheckedChanged;
            animationObjectPropertyPanel.ResetNodeExpandClick					+= tsmResetNodeExpand_Click;
            animationObjectPropertyPanel.OpenNodeExpandClick					+= tsmOpenNodeExpand_Click;
            animationObjectPropertyPanel.VisibleAllNodeClick					+= tsmVisibleAllNode_Click;
            animationObjectPropertyPanel.InvisibleAllNodeClick					+= tsmInvisibleAllNode_Click;
            animationObjectPropertyPanel.SearchTextChanged                      += SearchTextChanged;

            CurveView.PreviewKeyDown	+= CurveView_PreviewKeyDown;
            CurveTreeView.KeyUp += CurveView_KeyUp;
            CurveView.KeyUp += CurveView_KeyUp;
            CurveTreeView.KeyDown		+= CurveEditorPanel_KeyDown;
            CurveView.KeyDown	  		+= CurveEditorPanel_KeyDown;
            animationObjectPropertyPanel.CategoryView.KeyUp += CurveView_KeyUp;
            animationObjectPropertyPanel.CategoryView.KeyDown += CurveEditorPanel_KeyDown;
            CurveView.State.MouseDoubleClick += (e) => CurveView_MouseDoubleClick(this, e);
            tscFrameSnap.KeyDown += CurveEditorPanel_KeyDown;
            tscFrameSnap.KeyUp += CurveView_KeyUp;
            tscValueSnap.KeyDown += CurveEditorPanel_KeyDown;
            tscValueSnap.KeyUp += CurveView_KeyUp;
            tspPasteMethod.KeyDown += CurveEditorPanel_KeyDown;
            tspPasteMethod.KeyUp += CurveView_KeyUp;
            tspPastePosition.KeyDown += CurveEditorPanel_KeyDown;
            tspPastePosition.KeyUp += CurveView_KeyUp;

            animationObjectPropertyPanel.PreWrapChanged += cmbPreWrap_SelectedIndexChanged;
            animationObjectPropertyPanel.PostWrapChanged += cmbPostWrap_SelectedIndexChanged;
            animationObjectPropertyPanel.CompressEnableChanged += chkCompressEnable_CheckedChanged;
            animationObjectPropertyPanel.BinarizeVisibilityChanged += chkBinarizeVisibility_CheckedChanged;
            UpdateSelectedCurve += animationObjectPropertyPanel.UpdateCurveElement;
            animationObjectPropertyPanel.SetPanelGuiObjectID(targetGuiObjectID);

            animationObjectPropertyPanel_ = animationObjectPropertyPanel;
        }

        public void InvalidateCurveView()
        {
            CurveView.Invalidate();
        }

        public void UpdateForm(ObjectPropertyPage.UpdateFormInfo updateFormInfo)
        {
            /*
            if (isConfigLoaded_ == false)
            {
                //BeginInvoke((Action)(() => LoadConfig()));
                LoadConfig();
                isConfigLoaded_ = true;
            }*/

            if (IsForcedToClampValue)
            {
                tsbClampValue.Enabled = false;
                tsbClampValue.Checked = true;
            }
            else
            {
                tsbClampValue.Enabled = true;
                tsbClampValue.Checked = isClampValue;
            }
            UpdateFormInfo = updateFormInfo;
            {
                UpdatePause(true);
                UpdateCurveTreeView();
                MakeVisibledCurves();
                UpdateCurveElement();
                SetKeyParam();
                InvalidateCurveView();
                // 新規に開いたときは全表示にする
                if (UpdateFormInfo.TargetOrPageChanged && UpdateFormInfo.IsPageCreated)
                {
                    animationObjectPropertyPanel_.FitCurveTreeViewWidth();
                }
            }

            if (FormUpdated != null)
            {
                FormUpdated(this, EventArgs.Empty);
            }


            var editable = TargetGuiObjectID != GuiObjectID.SkeletalAnimation
                           && TargetGuiObjectID != GuiObjectID.ShapeAnimation;
            if (editable)
            {
                tsbMoveScaleTool.Enabled = true;
                tsbMoveScaleTool.Checked = KeyMoveScaleUtility.IsVisible;
                keyMoveScaleUtility.Display(KeyMoveScaleUtility.IsVisible);
            }
            else
            {
                tsbMoveScaleTool.Enabled = false;
                keyMoveScaleUtility.Visible = false;
            }

            UpdateFormInfo = null;
        }

        // 更新通知
        public event EventHandler FormUpdated;

        public void UpdateFormOnPageCreatedInternal()
        {
            if (isConfigLoaded_ == false)
            {
                //BeginInvoke((Action)(() => LoadConfig()));
                LoadConfig();
                isConfigLoaded_ = true;
            }

            // 最小化されているとサイズが正しく取れないので後回し
            if (this.IsHandleCreated)
            {
                BeginInvoke((Action)(() =>
                {
                    FitAllViewIn(VisibledCurves, true);
                }));
            }
            else
            {
                FitAllViewIn(VisibledCurves, true);
            }
        }

        private bool watchToolStripMove = false;
        public void BeforePageDeactivated()
        {
            watchToolStripMove = false;
        }

        public void AfterPageActivated()
        {
            watchToolStripMove = true;
            using (var suppressBlock = new UIControlEventSuppressBlock())
            {
                UpdateToolBarLayout();
            }
        }

        public bool CheckEditRestriction()
        {
            // 複数選択時は処理させない
            if (TargetGroup.GetObjects(TargetGuiObjectID).Count >= 2)
            {
                UIMessageBox.Warning(Strings.CurveEdit_MultiSelectOpration);
                return false;
            }

            return true;
        }

        public void VisibleAllNode()
        {
//			CurveTreeView.AfterCheck -= CurveTreeView_AfterCheck;		// TODO:効かない？
            suspendAfterCheck_ = true;
            {
                CurveTreeView.VisibleAllNode();
            }
//			CurveTreeView.AfterCheck += CurveTreeView_AfterCheck;		// TODO:効かない？
            suspendAfterCheck_ = false;

            MakeVisibledCurves();
        }

        public void InvisibleAllNode()
        {
//			CurveTreeView.AfterCheck -= CurveTreeView_AfterCheck;		// TODO:効かない？
            suspendAfterCheck_ = true;
            {
                CurveTreeView.InvisibleAllNode();
            }
//			CurveTreeView.AfterCheck += CurveTreeView_AfterCheck;		// TODO:効かない？
            suspendAfterCheck_ = false;

            MakeVisibledCurves();
        }

        private void UpdateCurveTreeView()
        {
            CurveTreeView.AfterCheck -= CurveTreeView_AfterCheck;
            {
                if (UpdateTreeView != null)
                {
                    UpdateTreeView(this, EventArgs.Empty);
                }

                UpdateTreeNodeIndices();
                var s = SelectedCurves;

                UpdateTreeViewSelected();

                CurveTreeView.Invalidate();
                UpdateNoNodeWarning();
            }
            CurveTreeView.AfterCheck += CurveTreeView_AfterCheck;
        }

        private void UpdatePause(bool stateChanged)
        {
            var animation = TargetGroup.Active as AnimationDocument;
            if (animation != null)
            {
                PlayToolStrip.Visible = true;
                tsbPause.Enabled = animation.PreviewingAnimationSets.Count == 1;
                double frame = 0;
                bool pausing = animation.PreviewingAnimationSets.Count == 1 && animation.PauseFrames.TryGetValue(animation.PreviewingAnimationSets.First(), out frame);
                tsbPause.Checked = pausing;
                CurveView.State.SetPause(pausing, frame);
            }
            else
            {
                PlayToolStrip.Visible = false;
                CurveView.State.SetPause(false, 0);
            }
        }

        // treeNodes_ を更新します
        private void UpdateTreeNodeIndices()
        {
            int index = 0;
            var treeNodeIndexDict	= new Dictionary<int, TreeNode>();
            {
                // 深さ優先でインデックスをふる
                foreach(TreeNode childNode in CurveTreeView.Nodes)
                {
                    UpdateTreeNodeIndicesInternal(childNode, treeNodeIndexDict, ref index);
                }
            }

            // 配列に変換する
            treeNodes_ = new TreeNode[index];
            for(int i = 0;i != index;++ i)
            {
                Debug.Assert(treeNodeIndexDict.ContainsKey(i));
                treeNodes_[i] = treeNodeIndexDict[i];
            }

            Debug.Assert(treeNodes_.All(x => x != null));
        }

        private void UpdateTreeNodeIndicesInternal(TreeNode node, Dictionary<int, TreeNode> dst, ref int index)
        {
            dst[index] = node;

            ++ index;

            foreach(TreeNode childNode in node.Nodes)
            {
                UpdateTreeNodeIndicesInternal(childNode, dst, ref index);
            }
        }

        private bool InUpdateTreeViewVisibled = false;

        internal void UpdateTreeViewVisibled()
        {
            if (!InUpdateTreeViewVisibled)
            {
                // 再帰呼び出しを防止
                InUpdateTreeViewVisibled = true;

                {
                    var newVisibledCurves = new List<IAnimationCurve>();
                    var hashSet = new HashSet<IAnimationCurve>(VisibledCurves, new IAnimationComparer());
                    foreach (var node in CurveTreeView.AllTreeNodesLeafFirst().OfType<CurveTreeNode>())
                    {
                        //var isVisibled = (node.Tag != null) && hashSet.Contains((IAnimationCurve)node.Tag);
                        //var isChecked = node.Checked || node.Nodes.OfType<TreeNode>().All(x => x.Checked);
                        var isChecked = node.Checked || (node.Nodes.Count > 0 && node.Nodes.OfType<TreeNode>().All(x => x.Checked));
                        CheckNodeIndividually(node, isChecked);
                    }
                }
            }

            InUpdateTreeViewVisibled = false;
        }

        private bool InUpdateTreeViewSelected = false;
        private void UpdateTreeViewSelected()
        {
            if (!InUpdateTreeViewSelected)
            {
                // 再帰呼び出しを防止
                InUpdateTreeViewSelected = true;

                {
                    List<IAnimationCurve> newSelectedCurves = new List<IAnimationCurve>();
                    HashSet<IAnimationCurve> hashSet = new HashSet<IAnimationCurve>(SelectedCurves, new IAnimationComparer());

                    foreach (var node in CurveTreeView.AllTreeNodesLeafFirst())
                    {
                        var isSelected = (node.Tag != null) && hashSet.Contains((IAnimationCurve)node.Tag);

                        CurveTreeView.NodeSelectState selectState;
                        CurveTreeView.HasCurveType hasKeyFrame;
                        CurveTreeView.Editable nodeEditable;
                        if (node.Tag == null)
                        {
                            bool allSelected = true;
                            bool notSelected = true;
                            bool allCurve = true;
                            bool notCurve = true;
                            nodeEditable = CurveTreeView.Editable.Yes;
                            foreach (var item in node.Nodes.OfType<TreeNode>())
                            {
                                var select = CurveTreeView.GetNodeState(item);
                                switch (select)
                                {
                                    case CurveTreeView.NodeSelectState.Unselected:
                                        allSelected = false;
                                        break;
                                    case CurveTreeView.NodeSelectState.Selected:
                                        notSelected = false;
                                        break;
                                    case CurveTreeView.NodeSelectState.SemiSelected:
                                        allSelected = false;
                                        notSelected = false;
                                        break;
                                }
                                var hasCurve = CurveTreeView.GetHasCurveType(item);
                                switch (hasCurve)
                                {
                                    case CurveTreeView.HasCurveType.NoCurve:
                                        allCurve = false;
                                        break;
                                    case CurveTreeView.HasCurveType.SomeCurve:
                                        allCurve = false;
                                        notCurve = false;
                                        break;
                                    case CurveTreeView.HasCurveType.AllCurve:
                                        notCurve = false;
                                        break;
                                }
                                var editable = CurveTreeView.GetEditable(item);
                                switch (editable)
                                {
                                    case CurveTreeView.Editable.No:
                                        if (nodeEditable == CurveTreeView.Editable.Yes)
                                        {
                                            nodeEditable = CurveTreeView.Editable.No;
                                        }
                                        break;
                                    case CurveTreeView.Editable.NoAndSelected:
                                        nodeEditable = CurveTreeView.Editable.NoAndSelected;
                                        break;
                                    case CurveTreeView.Editable.Yes:
                                        break;
                                }
                            }

                            selectState = notSelected ? CurveTreeView.NodeSelectState.Unselected : allSelected ? CurveTreeView.NodeSelectState.Selected : CurveTreeView.NodeSelectState.SemiSelected;
                            hasKeyFrame = notCurve ? CurveTreeView.HasCurveType.NoCurve : allCurve ? CurveTreeView.HasCurveType.AllCurve : CurveTreeView.HasCurveType.SomeCurve;
                        }
                        else
                        {
                            selectState = isSelected ? CurveTreeView.NodeSelectState.Selected : CurveTreeView.NodeSelectState.Unselected;
                            hasKeyFrame = ((IAnimationCurve)node.Tag).KeyFrames.Count > 0 ? CurveTreeView.HasCurveType.AllCurve : PropertyEdit.CurveTreeView.HasCurveType.NoCurve;
                            nodeEditable = ((CurveTreeNode)node).Info.IsBound ? CurveTreeView.Editable.Yes : isSelected ? CurveTreeView.Editable.NoAndSelected : CurveTreeView.Editable.No;
                        }

                        CurveTreeView.SetNodeState(node, selectState, hasKeyFrame, nodeEditable);

                        if (isSelected)
                        {
                            newSelectedCurves.Add((IAnimationCurve)node.Tag);
                        }
                    }

                    // 付随する処理を実行しない
                    SelectedCurves = newSelectedCurves;

                    // 要素表示を更新
                    UpdateCurveElement();
                }
                InUpdateTreeViewSelected = false;
            }
        }

        private class IAnimationComparer : IEqualityComparer<IAnimationCurve>
        {
            public bool Equals(IAnimationCurve c1, IAnimationCurve c2)
            {
                return (c1.ParentName == c2.ParentName) &&
                       (c1.Name == c2.Name) &&
                       (c1.ComponentIndex == c2.ComponentIndex);
            }

            public int GetHashCode(IAnimationCurve c1)
            {
                int code = 0;
                code = ShaderTypeUtility.MixHash(code, c1.ParentName);
                code = ShaderTypeUtility.MixHash(code, c1.Name);
                code = ShaderTypeUtility.MixValueHash(code, c1.ComponentIndex);
                return code;
            }
        }


        // ノード選択状態を作る
        private CurveTreeView.NodeSelectState MakeNodeSelectState(TreeNode node)
        {
            int selectCount   = 0;
            int unselectCount = 0;

            MakeNodeSelectStateInternal(node, ref selectCount, ref unselectCount);

            return
                ((selectCount >  0) && (unselectCount >  0)) ? CurveTreeView.NodeSelectState.SemiSelected :
                ((selectCount == 0) && (unselectCount == 0)) ? CurveTreeView.NodeSelectState.Unselected :
                (selectCount > 0)							 ? CurveTreeView.NodeSelectState.Selected :
                                                               CurveTreeView.NodeSelectState.Unselected;
        }

        private void MakeNodeSelectStateInternal(TreeNode node, ref int selectCount, ref int unselectCount)
        {
            if (node.Tag == null)
            {
                foreach(TreeNode childNode in node.Nodes)
                {
                    MakeNodeSelectStateInternal(childNode, ref selectCount, ref unselectCount);
                }
            }
            else
            {
                var curve = node.Tag as IAnimationCurve;
                if (SelectedCurves.Any(x => x.IsSameCurve(curve)))
                {
                    ++ selectCount;
                }
                else
                {
                    ++ unselectCount;
                }
            }
        }

        private CurveTreeView.HasCurveType HasKeyFrame(TreeNode node)
        {
            int allCount = 0;
            int noCount  = 0;

            HasKeyFrameInternal(node, ref allCount, ref noCount);

            return
                ((allCount >  0) && (noCount >  0)) ? CurveTreeView.HasCurveType.SomeCurve :
                ((allCount >  0) && (noCount == 0))	? CurveTreeView.HasCurveType.AllCurve :
                                                      CurveTreeView.HasCurveType.NoCurve;
        }

        private void HasKeyFrameInternal(TreeNode node, ref int allCount, ref int noCount)
        {
            if (node.Tag == null)
            {
                foreach(TreeNode childNode in node.Nodes)
                {
                    HasKeyFrameInternal(childNode, ref allCount, ref noCount);
                }
            }
            else
            {
                var curve = node.Tag as IAnimationCurve;
                if (curve.HasKeyFrame())
                {
                    ++ allCount;
                }
                else
                {
                    ++ noCount;
                }
            }
        }

        private CurveTreeView.Editable Editable(CurveTreeNode node)
        {
            CurveTreeView.Editable editable = CurveTreeView.Editable.Yes;
            if (node.Info.AnimationCurve != null)
            {
                if (!node.Info.IsBound)
                {
                    if (SelectedCurves.Any(x => x.IsSameCurve(node.Info.AnimationCurve)))
                    {
                        return CurveTreeView.Editable.NoAndSelected;
                    }
                    editable = CurveTreeView.Editable.No;
                }
            }

            foreach (CurveTreeNode child in node.Nodes)
            {
                switch (Editable(child))
                {
                    case PropertyEdit.CurveTreeView.Editable.NoAndSelected:
                        return PropertyEdit.CurveTreeView.Editable.NoAndSelected;
                    case PropertyEdit.CurveTreeView.Editable.No:
                        editable = PropertyEdit.CurveTreeView.Editable.No;
                        break;
                }
            }

            return editable;
        }

        private bool suspendAfterCheck_ = false;
        private void CurveTreeView_AfterCheck(object sender, TreeViewEventArgs e)
        {
            if (suspendAfterCheck_ == false)
            {
                suspendAfterCheck_ = true;
                {
                    if(e.Action != TreeViewAction.Unknown)
                    {
                        // 親のチェック状態を伝搬させる
                        if(e.Node.Nodes.Count > 0)
                        {
                            CheckAllChildNodes(e.Node, e.Node.Checked);
                        }

                        // 親に伝搬する
                        CheckAncestor(e.Node, e.Node.Checked);
                    }

                    var allCheckedCurveNodes = new List<TreeNode>();
                    {
                        foreach (TreeNode node in CurveTreeView.Nodes)
                        {
                            allCheckedCurveNodes.AddRange(CurveEditorUtility.CollectNode(node, x => x.Checked));
                        }
                    }

                    var	visibledCurves = new List<IAnimationCurve>();
                    {
                        allCheckedCurveNodes.ForEach(x => visibledCurves.Add((IAnimationCurve)x.Tag));
                    }


                    // VisibledCurvesを表示選択順に保つ
                    var newVisibled = visibledCurves.Except(VisibledCurves).ToList();
                    VisibledCurves = VisibledCurves.Intersect(visibledCurves).Union(newVisibled).ToList();

                    newVisibled.ForEach(curve => SelectAllKeys(curve, false));
                    SelectedCurves =
                        SelectedCurves.Intersect(visibledCurves)
                            .Union(newVisibled.Where(curve => curve.KeyFrames.Any(key => key.AnySelected)))
                            .ToList();
                }

                if (TreeViewAfterCheck != null)
                {
                    TreeViewAfterCheck(this, EventArgs.Empty);
                }

                suspendAfterCheck_ = false;
            }
        }

        // 個別のノードのチェック状態のみを更新
        public void CheckNodeIndividually(CurveTreeNode node, bool check)
        {
            suspendAfterCheck_ = true;
            node.Checked = check;
            suspendAfterCheck_ = false;
        }

        private bool isInKeyDown	= false;

        private void CurveView_PreviewKeyDown(object sender, PreviewKeyDownEventArgs e)
        {
            if (CurveView.IsDragging)
            {
                return;
            }

            if (isInKeyDown == false)
            {
                if ((e.Alt == false) && (e.Control == false) && (e.Shift == false))
                {
                    if (e.KeyData == Keys.Left || e.KeyData == Keys.Right)
                    {
                        ShiftKeySelection(e.KeyData);
                        e.IsInputKey = true;
                        isInKeyDown = true;
                    }
                }
            }
        }

        private void CurveView_KeyUp(object sender, KeyEventArgs e)
        {
            // スナップ・インサートの終了はドラッグ中でも可能にする
            if (e.KeyCode == Keys.I)
            {
                EndEditModeTypeInsert();
            }

            if (CurveView.IsDragging)
            {
                return;
            }

            isInKeyDown = false;

        }

        // キーの選択状態をシフトする
        private void ShiftKeySelection(Keys keyData)
        {
            if (keyData != Keys.Left && keyData != Keys.Right) return;

            foreach (var curve in SelectedCurves)
            {
                if (!curve.HasKeyFrame() || curve.KeyFrames.Count < 2)
                {
                    continue;
                }

                var selection = curve.KeyFrames.Select(x => x.Selected).ToArray();
                if (!selection.Any())
                {
                    continue;
                }

                var selIndex = keyData == Keys.Right ? -1 : 1;
                for (var i = 0; i < selection.Length; i++)
                {
                    selIndex = (selIndex + selection.Length) % selection.Length;
                    curve.KeyFrames[i].Selected = selection[selIndex];
                    selIndex++;
                }
            }

            // 選択キーでツールボックスのキー情報を更新。
            SetKeyParam();

            CurveView.Invalidate();
        }

        private void CurveView_MouseDoubleClick(object sender, EventArgs e)
        {
            foreach(var selectedCurve in SelectedCurves)
            {
                var treeNode = CurveTreeView.AllCurveNodes.FirstOrDefault(x => (x.Tag as IAnimationCurve).IsSame(selectedCurve));
                if (treeNode != null)
                {
                    treeNode.EnsureVisible();
                    CurveTreeView.SelectedNode = treeNode;
                    break;
                }
            }
        }

        private void CurveEditorPanel_KeyDown(object sender, KeyEventArgs e)
        {
            bool isPushedControl = (Control.ModifierKeys & Keys.Control) != 0;
            // キースナップのトグルはドラッグ中でも可能にする
            if (!isPushedControl && e.KeyCode == Keys.C)
            {
                tsbSnapToKey.Checked = !tsbSnapToKey.Checked;
            }

            if (CurveView.IsDragging)
            {
                return;
            }

            if (isInKeyDown == false)
            {
                if ((e.Alt == false) && (e.Control == false) && (e.Shift == false))
                {
                    isInKeyDown = true;
                }

                bool isPushedAlt = (Control.ModifierKeys & Keys.Alt) != 0;
                switch (e.KeyCode)
                {
                    case Keys.S:
                        AddKeyCurrentFrame();
                        break;
                    case Keys.Delete:
                        DeleteSelectedKeys();
                        break;
                    case Keys.Back:
                        DeleteSelectedKeys();
                        break;

                    case Keys.C:
                        if (isPushedControl)
                        {
                            CopyKeys();
                        }
                        break;
                    case Keys.V:
                        if (isPushedControl)
                        {
                            PasteKeys();
                        }
                        break;
                    case Keys.X:
                        if (isPushedControl)
                        {
                            CutKeys();
                        }
                        break;

                    case Keys.A:
                        FitAllViewIn(VisibledCurves, false, IsVisibilityAnimation());
                        break;
                    case Keys.F:
                        FitSelectedViewIn(IsVisibilityAnimation());
                        break;
                    case Keys.Add:
                        if (isPushedControl)
                        {
                            ScaleView(-50.0);
                        }
                        break;
                    case Keys.Subtract:
                        if (isPushedControl)
                        {
                            ScaleView(+50.0);
                        }
                        break;

                    case Keys.N:
                        SelectNextCurve(-1);
                        break;
                    case Keys.M:
                        SelectNextCurve(+1);
                        break;
                    case Keys.Oemcomma:
                        if (!isPushedAlt)
                        {
                            SelectNextKey(-1);
                        }
                        break;
                    case Keys.OemPeriod:
                        if (!isPushedAlt)
                        {
                            SelectNextKey(+1);
                        }
                        break;
                    case Keys.W:
                        EditMode = EditModeType.Multi;
                        break;
                    case Keys.E:
                        EditMode = EditModeType.Near;
                        break;
                    case Keys.R:
                        EditMode = EditModeType.Scale;
                        break;
                    case Keys.I:
                        BeginEditModeTypeInsert();
                        break;
                }
            }
            else
            {
                DebugConsole.WriteLine("CurveEditorPanel_KeyDown: isInKeyDown == true");
            }
        }

        public bool IsVisibilityAnimation()
        {
            switch (TargetGuiObjectID)
            {
                case GuiObjectID.BoneVisibilityAnimation:
                case GuiObjectID.MaterialVisibilityAnimation:
                    return true;
            }

            return false;
        }

        private EditModeType? beginTimeEditMode_;

        private void BeginEditModeTypeInsert()
        {
            beginTimeEditMode_ = EditMode;
            EditMode = EditModeType.Insert;
        }

        private void EndEditModeTypeInsert()
        {
            if (beginTimeEditMode_.HasValue)
            {
                EditMode = beginTimeEditMode_.Value;
                beginTimeEditMode_ = null;
        }
        }

        private void AddKeyCurrentFrame()
        {
            Point pos = CurveView.PointToClient(Cursor.Position);
            AddKey(
                EditableCurves.ToList(),
                pos,
                true,
                frame:CurveView.State.Pausing?(float)CurveView.State.PauseFrame:(float)CurveView.State.CurrentFrame);
        }

        private void DeleteSelectedKeys()
        {
            if (CheckEditRestriction() == false)
            {
                CurveView.Invalidate();
                return;
            }

            var commandSet = new EditCommandSet();
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                foreach (var curve in MovableCurves)
                {
                    int count = curve.KeyFrames.Count;
                    //var keys = curve.KeyFrames.FindAll(x => x.Selected == false);
                    List<KeyFrame> keys;
                    if (CurveView.IsAutoSplineSlope)
                    {
                        // スロープの自動調整
                        bool[] inSplines = new bool[count];
                        bool[] outSplines = new bool[count];
                        keys = new List<KeyFrame>();

                        for (int i = 0; i < count; i++)
                        {
                            if (curve.KeyFrames[i].Selected)
                            {
                                continue;
                            }

                            var key = curve.KeyFrames[i].Clone();
                            keys.Add(key);
                            bool inSpline;
                            bool outSpline;
                            CurveView.State.IsSplineSlope(curve.KeyFrames, i, -1, out inSpline, out outSpline);
                            inSplines[keys.Count -1] = inSpline;
                            outSplines[keys.Count - 1] = outSpline;
                        }

                        for (int i = 0; i < keys.Count; i++)
                        {
                            if (!inSplines[i] && !outSplines[i])
                            {
                                continue;
                            }

                            float slope = CurveView.State.GetSplineSlope(keys, i);
                            if (inSplines[i])
                            {
                                keys[i].InSlope = slope;
                            }

                            if (outSplines[i])
                            {
                                keys[i].OutSlope = slope;
                            }
                        }
                    }
                    else
                    {
                        keys = curve.KeyFrames.FindAll(x => x.Selected == false);
                    }

                    if (keys.Count != count)
                    {


                        commandSet.Add(
                            AnimationCurveEditCommand.Create(
                                TargetGroup,
                                TargetGuiObjectID,
                                curve,
                                keys
                            )
                        );
                    }
                }
            }
            if (commandSet.CommandCount > 0)
            {
                commandSet.Reverse();
                commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void CopyKeys()
        {
            copyData.CopyKeys(VisibledCurves);
        }

        private void PasteKeys()
        {
            if (EditableCurves.Any())
            {
                var pos = CurveView.PointToClient(Cursor.Position);

                var commandSet =
                    copyData.CreatePasteKeysCommand(
                        TargetGroup,
                        TargetGuiObjectID,
                        EditableCurves.ToList(),
                        CurveView,
                        pos.X,
                        pos.Y,
                        CurveView.State.Pausing ? (float)CurveView.State.PauseFrame : (float)CurveView.State.CurrentFrame,
                        (ConfigCommon.PasteMethodType)tspPasteMethod.SelectedIndex,
                        (ConfigCommon.PastePositionType)tspPastePosition.SelectedIndex
                    );

                if (commandSet.CommandCount > 0)
                {
                    commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                    TheApp.CommandManager.Add(commandSet.Execute());
                }
                else
                {
                    UIMessageBox.Warning(Strings.CurveEdit_NotFoundPasteData);
                }
            }
            else
            {
                UIMessageBox.Warning(Strings.CurveEdit_NotSelectTargetPaste);
            }
        }

        private void CutKeys()
        {
            copyData.CopyKeys(MovableCurves);
            DeleteSelectedKeys();
        }

        private void ScaleView(double factor)
        {
            int	centerScreenX = CurveView.CurveBgWidth  / 2;
            int	centerScreenY = CurveView.CurveBgHeight / 2;
            double beforeCurveX = CurveView.GetXInCurvePlane(centerScreenX);
            double beforeCurveY = CurveView.GetYInCurvePlane(centerScreenY);

            double moveDiffX = factor * CurveEditorConst.WheelSensitivity;
            double moveDiffY = factor * CurveEditorConst.WheelSensitivity;

            double speedX = CurveView.ScaleX * CurveEditorConst.DragSensitivity;
            double speedY = CurveView.ScaleY * CurveEditorConst.DragSensitivity;

            double newScaleX = CurveEditorUtility.ClampScaleX(CurveView.ScaleX + moveDiffX * speedX);
            double newScaleY = CurveEditorUtility.ClampScaleY(CurveView.ScaleY + moveDiffY * speedY);

            double ratioX = newScaleX / CurveView.ScaleX;
            double ratioY = newScaleY / CurveView.ScaleY;

            CurveView.ScaleX = newScaleX;
            CurveView.ScaleY = newScaleY;

            double moveBeforeScreenX;
            double moveBeforeScreenY;
            CurveView.MakeCurveViewScreenPos((float)beforeCurveX, (float)beforeCurveY, out moveBeforeScreenX, out moveBeforeScreenY);

            // マウス位置をスケールの中心にする
            CurveView.ScrollX += (moveBeforeScreenX - centerScreenX) / CurveView.ScaleX;
            CurveView.ScrollY += (moveBeforeScreenY - centerScreenY) / CurveView.ScaleY;
        }

        private void SelectNextCurve(int nextStep)
        {
            DebugConsole.WriteLine("未実装 -- CurveEditorPanel.SelectNextCurve()");
        }

        private void SelectNextKey(int nextStep)
        {
            float lastFrame;
            var anim = TargetGroup.Active;
            if (anim is AnimationDocument)
            {
                lastFrame = ((AnimationDocument)anim).FrameCount - 1;
            }
            else if (anim is CameraAnimation)
            {
                lastFrame = ((CameraAnimation)anim).Data.frame_count - 1;
            }
            else if (anim is LightAnimation)
            {
                lastFrame = ((LightAnimation)anim).Data.frame_count - 1;
            }
            else if (anim is FogAnimation)
            {
                lastFrame = ((FogAnimation)anim).Data.frame_count - 1;
            }
            else
            {
                return;
            }

            var currentFrame = App.AppContext.PreviewCurrentFrame;
            var frames = VisibledCurves.SelectMany(x => x.KeyFrames).Select(x => x.Frame).ToArray();
            if (!frames.Any())
            {
                return;
            }
            var sortedFrames = new SortedSet<float>(frames).Where(x => x >= 0.0f && x <= lastFrame).ToArray();
            var frame = currentFrame;
            if (nextStep > 0)
            {
                var candidates = sortedFrames.Where(f => f > currentFrame).ToArray();
                frame = candidates.Any() ? candidates.First() : sortedFrames.First();
            }
            else
            {
                var candidates = sortedFrames.Where(f => f < currentFrame).ToArray();
                frame = candidates.Any() ? candidates.Last() : sortedFrames.Last();
            }
            App.AppContext.PreviewCurrentFrame = frame;
        }

        private void CheckAllChildNodes(TreeNode treeNode, bool nodeChecked)
        {
            foreach(TreeNode node in treeNode.Nodes)
            {
                node.Checked = nodeChecked;
                if(node.Nodes.Count > 0)
                {
                    CheckAllChildNodes(node, nodeChecked);
                }
            }
        }

        private void CheckAncestor(TreeNode treeNode, bool nodeChecked)
        {
            if (treeNode.Parent != null && treeNode.Parent.Checked != nodeChecked && IsAllChildChecked(treeNode.Parent, nodeChecked))
            {
                treeNode.Parent.Checked = nodeChecked;
                CheckAncestor(treeNode.Parent, nodeChecked);
            }
        }

        private bool IsAllChildChecked(TreeNode treeNode, bool nodeChecked)
        {
            foreach (TreeNode node in treeNode.Nodes)
            {
                if (node.Checked != nodeChecked)
                {
                    return false;
                }
            }

            return true;
        }

        private MouseEventArgs lastMouseDownArgs_ = null;

        private void CurveTreeView_MouseDown(object sender, MouseEventArgs e)
        {
            lastMouseDownArgs_ = e;

            var selectedNode = CurveTreeView.GetNodeAt(e.X, e.Y);
            if (selectedNode != null && e.Button == MouseButtons.Left)
            {
                suspendAfterCheck_ = true;
                var hitTestInfo = CurveTreeView.HitTest(e.X, e.Y);

                if ((hitTestInfo.Location == TreeViewHitTestLocations.Indent) ||
                    (hitTestInfo.Location == TreeViewHitTestLocations.Image) ||
                    (hitTestInfo.Location == TreeViewHitTestLocations.Label) ||
                    (hitTestInfo.Location == TreeViewHitTestLocations.RightOfLabel))
                {
                    bool isIconClicked	= hitTestInfo.Location == TreeViewHitTestLocations.Image;
                    var isToggle		= ((Control.ModifierKeys & Keys.Control) != 0) || isIconClicked;
                    var isRangeSelect	=  (Control.ModifierKeys & Keys.Shift  ) != 0;

                    // 範囲選択
                    if (isRangeSelect && (oldSelectedNode_ != null) && (CurveTreeView.SelectedNode != null))
                    {
                        int oldIndex     = Array.IndexOf(treeNodes_, oldSelectedNode_);
                        int currentIndex = Array.IndexOf(treeNodes_, CurveTreeView.SelectedNode);

                        if ((oldIndex     != -1) &&
                            (currentIndex != -1))
                        {
                            var animationCurves = new List<IAnimationCurve>();
                            var selectedEmptyCurves = new List<IAnimationCurve>();

                            int minIndex = Math.Min(oldIndex, currentIndex);
                            int maxIndex = Math.Max(oldIndex, currentIndex);

                            {
                                for (int i = minIndex; i <= maxIndex; ++ i)
                                {
                                    var node = treeNodes_[i];
                                    node.Checked = true;
                                    CheckAncestor(node, true);

                                    if (i != maxIndex)
                                    {
                                        var curve = node.Tag as IAnimationCurve;
                                        if (curve != null)
                                        {
                                            animationCurves.Add(curve);
                                        }
                                    }
                                    else
                                    {
                                        CheckAllChildNodes(node, true);
                                        var childCurves = CurveEditorUtility.CollectNode(node).Select(x => x.Tag as IAnimationCurve);

                                        animationCurves = animationCurves.Concat(childCurves).ToList();
                                        animationCurves.ForEach(
                                            curve =>
                                                {
                                                    // キーのないカーブは編集対象とする
                                                    if (curve != null)
                                                    {
                                                        if (!curve.KeyFrames.Any())
                                                        {
                                                            selectedEmptyCurves.Add(curve);
                                                        }
                                                        else if (isIconClicked)
                                                        {
                                                            SelectAllKeys(curve, true);
                                                        }
                                                        else if (!this.SelectedCurves.Contains(curve))
                                                        {
                                                            SelectAllKeys(curve, false);
                                                        }
                                                    }
                                                });
                                    }
                                }
                            }

                            SelectedCurves = isIconClicked
                                                 ? SelectedCurves.Union(animationCurves).Union(selectedEmptyCurves).ToList()
                                                 : SelectedCurves.Union(selectedEmptyCurves).ToList();
                            VisibledCurves = VisibledCurves.Union(animationCurves).ToList();
                        }
                    }
                    else
                    if (isToggle)
                    {
                        var animationCurves = new List<IAnimationCurve>(SelectedCurves);
                        var removedCurves = new List<IAnimationCurve>();
                        var selectedEmptyCurves = new List<IAnimationCurve>();
                        {
                            // カーブの親ノードの編集状態をクリックした時はすべての子を同じ状態でトグル
                            var syncParent = isIconClicked && !(selectedNode.Tag is IAnimationCurve);
                            var syncValue = CurveTreeView.GetNodeState(selectedNode) != CurveTreeView.NodeSelectState.Selected;

                            var	foundCurves	= CurveEditorUtility.CollectNode(selectedNode);

                            foreach (var node in foundCurves)
                            {
                                Debug.Assert(node.Tag is IAnimationCurve);
                                var curve = node.Tag as IAnimationCurve;

                                bool doSelect = syncParent ? syncValue : !this.SelectedCurves.Contains(curve);

                                if (doSelect)
                                {
                                    if (!node.Checked || isIconClicked)
                                    {
                                        SelectAllKeys(curve, isIconClicked);
                                        animationCurves.Add(curve);
                                        // キーのないカーブは編集対象とする
                                        if (!curve.KeyFrames.Any())
                                        {
                                            selectedEmptyCurves.Add(curve);
                                        }
                                        node.Checked = true;
                                        this.CheckAncestor(node, true);
                                    }
                                    else
                                    {
                                        removedCurves.Add(curve);
                                        node.Checked = false;
                                        this.CheckAncestor(node, false);
                                    }
                                }
                                else
                                {
                                    SelectAllKeys(curve, false);
                                    animationCurves.Remove(curve);
                                    if (isIconClicked)
                                    {
                                    }
                                    else
                                    {
                                        if (node.Checked)
                                        {
                                            removedCurves.Add(curve);
                                            node.Checked = false;
                                            this.CheckAncestor(node, false);
                                        }
                                    }
                                }
                            }
                        }
                        SelectedCurves = isIconClicked
                                             ? animationCurves
                                             : SelectedCurves.Intersect(animationCurves).Except(removedCurves).Union(selectedEmptyCurves).ToList();
                        VisibledCurves = VisibledCurves.Union(animationCurves).Except(removedCurves).ToList();
                        oldSelectedNode_ = CurveTreeView.SelectedNode;
                    }
                    else
                    {
                        // 通常選択時は表示チェックをクリアする
                        CurveTreeView.SetCheck(false);
                        var animationCurves = new List<IAnimationCurve>();
                        var selectedEmptyCurves = new List<IAnimationCurve>();
                        {
                            var nodes = CurveEditorUtility.CollectNode(selectedNode, isIncludeNullNode:true);
                            if (!nodes.Contains(selectedNode))
                            {
                                nodes.Add(selectedNode);
                            }

                            foreach (var node in nodes)
                            {
                                var curve = node.Tag as IAnimationCurve;
                                if (curve != null)
                                {
                                    animationCurves.Add(curve);
                                    if (!this.SelectedCurves.Contains(curve))
                                    {
                                        SelectAllKeys(curve, false);
                                    }
                                    // キーのないカーブは編集対象とする
                                    if (!curve.KeyFrames.Any())
                                    {
                                        selectedEmptyCurves.Add(curve);
                                    }
                                }

                                if (node.Checked == false)
                                {
                                    node.Checked = true;
                                    CheckAncestor(node, true);
                                }
                            }
                        }
                        SelectedCurves = SelectedCurves.Intersect(animationCurves).Union(selectedEmptyCurves).ToList();
                        VisibledCurves = animationCurves;
                        oldSelectedNode_ = CurveTreeView.SelectedNode;
                    }
                }
                suspendAfterCheck_ = false;
            }

        }

        private static void SelectAllKeys(IAnimationCurve curve, bool isSelect)
        {
            if (curve == null || curve.KeyFrames == null)
            {
                return;
            }
            curve.KeyFrames.ForEach(
                key =>
                    {
                        key.Selected = isSelect;
                        key.AreaSelected = false;
                        key.InHandleSelected = false;
                        key.OutHandleSelected = false;
                    });
        }

        private void CurveTreeView_DoubleClick(object sender, EventArgs e)
        {
            if (lastMouseDownArgs_ == null)
            {
                return;
            }

            var selectedNode = CurveTreeView.GetNodeAt(lastMouseDownArgs_.X, lastMouseDownArgs_.Y);
            if (selectedNode != null)
            {
                var hitTestInfo = CurveTreeView.HitTest(lastMouseDownArgs_.X, lastMouseDownArgs_.Y);
                if ((hitTestInfo.Location == TreeViewHitTestLocations.Label) ||
                    (hitTestInfo.Location == TreeViewHitTestLocations.RightOfLabel))
                {
                    var curve = selectedNode.Tag as IAnimationCurve;
                    if (curve != null)
                    {
                        FitAllViewIn(Enumerable.Repeat(curve, 1), false);
                    }
                }
            }

            lastMouseDownArgs_ = null;
        }

        private void tsbAllViewIn_Click(object sender, EventArgs e)
        {
            FitAllViewIn(VisibledCurves, false, IsVisibilityAnimation());
        }

        private void tsbFrameRangeViewIn_Click(object sender, EventArgs e)
        {
            FitFrameRangeViewIn();
        }

        private void tsbValueClampViewIn_Click(object sender, EventArgs e)
        {
            FitValueClampViewIn();
        }

        private void tsbCurrentFrameCenter_Click(object sender, EventArgs e)
        {
            CurrentFrameCenterViewerIn();
        }

        public void FitAllViewIn(IEnumerable<IAnimationCurve> curves, bool initial, bool useMinMax = true)
        {
            // フレーム、値の最小最大を求めておく
            float frameMin = +float.MaxValue;
            float frameMax = -float.MaxValue;
            float valueMin = +float.MaxValue;
            float valueMax = -float.MaxValue;

            // 初期表示時は再生範囲全体にする
            if (initial)
            {
                frameMin = 0;
                frameMax = (float)FrameCount;
            }

            {
                // キーを持っているカーブのみを対象とする
                foreach(var curve in curves.Where(x => x.KeyFrames.Any()))
                {
                    if (useMinMax && (curve.MinFitValue != null) && (curve.MaxFitValue != null))
                    {
                        valueMin = Math.Min(valueMin, (float)(double)curve.MinFitValue);
                        valueMax = Math.Max(valueMax, (float)(double)curve.MaxFitValue);
                    }

                    float tempFrameMin, tempFrameMax;
                    if (CurveEditorUtility.MakeMinMaxFrame(curve, out tempFrameMin, out tempFrameMax))
                    {
                        frameMin = Math.Min(frameMin, tempFrameMin);
                        frameMax = Math.Max(frameMax, tempFrameMax);
                    }

                    float tempValueMin, tempValueMax;
                    if (CurveEditorUtility.MakeMinMaxValue(curve, out tempValueMin, out tempValueMax))
                    {
                        valueMin = Math.Min(valueMin, tempValueMin);
                        valueMax = Math.Max(valueMax, tempValueMax);
                    }
                }
            }

            if ((frameMin != +float.MaxValue) && (frameMax != -float.MaxValue) &&
                (valueMin != +float.MaxValue) && (valueMax != -float.MaxValue))
            {
                CurveView.SetFitView(frameMin, frameMax, valueMin, valueMax);
            }
            else// if (initial)
            {
                switch (TargetGuiObjectID)
                {
                    case GuiObjectID.MaterialAnimation:
                    case GuiObjectID.ShaderParameterAnimation:
                    case GuiObjectID.ColorAnimation:
                    case GuiObjectID.TextureSrtAnimation:
                    case GuiObjectID.TexturePatternAnimation:
                        {
                            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1531
                            // チェックされたカーブにキーがない場合に、Aキーが押されたときやマテリアルプロパティパネルからカーブエディタを
                            // 開いたときは ui_min, ui_max を考慮する必要があります。

                            // ■カーブにキーが１つもない場合
                            //   ui_min, ui_max から範囲を決める
                            {
                                if (valueMin == +float.MaxValue)
                                {
                                    foreach (var curve in curves.Where(x => x.MinFitValue.HasValue))
                                    {
                                        valueMin = Math.Min(valueMin, curve.MinFitValue.Value);
                                    }
                                }

                                if (valueMax == -float.MaxValue)
                                {
                                    foreach (var curve in curves.Where(x => x.MaxFitValue.HasValue))
                                    {
                                        valueMax = Math.Max(valueMax, curve.MaxFitValue.Value);
                                    }
                                }
                            }

                            // ■ui_min, ui_max も指定されていない場合
                            //   ui_min もしくは ui_max の片方のみ決定済みであれば、決定済みからのオフセットで決める。
                            //   両方共未決定であれば、絶対値で決める。
                            {
                                if (((valueMin == +float.MaxValue) && (valueMax == -float.MaxValue))
                                    || (valueMin == valueMax))
                                {
                                    valueMin = (TargetGuiObjectID == GuiObjectID.TexturePatternAnimation)? 0.0f : - 10.0f;
                                    valueMax = +10.0f;
                                }
                                else
                                    if (valueMin == +float.MaxValue)
                                    {
                                        valueMin = valueMax - 10.0f;
                                    }
                                    else
                                        if (valueMax == -float.MaxValue)
                                        {
                                            valueMax = valueMin + 10.0f;
                                        }
                            }
                        }
                        break;
                    default:
                        {
                            valueMin =
                                (TargetGuiObjectID == GuiObjectID.ShapeAnimation) ||
                                (TargetGuiObjectID == GuiObjectID.MaterialVisibilityAnimation) ||
                                (TargetGuiObjectID == GuiObjectID.BoneVisibilityAnimation) ? 0.0f : -10.0f;

                            valueMax =
                                (TargetGuiObjectID == GuiObjectID.ShapeAnimation) ||
                                (TargetGuiObjectID == GuiObjectID.MaterialVisibilityAnimation) ||
                                (TargetGuiObjectID == GuiObjectID.BoneVisibilityAnimation) ? 1.0f : 10.0f;
                        }
                        break;
                }

                Debug.Assert(valueMin <= valueMax);

                CurveView.SetFitView(0.0f, (float)FrameCount, valueMin, valueMax);
            }
        }

        private void FitSelectedViewIn(bool useMinMax)
        {
            FitCurvesViewIn(VisibledCurves, useMinMax);
        }

        private void FitCurvesViewIn(IEnumerable<IAnimationCurve> curves, bool useMinMax)
        {
            // フレーム、値の最小最大を求めておく
            float frameMin = float.MaxValue;
            float frameMax = float.MinValue;
            float valueMin = float.MaxValue;
            float valueMax = float.MinValue;
            bool anySelected = false;
            {
                foreach(var curve in curves)
                {
                    bool selected = false;
                    for (int i = 0; i < curve.KeyFrames.Count; i++)
                    {
                        if (i == 0)
                        {
                            if (curve.KeyFrames[i].Selected)
                            {
                                selected = true;
                                var key = curve.KeyFrames[i];
                                float frame = key.Frame;
                                float value = key.Value;
                                frameMin = Math.Min(frameMin, frame);
                                frameMax = Math.Max(frameMax, frame);
                                valueMin = Math.Min(valueMin, value);
                                valueMax = Math.Max(valueMax, value);
                            }
                        }
                        else
                        {
                            var left = curve.KeyFrames[i - 1];
                            var right = curve.KeyFrames[i];
                            if (left.Selected || right.Selected)
                            {
                                selected = true;
                                frameMin = Math.Min(frameMin, left.Frame);
                                frameMax = Math.Max(frameMax, right.Frame);
                                float min;
                                float max;
                                MathUtility.HermiteMinMax(left, right, out min, out max);

                                valueMin = Math.Min(valueMin, min);
                                valueMax = Math.Max(valueMax, max);
                            }
                        }
                    }

                    anySelected |= selected;
                    if (useMinMax && selected && curve.MinFitValue != null && curve.MaxFitValue != null)
                    {
                        valueMin = Math.Min(valueMin, (float)(double)curve.MinFitValue);
                        valueMax = Math.Max(valueMax, (float)(double)curve.MaxFitValue);
                    }
                }
            }

            if (anySelected)
            {
                CurveView.SetFitView(frameMin, frameMax, valueMin, valueMax);
            }
            else
            {
                FitAllViewIn(curves, false, useMinMax);
            }
        }

        private void FitFrameRangeViewIn()
        {
            // フレーム、値の最小最大を求めておく
            float frameMin = 0.0f;
            float frameMax = FrameCount;

            CurveView.SetFitX(frameMin, frameMax);
        }

        private void FitValueClampViewIn()
        {
            var clampMin = +float.MaxValue;
            var clampMax = -float.MaxValue;

            var isFitted = false;

            var valueRange = CurveView.MakeEditValueRange();

            if (valueRange.MinValue.HasValue)
            {
                clampMin = valueRange.MinValue.Value;
            }

            if (valueRange.MaxValue.HasValue)
            {
                clampMax = valueRange.MaxValue.Value;
            }

            if ((clampMin != +float.MaxValue) && (clampMax != -float.MaxValue))
            {
                CurveView.SetFitY(clampMin, clampMax);
                isFitted = true;
            }

            if (isFitted == false)
            {
                // 上限値、下限値両方共指定されていない
                if ((clampMin == +float.MaxValue) && (clampMax == -float.MaxValue))
                {
                    ;			// 何もしない
                }
                else
                // 上限値のみ指定されている
                if (clampMin == +float.MaxValue)
                {
                    // すべてのキーフレームの最小値を下限値とする。
                    // 上限値以上のキーは無視。
                    // キーがなければ固定値

                    var keyframe = VisibledCurves.SelectMany(x => x.KeyFrames).Where(x => x.Value < clampMax).OrderBy(x => x.Value).FirstOrDefault();
                    if (keyframe == null)
                    {
                        clampMin = clampMax - 10.0f;
                    }
                    else
                    {
                        clampMin = keyframe.Value;
                    }
                }
                else
                // 下限値のみ指定されている
                if (clampMax == -float.MaxValue)
                {
                    // すべてのキーフレームの最大値を上限値とする。
                    // 下限値以下のキーは無視。
                    // キーがなければ固定値

                    var keyframe = VisibledCurves.SelectMany(x => x.KeyFrames).Where(x => x.Value > clampMin).OrderByDescending(x => x.Value).FirstOrDefault();
                    if (keyframe == null)
                    {
                        clampMax = clampMin + 10.0f;
                    }
                    else
                    {
                        clampMax = keyframe.Value;
                    }
                }

                if ((clampMin != +float.MaxValue) && (clampMax != -float.MaxValue))
                {
                    CurveView.SetFitY(clampMin, clampMax);
                }
            }
        }

        private void CurrentFrameCenterViewerIn()
        {
            if (CurveView.State.Pausing)
            {
                CurveView.SetCenterFrame(CurveView.State.PauseFrame);
            }
            else
            {
                CurveView.SetCenterFrame(CurveView.State.CurrentFrame);
            }
        }

        private void tsbEditMode_Click(object sender, EventArgs e)
        {
            EditMode = (EditModeType)((UIToolStripButton)sender).Tag;
        }

        private void VisibleButton_CheckedChanged(object sender, EventArgs e)
        {
            UpdateConfig();
            InvalidateCurveView();
        }

        // 複数選択未対応
        private void KeyInterpolationTypeChange_Click(object sender, EventArgs e)
        {
            if (CheckEditRestriction() == false)
            {
                CurveView.Invalidate();
                return;
            }

            var commandSet = new EditCommandSet();
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                foreach (var curve in EditableCurves)
                {
                    if (curve.CurvePrimitiveType != PrimitiveTypeKind.Float)
                    {
                        continue;
                    }

                    var getAnimTarget = curve;

                    commandSet.Add(
                        AnimationCurveEditCommand.CreateInterpolationTypeEditCommand(
                            TargetGroup,
                            TargetGuiObjectID,
                            curve,
                            (InterpolationType)((ToolStripItem)sender).Tag));

                    commandSet.Add(new LazyCommand(() => AnimationCurveEditCommand.CreateQuantizeEditCommand(TargetGroup, TargetGuiObjectID, curve)));
                }
            }
            if (commandSet.CommandCount > 0)
            {
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void tsbToFlatSlope_Click(object sender, EventArgs e)
        {
            EditSelectedKeys(
                (key) => key.AnySelected,
                (key, curve) => {
                                    if (key.Selected || key.InHandleSelected)
                                    {
                                        key.InSlope = 0.0f;
                                    }
                                    if (key.Selected || key.OutHandleSelected)
                                    {
                                        key.OutSlope = 0.0f;
                                    }
                }
            );

            SetKeyParam();
        }

        private void tsbToLinearSlope_Click(object sender, EventArgs e)
        {
            DebugConsole.WriteLine("--------------------------------");

            EditSelectedKeys(
                (key) => key.AnySelected,
                (key, curve) =>
                {
                    DebugConsole.WriteLine(">>>{0},{1}", key, curve);

                    // 左端のin_slope、右端のout_slopeは水平に設定
                    var isFirstKey = false;
                    var isLastKey = false;
                    {
                        if ((key.Index == 0) && (key.InHandleSelected || key.Selected))
                        {
                            key.InSlope = 0.0f;
                            isFirstKey = true;
                        }

                        if ((key.Index == curve.KeyFrames.Count() - 1) && (key.OutHandleSelected || key.Selected))
                        {
                            key.OutSlope = 0.0f;
                            isLastKey = true;
                        }
                    }

                    // in_slope
                    if ((isFirstKey == false) && (key.InHandleSelected || key.Selected))
                    {
                        var priorKey = curve.KeyFrames[key.Index - 1];

                        double x = key.Frame - priorKey.Frame;
                        double y = key.Value - priorKey.Value;

                        x = Math.Max(x, 0.00001);

                        key.InSlope = (float)(y / x);
                    }

                    // out_slope
                    if ((isLastKey == false) && (key.OutHandleSelected || key.Selected))
                    {
                        var nextKey = curve.KeyFrames[key.Index + 1];

                        double x = nextKey.Frame - key.Frame;
                        double y = nextKey.Value - key.Value;

                        x = Math.Max(x, 0.00001);

                        key.OutSlope = (float)(y / x);
                    }
                }
            );

            SetKeyParam();
        }

        private void tsbToSplineSlope_Click(object sender, EventArgs e)
        {
            EditSelectedKeys(
                (key) => key.AnySelected,
                (key, curve) =>
                {
                    double x0, x1;
                    double y0, y1;
                    if (key.Index == 0)
                    {
                        x0 = key.Frame;
                        y0 = key.Value;
                    }
                    else
                    {
                        x0 = curve.KeyFrames[key.Index - 1].Frame;
                        y0 = curve.KeyFrames[key.Index - 1].Value;
                    }

                    if (key.Index == curve.KeyFrames.Count - 1)
                    {
                        x1 = key.Frame;
                        y1 = key.Value;
                    }
                    else
                    {
                        x1 = curve.KeyFrames[key.Index + 1].Frame;
                        y1 = curve.KeyFrames[key.Index + 1].Value;
                    }

                    double x = Math.Max(x1 - x0, 0.00001);
                    double y = y1 - y0;
                    float slope = (float)(y / x);
                    if (key.InHandleSelected || key.Selected)
                    {
                        key.InSlope = slope;
                    }

                    if (key.OutHandleSelected || key.Selected)
                    {
                        key.OutSlope = slope;
                    }
                }
            );

            SetKeyParam();
        }

        private void KeyIsUnionSlopeChange_Click(object sender, EventArgs e)
        {
            bool isUnionTangent = (bool)((ToolStripItem)sender).Tag;

            EditSelectedKeys(
                (key) => key.AnySelected,
                (key, curve) =>  key.IsUnionSlope = isUnionTangent
            );
        }

         void MakeVisibledCurves()
        {
            var visibledCurves = new List<IAnimationCurve>();

            MakeVisibledCurves(visibledCurves, CurveTreeView.Nodes);

            VisibledCurves = visibledCurves;

            CurveView.Invalidate();
        }

        private void MakeVisibledCurves(List<IAnimationCurve> dst, TreeNodeCollection nodes)
        {
            foreach(TreeNode node in nodes)
            {
                if (node.Checked && (node.Tag != null))
                {
                    dst.Add(node.Tag as IAnimationCurve);
     			}

                MakeVisibledCurves(dst, node.Nodes);
            }
        }

        // TODO: MaterialAnimation
        public static EditCommand CreateShaderParameterAnimsEditCommand(
            GuiObjectGroup targets,
            GuiObjectID id,
            ShaderParameterAnimation.ShaderParamAnimList shaderParamMatAnims)
        {
            return new GeneralGroupReferenceEditCommand<ShaderParameterAnimation.ShaderParamAnimList>(
                targets,
                id,
                ObjectUtility.MultipleClone(shaderParamMatAnims, targets.GetObjects(id).Count),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    Debug.Assert(target is ShaderParameterAnimation, "未実装");
                    var shaderParamAnim = target as ShaderParameterAnimation;
                    swap = shaderParamAnim.ShaderParamAnims;
                    shaderParamAnim.ShaderParamAnims = (ShaderParameterAnimation.ShaderParamAnimList)data;
                    Viewer.LoadOrReloadAnimation.Send(shaderParamAnim);
                }
            );
        }

        public static EditCommand CreateMaterialVisibilityMatAnimsEditCommand(
            GuiObjectGroup targets,
            GuiObjectID id,
            List<MaterialVisibilityAnimation.MaterialVisibilityMatAnim> materialVisibilityMatAnims)
        {
            return new GeneralGroupReferenceEditCommand<List<MaterialVisibilityAnimation.MaterialVisibilityMatAnim>>(
                        targets,
                        id,
                        ObjectUtility.MultipleClone(materialVisibilityMatAnims, targets.GetObjects(id).Count),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is MaterialVisibilityAnimation, "未実装");
                            var materialVisibilityAnimation = target as MaterialVisibilityAnimation;

                            swap = ObjectUtility.Clone(materialVisibilityAnimation.MaterialVisibilityMatAnims);
                            materialVisibilityAnimation.MaterialVisibilityMatAnims.Clear();
                            materialVisibilityAnimation.MaterialVisibilityMatAnims.AddRange(data as List<MaterialVisibilityAnimation.MaterialVisibilityMatAnim>);
                        }
                    );
        }

        public static EditCommand CreateTexturePatternMatAnimsEditCommand(GuiObjectGroup objects, List<TexturePatternAnimation.TexPatternMatAnim> texPatternMatAnims)
        {
            return new GeneralGroupReferenceEditCommand<List<TexturePatternAnimation.TexPatternMatAnim>>(
                        objects,
                        GuiObjectID.TexturePatternAnimation,
                        ObjectUtility.MultipleClone(texPatternMatAnims, objects.GetObjects(GuiObjectID.TexturePatternAnimation).Count),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is TexturePatternAnimation, "未実装");
                            var texturePatternAnim = target as TexturePatternAnimation;

                            swap = ObjectUtility.Clone(texturePatternAnim.TexPatternMatAnims);
                            texturePatternAnim.TexPatternMatAnims.Clear();
                            texturePatternAnim.TexPatternMatAnims.AddRange(data as List<TexturePatternAnimation.TexPatternMatAnim>);
                        }
                    );
        }

        private void tsmAllShowNode_CheckedChanged(object sender, EventArgs e)
        {
            MakeVisibledCurves();
            UpdateForm(new ObjectPropertyPage.UpdateFormInfo());
        }

        private void tsmSortNode_CheckedChanged(object sender, EventArgs e)
        {
            UpdateForm(new ObjectPropertyPage.UpdateFormInfo());
        }

        private void tsmShaderParamAnimGroupCondShow_CheckedChanged(object sender, EventArgs e)
        {
            MakeVisibledCurves();
            SelectedCurves = SelectedCurves.Intersect(CurveTreeView.AllCurveNodes.Select(x => x.Info.AnimationCurve)).ToList();
            UpdateForm(new ObjectPropertyPage.UpdateFormInfo());
        }

        private void tsmOnlyTargetShowNode_CheckedChanged(object sender, EventArgs e)
        {
            MakeVisibledCurves();
            SelectedCurves = SelectedCurves.Intersect(CurveTreeView.AllCurveNodes.Select(x => x.Info.AnimationCurve)).ToList();
            UpdateForm(new ObjectPropertyPage.UpdateFormInfo());
        }

        private void tsmShowKeyedMaterial_CheckedChanged(object sender, EventArgs e)
        {
            MakeVisibledCurves();
            UpdateForm(new ObjectPropertyPage.UpdateFormInfo());
        }

        private void tsmResetNodeExpand_Click(object sender, EventArgs e)
        {
            switch (TargetGuiObjectID)
            {
                case GuiObjectID.MaterialAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                    CurveTreeView.ResetShaderParameterNodeExpand();
                    return;
            }

            CurveTreeView.ResetNodeExpand();
        }

        private void tsmOpenNodeExpand_Click(object sender, EventArgs e)
        {
            CurveTreeView.OpenNodeExpand();
        }

        private void tsmVisibleAllNode_Click(object sender, EventArgs e)
        {
            VisibleAllNode();
        }

        private void tsmInvisibleAllNode_Click(object sender, EventArgs e)
        {
            InvisibleAllNode();
            SelectedCurves = new List<IAnimationCurve>();
        }

        private void SearchTextChanged(object sender, EventArgs e)
        {
            UpdateForm(new ObjectPropertyPage.UpdateFormInfo());
        }

        private void SetSelectedKeysFrame(float frame)
        {
            EditSelectedKeys(
                (key) => key.Selected,
                (key, curve) =>
                {
                    if (IsClampFrame)
                    {
                        frame = Math.Min(Math.Max(frame, 0.0f), FrameCount);
                    }
                    key.Frame = frame;
                }
            );
        }

        // 型による値の修正
        private static float FixValue(float value, PrimitiveTypeKind curvePrimitiveType)
        {
            switch (curvePrimitiveType)
            {
                case PrimitiveTypeKind.Bool:
                    value = (value >= 0.5f) ? 1.0f : 0.0f;
                    break;
                case PrimitiveTypeKind.Int:
                    value = (float)Math.Round(value, MidpointRounding.AwayFromZero);
                    break;
                case PrimitiveTypeKind.Uint:
                    value = (float)Math.Max(Math.Round(value, MidpointRounding.AwayFromZero), 0);
                    break;
            }

            return value;
        }

        private void SetSelectedKeysValue(float value)
        {
            EditSelectedKeys(
                (key) => key.Selected,
                (key, curve) => {
                    if (IsClampValue)
                    {
                        value = curve.ClampValue(value);
                    }
                    else
                    {
                        value = FixValue(value, curve.CurvePrimitiveType);
                    }

                    key.Value = value;
                }
            );
        }

        private void SetSelectedKeysInSlope(float slope)
        {
            EditSelectedKeys(
                (key) => key.Selected || key.InHandleSelected,
                (key, curve) => key.InSlope = slope
            );
        }

        private void SetSelectedKeysOutSlope(float slope)
        {
            EditSelectedKeys(
                (key) => key.Selected || key.OutHandleSelected,
                (key, curve) => key.OutSlope = slope
            );
        }

        private delegate bool IsKeySelectedDelegate(KeyFrame key);
        private delegate void EditSelectedKeysDelegate(KeyFrame key, IAnimationCurve curve);

        private void EditSelectedKeys(IsKeySelectedDelegate isKeySelected, EditSelectedKeysDelegate editSelectedKeysDelegate)
        {
            var commandSet = new EditCommandSet();
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                foreach(var visibleCurve in MovableCurves)
                {
                    visibleCurve.SetCurvesSequentialIndex();

                    var keyFrames = ObjectUtility.Clone(visibleCurve.KeyFrames);

                    bool edited = false;
                    foreach (var keyFrame in keyFrames)
                    {
                        if (isKeySelected(keyFrame))
                        {
                            editSelectedKeysDelegate(keyFrame, visibleCurve);
                            edited = true;
                        }
                    }

                    if (edited)
                    {
                        commandSet.Add(
                            AnimationCurveEditCommand.Create(
                                TargetGroup,
                                TargetGuiObjectID,
                                visibleCurve,
                                keyFrames
                            )
                        );
                    }
                }

                if (commandSet.CommandCount > 0)
                {
                    commandSet.OnPostEdit += (s, e) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                    TheApp.CommandManager.Add(commandSet.Execute());
                }
            }
        }

        private void Paste_SelectedIndexChanged(object sender, EventArgs e)
        {
            ApplicationConfig.Setting.CurveEditor.PasteMethod	= (ConfigCommon.PasteMethodType)tspPasteMethod.SelectedIndex;
        }

        private void tspPastePosition_SelectedIndexChanged(object sender, EventArgs e)
        {
            ApplicationConfig.Setting.CurveEditor.PastePosition	= (ConfigCommon.PastePositionType)tspPastePosition.SelectedIndex;
        }

        private void Snap_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (isConfigLoaded_)
            {
                ApplicationConfig.Setting.CurveEditor.FrameSnapFactor = FrameSnapFactor;
                if (tscValueSnap.Items.Count > 1)
                {
                    // 選択肢が複数ある時だけ更新
                    ApplicationConfig.Setting.CurveEditor.ValueSnapFactor = ValueSnapFactor;
                }
            }
        }

        private void tscFrame_ValueChanged(object sender, EventArgs e)
        {
            SetSelectedKeysFrame((sender as ToolStripFloatInputBox).Value);
        }

        private void tscValue_ValueChanged(object sender, EventArgs e)
        {
            float value = (sender as ToolStripFloatInputBox).Value;
            SetSelectedKeysValue(value);
        }

        private void tscInSlope_ValueChanged(object sender, EventArgs e)
        {
            SetSelectedKeysInSlope((sender as ToolStripFloatInputBox).Value);
        }

        private void tscOutSlope_ValueChanged(object sender, EventArgs e)
        {
            SetSelectedKeysOutSlope((sender as ToolStripFloatInputBox).Value);
        }

        private MaterialVisibilityAnimation.MaterialVisibilityMatAnim GetMaterialVisibilityAnim(string ParentName, ref GuiObject target)
        {
            var ParamAnim = (MaterialVisibilityAnimation)target;

            MaterialVisibilityAnimation.MaterialVisibilityMatAnim matAnim = ParamAnim.GetParamAnimFromId(ParentName);
            return matAnim;
        }

        private BoneVisibilityAnimation.BoneVisibilityBoneAnim GetBoneVisibilityAnim(string ParentName, ref GuiObject target)
        {
            var ParamAnim = (BoneVisibilityAnimation)target;

            BoneVisibilityAnimation.BoneVisibilityBoneAnim boneAnim = ParamAnim.GetParamAnimFromId(ParentName);
            return boneAnim;
        }

        private TexturePatternAnimation.PatternAnimTarget GetTexturePatternAnimTarget(string ParentName, string sampler_name, ref GuiObject target)
        {
            var ParamAnim = (TexturePatternAnimation)target;

            TexturePatternAnimation.PatternAnimTarget animTarget = ParamAnim.GetParamAnimFromId(ParentName, sampler_name);
            if (animTarget == null)
            {
                // 追加先がない場合は作る
                animTarget = AnimationCurveEditCommand.CreateTemporaryPatternAnimTarget(sampler_name);
                ParamAnim.TexPatternMatAnims.First(x => x.mat_name == ParentName).PatternAnimTargets.Add(animTarget);
            }

            return animTarget;
        }

        private void cmbPreWrap_SelectedIndexChanged(object sender, EventArgs e)
        {
            var commandSet = new EditCommandSet();
            {
                foreach (var curve0 in EditableCurves)
                {
                    // スコープを制限
                    var curve = curve0;
                    string parentName = curve.ParentName;
                    string curveName = curve.Name;
                    int curveComponentIndex = curve.ComponentIndex;

                    commandSet.Add(
                        new GeneralGroupValueEditCommand<curve_wrapType>(
                            TargetGroup,
                            TargetGuiObjectID,
                            (curve_wrapType)(((UIComboBox)sender).SelectedItemData),
                            delegate(ref GuiObject target, ref object data, ref object swap)
                            {
                                var dst = curve.GetAnimTarget(target);
                                swap = dst.pre_wrap;
                                dst.pre_wrap = (curve_wrapType)data;

                                curve.UpdateIsModified(target);

                                // HIOランタイム連携に送る
                                CurveExportType type = dst.ExportType;
                                if (type == CurveExportType.Dependent)
                                {
                                    Viewer.LoadOrReloadAnimation.Send((AnimationDocument)target.OwnerDocument);
                                }
                                else
                                {
                                    CurveEditorPanel.SendViewerEditCurve(target, curve, type);
                                }
                            }
                        )
                    );
                }
            }
            if (commandSet.CommandCount > 0)
            {
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void cmbPostWrap_SelectedIndexChanged(object sender, EventArgs e)
        {
            var commandSet = new EditCommandSet();
            {
                foreach (var curve0 in EditableCurves)
                {
                    // スコープを制限
                    var curve = curve0;
                    string parentName = curve.ParentName;
                    string curveName = curve.Name;
                    int curveComponentIndex = curve.ComponentIndex;
                    commandSet.Add(
                        new GeneralGroupValueEditCommand<curve_wrapType>(
                            TargetGroup,
                            TargetGuiObjectID,
                            (curve_wrapType)(((UIComboBox)sender).SelectedItemData),
                            delegate(ref GuiObject target, ref object data, ref object swap)
                            {
                                var dst = curve.GetAnimTarget(target);
                                swap = dst.post_wrap;
                                dst.post_wrap = (curve_wrapType)data;

                                curve.UpdateIsModified(target);

                                // HIOランタイム連携に送る
                                CurveExportType type = dst.ExportType;
                                if (type == CurveExportType.Constant || type == CurveExportType.Dependent)
                                {
                                    Viewer.LoadOrReloadAnimation.Send((AnimationDocument)target.OwnerDocument);
                                }
                                else
                                {
                                    CurveEditorPanel.SendViewerEditCurve(target, curve, type);
                                }
                            }
                        )
                    );
                }
            }
            if (commandSet.CommandCount > 0)
            {
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void chkCompressEnable_CheckedChanged(object sender, EventArgs e)
        {
            var commandSet = new EditCommandSet();
            {
                foreach (var curve in EditableCurves)
                {
                    string parentName = curve.ParentName;
                    string curveName = curve.Name;
                    int curveComponentIndex = curve.ComponentIndex;

                    if (TargetGroup.Active is BoneVisibilityAnimation)
                    {
                        commandSet.Add(
                            new GeneralGroupValueEditCommand<bool>(
                                TargetGroup,
                                TargetGuiObjectID,
                                (bool)(((UICheckBox)sender).Checked),
                                delegate(ref GuiObject target, ref object data, ref object swap)
                                {
                                    BoneVisibilityAnimation.BoneVisibilityBoneAnim dst = GetBoneVisibilityAnim(parentName, ref target);
                                    swap = dst.compress_enable;
                                    dst.compress_enable = (bool)data;

                                    curve.UpdateIsModified(target);

                                    // Viewerへ再転送
                                    Viewer.LoadOrReloadAnimation.Send((AnimationDocument)target);
                                }
                            )
                        );
                    }
                    else
                    {
                        Debug.Assert(false, "未実装");
                    }
                }
            }
            if (commandSet.CommandCount > 0)
            {
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void chkBinarizeVisibility_CheckedChanged(object sender, EventArgs e)
        {
            var commandSet = new EditCommandSet();
            {
                // バイナリ出力がOFFのとき該当のカーブが編集不可になるが、binarize_visibilityだけは編集させる
                var active = TargetGroup.Active;
                var bones = new HashSet<string>((DocumentManager.Models.Where(
                    model => model.AllAnimations.Contains(active))
                    .SelectMany(model => model.Bones, (model, bone) => bone.Name)).Distinct());
                var editableCurves = SelectedCurves.Where(
                    x => bones.Contains(GetBoneVisibilityAnim(x.ParentName, ref active).bone_name)).ToList();

                foreach (var curve in editableCurves)
                {
                    string parentName = curve.ParentName;
                    string curveName = curve.Name;
                    int curveComponentIndex = curve.ComponentIndex;

                    if (TargetGroup.Active is BoneVisibilityAnimation)
                    {
                        var curve1 = curve;
                        commandSet.Add(
                            new GeneralGroupValueEditCommand<bool>(
                                TargetGroup,
                                TargetGuiObjectID,
                                (bool) (((UICheckBox) sender).Checked),
                                delegate(ref GuiObject target, ref object data, ref object swap)
                                {
                                    BoneVisibilityAnimation.BoneVisibilityBoneAnim dst =
                                        GetBoneVisibilityAnim(parentName, ref target);
                                    swap = dst.binarize_visibility;
                                    dst.binarize_visibility = (bool) data;

                                    curve1.UpdateIsModified(target);

                                    // バイナリ出力がOFFの場合は圧縮可能のチェックボックスは編集できない
                                    animationObjectPropertyPanel_.SetChkCompressEnable(dst.binarize_visibility);

                                    // Viewerへ再転送
                                    AnimationDocument anim = target as AnimationDocument;
                                    Viewer.LoadOrReloadAnimation.Send(anim);
                                }
                                )
                            );
                    }
                    else
                    {
                        Debug.Assert(false, "未実装");
                    }
                }
            }
            if (commandSet.CommandCount > 0)
            {
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void UpdateCurveElement()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                UpdateSelectedCurve(this, EventArgs.Empty);
            }
        }

        public void SetInterpolationTypeButtons(bool enabled, bool clampValue, bool scaling)
        {
            tsbToHermiteKey.Enabled = enabled;
            tsbToStepKey.Enabled = enabled;
            tsbToLinearKey.Enabled = enabled;

            SetSlopeControlState(enabled);

            // スナップ
            if (!enabled)
            {
                using (var block = new UpdateBlock(tscValueSnap))
                {
                    tscValueSnap.Items.Clear();
                    tscValueSnap.Items.Add("1");
                    tscValueSnap.SelectedIndex = 0;
                }
            }

            tsbClampValue.Enabled = clampValue;
        }

        private void SetSlopeControlState(bool enabled)
        {
            tsbToSplineSlope.Enabled = enabled;
            tsbToLinearSlope.Enabled = enabled;
            tsbToFlatSlope.Enabled = enabled;

            tsbUnionTangent.Enabled = enabled;
            tsbBreakTangent.Enabled = enabled;

            tslSlopeLabel.Enabled = enabled;
            tscInSlope.Enabled = enabled;
            tscOutSlope.Enabled = enabled;

            tsbAutoSplineSlope.Enabled = enabled;
        }

        private void contextMenuStrip_SubMenuClickaterialShaderPage(object sender, EventArgs e)
        {
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            // 要素単体を指す場合はマテリアル全体まで遡る
            while (node.Level > 1)
            {
                node = (CurveTreeNode)node.Parent;
            }

            string materialName = null;
            string groupName = null;
            string paramName = null;
            string elemName = null;

            MakeShaderParamNameFromNode(
                node,
                out materialName,
                out groupName,
                out paramName,
                out elemName
            );

            Debug.Assert(materialName != null);

            // 対象のモデルを取得
            ToolStripMenuItem toolStripMenuItem = (ToolStripMenuItem)sender;
            Model model = DocumentManager.Models.FirstOrDefault(x => x.Name == toolStripMenuItem.Text);
            if (model == null)
            {
                return;
            }

            // 対象のマテリアルを取得
            Material material = model.Materials.FirstOrDefault(x => x.Name == node.Text);
            if (material == null)
            {
                return;
            }

            using (var wait = new WaitCursor())
            {
                // プロパティウィンドウを生成
                var objectGroup = new GuiObjectGroup(material);

                var dialog = new ObjectPropertyDialog(null, objectGroup);
                dialog.TargetFixed = true;

                // 現在のウィンドウの上に表示する
                Control parent = Parent;
                while (parent != null)
                {
                    if (parent is ObjectPropertyDialog)
                    {
                        dialog.SetLocationNextTo((ObjectPropertyDialog)parent);
                        break;
                    }
                    parent = parent.Parent;
                }

                // シェーダーを選択
                var materialPropertyPanel = (MaterialPropertyPanel)dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.Material);
                materialPropertyPanel.SelectShader(groupName, paramName, elemName);

                dialog.Show();
            }
        }

        private void contextMenuStrip_SubMenuClickaterialSamplerPage(object sender, EventArgs e)
        {
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            // サンプラ要素が単体が選択された場合
            string samplerName = null;
            if (node.Info.Category == CurveTreeNodeCategory.TexPatAnim)
            {
                samplerName = node.Text;
            }

            // 要素単体を指す場合はマテリアル全体まで遡る
            while (node.Level > 1)
            {
                node = (CurveTreeNode)node.Parent;
            }

            // 対象のモデルを取得
            ToolStripMenuItem toolStripMenuItem = (ToolStripMenuItem)sender;
            Model model = DocumentManager.Models.FirstOrDefault(x => x.Name == toolStripMenuItem.Text);
            if (model == null)
            {
                return;
            }

            // 対象のマテリアルを取得
            Material material = model.Materials.FirstOrDefault(x => x.Name == node.Text);
            if (material == null)
            {
                return;
            }

            // 対象のサンプラを取得
            samplerType sampler = null;
            if (samplerName != null)
            {
                sampler = material.sampler_array.sampler.FirstOrDefault(x => x.name == samplerName);
            }

            using (var wait = new WaitCursor())
            {
                // プロパティウィンドウを生成
                var objectGroup = new GuiObjectGroup(material);

                ObjectPropertyDialog dialog = new ObjectPropertyDialog(null, objectGroup);
                dialog.TargetFixed = true;

                // 現在のウィンドウの上に表示する
                Control parent = Parent;
                while (parent != null)
                {
                    if (parent is ObjectPropertyDialog)
                    {
                        dialog.SetLocationNextTo((ObjectPropertyDialog)parent);
                        break;
                    }
                    parent = parent.Parent;
                }

                // サンプラを選択
                var materialPropertyPanel = (MaterialPropertyPanel)dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.Material);
                materialPropertyPanel.SelectSampler(sampler);

                dialog.Show();
            }
        }

        // アニメーションが複数モデルにバインドされている際の一覧を作成
        private void SetupMaterialShaderPage(ToolStripMenuItem toolStripMenuItem)
        {
            // 対象のマテリアル名を取得
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            // 要素単体を指す場合はマテリアル全体まで遡る
            while (node.Level > 1)
            {
                node = (CurveTreeNode)node.Parent;
            }

            // 対象のマテリアルを持つモデル一覧を作成
            toolStripMenuItem.DropDownItems.Clear();
            foreach (Model model in DocumentManager.Models.Where(x => x.AllAnimations.Any(y => y == TargetGroup.Active)))
            {
                foreach (Material material in model.Materials.Where(x => x.Name == node.Text))
                {
                    ToolStripMenuItem item = new ToolStripMenuItem();
                    item.Text = model.Name;
                    item.Click += contextMenuStrip_SubMenuClickaterialShaderPage;
                    toolStripMenuItem.DropDownItems.Add(item);
                }
            }
        }

        private void SetupShowSamplerPage(ToolStripMenuItem toolStripMenuItem)
        {
            // 対象のマテリアル名を取得
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            // 要素単体を指す場合はマテリアル全体まで遡る
            while (node.Level > 1)
            {
                node = (CurveTreeNode)node.Parent;
            }

            // 対象のマテリアルを持つモデル一覧を作成
            toolStripMenuItem.DropDownItems.Clear();
            foreach (Model model in DocumentManager.Models.Where(x => x.AllAnimations.Any(y => y == TargetGroup.Active)))
            {
                foreach (Material material in model.Materials.Where(x => x.Name == node.Text))
                {
                    ToolStripMenuItem item = new ToolStripMenuItem();
                    item.Text = model.Name;
                    item.Click += contextMenuStrip_SubMenuClickaterialSamplerPage;
                    toolStripMenuItem.DropDownItems.Add(item);
                }
            }
        }

        public void CurveTreeView_ContextMenuPopup(object sender, ContextMenuPopupEventArgs e)
        {
            bool applyOriginal = false;
            bool applySpecificOriginal = false;
            bool modeTextureSRT = false;
            bool showSamplerPage = false;
            bool enableShowSamplerPage = false;
            bool showMaterialShaderPage = false;
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            switch (TargetGuiObjectID)
            {
                case GuiObjectID.MaterialAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                    switch (node.Info.Category)
                    {
                        case CurveTreeNodeCategory.ParamAnim:
                            applyOriginal = true;
                            applySpecificOriginal = true;
                            {
                                IHasShaderParameterAnimation animation = (IHasShaderParameterAnimation)TargetGroup.Active;
                                if (animation != null)
                                {
                                    ShaderParameterAnimation.ParamAnim paramAnim = animation.GetParamAnimFromId(getMaterialName(node), node.id);
                                    if (paramAnim != null)
                                    {
                                        modeTextureSRT = paramAnim.IsTexSRT;
                                    }
                                }
                            }
                            showMaterialShaderPage = true;
                            break;
                        case CurveTreeNodeCategory.MaterialAnim:
                        case CurveTreeNodeCategory.ShaderParamAnim:
                            applyOriginal = true;
                            break;
                        case CurveTreeNodeCategory.PerMaterialAnim:
                        case CurveTreeNodeCategory.ShaderParamMatAnim:
                        case CurveTreeNodeCategory.ShaderParamGroup:
                            showMaterialShaderPage = true;
                            applyOriginal = true;
                            break;
                        case CurveTreeNodeCategory.ShaderParamAnimTarget:
                            showMaterialShaderPage = true;
                            applySpecificOriginal = true;
                            break;
                    }

                    if (TargetGuiObjectID == GuiObjectID.MaterialAnimation)
                    {
                        goto case GuiObjectID.TexturePatternAnimation;
                    }
                    break;
                case GuiObjectID.TexturePatternAnimation:
                    switch (node.Info.Category)
                    {
                        case CurveTreeNodeCategory.TexPatAnim:
                            {
                                var parent = node.Parent as CurveTreeNode;
                                if (parent != null && (parent.Info.Category == CurveTreeNodeCategory.TexPatAnimMaterial || parent.Info.Category == CurveTreeNodeCategory.PerMaterialAnim))
                                {
                                    var materialName = parent.Info.Id;
                                    var samplerName = node.Info.Id;

                                    var samplerpair = DocumentManager.Materials
                                        .SelectMany((m) => m.ResolvedSamplers, (m, s) => new { material = m, sampler = s })
                                        .Where(x => x.material.Name == materialName && x.sampler.name == samplerName)
                                        .FirstOrDefault();

                                    if (samplerpair != null)
                                    {
                                        enableShowSamplerPage = true;
                                    }
                                }
                            }
                            showSamplerPage = true;
                            break;
                        case CurveTreeNodeCategory.PerMaterialAnim:
                        case CurveTreeNodeCategory.TexPatAnimMaterial:
                            {
                                var material = DocumentManager.Materials.FirstOrDefault(x => x.Name == node.Info.Id);
                                if (material != null)
                                {
                                    enableShowSamplerPage = true;
                                }
                            }
                            showSamplerPage = true;
                            break;
                    }
                    break;
                case GuiObjectID.SkeletalAnimation:
                case GuiObjectID.BoneVisibilityAnimation:
                case GuiObjectID.MaterialVisibilityAnimation:
                case GuiObjectID.CameraAnimation:
                case GuiObjectID.FogAnimation:
                case GuiObjectID.LightAnimation:
                    break;
            }

            cmiApplyOriginal.Visible = applyOriginal;
            cmiApplySpecificationOriginalCurve.Visible = applySpecificOriginal;
            cmiMode.Visible = modeTextureSRT;
            cmiShowSamplerPage.Visible = showSamplerPage;
            cmiShowSamplerPage.Enabled = enableShowSamplerPage;
            SetupShowSamplerPage(cmiShowSamplerPage);
            cmiShowMaterialShaderPage.Visible = showMaterialShaderPage;
            cmiShowMaterialShaderPage.Enabled = true;
            SetupMaterialShaderPage(cmiShowMaterialShaderPage);
            uiContextMenuStrip1.Show(CurveTreeView.PointToScreen(e.Location));
        }

        private IEnumerable<IAnimationCurve> SerializableDescendantAnimationCurves(CurveTreeNode node)
        {
            return from info in node.Info.Descendants()
                   let curve = info.AnimationCurve
                   where curve != null
                   let target = curve.GetAnimTarget(TargetGroup.Active)
                   where target != null && target.ExportType != CurveExportType.Ignored
                   select curve;
        }

        private IEnumerable<IAnimationCurve> HasNodesDescendantAnimationCurves(CurveTreeNode node)
        {
            return from info in node.Info.Descendants()
                   let curve = info.AnimationCurve
                   where curve != null
                   let target = curve.GetAnimTarget(TargetGroup.Active)
                   where target != null && target.KeyFrames.Count > 0
                   select curve;
        }

        private void cmiDelete_Click(object sender, EventArgs e)
        {
            var node = (CurveTreeNode)CurveTreeView.SelectedNode;
            var commandSet = new EditCommandSet();
            using (var filter = new Viewer.ViewerDrawSuppressBlock(AnimationCurveEditCommand.AnimationMessageFilter))
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                IEnumerable<IAnimationCurve> curves;
                switch (TargetGroup.Active.ObjectID)
                {
                    case GuiObjectID.CameraAnimation:
                    case GuiObjectID.LightAnimation:
                    case GuiObjectID.FogAnimation:
                        curves = HasNodesDescendantAnimationCurves(node);
                        break;
                    default:
                        curves = SerializableDescendantAnimationCurves(node);
                        break;
                }

                foreach (var curve in curves)
                {
                    AnimTarget animTarget = curve.CreateTemporaryTarget(TargetGroup.Active);
                    if (TargetGroup.Active.ObjectID == GuiObjectID.ShaderParameterAnimation)
                    {
                        ShaderParameterAnimationCurveTreeNodeInfo shaderCurve = curve as ShaderParameterAnimationCurveTreeNodeInfo;
                        ShaderParameterAnimation.ParamAnimTarget paramAnimTarget = animTarget as ShaderParameterAnimation.ParamAnimTarget;
                        if (shaderCurve != null && paramAnimTarget != null && paramAnimTarget.component_index == 0)
                        {
                            ShaderParameterAnimation.ParamAnim paramAnim = shaderCurve.GetAnimParam(TargetGroup.Active);
                            if (paramAnim != null && paramAnim.IsTexSRT)
                            {
                                ShaderParameterAnimation.SetTexSRTModeValue(paramAnim, shaderCurve.GetMaterialName(), paramAnimTarget);
                            }
                        }
                    }
                    else if (TargetGroup.Active.ObjectID == GuiObjectID.MaterialAnimation)
                    {
                        ShaderParameterAnimationCurveTreeNodeInfo2 shaderCurve = curve as ShaderParameterAnimationCurveTreeNodeInfo2;
                        ShaderParameterAnimation.ParamAnimTarget paramAnimTarget = animTarget as ShaderParameterAnimation.ParamAnimTarget;
                        if (shaderCurve != null && paramAnimTarget != null && paramAnimTarget.component_index == 0)
                        {
                            ShaderParameterAnimation.ParamAnim paramAnim = shaderCurve.GetAnimParam(TargetGroup.Active);
                            if (paramAnim != null && paramAnim.IsTexSRT)
                            {
                                ShaderParameterAnimation.SetTexSRTModeValue(paramAnim, shaderCurve.GetMaterialName(), paramAnimTarget);
                            }
                        }
                    }

                    commandSet.Add(
                        CreateAnimTargetSwapCommand(TargetGroup, TargetGroup.Active.ObjectID, curve, animTarget).Execute());
                }

                var updateBind = ((AnimationDocument)TargetGroup.Active.OwnerDocument).CreateUpdateBindCommand();
                if (updateBind != null)
                {
                    commandSet.Add(updateBind.Execute());
                }

                commandSet.Reverse();

                TheApp.CommandManager.Add(commandSet);
            }
        }

        public static bool CurveIndex(GuiObject target, IAnimationCurve curve, out int parentIndex, out int curveIndex, out int curveComponentIndex, out int curveIndexInBinary, out AnimTarget animTarget)
        {
            parentIndex = -1;
            curveIndex = 0;
            curveComponentIndex = 0;
            curveIndexInBinary = -1;
            animTarget = null;

            switch (target.ObjectID)
            {
                case GuiObjectID.MaterialAnimation:
                    {
                        var animation = target as MaterialAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        if (curve is ShaderParameterAnimationCurveTreeNodeInfo2)
                        {
                            var info = (ShaderParameterAnimationCurveTreeNodeInfo2)curve;
                            info.IntermediateFileIndices(animation, out parentIndex, out curveIndex, out curveComponentIndex);
                        }
                        else if (curve is TexturePatternAnimationCurveTreeNodeInfo2)
                        {
                            var info = (TexturePatternAnimationCurveTreeNodeInfo2)curve;
                            info.IntermediateFileIndices(animation, out parentIndex, out curveIndex);
                        }
                        else if (curve is MaterialVisibilityAnimationCurveTreeNodeInfo2)
                        {
                            var info = (MaterialVisibilityAnimationCurveTreeNodeInfo2)curve;
                            info.IntermediateFileIndices(animation, out parentIndex, out curveIndex);
                        }
                        break;
                    }
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                    {
                        var animation = target as ShaderParameterAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        {
                            var info = (ShaderParameterAnimationCurveTreeNodeInfo)curve;
                            info.IntermediateFileIndices(animation, out parentIndex, out curveIndex, out curveComponentIndex, out curveIndexInBinary);
                        }

                        break;
                    }

                case GuiObjectID.TexturePatternAnimation:
                    {
                        var animation = target as TexturePatternAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        {
                            var info = (TexturePatternAnimationCurveTreeNodeInfo)curve;
                            info.IntermediateFileIndices(animation, out parentIndex, out curveIndex);
                        }

                        break;
                    }

                case GuiObjectID.MaterialVisibilityAnimation:
                    {
                        var animation = target as MaterialVisibilityAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        {
                            var info = (MaterialVisibilityAnimationCurveTreeNodeInfo)curve;
                            info.IntermediateFileIndices(animation, out parentIndex, out curveIndexInBinary);
                        }

                        break;
                    }

                case GuiObjectID.BoneVisibilityAnimation:
                    {
                        var animation = target as BoneVisibilityAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var boneAnimTarget = animTarget as BoneVisibilityAnimation.BoneVisibilityBoneAnim;
                        if (boneAnimTarget == null || !boneAnimTarget.binarize_visibility)
                        {
                            return false;
                        }
                        var exportType = boneAnimTarget.ExportType;
                        if (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant)
                        {
                            return false;
                        }

                        {
                            var info = (BoneVisibilityAnimationCurveTreeNodeInfo)curve;
                            info.IntermediateFileIndices(animation, out parentIndex, out curveIndexInBinary);
                        }

                        break;
                    }

                case GuiObjectID.CameraAnimation:
                    {
                        var animation = target as CameraAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        var info = (CameraAnimationCurveTreeNodeInfo)curve;
                        info.IntermediateFileIndices(animation, out parentIndex, out curveIndex, out curveComponentIndex);

                        break;
                    }

                case GuiObjectID.LightAnimation:
                    {
                        var animation = target as LightAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        var info = (LightAnimationCurveTreeNodeInfo)curve;
                        info.IntermediateFileIndices(animation, out parentIndex, out curveIndex, out curveComponentIndex);

                        break;
                    }

                case GuiObjectID.FogAnimation:
                    {
                        var animation = target as FogAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        var info = (FogAnimationCurveTreeNodeInfo)curve;
                        info.IntermediateFileIndices(animation, out parentIndex, out curveIndex, out curveComponentIndex);

                        break;
                    }
                case GuiObjectID.ShapeAnimation:
                    {
                        var animation = target as ShapeAnimation;
                        animTarget = curve.GetAnimTarget(animation);
                        var exportType = animTarget.ExportType;
                        if (animTarget == null || (exportType != CurveExportType.Curve && exportType != CurveExportType.Constant))
                        {
                            return false;
                        }

                        var info = (ShapeAnimationCurveTreeNodeInfo)curve;
                        info.IntermediateFileIndices(animation, out parentIndex, out curveIndex);

                        break;
                    }
                default:
                    {
                        DebugConsole.WriteLine("SendViewerEditCurve.Execute() -- 未実装");
                        return false;
                    }
            }

            return true;
        }

        public static void SendViewerEditCurve(GuiObject target, IAnimationCurve curve, CurveExportType exportType)
        {
            int parentIndex;
            int curveIndex;
            int curveComponentIndex;
            int curveIndexInBinary;
            AnimTarget animTarget;
            if (CurveIndex(target, curve, out parentIndex, out curveIndex, out curveComponentIndex, out curveIndexInBinary, out animTarget))
            {
                Viewer.EditCurve.Send((AnimationDocument)target.OwnerDocument,
                    (uint)parentIndex,
                    (uint)curveIndex,
                    (uint)curveComponentIndex,
                    curve.IsRotate,
                    animTarget,
                    curveIndexInBinary,
                    target.ObjectID,
                    curve.CurvePrimitiveType,
                    true,
                    exportType
                    );
            }
        }

        /// <summary>
        /// コピー
        /// </summary>
        private void cmiCopy_Click(object sender, EventArgs e)
        {
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            CopiedData = Copy(node.Info, TargetGroup.Active);
        }

        /// <summary>
        /// ペースト
        /// </summary>
        private void cmiPaste_Click(object sender, EventArgs e)
        {
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            TheApp.CommandManager.Execute(CreatePasteCommand(TargetGroup, node.Info));
        }

        public static CopiedCurveNode CopiedData;

        /// <summary>
        /// 現在の状態をコピー
        /// </summary>
        public CopiedCurveNode Copy(CurveTreeInfo node, GuiObject animation)
        {
            Debug.Assert(node != null);
            Debug.Assert(node.Nodes != null);
            Debug.Assert(node.Nodes.All(x => x != null));
            var copiedNode =  new CopiedCurveNode()
            {
                Id = node.Id,
                Text = node.Text,
            };

            if (node.AnimationCurve != null)
            {
                // コピー元が無ければノードが生成される
                copiedNode.AnimTarget = ObjectUtility.Clone(
                    node.AnimationCurve.GetAnimTarget(animation) ??
                    node.AnimationCurve.CreateAnimTarget(animation));
                copiedNode.CurvePrimitiveType = node.AnimationCurve.CurvePrimitiveType;
            }

            copiedNode.Nodes = node.Nodes.Select(x => Copy(x, animation)).ToList();
            return copiedNode;
        }

        /// <summary>
        /// ペーストコマンドの作成
        /// </summary>
        public ICommand CreatePasteCommand(GuiObjectGroup targets, CurveTreeInfo node)
        {
            var commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
            var id = targets.Active.ObjectID;
            var pairs = ColorPair(node, CopiedData);
            if (!pairs.Any())
            {
                pairs = Pair(node, CopiedData);
            }

            foreach (var pair in pairs)
            {
                Debug.Assert(pair != null);
                var animationCurve = pair.Item1;
                var animTarget = pair.Item2;

                if (animationCurve.NonEditableKind != AnimationDocument.NonEditableKind.ShaderParamAnim_DisableByOptionVar)
                {
                    commandSet.Add(CreateAnimTargetEditCommand(targets, id, animationCurve, animTarget));
                    commandSet.Add(new LazyCommand(() => AnimationCurveEditCommand.CreateQuantizeEditCommand(targets, id, animationCurve)));
                }
            }

            return commandSet;
        }

        /// <summary>
        /// AnimTarget の編集
        /// </summary>
        public static ICommand CreateAnimTargetEditCommand(GuiObjectGroup targetGroup, GuiObjectID targetGuiObjectID, IAnimationCurve curve, AnimTarget animTarget)
        {
            return new GeneralGroupReferenceEditCommand<AnimTarget>(
                targetGroup,
                targetGuiObjectID,
                ObjectUtility.MultipleClone(animTarget, targetGroup.GetObjects(targetGuiObjectID).Count),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    AnimTarget curveAnimTarget = curve.GetAnimTarget(target);
                    if (curveAnimTarget == null)
                    {
                        curveAnimTarget = curve.CreateAnimTarget(target);
                    }
                    CurveExportType oldType = curveAnimTarget.ExportType;
                    var oldInterpolation = curveAnimTarget.CurveInterpolationType;
                    AnimTarget oldAnimTarget = new AnimTarget();
                    oldAnimTarget.PartialSet(curveAnimTarget);
                    swap = oldAnimTarget;
                    curveAnimTarget.PartialSet((AnimTarget)data);

                    curve.UpdateIsModified(target);

                    // 転送
                    CurveExportType type = curveAnimTarget.ExportType;
                    if (oldType == CurveExportType.Ignored || type == CurveExportType.Ignored || type == CurveExportType.Dependent || oldInterpolation != curve.CurveInterpolationType)
                    {
                        // 不正アニメーションチェック
                        ((AnimationDocument)target.OwnerDocument).CheckAndDisConnect();
                        Viewer.LoadOrReloadAnimation.Send((AnimationDocument)target.OwnerDocument);
                    }
                    else
                    {
                        // 不正アニメーションチェック
                        ((AnimationDocument)target.OwnerDocument).CheckAndDisConnect();
                        CurveEditorPanel.SendViewerEditCurve(target, curve, type);
                    }
                }
            );
        }

        public static ICommand CreateAnimTargetSwapCommand(GuiObjectGroup targetGroup, GuiObjectID targetGuiObjectID, IAnimationCurve curve, AnimTarget animTarget)
        {
            return new GeneralGroupReferenceEditCommand<AnimTarget>(
                targetGroup,
                targetGuiObjectID,
                ObjectUtility.MultipleClone(animTarget, targetGroup.GetObjects(targetGuiObjectID).Count),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    AnimTarget old = curve.GetAnimTarget(target);
                    if (old == null)
                    {
                        old = curve.CreateAnimTarget(target);
                    }
                    swap = old;
                    var oldInterpolation = old.CurveInterpolationType;
                    CurveExportType oldType = old.ExportType;
                    AnimTarget newAnimTarget = (AnimTarget)data;
                    newAnimTarget.IsModified = old.IsModified;
                    curve.SetAnimTarget(target, newAnimTarget);

                    curve.UpdateIsModified(target);

                    // 転送
                    CurveExportType type = newAnimTarget.ExportType;
                    if (oldType == CurveExportType.Ignored || type == CurveExportType.Ignored || type == CurveExportType.Dependent || oldInterpolation != curve.CurveInterpolationType)
                    {
                        // 不正アニメーションチェック
                        ((AnimationDocument)target.OwnerDocument).CheckAndDisConnect();
                        Viewer.LoadOrReloadAnimation.Send((AnimationDocument)target.OwnerDocument);
                    }
                    else
                    {
                        // 不正アニメーションチェック
                        ((AnimationDocument)target.OwnerDocument).CheckAndDisConnect();
                        CurveEditorPanel.SendViewerEditCurve(target, curve, type);
                    }
                }
            );
        }

        /// <summary>
        /// ペースト先とペースト元の組を作る
        /// 対応に誤りがある箇所は null を列挙する
        /// </summary>
        private IEnumerable<Tuple<IAnimationCurve, AnimTarget>> Pair(CurveTreeInfo curveNode, CopiedCurveNode copiedCurveNode)
        {
            if (curveNode.AnimationCurve != null && copiedCurveNode.AnimTarget != null)
            {
                // 型が一致するかどうかの確認
                if (curveNode.AnimationCurve.CurvePrimitiveType != copiedCurveNode.CurvePrimitiveType)
                {
                    return Enumerable.Repeat<Tuple<IAnimationCurve, AnimTarget>>(null, 1);
                }

                return Enumerable.Repeat(new Tuple<IAnimationCurve, AnimTarget>(curveNode.AnimationCurve, copiedCurveNode.AnimTarget), 1);
            }

            if (curveNode.Nodes.Count != copiedCurveNode.Nodes.Count)
            {
                return Enumerable.Repeat<Tuple<IAnimationCurve, AnimTarget>>(null, 1);
            }

            // TODO : 順序を変更しないようにする
            return curveNode.Nodes.OrderBy(x => x.Id).Zip(copiedCurveNode.Nodes.OrderBy(x => x.Id), (x, y) => new { x, y })
                .SelectMany(p => p.x.Id == p.y.Id ?
                    Pair(p.x, p.y) :
                    Enumerable.Repeat<Tuple<IAnimationCurve, AnimTarget>>(null, 1));
        }

        private IEnumerable<Tuple<IAnimationCurve, AnimTarget>> ColorPair(CurveTreeInfo curveNode, CopiedCurveNode copiedCurveNode)
        {
            var nodes = curveNode.Nodes;
            var copiedNodes = copiedCurveNode.Nodes;
            if (curveNode.AnimationCurve == null && copiedCurveNode.AnimTarget == null &&
                (nodes.Count == 3 || nodes.Count == 4) &&
                (copiedNodes.Count == 3 || copiedNodes.Count == 4))
            {
                string[] rgb = new string[] { "R", "G", "B" };
                int i;
                for (i = 0; i < 3; i++)
                {
                    if (nodes[i].AnimationCurve != null &&
                        copiedNodes[i].AnimTarget != null &&
                        nodes[i].AnimationCurve.CurvePrimitiveType == PrimitiveTypeKind.Float &&
                        copiedNodes[i].CurvePrimitiveType == PrimitiveTypeKind.Float &&
                        nodes[i].Text == rgb[i] &&
                        copiedNodes[i].Text == rgb[i])
                    {

                    }
                    else
                    {
                        break;
                    }
                }

                if (i == 3)
                {
                    return nodes.Zip(copiedNodes, (x, y) => new Tuple<IAnimationCurve, AnimTarget>(x.AnimationCurve, y.AnimTarget));
                }
            }

            return Enumerable.Empty<Tuple<IAnimationCurve, AnimTarget>>();
        }

        private IEnumerable<CurveTreeNode> getParamAnimNodes(CurveTreeNode node)
        {
            Func<CurveTreeNode, IEnumerable<CurveTreeNode>> children = x =>
                x.Info.Category != CurveTreeNodeCategory.ParamAnim ?
                x.Nodes.OfType<CurveTreeNode>() :
                Enumerable.Empty<CurveTreeNode>();

            return TreeUtility.PreOrder(node, children).Where(x => x.Info.Category == CurveTreeNodeCategory.ParamAnim);
        }

        private string getMaterialName(CurveTreeNode node)
        {
            while (node.Info.Category != CurveTreeNodeCategory.ShaderParamMatAnim)
            {
                node = (CurveTreeNode)node.Parent;
            }

            return node.id;
        }

        private IEnumerable<CurveTreeNode> getMaterialNodesFromShaderParamAnim(CurveTreeNode node)
        {
            List<CurveTreeNode> targets = new List<CurveTreeNode>();
            TreeNodeCollection children = node.Nodes;
            for (int i = 0; i < children.Count; i++)
            {
                CurveTreeNode child = (CurveTreeNode)children[i];
                if (child.Info.Category == CurveTreeNodeCategory.ShaderParamMatAnim || child.Info.Category == CurveTreeNodeCategory.PerMaterialAnim)
                {
                    targets.Add(child);
                }
            }

            return targets;
        }

        private void uiContextMenuStrip1_Opening(object sender, CancelEventArgs e)
        {
            // Enable の設定
            {
                var node = CurveTreeView.SelectedNode as CurveTreeNode;
                //bool allBound = !node.Info.Descendants().Any(x => (x.AnimationCurve != null && x.AnimationCurve.KeyFrames.Any() && !x.IsBound));
                var anims = node.Info.Descendants().Where(x => (x.AnimationCurve != null && x.AnimationCurve.KeyFrames.Any()));
                var allBound = anims.All(x => x.IsBound);
                var anyBound = anims.Any(x => x.IsBound);

                if (node != null)
                {
                    // カーブの編集可否を取得する。
                    var IsCurveEditable = false;
                    var editable = CurveTreeView.GetEditable(node);
                    if (editable == PropertyEdit.CurveTreeView.Editable.Yes)
                    {
                        IsCurveEditable = true;
                    }
                    // シェーダーアニメーションのときは編集不可でもペーストできるようにする
                    var isShaderParamAnim = TargetGroup.Active is IHasShaderParameterAnimation;

                    if (CopiedData != null && (allBound || (anyBound && isShaderParamAnim))
                        && (ColorPair(node.Info, CopiedData).Any() || Pair(node.Info, CopiedData).All(x => x != null))
                        && (IsCurveEditable || isShaderParamAnim))
                    {
                        cmiPaste.Enabled = true;
                    }
                    else
                    {
                        cmiPaste.Enabled = false;
                    }

                    // 削除
                    switch (TargetGroup.Active.ObjectID)
                    {
                        case GuiObjectID.CameraAnimation:
                        case GuiObjectID.LightAnimation:
                        case GuiObjectID.FogAnimation:
                            cmiDelete.Enabled = HasNodesDescendantAnimationCurves(node).Any();
                            break;
                        case GuiObjectID.SkeletalAnimation:
                        case GuiObjectID.ShapeAnimation:
                            cmiDelete.Enabled = false;
                            break;
                        default:
                            cmiDelete.Enabled = SerializableDescendantAnimationCurves(node).Any();
                            break;
                    }

                    // cmiShowMaterialShaderPage
                    {
                        cmiShowMaterialShaderPage.Enabled = false;

                        var target = TargetGroup.Active;
                        if (target is IHasShaderParameterAnimation)
                        {
                            // バインドされている
                            if (DocumentManager.Models.SelectMany(x => x.AllAnimations).Any(x => x == target))
                            {
                                string materialName = null;
                                string groupName = null;
                                string paramName = null;
                                string elemName = null;

                                MakeShaderParamNameFromNode(
                                    node,
                                    out materialName,
                                    out groupName,
                                    out paramName,
                                    out elemName
                                );

                                cmiShowMaterialShaderPage.Enabled = DocumentManager.Materials.Any(x => x.Name == materialName);
                            }
                        }
                    }

                    // オリジナル適用
                    if (allBound)
                    {
                        switch (node.Info.Category)
                        {
                            case CurveTreeNodeCategory.ParamAnim:
                                {
                                    // 対象のアニメーション取得
                                    IHasShaderParameterAnimation animation = (IHasShaderParameterAnimation)TargetGroup.Active;

                                    // このノードの指し示すParamAnimが対象となる。
                                    if (!animation.IsExistOriginalAnim())
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                        cmiApplySpecificationOriginalCurve.Enabled = false;
                                        break;
                                    }

                                    List<CurveTreeNode> nodes = new List<CurveTreeNode>();
                                    nodes.Add(node);
                                    if (animation.IsExistOriginalParamAnim(getMaterialName(node), nodes) && IsCurveEditable)
                                    {
                                        cmiApplyOriginal.Enabled = true;
                                    }
                                    else
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                    }

                                    // 特定カーブを適用できるか？を判定する。
                                    if (animation.IsExistOriginalParamAnim(getMaterialName(node), node) && IsCurveEditable)
                                    {
                                        cmiApplySpecificationOriginalCurve.Enabled = true;
                                    }
                                    else
                                    {
                                        cmiApplySpecificationOriginalCurve.Enabled = false;
                                    }
                                }
                                break;
                            case CurveTreeNodeCategory.PerMaterialAnim:
                            case CurveTreeNodeCategory.ShaderParamMatAnim:
                                {
                                    // 対象のアニメーション取得
                                    IHasShaderParameterAnimation animation = (IHasShaderParameterAnimation)TargetGroup.Active;

                                    if (!animation.IsExistOriginalAnim())
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                        break;
                                    }

                                    // このマテリアルが指し示すParamMatAnimが保持するParamAnimが対象となる。
                                    IEnumerable<CurveTreeNode> nodes = getParamAnimNodes(node);
                                    if (animation.IsExistOriginalParamAnim(node.id, nodes) && IsCurveEditable)
                                    {
                                        cmiApplyOriginal.Enabled = true;
                                    }
                                    else
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                    }
                                }
                                break;
                            case CurveTreeNodeCategory.MaterialAnim:
                            case CurveTreeNodeCategory.ShaderParamAnim:
                                {
                                    cmiShowMaterialShaderPage.Enabled = false;

                                    // 対象のアニメーション取得
                                    IHasShaderParameterAnimation animation = (IHasShaderParameterAnimation)TargetGroup.Active;

                                    if (!animation.IsExistOriginalAnim())
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                        break;
                                    }
                                    // (ShaderParameterAnimation)TargetGroup.Activeの保持するアニメーションすべてが対象となる。
                                    IEnumerable<CurveTreeNode> materialNodes = getMaterialNodesFromShaderParamAnim(node);
                                    bool exist = false;
                                    foreach (var materialNode in materialNodes)
                                    {
                                        IEnumerable<CurveTreeNode> nodes = getParamAnimNodes(materialNode);
                                        if (animation.IsExistOriginalParamAnim(materialNode.id, nodes))
                                        {
                                            exist = true;
                                            break;
                                        }
                                    }
                                    if (exist && IsCurveEditable)
                                    {
                                        cmiApplyOriginal.Enabled = true;
                                    }
                                    else
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                    }
                                }
                                break;
                            case CurveTreeNodeCategory.ShaderParamGroup:
                                {
                                    // 対象のアニメーション取得
                                    IHasShaderParameterAnimation animation = (IHasShaderParameterAnimation)TargetGroup.Active;

                                    if (!animation.IsExistOriginalAnim())
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                        break;
                                    }
                                    // 配下のParamAnimをすべて列挙する。
                                    IEnumerable<CurveTreeNode> nodes = getParamAnimNodes(node);
                                    if (animation.IsExistOriginalParamAnim(getMaterialName(node), nodes) && IsCurveEditable)
                                    {
                                        cmiApplyOriginal.Enabled = true;
                                    }
                                    else
                                    {
                                        cmiApplyOriginal.Enabled = false;
                                    }
                                }
                                break;
                            case CurveTreeNodeCategory.ShapeAnimTarget:
                                break;
                            case CurveTreeNodeCategory.ShaderParamAnimTarget:
                                {
                                    // 対象のアニメーション取得
                                    IHasShaderParameterAnimation animation = (IHasShaderParameterAnimation)TargetGroup.Active;
                                    if (animation == null)
                                    {
                                        cmiApplySpecificationOriginalCurve.Enabled = false;
                                        break;
                                    }
                                    bool floatCurve = node.Info.AnimationCurve != null && node.Info.AnimationCurve.CurvePrimitiveType == PrimitiveTypeKind.Float;
                                    if (animation.IsExistOriginalAnim() && IsCurveEditable && floatCurve)
                                    {
                                        cmiApplySpecificationOriginalCurve.Enabled = true;
                                    }
                                    else
                                    {
                                        cmiApplySpecificationOriginalCurve.Enabled = false;
                                    }
                                    break;
                                }
                            default:
                                //Debug.Assert(false);
                                break;
                        }
                    }
                    else
                    {
                        cmiApplyOriginal.Enabled = false;
                        cmiApplySpecificationOriginalCurve.Enabled = false;
                    }

                    // TexSRTのモード変更
                    if (allBound)
                    {
                        cmiMode.Enabled = true;
                        if (node.Info.Category == CurveTreeNodeCategory.ParamAnim && IsCurveEditable)
                        {
                            if (node.Info.Nodes != null &&
                                node.Info.Nodes.Any())
                            {
                                CurveTreeInfo child = node.Info.Nodes[0];
                                if (child.AnimationCurve != null &&
                                    child.AnimationCurve.KeyFrames != null &&
                                    child.AnimationCurve.KeyFrames.Any())
                                {
                                    float mode = child.AnimationCurve.KeyFrames[0].Value;
                                    cmiMode_Maya.Checked = false;
                                    cmiMode_3dsMax.Checked = false;
                                    cmiMode_Softimage.Checked = false;
                                    if (mode == 0.0f)
                                    {
                                        cmiMode_Maya.Checked = true;
                                    }
                                    else if (mode == 1.0f)
                                    {
                                        cmiMode_3dsMax.Checked = true;
                                    }
                                    else if (mode == 2.0f)
                                    {
                                        cmiMode_Softimage.Checked = true;
                                    }
                                }
                            }
                        }
                        else
                        {
                            cmiMode.Enabled = false;
                        }
                    }
                    else
                    {
                        cmiMode.Enabled = false;
                    }
                }

                if (node.Info.AnimationCurve == null)
                {
                    cmiShowSelectedCurveOnly.Visible = false;
                    cmiShowSelectedFolderCurveOnly.Visible = true;
                }
                else
                {
                    cmiShowSelectedCurveOnly.Visible = true;
                    cmiShowSelectedFolderCurveOnly.Visible = false;
                }
            }
        }

        /// <summary>
        /// オリジナルを適用
        /// </summary>
        private void cmiApplyOriginal_Click(object sender, EventArgs e)
        {
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                return;
            }

            // 対象のアニメーション取得
            var animation = (IHasShaderParameterAnimation)TargetGroup.Active;
            var commandSet = new EditCommandSet();

            switch (node.Info.Category)
            {
                case CurveTreeNodeCategory.ParamAnim:
                    {
                        // このノードの指し示すParamAnimが対象となる。
                        var nodes = new List<CurveTreeNode> {node};
                        animation.UpdateParamAnimFromOriginalAnim(getMaterialName(node), nodes, commandSet);
                    }
                    break;
                case CurveTreeNodeCategory.PerMaterialAnim:
                case CurveTreeNodeCategory.ShaderParamMatAnim:
                    {
                        // このマテリアルが指し示すParamMatAnimが保持するParamAnimが対象となる。
                        IEnumerable<CurveTreeNode> nodes = getParamAnimNodes(node);
                        animation.UpdateParamAnimFromOriginalAnim(node.id, nodes, commandSet);
                    }
                    break;
                case CurveTreeNodeCategory.MaterialAnim:
                case CurveTreeNodeCategory.ShaderParamAnim:
                    {
                        // (ShaderParameterAnimation)TargetGroup.Activeの保持するアニメーションすべてが対象となる。
                        IEnumerable<CurveTreeNode> materialNodes = getMaterialNodesFromShaderParamAnim(node);
                        foreach (var materialNode in materialNodes)
                        {
                            IEnumerable<CurveTreeNode> nodes = getParamAnimNodes(materialNode);
                            animation.UpdateParamAnimFromOriginalAnim(materialNode.id, nodes, commandSet);
                        }
                    }
                    break;
                case CurveTreeNodeCategory.ShaderParamGroup:
                    {
                        // 配下のParamAnimをすべて列挙する。
                        IEnumerable<CurveTreeNode> nodes = getParamAnimNodes(node);
                        animation.UpdateParamAnimFromOriginalAnim(getMaterialName(node), nodes, commandSet);
                    }
                    break;
                case CurveTreeNodeCategory.ShapeAnimTarget:
                    // シェイプアニメーション用
                    break;
            }

            if (commandSet.CommandCount > 0)
            {
                TheApp.CommandManager.Add(commandSet);
            }

        }

        /// <summary>
        /// 特定のオリジナルカーブを適用する
        /// </summary>
        private void cmiApplySpecificationOriginalCurve_Click(object sender, EventArgs e)
        {
            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                // まぁ、こんなことは無い。
                return;
            }

            // 対象のアニメーション取得
            IHasShaderParameterAnimation animation = (IHasShaderParameterAnimation)TargetGroup.Active;
            string materialName = getMaterialName(node);

            bool modeElement = false;
            bool modeColor = false;
            switch (node.Info.Category)
            {
                case CurveTreeNodeCategory.ParamAnim:
                    {
                        ShaderParameterAnimation.ParamAnim paramAnim = animation.GetParamAnimFromId(materialName, node.id);
                        if (paramAnim == null)
                        {
                            return;
                        }
                        if (paramAnim.type == shader_param_typeType.@float)
                        {
                            modeElement = true;
                        }
                        else if (paramAnim.type == shader_param_typeType.float4 ||
                                 paramAnim.type == shader_param_typeType.float3)
                        {
                            modeColor = true;
                        }

                        // 作成するアニメーションをダイアログで確認する。
                        using (var dialog = new OriginalAnimOverwriteSelectionDialog(animation.OriginalMaterialAnimArray/*Data.original_material_anim_array*/, modeElement, modeColor))
                        {
                            if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK)
                            {
                                EditCommandSet commandSet = new EditCommandSet();

                                if (modeElement)
                                {
                                    // このノードのParamAnimが対象で、オリジナルのカーブの要素で適用する。
                                    animation.UpdateParamAnimFromOriginalAnimTarget(materialName, node, 0,
                                                                                    dialog.OverWriteColorAnim, dialog.OverWriteColorAnimTarget,
                                                                                    dialog.OverWriteTexSRTAnim, dialog.OverWriteTexSRTAnimTarget,
                                                                                    commandSet);
                                }
                                else
                                {
                                    // このノードの指し示すParamAnimが対象となる。
                                    animation.UpdateParamAnimFromSpecificOriginalAnim(materialName, node, dialog.OverWriteColorAnim, dialog.OverWriteTexSRTAnim, commandSet);
                                }

                                TheApp.CommandManager.Add(commandSet);
                            }
                        }
                        break;
                    }
                case CurveTreeNodeCategory.ShaderParamAnimTarget:
                    {
                        modeElement = true;
                        // 作成するアニメーションをダイアログで確認する。
                        using (var dialog = new OriginalAnimOverwriteSelectionDialog(animation.OriginalMaterialAnimArray, modeElement, modeColor))
                        {
                            if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK)
                            {
                                var parentNode = (CurveTreeNode)node.Parent;
                                int componentIndex = 0;
                                foreach (var targetNode in parentNode.Nodes)
                                {
                                    if (targetNode == node)
                                    {
                                        break;
                                    }
                                    componentIndex++;
                                }

                                var commandSet = new EditCommandSet();
                                // このノードのParamAnimが対象で、オリジナルのカーブの要素で適用する。
                                animation.UpdateParamAnimFromOriginalAnimTarget(materialName, parentNode, componentIndex,
                                                                                dialog.OverWriteColorAnim, dialog.OverWriteColorAnimTarget,
                                                                                dialog.OverWriteTexSRTAnim, dialog.OverWriteTexSRTAnimTarget,
                                                                                commandSet);


                                TheApp.CommandManager.Add(commandSet);
                            }
                        }


                        break;
                    }
            }


        }

        private void tsbClampFrame_Click(object sender, EventArgs e)
        {
            UpdateConfig();
            CurveView.Invalidate();
        }

        private void tsbClampValue_Click(object sender, EventArgs e)
        {
            isClampValue = ApplicationConfig.Setting.CurveEditor.IsClampValue = tsbClampValue.Checked;
            CurveView.Invalidate();
        }

        private void cmiMode_Click(object sender, EventArgs e)
        {
            // モード変更(Maya/3dsMax/Softimage)
            float value = 0.0f;
            if (sender == cmiMode_Maya)
            {
                value = 0.0f;
            }
            else if (sender == cmiMode_3dsMax)
            {
                value = 1.0f;
            }
            else if (sender == cmiMode_Softimage)
            {
                value = 2.0f;
            }


            var node = CurveTreeView.SelectedNode as CurveTreeNode;
            if (node == null)
            {
                // まぁ、こんなことは無い。
                return;
            }
            if (node.Info.Category != CurveTreeNodeCategory.ParamAnim)
            {
                // まぁ、こんなことは無い。
                return;
            }
            var modeNode = node.Info.Nodes[0];


            // 対象のアニメーション取得
            EditCommandSet commandSet = new EditCommandSet();
            KeyFrame keyframe = modeNode.AnimationCurve.KeyFrames.FirstOrDefault();
            if (keyframe == null)
            {
                keyframe = new KeyFrame();
            }

            commandSet.Add(CreateKeyFrameValueEditCommand(TargetGroup.Active, modeNode.AnimationCurve, keyframe, value));

            TheApp.CommandManager.Add(commandSet.Execute());

        }


        /// <summary>
        /// テクスチャパターンアニメーションカーブのツリーからマテリアルのサンプラを開く
        /// </summary>
        private void cmiShowSamplerPage_Click(object sender, EventArgs e)
        {
            // モデルの選択が入るので、ここではプロパティウインドウを開かない
        }

        private void MakeShaderParamNameFromNode(
            CurveTreeNode node,
            out string materialName,
            out string groupName,
            out string paramName,
            out string elemName
        )
        {
            materialName = null;
            groupName = null;
            paramName = null;
            elemName = null;

            while(node != null)
            {
                switch (node.Info.Category)
                {
                    // 例)PierrotAllDance2
                    case CurveTreeNodeCategory.ShaderParamAnim:
                        break;

                    // 例)PierrotAllMat
                    case CurveTreeNodeCategory.PerMaterialAnim:
                    case CurveTreeNodeCategory.ShaderParamMatAnim:
                        materialName = node.Info.Id;
                        break;

                    // 例)基本
                    case CurveTreeNodeCategory.ShaderParamGroup:
                        if (groupName == null)
                        {
                            groupName = node.Info.Id;
                        }
                        break;

                    // 例)テクスチャ行列
                    case CurveTreeNodeCategory.ParamAnim:
                        paramName = node.Info.Id;
                        break;

                    // 例)ScaleU
                    case CurveTreeNodeCategory.ShaderParamAnimTarget:
                        elemName = node.Info.Id;
                        break;
                }

                node = node.Parent as CurveTreeNode;
            }
        }

        private void cmiShowMaterialShaderPage_Click(object sender, EventArgs e)
        {
            // モデルの選択が入るので、ここではプロパティウインドウを開かない
        }

        private void cmiShowSelectedCurveOnly_Click(object sender, EventArgs e)
        {

            var node = (CurveTreeNode)CurveTreeView.SelectedNode;
            if (node == null || node.Info == null || node.Info.AnimationCurve == null)
            {
                return;
            }
            InvisibleAllNode();
            CheckNodeIndividually(node, true);

            var curve = node.Info.AnimationCurve;

            var animationCurves = new List<IAnimationCurve>(1) {curve};
            SelectedCurves = new List<IAnimationCurve>(animationCurves);

            UpdateTreeViewVisibled();
            UpdateCurveTreeView();
            VisibledCurves = new List<IAnimationCurve>(SelectedCurves);
        }

        private void cmiShowSelectedFolderCurveOnly_Click(object sender, EventArgs e)
        {
            InvisibleAllNode();

            var node = (CurveTreeNode)CurveTreeView.SelectedNode;
            if (node == null || node.Info == null)
            {
                return;
            }

            var selectedNodes = CurveTreeView.TreeNodes(node, x => x.Info.AnimationCurve != null);
            var curves = selectedNodes.Select(x => x.Info.AnimationCurve);
            var selectedCurves = new List<IAnimationCurve>(curves);

            foreach (var cnode in selectedNodes)
            {
                CheckNodeIndividually(cnode, true);
            }
            SelectedCurves = new List<IAnimationCurve>(selectedCurves);

            UpdateTreeViewVisibled();
            UpdateCurveTreeView();
            VisibledCurves = new List<IAnimationCurve>(SelectedCurves);

        }

        private ICommand CreateKeyFrameValueEditCommand(GuiObject target, IAnimationCurve curve, KeyFrame keyFrame, float value)
        {
            GuiObjectGroup targetGroup = new GuiObjectGroup(target);

            var newKeyFrames = ObjectUtility.Clone(curve.KeyFrames);
            KeyFrame newKeyFrame;
            if (newKeyFrames.Any())
            {
                newKeyFrame = newKeyFrames[keyFrame.Index];
            }
            else
            {
                newKeyFrame = new KeyFrame();
                newKeyFrames.Add(newKeyFrame);
            }
            newKeyFrame.Value = value;

            return
                AnimationCurveEditCommand.Create(
                    targetGroup,
                    target.ObjectID,
                    curve,
                    newKeyFrames
                );
        }

        private static GroupEditCommand CreateEditCommand_frame_count(GuiObjectGroup targets, int frameCount)
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    targets,
                    targets.Active.ObjectID,
                    frameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var targetId = target.ObjectID;

                        if (   targetId != GuiObjectID.SceneAnimation
                            && targetId != GuiObjectID.SkeletalAnimation
                            && target is AnimationDocument
                            )
                        {
                            var animDoc = target as AnimationDocument;
                            animDoc.FrameCount = frameCount;
                            AnimationDocument.NotifyFrameCountChanged(animDoc, EventArgs.Empty);
                            Viewer.LoadOrReloadAnimation.Send(animDoc);
                        }
                        else if (targetId == GuiObjectID.CameraAnimation)
                        {
                            var anim = target as CameraAnimation;
                            anim.Data.frame_count = frameCount;
                            AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                        else if (targetId == GuiObjectID.FogAnimation)
                        {
                            var anim = target as FogAnimation;
                            anim.Data.frame_count = frameCount;
                            AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                        else if (targetId == GuiObjectID.LightAnimation)
                        {
                            var anim = target as LightAnimation;
                            anim.Data.frame_count = frameCount;
                            AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }

                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    targets.Active.ObjectID,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var targetId = target.ObjectID;

                        if (targetId != GuiObjectID.SceneAnimation
                            && targetId != GuiObjectID.SkeletalAnimation
                            && target is AnimationDocument
                            )
                        {
                            var anim = target as AnimationDocument;
                            anim.Loop = value;
                            Viewer.LoadOrReloadAnimation.Send(anim);
                        }
                        else if (targetId == GuiObjectID.CameraAnimation)
                        {
                            var anim = target as CameraAnimation;
                            anim.Data.loop = value;
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                        else if (targetId == GuiObjectID.FogAnimation)
                        {
                            var anim = target as FogAnimation;
                            anim.Data.loop = value;
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                        else if (targetId == GuiObjectID.LightAnimation)
                        {
                            var anim = target as LightAnimation;
                            anim.Data.loop = value;
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                    }
                );
        }


        private void tsbClampCurrentFrame_CheckedChanged(object sender, EventArgs e)
        {
            if (tsbClampCurrentFrame.Checked)
            {
                CurveView.State.CurrentFrame = Math.Min(FrameCount, Math.Max(CurveView.State.CurrentFrame, 0));
            }

            UpdateConfig();
        }

        private void UpdateConfig()
        {
            ApplicationConfig.Setting.CurveEditor.IsClampFrame = tsbClampFrame.Checked;

            ApplicationConfig.Setting.CurveEditor.IsVisibleFrameValue = IsVisibleFrameValue;
            ApplicationConfig.Setting.CurveEditor.IsVisibleSlope = IsVisibleSlope;
            ApplicationConfig.Setting.CurveEditor.IsVisibleMinMax = IsVisibleMinMax;
            ApplicationConfig.Setting.CurveEditor.IsVisibleCurrentValue = IsVisibleCurrentValue;
            ApplicationConfig.Setting.CurveEditor.IsVisibleCurveName = IsVisibleCurveName;
            ApplicationConfig.Setting.CurveEditor.IsClampCurrentFrame	= tsbClampCurrentFrame.Checked;
            ApplicationConfig.Setting.CurveEditor.IsAutoSplineSlope = tsbAutoSplineSlope.Checked;
            ApplicationConfig.Setting.CurveEditor.IsSnapToOtherKey = tsbSnapToKey.Checked;
        }

        private void ToolStrip_Move(object sender, EventArgs e)
        {
            if (isConfigLoaded_ && watchToolStripMove)
            {
                // Todo:抑制中でもイベントが発生ししてしまうようなので直接みる
                if (UIControlEventSuppressBlock.Enabled == false)
                {
                    ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Edit = new AppConfig.CurveEditor.ToolStripLocationType.State()
                    {
                        Location = EditToolStrip.Location,
                        Collapsed = EditToolStrip.PreferredSize != EditToolStrip.Size,
                    };
                    ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Fit = new AppConfig.CurveEditor.ToolStripLocationType.State()
                    {
                        Location = FitToolStrip.Location,
                        Collapsed = FitToolStrip.PreferredSize != FitToolStrip.Size,
                    };
                    ApplicationConfig.Setting.CurveEditor.ToolStripLocation.KeyFrame = new AppConfig.CurveEditor.ToolStripLocationType.State()
                    {
                        Location = KeyFrameToolStrip.Location,
                        Collapsed = KeyFrameToolStrip.PreferredSize != KeyFrameToolStrip.Size,
                    };
                    ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Play = new AppConfig.CurveEditor.ToolStripLocationType.State()
                    {
                        Location = PlayToolStrip.Location,
                        Collapsed = PlayToolStrip.PreferredSize != PlayToolStrip.Size,
                    };
                    ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Paste = new AppConfig.CurveEditor.ToolStripLocationType.State()
                    {
                        Location = PasteToolStrip.Location,
                        Collapsed = PasteToolStrip.PreferredSize != PasteToolStrip.Size,
                    };
                    ApplicationConfig.Setting.CurveEditor.ToolStripLocation.Snap = new AppConfig.CurveEditor.ToolStripLocationType.State()
                    {
                        Location = SnapToolStrip.Location,
                        Collapsed = SnapToolStrip.PreferredSize != SnapToolStrip.Size,
                    };
                }
            }
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // ドラッグ中は無視する
            if (CurveView.IsDragging && ((keyData & Keys.Z) == Keys.Z || (keyData & Keys.Control) == Keys.Control))
            {
                return true;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        private void tsbPause_Click(object sender, EventArgs e)
        {
            var iPause = TargetGroup.Active as IPause;
            if (iPause != null)
            {
                iPause.SetPause(tsbPause.Checked, CurveView.State.CurrentFrame);
                App.AppContext.OnPauseChanged(true);
            }
        }

        public void ScaleSelectedKeysFrameByUtil(float axis, float frameScale)
        {
            // 入力コントロールを確定させる
            this.ActiveControl = this.CurveView;

            this.EditSelectedKeys(
                (key) => key.Selected,
                (key, curve) =>
                    {
                        // フレームと値
                        var newFrame = (key.Frame - axis) * frameScale + axis;

                        // クランプ
                        if (this.IsClampFrame)
                        {
                            newFrame = Math.Min(Math.Max(newFrame, 0.0f), this.FrameCount);
                        }

                        key.Frame = newFrame;

                        // スロープを縦横比でスケーリングする
                        if (!(Math.Abs(frameScale) > 0.00001f))
                        {
                            return;
                        }

                        var scale = 1.0f / frameScale;

                        if (frameScale > 0.0f)
                        {
                            // フレームのスケールが正数の時
                            key.InSlope *= scale;
                            key.OutSlope *= scale;
                        }
                        else
                        {
                            // フレームのスケールが負数の時
                            var newInSlope = key.OutSlope * scale;
                            var newOutSlope = key.InSlope * scale;

                            key.InSlope = newInSlope;
                            key.OutSlope = newOutSlope;
                        }
                    });
        }

        public void ScaleSelectedKeysValueByUtil(float axis, float scale)
        {
            // 入力コントロールを確定させる
            this.ActiveControl = this.CurveView;

            this.EditSelectedKeys(
                (key) => key.Selected,
                (key, curve) =>
                    {
                        // フレームと値
                        var newValue = (key.Value - axis) * scale + axis;

                        // クランプ
                        newValue = FixValue(newValue, curve.CurvePrimitiveType);

                        if (this.IsClampValue)
                        {
                            newValue = curve.ClampValue(newValue);
                        }

                        key.Value = newValue;

                        key.InSlope *= scale;
                        key.OutSlope *= scale;
                    });
        }

        private void uiContextMenuStripCurveView_Opening(object sender, CancelEventArgs e)
        {

            cmiCurveViewShowSelectedCurveOnly.Enabled = (SelectedCurves.Count > 0);
            if ((CurveView.State.DragBegin.ModifierKeys & Keys.Alt) != 0)
            {
                e.Cancel = true;
            }
        }

        private void cmiCurveViewShowSelectedCurveOnly_Click(object sender, EventArgs e)
        {
            InvisibleAllNode();

            var selectedCurves = new List<IAnimationCurve>(SelectedCurves);
            var selectedNodes = CurveTreeView.AllCurveNodes
                .Where(x => x.Tag is IAnimationCurve)
                .Where(x => selectedCurves.Contains((IAnimationCurve) x.Tag));

            foreach (var node in selectedNodes)
            {
                CheckNodeIndividually(node, true);
            }
            SelectedCurves = new List<IAnimationCurve>(selectedCurves);

            UpdateTreeViewVisibled();
            UpdateCurveTreeView();
            VisibledCurves = new List<IAnimationCurve>(SelectedCurves);
        }

        private void cmiInsertKey_Click(object sender, EventArgs e)
        {
            if (!EditableCurves.Any())
            {
                return;
            }
            AddKeyCurrentFrame();
        }

        private void animationSettingPopupButton_Click(object sender, EventArgs e)
        {
            var btn = sender as AnimationSettingPopupButton;
            Debug.Assert(btn != null);
            btn.TargetGroup = TargetGroup;
        }

        private void tsbAutoSplineSlope_Click(object sender, EventArgs e)
        {
            UpdateConfig();
        }

        private void CurveEditorPanel_VisibleChanged(object sender, EventArgs e)
        {
        }

        public void MoveSelectedKeysByUtil(float frame, float value)
        {
            // 入力コントロールを確定させる
            ActiveControl = CurveView;

            EditSelectedKeys(
                (key) => key.Selected,
                (key, curve) =>
                {
                    // フレームと値
                    var newValue = key.Value + value;
                    var newFrame = key.Frame + frame;

                    // クランプ
                    newValue = FixValue(newValue, curve.CurvePrimitiveType);

                    if (IsClampValue)
                    {
                        newValue = curve.ClampValue(newValue);
                    }
                    if (IsClampFrame)
                    {
                        newFrame = Math.Min(Math.Max(newFrame, 0.0f), FrameCount);
                    }

                    key.Value = newValue;
                    key.Frame = newFrame;
                }
            );

        }

        public void EnableAnimationSettingPopupButton(bool flag)
        {
            animationSettingPopupButton.Enabled = flag;
        }

        private void tsbMoveScaleTool_Click(object sender, EventArgs e)
        {
//            keyMoveScaleUtility.Visible = tsbMoveScaleTool.Checked;
            keyMoveScaleUtility.Display(tsbMoveScaleTool.Checked);
        }

        #region カラーのコピペ

        /// <summary>
        /// コピーまたはペーストの対象
        /// </summary>
        private CurveView.ColorSet ContextMenuColorSet;

        /// <summary>
        /// ペースト時にキーを追加するか?
        /// </summary>
        private bool InsertKeyOnPasteColor = false;

        void CurveView_ContextMenuPopup(object sender, ContextMenuPopupEventArgs e)
        {
            if ((CurveView.State.DragBegin.ModifierKeys & Keys.Alt) != 0)
            {
                // 何もしない
                return;
            }

            InsertKeyOnPasteColor = false;

            cmiInsertKey.Enabled = EditableCurves.Any();
            cmiCurveViewShowSelectedCurveOnly.Enabled = (SelectedCurves.Count > 0);

            bool colorEdit = false;
            switch (TargetGroup.Active.ObjectID)
            {
                case GuiObjectID.MaterialAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.LightAnimation:
                case GuiObjectID.FogAnimation:
                    colorEdit = true;
                    break;
            }

            cmiCurveViewCopyColor.Visible = cmiCurveViewPasteColor.Visible = colorEdit;
            if (colorEdit)
            {
                ContextMenuColorSet = CurveView.State.GetColorKey(e.X, e.Y);
                if (ContextMenuColorSet != null)
                {
                    cmiCurveViewCopyColor.Enabled = true;
                    var mode = ContextMenuColorSet.Curves.Any(x => x.ColorComponentIndex == 3) ? ColorEditMode.RGBA : ColorEditMode.RGB;
                    cmiCurveViewPasteColor.Enabled = ColorEditPanel.CanPaste(mode);
                }
                else
                {
                    bool inColorBar = CurveView.State.IsInHRulerColorBar(e.X, e.Y);
                    if (inColorBar)
                    {
                        InsertKeyOnPasteColor = true;

                        var frame = (float)CurveView.GetXInCurvePlane(e.X);
                        float HDRFactor;
                        bool hasA;
                        var color = CurveView.MakeColorFromColorCurve(frame, out HDRFactor, out hasA);

                        var colorGroup = CurveView.Painter.MakeColorCurveGroup(true);
                        var curves = colorGroup.SelectMany(x => x.Value).ToList();
                        var mode = curves.Any(x => x.ColorComponentIndex == 3) ? ColorEditMode.RGBA : ColorEditMode.RGB;
                        ContextMenuColorSet = new PropertyEdit.CurveView.ColorSet()
                        {
                            Frame = frame,
                            Curves = curves,
                            Color = color,
                        };

                        if (colorGroup.Count == 1)
                        {
                            cmiCurveViewCopyColor.Enabled = true;
                            cmiCurveViewPasteColor.Enabled = ColorEditPanel.CanPaste(mode);
                        }
                        else if (colorGroup.Count == 0)
                        {
                            cmiCurveViewCopyColor.Enabled = false;
                            cmiCurveViewPasteColor.Enabled = false;
                        }
                        else
                        {
                            cmiCurveViewCopyColor.Enabled = false;
                            cmiCurveViewPasteColor.Enabled = ColorEditPanel.CanPaste(mode);
                        }
                    }
                    else
                    {
                        cmiCurveViewCopyColor.Enabled = cmiCurveViewPasteColor.Enabled = false;
                    }
                }
            }

            uiContextMenuStripCurveView.Show(CurveView.PointToScreen(e.Location));
        }

        private void cmiCurveViewCopyColor_Click(object sender, EventArgs e)
        {
            ColorEditPanel.CopiedColor = ContextMenuColorSet.Color;
            ColorEditPanel.CopiedEditMode = ContextMenuColorSet.Curves.Any(x => x.ColorComponentIndex == 3) ? ColorEditMode.RGBA : ColorEditMode.RGB;
        }

        private void cmiCurveViewPasteColor_Click(object sender, EventArgs e)
        {
            var curves = ContextMenuColorSet.Curves.Where(x =>
                ColorEditPanel.CopiedEditMode == ColorEditMode.RGBA ||
                x.ColorComponentIndex < 3).ToArray();

            if (InsertKeyOnPasteColor)
            {
                // 追加
                CurveView.State.InsertColorKey(curves, ContextMenuColorSet.Frame, ColorEditPanel.CopiedColor.Value);
            }
            else
            {
                // 変更
                var commandSet = new EditCommandSet();

                commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                var color = ColorEditPanel.CopiedColor.Value;
                var values = new[] { color.R, color.G, color.B, color.A };
                foreach (var curve in curves)
                {
                    curve.SetCurvesSequentialIndex();
                    var keyFrames = ObjectUtility.Clone(curve.KeyFrames);
                    var key = keyFrames.FirstOrDefault(x => x.Frame == ContextMenuColorSet.Frame);
                    if (key != null)
                    {
                        key.Value = values[curve.ColorComponentIndex];
                        commandSet.Add(
                            AnimationCurveEditCommand.Create(
                                TargetGroup,
                                TargetGuiObjectID,
                                curve,
                                keyFrames));
                    }
                }

                if (commandSet.CommandCount > 0)
                {
                    commandSet.OnPostEdit += (s, a) => Viewer.HioUtility.FlushCurves(TargetGroup, TargetGuiObjectID);
                    TheApp.CommandManager.Add(commandSet.Execute());
                }
            }
        }

        #endregion

        private void CurveView_DragOver(object sender, DragEventArgs e)
        {
            e.Effect = this.IsTextureAcceptable(e) ? DragDropEffects.Copy : DragDropEffects.None;
        }

        private bool IsTextureAcceptable(DragEventArgs e)
        {
            // テクスチャパターンアニメーションでない場合はドロップ不可
            if (this.TargetGroup.Active == null)
            {
                return false;
            }
            bool isTexturePattern =
                (this.TargetGroup.Active.ObjectID == GuiObjectID.TexturePatternAnimation) ||
                ((this.TargetGroup.Active.ObjectID == GuiObjectID.MaterialAnimation) && CurveView.IsSelectedTexturePatternCurve);
            if (!isTexturePattern)
            {
                return false;
            }

            // カーブが選択されていない場合はドロップ不可
            if (!this.SelectedCurves.Any())
            {
                return false;
            }

            var patternAnim = this.TargetGroup.Active as IHasTexturePatternAnimation;
            if (patternAnim == null)
            {
                return false;
            }

            // ファイルであり、モーダルでないかファイルダイアログが開いている時はドラッグＯＫ
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                if (UIForm.IsModalState(TheApp.MainFrame) && !DialogUtility.IsOpened)
                {
                    return false;
                }

                // 追加できるのは1ファイルのみ
                var files = (string[])e.Data.GetData(DataFormats.FileDrop);
                if (files == null || files.Count() != 1)
                {
                    return false;
                }
                var file = files.First();
                var ext = Path.GetExtension(file);
                var filebody = Path.GetFileNameWithoutExtension(file);
                // テクスチャファイル以外がドラッグされたときはドロップ不可
                if (ObjectIDUtility.ExtToId(ext) != GuiObjectID.Texture && ObjectIDUtility.IsTgaExtension(ext) == false)
                {
                    return false;
                }

                // 既にテクスチャパターンに登録されているテクスチャと同名がドラッグされたときはドロップ不可
                if (patternAnim.TexPatterns.Any(pat => pat.tex_name.Equals(filebody, StringComparison.OrdinalIgnoreCase)))
                {
                    return false;
                }
            }
            else if (e.Data.GetDataPresent(typeof(FileTreeView.ViewItem)))
            {
                var source = (FileTreeView.ViewItem)e.Data.GetData(typeof(FileTreeView.ViewItem));
                if (source == null)
                {
                    return false;
                }

                var texture = source.Tag as Texture;
                if (texture == null)
                {
                    return false;
                }
                // 既にテクスチャパターンに登録されているテクスチャと同名別パスがドラッグされたとき
                if (
                    patternAnim.TexPatterns.Any(
                            pat =>
                            pat.tex_name.Equals(texture.Name, StringComparison.OrdinalIgnoreCase)
                            && patternAnim.GetReferenceTexture(pat.tex_name) != texture))
                {
                    return false;
                }
            }
            return true;
        }

        private void CurveView_DragDrop(object sender, DragEventArgs e)
        {
            if (!this.IsTextureAcceptable(e))
            {
                return;
            }

            var commandSets = new EditCommandSet();

            var pos = this.CurveView.PointToClient(new Point(e.X, e.Y));
            var frame = (float)this.CurveView.GetXInCurvePlane(pos.X);
            if (this.FrameSnapFactor > 0)
            {
                frame = (float)(Math.Round(frame / FrameSnapFactor, MidpointRounding.AwayFromZero) * FrameSnapFactor);
            }

            Texture texture;
            List<Texture> patternTextures;
            switch (this.TargetGroup.Active.ObjectID)
            {
                case GuiObjectID.TexturePatternAnimation:
                    {
                        var textures = TexturePatternAnimationPatternPage.DroppedTextures(e, commandSets);
                        if (textures == null || textures.Count() != 1)
                        {
                            return;
                        }
                        texture = textures.First();
                        var patternAnim = this.TargetGroup.Active as TexturePatternAnimation;
                        if (patternAnim == null)
                        {
                            return;
                        }

                        commandSets.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                        // テクスチャパターンアニメーションに登録されていないテクスチャの場合は追加
                        patternTextures = patternAnim.TexPatterns.Select(x => patternAnim.GetReferenceTexture(x.tex_name)).ToList();
                        if (patternTextures.All(x => x != texture))
                        {
                            var addTexCommandSets = new EditCommandSet();
                            TexturePatternAnimationPatternPage.AddTextureToTexturePattern(addTexCommandSets, textures, patternAnim);
                            commandSets.Add(addTexCommandSets.Execute());
                            patternTextures = patternAnim.TexPatterns.Select(x => patternAnim.GetReferenceTexture(x.tex_name)).ToList();
                        }

                        break;
                    }
                case GuiObjectID.MaterialAnimation:
                    {
                        var textures = MaterialAnimationPatternPage.DroppedTextures(e, commandSets);
                        if (textures == null || textures.Count() != 1)
                        {
                            return;
                        }
                        texture = textures.First();
                        var patternAnim = this.TargetGroup.Active as MaterialAnimation;
                        if (patternAnim == null)
                        {
                            return;
                        }

                        commandSets.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
                        // テクスチャパターンアニメーションに登録されていないテクスチャの場合は追加
                        patternTextures = patternAnim.TexPatterns.Select(x => patternAnim.GetReferenceTexture(x.tex_name)).ToList();
                        if (patternTextures.All(x => x != texture))
                        {
                            var addTexCommandSets = new EditCommandSet();
                            MaterialAnimationPatternPage.AddTextureToTexturePattern(addTexCommandSets, textures, patternAnim);
                            commandSets.Add(addTexCommandSets.Execute());
                            patternTextures = patternAnim.TexPatterns.Select(x => patternAnim.GetReferenceTexture(x.tex_name)).ToList();
                        }

                        break;
                    }
                default:
                    {
                        return;
                    }
            }

            var value = patternTextures.IndexOf(texture);
            Debug.Assert(value >= 0);

            foreach (var curve in this.SelectedCurves)
            {
                var keys = new List<KeyFrame>(curve.KeyFrames);
                var key = curve.CreateKey(
                    frame,
                    value,
                    0.0f,
                    0.0f,
                    this.FrameSnapFactor,
                    this.ValueSnapFactor,
                    this.IsClampValue,
                    this.IsClampFrame,
                    this.FrameCount);
                keys.Add(key);
                commandSets.Add(AnimationCurveEditCommand.Create(this.TargetGroup, this.TargetGuiObjectID, curve, keys).Execute());
            }
            commandSets.Reverse();
            commandSets.OnPostEdit += (s, a) => Viewer.HioUtility.FlushCurves(this.TargetGroup, this.TargetGuiObjectID);
            TheApp.CommandManager.Add(commandSets);
        }

        private void tsbSnapToKey_CheckedChanged(object sender, EventArgs e)
        {
            //var isSnapToOtherKey = tsbSnapToKey.Checked;
            ApplicationConfig.Setting.CurveEditor.IsSnapToOtherKey = IsSnapToKey;
            UpdateSnapToolbar();
            CurveView.Invalidate();
        }

        private void UpdateSnapToolbar()
        {
            SnapToolStrip.EnableToolStripItems(IsEditableAnimation);
            if (IsEditableAnimation)
            {
                lblSnapToMultiply.Enabled = !IsSnapToKey;
                tscFrameSnap.Enabled = !IsSnapToKey;
                tscValueSnap.Enabled = !IsSnapToKey;
            }
        }

        public void UpdateNoNodeWarning()
        {
            var filterd = !animationObjectPropertyPanel_.IsAllShowNode ||
                          animationObjectPropertyPanel_.IsOnlyTargetShowNode ||
                          !animationObjectPropertyPanel_.IsShaderParamAnimGroupCondShow ||
                          animationObjectPropertyPanel_.SearchTexts.Any(s => !string.IsNullOrEmpty(s)) ||
                          animationObjectPropertyPanel_.IsShowKeyedMaterial;

            var hasAnimation = false;
            if (TargetGroup.Active is AnimationDocument)
            {
                var animation = (AnimationDocument)TargetGroup.Active;
                hasAnimation = animation != null && animation.HasAnimation;
            }
            else if (TargetGroup.Active is ISceneAnimationObject)
            {
                var animation = (ISceneAnimationObject)TargetGroup.Active;
                hasAnimation = animation != null && animation.HasAnimation;
            }

            var isAnimatableNodeShown = CurveTreeView.AllAnimatableNodes.Any();
            animationObjectPropertyPanel_.ShowNoNodeWarning(filterd && hasAnimation && !isAnimatableNodeShown);
        }
    }

    public class CurveTreeNode : TreeNode
    {
        // 名前
        // Text が実装に縛られないようにするために node に id を割り振ります。
        public string id { get { Debug.Assert(Info.Id != null); return Info.Id; } }

        public CurveTreeInfo Info;
        public bool IsSameStructure(TreeNodeCollection nodes)
        {
            if (nodes.Count == 1)
            {
                var node = nodes[0] as CurveTreeNode;
                if (node != null)
                {
                    return IsSameStructure(node);
                }
            }

            return false;
        }

        // IsSameStructure が true のときに呼ぶこと
        public void CopyInfo(CurveTreeNode sourceTreeNode)
        {
            Info = sourceTreeNode.Info;
            Tag = sourceTreeNode.Tag;
            for (int i = 0; i < Nodes.Count; i++)
            {
                ((CurveTreeNode)Nodes[i]).CopyInfo((CurveTreeNode)sourceTreeNode.Nodes[i]);
            }
        }

        public bool IsSameStructure(CurveTreeNode node)
        {
            if (Text == node.Text && id == node.id && Info.IsBound == node.Info.IsBound)
            {
                if (Nodes.Count == node.Nodes.Count)
                {
                    Debug.Assert(Nodes.OfType<TreeNode>().All(x => x is CurveTreeNode) && node.Nodes.OfType<TreeNode>().All(x => x is CurveTreeNode));
                    return Nodes.OfType<CurveTreeNode>().Zip(node.Nodes.OfType<CurveTreeNode>(), (x, y) => x.IsSameStructure(y)).All(x => x);
                }
            }

            return false;
        }
    }

    /// <summary>
    /// コピペ用のデータ
    /// </summary>
    public class CurveTreeInfo
    {
        public bool									ShowEmptyNode;
        public string								Text;
        public string								Id;
        public List<CurveTreeInfo>					Nodes			= new List<CurveTreeInfo>();
        public Color?								ForeColor;
        public IAnimationCurve						AnimationCurve;
        public CurveTreeNodeCategory				Category;
        public bool									IsBound			= true;
        public AnimationDocument.NonEditableKind	NonEditableKind	= AnimationDocument.NonEditableKind.Editable;
        public object								NonEditableKindDisplayAux	= null;	// NonEditableKind 表示用補助情報
        public bool									IsModified		= false;
        //
        public Image								Image				= null;
        public string								ImageToolTipString	= null;

        public CurveTreeInfo TrimInvisibleNodes()
        {
            if (!ShowEmptyNode && AnimationCurve != null)
            {
                if (!AnimationCurve.HasKeyFrame())
                {
                    return null;
                }
            }

            Nodes = Nodes.Select(x => x.TrimInvisibleNodes()).Where(x => x != null).ToList();
            if (!ShowEmptyNode && AnimationCurve == null && Nodes.Count == 0)
            {
                return null;
            }

            return this;
        }

        public CurveTreeNode ConvertToTreeNode()
        {
            /*
            if (!ShowEmptyNode && AnimationCurve != null)
            {
                if (!AnimationCurve.HasKeyFrame())
                {
                    return null;
                }
            }
            */

            if (AnimationCurve != null)
            {
                AnimationCurve.IsEditable					= IsBound;
                AnimationCurve.NonEditableKind				= NonEditableKind;
                AnimationCurve.NonEditableKindDisplayAux	= NonEditableKindDisplayAux;
            }

            // TextureSRTの場合で、modeの場合、CurveTreeNodeは作成しない。(非表示になるように)
            if (Category == CurveTreeNodeCategory.ShaderParamAnimTarget)
            {
                // 判定
                if (Id == "Mode")
                {
                    return null;
                }
            }

            var node = new CurveTreeNode()
            {
                Text		= Text,
                ToolTipText	= (NonEditableKind != AnimationDocument.NonEditableKind.Editable) ?
                                    UIText.EnumValue(NonEditableKind, NonEditableKindDisplayAux) :
                                    null,
                Info		= this,
                Tag			= AnimationCurve
            };

            if (string.IsNullOrEmpty(ImageToolTipString) == false)
            {
                node.ToolTipText = ImageToolTipString;
            }

            /*
            bool expand = true;
            switch (Category)
            {
                case CurveTreeNodeCategory.ParamAnim:
                case CurveTreeNodeCategory.ShaderParamAnimTarget:
                case CurveTreeNodeCategory.ShaderParamGroup:
                    expand = false;
                    break;
            }

            if (expand)
            {
                node.Expand();
            }*/

            if (ForeColor.HasValue)
            {
                node.ForeColor = ForeColor.Value;
            }
            else if (AnimationCurve != null)
            {
                node.ForeColor = AnimationCurve.CurveColor;
            }

            foreach (var child in Nodes)
            {
                var childNode = child.ConvertToTreeNode();
                if (childNode != null)
                {
                    node.Nodes.Add(childNode);
                }
            }

            if (!ShowEmptyNode && AnimationCurve == null && node.Nodes.Count == 0)
            {
                return null;
            }

            return node;
        }

        // 自分自身も含めた子孫
        public IEnumerable<CurveTreeInfo> Descendants()
        {
            return Enumerable.Repeat(this, 1).Concat(Nodes.SelectMany(x => x.Descendants()));
        }

        // 文字列によるフィルター
        public static void FilterCurveTreeInfoRoot(CurveTreeInfo root, List<string> texts)
        {
            if (texts.Count > 0)
            {
                root.Nodes = root.Nodes.Where(x => x.FilterCurveTreeInfo(texts)).ToList();
            }
        }

        private bool FilterCurveTreeInfo(List<string> texts)
        {
            // 非表示のノードについて一旦 true を返す
            if (Category == CurveTreeNodeCategory.ShaderParamAnimTarget)
            {
                // 判定
                if (Id == "Mode")
                {
                    return true;
                }
            }

            // 大文字小文字の違いを除いてすべての文字列を含めば OK
            if (texts.All(x => Thread.CurrentThread.CurrentUICulture.CompareInfo.IndexOf(Text, x, CompareOptions.IgnoreCase) >= 0))
            {
                return true;
            }

            Nodes = Nodes.Where(x => x.FilterCurveTreeInfo(texts)).ToList();

            // 子が非表示のノードだけのときは表示しない
            if (Nodes.Count == 1 && Nodes[0].Category == CurveTreeNodeCategory.ShaderParamAnimTarget && Nodes[0].Id == "Mode")
            {
                return false;
            }

            return Nodes.Any();
        }
    }

    public enum CurveTreeNodeCategory
    {
        Default,
        ShaderParamAnim,
        ShaderParamMatAnim,
        ParamAnim,
        ShaderParamAnimTarget,
        ShaderParamGroup,
        ShapeAnimTarget,
        TexPatAnimMaterial,
        TexPatAnim,
        MaterialAnim,
        PerMaterialAnim,
    }

    /// <summary>
    /// ペースト用のノードの情報
    /// </summary>
    public class CopiedCurveNode
    {
        public string Id;
        public string Text;
        public List<CopiedCurveNode> Nodes = new List<CopiedCurveNode>();

        // 以下は葉のみ意味を持つ情報
        public AnimTarget AnimTarget;
        public PrimitiveTypeKind CurvePrimitiveType;
    }
}
