﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class UserDataPage : ObjectPropertyPage
    {
        /// <summary>
        /// アクティブターゲット。
        /// </summary>
        private GuiObject ActiveTarget
        {
            get { return Targets.Active; }
        }

        public UserDataPage() :
            base(PropertyPageID.UserData)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_property_window_common_page_a_user_data;
            }
        }

        public static UserDataPage CreateInstance(object arg)
        {
            return new UserDataPage();
        }

        protected override void InitializeFormInternal()
        {
            ;
        }

        private bool isFirstUpdate = true;

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            var iUserData = Targets.Active as IHasUserData;
            if (iUserData == null)
            {
                Debug.Assert(false);
                return;
            }

            // 編集不可のファイルの場合は無効にする
            //if (!Targets.Active.Editable)
            //{
            //    Enabled = false;
            //    return;
            //}
            //Enabled = true;

            Enabled = Targets.Active.Editable;

            uiModifiedMarkLabel1.IsModified = iUserData.UserDataArrayChanged;
            UserDataArray userDataArray = iUserData.UserDataArray;

            chbHexMode.Checked = ApplicationConfig.Setting.PropertyEdit.UserDataPageHexMode;

            {
                int selectIndex = -1;
                if (lvwUserData.SelectedIndices.Count > 0)
                {
                    selectIndex = lvwUserData.SelectedIndices[0];
                }

                lvwUserData.BeginUpdate();
                {
                    lvwUserData.Items.Clear();
                    if (userDataArray != null)
                    {
                        foreach (UserDataArray.UserData userData in userDataArray.Data)
                        {
                            lvwUserData.Items.Add(CreateListItem(userData, ApplicationConfig.Setting.PropertyEdit.UserDataPageHexMode));
                        }
                    }
                }
                lvwUserData.EndUpdate();

                if (lvwUserData.Items.Count > 0)
                {
                    if ((selectIndex != -1) && (selectIndex < lvwUserData.Items.Count))
                    {
                        lvwUserData.Items[selectIndex].Selected = true;
                        lvwUserData.Items[selectIndex].Focused = true;
                    }
                    else
                    {
                        btnAdd.Select();
                    }
                }
                else
                {
                    lvwUserData_SelectedIndexChanged(null, null);
                }
            }

            // ページを開いた直後は最後を選択状態にする
            if(isFirstUpdate)
            {
                if (lvwUserData.Items.Count > 0)
                {
                    var lastIndex = lvwUserData.Items.Count - 1;

                    lvwUserData.Items[lastIndex].Selected = true;
                    lvwUserData.Items[lastIndex].Focused = true;
                }

                isFirstUpdate = false;
            }

            UpdateState();
        }

        public static bool IsModified(IHasUserData iUserData)
        {
            return iUserData != null && iUserData.UserDataArrayChanged;
        }

        private void lvwUserData_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateState();
        }

        private void lvwUserData_SelectionChanged(object sender, EventArgs e)
        {
            UpdateState();
        }

        private void UpdateState()
        {
            bool isSelected = lvwUserData.SelectedIndex != -1;

            btnEdit.Enabled		= isSelected;
            btnDelete.Enabled	= isSelected;
            btnDown.Enabled		= isSelected && (lvwUserData.SelectedIndex >= 0) && (lvwUserData.SelectedIndex <= lvwUserData.Items.Count - 2);;
            btnUp.Enabled		= isSelected && (lvwUserData.SelectedIndex > 0);;
        }

        /// <summary>
        /// リストアイテムの作成
        /// </summary>
        private ListViewItem CreateListItem(UserDataArray.UserData userData, bool hexMode)
        {
            ListViewItem item = new ListViewItem(userData.Name);
            item.Tag = userData;

            string elementTypeName =
                (userData is UserDataArray.UserDataInt) ? Strings.UserData_Type_Int :
                (userData is UserDataArray.UserDataFloat) ? Strings.UserData_Type_Float :
                (userData is UserDataArray.UserDataString) ? Strings.UserData_Type_String :
                (userData is UserDataArray.UserDataWString) ? Strings.UserData_Type_WString :
                (userData is UserDataArray.UserDataStream) ? Strings.UserData_Type_Stream:
                                                                        Strings.UserData_Type_String;
            int count = 0;
            StringBuilder valueString = new StringBuilder();
            {
                if (userData is UserDataArray.UserDataInt)
                {
                    UserDataArray.UserDataInt intItem = userData as UserDataArray.UserDataInt;
                    count = intItem.Item.Count;
                    SetIntStream(intItem, valueString, hexMode);
                }
                else if (userData is UserDataArray.UserDataFloat)
                {
                    UserDataArray.UserDataFloat floatItem = userData as UserDataArray.UserDataFloat;
                    count = floatItem.Item.Count;
                    SetFloatStream(floatItem, valueString);
                }
                else if (userData is UserDataArray.UserDataString)
                {
                    UserDataArray.UserDataString stringItem = userData as UserDataArray.UserDataString;
                    count = stringItem.Item.Count;
                    SetStringStream(stringItem, valueString);
                }
                else if (userData is UserDataArray.UserDataWString)
                {
                    UserDataArray.UserDataWString wstringItem = userData as UserDataArray.UserDataWString;
                    count = wstringItem.Item.Count;
                    SetWStringStream(wstringItem, valueString);
                }
                else if (userData is UserDataArray.UserDataStream)
                {
                    UserDataArray.UserDataStream streamItem = userData as UserDataArray.UserDataStream;
                    count = streamItem.Item.Count;
                    SetStreamStream(streamItem, valueString);
                }
            }

            item.SubItems.Add(elementTypeName);
            item.SubItems.Add(count.ToString());
            item.SubItems.Add(valueString.ToString());

            return item;
        }

        private void SetIntStream(UserDataArray.UserDataInt intItem, StringBuilder valueString, bool hexMode)
        {
            foreach (var item in intItem.Item)
            {
                valueString.AppendFormat(
                    hexMode ?
                    "{0:x08} " :
                    "{0:d} ",
                    item
                );
            }
        }

        private void SetFloatStream(UserDataArray.UserDataFloat floatItem, StringBuilder valueString)
        {
            foreach (var item in floatItem.Item)
            {
                valueString.Append(item.ToString());
                valueString.Append(" ");
            }
        }

        private void SetStringStream(UserDataArray.UserDataString stringArrayItem, StringBuilder valueString)
        {
            foreach (var item in stringArrayItem.Item)
            {
                if (!string.IsNullOrEmpty(item))
                {
                    valueString.Append(item);
                    valueString.Append(" ");
                }
            }
        }

        private void SetWStringStream(UserDataArray.UserDataWString wstringArrayItem, StringBuilder valueString)
        {
            foreach (var item in wstringArrayItem.Item)
            {
                if (!string.IsNullOrEmpty(item))
                {
                    valueString.Append(item);
                    valueString.Append(" ");
                }
            }
        }

        private void SetStreamStream(UserDataArray.UserDataStream streamItem, StringBuilder valueString)
        {
            foreach (var item in streamItem.Item)
            {
                valueString.AppendFormat(
                    "{0:x02} ",
                    item
                );
            }
        }

        /// <summary>
        /// ユーザーデータの作成
        /// </summary>
        private ListViewItem CreateUserData(
            UserDataDialog dialog, int index, ref UserDataArray userDataArray)
        {
            UserDataArray.UserData userData = dialog.GetResult();
            userDataArray.InsertUserData(userData, index);
            ListViewItem item = CreateListItem(userData, ApplicationConfig.Setting.PropertyEdit.UserDataPageHexMode);
            lvwUserData.Items.Insert(index, item);
            return item;
        }

        private void PostEdit(object sender, EventArgs e)
        {
            PostEdit(Targets);
        }

        private static void PostEdit(GuiObjectGroup taregets)
        {
            Viewer.SuspendUnloadTexture.Send(true);

            var editables = taregets.GetObjects(taregets.Active.ObjectID).Where(x => x.Editable).ToArray();

            // ユーザーデータ対象。
            var models = editables.OfType<Model>().ToArray();
            var animations = editables.OfType<AnimationDocument>().ToArray();
            var textures = editables.OfType<Texture>().ToArray();
            var materials = editables.OfType<Material>().ToArray();
            var bones = editables.OfType<Bone>().ToArray();

            // textures を参照しているモデル。
            var textureModels = textures.SelectMany(x => DocumentManager.Models.Where(y => y.ReferenceDocuments.Contains(x))).Distinct().ToArray();

            // textures を参照しているアニメーション。
            var textureAnimations = textures.SelectMany(x => DocumentManager.Animations.Where(y => y.ReferenceDocuments.Contains(x))).Distinct().ToArray();

            // materials を参照しているドキュメント。
            var materialDocuments = materials.Select(x => x.OwnerDocument).Distinct().ToArray();

            // bones を参照しているドキュメント。
            var boneDocuments = bones.Select(x => x.OwnerDocument).Distinct().ToArray();

            // リロードするモデル。
            var reloadModels = models.Concat(textureModels.Concat(materialDocuments.OfType<Model>().Concat(boneDocuments.OfType<Model>()))).Distinct().ToArray();

            // リロードするアニメーション。
            var reloadAnimations = animations.Concat(textureAnimations.Concat(materialDocuments.OfType<AnimationDocument>().Concat(boneDocuments.OfType<AnimationDocument>()))).Distinct().ToArray();

            // Editへは、Modelの参照データ(テクスチャ)を一緒にバイナリ化しないといけないので、
            // ここで送ると、参照データが読み込まれていない可能性がある。
            // なので、HioUtilityに、送信データを保持させる。
            foreach (var model in reloadModels)
            {
                Viewer.HioUtility.AddViewerDrawSuppressBlockModel(model);
                Viewer.ViewerUtility.SendAnimationSet(model);
            }

            // Editへは、Animationの参照データ(テクスチャ)を一緒にバイナリ化しないといけないので、
            // ここで送ると、参照データが読み込まれていない可能性がある。
            // なので、HioUtilityに、送信データを保持させる。
            foreach (var animation in reloadAnimations)
            {
                Viewer.HioUtility.AddViewerDrawSuppressBlockAnim(animation);
            }

            Viewer.SuspendUnloadTexture.Send(false);
        }


#region イベント
        private void chbHexMode_CheckedChanged(object sender, EventArgs e)
        {
            ApplicationConfig.Setting.PropertyEdit.UserDataPageHexMode = chbHexMode.Checked;
            UpdateForm(false, false);
        }

        private void btnUp_Click(object sender, EventArgs e)
        {
            if (lvwUserData.SelectedItems.Count == 0) { return; }
            Debug.Assert(lvwUserData.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwUserData.SelectedItems[0];
            int index = lvwUserData.Items.IndexOf(item);
            if (index == 0) { return; }

            lvwUserData.BeginUpdate();
            {
                lvwUserData.Items.RemoveAt(index);
                lvwUserData.Items.Insert(index - 1, item);
                lvwUserData.EndUpdate();
            }
            btnUp.Focus();

            // コマンド発行
            IHasUserData iUserData = Targets.Active as IHasUserData;
            if (iUserData == null)
            {
                return;
            }
            UserDataArray userDataArray = ObjectUtility.Clone(iUserData.UserDataArray);
            userDataArray.RemoveUserData(index);
            userDataArray.InsertUserData((UserDataArray.UserData)item.Tag, index - 1);
            var commandSet = new EditCommandSet();
            commandSet.Add(CreateEditCommand_UserDataArray(Targets, userDataArray, Targets.Active.ObjectID));
            commandSet.OnPostEdit += new EventHandler(PostEdit);
            TheApp.CommandManager.Add(commandSet.Execute());
        }

        private void btnDown_Click(object sender, EventArgs e)
        {
            if (lvwUserData.SelectedItems.Count == 0) { return; }
            Debug.Assert(lvwUserData.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwUserData.SelectedItems[0];
            int index = lvwUserData.Items.IndexOf(item);
            if ((index + 1) == lvwUserData.Items.Count) { return; }

            lvwUserData.BeginUpdate();
            {
                lvwUserData.Items.RemoveAt(index);
                lvwUserData.Items.Insert(index + 1, item);
                lvwUserData.EndUpdate();
            }
            btnDown.Focus();

            // コマンド発行
            IHasUserData iUserData = Targets.Active as IHasUserData;
            if (iUserData == null)
            {
                return;
            }
            UserDataArray userDataArray = ObjectUtility.Clone(iUserData.UserDataArray);
            userDataArray.RemoveUserData(index);
            userDataArray.InsertUserData((UserDataArray.UserData)item.Tag, index + 1);
            var commandSet = new EditCommandSet();
            commandSet.Add(CreateEditCommand_UserDataArray(Targets, userDataArray, Targets.Active.ObjectID));
            commandSet.OnPostEdit += new EventHandler(PostEdit);
            TheApp.CommandManager.Add(commandSet.Execute());
        }

        private void btnAdd_Click(object sender, EventArgs e)
        {
            IHasUserData iUserData = Targets.Active as IHasUserData;
            if (iUserData == null)
            {
                return;
            }

            UserDataArray userDataArray = ObjectUtility.Clone(iUserData.UserDataArray);
            if (userDataArray == null)
            {
                userDataArray = new UserDataArray();
            }

            using (UserDataDialog dialog = new UserDataDialog(ApplicationConfig.Setting.PropertyEdit.UserDataPageHexMode, userDataArray))
            {
                DialogResult result = dialog.ShowDialog(Owner.Owner);

                if (result == DialogResult.OK)
                {
                    int index;
                    bool selected = false;
                    if (lvwUserData.SelectedIndices.Count == 0)
                    {
                        index = lvwUserData.Items.Count;
                    }
                    else
                    {
                        Debug.Assert(lvwUserData.SelectedIndices.Count == 1);
                        selected = true;
                        index = lvwUserData.SelectedIndices[0];
                    }

                    // カーソル位置の次の位置
                    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1627
                    ++ index;
                    index = Math.Min(index, lvwUserData.Items.Count);

                    // データの追加
                    ListViewItem item = CreateUserData(dialog, index, ref userDataArray);

                    // コマンド発行
                    var commandSet = new EditCommandSet();
                    commandSet.Add(CreateEditCommand_UserDataArray(Targets, userDataArray, Targets.Active.ObjectID));
                    commandSet.OnPostEdit += new EventHandler(PostEdit);
                    TheApp.CommandManager.Add(commandSet.Execute());

                    item.Selected = selected;
                }
            }
        }

        private void btnDelete_Click(object sender, EventArgs e)
        {
            if (lvwUserData.SelectedItems.Count == 0) { return; }
            Debug.Assert(lvwUserData.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwUserData.SelectedItems[0];
            int index = lvwUserData.Items.IndexOf(item);
            int targetIndex = index;
            lvwUserData.Items.Remove(item);
            if (lvwUserData.Items.Count != 0)
            {
                if (index == lvwUserData.Items.Count) { index--; }
                lvwUserData.Items[index].Selected = true;
                lvwUserData.Items[index].Focused = true;
            }
            btnDelete.Focus();

            // 削除
            IHasUserData iUserData = Targets.Active as IHasUserData;
            UserDataArray userDataList = ObjectUtility.Clone(iUserData.UserDataArray);
            userDataList.RemoveUserData(targetIndex);

            // コマンド発行
            var commandSet = new EditCommandSet();
            commandSet.Add(CreateEditCommand_UserDataArray(Targets, userDataList, Targets.Active.ObjectID));
            commandSet.OnPostEdit += new EventHandler(PostEdit);
            TheApp.CommandManager.Add(commandSet.Execute());
        }

        private void btnEdit_Click(object sender, EventArgs e)
        {
            Debug.Assert(lvwUserData.SelectedIndices.Count == 1);
            ListViewItem item = lvwUserData.SelectedItems[0];
            UserDataArray.UserData userData = (UserDataArray.UserData)item.Tag;

            int index = lvwUserData.SelectedIndices[0];

            IHasUserData iUserData = Targets.Active as IHasUserData;
            using (UserDataDialog dialog = new UserDataDialog(ApplicationConfig.Setting.PropertyEdit.UserDataPageHexMode, iUserData.UserDataArray, userData))
            {
                DialogResult result = dialog.ShowDialog(Owner.Owner);

                if (result == DialogResult.OK)
                {
                    // データの削除
                    UserDataArray userDataList = ObjectUtility.Clone(iUserData.UserDataArray);
                    userDataList.RemoveUserData(index);

                    // データの追加
                    lvwUserData.BeginUpdate();
                    {
                        lvwUserData.Items.RemoveAt(index);
                        ListViewItem newItem = CreateUserData(dialog, index, ref userDataList);

                        // コマンド発行
                        var commandSet = new EditCommandSet();
                        commandSet.Add(CreateEditCommand_UserDataArray(Targets, userDataList, Targets.Active.ObjectID));
                        commandSet.OnPostEdit += new EventHandler(PostEdit);
                        TheApp.CommandManager.Add(commandSet.Execute());

                        newItem.Selected = true;
                        newItem.Focused = true;
                    }
                    lvwUserData.EndUpdate();
                }
            }
        }

        private void lvwUserData_DoubleClick(object sender, EventArgs e)
        {
            if (lvwUserData.SelectedItems.Count > 0)
            {
                if (lvwUserData.SelectedItems.Count == 1)
                {
                    btnEdit_Click(null, null);
                }
            }
        }
#endregion

#region コマンド
        public static GroupEditCommand CreateEditCommand_UserDataArray(GuiObjectGroup targets, UserDataArray userDataArray, GuiObjectID id)
        {
            var editables = targets.GetObjects(id).Where(x => x.Editable).ToArray();
            var array = editables.OfType<IHasUserData>().Select(x =>
            {
                var cloned = ObjectUtility.Clone(userDataArray);
                bool modified = !x.SavedUserDataArray.IsSame(cloned);
                return new Tuple<UserDataArray, bool>(cloned, modified);
            }).ToArray();
            return
                new GeneralGroupReferenceEditCommand<Tuple<UserDataArray, bool>>(
                    new GuiObjectGroup(editables),
                    id,
                    array,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var obj = target as IHasUserData;

                        swap = new Tuple<UserDataArray, bool>(obj.UserDataArray, obj.UserDataArrayChanged);
                        var tuple = (Tuple<UserDataArray, bool>)data;
                        obj.UserDataArray = tuple.Item1;
                        obj.UserDataArrayChanged = tuple.Item2;
                    }
                );
        }
#endregion

#region コピー＆ペースト
        private class CopyData
        {
            public UserDataArray UserDataArray { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return (ActiveTarget != null) && ActiveTarget.Editable;
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return (ActiveTarget != null) && ActiveTarget.Editable;
        }


        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(GuiObject target)
        {
            IHasUserData iUserData = target as IHasUserData;
            if (iUserData == null)
            {
                // ユーザーデータがない
                Debug.Assert(false);
                return null;
            }

            return
                new CopyData()
                {
                    UserDataArray = (target as IHasUserData).UserDataArray,
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var editables = targets.Objects.Where(x => x.Editable);
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_UserDataArray(new GuiObjectGroup(editables), copyData.UserDataArray, targets.Active.ObjectID));
                commandSet.OnPostEdit += (ss, ee) =>
                {
                    PostEdit(new GuiObjectGroup(editables));
                };
            }

            return commandSet.Execute();
        }
#endregion
    }
}
