﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using App.res;
using System.IO;

namespace App.Controls
{
    public partial class SearchPathDialog : OkCancelDialog
    {
        private string initialPath;
        private bool initialRecursive;
        private string basePath;
        private List<string> paths;
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SearchPathDialog(string path, string modelPath, bool recursive, List<string> paths)
        {
            InitializeComponent();
            tbxLabel.Text = initialPath = path;
            cbxRecursive.Checked = initialRecursive = recursive;
            this.paths = paths;
            basePath = string.Empty;
            try
            {
                if (!string.IsNullOrEmpty(modelPath))
                {
                    basePath = Path.GetFullPath(Path.GetDirectoryName(modelPath));
                }
            }
            catch
            {
            }

            ltbModelPath.Text = string.IsNullOrEmpty(basePath) ? "": basePath + "\\";
            btnOutput.Enabled = !string.IsNullOrEmpty(basePath);
            UpdateRootedPath();
        }

        public string FolderPath
        {
            get
            {
                return tbxLabel.Text != null ? tbxLabel.Text.Replace('\\', '/'): null;
            }
            set
            {
                tbxLabel.Text = value;
            }
        }

        public bool Recursive
        {
            get
            {
                return cbxRecursive.Checked;
            }
        }

        public void UpdateRootedPath()
        {
            if (FolderPath == string.Empty)
            {
                ltbRootedPath.Text = "";
            }
            else
            {
                ltbRootedPath.Text = GetRootedPath();
            }
        }

        public string GetRootedPath()
        {
            string path = "";
            try
            {
                path = Environment.ExpandEnvironmentVariables(FolderPath);
                if (!System.IO.Path.IsPathRooted(path))
                {
                    if (!string.IsNullOrEmpty(basePath))
                    {
                        path = Path.GetFullPath(Path.Combine(basePath, path));
                    }
                    else
                    {
                        path = "";
                    }
                }
                else
                {
                    path = Path.GetFullPath(path);
                }
            }
            catch
            {
                path = "";
            }

            return path;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            if (string.IsNullOrEmpty(FolderPath))
            {
                UIMessageBox.Warning(Strings.ModelSearchPathDialog_Empty);
                return false;
            }
            else if (initialPath == FolderPath && initialRecursive == Recursive)
            {
                // データが変わっていなければキャンセル
                DialogResult = DialogResult.Cancel;
            }
            else if (initialPath != FolderPath && paths.Contains(FolderPath))
            {
                UIMessageBox.Warning(string.Format(Strings.ModelSearchPathDialog_Duplicate, FolderPath));
                return false;
            }
            else if (initialPath != FolderPath)
            {
                // パスがあるかどうか？
                var showWarning = false;
                // パスを置き換えるかどうか？
                var replacePath = false;
                var path = FolderPath;
                try
                {
                    if (FolderPath.Contains("%"))
                    {
                        path = Environment.ExpandEnvironmentVariables(path);
                    }
                    else if (Path.IsPathRooted(path))
                    {
                        path = PathUtility.MakeRelativePath(basePath + "/", path + "/");
                        replacePath = true;
                    }

                    if (Path.IsPathRooted(path))
                    {
                        if (!Directory.Exists(path))
                        {
                            showWarning = true;
                        }
                    }
                    else if (!string.IsNullOrEmpty(basePath))
                    {
                        if (!Directory.Exists(Path.Combine(basePath, path)))
                        {
                            showWarning = true;
                        }
                    }
                }
                catch
                {
                    UIMessageBox.Warning(string.Format(Strings.ModelSearchPathDialog_InvalidChar, FolderPath));
                    return false;
                }

                if (showWarning)
                {
                    if (!UIMessageBox.YesNo(Strings.ModelSearchPathDialog_PathNotExist))
                    {
                        return false;
                    }
                }
                if (path.EndsWith("/") || path.EndsWith("\\"))
                {
                    path = path.TrimEnd('/', '\\');
                    replacePath = true;
                }
                if (replacePath)
                {
                    FolderPath = path;
                }
            }

            return true;
        }



        private void btnOutput_Click(object sender, EventArgs e)
        {
            using (var dialog = new FolderBrowserDialog())
            {
                dialog.Description = Strings.IO_SaveDirectorySpecify;

                string initialPath = GetRootedPath();
                if (!string.IsNullOrEmpty(initialPath) && Directory.Exists(initialPath))
                {
                    dialog.SelectedPath = initialPath;
                }
                else if (!string.IsNullOrEmpty(basePath) && Directory.Exists(basePath))
                {
                    dialog.SelectedPath = basePath;
                }


                if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK)
                {
                    string path = dialog.SelectedPath;
                    try
                    {
                        if (Directory.Exists(basePath))
                        {
                            path = PathUtility.MakeRelativePath(basePath + "/", path + "/");
                            if (string.Empty == path)
                            {
                                path = ".";
                            }
                            else if (path.EndsWith("/") || path.EndsWith("\\"))
                            {
                                path = path.TrimEnd('/', '\\');
                            }
                        }
                    }
                    catch
                    {
                        path = dialog.SelectedPath;
                    }
                    FolderPath = path;
                }
            }
            UpdateRootedPath();
        }

        private void tbxLabel_TextChanged(object sender, EventArgs e)
        {
            UpdateRootedPath();
        }
    }
}
