﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;

using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.FileView;
using App.res;
using App.Utility;

using ConfigCommon;

using Viewer;

namespace App.PropertyEdit
{
    public partial class TexturePatternAnimationPatternPage : TexturePatternAnimationPropertyPage
    {
        // ３２サイズイメージリスト
        private static readonly ImageList Iml32Size = new ImageList()
        {
            ImageSize = new Size(32 + 2, 32 + 2),
            ColorDepth = ColorDepth.Depth8Bit,
            TransparentColor = Color.Magenta,
        };

        // ６４サイズイメージリスト
        private static readonly ImageList Iml64Size = new ImageList()
        {
            ImageSize = new Size(64 + 2, 64 + 2),
            ColorDepth = ColorDepth.Depth8Bit,
            TransparentColor = Color.Magenta,
        };

        private Size? InitialListViewPanelClientSize = null;
        private Rectangle? InitialListViewBounds = null;

        public TexturePatternAnimationPatternPage() :
            base(PropertyPageID.TexturePatternAnimationPattern)
        {
            InitializeComponent();

            // 行の高さを変更
            lvwList.SmallImageList = Iml32Size;
            lvwList.LargeImageList = Iml64Size;

/*
            lvwList.DragOver += (sender, args) => { };
            lvwList.DragDrop += (sender, args) => { };
            lvwList.AllowDrop = true;
*/
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_animation_property_window_pattern_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new TexturePatternAnimationPatternPage();
        }

        protected override void InitializeFormInternal()
        {
            if (!InitialListViewPanelClientSize.HasValue)
            {
                InitialListViewPanelClientSize = uiSplitContainer1.Panel1.ClientSize;
            }
            if (!InitialListViewBounds.HasValue)
            {
                InitialListViewBounds = lvwList.Bounds;
            }
        }

        private bool isFirstUpdate = true;

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            lblPattern.IsModified = ActiveTarget.IsPatternChanged();
            UpdateList();
            UpdateImage();
            UpdateState();

            // ページを開いた直後は最後を選択状態にする
            if (isFirstUpdate)
            {
                if (lvwList.Items.Count > 0)
                {
                    var lastIndex = lvwList.Items.Count - 1;

                    lvwList.Items[lastIndex].Selected = true;
                    lvwList.Items[lastIndex].Focused = true;
                }

                // ページを開いた直後は、リストビューの「名前」列の文字列が途中で切れないように調整する。
                var colWidth = lvwList.Items.OfType<ListViewItem>().Select(x => TextRenderer.MeasureText(x.SubItems[clhName.Index].Text, x.UseItemStyleForSubItems ? x.Font : x.SubItems[clhName.Index].Font).Width + 10).Concat(new int[] { 0 }).Max();
                if (lvwList.Columns[clhName.Index].Width < colWidth)
                {
                    lvwList.Columns[clhName.Index].Width = colWidth;
                }

                isFirstUpdate = false;
            }
        }

        public static bool IsModified(TexturePatternAnimation activeTarget)
        {
            return activeTarget != null && activeTarget.IsPatternChanged();
        }

        private void UpdateList()
        {
            var target = ActiveTarget;

            int selectIndex = -1;
            if (lvwList.SelectedIndices.Count > 0)
            {
                selectIndex = lvwList.SelectedIndices[0];
            }

            using (var ub = new UpdateBlock(lvwList))
            {
                lvwList.SetItemCount(target.TexPatterns.Count);

                int index = 0;
                foreach (var texPattern in target.TexPatterns)
                {
                    ListViewItem item = lvwList.Items[index];
                    item.Tag = texPattern.tex_name;

                    item.SubItems[clhIndex.Index].Text = index.ToString();
                    item.SubItems[clhName.Index].Text = texPattern.tex_name;
                    item.SubItems[clhImage.Index].Text = string.Empty;

                    ++index;
                }
            }

            if (lvwList.Items.Count > 0)
            {
                if ((selectIndex != -1) && (selectIndex < lvwList.Items.Count))
                {
                    lvwList.Items[selectIndex].Selected = true;
                    lvwList.Items[selectIndex].Focused = true;
                }
                else
                {
                    btnMultiAdd.Select();
                }
            }
            else
            {
                this.LvwList_SelectionChanged(null, null);
            }
        }

        private void UpdateImage()
        {
            var texName = lvwList.SelectedItemData as string;

            Texture texture = null;
            if (texName != null)
            {
                texture = ActiveTarget.GetReferenceTexture(texName);
            }

            tvpImage.Target = texture;

            if (texture != null)
            {
                ltbTexName.ForeColor = SystemColors.WindowText;

                string name = texture.Name;
                if (DocumentManager.Textures.Any(x => x.Name == name && x != texture))
                {
                    string indexText = DocumentManager.GetSameNameIndexText(texture, false);
                    if (indexText.Length > 0)
                    {
                        name += string.Format("  ({0} {1})", texture.FileName, indexText);
                    }
                    else
                    {
                        name += string.Format("  ({0})", texture.FileName);
                    }
                }

                ltbTexName.Text = name;
                ltbTexFullPath.Text = texture.FilePath;

                var preset = ApplicationConfig.Preset.TexturePresets.FirstOrDefault(x => x.quantize_type == texture.Data.texture_info.quantize_type);

                ltbHint.Text = texture.Data.texture_info.hint;

                ltbDccPreset.Text = texture.Data.texture_info.dcc_preset;
                ltbPresetFormat.Text = (preset == null) ? string.Empty : preset.Name;

                ltbFormat.Text = UIText.EnumValue(texture.Data.texture_info.quantize_type);
                ltbMipmapGenFilter.Text = TextureSelectPanel.mipmapGenFilters_.ContainsKey(texture.Data.texture_info.mip_gen_filter) ? TextureSelectPanel.mipmapGenFilters_[texture.Data.texture_info.mip_gen_filter] : string.Empty;
                ccsPanel.Value = texture.Data.texture_info.comp_sel;
                ltbWeightedCompress.Text = UIText.FlagEnableDisable(texture.Data.texture_info.weighted_compress);
                rgbaTable1.Values = texture.Data.texture_info.linear;

                ltbDimension.Text = UIText.EnumValue(texture.Data.texture_info.dimension);
                ltbWidth.Text = texture.Data.texture_info.width.ToString();
                ltbHeight.Text = texture.Data.texture_info.height.ToString();

                ltbDepth.Text = texture.Data.texture_info.depth.ToString();
                ltbVramSize.Text = DataSize.XBytesText(texture.Data.texture_info.size);
                ltbTexMipmapLevel.Text = texture.Data.texture_info.mip_level.ToString();
                ltbTexMipmapMinSize.Text = UIText.MipmapMinSize(texture.Data.texture_info);

                ltbInitialSwizzle.Text = texture.Data.texture_info.initial_swizzle.ToString();
                ltbOriginalImageHash.Text = texture.Data.texture_info.original_image_hash;

                btnPropertyDialog.Enabled = true;
            }
            else
            {
                ltbTexName.ForeColor = Const.DisreferenceColor;
                ltbTexName.Text = texName;
                ltbTexFullPath.Text = string.Empty;

                ltbHint.Text = string.Empty;

                ltbDccPreset.Text = string.Empty;
                ltbPresetFormat.Text = string.Empty;

                ltbFormat.Text = string.Empty;
                ltbMipmapGenFilter.Text = string.Empty;
                ccsPanel.Value = null;
                ltbWeightedCompress.Text = string.Empty;
                rgbaTable1.Values = null;

                ltbDimension.Text = string.Empty;
                ltbWidth.Text = string.Empty;
                ltbHeight.Text = string.Empty;

                ltbDepth.Text = string.Empty;
                ltbVramSize.Text = string.Empty;
                ltbTexMipmapLevel.Text = string.Empty;
                ltbTexMipmapMinSize.Text = string.Empty;

                ltbInitialSwizzle.Text = string.Empty;
                ltbOriginalImageHash.Text = string.Empty;

                btnPropertyDialog.Enabled = false;
            }

            {
                ltbHint.Enabled = texture != null;

                ltbDccPreset.Enabled = texture != null;
                ltbPresetFormat.Enabled = texture != null;

                ltbFormat.Enabled = texture != null;
                ltbMipmapGenFilter.Enabled = texture != null;
                ccsPanel.Enabled = texture != null;
                ltbWeightedCompress.Enabled = texture != null;
                rgbaTable1.Enabled = texture != null;

                ltbDimension.Enabled = texture != null;
                ltbWidth.Enabled = texture != null;
                ltbHeight.Enabled = texture != null;

                ltbDepth.Enabled = texture != null;
                ltbVramSize.Enabled = texture != null;
                ltbTexMipmapLevel.Enabled = texture != null;
                ltbTexMipmapMinSize.Enabled = texture != null;

                ltbInitialSwizzle.Enabled = texture != null;
                ltbOriginalImageHash.Enabled = texture != null;
            }
        }

        private void UpdateState()
        {
            bool isSelected = lvwList.SelectedIndex != -1;

            grpImage.Enabled    = isSelected;
            btnMoveUp.Enabled   = isSelected && (lvwList.SelectedIndex > 0);
            btnMoveDown.Enabled = isSelected && (lvwList.SelectedIndex >= 0) && (lvwList.SelectedIndex <= lvwList.Items.Count - 2);
            btnDelete.Enabled   = isSelected;
        }

        private void LvwList_CustomDrawItem(object sender, CustomDrawListViewItemEventArgs e)
        {
            if (e.Item != null)
            {
                var texName = e.Item.Tag as string;
                var texture = ActiveTarget.GetReferenceTexture(texName);

                if (e.ColumnIndex == 0)
                {
                    if (texture != null)
                    {
                        GraphicsUtility.DrawTextureImage(texture, e.Graphics, e.Bounds);
                    }

                    e.Handled = true;
                }

                // 文字色
                {
                    e.SpecificForeColor = (texture == null) ? Const.DisreferenceColor : SystemColors.WindowText;
                }
            }
        }

        private void BtnDelete_Click(object sender, EventArgs e)
        {
            if (Manager.Instance.IsConnected)
            {
                if (ActiveTarget.AllKeys().Any(x => x.Value >= ActiveTarget.TexPatterns.Count - 1))
                {
                    if (UIMessageBox.OkCancel(Strings.TexturePatternAnimation_OutOfRange_Delete))
                    {
                        Manager.Instance.Close();
                    }
                    else
                    {
                        return;
                    }
                }
            }

            var target = ActiveTarget;
            var texPatterns = ObjectUtility.Clone(target.TexPatterns);
            texPatterns.RemoveAt(lvwList.SelectedIndices[0]);

            int index = Math.Min(lvwList.SelectedIndices[0], texPatterns.Count - 1);

            var commandSet = new EditCommandSet();
            commandSet.Add(CreateEditCommand_TexPatterns(Targets, texPatterns, true));
            commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(Targets)));
            TheApp.CommandManager.Execute(commandSet);

            if ((index >= 0) && (index <= lvwList.Items.Count - 1))
            {
                lvwList.Items[index].Selected = true;
                lvwList.Items[index].Focused = true;
            }
        }

        private void BtnMoveUp_Click(object sender, EventArgs e)
        {
            int selectedIndex = lvwList.SelectedIndices[0];

            if (selectedIndex >= 1)
            {
                var target = ActiveTarget;
                var texPatterns = ObjectUtility.Clone(target.TexPatterns);

                var dst = texPatterns[selectedIndex - 1];
                var src = texPatterns[selectedIndex];

                texPatterns[selectedIndex - 1] = src;
                texPatterns[selectedIndex] = dst;

                using (var sb = new App.AppContext.PropertyChangedSuppressBlock())
                {
                    TheApp.CommandManager.Execute(CreateEditCommand_TexPatterns(Targets, texPatterns, true));
                }

                lvwList.Items[selectedIndex - 1].Selected = true;
                lvwList.Items[selectedIndex - 1].Focused = true;

                App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
            }
        }

        private void BtnMoveDown_Click(object sender, EventArgs e)
        {
            int selectedIndex = lvwList.SelectedIndices[0];

            if (selectedIndex <= (lvwList.Items.Count - 2))
            {
                var target = ActiveTarget;
                var texPatterns = ObjectUtility.Clone(target.TexPatterns);

                var dst = texPatterns[selectedIndex + 1];
                var src = texPatterns[selectedIndex];

                texPatterns[selectedIndex + 1] = src;
                texPatterns[selectedIndex] = dst;

                using (var sb = new App.AppContext.PropertyChangedSuppressBlock())
                {
                    TheApp.CommandManager.Execute(CreateEditCommand_TexPatterns(Targets, texPatterns, true));
                }

                lvwList.Items[selectedIndex + 1].Selected = true;
                lvwList.Items[selectedIndex + 1].Focused = true;

                App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(target, null));
            }
        }

        private void LvwList_SelectionChanged(object sender, EventArgs e)
        {
            UpdateState();
            UpdateImage();
        }

        private void BtnChange_Click(object sender, EventArgs e)
        {
            var texName = lvwList.SelectedItemData as string;

            Texture current = null;
            if (texName != null)
            {
                current = ActiveTarget.GetReferenceTexture(texName);
            }
            var remains = ActiveTarget.TexPatterns.Where((x, y) => y != lvwList.SelectedIndices[0])
                .Select(x => x.tex_name).Where(x => !string.IsNullOrEmpty(x)).Distinct()
                .ToDictionary(x => x, x => { string path; ActiveTarget.ReferenceTexturePaths.TryGetValue(x, out path); return path; });
            Predicate<Texture> canAssign = x =>
            {
                string path;
                if (remains.TryGetValue(x.Name, out path))
                {
                    if (string.Compare(path, x.FilePath, StringComparison.OrdinalIgnoreCase) != 0)
                    {
                        return false;
                    }
                }

                return true;
            };

            using (var dialog = new TextureSelectDialog(current, canAssign))
            {
                if (dialog.ShowDialog(Owner.Owner) == DialogResult.OK)
                {
                    var newTex = dialog.SelectedTex;
                    if (newTex == current)
                    {
                        return;
                    }

                    var target = ActiveTarget;
                    var texPatterns = ObjectUtility.Clone(target.TexPatterns);

                    texPatterns.RemoveAt(lvwList.SelectedIndices[0]);

                    texPatterns.Insert(
                        lvwList.SelectedIndices[0],
                        new TexturePatternAnimation.TexPattern()
                        {
                            tex_name = newTex.Name,
                        });

                    var commandSet = new EditCommandSet();
                    commandSet.Add(CreateEditCommand_TexPatterns(Targets, texPatterns, false));
                    commandSet.Add(CreateEditCommand_ReferenceTexturePath(Targets, newTex.Name, newTex.FilePath));
                    commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(Targets)));
                    var anims = Targets.GetObjects(GuiObjectID.TexturePatternAnimation).OfType<TexturePatternAnimation>().ToArray();
                    commandSet.OnPostEdit += (s, a) => {
                        foreach (var anim in anims)
                        {
                            // 不正アニメーションチェック
                            anim.CheckAndDisConnect();
                            LoadOrReloadAnimation.Send(anim);
                        }
                    };
                    TheApp.CommandManager.Execute(commandSet);
                }
            }
        }

        private void LvwList_DoubleClick(object sender, EventArgs e)
        {
            this.BtnChange_Click(sender, e);
        }

        #region コマンド
        public static GroupEditCommand CreateEditCommand_TexPatterns(GuiObjectGroup targets, List<TexturePatternAnimation.TexPattern> texPatterns, bool sendViewer)
        {
            return
                new GeneralGroupReferenceEditCommand<List<TexturePatternAnimation.TexPattern>>(
                    targets,
                    GuiObjectID.TexturePatternAnimation,
                    ObjectUtility.MultipleClone(texPatterns, targets.GetObjects(GuiObjectID.TexturePatternAnimation).Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as TexturePatternAnimation;

                        swap = ObjectUtility.Clone(anim.TexPatterns);
                        anim.TexPatterns = data as List<TexturePatternAnimation.TexPattern>;

                        // 不正アニメーションチェック
                        if (sendViewer)
                        {
                            anim.CheckAndDisConnect();
                            LoadOrReloadAnimation.Send(anim);
                        }
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        // リロードはまとめて送れないのでEditDelegateで送る
                    });
        }

        public static GroupEditCommand CreateEditCommand_ReferenceTexturePath(GuiObjectGroup targets, string texName, string filePath)
        {
            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                GuiObjectID.TexturePatternAnimation,
                targets.GetObjects(GuiObjectID.TexturePatternAnimation).Select(x => filePath),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var anim = (TexturePatternAnimation)target;
                    string old;
                    anim.ReferenceTexturePaths.TryGetValue(texName, out old);
                    swap = old;
                    string path = (string)data;
                    if (path == null)
                    {
                        anim.ReferenceTexturePaths.Remove(texName);
                    }
                    else
                    {
                        anim.ReferenceTexturePaths[texName] = path;
                    }
                });
        }

        public static GroupEditCommand CreateEditCommand_RemoveTexturePaths(GuiObjectGroup targets)
        {
            return new GeneralGroupReferenceEditCommand<Dictionary<string, string>>(
                targets,
                GuiObjectID.TexturePatternAnimation,
                targets.GetObjects(GuiObjectID.TexturePatternAnimation).OfType<TexturePatternAnimation>().Select(x =>
                {
                    var texNames = (from pattern in x.TexPatterns
                                    where !string.IsNullOrEmpty(pattern.tex_name)
                                    select pattern.tex_name).Distinct();
                    return texNames.Where(y => x.ReferenceTexturePaths.ContainsKey(y)).ToDictionary(y => y, y => x.ReferenceTexturePaths[y]);
                }),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var anim = (TexturePatternAnimation)target;
                    swap = anim.ReferenceTexturePaths;
                    anim.ReferenceTexturePaths = (Dictionary<string, string>)data;
                });
        }
        #endregion

        #region コピー＆ペースト
        public class CopyData
        {
            public List<TexturePatternAnimation.TexPattern> TexPatterns { get; set; }
            public Dictionary<string, string> TexturePaths { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(TexturePatternAnimation target)
        {
            var paths = target.TexPatterns.Select(x => x.tex_name).Where(x => !string.IsNullOrEmpty(x)).Distinct()
                .ToDictionary(x => x, x => { string path; target.ReferenceTexturePaths.TryGetValue(x, out path); return path; });
            return
                new CopyData()
                {
                    TexPatterns = ObjectUtility.Clone(target.TexPatterns),
                    TexturePaths = paths
                };
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            if (!CanPaste((CopyData)copiedObject))
            {
                return false;
            }

            return base.CanPaste(copiedObjectInfo, copiedObject);
        }

        public static bool CanPaste(CopyData data)
        {
            foreach (var keyValue in data.TexturePaths)
            {
                if (!DocumentManager.Textures.Any(
                    x => x.Name == keyValue.Key && string.Compare(x.FilePath, keyValue.Value, StringComparison.OrdinalIgnoreCase) == 0))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            if (Manager.Instance.IsConnected &&
                ActiveTarget.AllKeys()
                .Any(x => x.Value < 0 || ((CopyData)pasteObject).TexPatterns.Count <= x.Value))
            {
                if (UIMessageBox.OkCancel(Strings.TexturePatternAnimation_OutOfRange_Paste))
                {
                    Manager.Instance.Close();
                }
                else
                {
                    return;
                }
            }

            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_TexPatterns(targets, copyData.TexPatterns, false));
                foreach (var keyValue in copyData.TexturePaths)
                {
                    commandSet.Add(CreateEditCommand_ReferenceTexturePath(targets, keyValue.Key, keyValue.Value));
                }
                commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(targets)));
                var anims = targets.GetObjects(GuiObjectID.TexturePatternAnimation).OfType<TexturePatternAnimation>().ToArray();
                commandSet.OnPostEdit += (s, e) =>
                {
                    foreach (var anim in anims)
                    {
                        // 不正アニメーションチェック
                        anim.CheckAndDisConnect();
                        LoadOrReloadAnimation.Send(anim);
                    }
                };
            }

            return commandSet.Execute();
        }
        #endregion

        private void BtnPropertyDialog_Click(object sender, EventArgs e)
        {
            var texname = lvwList.SelectedItemData as string;
            if (texname != null)
            {
                var texture = ActiveTarget.GetReferenceTexture(texname);
                if (texture != null)
                {
                    using (var wait = new WaitCursor())
                    {
                        // プロパティウィンドウを生成
                        var objectGroup = new GuiObjectGroup(texture);

                        ObjectPropertyDialog dialog = new ObjectPropertyDialog(null, objectGroup);
                        dialog.TargetFixed = true;

                        // 現在のウィンドウの上に表示する
                        Control parent = Owner;
                        while (parent != null)
                        {
                            if (parent is ObjectPropertyDialog)
                            {
                                dialog.SetLocationNextTo((ObjectPropertyDialog)parent);
                                break;
                            }

                            parent = parent.Parent;
                        }

                        dialog.Show();
                    }
                }
            }
        }

        private void BtnMultiAdd_Click(object sender, EventArgs e)
        {
            var texPatternAnim = this.ActiveTarget;
            Debug.Assert(texPatternAnim != null, "texPatternAnim != null");

            using (var dialog = new AddMultiTexureToTexturePatternDialog(()=>texPatternAnim.TexPatterns))
            {
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    var selectedTextures = dialog.SelectedTextures;
                    if (selectedTextures == null || !selectedTextures.Any())
                    {
                        return;
                    }
                    var commandSet = new EditCommandSet();

                    this.AddTextures(commandSet, selectedTextures, (this.lvwList.SelectedItemData != null) ? this.lvwList.SelectedIndices[0] + 1 : texPatternAnim.TexPatterns.Count);

                    TheApp.CommandManager.Execute(commandSet);
                }
            }
        }

        private void AddTextures(EditCommandSet commandSet, IEnumerable<Texture> selectedTextures, int index)
        {
            var animation = this.ActiveTarget;
            AddTextureToTexturePattern(commandSet, selectedTextures, animation, index);
        }

        public static void AddTextureToTexturePattern(
            EditCommandSet commandSet,
            IEnumerable<Texture> selectedTextures,
            TexturePatternAnimation animation,
            int insertIndex = -1)
        {
            var texPatterns = ObjectUtility.Clone(animation.TexPatterns);
            insertIndex = insertIndex < 0 ? texPatterns.Count : Math.Min(insertIndex, texPatterns.Count);
            var groups = new GuiObjectGroup(animation);

            foreach (var tex in selectedTextures)
            {
                texPatterns.Insert(insertIndex++, new TexturePatternAnimation.TexPattern() { tex_name = tex.Name });
                commandSet.Add(CreateEditCommand_ReferenceTexturePath(groups, tex.Name, tex.FilePath));
            }
            commandSet.Add(CreateEditCommand_TexPatterns(groups, texPatterns, false));

            // 複数編集がないので　CreateEditCommand_RemoveTexturePaths は不要
            commandSet.OnPostEdit += (s, a) =>
                {
                    // 不正アニメーションチェック
                    animation.CheckAndDisConnect();
                    LoadOrReloadAnimation.Send(animation);
                };
        }

        private void LvwList_DragOver(object sender, DragEventArgs e)
        {
            var pos = this.lvwList.PointToClient(new Point(e.X, e.Y));
            var targetItem = this.lvwList.GetItemAt(pos.X, pos.Y);

            CheckDropState(e, targetItem);
        }

        private void CheckDropState(DragEventArgs e, ListViewItem targetItem)
        {
            e.Effect = DragDropEffects.None;
            // ファイルであり、モーダルでないかファイルダイアログが開いている時はドラッグＯＫ
            var patterns = ActiveTarget.TexPatterns.Select(x => x.tex_name.ToLower()).ToArray();
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                if (UIForm.IsModalState(TheApp.MainFrame) && !DialogUtility.IsOpened)
                {
                    return;
                }

                var filepaths = (string[])e.Data.GetData(DataFormats.FileDrop);
                if (filepaths == null)
                {
                    return;
                }

                // テクスチャファイル以外がドラッグされたときはドロップ不可
                if (filepaths.Select(Path.GetExtension)
                    .Any(x => ObjectIDUtility.ExtToId(x) != GuiObjectID.Texture && ObjectIDUtility.IsTgaExtension(x) == false))
                {
                    return;
                }

                // 同じファイル名がドラッグされたときはドロップ不可
                var filenames = filepaths.Select(s => Path.GetFileNameWithoutExtension(s).ToLower()).ToArray();
                if (!filenames.Any() || filenames.Distinct().Count() != filepaths.Count())
                {
                    return;
                }

                // 既に開かれているテクスチャと同名がドラッグされたときはドロップ不可
                // ただしドロップ先とファイルが同名の時だけ許可

                if (filenames.Intersect(patterns).Any())
                {
                    if (targetItem == null || !(targetItem.Tag is string) || filenames.Count() != 1
                        || string.Compare(filenames[0], (string)targetItem.Tag, StringComparison.OrdinalIgnoreCase) != 0)
                    {
                        return;
                    }

                    // ドロップ先と同一パスの場合はドロップ不可
                    var currentTex = this.ActiveTarget.GetReferenceTexture((string)targetItem.Tag);
                    if (currentTex == null
                        || string.Compare(filepaths[0], currentTex.FilePath, StringComparison.OrdinalIgnoreCase) == 0)
                    {
                        return;
                    }

                    // ドロップ先のテクスチャがパターン内で複数使われている場合はドロップ不可
                    if (patterns.Count(x => string.Compare(x, currentTex.Name, StringComparison.OrdinalIgnoreCase) == 0) > 1)
                    {
                        return;
                    }
                }

                // 既存テクスチャ上にファイルがひとつドラッグされたときは、上書き(移動)。そうでない場合はコピー（追加）
                e.Effect = (targetItem != null && ModifierKeys != Keys.Control && filepaths.Count() == 1) ? DragDropEffects.Move : DragDropEffects.Copy;
                return;
            }
            else if (e.Data.GetDataPresent(typeof(FileTreeView.ViewItem)))
            {
                var source = (FileTreeView.ViewItem)e.Data.GetData(typeof(FileTreeView.ViewItem));
                if (source == null)
                {
                    return;
                }

                var texture = source.Tag as Texture;
                if (texture != null)
                {
                    // 既に開かれているテクスチャと同名がドラッグされたときはドロップ不可
                    // ただしドロップ先と同名の時だけ許可
                    if (patterns.Any(x => string.Compare(x, texture.Name, StringComparison.OrdinalIgnoreCase) == 0))
                    {
                        if (targetItem == null ||
                            !(targetItem.Tag is string) ||
                            string.Compare(texture.Name, (string)targetItem.Tag, StringComparison.OrdinalIgnoreCase) != 0)
                        {
                            return;
                        }
                        // ドロップ先と同一テクスチャの場合はドロップ不可
                        var currentTex = this.ActiveTarget.GetReferenceTexture((string)targetItem.Tag);
                        if (currentTex == texture)
                        {
                            return;
                        }
                        // ドロップ先のテクスチャがパターン内で複数使われている場合はドロップ不可
                        if (patterns.Count(x => string.Compare(x, currentTex.Name, StringComparison.OrdinalIgnoreCase) == 0) > 1)
                        {
                            return;
                        }
                    }
                    // 既存テクスチャ上にドラッグされたときは、上書き(移動)。そうでない場合はコピー（追加）
                    e.Effect = (targetItem != null && ModifierKeys != Keys.Control) ? DragDropEffects.Move: DragDropEffects.Copy;
                    return;
                }
            }
        }

        private void LvwList_DragDrop(object sender, DragEventArgs e)
        {
            var pos = this.lvwList.PointToClient(new Point(e.X, e.Y));
            var targetItem = this.lvwList.GetItemAt(pos.X, pos.Y);
            CheckDropState(e, targetItem);

            if (e.Effect == DragDropEffects.None)
            {
                return;
            }

            var commandSets = new EditCommandSet();
            var textures = DroppedTextures(e, commandSets);
            if (textures == null)
            {
                return;
            }

            var target = this.ActiveTarget;

            if (e.Effect != DragDropEffects.Copy && textures.Count() == 1 && targetItem != null && targetItem.Tag is string)
            {
                // パターンを上書き
                var oldName = (string)targetItem.Tag;
                var newTex = textures.First();
                var texPatterns = ObjectUtility.Clone(target.TexPatterns);

                try
                {
                    var index = texPatterns.FindIndex(x => string.Compare(x.tex_name, oldName, StringComparison.OrdinalIgnoreCase) == 0);
                    texPatterns.RemoveAt(index);

                    texPatterns.Insert(
                        index,
                        new TexturePatternAnimation.TexPattern()
                        {
                            tex_name = newTex.Name,
                        });

                    var replaceCommandSets = new EditCommandSet();
                    replaceCommandSets.Add(CreateEditCommand_TexPatterns(this.Targets, texPatterns, false));
                    replaceCommandSets.Add(CreateEditCommand_ReferenceTexturePath(this.Targets, newTex.Name, newTex.FilePath));
                    replaceCommandSets.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(this.Targets)));
                    var anims = this.Targets.GetObjects(GuiObjectID.TexturePatternAnimation).OfType<TexturePatternAnimation>().ToArray();
                    replaceCommandSets.OnPostEdit += (s, a) =>
                    {
                        foreach (var anim in anims)
                        {
                            // 不正アニメーションチェック
                            anim.CheckAndDisConnect();
                            LoadOrReloadAnimation.Send(anim);
                        }
                    };
                    commandSets.Add(replaceCommandSets.Execute());
                }
                catch (Exception)
                {
                    return;
                }
            }
            else
            {
                // 追加
                try
                {
                    int index;
                    if (targetItem != null && targetItem.Tag is string)
                    {
                        var oldName = (string)targetItem.Tag;
                        index = target.TexPatterns.FindIndex(x => string.Compare(x.tex_name, oldName, StringComparison.OrdinalIgnoreCase) == 0) + 1;
                    }
                    else
                    {
                        index = target.TexPatterns.Count;
                    }

                    var addTexCommandSets = new EditCommandSet();
                    this.AddTextures(addTexCommandSets, textures, index);
                    commandSets.Add(addTexCommandSets.Execute());
                }
                catch (Exception)
                {
                    return;
                }
            }
            commandSets.Reverse();
            TheApp.CommandManager.Add(commandSets);
        }

        public static Texture[] DroppedTextures(DragEventArgs e, EditCommandSet commandSets)
        {
            Texture[] textures;
            var files = (string[])e.Data.GetData(DataFormats.FileDrop);
            if (files != null) // ファイルをドロップ
            {
                if (
                    files.Select(Path.GetExtension)
                        .Any(x => ObjectIDUtility.ExtToId(x) != GuiObjectID.Texture && ObjectIDUtility.IsTgaExtension(x) == false))
                {
                    return null;
                }

                // 既にファイルが開かれているか確認
                var preOpenFiles =
                    files.Where(
                        file =>
                        DocumentManager.Textures.Any(
                            tex => file.Equals(tex.FilePath, StringComparison.OrdinalIgnoreCase))).ToArray();

                if (preOpenFiles.Any())
                {
                    using (var dialog = new OkCancelTextBoxDialog())
                    {
                        dialog.lblDescription.Text = Strings.TexturePatternAnimationPatternPage_DropedTextures_WarnSameFile;
                        foreach (var file in preOpenFiles)
                        {
                            dialog.AddLine(file);
                        }

                        var result = dialog.ShowDialog();
                        if (result == DialogResult.Cancel)
                        {
                            return null;
                        }
                    }
                }

                // ファイルを開く
                DocumentManager.LoadFromFile(
                    files.Except(preOpenFiles).Select(x => new DocumentManager.PathWithName(x, null)),
                    commandSet: commandSets,
                    isUpdateRecentlyUsedFiles: true,
                    isShowErrorDialog: false);

                textures = files.Select(x => DocumentManager.Textures.FirstOrDefault(y => string.Compare(x, y.FilePath, StringComparison.OrdinalIgnoreCase) == 0))
                    .Where(x => x != null).ToArray();
                if (textures.Length == 0)
                {
                    return null;
                }
            }
            else // ツリービューからテクスチャをドロップ
            {
                var source = (FileTreeView.ViewItem)e.Data.GetData(typeof(FileTreeView.ViewItem));
                if (source == null)
                {
                    return null;
                }

                var data = source.Tag as Texture;
                if (data == null)
                {
                    return null;
                }

                textures = new[] { data };
            }
            return textures;
        }

        private void uiSplitContainer1_SplitterMoved(object sender, SplitterEventArgs e)
        {
            // lvwList サイズを調整する。
            // uiSplitContainer1.Panel1 のサイズ変更関連イベント内で lvwList サイズ変更すると
            // uiSplitContainer1.Panel1.AutoSize が正常に作用しないのでここで調整している。
            if (InitialListViewPanelClientSize.HasValue && InitialListViewBounds.HasValue)
            {
                var lvwListSize = lvwList.Size;

                // lvwList サイズを調整する。
                if (InitialListViewPanelClientSize.Value.Width < uiSplitContainer1.Panel1.ClientSize.Width)
                {
                    lvwListSize.Width = InitialListViewBounds.Value.Width + (uiSplitContainer1.Panel1.ClientSize.Width - InitialListViewPanelClientSize.Value.Width);
                }
                else
                {
                    lvwListSize.Width = InitialListViewBounds.Value.Width;
                }
                if (InitialListViewPanelClientSize.Value.Height < uiSplitContainer1.Panel1.ClientSize.Height)
                {
                    lvwListSize.Height = InitialListViewBounds.Value.Height + (uiSplitContainer1.Panel1.ClientSize.Height - InitialListViewPanelClientSize.Value.Height);
                }
                else
                {
                    lvwListSize.Height = InitialListViewBounds.Value.Height;
                }

                // uiSplitContainer1.Panel1 でのスクロールバー表示などによって
                // lvwList の端が隠れないように調整。
                // lvwList のスクロールバーに uiSplitContainer1.Panel1 のスクロールバーが重なり、
                // lvwList をスクロールできなくなることを防ぐ。
                if (uiSplitContainer1.Panel1.Size.Width > uiSplitContainer1.Panel1.ClientSize.Width)
                {
                    lvwListSize.Width -= uiSplitContainer1.Panel1.Size.Width - uiSplitContainer1.Panel1.ClientSize.Width;
                }
                if (uiSplitContainer1.Panel1.Size.Height > uiSplitContainer1.Panel1.ClientSize.Height)
                {
                    lvwListSize.Height -= uiSplitContainer1.Panel1.Size.Height - uiSplitContainer1.Panel1.ClientSize.Height;
                }

                // 調整値でリサイズ。
                lvwList.Size = lvwListSize;
            }
        }
    }
}
