﻿using System;
using App.Command;
using App.Data;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class TexturePatternAnimationGeneralPage : TexturePatternAnimationPropertyPage
    {
        public TexturePatternAnimationGeneralPage() :
            base(PropertyPageID.TexturePatternAnimationGeneral)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_animation_property_window_general_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new TexturePatternAnimationGeneralPage();
        }

        protected override void InitializeFormInternal()
        {
            iepFrameSize.SequentialValueChanged +=
                (s, e) =>
                {
                    if (!e.Changing)
                    {
                        TheApp.CommandManager.Execute(CreateEditCommand_frame_count(Targets, iepFrameSize.Value));
                    }
                };
            cbxLoop.CheckedChanged +=
                (s, e) => TheApp.CommandManager.Execute(CreateEditCommand_loop(Targets, cbxLoop.Checked));
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            TexturePatternAnimation target = ActiveTarget;

            var info = target.Data.tex_pattern_anim_info;
            lblFrameSize.IsModified = target.IsValueChanged(x => x.tex_pattern_anim_info.frame_count);
            iepFrameSize.Value = info.frame_count;

            cbxLoop.IsModified = target.IsValueChanged(x => x.tex_pattern_anim_info.loop);
            cbxLoop.Checked = info.loop;

            ltbFrameResolution.Text	= target.Data.tex_pattern_anim_info.frame_resolution.ToString();
            ltbBakeAll.Text			= UIText.FlagYesNo(target.Data.tex_pattern_anim_info.bake_all);

            ltbDccPreset.Text = ActiveTarget.Data.tex_pattern_anim_info.dcc_preset;
            ltbDccStartFrame.Text = ActiveTarget.Data.tex_pattern_anim_info.dcc_start_frame.ToString();
            ltbDccEndFrame.Text = ActiveTarget.Data.tex_pattern_anim_info.dcc_end_frame.ToString();
            ltbDccFps.Text = ActiveTarget.Data.tex_pattern_anim_info.dcc_fps.ToString();
        }


        public static bool IsModified(TexturePatternAnimation activeTarget)
        {
            return activeTarget != null &&
                (activeTarget.IsValueChanged(x => x.tex_pattern_anim_info.frame_count) ||
                activeTarget.IsValueChanged(x => x.tex_pattern_anim_info.loop));
        }

        #region コマンド
        public static GroupEditCommand CreateEditCommand_frame_count(GuiObjectGroup targets, int FrameCount)
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    targets,
                    GuiObjectID.TexturePatternAnimation,
                    FrameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        TexturePatternAnimation anim = target as TexturePatternAnimation;

                        swap = anim.Data.tex_pattern_anim_info.frame_count;
                        anim.Data.tex_pattern_anim_info.frame_count = (int)data;
                        AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);

                        // 不正アニメーションチェック
                        anim.CheckAndDisConnect();
                        // Viewerへ再転送
                        Viewer.LoadOrReloadAnimation.Send(anim);
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                    }
                );
        }

        public static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.TexturePatternAnimation,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        TexturePatternAnimation anim = target as TexturePatternAnimation;

                        swap = anim.Data.tex_pattern_anim_info.loop;
                        anim.Data.tex_pattern_anim_info.loop = (bool)data;

                        // 不正アニメーションチェック
                        anim.CheckAndDisConnect();
                        // Viewerへ再転送
                        Viewer.LoadOrReloadAnimation.Send(anim);
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                    }
                );
        }
        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            public int												frame_count { get; set; }
            public bool												loop { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }


        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(TexturePatternAnimation target)
        {
            return
                new CopyData()
                {
                    frame_count					= target.Data.tex_pattern_anim_info.frame_count,
                    loop						= target.Data.tex_pattern_anim_info.loop,
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_frame_count(targets, copyData.frame_count));
                commandSet.Add(CreateEditCommand_loop(targets, copyData.loop));
            }

            return commandSet.Execute();
        }
        #endregion
    }
}
