﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Command;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class TexturePatternAnimationCurveEditPage : TexturePatternAnimationPropertyPage, IAnimationEditPage
    {
        public TexturePatternAnimationCurveEditPage() :
            base(PropertyPageID.TexturePatternAnimationCurveEdit)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_curve_editor;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new TexturePatternAnimationCurveEditPage();
        }

        protected override void InitializeFormInternal()
        {
            AnimationObjectPropertyPanel animationObjectPropertyPanel = Owner as AnimationObjectPropertyPanel;
            Debug.Assert(animationObjectPropertyPanel != null);

            CurveEditorPanel.Initialize(animationObjectPropertyPanel, ActiveTarget.ObjectID);

            CurveEditorPanel.UpdateTreeView       += (s, e) => UpdateTreeView();
            CurveEditorPanel.ChangeSelectedCurves += (s, e) => UpdateSelected();
            CurveEditorPanel.ChangeVisibledCurves += (s, e) => UpdateVisibled();
            CurveEditorPanel.GetFrameCount        = () => ActiveTarget.Data.tex_pattern_anim_info.frame_count;
            CurveEditorPanel.SetInterpolationTypeButtons(false, true, false);
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width),
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height));
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && !ObjectPropertyDialog.InternallyChangingSize.IsChanging &&
                Owner.Owner.WindowState == System.Windows.Forms.FormWindowState.Normal)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height = Height;
            }

            base.OnSizeChanged(e);
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            CurveEditorPanel.TargetGroup = Targets;

            CurveEditorPanel.UpdateForm(updateFormInfo);
        }

        protected override void UpdateFormOnPageCreatedInternal()
        {
            CurveEditorPanel.UpdateFormOnPageCreatedInternal();
        }

        public override void BeforePageDeactivated()
        {
            CurveEditorPanel.BeforePageDeactivated();
        }

        public override void AfterPageActiveted(ObjectPropertyPage oldPage)
        {
            CurveEditorPanel.AfterPageActivated();
        }

        private void UpdateSelected()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private void UpdateVisibled()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private IEnumerable<Model> ParentModels
        {
            get
            {
                return
                    from model in DocumentManager.Models
                    from anim in model.AllAnimations
                    where anim == ActiveTarget
                    select model;
            }
        }

        private Dictionary<string, bool>				visibledNodes_		= new Dictionary<string, bool>();
        private Dictionary<string, bool>	isExpandedNodes_	= new Dictionary<string, bool>();

        private void UpdateTreeView()
        {
            using (var sb = new UIControlEventSuppressBlock())
            {
                // ノードの状態を保存しておく
                if (CurveEditorPanel.CurveTreeView.Nodes.Count > 0)
                {
                    var root = (CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0];
                    foreach (var tuple in CurveTreeView.NodesAndFullPath(root, root.id))
                    {
                        // チェック状態
                        visibledNodes_[tuple.Item2] = tuple.Item1.Checked;

                        // 開閉状態
                        isExpandedNodes_[tuple.Item2] = tuple.Item1.Nodes.Count == 0 || tuple.Item1.IsExpanded;
                    }
                }
                IEnumerable<Model> parentModels = ParentModels;

                bool isAllShowNode = (Owner as AnimationObjectPropertyPanel).IsAllShowNode;

                // つくり直す
                {
                    var rootNode = new CurveTreeInfo()
                    {
                        Text = ActiveTarget.Name,
                        Id = ActiveTarget.Name,
                        ShowEmptyNode = true,
                    };

                    foreach (var texPatternMatAnims in ActiveTarget.TexPatternMatAnims)
                    {
                        var materialName = texPatternMatAnims.mat_name;
                        var materials = (from model in parentModels
                                         from material in model.Materials
                                         where material.Name == materialName
                                         select material).ToArray();
                        var materialNode = new CurveTreeInfo()
                        {
                            Text = materialName,
                            Id = materialName,
                            ShowEmptyNode = true,
                            Category = CurveTreeNodeCategory.TexPatAnimMaterial,
                        };

                        {
                            var addedKey = new Dictionary<string, bool>();

                            // 親のモデルから構築する
                            var query = new List<string>();
                            foreach (var sampler in materials.SelectMany(x => x.ResolvedSamplers))
                            {
                                // データで追加していないものを追加する
                                if (addedKey.ContainsKey(sampler.name) == false)
                                {
                                    query.Add(sampler.name);
                                    addedKey.Add(sampler.name, true);
                                }
                            }

                            // データから構築する
                            foreach (var patternAnimTarget in texPatternMatAnims.PatternAnimTargets)
                            {
                                // データで追加していないものを追加する
                                if (addedKey.ContainsKey(patternAnimTarget.sampler_name) == false)
                                {
                                    query.Add(patternAnimTarget.sampler_name);
                                    addedKey.Add(patternAnimTarget.sampler_name, false);
                                }
                            }

                            foreach (var sampler in query)
                            {
                                var nodeInfo = new TexturePatternAnimationCurveTreeNodeInfo(ActiveTarget, materialName, sampler);
                                if (nodeInfo.GetAnimTarget(ActiveTarget) == null)
                                {
                                    Debug.Assert(false);
                                    nodeInfo.CreateAnimTarget(ActiveTarget);
                                }
                                var animTarget = nodeInfo.GetAnimTarget(ActiveTarget);
                                if (!materials.Any() && !nodeInfo.HasKeyFrame())
                                {
                                    continue;
                                }

                                var paramNode = new CurveTreeInfo()
                                {
                                    Text			= sampler,
                                    Id				= sampler,
                                    AnimationCurve	= nodeInfo,
                                    IsBound			= addedKey[sampler] || !materials.Any(),
                                    NonEditableKind	= addedKey[sampler] ? AnimationDocument.NonEditableKind.Editable :
                                                                          AnimationDocument.NonEditableKind.TexturePattern_NotFoundSampler,
                                    ShowEmptyNode	= isAllShowNode,
                                    IsModified		= animTarget.IsModified,
                                    Category = CurveTreeNodeCategory.TexPatAnim,
                                };

                                if (Viewer.Manager.Instance.IsConnected &&
                                    ActiveTarget.TexPatterns.Count == 0)
                                {
                                    paramNode.IsBound = false;
                                }

                                materialNode.Nodes.Add(paramNode);
                            }

                            materialNode.IsBound = materialNode.Nodes.Any(x => x.IsBound);
                        }
                        if (materialNode.Nodes.Any())
                        {
                            rootNode.Nodes.Add(materialNode);
                        }
                    }

                    rootNode.TrimInvisibleNodes();

                    // 文字列によるフィルター
                    CurveTreeInfo.FilterCurveTreeInfoRoot(rootNode, (Owner as AnimationObjectPropertyPanel).SearchTexts);

                    var nodes = rootNode.ConvertToTreeNode();
                    if (nodes.IsSameStructure(CurveEditorPanel.CurveTreeView.Nodes) == false)
                    {
                        if ((CurveEditorPanel.UpdateFormInfo == null) || (CurveEditorPanel.UpdateFormInfo.TargetOrPageChanged == false))
                        {
                            // つくり直す


                            foreach (var tuple in CurveTreeView.NodesAndFullPath((CurveTreeNode)nodes, nodes.id))
                            {
                                // チェック状態を設定する
                                bool isChecked = false;
                                if (visibledNodes_.TryGetValue(tuple.Item2, out isChecked))
                                {
                                    if (isChecked != tuple.Item1.Checked)
                                    {
                                        tuple.Item1.Checked = isChecked;
                                    }
                                }

                                bool isExpanded = false;
                                if (isExpandedNodes_.TryGetValue(tuple.Item2, out isExpanded))
                                {
                                    if (isExpanded)
                                    {
                                        if (!tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Expand();
                                        }
                                    }
                                    else
                                    {
                                        if (tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Collapse();
                                        }
                                    }
                                }
                            }

                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);
                        }
                        else
                        {
                            nodes.ExpandAll();

                            // つくり直す
                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);

                            // 新規に開いたときは全表示にする
                            CurveEditorPanel.VisibleAllNode();
                        }
                    }
                    else
                    {
                        ((CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0]).CopyInfo(nodes);
                    }
                }
            }
        }

        public void Select(System.Collections.Generic.IEnumerable<string> materials, string sampler)
        {
            CurveEditorPanel.SelectedCurves = CurveEditorPanel.AllCurves.Where(x => materials.Contains(x.ParentName) && x.Name == sampler).ToList();
        }

        /// <summary>
        /// カーブを選択しダイアログを開いた後にフィットする
        /// </summary>
        public void FitAfterSelect()
        {
            // スケールの設定
            CurveEditorPanel.FitAllViewIn(CurveEditorPanel.SelectedCurves, true);
        }

        #region コピー＆ペースト
        public class CopyData
        {
            public int frame_count { get; set; }
            public bool loop { get; set; }
            public List<TexturePatternAnimation.TexPatternMatAnim> TexPatternMatAnims { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(TexturePatternAnimation target)
        {
            return
                new CopyData()
                {
                    frame_count = target.Data.tex_pattern_anim_info.frame_count,
                    loop = target.Data.tex_pattern_anim_info.loop,
                    TexPatternMatAnims = ObjectUtility.Clone(target.TexPatternMatAnims),
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            if (Viewer.Manager.Instance.IsConnected &&
                TexturePatternAnimation.AllKeys(((CopyData)pasteObject).TexPatternMatAnims)
                .Any(x => x.Value < 0 || ActiveTarget.TexPatterns.Count <= x.Value))
            {
                if (UIMessageBox.OkCancel(res.Strings.TexturePatternAnimation_OutOfRange_Paste))
                {
                    Viewer.Manager.Instance.Close();
                }
                else
                {
                    return;
                }
            }

            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
            using (var block = new Viewer.ViewerDrawSuppressBlock(AnimationCurveEditCommand.AnimationMessageFilter))
            {
                var copyData = (CopyData)pasteObject;
                commandSet.Add(TexturePatternAnimationGeneralPage.CreateEditCommand_frame_count(targets, copyData.frame_count).Execute());
                commandSet.Add(TexturePatternAnimationGeneralPage.CreateEditCommand_loop(targets, copyData.loop).Execute());
                commandSet.Add(
                    new GeneralGroupReferenceEditCommand<List<TexturePatternAnimation.TexPatternMatAnim>>(
                        targets,
                        GuiObjectID.TexturePatternAnimation,
                        ObjectUtility.MultipleClone(copyData.TexPatternMatAnims, targets.Objects.Count),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is TexturePatternAnimation);
                            var texPatternAnim = target as TexturePatternAnimation;

                            swap = ObjectUtility.Clone(texPatternAnim.TexPatternMatAnims);
                            texPatternAnim.TexPatternMatAnims.Clear();
                            texPatternAnim.TexPatternMatAnims.AddRange(data as List<TexturePatternAnimation.TexPatternMatAnim>);

                            texPatternAnim.UpdateIsModifiedAnimTargetAll();
                            Viewer.LoadOrReloadAnimation.Send(texPatternAnim);
                        }
                    ).Execute()
                );

                var active = (TexturePatternAnimation)targets.Active;
                var command = active.CreateUpdateBindCommand();
                if (command != null)
                {
                    commandSet.Add(command.Execute());
                }
            }

            commandSet.Reverse();
            return commandSet;
        }
        #endregion
    }
}
