﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using App.ConfigData;
using App.Controls;
using App.Data;
using System.Linq;
using System.Collections.Generic;
using App.Command;
using App.Utility;
using ConfigCommon;
using System.Drawing;

namespace App.PropertyEdit
{
    public partial class SkeletalAnimationPreviewPage : SkeletalAnimationPropertyPage
    {
        public SkeletalAnimationPreviewPage() :
            base(PropertyPageID.SkeletalAnimationPreview)
        {
            InitializeComponent();
        }

        // TODO: PageKey
        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_animation_property_window;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new SkeletalAnimationPreviewPage();
        }

        protected override void InitializeFormInternal()
        {
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            lblRetargetingModelName.IsModified = ActiveTarget.IsPreviewRetargetingHostModelNameChanged();
            cbxMotionMirroring.IsModified = ActiveTarget.IsPreviewMotionMirroringChanged();

            using (new UpdateBlock(cmbRetargetingModelName))
            {
                cmbRetargetingModelName.Items.Clear();

                var models = DocumentManager.Models.Select(x => x.Name)
                    .Concat(Enumerable.Repeat(ActiveTarget.RetargetingHostModelName, 1))
                    .Where(x => x != null)
                    .OrderBy(x => x)
                    .Distinct();
                foreach (var name in models)
                {
                    cmbRetargetingModelName.AddItem(name, name);
                }
                cmbRetargetingModelName.AddItem(res.Strings.SkeletalAnimation_Preview_NotSelect, null);

                cmbRetargetingModelName.SelectedItemData = ActiveTarget.RetargetingHostModelName;

                cbxMotionMirroring.Checked = ActiveTarget.MotionMirroring;
            }
        }

        public static bool IsModified(SkeletalAnimation activeTarget)
        {
            return activeTarget != null &&
                (activeTarget.IsPreviewRetargetingHostModelNameChanged() || activeTarget.IsPreviewMotionMirroringChanged());
        }

        private void cmbRetargetingModelName_SelectedIndexChanged(object sender, System.EventArgs e)
        {
            TheApp.CommandManager.Execute(
                CreateSetRetargetingHostModelCommand(
                    Targets.Objects.OfType<SkeletalAnimation>(),
                    (string)cmbRetargetingModelName.SelectedItemData));

        }

        public static EditCommand CreateSetRetargetingHostModelCommand(IEnumerable<SkeletalAnimation> objs, string modelName)
        {
            var anims = objs.ToArray();
            var targets = Enumerable.Repeat(modelName, anims.Length).ToArray();
            return new GeneralGroupValueEditCommand<int>(
                new GuiObjectGroup(DocumentManager.ProjectDocument),
                GuiObjectID.Project,
                0,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    for (int i = 0; i < anims.Length; i++)
                    {
                        var anim = anims[i];
                        var tmp = anim.RetargetingHostModelName;
                        anim.RetargetingHostModelName = targets[i];
                        targets[i] = tmp;
                        Viewer.SetRetargetHostModel.Send(anim);
                    }
                });
        }

        #region コピー＆ペースト
        private class CopyData
        {
            //
            public string RetargetingModelName { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return CanCopy(ActiveTarget);
        }

        public static bool CanCopy(SkeletalAnimation ActiveTarget)
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(SkeletalAnimation target)
        {
            return
                new CopyData()
                {
                    RetargetingModelName = target.RetargetingHostModelName,
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateSetRetargetingHostModelCommand(
                    targets.Objects.OfType<SkeletalAnimation>(),
                    copyData.RetargetingModelName));
            }
            return commandSet.Execute();
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return CanPaste(Targets, copiedObjectInfo, copiedObject);
        }

        public static bool CanPaste(GuiObjectGroup targets, object copiedObjectInfo, object copiedObject)
        {
            return true;
        }
        #endregion

        private void cmbRetargetingModelName_CustomDrawItem(object sender, CustomDrawListControlItemEventArgs e)
        {
            if (e.Item != null)
            {
                e.ForeColor = e.Item.Data == null || DocumentManager.Models.Any(x => x.Name == (string)e.Item.Data)
                    ? SystemColors.ControlText
                    : Color.Red;
            }
        }

        private void cbxMotionMirroring_CheckedChanged(object sender, System.EventArgs e)
        {
            TheApp.CommandManager.Execute(
                CreateSetMotionMirroringCommand(
                    Targets.Objects.OfType<SkeletalAnimation>()));

        }

        public static EditCommand CreateSetMotionMirroringCommand(IEnumerable<SkeletalAnimation> objs)
        {
            var anims = objs.ToArray();
            return new GeneralGroupValueEditCommand<int>(
                new GuiObjectGroup(DocumentManager.ProjectDocument),
                GuiObjectID.Project,
                0,
                delegate (ref GuiObject target, ref object data, ref object swap)
                {
                    for (int i = 0; i < anims.Length; i++)
                    {
                        var anim = anims[i];
                        anim.MotionMirroring = !anim.MotionMirroring;
                        G3dHioLibProxy.Hio.SetPlayMotionMirroringEnabled(anim, anim.MotionMirroring);
                    }
                });
        }
    }
}
