﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class ShapeAnimationCurveEditPage : ShapeAnimationPropertyPage, IAnimationEditPage
    {
        public ShapeAnimationCurveEditPage() :
            base(PropertyPageID.ShapeAnimationCurveEdit)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_curve_editor;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new ShapeAnimationCurveEditPage();
        }

        protected override void InitializeFormInternal()
        {
                AnimationObjectPropertyPanel animationObjectPropertyPanel = Owner as AnimationObjectPropertyPanel;
            Debug.Assert(animationObjectPropertyPanel != null);

            CurveEditorPanel.Initialize(animationObjectPropertyPanel, ActiveTarget.ObjectID);

            CurveEditorPanel.UpdateTreeView       += (s, e) => UpdateTreeView();
            CurveEditorPanel.ChangeSelectedCurves += (s, e) => UpdateSelected();
            CurveEditorPanel.ChangeVisibledCurves += (s, e) => UpdateVisibled();
            CurveEditorPanel.GetFrameCount        = () => ActiveTarget.Data.shape_anim_info.frame_count;
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width),
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height));
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && !ObjectPropertyDialog.InternallyChangingSize.IsChanging &&
                Owner.Owner.WindowState == System.Windows.Forms.FormWindowState.Normal)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height = Height;
            }

            base.OnSizeChanged(e);
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            CurveEditorPanel.TargetGroup = Targets;

            CurveEditorPanel.UpdateForm(updateFormInfo);
        }

        protected override void UpdateFormOnPageCreatedInternal()
        {
            CurveEditorPanel.UpdateFormOnPageCreatedInternal();
        }

        public override void BeforePageDeactivated()
        {
            CurveEditorPanel.BeforePageDeactivated();
        }

        public override void AfterPageActiveted(ObjectPropertyPage oldPage)
        {
            CurveEditorPanel.AfterPageActivated();
        }

        private void UpdateSelected()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private void UpdateVisibled()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private Dictionary<string, bool> visibledNodes_ = new Dictionary<string, bool>();
        private readonly Dictionary<string, bool> isExpandedNodes_ = new Dictionary<string, bool>();

        private void UpdateTreeView()
        {
            using (var sb = new UIControlEventSuppressBlock())
            using (var ub = new UpdateBlock(CurveEditorPanel.CurveTreeView))
            {
                // ノードの状態を保存しておく
                if (CurveEditorPanel.CurveTreeView.Nodes.Count > 0)
                {
                    var root = (CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0];
                    foreach (var tuple in CurveTreeView.NodesAndFullPath(root, root.id))
                    {
                        // チェック状態
                        visibledNodes_[tuple.Item2] = tuple.Item1.Checked;

                        // 開閉状態
                        isExpandedNodes_[tuple.Item2] = tuple.Item1.Nodes.Count == 0 || tuple.Item1.IsExpanded;
                    }
                }

                int oldScrollX = CurveEditorPanel.CurveTreeView.ScrollX;
                int oldScrollY = CurveEditorPanel.CurveTreeView.ScrollY;

                bool isAllShowNode = (Owner as AnimationObjectPropertyPanel).IsAllShowNode;

                // つくり直す
                {
                    var rootNode = new CurveTreeInfo()
                    {
                        Text = ActiveTarget.Name,
                        Id = ActiveTarget.Name,
                        ShowEmptyNode = true,
                    };

                    // 実体のあるもの
                    foreach (var shape in ActiveTarget.VertexShapeAnims)
                    {
                        var node = new CurveTreeInfo()
                        {
                            Text = shape.shape_name,
                            Id = shape.shape_name,
                        };
                        rootNode.Nodes.Add(node);

                        //IEnumerable<string> keys = shape.ShapeAnims.Select(x => x.key_shape_name);
                        var keys = (from model in DocumentManager.Models
                                    where model.AllAnimations.Contains(ActiveTarget)
                                    from modelShape in model.Shapes
                                    where modelShape.Name == shape.shape_name && modelShape.Data.key_shape_array != null
                                    from key_shape in modelShape.Data.key_shape_array.key_shape
                                    select key_shape.name).Distinct().ToList();
                        var modelKeys = keys.ToDictionary(x => x, x => true);
                        foreach (var key in shape.ShapeAnims.Select(x => x.key_shape_name))
                        {
                            if (!modelKeys.ContainsKey(key))
                            {
                                modelKeys.Add(key, false);
                                keys.Add(key);
                            }
                        }
                        //keys = addition.Union(keys);

                        foreach (var key in keys)
                        {
//							if (key != shape.base_name)
                            {
                                var AnimationCurve = new ShapeAnimationCurveTreeNodeInfo(ActiveTarget, shape.shape_name, key);
                                if (AnimationCurve.GetAnimTarget(ActiveTarget) == null)
                                {
                                    Debug.Assert(false);
                                    AnimationCurve.CreateAnimTarget(ActiveTarget);
                                }

                                var animTarget = AnimationCurve.GetAnimTarget(ActiveTarget);
                                var nodeName = (key != shape.base_name) ? key : string.Format(Strings.ShapeAnimationCurveEditPage_UpdateTreeView_BaseShape, key);
                                var keyNode = new CurveTreeInfo()
                                {
                                    Text = nodeName,
                                    Id = key,
                                    AnimationCurve = AnimationCurve,
                                    Category = CurveTreeNodeCategory.ShapeAnimTarget,
                                    IsBound = false,
                                    NonEditableKind = AnimationDocument.NonEditableKind.ShapeAnimation_NotEditable,
                                    ShowEmptyNode = isAllShowNode,
                                    IsModified = animTarget.IsModified,
                                };

                                node.Nodes.Add(keyNode);
                            }
                        }

                        node.IsBound = node.Nodes.Any(x => x.IsBound);
                    }

                    rootNode.TrimInvisibleNodes();

                    // 文字列によるフィルター
                    CurveTreeInfo.FilterCurveTreeInfoRoot(rootNode, (Owner as AnimationObjectPropertyPanel).SearchTexts);

                    var nodes = rootNode.ConvertToTreeNode();
                    if (nodes.IsSameStructure(CurveEditorPanel.CurveTreeView.Nodes) == false)
                    {
                        if ((CurveEditorPanel.UpdateFormInfo == null) || (CurveEditorPanel.UpdateFormInfo.TargetOrPageChanged == false))
                        {
                            // つくり直す


                            foreach (var tuple in CurveTreeView.NodesAndFullPath((CurveTreeNode)nodes, nodes.id))
                            {
                                // チェック状態を設定する
                                bool isChecked = false;
                                if (visibledNodes_.TryGetValue(tuple.Item2, out isChecked))
                                {
                                    if (isChecked != tuple.Item1.Checked)
                                    {
                                        tuple.Item1.Checked = isChecked;
                                    }
                                }

                                bool isExpanded = false;
                                if (isExpandedNodes_.TryGetValue(tuple.Item2, out isExpanded))
                                {
                                    if (isExpanded)
                                    {
                                        if (!tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Expand();
                                        }
                                    }
                                    else
                                    {
                                        if (tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Collapse();
                                        }
                                    }
                                }
                            }

                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);
                        }
                        else
                        {
                            nodes.ExpandAll();

                            // つくり直す
                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);

                            // 新規に開いたときは全表示にする
                            CurveEditorPanel.VisibleAllNode();
                        }
                    }
                    else
                    {
                        ((CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0]).CopyInfo(nodes);
                    }
                }

                CurveEditorPanel.CurveTreeView.ScrollX = oldScrollX;
                CurveEditorPanel.CurveTreeView.ScrollY = oldScrollY;
            }
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return false;
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return false;
        }
        #endregion
    }
}
