﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class ShapeVertexPage : ShapePropertyPage
    {
        public ShapeVertexPage() :
            base(PropertyPageID.ShapePolygon)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_shape_property_window_vertex_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new ShapeVertexPage();
        }

        protected override void InitializeFormInternal()
        {
        }

        private List<VertexInfo> vertexInfos;
        private List<ColumnInfo> columnInfos;
        private readonly Dictionary<string, bool> checkedDictionary = new Dictionary<string, bool>();

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            var model = DocumentManager.Models.First(x => x.Shapes.Contains(ActiveTarget));
            var vertex = model.Vertices[ActiveTarget.Data.shape_info.vertex_index];
            var material = model.Materials.First(x => x.Name == ActiveTarget.Data.shape_info.mat_name);
            var shader = material.MaterialShaderAssign;
            lowTextBox1.Text = string.IsNullOrEmpty(shader.ShaderName)? res.Strings.Flag_NotAssigned: shader.ShaderName;

            var streams = model.BinaryStreams;

            // columnInfos の更新
            columnInfos = new List<ColumnInfo>();
            ColumnInfo columnInfo;

            // メッシュ
            columnInfo = new ColumnInfo();
            columnInfo.Text = clhMeshIndex.Text;
            columnInfo.Width = clhMeshIndex.Width;
            columnInfo.ItemText = x => x.meshIndex.ToString();
            columnInfo.ItemComparison = (x, y) => (x.meshIndex.CompareTo(y.meshIndex));
            columnInfos.Add(columnInfo);

            // サブメッシュ
            columnInfo = new ColumnInfo();
            columnInfo.Text = clhSubmeshIndex.Text;
            columnInfo.Width = clhSubmeshIndex.Width;
            columnInfo.ItemText = x => x.submeshIndex.ToString();
            columnInfo.ItemComparison = (x, y) => (x.submeshIndex.CompareTo(y.submeshIndex));
            columnInfos.Add(columnInfo);

            // ポリゴン
            columnInfo = new ColumnInfo();
            columnInfo.Text = clhPolygonIndex.Text;
            columnInfo.Width = clhPolygonIndex.Width;
            columnInfo.ItemText = x => string.Format("{0}[{1}]", x.TriangleIndex, x.IndexInTriangle);
            columnInfo.ItemComparison = (x, y) => {
                int diff = x.TriangleIndex.CompareTo(y.TriangleIndex);
                if (diff == 0)
                {
                    return x.IndexInTriangle.CompareTo(y.IndexInTriangle);
                }
                return diff;
            };
            columnInfos.Add(columnInfo);

            // 頂点インデックス
            columnInfo = new ColumnInfo();
            columnInfo.Text = clhVertexIndex.Text;
            columnInfo.Width = clhVertexIndex.Width;
            columnInfo.ItemText = x => x.vertexIndex.ToString();
            columnInfo.ItemComparison = (x, y) => (x.vertexIndex.CompareTo(y.vertexIndex));
            columnInfos.Add(columnInfo);

            // 属性一覧の更新
            {
                List<ListViewItem> newAttributeItems = new List<ListViewItem>();
                foreach (var attrib in vertex.vtx_attrib_array.vtx_attrib)
                {
                    // lvwAttribute のアイテム設定
                    bool showColumn = true;
                    ListViewItem item = new ListViewItem();
                    item.Tag = attrib.name;
                    newAttributeItems.Add(item);
                    foreach (var col in lvwAttribute.Columns)
                    {
                        item.SubItems.Add("");
                    }

                    // 名前
                    item.SubItems[clhName.Index].Text = attrib.name;

                    // ヒント
                    item.SubItems[clhHint.Index].Text = attrib.hint;

                    // 型
                    item.SubItems[clhType.Index].Text = UIText.EnumValue(attrib.type);

                    // 頂点数
                    item.SubItems[clhCount.Index].Text = attrib.count.ToString();

                    // 量子化型
                    item.SubItems[clhQuantizeType.Index].Text = UIText.EnumValue(attrib.quantize_type);

                    // バッファインデックス
                    var buffers = vertex.VtxBuffers();
                    var indices = (from i in Enumerable.Range(0, buffers.Length)
                                   where buffers[i].input_array.input.Any(x => x.attrib_index == attrib.attrib_index)
                                   select i).ToArray();
                    var builder = new StringBuilder();
                    for (int i = 0; i < indices.Length; i++)
                    {
                        if (i > 0)
                        {
                            builder.Append(", ");
                        }
                        builder.Append(indices[i]);
                    }
                    item.SubItems[clhBufferIndex.Index].Text = builder.ToString();

                    // 割り当てられているか
                    item.SubItems[clhProgram.Index].Text = shader.AttribAssigns.Any(x => x.attrib_name == attrib.name) ?
                        res.Strings.Flag_Use :
                        res.Strings.Flag_NotUse;

                    if (checkedDictionary.ContainsKey(attrib.name))
                    {
                        showColumn = checkedDictionary[attrib.name];
                    }

                    item.Checked = showColumn;
                    checkedDictionary[attrib.name] = showColumn;

                    var stream = streams[attrib.stream_index].GetIList();
                    int count = (int)attrib.type % 4 + 1;
                    for (int i = 0; i < count; i++)
                    {
                        columnInfo = new ColumnInfo();
                        switch (i)
                        {
                            case 0:
                                columnInfo.Text = (count == 1) ? attrib.hint : attrib.hint + ".x";
                                break;
                            case 1:
                                columnInfo.Text = attrib.hint + ".y";
                                break;
                            case 2:
                                columnInfo.Text = attrib.hint + ".z";
                                break;
                            case 3:
                                columnInfo.Text = attrib.hint + ".w";
                                break;
                        }
                        int ii = i; // 局所化
                        string key = attrib.name;
                        columnInfo.ItemText = x => stream[x.vertexIndex * count + ii].ToString();
                        columnInfo.ItemComparison = (x, y) => ((IComparable)stream[x.vertexIndex * count + ii]).CompareTo(stream[y.vertexIndex * count + ii]);
                        columnInfo.IsVisible =
                            () =>
                            {
                                return
                                checkedDictionary.ContainsKey(key) &&
                            checkedDictionary[key];
                            };
                        columnInfos.Add(columnInfo);
                    }
                }
                lvwAttribute.BeginUpdate();
                lvwAttribute.Items.Clear();
                lvwAttribute.Items.AddRange(newAttributeItems.ToArray());
                lvwAttribute.EndUpdate();
            }

            UpdateLvwVertex();
        }

        public static bool IsModified(Shape activeTarget)
        {
            return activeTarget != null && activeTarget.IsStreamModified;
        }

        private void UpdateLvwVertex()
        {
            using(var lockWindowUpdate = new LockWindowUpdate(this))
            {
                var model = DocumentManager.Models.First(x => x.Shapes.Contains(ActiveTarget));
                var streams = model.BinaryStreams;

                using (var block = new UpdateBlock(lvwVetex))
                {
                    // おまじない
                    Win32.NativeMethods.SendMessage(lvwVetex.Handle, Win32.LVM.LVM_SCROLL, IntPtr.Zero, IntPtr.Zero);
                    lvwVetex.VirtualListSize = 0;
                    lvwVetex.VirtualMode = false;
                    lvwVetex.Columns.Clear();

                    foreach (var info in columnInfos.Where(x => x.IsVisible()))
                    {
                        var header = new ColumnHeader()
                        {
                            Text = info.Text,
                            TextAlign = HorizontalAlignment.Right,
                            Width = info.Width,
                            Tag = info,
                        };

                        lvwVetex.Columns.Add(header);
                    }

                    // vertexInfos の更新
                    int totalItemCount = 0;
                    foreach(var mesh in ActiveTarget.Data.mesh_array.mesh)
                    {
                        if (mesh.mode == mesh_modeType.triangles)
                        {
                            totalItemCount += mesh.count;
                        }
                        else
                        {
                            totalItemCount += mesh.submesh_array.submesh.Sum(x => 3 * (x.count - 2));
                        }
                    }
                    vertexInfos = new List<VertexInfo>(totalItemCount);

                    var vertex = model.Vertices[ActiveTarget.Data.shape_info.vertex_index];
                    // メッシュ0用のオフセット0を追加
                    var lodOffsets = new int[] {0};
                    if (vertex.lod_offset != null && vertex.lod_offset.Value.Any())
                    {
                        lodOffsets = lodOffsets.Concat(G3dDataParser.ParseIntArray(vertex.lod_offset.Value)).ToArray();
                    }

                    var meshIndex = 0;
                    foreach (var mesh in ActiveTarget.Data.mesh_array.mesh)
                    {
                        var submeshIndex = 0;
                        var triangleIndex = 0;
                        var indexStream = streams[mesh.stream_index].GetIList();
                        var vertexIndexOffset = 0;

                        foreach (var submesh in mesh.submesh_array.submesh)
                        {
                            int triangleCount = mesh.mode == mesh_modeType.triangles ? submesh.count / 3 : submesh.count - 2;
                            if (meshIndex < lodOffsets.Count())
                            {
                                vertexIndexOffset = lodOffsets[meshIndex];
                            }

                            for (int i = 0; i < triangleCount; i++)
                            {
                                for (int j = 0; j < 3; j++)
                                {
                                    var vertexInfo = new VertexInfo();
                                    int vertexIndexIndex = 0;
                                    switch (mesh.mode)
                                    {
                                        case mesh_modeType.triangles:
                                            vertexIndexIndex = i * 3 + j;
                                            break;
                                        case mesh_modeType.triangle_strip:
                                            vertexIndexIndex = i + j;
                                            break;
                                        case mesh_modeType.triangle_fan:
                                            vertexIndexIndex = j == 0 ? 0 : i + j;
                                            break;
                                    }
                                    vertexInfo.meshIndex = meshIndex;
                                    vertexInfo.submeshIndex = submeshIndex;
                                    vertexInfo.TriangleIndex = triangleIndex;
                                    vertexInfo.IndexInTriangle = j;
                                    vertexInfo.vertexIndex = (int)indexStream[vertexIndexIndex + submesh.offset] + vertexIndexOffset;

                                    // 属性の追加
                                    vertexInfos.Add(vertexInfo);
                                }
                                triangleIndex++;
                            }
                            submeshIndex++;
                        }
                        meshIndex++;
                    }

                    // バーチャルリストサイズの更新

                    lvwVetex.VirtualMode = true;
                    lvwVetex.VirtualListSize = totalItemCount;
                }
            }
        }

        private void lvwAttribute_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            var visibleColumnInfos = columnInfos.Where(x => x.IsVisible()).ToArray();
            if (visibleColumnInfos.Length <= e.Column)
            {
                return;
            }

            // おまじない
            Win32.NativeMethods.SendMessage(lvwVetex.Handle, Win32.LVM.LVM_SCROLL, IntPtr.Zero, IntPtr.Zero);

            lvwVetex.BeginUpdate();
            {
                int tmp = lvwVetex.VirtualListSize;
                lvwVetex.VirtualListSize = 0;
                lvwVetex.VirtualMode = false;
                var columnInfo = visibleColumnInfos[e.Column];
                if (columnInfo.AscendingOrder)
                {
                    vertexInfos = new List<VertexInfo>(vertexInfos.OrderByDescending(x => x, columnInfo));
                    columnInfo.AscendingOrder = false;
                }
                else
                {
                    vertexInfos = new List<VertexInfo>(vertexInfos.OrderBy(x => x, columnInfo));
                    columnInfo.AscendingOrder = true;
                }
                lvwVetex.VirtualMode = true;
                lvwVetex.VirtualListSize = tmp;
            }
            lvwVetex.EndUpdate();
            lvwVetex.HideToolTip();
        }

        class VertexInfo
        {
            public int meshIndex;
            public int submeshIndex;
            public int vertexIndex;
            public int TriangleIndex;
            public int IndexInTriangle;
        }

        private void lvwAttribute_RetrieveVirtualItem(object sender, RetrieveVirtualItemEventArgs e)
        {
            var vertexInfo = vertexInfos[e.ItemIndex];
            ListViewItem item = new ListViewItem(lvwVetex.Columns.OfType <ColumnHeader>().Select(x => ((ColumnInfo)x.Tag).ItemText(vertexInfo)).ToArray());
            e.Item = item;
        }

        private class ColumnInfo : IComparer<VertexInfo>
        {
            public string Text = string.Empty;
            public Comparison<VertexInfo> ItemComparison = (x, y) => 0;
            public Func<VertexInfo, string> ItemText = x=>string.Empty;
            public int Width = 90;
            public Func<bool> IsVisible = () => true;
            public bool AscendingOrder = false;
            int IComparer<VertexInfo>.Compare(VertexInfo x, VertexInfo y)
            {
                return ItemComparison(x, y);
            }
        }

        private void lvwAttribute_ItemCheck(object sender, ItemCheckEventArgs e)
        {
            checkedDictionary[(string)lvwAttribute.Items[e.Index].Tag] = e.NewValue == CheckState.Checked;

            UpdateLvwVertex();
        }
    }
}
