﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using App.Controls;
using App.Utility;
using App.res;

namespace App.PropertyEdit
{
    public partial class ShaderDefinitionSourceCodeSubPage
    {
        private string Word
        {
            get { return tspSearchWord.Text; }
            set { tspSearchWord.Text = value; }
        }

        private void InitializeSearch()
        {
            // 動作前提です。
            Debug.Assert(tbxSourceCode.WordWrap == false);

            tsbSearchPrior.Tag	= ShaderDefinitionPropertyPanel.SearchDir.Prior;
            tsbSearchNext.Tag	= ShaderDefinitionPropertyPanel.SearchDir.Next;

            CancelOldSearchResult();
        }

        private const int MaxHistory = 20;

        // コンフィグを読み出す
        private void LoadConfig()
        {
            tspSearchWord.SelectedIndexChanged -= tspSearchWord_SelectedIndexChanged;
            {
                tsbSearchAllTarget.Checked	= ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.AllTarget;
                tsbCaseSensitive.Checked	= ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.CaseSensitive;
                tsbRegex.Checked			= ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.Regex;

                var word = Word;

                tspSearchWord.Items.Clear();
                foreach(var item in ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.Histories.Take(MaxHistory))
                {
                    tspSearchWord.Items.Add(item);
                }

                Word = word;
            }
            tspSearchWord.SelectedIndexChanged += tspSearchWord_SelectedIndexChanged;
        }

        private void State_CheckedChanged(object sender, EventArgs e)
        {
            ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.AllTarget		= tsbSearchAllTarget.Checked;
            ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.CaseSensitive	= tsbCaseSensitive.Checked;
            ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.Regex			= tsbRegex.Checked;
        }

        private void tsbSearch_Click(object sender, EventArgs e)
        {
            SearchWord(Word, (ShaderDefinitionPropertyPanel.SearchDir)(sender as UIToolStripButton).Tag);
        }

        private void tspSearchWord_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                SearchWord(Word, e.Shift ? ShaderDefinitionPropertyPanel.SearchDir.Prior : ShaderDefinitionPropertyPanel.SearchDir.Next);
            }
        }

        private void tspSearchWord_SelectedIndexChanged(object sender, EventArgs e)
        {
//			Word = (string)tspSearchWord.Items[tspSearchWord.SelectedIndex];

            CancelOldSearchResult();
            SearchWord(Word, ShaderDefinitionPropertyPanel.SearchDir.Next);
        }

        private void tsbSearchTarget_Click(object sender, EventArgs e)
        {
            ;
        }

        private void tsbCaseSensitive_Click(object sender, EventArgs e)
        {
            CancelOldSearchResult();
            SearchWord(Word, ShaderDefinitionPropertyPanel.SearchDir.Next);
        }

        private void tsbRegex_Click(object sender, EventArgs e)
        {
            CancelOldSearchResult();
            SearchWord(Word, ShaderDefinitionPropertyPanel.SearchDir.Next);
        }

        // 前回の強調表示をキャンセル
        private void CancelOldSearchResult()
        {
            // スクロール位置を記録
            var scrollPos = tbxSourceCode.ScrollPos;

            // 検索結果をクリア
            tbxSourceCode.Clear();
            tbxSourceCode.Text = SourceCode;

            // スクロール位置を戻す
            tbxSourceCode.ScrollPos = scrollPos;

            currentWord_	= null;
            wordPositions_.Clear();
            wordSelectIndex_ = -1;
        }

        private class WordPositions
        {
            public int X{ get; set; }
            public int Y{ get; set; }
            public int Offset{ get; set; }
            public int Length{ get; set; }
        }

        private string				currentWord_		= null;
        private readonly List<WordPositions>	wordPositions_		= new List<WordPositions>();
        private int					wordSelectIndex_	= -1;

        private static readonly Regex escapeRegex_ = new Regex(@"[\.\$\^\{\[\(\|\)\*\+\?\\]", RegexOptions.Compiled);

        private void UpdateFormSearchWord(bool isCaseSensitive, bool isRegex)
        {
            LoadConfig();

            var lines = (new Regex("\n")).Split(SourceCode);
            var lineIndex = 0;

            // 正規表現オプションを作る
            var regexOptions = RegexOptions.None;
            {
                if (isCaseSensitive == false)
                {
                    regexOptions |= RegexOptions.IgnoreCase;
                }
            }

            Regex searchRegex = null;
            {
                if (isRegex)
                {
                    searchRegex = new Regex(Word, regexOptions);
                }
                else
                {
                    // 正規表現キーワードをエスケープする
                    var escaped = escapeRegex_.Replace(Word, @"\$0");
                    searchRegex = new Regex(escaped, regexOptions);
                }
            }

            // ソースと検索内容によっては時間が掛かるので待機カーソルに変更
            using (WaitCursor wait = new WaitCursor())
            {
                foreach (var line in lines)
                {
                    foreach (Match m in searchRegex.Matches(line))
                    {
                        foreach (Group g in m.Groups)
                        {
                            if (g.Length > 0)
                            {
                                // ソースの行数が多いと処理負荷が高いため、検出されてからオフセットを取得する
                                var lineOffset = tbxSourceCode.GetFirstCharIndexFromLine(lineIndex);

                                wordPositions_.Add(
                                    new WordPositions()
                                    {
                                        X = g.Index,
                                        Y = lineIndex,
                                        Offset = lineOffset + g.Index,
                                        Length = g.Length
                                    }
                                );
                            }
                        }
                    }

                    ++lineIndex;
                }
            }

            SetColorFromWordPositions();
        }

        private void SetColorFromWordPositions()
        {
            // ソースと検索内容によっては時間が掛かるので待機カーソルに変更
            using (WaitCursor wait = new WaitCursor())
            {
                foreach (var wordPosition in wordPositions_)
                {
                    tbxSourceCode.Select(wordPosition.Offset, wordPosition.Length);
                    var bcol = tbxSourceCode.SelectionBackColor;
                    tbxSourceCode.SelectionBackColor = Color.Red;
                    tbxSourceCode.SelectionColor = Color.White;
                    tbxSourceCode.Select(wordPosition.Offset, 0);
                    tbxSourceCode.SelectionBackColor = bcol;
                }
            }
        }

        private void SearchWord(string word, ShaderDefinitionPropertyPanel.SearchDir dir)
        {
            var isAllTarget		= tsbSearchAllTarget.Checked;

            var sourceCode = SourceCode;

            if (string.IsNullOrEmpty(word))
            {
                CancelOldSearchResult();
                return;
            }

            // 履歴
            using(var lockWindowUpdate = new LockWindowUpdate(tspSearchWord.Control))
            {
                tspSearchWord.SelectedIndexChanged -= tspSearchWord_SelectedIndexChanged;
                {
                    tspSearchWord.Items.Remove(word);
                    tspSearchWord.Items.Insert(0, word);

                    while (tspSearchWord.Items.Count > MaxHistory)
                    {
                        tspSearchWord.Items.RemoveAt(tspSearchWord.Items.Count - 1);
                    }

                    ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.Histories = new List<string>();
                    foreach(string item in tspSearchWord.Items)
                    {
                        ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionSourceCode.Histories.Add(item);
                    }

                    tspSearchWord.Text = (string)tspSearchWord.Items[0];
                }
                tspSearchWord.SelectedIndexChanged += tspSearchWord_SelectedIndexChanged;
            }

            using(var lockWindowUpdate = new LockWindowUpdate(tbxSourceCode))
            {
                try
                {
                    // 新規検索開始
                    if (word != currentWord_)
                    {
                        // 前回の強調表示をキャンセル
                        tbxSourceCode.Clear();
                        tbxSourceCode.Text = sourceCode;

                        // 現在のワードとする。
                        currentWord_ = word;
                        wordPositions_.Clear();
                        wordSelectIndex_ = -1;

                        var isCaseSensitive	= tsbCaseSensitive.Checked;
                        var isRegex			= tsbRegex.Checked;

                        UpdateFormSearchWord(isCaseSensitive, isRegex);

                        if (wordPositions_.Any())
                        {
                            var pos = wordPositions_.First();
                            SetCurrentSelectWord(pos);
                            wordSelectIndex_ = 0;
                        }
                    }
                    // スクロール位置を移動
                    else
                    {
                        if (wordPositions_.Any())
                        {
                            // 初めての検索
                            if (wordSelectIndex_ == -1)
                            {
                                wordSelectIndex_ = dir == ShaderDefinitionPropertyPanel.SearchDir.Prior ? wordPositions_.Count() - 1 : 0;

                                SetColorFromWordPositions();

                                var pos = wordPositions_[wordSelectIndex_];
                                SetCurrentSelectWord(pos);
                            }
                            else
                            {
                                var old = wordSelectIndex_;

                                wordSelectIndex_ += dir == ShaderDefinitionPropertyPanel.SearchDir.Prior ? -1 : +1;

                                var isChangeCode = false;
                                {
                                    if (isAllTarget)
                                    {
                                        isChangeCode =
                                            (wordSelectIndex_ == -1) ||
                                            (wordSelectIndex_ == wordPositions_.Count());
                                    }
                                    else
                                    {
                                        if (wordSelectIndex_ == -1)
                                        {
                                            wordSelectIndex_ = wordPositions_.Count() - 1;
                                        }

                                        if (wordSelectIndex_ == wordPositions_.Count())
                                        {
                                            wordSelectIndex_ = 0;
                                        }
                                    }
                                }

                                if (isChangeCode)
                                {
                                    SelectSourceCode(word, dir);
                                }
                                else
                                {
                                    // 前回の選択位置だけ強調表示を戻す
                                    var pos = wordPositions_[old];
                                    ClearSelectWord(pos);

                                    // 新しい選択位置を強調表示する
                                    pos = wordPositions_[wordSelectIndex_];
                                    SetCurrentSelectWord(pos);
                                }
                            }
                        }
                        else
                        {
                            // 一つもヒットしなければ次に移動
                            if (isAllTarget)
                            {
                                SelectSourceCode(word, dir);
                            }
                        }
                    }
                }
                catch(ArgumentException e)
                {
                    // 例外が起きたら無視。
                    // 例外が起きるケースとして、正規表現文字列から正規表現オブジェクトが作れないとき。など

                    UIMessageBox.Error(Strings.ShaderDefinitionSourceCodeSubPage_ErrorSearchRegex, e.Message);

                    CancelOldSearchResult();
                }
                catch(Exception e)
                {
                    // 未定義のエラー
                    DebugConsole.WriteLine(string.Format("error : {0}, {1}", e.GetType(), e.Message));
#if DEBUG
                    MessageBox.Show(string.Format("error : {0}, {1}", e.GetType(), e.Message));
#endif

                    CancelOldSearchResult();
                }
            }
        }

        // 選択単語の設定を行う
        //	色の変更
        //	スクロール
        private void SetCurrentSelectWord(WordPositions pos)
        {
            tbxSourceCode.Select(pos.Offset, pos.Length);
            var bcol = tbxSourceCode.SelectionBackColor;
            tbxSourceCode.SelectionBackColor = Color.Blue;
            tbxSourceCode.SelectionColor = Color.White;
            tbxSourceCode.Select(pos.Offset, 0);
            tbxSourceCode.SelectionBackColor = bcol;
            tbxSourceCode.ScrollToCaret();
        }

        private void ClearSelectWord(WordPositions pos)
        {
            tbxSourceCode.Select(pos.Offset, pos.Length);
            var bcol = tbxSourceCode.SelectionBackColor;
            tbxSourceCode.SelectionBackColor = Color.Red;
            tbxSourceCode.SelectionColor = Color.White;
            tbxSourceCode.Select(pos.Offset, 0);
            tbxSourceCode.SelectionBackColor = bcol;
        }

        private void SelectSourceCode(string word, ShaderDefinitionPropertyPanel.SearchDir dir)
        {
            Debug.Assert(Owner is ShaderDefinitionPropertyPanel);
            var owner = Owner as ShaderDefinitionPropertyPanel;

            using(var lockWindowUpdate = new LockWindowUpdate(owner.Owner))
            {
                var nodeIndex = owner.CurrentSourceCodeNodeIndex;
                if (nodeIndex != -1)
                {
                    var oldNodeIndex = nodeIndex;

                    var isCaseSensitive	= tsbCaseSensitive.Checked;
                    var isRegex			= tsbRegex.Checked;

                    ShaderDefinitionSourceCodeSubPage nextPage = null;
                    {
                        while (true)
                        {
//							nodeIndex += dir == ShaderDefinitionPropertyPanel.SearchDir.Prior ? -1 : +1;
                            nodeIndex = owner.MakeNextSourceCodeNodeIndex(nodeIndex, dir);

                            if (nodeIndex != oldNodeIndex)
                            {
                                // ページを切り替える

                                var page = owner.GetPropertyCategoryNodeFromNodeIndex(nodeIndex).PropertyPage as ShaderDefinitionSourceCodeSubPage;

                                // 検索語を伝搬させる
                                CopySearchConditions(page);
                                page.CancelOldSearchResult();
                                page.SearchWord(page.Word, dir);

                                // 検索語が一つ以上あれば
                                if (page.wordPositions_.Any())
                                {
                                    owner.CurrentSourceCodeNodeIndex = nodeIndex;

                                    nextPage = owner.ActivePage as ShaderDefinitionSourceCodeSubPage;

                                    // 検索語を伝搬させる
                                    Debug.Assert(nextPage != null);
                                    CopySearchConditions(nextPage);
                                    nextPage.CancelOldSearchResult();
                                    nextPage.SearchWord(nextPage.Word, dir);

                                    break;
                                }
                            }
                            else
                            {
                                // 移動できない
                                owner.CurrentSourceCodeNodeIndex = oldNodeIndex;
                                nextPage = null;
                                break;
                            }
                        }
                    }

                    if (nextPage != null)
                    {
                        // 検索後が見つかった時
                        CopySearchConditions(nextPage);
                        OneShotIdleProcess.Execute(() =>
                        {
                            if (!nextPage.IsDisposed)
                            {
                                nextPage.tspSearchWord.Focus();
                            }
                        });
                    }
                    else
                    {
                        // 検索後が見つからなかった時

                        CancelOldSearchResult();
                        SearchWord(Word, dir);

                        OneShotIdleProcess.Execute(() =>
                        {
                            if (!tspSearchWord.IsDisposed)
                            {
                                tspSearchWord.Focus();
                            }
                        });
                    }
                }
            }
        }

        private void CopySearchConditions(ShaderDefinitionSourceCodeSubPage nextPage)
        {
            nextPage.Word = Word;
            nextPage.tsbCaseSensitive.Checked = tsbCaseSensitive.Checked;
            nextPage.tsbRegex.Checked = tsbRegex.Checked;
            nextPage.tsbSearchAllTarget.Checked = tsbSearchAllTarget.Checked;
        }
    }
}
