﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class SceneAnimationGeneralPage : SceneAnimationPropertyPage
    {
        public SceneAnimationGeneralPage() :
            base(PropertyPageID.SceneAnimationGeneral)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_animation_property_window_general_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new SceneAnimationGeneralPage();
        }

        protected override void InitializeFormInternal()
        {
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            lblFrameResolution.IsModified = ActiveTarget.IsValueChanged(x => x.frame_resolution);
            ltbFrameResolution.Text = ActiveTarget.Data.scene_anim_info.frame_resolution.ToString();

            lblBakeAll.IsModified = ActiveTarget.IsValueChanged(x => x.bake_all);
            ltbBakeAll.Text			= UIText.FlagYesNo(ActiveTarget.Data.scene_anim_info.bake_all);

            lblBakeColor.IsModified = ActiveTarget.IsValueChanged(x => x.bake_tolerance_color);
            ltbBakeColor.Text				 = ActiveTarget.Data.scene_anim_info.bake_tolerance_color.ToString();

            lblBakeTexRotate.IsModified = ActiveTarget.IsValueChanged(x => x.bake_tolerance_rotate);
            ltbBakeTexRotate.Text			= ActiveTarget.Data.scene_anim_info.bake_tolerance_rotate.ToString();

            lblBakeTexTranslate.IsModified = ActiveTarget.IsValueChanged(x => x.bake_tolerance_translate);
            ltbBakeTexTranslate.Text		= ActiveTarget.Data.scene_anim_info.bake_tolerance_translate.ToString();

            lblQuantizeTexRotate.IsModified = ActiveTarget.IsValueChanged(x => x.quantize_tolerance_rotate);
            fepQuantizeTexRotate.Value		= ActiveTarget.Data.scene_anim_info.quantize_tolerance_rotate;

            lblQuantizeTexTranslate.IsModified = ActiveTarget.IsValueChanged(x => x.quantize_tolerance_translate);
            fepQuantizeTexTranslate.Value	= ActiveTarget.Data.scene_anim_info.quantize_tolerance_translate;

            lblPreset.IsModified = ActiveTarget.IsStringChanged(x => x.dcc_preset);
            ltbDccPreset.Text = ActiveTarget.Data.scene_anim_info.dcc_preset;

            lblDccMagnify.IsModified = ActiveTarget.IsValueChanged(x => x.dcc_magnify);
            ltbDccMagnify.Text = ActiveTarget.Data.scene_anim_info.dcc_magnify.ToString();

            lblDccStartFrame.IsModified = ActiveTarget.IsValueChanged(x => x.dcc_start_frame);
            ltbDccStartFrame.Text = ActiveTarget.Data.scene_anim_info.dcc_start_frame.ToString();

            lblDccEndFrame.IsModified = ActiveTarget.IsValueChanged(x => x.dcc_end_frame);
            ltbDccEndFrame.Text = ActiveTarget.Data.scene_anim_info.dcc_end_frame.ToString();

            lblDccFps.IsModified = ActiveTarget.IsValueChanged(x => x.dcc_fps);
            ltbDccFps.Text = ActiveTarget.Data.scene_anim_info.dcc_fps.ToString();

            UpdateCompress(ActiveTarget);

            if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
            {
                gbxQuantize.Visible = false;
                gbxQuantizationReductionRate.Visible = false;
            }
        }

        public static bool IsModified(SceneAnimation activeTarget)
        {
            return activeTarget != null &&
                (activeTarget.IsValueChanged(x => x.frame_resolution) ||
                activeTarget.IsValueChanged(x => x.bake_all) ||
                activeTarget.IsValueChanged(x => x.bake_tolerance_color) ||
                activeTarget.IsValueChanged(x => x.bake_tolerance_rotate) ||
                activeTarget.IsValueChanged(x => x.bake_tolerance_translate) ||
                activeTarget.IsValueChanged(x => x.quantize_tolerance_rotate) ||
                activeTarget.IsValueChanged(x => x.quantize_tolerance_translate) ||
                activeTarget.IsStringChanged(x => x.dcc_preset) ||
                activeTarget.IsValueChanged(x => x.dcc_magnify) ||
                activeTarget.IsValueChanged(x => x.dcc_start_frame) ||
                activeTarget.IsValueChanged(x => x.dcc_end_frame) ||
                activeTarget.IsValueChanged(x => x.dcc_fps));
        }

        private void UpdateCompress(SceneAnimation target)
        {
            UpdateQuantizationResult(lstQuantizationResult,
                0, 0,
                0, GetKeyDataSizeRotate(false, target),
                0, GetKeyDataSizeTranslate(false, target));
        }

        private int GetKeyDataSizeRotate(bool isBase, SceneAnimation target)
        {
            int size = 0;
            Predicate<camera_anim_target_targetType> rotateCamera = x => {
                switch (x)
                {
                    case camera_anim_target_targetType.rotate_x:
                    case camera_anim_target_targetType.rotate_y:
                    case camera_anim_target_targetType.rotate_z:
                    case camera_anim_target_targetType.twist:
                    case camera_anim_target_targetType.persp_fovy:
                        return true;
                }
                return false;
            };
            size += (from anim in target.CameraAnims
                     from animTarget in anim.CameraAnimTargets
                     where rotateCamera(animTarget.targetType)
                     let factor = isBase ? 4 : IfUtility.GetByteSizeFromCurveKeyType(animTarget.QuantizationInfo.key_type)
                     select factor * animTarget.KeyFrames.Count).DefaultIfEmpty().Aggregate((x, y) => x + y);
            Predicate<light_anim_target_targetType> rotateLight = x =>
            {
                switch (x)
                {
                    case light_anim_target_targetType.direction_x:
                    case light_anim_target_targetType.direction_y:
                    case light_anim_target_targetType.direction_z:
                    case light_anim_target_targetType.angle_attn_start:
                    case light_anim_target_targetType.angle_attn_end:
                        return true;
                }
                return false;
            };
            size += (from anim in target.LightAnims
                     from animTarget in anim.LightAnimTargets
                     where rotateLight(animTarget.targetType)
                     let factor = isBase ? 4 : IfUtility.GetByteSizeFromCurveKeyType(animTarget.QuantizationInfo.key_type)
                     select factor * animTarget.KeyFrames.Count).DefaultIfEmpty().Aggregate((x, y) => x + y);
            return size;
        }

        private int GetKeyDataSizeTranslate(bool isBase, SceneAnimation target)
        {
            int size = 0;
            Predicate<camera_anim_target_targetType> translateCamera = x =>
            {
                switch (x)
                {
                    case camera_anim_target_targetType.position_x:
                    case camera_anim_target_targetType.position_y:
                    case camera_anim_target_targetType.position_z:
                    case camera_anim_target_targetType.aim_x:
                    case camera_anim_target_targetType.aim_y:
                    case camera_anim_target_targetType.aim_z:
                    case camera_anim_target_targetType.near:
                    case camera_anim_target_targetType.far:
                    case camera_anim_target_targetType.ortho_height:
                        return true;
                }
                return false;
            };
            size += (from anim in target.CameraAnims
                     from animTarget in anim.CameraAnimTargets
                     where translateCamera(animTarget.targetType)
                     let factor = isBase ? 4 : IfUtility.GetByteSizeFromCurveKeyType(animTarget.QuantizationInfo.key_type)
                     select factor * animTarget.KeyFrames.Count).DefaultIfEmpty().Aggregate((x, y) => x + y);
            Predicate<light_anim_target_targetType> translateLight = x =>
            {
                switch (x)
                {
                    case light_anim_target_targetType.position_x:
                    case light_anim_target_targetType.position_y:
                    case light_anim_target_targetType.position_z:
                    case light_anim_target_targetType.aim_x:
                    case light_anim_target_targetType.aim_y:
                    case light_anim_target_targetType.aim_z:
                    case light_anim_target_targetType.dist_attn_start:
                    case light_anim_target_targetType.dist_attn_end:
                        return true;
                }
                return false;
            };
            size += (from anim in target.LightAnims
                     from animTarget in anim.LightAnimTargets
                     where translateLight(animTarget.targetType)
                     let factor = isBase ? 4 : IfUtility.GetByteSizeFromCurveKeyType(animTarget.QuantizationInfo.key_type)
                     select factor * animTarget.KeyFrames.Count).DefaultIfEmpty().Aggregate((x, y) => x + y);
            size += (from anim in target.FogAnims
                     from animTarget in anim.FogAnimTargets
                     let factor = isBase ? 4 : IfUtility.GetByteSizeFromCurveKeyType(animTarget.QuantizationInfo.key_type)
                     select factor * animTarget.KeyFrames.Count).DefaultIfEmpty().Aggregate((x, y) => x + y);
            return size;
        }

        #region コマンド

        public static ICommand CreateEditCommand_quantize(
            GuiObjectGroup targets,
            GuiObjectID id,
            float	tex_rotate,
            float	tex_translate,
            bool reload
        )
        {
            try
            {
                var quantizeData = new List<SceneAnimation.QuantizeToleranceData>();
                foreach (Document targetObject in targets.GetObjects(id))
                {
                    quantizeData.Add(new SceneAnimation.QuantizeToleranceData()
                    {
                        QuantizeToleranceTextureRotate = tex_rotate,
                        QuantizeToleranceTextureTranslate = tex_translate
                    });
                }

                EditCommandSet commandSet = new EditCommandSet();
                commandSet.Add(new GeneralGroupReferenceEditCommand<SceneAnimation.QuantizeToleranceData>(
                        targets,
                        targets.Active.ObjectID,
                        quantizeData,
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is SceneAnimation);
                            var anim	= target as SceneAnimation;

                            swap =
                                new SceneAnimation.QuantizeToleranceData()
                                {
                                    ////
                                    QuantizeToleranceTextureRotate = anim.Data.scene_anim_info.quantize_tolerance_rotate,
                                    QuantizeToleranceTextureTranslate = anim.Data.scene_anim_info.quantize_tolerance_translate
                                };

                            var qtd = data as SceneAnimation.QuantizeToleranceData;
                            ////
                            anim.Data.scene_anim_info.quantize_tolerance_rotate = qtd.QuantizeToleranceTextureRotate;
                            anim.Data.scene_anim_info.quantize_tolerance_translate	= qtd.QuantizeToleranceTextureTranslate;
                        },
                        postEditDelegate : (editTargets, data) =>
                        {
                            ;	// リロードはまとめて送れないのでEditDelegateで送る
                        }
                    ));
                var anims = targets.GetObjects(id).OfType<SceneAnimation>().ToArray();
                foreach (var tmp in anims)
                {
                    var targetObject = tmp;
                    commandSet.Add(new LazyCommand(() => AnimationCurveEditCommand.CreateQuantizeAnalyseAllCommand(targetObject, false)));
                }

                if (reload)
                {
                    commandSet.OnPostEdit += (s, args)
                        =>
                    {
                        foreach (var anim in anims)
                        {
                            Viewer.LoadOrReloadAnimation.Send(anim);
                        }
                    };
                }
                return commandSet;
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : SceneAnimationGeneralPage.CreateEditCommand_quantize() : {0}", e.Message);

                return null;
            }
        }

        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            public float	quantize_tolerance_tex_rotate		{ get; set; }
            public float	quantize_tolerance_tex_translate	{ get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(SceneAnimation target)
        {
            return
                new CopyData()
                {
                    quantize_tolerance_tex_rotate = target.Data.scene_anim_info.quantize_tolerance_rotate,
                    quantize_tolerance_tex_translate = target.Data.scene_anim_info.quantize_tolerance_translate
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                var command =
                    CreateEditCommand_quantize(
                        targets,
                        targets.Active.ObjectID,
                        //
                        copyData.quantize_tolerance_tex_rotate,
                        copyData.quantize_tolerance_tex_translate,
                        true
                    );

                if (command != null)
                {
                    commandSet.Add(command);
                }
            }

            return commandSet.Execute();
        }
        #endregion

        #region イベント
        private void fepQuantizeFrame_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if(e.Changing)
            {
                try
                {
                    Document tempDoc = null;
                    using (var currentTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.ShaderParamAnimBinaryExtension))
                    {
                        // 実質一つだけのとき
                        (new DocumentSaver()).WriteDocument(ActiveTarget, currentTempFileName.Path, false);
                        {
                            bool isSuccess = DocumentManager.Read(new DocumentManager.PathWithName(currentTempFileName.Path, string.Empty), out tempDoc);
                            Debug.Assert(isSuccess && (tempDoc is SceneAnimation));
                        }
                    }

                    // テンポラリに出力したドキュメントの実体
                    var target = tempDoc as SceneAnimation;

                    // 最適化
                    target.Data.scene_anim_info.quantize_tolerance_rotate = fepQuantizeTexRotate.Value;
                    target.Data.scene_anim_info.quantize_tolerance_translate = fepQuantizeTexTranslate.Value;
                    target.Optimize();
                    UpdateCompress(target);
                }
                catch
                {
                }
            }
            else
            {
                var command =
                    CreateEditCommand_quantize(
                        Targets,
                        ActiveTarget.ObjectID,
                        //
                        fepQuantizeTexRotate.Value,
                        fepQuantizeTexTranslate.Value,
                        true
                    );

                if (command != null)
                {
                    TheApp.CommandManager.Execute(command);
                }
            }
        }

        private void fepQuantizeTexRotate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void fepQuantizeTexTranslate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }
        #endregion


    }
}
