﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Controls;
using App.Data;
using App.Utility;

namespace App.PropertyEdit
{
    public partial class ObjectPropertyPage : UIUserControl
    {
        // オーナー
        private ObjectPropertyPanel _owner = null;

        // デフォルトページサイズ
        private Size _defaultPageSize = Size.Empty;

        /// <summary>
        /// デフォルトページサイズ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public virtual Size DefaultPageSize
        {
            get { return _defaultPageSize; }
        }

        /// <summary>
        /// デフォルトページサイズ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public virtual bool ChangeDockStyle
        {
            get { return true; }
        }

        /// <summary>
        /// ページＩＤ。
        /// </summary>
        public readonly PropertyPageID PageID;

        /// <summary>
        /// オーナー。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ObjectPropertyPanel Owner
        {
            get { return _owner; }
            set { _owner = value; }
        }

        /// <summary>
        /// ターゲット。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public GuiObjectGroup Targets
        {
            get { return _owner.Targets; }
        }

        /// <summary>
        /// 親のスクロールバーを表示させるか(false の場合はthis の AutoScroll が false の場合だけ表示する)
        /// </summary>
        public bool ShowParentScrollBar
        {
            get { return showParentScrollBar_; }
            set { showParentScrollBar_ = value; }
        }
        private bool showParentScrollBar_ = true;

        /// <summary>
        /// 親のスクロールバーを表示しない(this の AutoScroll に関わらず表示しない)
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool HideParentScrollBar { get; set; }

        // デザイナ用
        public ObjectPropertyPage()
        {
            InitializeComponent();
        }

        public ObjectPropertyPage(PropertyPageID propertyPageID)
        {
            InitializeComponent();

            PageID = propertyPageID;
        }

        //---------------------------------------------------------------------------
        // フォーム初期化
        //---------------------------------------------------------------------------
        /// <summary>
        /// フォーム初期化。
        /// </summary>
        public void InitializeForm()
        {
            DebugConsole.WriteLine("initialize property page[{0}]", GetType().Name);

            // デフォルトページサイズを初期化
            _defaultPageSize = Size;

            // フォーム初期化
            InitializeFormInternal();
        }

        /// <summary>
        /// フォーム初期化（内部処理用）。
        /// </summary>
        protected virtual void InitializeFormInternal()
        {
            // 派生クラスで実装する
            // 抽象にしないのはフォームデザイナでインスタンス化させる為
            Debug.Assert(false);
        }

        //---------------------------------------------------------------------------
        // フォーム更新
        //---------------------------------------------------------------------------
        public class UpdateFormInfo
        {
            public bool TargetOrPageChanged{ get; set; }
            public bool IsPageCreated{ get; set; }
        }

        public virtual PropertyCategoryNode CategoryNode
        {
            get;
            set;
        }

        /// <summary>
        /// ページがアクティブになった後の処理
        /// </summary>
        public virtual void AfterPageActiveted(ObjectPropertyPage oldPage)
        {
        }

        public virtual void BeforePageDeactivated()
        {
        }
        /// <summary>
        /// フォーム更新。
        /// </summary>
        public void UpdateForm(bool targetOrPageChanged, bool isPageCreated)
        {
            DebugConsole.WriteLine("update property page:[{0}]", GetType().Name);

            // フォーム更新
            using (var block = new UIControlEventSuppressBlock())
            {
                UpdateFormInternal(
                    new UpdateFormInfo()
                    {
                        TargetOrPageChanged = targetOrPageChanged,
                        IsPageCreated = isPageCreated,
                    }
                );
            }
        }

        public void UpdateFormOnPageCreated()
        {
            // フォーム更新
            using (var block = new UIControlEventSuppressBlock())
            {
                UpdateFormOnPageCreatedInternal();
            }
        }

        /// <summary>
        /// フォーム更新（内部処理用）。
        /// </summary>
        protected virtual void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            // 派生クラスで実装する
            // 抽象にしないのはフォームデザイナでインスタンス化させる為
            Debug.Assert(false);
        }

        /// <summary>
        /// フォーム更新（内部処理用）。
        /// </summary>
        protected virtual void UpdateFormOnPageCreatedInternal()
        {
            ;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnVisibleChanged(EventArgs e)
        {
            if (DesignMode) { return; }

            base.OnVisibleChanged(e);
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public virtual bool CanCopy() { return false; }

        /// <summary>
        /// コピー。
        /// </summary>
        public virtual object Copy(ref object copyObjectInfo) { return null; }

        /// <summary>
        /// ペーストが有効か。
        /// </summary>
        public virtual bool CanPaste(object copiedObjectInfo, object copiedObject){ return true; }

        /// <summary>
        /// ペースト。
        /// </summary>
        public virtual void Paste(object pasteObject) { Debug.Assert(false); }
        #endregion

        // ヘルプのパスの取得
        public virtual string HelpPath
        {
            get
            {
                return null;
            }
        }

        public virtual HelpUtility.PageKey HelpKey
        {
            get
            {
                return HelpUtility.PageKey.p_3deditor;
            }
        }

        protected enum QuantizationResultRow
        {
            Scale,
            Rotate,
            Translate,
            Total,
        }

        protected enum QuantizationResultColumn
        {
            Name,
            UncompressDataSize,
            CompressDataSize,
            ReductionRate,
        }

        protected void SetQuantizationResult(UIListView listView, QuantizationResultRow row, QuantizationResultColumn column, string text)
        {
            listView.Items[(int)row].SubItems[(int)column].Text = text;
        }

        protected void SetQuantizationResult(UIListView listView, QuantizationResultRow row, int uncompressDataSize, int compressDataSize)
        {
            SetQuantizationResult(listView, row, QuantizationResultColumn.UncompressDataSize, uncompressDataSize.ToString());
            SetQuantizationResult(listView, row, QuantizationResultColumn.CompressDataSize, compressDataSize.ToString());

            if (uncompressDataSize > 0)
            {
                float reductionRate = (float)compressDataSize / uncompressDataSize;
                reductionRate = (1 - reductionRate) * 100;
                SetQuantizationResult(listView, row, QuantizationResultColumn.ReductionRate, reductionRate.ToString("0.00"));
            }
            else
            {
                SetQuantizationResult(listView, row, QuantizationResultColumn.ReductionRate, "");
            }
        }

        // 量子化による削減結果を更新する
        protected void UpdateQuantizationResult(UIListView listView,
                                                int scaleUncompressDataSize, int scaleCompressDataSize,
                                                int rotateUncompressDataSize, int rotateCompressDataSize,
                                                int translateUncompressDataSize, int translateCompressDataSize)
        {
            // スケール・回転・移動
            SetQuantizationResult(listView, QuantizationResultRow.Scale, scaleUncompressDataSize, scaleCompressDataSize);
            SetQuantizationResult(listView, QuantizationResultRow.Rotate, rotateUncompressDataSize, rotateCompressDataSize);
            SetQuantizationResult(listView, QuantizationResultRow.Translate, translateUncompressDataSize, translateCompressDataSize);

            // 全体
            int totalUncompressDataSize = scaleUncompressDataSize;
            totalUncompressDataSize += rotateUncompressDataSize;
            totalUncompressDataSize += translateUncompressDataSize;
            int totalCompressDataSize = scaleCompressDataSize;
            totalCompressDataSize += rotateCompressDataSize;
            totalCompressDataSize += translateCompressDataSize;
            SetQuantizationResult(listView, QuantizationResultRow.Total, totalUncompressDataSize, totalCompressDataSize);
        }
    }

    #region ModelPropertyPage
    /// <summary>
    /// モデルプロパティページクラス。
    /// </summary>
    public class ModelPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected ModelPropertyPage() {}

        // コンストラクタ。
        protected ModelPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Model ActiveTarget
        {
            get { return (Model)base.Targets.Active; }
        }
    }
    #endregion

    #region TexturePropertyPage
    /// <summary>
    /// テクスチャプロパティページクラス。
    /// </summary>
    public class TexturePropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected TexturePropertyPage() {}

        // コンストラクタ。
        protected TexturePropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Texture ActiveTarget
        {
            get { return (Texture)base.Targets.Active; }
        }
    }
    #endregion

    #region MaterialPropertyPage
    /// <summary>
    /// マテリアルプロパティページクラス。
    /// </summary>
    public class MaterialPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected MaterialPropertyPage() {}

        // コンストラクタ。
        protected MaterialPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Material ActiveTarget
        {
            get { return (Material)base.Targets.Active; }
        }
    }
    #endregion

    #region ShapePropertyPage
    /// <summary>
    /// シェイププロパティページクラス。
    /// </summary>
    public class ShapePropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected ShapePropertyPage() { }

        // コンストラクタ。
        protected ShapePropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Shape ActiveTarget
        {
            get { return (Shape)base.Targets.Active; }
        }
    }
    #endregion

    #region SkeletalAnimationPropertyPage
    /// <summary>
    /// スケルタルアニメーションプロパティページクラス。
    /// </summary>
    public class SkeletalAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected SkeletalAnimationPropertyPage() {}

        // コンストラクタ。
        protected SkeletalAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public SkeletalAnimation ActiveTarget
        {
            get { return (SkeletalAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region ShapeAnimationPropertyPage
    /// <summary>
    /// シェイプアニメーションプロパティページクラス。
    /// </summary>
    public class ShapeAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected ShapeAnimationPropertyPage() {}

        // コンストラクタ。
        protected ShapeAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ShapeAnimation ActiveTarget
        {
            get { return (ShapeAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region MaterialAnimationPropertyPage
    /// <summary>
    /// シェーダーパラメーターアニメーションプロパティページクラス。
    /// </summary>
    public class MaterialAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected MaterialAnimationPropertyPage() { }

        // コンストラクタ。
        protected MaterialAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public MaterialAnimation ActiveTarget
        {
            get { return (MaterialAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region ShaderParameterAnimationPropertyPage
    /// <summary>
    /// シェーダーパラメーターアニメーションプロパティページクラス。
    /// </summary>
    public class ShaderParameterAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected ShaderParameterAnimationPropertyPage() {}

        // コンストラクタ。
        protected ShaderParameterAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ShaderParameterAnimation ActiveTarget
        {
            get { return (ShaderParameterAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region MaterialVisibilityAnimationPropertyPage
    /// <summary>
    /// マテリアルビジビリティアニメーションプロパティページクラス。
    /// </summary>
    public class MaterialVisibilityAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected MaterialVisibilityAnimationPropertyPage() {}

        // コンストラクタ。
        protected MaterialVisibilityAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public MaterialVisibilityAnimation ActiveTarget
        {
            get { return (MaterialVisibilityAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region BoneVisibilityAnimationPropertyPage
    /// <summary>
    /// ボーンビジビリティアニメーションプロパティページクラス。
    /// </summary>
    public class BoneVisibilityAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected BoneVisibilityAnimationPropertyPage() {}

        // コンストラクタ。
        protected BoneVisibilityAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public BoneVisibilityAnimation ActiveTarget
        {
            get { return (BoneVisibilityAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region TexturePatternAnimationPropertyPage
    /// <summary>
    /// テクスチャパターンアニメーションプロパティページクラス。
    /// </summary>
    public class TexturePatternAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected TexturePatternAnimationPropertyPage() {}

        // コンストラクタ。
        protected TexturePatternAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public TexturePatternAnimation ActiveTarget
        {
            get { return (TexturePatternAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region SceneAnimationPropertyPage
    /// <summary>
    /// シーンアニメーションプロパティページクラス。
    /// </summary>
    public class SceneAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected SceneAnimationPropertyPage() { }

        // コンストラクタ。
        protected SceneAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public SceneAnimation ActiveTarget
        {
            get { return (SceneAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region CameraAnimationPropertyPage
    /// <summary>
    /// マテリアルビジビリティアニメーションプロパティページクラス。
    /// </summary>
    public class CameraAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected CameraAnimationPropertyPage() { }

        // コンストラクタ。
        protected CameraAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public CameraAnimation ActiveTarget
        {
            get { return (CameraAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region LightAnimationPropertyPage
    /// <summary>
    /// マテリアルビジビリティアニメーションプロパティページクラス。
    /// </summary>
    public class LightAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected LightAnimationPropertyPage() { }

        // コンストラクタ。
        protected LightAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public LightAnimation ActiveTarget
        {
            get { return (LightAnimation)base.Targets.Active; }
        }
    }
    #endregion

    #region FogAnimationPropertyPage
    /// <summary>
    /// マテリアルビジビリティアニメーションプロパティページクラス。
    /// </summary>
    public class FogAnimationPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected FogAnimationPropertyPage() { }

        // コンストラクタ。
        protected FogAnimationPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FogAnimation ActiveTarget
        {
            get { return (FogAnimation)base.Targets.Active; }
        }
    }
    #endregion
    #region ShaderDefinitionPropertyPage
    /// <summary>
    /// シェーダー定義プロパティページクラス。
    /// </summary>
    public class ShaderDefinitionPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected ShaderDefinitionPropertyPage() {}

        // コンストラクタ。
        protected ShaderDefinitionPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID){}

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ShaderDefinition ActiveTarget
        {
            get { return (ShaderDefinition)base.Targets.Active; }
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionPageSize.Width),
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionPageSize.Height));
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && (Targets.Active is ShaderDefinition) && !ObjectPropertyDialog.InternallyChangingSize.IsChanging &&
                Owner.Owner.WindowState == System.Windows.Forms.FormWindowState.Normal)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.ShaderDefinitionPageSize.Height = Height;
            }

            base.OnSizeChanged(e);
        }

        public override bool ChangeDockStyle
        {
            get
            {
                return false;
            }
        }
    }
    #endregion

    #region BonePropertyPage
    /// <summary>
    /// モデルプロパティページクラス。
    /// </summary>
    public class BonePropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected BonePropertyPage() { }

        // コンストラクタ。
        protected BonePropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Bone ActiveTarget
        {
            get { return (Bone)base.Targets.Active; }
        }
    }
    #endregion

    #region ProjectPropertyPage
    /// <summary>
    /// プロジェクトプロパティページクラス。
    /// </summary>
    public class ProjectPropertyPage : ObjectPropertyPage
    {
        // コンストラクタ。デザイナ用
        protected ProjectPropertyPage() { }

        // コンストラクタ。
        protected ProjectPropertyPage(PropertyPageID propertyPageID) :
            base(propertyPageID) { }

        // アクティブターゲット。
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Model ActiveTarget
        {
            get { return (Model)base.Targets.Active; }
        }
    }
    #endregion
}
