﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class MaterialAnimationGeneralPage : MaterialAnimationPropertyPage
    {
        public MaterialAnimationGeneralPage() :
            base(PropertyPageID.MaterialAnimationGeneral)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_animation_property_window_general_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new MaterialAnimationGeneralPage();
        }

        protected override void InitializeFormInternal()
        {
            iepFrameSize.SequentialValueChanged +=
                (s, e) =>
                {
                    if (!e.Changing)
                    {
                        TheApp.CommandManager.Execute(CreateEditCommand_frame_count(Targets, iepFrameSize.Value));
                    }
                };
            cbxLoop.CheckedChanged +=
                (s, e) => TheApp.CommandManager.Execute(CreateEditCommand_loop(Targets, cbxLoop.Checked));
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            var target = ActiveTarget;

            if (target.ObjectID == GuiObjectID.ColorAnimation ||
                ApplicationConfig.DefaultValue.DisableAnimationQuantize)
            {
                lblFrameLoop.Visible = true;
                lblFrameLoopQuantize.Visible = false;
            }
            else
            {
                lblFrameLoop.Visible = false;
                lblFrameLoopQuantize.Visible = true;
            }

            var info = target.Data.material_anim_info;
            lblFrameSize.IsModified = target.IsValueChanged(x => x.frame_count);
            iepFrameSize.Value = info.frame_count;

            cbxLoop.IsModified = target.IsValueChanged(x => x.loop);
            cbxLoop.Checked = info.loop;

            lblFrameResolution.IsModified = target.IsValueChanged(x => x.frame_resolution);
            ltbFrameResolution.Text	= target.Data.material_anim_info.frame_resolution.ToString();

            lblBakeAll.IsModified = target.IsValueChanged(x => x.bake_all);
            ltbBakeAll.Text			= UIText.FlagYesNo(target.Data.material_anim_info.bake_all);

            lblDccPreset.IsModified = target.IsStringChanged(x => x.dcc_preset);
            ltbDccPreset.Text = target.Data.material_anim_info.dcc_preset;

            lblDccStartFrame.IsModified = target.IsValueChanged(x => x.dcc_start_frame);
            ltbDccStartFrame.Text = target.Data.material_anim_info.dcc_start_frame.ToString();

            lblDccEndFrame.IsModified = target.IsValueChanged(x => x.dcc_end_frame);
            ltbDccEndFrame.Text = target.Data.material_anim_info.dcc_end_frame.ToString();

            lblDccFps.IsModified = target.IsValueChanged(x => x.dcc_fps);
            ltbDccFps.Text = target.Data.material_anim_info.dcc_fps.ToString();

            lblBakeColor.IsModified = target.IsValueChanged(x => x.bake_tolerance_color);
            ltbBakeColor.Text				= target.Data.material_anim_info.bake_tolerance_color.ToString();

            lblBakeTexScale.IsModified = target.IsValueChanged(x => x.bake_tolerance_tex_scale);
            ltbBakeTexScale.Text			= target.Data.material_anim_info.bake_tolerance_tex_scale.ToString();

            lblBakeTexRotate.IsModified = target.IsValueChanged(x => x.bake_tolerance_tex_rotate);
            ltbBakeTexRotate.Text			= target.Data.material_anim_info.bake_tolerance_tex_rotate.ToString();
            ltbBakeTexTranslate.Text		= target.Data.material_anim_info.bake_tolerance_tex_translate.ToString();

            lblQuantizeTexScale.IsModified = target.IsValueChanged(x => x.quantize_tolerance_tex_scale);
            fepQuantizeTexScale.Value       = target.Data.material_anim_info.quantize_tolerance_tex_scale;

            lblQuantizeTexRotate.IsModified = target.IsValueChanged(x => x.quantize_tolerance_tex_rotate);
            fepQuantizeTexRotate.Value= target.Data.material_anim_info.quantize_tolerance_tex_rotate;

            lblQuantizeTexTranslate.IsModified = target.IsValueChanged(x => x.quantize_tolerance_tex_translate);
            fepQuantizeTexTranslate.Value       = target.Data.material_anim_info.quantize_tolerance_tex_translate;

            UpdateControlEnabled();

            UpdateCompress(target);
        }

        public static bool IsModified(MaterialAnimation target)
        {
            return target != null &&
                (target.IsValueChanged(x => x.frame_count) ||
                target.IsValueChanged(x => x.frame_resolution) ||
                target.IsValueChanged(x => x.bake_all) ||
                target.IsValueChanged(x => x.loop) ||
                target.IsStringChanged(x => x.dcc_preset) ||
                target.IsValueChanged(x => x.dcc_start_frame) ||
                target.IsValueChanged(x => x.dcc_end_frame) ||
                target.IsValueChanged(x => x.dcc_fps) ||
                target.IsValueChanged(x => x.bake_tolerance_color) ||
                target.IsValueChanged(x => x.bake_tolerance_tex_scale) ||
                target.IsValueChanged(x => x.bake_tolerance_tex_rotate) ||
                target.IsValueChanged(x => x.quantize_tolerance_tex_scale) ||
                target.IsValueChanged(x => x.quantize_tolerance_tex_rotate) ||
                target.IsValueChanged(x => x.quantize_tolerance_tex_translate));
        }

        private void UpdateCompress(MaterialAnimation target)
        {
            UpdateQuantizationResult(lstQuantizationResult,
                target.TextureScaleUncompressDataSize, target.TextureScaleCompressDataSize,
                target.TextureRotateUncompressDataSize, target.TextureRotateCompressDataSize,
                target.TextureTranslateUncompressDataSize, target.TextureTranslateCompressDataSize);
        }

        private void UpdateControlEnabled()
        {
            var id = ActiveTarget.ObjectID;
            bool isEnabledColor  = (id == GuiObjectID.MaterialAnimation) || (id == GuiObjectID.ColorAnimation);
            bool isEnabledTexSrt = (id == GuiObjectID.MaterialAnimation) || (id == GuiObjectID.TextureSrtAnimation);

            ltbBakeColor.Enabled			= isEnabledColor;
            ltbBakeTexScale.Enabled			= isEnabledTexSrt;
            ltbBakeTexRotate.Enabled		= isEnabledTexSrt;
            ltbBakeTexTranslate.Enabled		= isEnabledTexSrt;

            fepQuantizeTexScale.Enabled		= isEnabledTexSrt;
            fepQuantizeTexRotate.Enabled	= isEnabledTexSrt;
            fepQuantizeTexTranslate.Enabled	= isEnabledTexSrt;

            lblBakeColor.Enabled			= isEnabledColor;
            lblBakeTex.Enabled				= isEnabledTexSrt;
            lblBakeTexScale.Enabled			= isEnabledTexSrt;
            lblBakeTexRotate.Enabled		= isEnabledTexSrt;
            lblBakeTexTranslate.Enabled		= isEnabledTexSrt;

            lblQuantizeTex.Enabled			= isEnabledTexSrt;
            lblQuantizeTexScale.Enabled		= isEnabledTexSrt;
            lblQuantizeTexRotate.Enabled	= isEnabledTexSrt;
            lblQuantizeTexTranslate.Enabled	= isEnabledTexSrt;

            if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
            {
                gbxQuantize.Visible = false;
                gbxQuantizationReductionRate.Visible = false;
            }
        }

        #region コマンド
        public static GroupEditCommand CreateEditCommand_frame_count(GuiObjectGroup targets, int FrameCount)
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    targets,
                    targets.Active.ObjectID,
                    FrameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as MaterialAnimation;

                        swap = anim.Data.material_anim_info.frame_count;
                        anim.Data.material_anim_info.frame_count = (int)data;
                        AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);

                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                );
        }

        public static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    targets.Active.ObjectID,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as MaterialAnimation;

                        swap = anim.Data.material_anim_info.loop;
                        anim.Data.material_anim_info.loop = (bool)data;
                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                );
        }

#if false
        public static EditCommand CreateEditCommand_quantize(
            GuiObjectGroup targets,
            GuiObjectID id,
            float	tex_scale,
            float	tex_rotate,
            float	tex_translate,
            bool	isLoadAnimation
        )
        {

            var quantizeData = new List<MaterialAnimation.QuantizeToleranceData>();
            foreach (MaterialAnimation targetObject in targets.GetObjects(id))
            {
                quantizeData.Add(new MaterialAnimation.QuantizeToleranceData()
                {
                    QuantizeToleranceTextureScale = tex_scale,
                    QuantizeToleranceTextureRotate = tex_rotate,
                    QuantizeToleranceTextureTranslate = tex_translate
                });

            }

            //return
            var commandSet = new EditCommandSet();
            commandSet.Add(new GeneralGroupReferenceEditCommand<MaterialAnimation.QuantizeToleranceData>(
                    targets,
                    targets.Active.ObjectID,
                    quantizeData,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is MaterialAnimation);
                        var anim	= target as MaterialAnimation;

                        swap =
                            new MaterialAnimation.QuantizeToleranceData()
                            {
                                PerMaterialAnims	= anim.PerMaterialAnims,
                                //
                                QuantizeToleranceTextureScale		= anim.Data.material_anim_info.quantize_tolerance_tex_scale,
                                QuantizeToleranceTextureRotate		= anim.Data.material_anim_info.quantize_tolerance_tex_rotate,
                                QuantizeToleranceTextureTranslate	= anim.Data.material_anim_info.quantize_tolerance_tex_translate
                            };

                        var qtd = data as MaterialAnimation.QuantizeToleranceData;

                        anim.Data.material_anim_info.quantize_tolerance_tex_scale		= qtd.QuantizeToleranceTextureScale;
                        anim.Data.material_anim_info.quantize_tolerance_tex_rotate		= qtd.QuantizeToleranceTextureRotate;
                        anim.Data.material_anim_info.quantize_tolerance_tex_translate	= qtd.QuantizeToleranceTextureTranslate;
                        if (qtd.PerMaterialAnims == null)
                        {
                            anim.PerMaterialAnims = ObjectUtility.Clone(anim.PerMaterialAnims);
                            anim.Optimize();
                        }
                        else
                        {
                            anim.PerMaterialAnims = qtd.PerMaterialAnims;
                        }

                        anim.UpdateIsModifiedAnimTargetAll();

                        if (isLoadAnimation)
                        {
                            Viewer.LoadAnimation2.Send(anim);
                        }
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                    }
                ));
            foreach (MaterialAnimation tmp in targets.GetObjects(id))
            {
                // スコープを制限
                var targetObject = tmp;
                commandSet.Add(new LazyCommand(() => targetObject.CreateUpdateBindCommand()));
            }

            return commandSet;
        }
#endif
        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            public int frame_count { get; set; }
            public bool loop { get; set; }
            public float	quantize_tolerance_tex_scale		{ get; set; }
            public float	quantize_tolerance_tex_rotate		{ get; set; }
            public float	quantize_tolerance_tex_translate	{ get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }


        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(MaterialAnimation target)
        {
            return
                new CopyData()
                {
                    frame_count = target.Data.material_anim_info.frame_count,
                    loop = target.Data.material_anim_info.loop,
                    quantize_tolerance_tex_scale		= target.Data.material_anim_info.quantize_tolerance_tex_scale,
                    quantize_tolerance_tex_rotate		= target.Data.material_anim_info.quantize_tolerance_tex_rotate,
                    quantize_tolerance_tex_translate	= target.Data.material_anim_info.quantize_tolerance_tex_translate
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_frame_count(targets, copyData.frame_count));
                commandSet.Add(CreateEditCommand_loop(targets, copyData.loop));
                var id = targets.Active.ObjectID;
                //if ((id == GuiObjectID.MaterialAnimation) || (id == GuiObjectID.TextureSrtAnimation))
                {
                    commandSet.Add(
                        MaterialAnimationSettingPopup.CreateEditCommand_quantize(
                            targets,
                            targets.Active.ObjectID,
                        //
                            copyData.quantize_tolerance_tex_scale,
                            copyData.quantize_tolerance_tex_rotate,
                            copyData.quantize_tolerance_tex_translate,
                            true
                        )
                    );
                }
            }

            return commandSet.Execute();
        }
        #endregion

        #region イベント
        private void iepFrameSize_SequentialValueChanged(object sender, Controls.SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {

            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_frame_count(Targets, iepFrameSize.Value));
            }

        }

        private void fepQuantizeFrame_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if(e.Changing)
            {
                Document tempDoc = null;
                using (var currentTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.ShaderParamAnimBinaryExtension))
                {
                    // 実質一つだけのとき
                    (new DocumentSaver()).WriteDocument(ActiveTarget, currentTempFileName.Path, false);
                    {
                        bool isSuccess = DocumentManager.Read(new DocumentManager.PathWithName(currentTempFileName.Path, string.Empty), out tempDoc);
                        Debug.Assert(isSuccess && (tempDoc is MaterialAnimation));
                    }
                }

                // テンポラリに出力したドキュメントの実体
                var target = tempDoc as MaterialAnimation;

                // 最適化
                target.Data.material_anim_info.quantize_tolerance_tex_scale = fepQuantizeTexScale.Value;
                target.Data.material_anim_info.quantize_tolerance_tex_rotate = fepQuantizeTexRotate.Value;
                target.Data.material_anim_info.quantize_tolerance_tex_translate = fepQuantizeTexTranslate.Value;
                target.OptimizeParamAnims();
                UpdateCompress(target);
            }
            else
            {
                TheApp.CommandManager.Execute(
                    MaterialAnimationSettingPopup.CreateEditCommand_quantize(
                        Targets,
                        ActiveTarget.ObjectID,
                        fepQuantizeTexScale.Value,
                        fepQuantizeTexRotate.Value,
                        fepQuantizeTexTranslate.Value,
                        true
                    )
                );
            }
        }

        private void fepQuantizeTexScale_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void fepQuantizeTexRotate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void fepQuantizeTexTranslate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }
        #endregion
    }
}
