﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;
using System.Text.RegularExpressions;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public class LabelHelper
    {

        public string paramName;
        public string paramLabel;

        private static Dictionary<string, string> dummy = new Dictionary<string, string>();
        public string LinkLabel;
        public string CustomLinkLabel;
        public bool LinkError;
        public string LinkGroup;

        public Color? color;
        public Color? backGroundColor;
#region 属性指定
        // 文字列だけ
        public string paramLabelPlane;
        public ParamType linkTargetType;
        public string linkTarget;
        public static Regex regex = new Regex(@":(ref|color|background-color){([^}]*)}\s*");
        public void UpdateLabelAttribute()
        {
            paramLabelPlane = paramLabel;
            color = null;
            backGroundColor = null;
            bool success = false;
            if (!string.IsNullOrEmpty(paramLabel))
            {
                var ms = regex.Matches(paramLabel);
                if (ms.Count > 0)
                {
                    success = true;
                    int lastIndex = ms[0].Index;

                    foreach (Match m in ms)
                    {
                        if (m.Index != lastIndex)
                        {
                            success = false;
                            break;
                        }

                        if (m.Groups.Count == 3)
                        {
                            if (m.Groups[1].Value == "ref")
                            {
                                var value = m.Groups[2].Value;
                                var pos = value.IndexOf('/');
                                if (pos < 0)
                                {
                                    success = false;
                                    break;
                                }
                                if (Enum.TryParse(value.Substring(0, pos), out linkTargetType))
                                {
                                    linkTarget = value.Substring(pos + 1);
                                }
                                else
                                {
                                    success = false;
                                    break;
                                }
                            }
                            else if (m.Groups[1].Value == "color")
                            {
                                var value = m.Groups[2].Value;
                                try
                                {
                                    color = ColorTranslator.FromHtml(value);
                                }
                                catch
                                {
                                    success = false;
                                }
                            }
                            else if (m.Groups[1].Value == "background-color")
                            {
                                var value = m.Groups[2].Value;
                                try
                                {
                                    backGroundColor = ColorTranslator.FromHtml(value);
                                }
                                catch
                                {
                                    success = false;
                                }
                            }
                        }
                        else
                        {
                            Debug.Assert(false);
                            success = false;
                            break;
                        }
                        lastIndex = m.Index + m.Length;
                    }

                    if (success && lastIndex == paramLabel.Length)
                    {
                        paramLabelPlane = paramLabel.Substring(0, ms[0].Index);
                    }
                }
            }

            if (!success)
            {
                paramLabelPlane = paramLabel;
            }
        }

        public static string StripAttributes(string label)
        {
            var ms = regex.Matches(label);
            if (ms.Count > 0)
            {
                int lastIndex = ms[0].Index;
                foreach (Match m in ms)
                {
                    if (m.Index != lastIndex)
                    {
                        return label;
                    }
                    lastIndex = m.Index + m.Length;
                }

                return label.Substring(0, ms[0].Index);
            }

            return label;
        }

        public LinkClickedEventArgs ClickArgs()
        {
            return new LinkClickedEventArgs()
            {
                Group = LinkGroup,
                ParamType = linkTargetType,
                Target = linkTarget,
            };
        }
#endregion
        public string customLabel;

        public bool showId;

        public bool showOriginalLabel;

        private bool IsParentLabel = false;

        public string CustumLabelForEdit => IsParentLabel ? string.Empty : customLabel;

        public string GetLabelString(bool colon = true)
        {
            // カスタムラベルがあるとき
            if (!string.IsNullOrEmpty(customLabel))
            {
                if (showId)
                {
                    if (string.IsNullOrEmpty(paramLabel) || !showOriginalLabel)
                    {
                        return string.Format("{0} ({1}){2}", customLabel, paramName, colon ? ":" : "");
                    }
                    else
                    {
                        return string.Format("{0} ({1}|{2}){3}", customLabel, paramName, paramLabel, colon ? ":" : "");
                    }
                }
                else
                {
                    if (!showOriginalLabel)
                    {
                        return customLabel + (colon ? " :" : "");
                    }
                    else
                    {
                        if (!string.IsNullOrEmpty(paramLabelPlane))
                        {
                            return string.Format("{0} ({1}){2}", customLabel, paramLabelPlane, colon ? ":" : "");
                        }
                        else
                        {
                            return string.Format("{0} ({1}){2}", customLabel, paramName, colon ? ":" : "");
                        }
                    }
                }
            }

            // ラベルがあるとき
            if (!string.IsNullOrEmpty(paramLabelPlane))
            {
                if (showId)
                {
                    if (paramLabel == paramLabelPlane)
                    {
                        return string.Format("{0} ({1}){2}", paramLabelPlane, paramName, colon?":":"");
                    }
                    else
                    {
                        return string.Format("{0} ({1}|{2}){3}", paramLabelPlane, paramName, paramLabel, colon?":":"");
                    }
                }
                else
                {
                    return paramLabelPlane + (colon ? " :" : "");
                }
            }

            //if (!LinkError)
            {
                // カスタムリンクラベルがあるとき
                if (!string.IsNullOrEmpty(CustomLinkLabel))
                {
                    if (showId)
                    {
                        if (string.IsNullOrEmpty(LinkLabel) || !showOriginalLabel)
                        {
                            if (string.IsNullOrEmpty(paramLabel))
                            {
                                return string.Format("{0} ({1}){2}", CustomLinkLabel, paramName, colon ? ":" : "");
                            }
                            else
                            {
                                return string.Format("{0} ({1}|{2}){3}", CustomLinkLabel, paramName, paramLabel, colon ? ":" : "");
                            }
                        }
                        else
                        {
                            if (string.IsNullOrEmpty(paramLabel))
                            {
                                return string.Format("{0} ({1}/{2}){3}", CustomLinkLabel, LinkLabel, paramName, colon ? ":" : "");
                            }
                            else
                            {
                                return string.Format("{0} ({1}/{2}|{3}){4}", CustomLinkLabel, LinkLabel, paramName, paramLabel, colon ? ":" : "");
                            }
                        }
                    }
                    else
                    {
                        if (!showOriginalLabel)
                        {
                            return CustomLinkLabel + (colon ? " :" : "");
                        }
                        else
                        {
                            if (!string.IsNullOrEmpty(LinkLabel))
                            {
                                var stripped = StripAttributes(LinkLabel);
                                if (!string.IsNullOrEmpty(stripped))
                                {
                                    return string.Format("{0} ({1}){2}", CustomLinkLabel, stripped, colon ? ":" : "");
                                }
                            }

                            Debug.Assert(!string.IsNullOrEmpty(linkTarget));
                            return string.Format("{0} ({1}){2}", CustomLinkLabel, linkTarget, colon ? ":" : "");
                        }
                    }
                }

                // リンクラベルがあるとき
                if (!string.IsNullOrEmpty(LinkLabel))
                {
                    // アトリビュートを除く
                    var stripped = StripAttributes(LinkLabel);
                    if (!string.IsNullOrEmpty(stripped))
                    {
                        if (showId)
                        {
                            if (string.IsNullOrEmpty(paramLabel))
                            {
                                return string.Format("{0} ({1}){2}", stripped, paramName, colon ? ":" : "");
                            }
                            else
                            {
                                return string.Format("{0} ({1}|{2}){3}", stripped, paramName, paramLabel, colon ? ":" : "");
                            }
                        }
                        else
                        {
                            return stripped + (colon ? " :" : "");
                        }
                    }
                }
            }

            // ラベルの属性があるとき(paramLabelPlane は空なので pramLabel は属性だけ)
            if (!string.IsNullOrEmpty(paramLabel))
            {
                if (showId)
                {
                    return string.Format("{0} ({1}|{2}){3}", paramName, paramName, paramLabel, colon ? ":" : "");
                }
                else
                {
                    return paramName + (colon ? " :" : "");
                }
            }

            // paramName しかないとき
            return paramName + (colon?" :":"");
        }

        public string GetLinkLabelString()
        {
            if (!string.IsNullOrEmpty(CustomLinkLabel))
            {
                return StripAttributes(CustomLinkLabel);
            }

            if (!string.IsNullOrEmpty(LinkLabel))
            {
                // リンクラベルがあるとき
                if (!string.IsNullOrEmpty(LinkLabel))
                {
                    // アトリビュートを除く
                    var stripped = StripAttributes(LinkLabel);
                    if (!string.IsNullOrEmpty(stripped))
                    {
                        return stripped;
                    }
                }
            }

            if (!string.IsNullOrEmpty(linkTarget))
            {
                return linkTarget;
            }

            return string.Empty;
        }

        private string FormatLabelString(string mainLabel, string subLabel, bool colon)
        {
            if (showId)
            {
                if (string.IsNullOrEmpty(subLabel))
                {
                    if (string.IsNullOrEmpty(paramLabel))
                    {
                        return string.Format("{0} ({1}){2}", mainLabel, paramName, colon ? ":" : "");
                    }
                    else
                    {
                        return string.Format("{0} ({1}|{2}){3}", mainLabel, paramName, paramLabel, colon ? ":" : "");
                    }
                }
                else
                {
                    if (string.IsNullOrEmpty(paramLabel))
                    {
                        return string.Format("{0} ({1}/{2}){3}", mainLabel, subLabel, paramName, colon ? ":" : "");
                    }
                    else
                    {
                        return string.Format("{0} ({1}/{2}|{3}){4}", mainLabel, subLabel, paramName, paramLabel, colon ? ":" : "");
                    }
                }
            }
            else
            {
                return mainLabel + (colon?" :":"");
            }
        }

        private static readonly Image linkImage = App.Properties.Resources.PropertyEdit_Link;

        public static LinkButton CreateLinkButton(int i = 0)
        {
            var btnLink = new LinkButton();
            btnLink.Image = linkImage;
            btnLink.Location = new System.Drawing.Point(106, 2);
            btnLink.Name = "btnLink" + i;
            btnLink.Size = new System.Drawing.Size(24, 18);
            btnLink.TabIndex = 0;
            return btnLink;
        }

        public bool SetLabel(
            string ParamName,
            string ParamLabel,
            string label,
            CustomUI customUI,
            Definition.ShadingModelTable table,
            Predicate<string> visibleGroups,
            HashSet<string> visiblePages,
            bool showId,
            bool showOriginalLabel,
            bool isParentLabel =false)
        {
            bool update = false;
            if (paramName != ParamName)
            {
                paramName = ParamName;
                update = true;
            }
            if (paramLabel != ParamLabel)
            {
                paramLabel = ParamLabel;
                UpdateLabelAttribute();
                update = true;
            }

            if (customLabel != label)
            {
                customLabel = label;
                update = true;
            }

            if (IsParentLabel != isParentLabel)
            {
                IsParentLabel = isParentLabel;
                update = true;
            }

            if (this.showId != showId)
            {
                this.showId = showId;
                update = true;
            }

            if (this.showOriginalLabel != showOriginalLabel)
            {
                this.showOriginalLabel = showOriginalLabel;
                update = true;
            }

            string linkLabel = null;
            bool linkError = false;
            string customLinkLabel = null;
            string linkGroup = null;
            if (linkTarget != null)
            {
                switch (linkTargetType)
                {
                    case ShaderParamControls.ParamType.attrib_var:
                        GetReference(linkTarget, table.attribTable, customUI.attrib_vars, ShaderTypeUtility.Label, ShaderTypeUtility.Group,
                            out linkError, out linkLabel, out customLinkLabel, out linkGroup);
                        break;
                    case ShaderParamControls.ParamType.option_var:
                        GetReference(linkTarget, table.optionTable, customUI.option_vars, ShaderTypeUtility.Label, ShaderTypeUtility.Group,
                            out linkError, out linkLabel, out customLinkLabel, out linkGroup);
                        break;
                    case ShaderParamControls.ParamType.sampler_var:
                        GetReference(linkTarget, table.samplerTable, customUI.sampler_vars, ShaderTypeUtility.Label, ShaderTypeUtility.Group,
                            out linkError, out linkLabel, out customLinkLabel, out linkGroup);
                        break;
                    case ShaderParamControls.ParamType.uniform_var:
                        GetReference(linkTarget, table.uniformTable, customUI.uniform_vars, ShaderTypeUtility.Label, ShaderTypeUtility.Group,
                            out linkError, out linkLabel, out customLinkLabel, out linkGroup);
                        break;
                    case ShaderParamControls.ParamType.render_info_slot:
                        GetReference(linkTarget, table.renderInfoTable, customUI.render_info_slots, ShaderTypeUtility.Label, ShaderTypeUtility.Group,
                            out linkError, out linkLabel, out customLinkLabel, out linkGroup);
                        break;
                    case ShaderParamControls.ParamType.group:
                        GetReference(linkTarget, table.groupTable, customUI.groups, ShaderTypeUtility.Label, x => x.name,
                            out linkError, out linkLabel, out customLinkLabel, out linkGroup);
                        break;
                    case ShaderParamControls.ParamType.page:
                        GetReference(linkTarget, table.pageTable, dummy, ShaderTypeUtility.Label, x => null,
                            out linkError, out linkLabel, out customLinkLabel, out linkGroup);
                        linkError |= !visiblePages.Contains(linkTarget);
                        break;
                }
            }

            LinkGroup = linkGroup;
            if (!string.IsNullOrEmpty(LinkGroup))
            {
                linkError |= !visibleGroups(LinkGroup);
            }

            if (CustomLinkLabel != customLinkLabel)
            {
                CustomLinkLabel = customLinkLabel;
                update = true;
            }

            if (LinkLabel != linkLabel)
            {
                LinkLabel = linkLabel;
                update = true;
            }

            if (LinkError != linkError)
            {
                LinkError = linkError;
                update = true;
            }

            return update;
        }

        private void GetReference<T>(
            string linkTarget,
            Dictionary<string, T> table,
            Dictionary<string, string> customTable,
            Func<T, string> label,
            Func<T, string> group,
            out bool linkError,
            out string linkLabel,
            out string customLinkLabel,
            out string linkGroup)
        {
            T item;
            linkLabel = null;
            customLinkLabel = null;
            linkError = false;
            linkGroup = null;
            if (!table.TryGetValue(linkTarget, out item))
            {
                linkError = true;
                return;
            }
            customTable.TryGetValue(linkTarget, out customLinkLabel);
            linkLabel = label(item);
            linkGroup = group(item);
        }

        private static HashSet<string> dummyHashSet = new HashSet<string>();
        public static string GetLabelForFilter(string id, string label, Dictionary<string,string> idToCustomLabel, CustomUI customUI, Definition.ShadingModelTable table)
        {
            if (table == null)
            {
                return id;
            }

            string customLabel;
            idToCustomLabel.TryGetValue(id, out customLabel);

            var helper = new LabelHelper();
            helper.SetLabel(id, label, customLabel, customUI, table, x => true, dummyHashSet, false, false);
            return helper.GetLabelString(false);
        }
    }
}
