﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditRenderInfo : EditRenderInfoBase
    {
        public string[] Value
        {
            get
            {
                return comboBoxes.Select(x => (string)x.SelectedItemData).ToArray();
            }
            set
            {
                for (int i= 0; i< Row*Column; i++)
                {
                    comboBoxes[i].SelectedItemData = value[i];
                }
            }
        }

        readonly Controls.UIComboBox[] comboBoxes;
        readonly int[] widths;

        public int Row { get; private set; }
        public int Column { get; private set; }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public render_info_slotType type;
        public const int minItemWidth = 100;
        public const int textMargin = 30;
        public EditRenderInfo(int row, int column, render_info_slotType type)
        {
            InitializeComponent();
            Row = row;
            Column = column;
            Debug.Assert(row * column > 0);
            this.type = type;
            comboBoxes = new UIComboBox[Row * Column];
            widths = new int[Row * Column];
            for (int i = 0; i < Row; i++)
            {
                for (int j = 0; j < Column; j++)
                {
                    // Row mager
                    int k = Column * i + j;
                    var control = new UIComboBox()
                    {
                        Width = minItemWidth,
                    };
                    control.Tag = 1u << k;
                    control.SelectedIndexChanged += cbxValue_ValueChanged;
                    control.CustomDrawItem += cbxValue_CustomDrawItem;
                    Controls.Add(control);
                    comboBoxes[k] = control;
                }
            }
        }

        public void VisibleComboBox(bool visible)
        {
            foreach (var comboBox in comboBoxes)
            {
                comboBox.Visible = false;
            }
        }

        public override void Align()
        {

            {
                int width = DefaultWidth - MarginX + marginBetweenColumn;
                for (int i = 0; i < Row; i++)
                {
                    for (int j = 0; j < Column; j++)
                    {
                        Control control = comboBoxes[Column * i + j];
                        control.Location = new Point(MarginX + (width * j) / 4, MarginY + subControlHeight * i);
                    }
                }
            }
        }

        private ChoiceHelper choiceHelper = new ChoiceHelper();

        Button[] linkButtons;
        public bool UpdateLinkButtons(bool updated)
        {
            var hashSet = choiceHelper.LabelHelpers.Where(x => x.Value.linkTarget != null);
            if (hashSet.Any())
            {
                if (linkButtons == null)
                {
                    linkButtons = new LinkButton[comboBoxes.Length];
                    int index = 0;
                    foreach (var comboBox in comboBoxes)
                    {
                        var button = LabelHelper.CreateLinkButton();
                        //button.Index = index;
                        linkButtons[index] = button;
                        var comboBox0 = comboBox;
                        button.Click += (s, e) =>
                            {
                                var value = (string)comboBox0.SelectedItemData;
                                var helper = choiceHelper.LabelHelpers[value];
                                OnLinkClicked(helper.ClickArgs());
                            };
                        Controls.Add(button);
                        index++;
                    }
                    updated = true;
                }

                {
                    int index = 0;
                    foreach (var comboBox in comboBoxes)
                    {
                        var button = linkButtons[index];
                        if (updated)
                        {
                            button.Location = new Point(comboBox.Right + 5, comboBox.Top + 2);
                        }
                        var data = (string)comboBox.SelectedItemData;
                        LabelHelper helper;
                        button.Enabled = data != null &&
                            choiceHelper.LabelHelpers.TryGetValue(data, out helper) &&
                            helper.linkTarget != null &&
                            !helper.LinkError;
                        index++;
                    }
                }
            }
            else
            {
                if (linkButtons != null)
                {
                    foreach (var button in linkButtons.Reverse())
                    {
                        Controls.Remove(button);
                        button.Dispose();
                    }
                    linkButtons = null;
                    updated = true;
                }
            }

            UpdateLinkToolTips(false);

            return updated;
        }

        public override void UpdateLinkToolTip(bool force)
        {
            base.UpdateLinkToolTip(force);
            UpdateLinkToolTips(force);
        }

        private void UpdateLinkToolTips(bool force)
        {
            try
            {
                if (linkButtons != null && parentHint != null)
                {
                    int index = 0;
                    foreach (var comboBox in comboBoxes)
                    {
                        var data = (string)comboBox.SelectedItemData;
                        LabelHelper helper = null;
                        if (data != null)
                        {
                            choiceHelper.LabelHelpers.TryGetValue(data, out helper);
                        }
                        var toolTip = helper != null ? helper.GetLinkLabelString(): "";
                        var current = parentHint.GetToolTip(linkButtons[index]);
                        if (toolTip != current || force)
                        {
                            parentHint.SetToolTip(linkButtons[index], toolTip);
                        }
                        index++;
                    }
                }
            }
            catch
            {
                Debug.Assert(false);
            }
        }

        // 有効な選択肢
        private readonly List<string> validData = new List<string>();
        public override bool SetValues(Material material, List<string> values, bool candidateModified, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel, Material.ValueResolvedState valueResolvedState)
        {
            Dictionary<string, string> aliases = new Dictionary<string,string>();
            validData.Clear();

            string[] defaultValues = new string[0];

            bool validDefault = true;

            bool packFromHio = false;
            // ビューアからの選択肢を設定
            // 数値型の場合はfsdb で選択肢が設定されているので来ないはず
            if (Viewer.Manager.Instance.IsConnected && type != null && type.type == render_info_slot_typeType.@string)
            {
                var candidates = material.RenderInfoPackFromHio.StringItems.FirstOrDefault(x => x.name == ParamName);
                if (candidates != null)
                {
                    packFromHio = true;
                    foreach(var value in candidates.values)
                    {
                        validData.Add(value.Choice);
                        aliases[value.Choice] = value.Alias;
                    }

                    // defaultValues を設定
                    defaultValues = candidates.defaults.ToArray();
                }
            }

            // シェーダー定義から選択肢を選択肢
            if (type != null && !packFromHio)
            {
                validData.AddRange(IfShaderOptionUtility.Enum(type.choice));
                aliases = IfShaderOptionUtility.Alias(type.choice);
                defaultValues = G3dDataParser.Tokenize(type.@default);
                validDefault = defaultValues.All(x => validData.Contains(x));
            }
            lblParameter.BackColor = GetBgColor(candidateModified, packFromHio, validData.Any());
            choiceHelper.SetLabel(aliases, customUI, table, visibleGroups, visiblePages, showId, showOriginalLabel);

            var valueArray = values.Concat(Enumerable.Repeat("", Row * Column)).Take(Row * Column).ToArray();
            using (var g = CreateGraphics())
            {
                for (int i = 0; i < Row * Column; i++)
                {
                    var control = comboBoxes[i];

                    // 正当な選択肢がない場合は無効にする
                    control.Enabled = validData.Count > 0 && validDefault;

                    string[] data;
                    if (control.Enabled)
                    {
                        // 範囲外は選択肢に値は入れない
                        data = Enumerable.Repeat("", (type != null && type.optional) ? 1: 0).Concat(validData).Distinct().ToArray();
                    }
                    else
                    {
                        data = Enumerable.Repeat("", (type != null && type.optional) ? 1 : 0).Concat(validData).Concat(Enumerable.Repeat(valueArray[i], 1)).Distinct().ToArray();
                    }

                    if (data.Count() != control.Items.Count ||
                        !data.Zip(control.Items.OfType<UIListControlItem>(), (x, y) => (x == (string)y.Data && (choiceHelper.labels.ContainsKey(x) ? choiceHelper.labels[x]: x) == y.Text)).All(x => x))
                    {
                        control.BeginUpdate();
                        control.Items.Clear();
                        widths[i] = minItemWidth;
                        foreach (var d in data)
                        {
                            string text = choiceHelper.labels.ContainsKey(d) ? choiceHelper.labels[d] : d;
                            control.AddItem(text, d);
                            widths[i] = Math.Max(widths[i], (int)g.MeasureString(text, control.Font).Width + textMargin);
                        }
                        control.EndUpdate();
                    }
                }
            }
            Value = valueArray;
            int width = widths.Max();
            bool updated = false;
            if (width != comboBoxes[0].Width)
            {
                foreach (var item in comboBoxes)
                {
                    item.Width = width;
                }
                updated = true;
            }

            return UpdateLinkButtons(updated);
        }


        private void cbxValue_ValueChanged(object sender, EventArgs e)
        {
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.ToList(),
                type = type.Type(),
            };
            InvokeValueChanged(this, args);
        }

        private void cbxValue_CustomDrawItem(object sender, CustomDrawListControlItemEventArgs e)
        {
            if (e.Item != null)
            {
                var data = (string)e.Item.Data;
                LabelHelper helper = null;
                if (data != null)
                {
                    choiceHelper.LabelHelpers.TryGetValue(data, out helper);
                }
                e.ForeColor = helper != null && helper.color.HasValue ? helper.color.Value:
                    data == string.Empty || validData.Contains(data) ? SystemColors.ControlText : Color.Red;
                e.BackColor = helper != null && helper.backGroundColor.HasValue ? helper.backGroundColor.Value : SystemColors.Window;
            }
        }

        public bool IsTextBlock { get; set; } = false;
    }
}
