﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditIntSliders : ShaderParamControl
    {
        public int[] Value
        {
            get
            {
                return intUpDowns.Select(x => x.Value).ToArray();
            }
            set
            {
                for (int i= 0; i< Row; i++)
                {
                    intUpDowns[i].Value = value[i];
                    intTrackBars[i].Value = value[i];
                }
                updateThumColor();
            }
        }

        void updateThumColor()
        {
            for (int i = 0; i < Row; i++)
            {
                var value = intTrackBars[i].Value;
                intTrackBars[i].DrawRedThumb = value < _minimum || _maximum < value;
            }
        }

        private int defaultMin = -1;
        private int _minimum = -1;
        public void SetMinimumValue(int minimum, int inputMin)
        {
            _minimum = minimum;
            for (int i = 0; i < Row; i++)
            {
                intUpDowns[i].InputMin = inputMin;
                intUpDowns[i].Minimum = minimum;
                intTrackBars[i].Minimum = minimum;
            }

            defaultMin = minimum;
            lblMinimum.Text = _minimum.ToString();
            updateButtonEnable();
            updateThumColor();
        }

        private int defaultMax = 1;
        private int _maximum = 1;
        public void SetMaximumValue(int maximum, int inputMax)
        {
            _maximum = maximum;
            for (int i = 0; i < Row; i++)
            {
                intUpDowns[i].InputMax = inputMax;
                intUpDowns[i].Maximum = maximum;
                intTrackBars[i].Maximum = maximum;
            }

            defaultMax = maximum;
            lblMaximum.Text = _maximum.ToString();
            updateButtonEnable();
            updateThumColor();
        }

        public void SetDefaultMinValue()
        {
            Value = Enumerable.Repeat(defaultMin, Row).ToArray();
        }

        public void UpdateIncrement()
        {
            var Increment = 1;//(int)Math.Pow(10, Math.Max(0, Math.Round(Math.Log10(Math.Max(0.1, (double)intUpDowns[0].Maximum - (double)intUpDowns[0].Minimum))) - 2));
            var LargeIncrement = (int)(Increment * 10);
            for (int i = 0; i < Row; i++)
            {
                intUpDowns[i].Increment = Increment;
                intUpDowns[i].LargeIncrement = LargeIncrement;
                intTrackBars[i].SmallChange = Increment;
                intTrackBars[i].LargeChange = LargeIncrement;
            }
        }

        readonly UILabel lblMinimum;
        readonly UILabel lblMaximum;
        readonly UIButton btnRangeDown;
        readonly UIButton btnRangeUp;
        readonly IntUpDown[] intUpDowns;
        readonly IntTrackBar[] intTrackBars;
        readonly UICheckBox linkValueCheck;
        System.Drawing.Rectangle? initialBounds = null;
        int row = 0;

        public int Row
        {
            get
            {
                return row;
            }
            private set
            {
                row = value;
                NumComponentCurveEditorOpeningButtons = row;
            }
        }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public shader_param_typeType type;
        public bool option = false;

        public static HashSet<string> CheckedNames = new HashSet<string>();
        public override string ParamName
        {
            get
            {
                return base.ParamName;
            }
            set
            {
                base.ParamName = value;
                if (linkValueCheck != null)
                {
                    linkValueCheck.Checked = CheckedNames.Contains(value);
                }
            }
        }

        private void chkLinkValueCheck_CheckedChanged(object sender, EventArgs e)
        {
            if (linkValueCheck.Checked)
            {
                CheckedNames.Add(ParamName);
            }
            else
            {
                CheckedNames.Remove(ParamName);
            }
        }

        private static readonly Bitmap imgValueEditPanelRangeDown_ = App.Properties.Resources.Control_ValueEditPanelRangeDown;
        private static readonly Bitmap imgValueEditPanelRangeUp_ = App.Properties.Resources.Control_ValueEditPanelRangeUp;

        ParamType paramType;
        public override ParamType ParamType
        {
            get { return paramType; }
        }

        public EditIntSliders(int row, shader_param_typeType type)
        {
            InitializeComponent();

            Row = row;
            this.type = type;
            intUpDowns = new IntUpDown[Row];
            intTrackBars = new IntTrackBar[Row];
            for (int i = 0; i < Row; i++)
            {
                var control = new IntUpDown();
                control.EnableInputMinMax = true;
                control.Tag = i;
                control.Width = 74;
                control.Minimum = -100;
                control.Maximum = 100;
                control.Value = 0;
                control.SequentialValueChanged += edpValue_SequentialValueChanged;
                Controls.Add(control);
                intUpDowns[i] = control;

                var bar = new IntTrackBar();
                bar.Width = 240;
                intTrackBars[i] = bar;
                Controls.Add(bar);
                bar.Tag = i;
                bar.SequentialValueChanged += bar_SequentialValueChanged;
            }

            if (Row != 1) // add link checkbox
            {
                var chk = new UICheckBox();
                chk.Text = Strings.ShaderParamControl_LinkScale;
                chk.CheckedChanged += chkLinkValueCheck_CheckedChanged;
                chk.AutoSize = true;
                Controls.Add(chk);
                linkValueCheck = chk;
            }
            btnRangeDown = new UIButton();
            btnRangeUp = new UIButton();
            btnRangeDown.Size = btnRangeUp.Size = new Size(24, 16);
            btnRangeDown.Image = imgValueEditPanelRangeDown_;
            btnRangeUp.Image = imgValueEditPanelRangeUp_;
            btnRangeDown.Click += btnRange_Click;
            btnRangeUp.Click += btnRange_Click;
            Controls.Add(btnRangeDown);
            Controls.Add(btnRangeUp);
            lblMinimum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopLeft,
            };
            lblMaximum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopRight,
            };
            lblMinimum.Height = lblMaximum.Height = 13;
            Controls.Add(lblMinimum);
            Controls.Add(lblMaximum);
            paramType = ParamType.uniform_var;
        }

        public EditIntSliders(option_varType option)
        {
            InitializeComponent();
            Row = 1;
            this.option = true;
            ParamName = option.id;
            intUpDowns = new IntUpDown[1];
            var control = new IntUpDown();
            control.EnableInputMinMax = false;
            control.Width = 74;
            control.SequentialValueChanged += edpValue_SequentialValueChanged;
            int min, max;
            IfShaderAssignUtility.TryParseIntRange(option.choice, out min, out max);
            control.Minimum = control.InputMin = min;
            control.Maximum = control.InputMax = max;
            control.Value = 0;
            Controls.Add(control);
            intUpDowns[0] = control;

            intTrackBars = new IntTrackBar[1];
            var bar = new IntTrackBar();
            bar.Width = 240;
            intTrackBars[0] = bar;
            Controls.Add(bar);
            bar.SequentialValueChanged += bar_SequentialValueChanged;
            bar.Minimum = min;
            bar.Maximum = max;
            bar.Value = 0;

            _minimum = min;
            _maximum = max;
            /*
            btnRangeDown = new UIButton();
            btnRangeUp = new UIButton();
            btnRangeDown.Size = btnRangeUp.Size = new Size(24, 16);
            btnRangeDown.Image = imgValueEditPanelRangeDown_;
            btnRangeUp.Image = imgValueEditPanelRangeUp_;
            btnRangeDown.Click += btnRange_Click;
            btnRangeUp.Click += btnRange_Click;
            Controls.Add(btnRangeDown);
            Controls.Add(btnRangeUp);*/
            lblMinimum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopLeft,
            };
            lblMaximum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopRight,
            };
            lblMinimum.Text = min.ToString();
            lblMaximum.Text = max.ToString();
            lblMinimum.Height = lblMaximum.Height = 13;
            Controls.Add(lblMinimum);
            Controls.Add(lblMaximum);
            paramType = ParamType.option_var;
        }

        private void btnRange_Click(object sender, EventArgs e)
        {
            Debug.Assert(!option);
            var range = new IntRangeProperty(_minimum, _maximum, 1, 1, intUpDowns[0].InputMin, intUpDowns[0].InputMax, defaultMin, defaultMax);
            range.Scale(sender == btnRangeUp);
            updateScale(range.Minimum, range.Maximum);
        }

        private void updateScale(int min, int max)
        {
            Debug.Assert(!option);
            _minimum = min;
            _maximum = max;
            int Increment = 1;// (int)Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.1, (double)max - (double)min))) - 2);
            int LargeIncrement = Increment * 10;
            for (int i = 0; i < Row; i++)
            {
                intUpDowns[i].Minimum = min;
                intTrackBars[i].Minimum = min;
                intUpDowns[i].Maximum = max;
                intTrackBars[i].Maximum = max;

                intUpDowns[i].Increment = Increment;
                intUpDowns[i].LargeIncrement = LargeIncrement;
                intTrackBars[i].SmallChange = Increment;
                intTrackBars[i].LargeChange = LargeIncrement;
            }

            lblMinimum.Text = _minimum.ToString();
            lblMaximum.Text = _maximum.ToString();
            updateButtonEnable();
            updateThumColor();
        }

        private void updateButtonEnable()
        {
            Debug.Assert(!option);
            var range = new IntRangeProperty(_minimum, _maximum, 1, 1, intUpDowns[0].InputMin, intUpDowns[0].InputMax, defaultMin, defaultMax);
            btnRangeUp.Enabled = range.Scale(true);
            range = new IntRangeProperty(_minimum, _maximum, 1, 1, intUpDowns[0].InputMin, intUpDowns[0].InputMax, defaultMin, defaultMax);
            btnRangeDown.Enabled = range.Scale(false);
        }

        public override void Align()
        {
            if (intUpDowns == null)
            {
                return;
            }

            {
                int width = DefaultWidth - MarginX + marginBetweenColumn;
                for (int i = 0; i < Row; i++)
                {
                    Control control = intUpDowns[i];
                    //control.Size = new Size(width / 4 - marginBetweenColumn, subControlHeight);
                    control.Location = new Point(MarginX, MarginY + subControlHeight * i);

                    var bar = intTrackBars[i];
                    bar.Location = new Point(MarginX + control.Size.Width, MarginY + subControlHeight * i - 1);
                }
                if (!option)
                {
                    btnRangeDown.Location = new Point(intTrackBars[0].Location.X + intTrackBars[0].Width / 2 - 2 - btnRangeDown.Width, MarginY + subControlHeight * Row);
                    btnRangeUp.Location = new Point(intTrackBars[0].Location.X + intTrackBars[0].Width / 2 + 2, MarginY + subControlHeight * Row);
                    lblMinimum.Location = new Point(intTrackBars[0].Location.X + 3, MarginY + subControlHeight * Row);
                    lblMinimum.Width = btnRangeDown.Location.X - lblMinimum.Location.X;
                    lblMaximum.Location = new Point(btnRangeUp.Right + 1, MarginY + subControlHeight * Row);
                    lblMaximum.Width = lblMinimum.Width;
                }
                else
                {
                    lblMinimum.Location = new Point(intTrackBars[0].Location.X + 3, MarginY + subControlHeight);
                    lblMinimum.Width = (intTrackBars[0].Width - 6) / 2;
                    lblMaximum.Location = new Point(lblMinimum.Right, MarginY + subControlHeight);
                    lblMaximum.Width = lblMinimum.Width;
                }

                if (linkValueCheck != null && Row != 1 && !option) // link value checkbox
                {
                    linkValueCheck.Location = new Point(MarginX, MarginY + subControlHeight * Row);
                }
            }

            // トラックバーの右横に「カーブエディタを開く」ボタンを配置。
            bool showButtons = false;
            do
            {
                var numIntTrackBars = intTrackBars.Length;
                if ((numIntTrackBars <= 1) || (numIntTrackBars != ComponentCurveEditorOpeningButtons.Count))
                {
                    break;
                }

                // トラックバーの右横に、ボタンを配置。
                for (int i = 0; i < numIntTrackBars; i++)
                {
                    var intTrackBar = intTrackBars[i];
                    ComponentCurveEditorOpeningButtons[i].Location = new System.Drawing.Point(intTrackBar.Right, intTrackBar.Top);
                }

                // コントロールサイズ拡大の妨げになるアンカーを一時的に外しておく。
                var oColorEditPanelAnchors = new System.Windows.Forms.AnchorStyles[Controls.Count];
                for (int i = 0; i < Controls.Count; i++)
                {
                    oColorEditPanelAnchors[i] = Controls[i].Anchor;
                    Controls[i].Anchor &= ~(System.Windows.Forms.AnchorStyles.Right | System.Windows.Forms.AnchorStyles.Bottom);
                }

                // 成分毎の「カーブエディタで開く」ボタンを追加する前の境界を取得しておく。
                if (!initialBounds.HasValue)
                {
                    initialBounds = Bounds;
                }

                // すべてのコントロールを内包する境界。
                // 座標はクラス外部で配置されるので、サイズだけを更新。
                var bounds = ComponentCurveEditorOpeningButtons.Select(x => x.Bounds).Concat(Enumerable.Repeat(initialBounds.Value, 1)).Aggregate(System.Drawing.Rectangle.Union);
                Size = bounds.Size;

                // アンカーを戻す。
                for (int i = 0; i < Controls.Count; i++)
                {
                    Controls[i].Anchor = oColorEditPanelAnchors[i];
                }

                showButtons = true;
            }
            while (false);

            if (showButtons)
            {
                // 成分毎の「カーブエディタで開く」ボタンを追加する。
                Controls.AddRange(ComponentCurveEditorOpeningButtons.Where(x => !Controls.Contains(x)).ToArray());
            }
            else
            {
                foreach (var button in ComponentCurveEditorOpeningButtons)
                {
                    Controls.Remove(button);
                }

                if (initialBounds.HasValue)
                {
                    // 座標はクラス外部で配置されるので、サイズだけを更新。
                    Size = initialBounds.Value.Size;
                }
            }
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            if (option)
            {
                value = value.Trim(new char[] { '(', ')' });
            }
            int[] paramArray = G3dDataParser.ParseIntArray(value);
            if (paramArray == null || paramArray.Length != Row)
            {
                Enabled = false;
                return false;
            }

            if (!option)
            {
                for (int i = 0; i < Row; i++)
                {
                    if (Value[i] != paramArray[i])
                    {
                        FitRange(paramArray[i]);
                    }
                }
            }

            Value = paramArray;
            return false;
        }

        private void bar_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (!option)
            {
                var args = new ShaderParamValueChangedEventArgs(type);
                uint elemBits;
                int value = ((IntTrackBar)sender).Value;
                int[] values;
                // 連動がオンのときはUIを同じ値で更新しまとめて一つのイベントにする
                if (linkValueCheck != null && linkValueCheck.Checked)
                {
                    values = Enumerable.Repeat<int>(value, Row).ToArray();
                    elemBits = (1u << Row) - 1;
                }
                else
                {
                    values = Value;
                    int i = (int)(((Control)sender).Tag);
                    elemBits = 1u << i;
                    values[i] = value;
                }
                using (var block = new UIControlEventSuppressBlock())
                {
                    for (int i = 0; i < Row; i++)
                    {
                        intUpDowns[i].Value = values[i];
                        if (i != (int)(((Control)sender).Tag))
                        {
                            intTrackBars[i].Value = values[i];
                        }
                    }
                    updateThumColor();
                }

                args.ParamName = ParamName;
                args.ParamValue = values;
                args.ElementBits = elemBits;
                args.SequentialValueChangedEventArgs = e;

                InvokeValueChanged(this, args);
            }
            else
            {
                using (var block = new UIControlEventSuppressBlock())
                {
                    intUpDowns[0].Value = ((IntTrackBar)sender).Value;
                    updateThumColor();
                }
                var args = new OptionValueChangedEventArgs();
                {
                    args.ParamName = ParamName;
                    args.ParamValue = Value[0].ToString();
                    args.SequentialValueChangedEventArgs = e;
                }
                InvokeValueChanged(this, args);
            }
        }

        private void edpValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (!option)
            {
                var args = new ShaderParamValueChangedEventArgs(type);
                uint elemBits = 0;

                // 連動がオンのときはUIを同じ値で更新しまとめて一つのイベントにする
                if (linkValueCheck != null && linkValueCheck.Checked)
                {
                    var modifiedPanel = sender as IntUpDown;
                    Debug.Assert(modifiedPanel != null);

                    using (var block = new UIControlEventSuppressBlock())
                    {
                        foreach (var panel in intUpDowns)
                        {
                            if (modifiedPanel != panel)
                            {
                                panel.Value = modifiedPanel.Value;
                            }
                            elemBits |= 1u<<(int)panel.Tag;
                        }
                    }
                }
                else
                {
                    elemBits = 1u<<(int)(((Control)sender).Tag);
                }

                var value = ((IntUpDown)sender).Value;
                FitRange(value);

                args.ParamName = ParamName;
                args.ParamValue = Value;
                args.ElementBits = elemBits;
                args.SequentialValueChangedEventArgs = e;

                InvokeValueChanged(this, args);
            }
            else
            {
                var args = new OptionValueChangedEventArgs();
                {
                    args.ParamName = ParamName;
                    args.ParamValue = Value[0].ToString();
                    args.SequentialValueChangedEventArgs = e;
                }
                InvokeValueChanged(this, args);
            }
        }

        void FitRange(int value)
        {
            Debug.Assert(!option);
            if (value < _minimum || _maximum < value)
            {

                var range = new IntRangeProperty(_minimum, _maximum, 1, 1, intUpDowns[0].InputMin, intUpDowns[0].InputMax, defaultMin, defaultMax);
                while (!range.Contains(value) && range.Scale(true))
                {
                }
                if (range.Contains(value))
                {
                    updateScale(range.Minimum, range.Maximum);
                }
            }
        }

        public override void UpdateEnabled(bool value)
        {
            base.UpdateEnabled(value);
            if (value && !option)
            {
                updateButtonEnable();
            }
        }
    }
}
