﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditFloatSliders : ShaderParamControl
    {
        // カラーの表示切替
        public event ColorControlChangedEventHandler ColorControlChanged = null;

        protected override void labelContextMenuStrip_Opening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            // コンテキストメニューの構築
            base.labelContextMenuStrip_Opening(sender, e);
            if (ColorControlChanged != null)
            {
                var cmiColorControl = new System.Windows.Forms.ToolStripMenuItem();
                cmiColorControl.Size = new System.Drawing.Size(152, 22);
                cmiColorControl.Text = Strings.ShaderParamControl_ShowAsColor;
                cmiColorControl.Click += (s, args) =>
                {
                    // フォーカスを外す
                    ActiveControl = null;

                    ColorControlChanged(s, new ColorControlChangedEventArgs()
                    {
                        ParamName = ParamName,
                        ShowColor = true,
                    });
                };
                labelContextMenuStrip.Items.Add(cmiColorControl);
            }
        }

        public float[] Value
        {
            get
            {
                return floatUpDowns.Select(x => x.Value).ToArray();
            }
            set
            {
                for (int i= 0; i< Row; i++)
                {
                    floatUpDowns[i].Value = value[i];
                    floatTrackBars[i].Value = value[i];
                }
                updateThumColor();
            }
        }

        void updateThumColor()
        {
            for (int i = 0; i < Row; i++)
            {
                var value = floatTrackBars[i].Value;
                floatTrackBars[i].DrawRedThumb = value < _minimum || _maximum < value;
            }
        }

        static string FloatToString(float val)
        {
            var str = val.ToString("R");
            if (str.Length > 14)
            {
                str = val.ToString("0.#########E+0");
            }
            return str;
        }

        private float defaultMin = -1;
        private float _minimum = -1;
        public void SetMinimumValue(float minimum, float inputMin)
        {
            _minimum = minimum;
            for (int i = 0; i < Row; i++)
            {
                floatUpDowns[i].InputMin = inputMin;
                floatUpDowns[i].Minimum = minimum;
                floatTrackBars[i].Minimum = minimum;
            }

            defaultMin = minimum;
            lblMinimum.Text = FloatToString(_minimum);
            updateButtonEnable();
            updateThumColor();
        }

        private float defaultMax = 1;
        private float _maximum = 1;
        public void SetMaximumValue(float maximum, float inputMax)
        {
            _maximum = maximum;
            for (int i = 0; i < Row; i++)
            {
                floatUpDowns[i].InputMax = inputMax;
                floatUpDowns[i].Maximum = maximum;
                floatTrackBars[i].Maximum = maximum;
            }

            defaultMax = maximum;

            lblMaximum.Text = FloatToString(_maximum);
            updateButtonEnable();
            updateThumColor();
        }

        public void SetDefaultMinValue()
        {
            Value = Enumerable.Repeat(defaultMin, Row).ToArray();
        }

        public void UpdateIncrement()
        {
            double Increment = Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, (double)floatUpDowns[0].Maximum - (double)floatUpDowns[0].Minimum))) - 2));
            double LargeIncrement = Increment * 10;
            for (int i = 0; i < Row; i++)
            {
                floatUpDowns[i].Increment = Increment;
                floatUpDowns[i].LargeIncrement = LargeIncrement;
                floatTrackBars[i].SmallChange = Increment;
                floatTrackBars[i].LargeChange = LargeIncrement;
            }
        }

        readonly UILabel lblMinimum;
        readonly UILabel lblMaximum;
        readonly UIButton btnRangeDown;
        readonly UIButton btnRangeUp;
        readonly Controls.FloatUpDown[] floatUpDowns;
        readonly Controls.FloatTrackBar[] floatTrackBars;
        readonly UICheckBox linkValueCheck;
        System.Drawing.Rectangle? initialBounds = null;
        int row = 0;


        public int Row
        {
            get
            {
                return row;
            }
            private set
            {
                row = value;
                NumComponentCurveEditorOpeningButtons = row;
            }
        }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public shader_param_typeType type;

        public static HashSet<string> CheckedNames = new HashSet<string>();
        public override string ParamName
        {
            get
            {
                return base.ParamName;
            }
            set
            {
                base.ParamName = value;
                if (linkValueCheck != null)
                {
                    linkValueCheck.Checked = CheckedNames.Contains(value);
                }
            }
        }

        private void chkLinkValueCheck_CheckedChanged(object sender, EventArgs e)
        {
            if (linkValueCheck.Checked)
            {
                CheckedNames.Add(ParamName);
            }
            else
            {
                CheckedNames.Remove(ParamName);
            }
        }

        private static readonly Bitmap imgValueEditPanelRangeDown_ = App.Properties.Resources.Control_ValueEditPanelRangeDown;
        private static readonly Bitmap imgValueEditPanelRangeUp_ = App.Properties.Resources.Control_ValueEditPanelRangeUp;

        public override ParamType ParamType
        {
            get { return ParamType.uniform_var; }
        }

        public EditFloatSliders(int row, shader_param_typeType type)
        {
            InitializeComponent();

            Row = row;
            this.type = type;
            floatUpDowns = new FloatUpDown[Row];
            floatTrackBars = new FloatTrackBar[Row];
            for (int i = 0; i < Row; i++)
            {
                var control = new FloatUpDown();
                control.EnableInputMinMax = true;
                control.Tag = i;
                control.Width = 74;
                //control.SliderRangeAdjustable = true;
                control.Minimum = -100;
                control.Maximum = 100;
                control.Value = 0;
                control.SequentialValueChanged += edpValue_SequentialValueChanged;
                Controls.Add(control);
                floatUpDowns[i] = control;

                var bar = new FloatTrackBar();
                bar.Width = 240;
                floatTrackBars[i] = bar;
                Controls.Add(bar);
                bar.Tag = i;
                bar.SequentialValueChanged += bar_SequentialValueChanged;
            }
            if (Row != 1) // add link checkbox
            {
                var chk = new UICheckBox();
                chk.Text = Strings.ShaderParamControl_LinkScale;
                chk.CheckedChanged += chkLinkValueCheck_CheckedChanged;
                chk.AutoSize = true;
                Controls.Add(chk);
                linkValueCheck = chk;
            }
            btnRangeDown = new UIButton();
            btnRangeUp = new UIButton();
            btnRangeDown.Size = btnRangeUp.Size = new Size(24, 16);
            btnRangeDown.Image = imgValueEditPanelRangeDown_;
            btnRangeUp.Image = imgValueEditPanelRangeUp_;
            btnRangeDown.Click += btnRange_Click;
            btnRangeUp.Click += btnRange_Click;
            Controls.Add(btnRangeDown);
            Controls.Add(btnRangeUp);
            lblMinimum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopLeft,
            };
            lblMaximum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopRight,
            };
            lblMinimum.Height = lblMaximum.Height = 13;
            Controls.Add(lblMinimum);
            Controls.Add(lblMaximum);
        }

        private void btnRange_Click(object sender, EventArgs e)
        {
            var range = new FloatRangeProperty(_minimum, _maximum, 1, 1, floatUpDowns[0].InputMin, floatUpDowns[0].InputMax, defaultMin, defaultMax);
            range.Scale(sender == btnRangeUp);
            updateScale(range.Minimum, range.Maximum);
        }

        private void updateScale(float min, float max)
        {
            _minimum = min;
            _maximum = max;
            double Increment = Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, (double)floatUpDowns[0].Maximum - (double)floatUpDowns[0].Minimum))) - 2));
            double LargeIncrement = Increment * 10;
            for (int i = 0; i < Row; i++)
            {
                floatUpDowns[i].Minimum = min;
                floatTrackBars[i].Minimum = min;
                floatUpDowns[i].Maximum = max;
                floatTrackBars[i].Maximum = max;

                floatUpDowns[i].Increment = Increment;
                floatUpDowns[i].LargeIncrement = LargeIncrement;
                floatTrackBars[i].SmallChange = Increment;
                floatTrackBars[i].LargeChange = LargeIncrement;
            }

            lblMinimum.Text = FloatToString(_minimum);
            lblMaximum.Text = FloatToString(_maximum);
            updateButtonEnable();
            updateThumColor();
        }

        private void updateButtonEnable()
        {
            var range = new FloatRangeProperty(_minimum, _maximum, 1, 1, floatUpDowns[0].InputMin, floatUpDowns[0].InputMax, defaultMin, defaultMax);
            btnRangeUp.Enabled = range.Scale(true);
            range = new FloatRangeProperty(_minimum, _maximum, 1, 1, floatUpDowns[0].InputMin, floatUpDowns[0].InputMax, defaultMin, defaultMax);
            btnRangeDown.Enabled = range.Scale(false);
        }

        public override void Align()
        {
            if (floatUpDowns == null)
            {
                return;
            }

            {
                int width = DefaultWidth - MarginX + marginBetweenColumn;
                for (int i = 0; i < Row; i++)
                {
                    Control control = floatUpDowns[i];
                    //control.Size = new Size(width / 4 - marginBetweenColumn, subControlHeight);
                    control.Location = new Point(MarginX, MarginY + subControlHeight * i);

                    var bar = floatTrackBars[i];
                    bar.Location = new Point(MarginX + control.Size.Width, MarginY + subControlHeight * i-1);
                }
                btnRangeDown.Location = new Point(floatTrackBars[0].Location.X + floatTrackBars[0].Width / 2 - 2 - btnRangeDown.Width, MarginY + subControlHeight * Row);
                btnRangeUp.Location = new Point(floatTrackBars[0].Location.X + floatTrackBars[0].Width / 2 + 2, MarginY + subControlHeight * Row);
                lblMinimum.Location = new Point(floatTrackBars[0].Location.X + 3, MarginY + subControlHeight * Row);
                lblMinimum.Width = btnRangeDown.Location.X - lblMinimum.Location.X;
                lblMaximum.Location = new Point(btnRangeUp.Right + 1, MarginY + subControlHeight * Row);
                lblMaximum.Width = lblMinimum.Width;
                if (linkValueCheck != null && Row != 1) // link value checkbox
                {
                    linkValueCheck.Location = new Point(MarginX, MarginY + subControlHeight * Row);
                }
            }

            // トラックバーの右横に「カーブエディタを開く」ボタンを配置。
            bool showButtons = false;
            do
            {
                var numFloatTrackBars = floatTrackBars.Length;
                if ((numFloatTrackBars <= 1) || (numFloatTrackBars !=  ComponentCurveEditorOpeningButtons.Count))
                {
                    break;
                }

                // トラックバーの右横に、ボタンを配置。
                for (int i = 0; i < numFloatTrackBars; i++)
                {
                    var floatTrackBar = floatTrackBars[i];
                    ComponentCurveEditorOpeningButtons[i].Location = new System.Drawing.Point(floatTrackBar.Right, floatTrackBar.Top);
                }

                // コントロールサイズ拡大の妨げになるアンカーを一時的に外しておく。
                var oColorEditPanelAnchors = new System.Windows.Forms.AnchorStyles[Controls.Count];
                for (int i = 0; i < Controls.Count; i++)
                {
                    oColorEditPanelAnchors[i] = Controls[i].Anchor;
                    Controls[i].Anchor &= ~(System.Windows.Forms.AnchorStyles.Right | System.Windows.Forms.AnchorStyles.Bottom);
                }

                // 成分毎の「カーブエディタで開く」ボタンを追加する前の境界を取得しておく。
                if (!initialBounds.HasValue)
                {
                    initialBounds = Bounds;
                }

                // すべてのコントロールを内包する境界。
                // 座標はクラス外部で配置されるので、サイズだけを更新。
                var bounds = ComponentCurveEditorOpeningButtons.Select(x => x.Bounds).Concat(Enumerable.Repeat(initialBounds.Value, 1)).Aggregate(System.Drawing.Rectangle.Union);
                Size = bounds.Size;

                // アンカーを戻す。
                for (int i = 0; i < Controls.Count; i++)
                {
                    Controls[i].Anchor = oColorEditPanelAnchors[i];
                }

                showButtons = true;
            }
            while (false);

            if (showButtons)
            {
                // 成分毎の「カーブエディタで開く」ボタンを追加する。
                Controls.AddRange(ComponentCurveEditorOpeningButtons.Where(x => !Controls.Contains(x)).ToArray());
            }
            else
            {
                foreach (var button in ComponentCurveEditorOpeningButtons)
                {
                    Controls.Remove(button);
                }

                if (initialBounds.HasValue)
                {
                    // 座標はクラス外部で配置されるので、サイズだけを更新。
                    Size = initialBounds.Value.Size;
                }
            }
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            float[] paramArray = G3dDataParser.ParseFloatArray(value);
            if (paramArray == null || paramArray.Length != Row)
            {
                Enabled = false;
                return false;
            }

            for (int i = 0; i < Row; i++)
            {
                if (Value[i] != paramArray[i])
                {
                    FitRange(paramArray[i]);
                }
            }
            Value = paramArray;
            return false;
        }

        private void bar_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {

            var args = new ShaderParamValueChangedEventArgs(type);
            uint elemBits;
            float value = ((FloatTrackBar)sender).Value;
            float[] values;
            // 連動がオンのときはUIを同じ値で更新しまとめて一つのイベントにする
            if (linkValueCheck != null && linkValueCheck.Checked)
            {
                values = Enumerable.Repeat<float>(value, Row).ToArray();
                elemBits = (1u << Row) - 1;
            }
            else
            {
                values = Value;
                int i = (int)(((Control)sender).Tag);
                elemBits = 1u << i;
                values[i] = value;
            }
            using (var block = new UIControlEventSuppressBlock())
            {
                for (int i = 0; i < Row; i++)
                {
                    floatUpDowns[i].Value = values[i];
                    if (i != (int)(((Control)sender).Tag))
                    {
                        floatTrackBars[i].Value = values[i];
                    }
                }
                updateThumColor();
            }

            args.ParamName = ParamName;
            args.ParamValue = values;
            args.ElementBits = elemBits;
            args.SequentialValueChangedEventArgs = e;

            InvokeValueChanged(this, args);
        }

        private void edpValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var args = new ShaderParamValueChangedEventArgs(type);
            uint elemBits = 0;

            // 連動がオンのときはUIを同じ値で更新しまとめて一つのイベントにする
            if (linkValueCheck != null && linkValueCheck.Checked)
            {
                var modifiedPanel = sender as FloatUpDown;
                Debug.Assert(modifiedPanel != null);

                using (var block = new UIControlEventSuppressBlock())
                {
                    foreach (var panel in floatUpDowns)
                    {
                        if (modifiedPanel != panel)
                        {
                            panel.Value = modifiedPanel.Value;
                        }
                        elemBits |= 1u<<(int) panel.Tag;
                    }
                }
            }
            else
            {
                elemBits = 1u<<(int)(((Control)sender).Tag);
            }

            var value = ((FloatUpDown)sender).Value;
            FitRange(value);

            args.ParamName = ParamName;
            args.ParamValue = Value;
            args.ElementBits = elemBits;
            args.SequentialValueChangedEventArgs = e;

            InvokeValueChanged(this, args);
        }

        void FitRange(float value)
        {
            if (value < _minimum || _maximum < value)
            {

                var range = new FloatRangeProperty(_minimum, _maximum, 1, 1, floatUpDowns[0].InputMin, floatUpDowns[0].InputMax, defaultMin, defaultMax);
                while (!range.Contains(value) && range.Scale(true))
                {
                }
                if (range.Contains(value))
                {
                    updateScale(range.Minimum, range.Maximum);
                }
            }
        }

        public override void UpdateEnabled(bool value)
        {
            base.UpdateEnabled(value);
            if (value)
            {
                updateButtonEnable();
            }
        }
    }
}
