﻿using System;
using System.Collections.Generic;
using System.Collections;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class MaterialRenderStatePage : MaterialPropertyPage
    {
        private class OperatorLabel
        {
            private readonly string[] labelList;

            public OperatorLabel(string label0, string label1, string label2)
            {
                labelList = new[]{label0, label1, label2};
            }

            public string this[int index]
            {
                get{
                    return labelList[index];
                }
            }
        }

        private static readonly Dictionary<color_blend_opType, OperatorLabel>	operatorLabels = new Dictionary<color_blend_opType, OperatorLabel>()
        {
            {color_blend_opType.add,			new OperatorLabel("",		"+",	"")},
            {color_blend_opType.src_minus_dst,	new OperatorLabel("",		"-",	"")},
            {color_blend_opType.dst_minus_src,	new OperatorLabel("-",		"+",	"")},
            {color_blend_opType.min,			new OperatorLabel("Min(",	", ",	")")},
            {color_blend_opType.max,			new OperatorLabel("Max(",	", ",	")")},
        };

        public enum BlendPresetType
        {
            Custom,
            Blend,
            Add,
            Sub,
            Mul,
        }

        private class BlendPreset
        {
            public color_blend_opType			ColorBlendOp{	get; private set; }
            public color_blend_rgb_funcType		ColorSrc{		get; private set; }
            public color_blend_rgb_funcType		ColorDst{		get; private set; }
            public color_blend_opType			AlphaBlendOp{	get; private set; }
            public color_blend_alpha_funcType	AlphaSrc{		get; private set; }
            public color_blend_alpha_funcType	AlphaDst{		get; private set; }

            public BlendPreset(
                color_blend_opType			colorBlendOp,
                color_blend_rgb_funcType	colorSrc,
                color_blend_rgb_funcType	colorDst
            ) : this(colorBlendOp, colorSrc, colorDst, color_blend_opType.add, color_blend_alpha_funcType.one, color_blend_alpha_funcType.zero){}

            public BlendPreset(
                color_blend_opType			colorBlendOp,
                color_blend_rgb_funcType	colorSrc,
                color_blend_rgb_funcType	colorDst,
                color_blend_opType			alphaBlendOp,
                color_blend_alpha_funcType	alphaSrc,
                color_blend_alpha_funcType	alphaDst
            )
            {
                ColorBlendOp 	= colorBlendOp;
                ColorSrc	 	= colorSrc;
                ColorDst	 	= colorDst;
                AlphaBlendOp	= alphaBlendOp;
                AlphaSrc		= alphaSrc;
                AlphaDst		= alphaDst;
            }

            public static bool operator ==(BlendPreset a, BlendPreset b)
            {
                return
                    (a.ColorBlendOp	== b.ColorBlendOp) &&
                    (a.ColorSrc		== b.ColorSrc) &&
                    (a.ColorDst		== b.ColorDst) &&
                    (a.AlphaBlendOp	== b.AlphaBlendOp) &&
                    (a.AlphaSrc		== b.AlphaSrc) &&
                    (a.AlphaDst		== b.AlphaDst);
            }

            public static bool operator !=(BlendPreset a, BlendPreset b)
            {
                return (a == b) == false;
            }

            public override int GetHashCode()
            {
                // TODO: 削除 GetHashCode が間違い
                Debug.Assert(false);

                return base.GetHashCode();
            }

            public override bool Equals(object obj)
            {
                // TODO: 削除 GetHashCode が間違い
                Debug.Assert(false);

                if ((obj == null) ||
                    (obj is BlendPreset) == false)
                {
                    return false;
                }
                else
                {
                    return this == (obj as BlendPreset);
                }
            }
        }

        private static readonly Dictionary<BlendPresetType, BlendPreset>	blendPresets = new Dictionary<BlendPresetType, BlendPreset>()
        {
            {BlendPresetType.Blend,	new BlendPreset(color_blend_opType.add,				color_blend_rgb_funcType.src_alpha,	color_blend_rgb_funcType.one_minus_src_alpha)},
            {BlendPresetType.Add,	new BlendPreset(color_blend_opType.add,				color_blend_rgb_funcType.src_alpha,	color_blend_rgb_funcType.one)},
            {BlendPresetType.Sub,	new BlendPreset(color_blend_opType.dst_minus_src,	color_blend_rgb_funcType.src_alpha,	color_blend_rgb_funcType.one)},
            {BlendPresetType.Mul,	new BlendPreset(color_blend_opType.add,				color_blend_rgb_funcType.zero,		color_blend_rgb_funcType.src_color)},
        };

        public MaterialRenderStatePage() :
            base(PropertyPageID.MaterialRenderState)
        {
            InitializeComponent();
            cepBlendColor.ColorPickerText = res.Strings.ColorPicker;
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_material_property_window;
            }
        }

        private static bool NeedToReload(Material target)
        {
            return (!string.IsNullOrEmpty(App.ConfigData.ApplicationConfig.FileIo.BuildFsv.ScriptFilePath) ||
                App.ConfigData.ApplicationConfig.FileIo.CreateShaderVariationCommand.HasCommand) &&
                Viewer.Manager.Instance.IsConnected &&
                DocumentManager.OptimizeShader &&
                target.OptimizeShader;
            // target.OptimizeShaderOnReload は条件に入れない
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new MaterialRenderStatePage();
        }

        protected override void InitializeFormInternal()
        {
            using(var ub = new UpdateBlock(cmbRenderState))
            {
                foreach (render_state_modeType value in Enum.GetValues(typeof(render_state_modeType)))
                {
                    cmbRenderState.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbBlendMode))
            {
                foreach (render_state_blend_modeType value in Enum.GetValues(typeof(render_state_blend_modeType)))
                {
                    cmbBlendMode.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbDepthFunc))
            {
                foreach (depth_test_funcType value in Enum.GetValues(typeof(depth_test_funcType)))
                {
                    cmbDepthFunc.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbAlphaFunc))
            {
                foreach (alpha_test_funcType value in Enum.GetValues(typeof(alpha_test_funcType)))
                {
                    cmbAlphaFunc.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbColorBlendEquation))
            {
                foreach (color_blend_opType value in Enum.GetValues(typeof(color_blend_opType)))
                {
                    cmbColorBlendEquation.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbColorBlendSrc))
            {
                foreach (color_blend_rgb_funcType value in Enum.GetValues(typeof(color_blend_rgb_funcType)))
                {
                    cmbColorBlendSrc.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbColorBlendDst))
            {
                foreach (color_blend_rgb_funcType value in Enum.GetValues(typeof(color_blend_rgb_funcType)))
                {
                    cmbColorBlendDst.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbAlphaBlendEquation))
            {
                foreach (color_blend_opType value in Enum.GetValues(typeof(color_blend_opType)))
                {
                    cmbAlphaBlendEquation.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbAlphaBlendSrc))
            {
                foreach (color_blend_alpha_funcType value in Enum.GetValues(typeof(color_blend_alpha_funcType)))
                {
                    cmbAlphaBlendSrc.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbAlphaBlendDst))
            {
                foreach (color_blend_alpha_funcType value in Enum.GetValues(typeof(color_blend_alpha_funcType)))
                {
                    cmbAlphaBlendDst.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbBlendLogicOp))
            {
                foreach (logical_blend_opType value in Enum.GetValues(typeof(logical_blend_opType)))
                {
                    cmbBlendLogicOp.AddItem(UIText.EnumValue(value), value);
                }
            }

            using(var ub = new UpdateBlock(cmbBlendPreset))
            {
                foreach (BlendPresetType value in Enum.GetValues(typeof(BlendPresetType)))
                {
                    cmbBlendPreset.AddItem(UIText.EnumValue(value), value);
                }
            }

            cepBlendColor.IsDefaultLinear = ConfigData.ApplicationConfig.Color.GammaCorrection;
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            mlbRenderState.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.mode);
            cmbRenderState.SelectedItemData	= ActiveTarget.Data.render_state.mode;

            bool isModeCustom = ActiveTarget.Data.render_state.mode == render_state_modeType.custom;

//			if (isModeCustom)
            {
                mcbDepthEnable.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.depth_test.enable);
                mcbDepthEnable.Checked			= ActiveTarget.Data.render_state.depth_test.enable;
                mcbDepthWrite.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.depth_test.write);
                mcbDepthWrite.Checked			= ActiveTarget.Data.render_state.depth_test.write;
                mlbDepthFunc.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.depth_test.func);
                cmbDepthFunc.SelectedItemData	= ActiveTarget.Data.render_state.depth_test.func;

                mcbAlphaEnable.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.alpha_test.enable);
                mcbAlphaEnable.Checked			= ActiveTarget.Data.render_state.alpha_test.enable;
                mlbAlphaFunc.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.alpha_test.func);
                cmbAlphaFunc.SelectedItemData	= ActiveTarget.Data.render_state.alpha_test.func;
                mlbAlphaCompareValue.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.alpha_test.value);
                ftbAlphaCompareValue.Value		= ActiveTarget.Data.render_state.alpha_test.value;

                mlbBlendMode.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.blend_mode);
                cmbBlendMode.SelectedItemData			= ActiveTarget.Data.render_state.blend_mode;
                mlbColorBlendEquation.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.color_blend.rgb_op);
                cmbColorBlendEquation.SelectedItemData	= ActiveTarget.Data.render_state.color_blend.rgb_op;
                mgbColorBlendSrc.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.color_blend.rgb_src_func);
                cmbColorBlendSrc.SelectedItemData		= ActiveTarget.Data.render_state.color_blend.rgb_src_func;
                mgbColorBlendDst.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.color_blend.rgb_dst_func);
                cmbColorBlendDst.SelectedItemData		= ActiveTarget.Data.render_state.color_blend.rgb_dst_func;
                mlbAlphaBlendEquation.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.color_blend.alpha_op);
                cmbAlphaBlendEquation.SelectedItemData	= ActiveTarget.Data.render_state.color_blend.alpha_op;
                mgbAlphaBlendSrc.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.color_blend.alpha_src_func);
                cmbAlphaBlendSrc.SelectedItemData		= ActiveTarget.Data.render_state.color_blend.alpha_src_func;
                mgbAlphaBlendDst.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.color_blend.alpha_dst_func);
                cmbAlphaBlendDst.SelectedItemData		= ActiveTarget.Data.render_state.color_blend.alpha_dst_func;
                mgbBlendColor.IsModified = ActiveTarget.IsSequenceValueChanged(x => x.render_state.color_blend.const_color);
                cepBlendColor.Color						= new RgbaColor(
                                                            ActiveTarget.Data.render_state.color_blend.const_color[0],
                                                            ActiveTarget.Data.render_state.color_blend.const_color[1],
                                                            ActiveTarget.Data.render_state.color_blend.const_color[2],
                                                            ActiveTarget.Data.render_state.color_blend.const_color[3]
                                                        );
                mlbBlendLogicOp.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.logical_blend.op);
                cmbBlendLogicOp.SelectedItemData		= ActiveTarget.Data.render_state.logical_blend.op;
            }

            gbxDepthTest.Enabled		= isModeCustom;

            // アルファテストの有効/無効は、opaque、maskの時は触れないようにする。
            mcbAlphaEnable.Enabled = !(ActiveTarget.Data.render_state.mode == render_state_modeType.opaque ||
                                      ActiveTarget.Data.render_state.mode == render_state_modeType.mask);
            // 不透明の場合は、アルファテストが無効なので、アルファテストのグループボックスを無効とします。
            bool isModeOpaque = (ActiveTarget.Data.render_state.mode == render_state_modeType.opaque);
            gbxAlphaTest.Enabled = !isModeOpaque;

            // アルファテストの有効/無効に合わせて、比較条件と比較値の有効/無効を切り替えます。
            cmbAlphaFunc.Enabled = mcbAlphaEnable.Checked;
            fudAlphaCompareValue.Enabled = mcbAlphaEnable.Checked;
            ftbAlphaCompareValue.Enabled = mcbAlphaEnable.Checked;

            // Min, Max のときは係数を使わない
            switch (ActiveTarget.Data.render_state.color_blend.rgb_op)
            {
                case color_blend_opType.min:
                case color_blend_opType.max:
                    cmbColorBlendSrc.Enabled = false;
                    cmbColorBlendDst.Enabled = false;
                    lblColorBlendSrc.Text = res.Strings.MaterialRenderStatePage_SourceColor;
                    lblColorBlendDst.Text = res.Strings.MaterialRenderStatePage_DestinationColor;
                    break;
                default:
                    cmbColorBlendSrc.Enabled = true;
                    cmbColorBlendDst.Enabled = true;
                    lblColorBlendSrc.Text = res.Strings.MaterialRenderStatePage_SourceColorx;
                    lblColorBlendDst.Text = res.Strings.MaterialRenderStatePage_DestinationColorx;
                    break;
            }

            switch (ActiveTarget.Data.render_state.color_blend.alpha_op)
            {
                case color_blend_opType.min:
                case color_blend_opType.max:
                    cmbAlphaBlendSrc.Enabled = false;
                    cmbAlphaBlendDst.Enabled = false;
                    lblAlphaBlendSrc.Text = res.Strings.MaterialRenderStatePage_SourceAlpha;
                    lblAlphaBlendDst.Text = res.Strings.MaterialRenderStatePage_DestinationAlpha;
                    break;
                default:
                    cmbAlphaBlendSrc.Enabled = true;
                    cmbAlphaBlendDst.Enabled = true;
                    lblAlphaBlendSrc.Text = res.Strings.MaterialRenderStatePage_SourceAlphax;
                    lblAlphaBlendDst.Text = res.Strings.MaterialRenderStatePage_DestinationAlphax;
                    break;
            }


            // 深度テストの有効/無効は、customの時以外は触れないようにする。
            mcbDepthEnable.Enabled = isModeCustom;
            // 深度テストの深度書き込みは、customの時以外は触れないようにする。
            mcbDepthWrite.Enabled = isModeCustom;
            // 深度テストの関数は、customの時以外は触れないようにする。
            cmbDepthFunc.Enabled = isModeCustom;

            // 深度テストの有効/無効に合わせて、比較条件と書き込みの有効/無効を切り替えます。
            cmbDepthFunc.Enabled = mcbDepthEnable.Checked;
            mcbDepthWrite.Enabled = mcbDepthEnable.Checked;

            cmbBlendMode.Enabled = isModeCustom;
            mlbBlendMode.Enabled = isModeCustom;

            // 不透明、抜きではブレンドモードがブレンドなしですので、「カラーブレンド」「アルファブレンド」「ブレンドカラー」の編集 UI は無効
            bool isBlendNone = (ActiveTarget.Data.render_state.mode == render_state_modeType.opaque ||
                                ActiveTarget.Data.render_state.mode == render_state_modeType.mask);
            grpColorBlend.Enabled = !isBlendNone && (ActiveTarget.Data.render_state.blend_mode == render_state_blend_modeType.color);
            grpAlphaBlend.Enabled = !isBlendNone && (ActiveTarget.Data.render_state.blend_mode == render_state_blend_modeType.color);
            mgbBlendColor.Enabled = !isBlendNone && (ActiveTarget.Data.render_state.blend_mode == render_state_blend_modeType.color);
            grpLogicOperation.Enabled = isModeCustom && (ActiveTarget.Data.render_state.blend_mode == render_state_blend_modeType.logic);

            // ブレンドプリセット
            {
                cmbBlendPreset.SelectedItemData = MakeBlendPreset();
                lblBlendPreset.Enabled = ActiveTarget.Data.render_state.blend_mode == render_state_blend_modeType.color;
                cmbBlendPreset.Enabled = ActiveTarget.Data.render_state.blend_mode == render_state_blend_modeType.color;
            }

            {
                OperatorLabel colorLabel = operatorLabels[ActiveTarget.Data.render_state.color_blend.rgb_op];
                OperatorLabel alphaLabel = operatorLabels[ActiveTarget.Data.render_state.color_blend.alpha_op];

                lblColorOp0.Text = colorLabel[0];
                lblColorOp1.Text = colorLabel[1];
                lblColorOp2.Text = colorLabel[2];
                lblAlphaOp0.Text = alphaLabel[0];
                lblAlphaOp1.Text = alphaLabel[1];
                lblAlphaOp2.Text = alphaLabel[2];
            }
        }

        public static bool IsModified(Material activeTarget)
        {
            return activeTarget != null &&
                (activeTarget.IsValueChanged(x => x.render_state.mode) ||
                activeTarget.IsValueChanged(x => x.render_state.depth_test.enable) ||
                activeTarget.IsValueChanged(x => x.render_state.depth_test.write) ||
                activeTarget.IsValueChanged(x => x.render_state.depth_test.func) ||
                activeTarget.IsValueChanged(x => x.render_state.alpha_test.enable) ||
                activeTarget.IsValueChanged(x => x.render_state.alpha_test.func) ||
                activeTarget.IsValueChanged(x => x.render_state.alpha_test.value) ||
                activeTarget.IsValueChanged(x => x.render_state.blend_mode) ||
                activeTarget.IsValueChanged(x => x.render_state.color_blend.rgb_op) ||
                activeTarget.IsValueChanged(x => x.render_state.color_blend.rgb_src_func) ||
                activeTarget.IsValueChanged(x => x.render_state.color_blend.rgb_dst_func) ||
                activeTarget.IsValueChanged(x => x.render_state.color_blend.alpha_op) ||
                activeTarget.IsValueChanged(x => x.render_state.color_blend.alpha_src_func) ||
                activeTarget.IsValueChanged(x => x.render_state.color_blend.alpha_dst_func) ||
                activeTarget.IsSequenceValueChanged(x => x.render_state.color_blend.const_color) ||
                activeTarget.IsValueChanged(x => x.render_state.logical_blend.op));
        }

        private BlendPresetType MakeBlendPreset()
        {
            var current = new BlendPreset(
                ActiveTarget.Data.render_state.color_blend.rgb_op,
                ActiveTarget.Data.render_state.color_blend.rgb_src_func,
                ActiveTarget.Data.render_state.color_blend.rgb_dst_func,
                ActiveTarget.Data.render_state.color_blend.alpha_op,
                ActiveTarget.Data.render_state.color_blend.alpha_src_func,
                ActiveTarget.Data.render_state.color_blend.alpha_dst_func
            );

            foreach (var preset in blendPresets)
            {
                if (preset.Value == current)
                {
                    return preset.Key;
                }
            }

            return BlendPresetType.Custom;
        }

        #region コマンド
        private static GroupEditCommand CreateEditCommand_render_state_mode(GuiObjectGroup targets, render_state_modeType mode, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<render_state_modeType>(
                    targets,
                    GuiObjectID.Material,
                    mode,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.mode;
                        material.Data.render_state.mode = (render_state_modeType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialRenderStateMode.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_depth_test_enable(GuiObjectGroup targets, bool enable, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.Material,
                    enable,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.depth_test.enable;
                        material.Data.render_state.depth_test.enable = (bool)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialDepthTestEnable.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_depth_test_write(GuiObjectGroup targets, bool write, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.Material,
                    write,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.depth_test.write;
                        material.Data.render_state.depth_test.write = (bool)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialDepthWriteEnable.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }


        private static GroupEditCommand CreateEditCommand_render_state_depth_test_func(GuiObjectGroup targets, depth_test_funcType func, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<depth_test_funcType>(
                    targets,
                    GuiObjectID.Material,
                    func,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.depth_test.func;
                        material.Data.render_state.depth_test.func = (depth_test_funcType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialDepthFunc.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_alpha_test_enable(GuiObjectGroup targets, bool enable, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.Material,
                    enable,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.alpha_test.enable;
                        material.Data.render_state.alpha_test.enable = (bool)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialAlphaTestEnable.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_alpha_test_func(GuiObjectGroup targets, alpha_test_funcType func, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<alpha_test_funcType>(
                    targets,
                    GuiObjectID.Material,
                    func,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.alpha_test.func;
                        material.Data.render_state.alpha_test.func = (alpha_test_funcType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialAlphaFunc.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_alpha_test_value(GuiObjectGroup targets, float value)
        {
            return
                new GeneralGroupValueEditCommand<float>(
                    targets,
                    GuiObjectID.Material,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.alpha_test.value;
                        material.Data.render_state.alpha_test.value = (float)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialAlphaRefValue.Send(editTargets, data, 0xFFFFFFFF, 0xFFFFFFFF);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_blend_mode(GuiObjectGroup targets, render_state_blend_modeType mode, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<render_state_blend_modeType>(
                    targets,
                    GuiObjectID.Material,
                    mode,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.blend_mode;
                        material.Data.render_state.blend_mode = (render_state_blend_modeType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialBlendMode.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_color_blend_rgb_op(GuiObjectGroup targets, color_blend_opType op, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<color_blend_opType>(
                    targets,
                    GuiObjectID.Material,
                    op,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.color_blend.rgb_op;
                        material.Data.render_state.color_blend.rgb_op = (color_blend_opType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialColorCombine.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_color_blend_rgb_src_func(GuiObjectGroup targets, color_blend_rgb_funcType func, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<color_blend_rgb_funcType>(
                    targets,
                    GuiObjectID.Material,
                    func,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.color_blend.rgb_src_func;
                        material.Data.render_state.color_blend.rgb_src_func = (color_blend_rgb_funcType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialColorSrcBlend.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_color_blend_rgb_dst_func(GuiObjectGroup targets, color_blend_rgb_funcType func, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<color_blend_rgb_funcType>(
                    targets,
                    GuiObjectID.Material,
                    func,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.color_blend.rgb_dst_func;
                        material.Data.render_state.color_blend.rgb_dst_func = (color_blend_rgb_funcType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialColorDstBlend.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_blend_control_alpha_op(GuiObjectGroup targets, color_blend_opType op, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<color_blend_opType>(
                    targets,
                    GuiObjectID.Material,
                    op,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.color_blend.alpha_op;
                        material.Data.render_state.color_blend.alpha_op = (color_blend_opType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialAlphaCombine.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }


        private static GroupEditCommand CreateEditCommand_render_state_blend_control_alpha_src_func(GuiObjectGroup targets, color_blend_alpha_funcType func, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<color_blend_alpha_funcType>(
                    targets,
                    GuiObjectID.Material,
                    func,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.color_blend.alpha_src_func;
                        material.Data.render_state.color_blend.alpha_src_func = (color_blend_alpha_funcType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialAlphaSrcBlend.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_blend_control_alpha_dst_func(GuiObjectGroup targets, color_blend_alpha_funcType func, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<color_blend_alpha_funcType>(
                    targets,
                    GuiObjectID.Material,
                    func,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.color_blend.alpha_dst_func;
                        material.Data.render_state.color_blend.alpha_dst_func = (color_blend_alpha_funcType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialAlphaDstBlend.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_color_blend_const_color(GuiObjectGroup targets, float[] color, uint modifiedPropertyElements)
        {
            return
                new GeneralGroupReferenceEditCommand<float[]>(
                    targets,
                    GuiObjectID.Material,
                    ObjectUtility.MultipleClone(color, targets.Objects.Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.color_blend.const_color;

                        var newData = ObjectUtility.Clone((float[])data);

                        for(int i = 0;i != 4;++ i)
                        {
                            if ((modifiedPropertyElements & (1 << i)) == 0)
                            {
                                newData[i] = material.Data.render_state.color_blend.const_color[i];
                            }
                        }

                        material.Data.render_state.color_blend.const_color = newData;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        var colors = data.Select(
                            x =>
                            {
                                var c = (float[])x;

                                return (object)(new RgbaColor(c[0], c[1], c[2], c[3]));
                            }
                        );

                        Viewer.SetMaterialConstantColor.Send(editTargets, colors.ToArray(), 0xFFFFFFFF, 0xFFFFFFFF);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_render_state_logical_blend_op(GuiObjectGroup targets, logical_blend_opType op, bool reloadModel)
        {
            return
                new GeneralGroupValueEditCommand<logical_blend_opType>(
                    targets,
                    GuiObjectID.Material,
                    op,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.logical_blend.op;
                        material.Data.render_state.logical_blend.op = (logical_blend_opType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialLogicOp.Send(editTargets, data, editTargets.OfType<Material>().Select(NeedToReload).ToArray(), reloadModel);
                    }
                );
        }
        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            public render_state_modeType		render_state_mode{								get; set; }
            public bool							render_state_depth_test_enable{					get; set; }
            public bool							render_state_depth_test_write{					get; set; }
            public depth_test_funcType			render_state_depth_test_func{					get; set; }
            public bool							render_state_alpha_test_enable{					get; set; }
            public alpha_test_funcType 			render_state_alpha_test_func{					get; set; }
            public float						render_state_alpha_test_value{					get; set; }
            public render_state_blend_modeType	render_state_blend_mode{						get; set; }
            public color_blend_opType			render_state_color_blend_rgb_op{		get; set; }
            public color_blend_rgb_funcType		render_state_color_blend_rgb_src_func{	get; set; }
            public color_blend_rgb_funcType		render_state_color_blend_rgb_dst_func{	get; set; }
            public color_blend_opType			render_state_blend_control_alpha_op{			get; set; }
            public color_blend_alpha_funcType	render_state_blend_control_alpha_src_func{		get; set; }
            public color_blend_alpha_funcType	render_state_blend_control_alpha_dst_func{		get; set; }
            public float[]						render_state_color_blend_const_color{	get; set; }
            public logical_blend_opType			render_state_logical_blend_op{			get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(Material target)
        {
            return
                new CopyData()
                {
                    render_state_mode								= target.Data.render_state.mode,
                    render_state_depth_test_enable					= target.Data.render_state.depth_test.enable,
                    render_state_depth_test_write					= target.Data.render_state.depth_test.write,
                    render_state_depth_test_func					= target.Data.render_state.depth_test.func,
                    render_state_alpha_test_enable					= target.Data.render_state.alpha_test.enable,
                    render_state_alpha_test_func					= target.Data.render_state.alpha_test.func,
                    render_state_alpha_test_value					= target.Data.render_state.alpha_test.value,
                    render_state_blend_mode							= target.Data.render_state.blend_mode,
                    render_state_color_blend_rgb_op			= target.Data.render_state.color_blend.rgb_op,
                    render_state_color_blend_rgb_src_func	= target.Data.render_state.color_blend.rgb_src_func,
                    render_state_color_blend_rgb_dst_func	= target.Data.render_state.color_blend.rgb_dst_func,
                    render_state_blend_control_alpha_op				= target.Data.render_state.color_blend.alpha_op,
                    render_state_blend_control_alpha_src_func		= target.Data.render_state.color_blend.alpha_src_func,
                    render_state_blend_control_alpha_dst_func		= target.Data.render_state.color_blend.alpha_dst_func,
                    render_state_color_blend_const_color	= ObjectUtility.Clone(target.Data.render_state.color_blend.const_color),
                    render_state_logical_blend_op			= target.Data.render_state.logical_blend.op,
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject, true));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject, bool reloadModel)
        {
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_render_state_mode(									targets, copyData.render_state_mode, false));
                commandSet.Add(CreateEditCommand_render_state_depth_test_enable(					targets, copyData.render_state_depth_test_enable, false));
                commandSet.Add(CreateEditCommand_render_state_depth_test_write(						targets, copyData.render_state_depth_test_write, false));
                commandSet.Add(CreateEditCommand_render_state_depth_test_func(						targets, copyData.render_state_depth_test_func, false));
                commandSet.Add(CreateEditCommand_render_state_alpha_test_enable(					targets, copyData.render_state_alpha_test_enable, false));
                commandSet.Add(CreateEditCommand_render_state_alpha_test_func(						targets, copyData.render_state_alpha_test_func, false));
                commandSet.Add(CreateEditCommand_render_state_alpha_test_value(						targets, copyData.render_state_alpha_test_value));
                commandSet.Add(CreateEditCommand_render_state_blend_mode(							targets, copyData.render_state_blend_mode, false));
                commandSet.Add(CreateEditCommand_render_state_color_blend_rgb_op(			targets, copyData.render_state_color_blend_rgb_op, false));
                commandSet.Add(CreateEditCommand_render_state_color_blend_rgb_src_func(		targets, copyData.render_state_color_blend_rgb_src_func, false));
                commandSet.Add(CreateEditCommand_render_state_color_blend_rgb_dst_func(		targets, copyData.render_state_color_blend_rgb_dst_func, false));
                commandSet.Add(CreateEditCommand_render_state_blend_control_alpha_op(				targets, copyData.render_state_blend_control_alpha_op, false));
                commandSet.Add(CreateEditCommand_render_state_blend_control_alpha_src_func(			targets, copyData.render_state_blend_control_alpha_src_func, false));
                commandSet.Add(CreateEditCommand_render_state_blend_control_alpha_dst_func(			targets, copyData.render_state_blend_control_alpha_dst_func, false));
                commandSet.Add(CreateEditCommand_render_state_color_blend_const_color(		targets, ObjectUtility.Clone(copyData.render_state_color_blend_const_color), 0xFFFFFFFF));
                commandSet.Add(CreateEditCommand_render_state_logical_blend_op(				targets, copyData.render_state_logical_blend_op, false));
            }
            var materials = targets.Objects.OfType<Material>().ToArray();
            commandSet.OnPostEdit += (s, e) =>
                {
                    if (reloadModel)
                    {
                        Viewer.SetMaterialRenderStateAllOrReload.Send(materials.Where(NeedToReload).ToArray());
                    }
                };
            return commandSet.Execute();
        }
        #endregion


        private void SetCmdAlphaEnable(render_state_modeType modeType, EditCommandSet commandSet)
        {
            // <render_state> mode が opaque であれば false を、 mask であれば true を指定します
            //bool alphaEnableOld = mcbAlphaEnable.Checked;
            bool alphaEnable = false;
            switch (modeType)
            {
                case render_state_modeType.opaque:
                    alphaEnable = false;
                    break;
                case render_state_modeType.mask:
                    alphaEnable = true;
                    break;
                case render_state_modeType.translucent:
                case render_state_modeType.custom:
                    return;
            }
            commandSet.Add(CreateEditCommand_render_state_alpha_test_enable(Targets, alphaEnable, false));
        }

        private void SetCmdDepthEnable(render_state_modeType modeType, EditCommandSet commandSet)
        {
            // <render_state> mode が custom でなければ true を指定します
            if (modeType != render_state_modeType.custom)
            {
                commandSet.Add(CreateEditCommand_render_state_depth_test_enable(Targets, true, false));
            }
        }

        private void SetCmdDepthWrite(render_state_modeType modeType, EditCommandSet commandSet)
        {
            // <render_state> mode が opaque か mask であれば true を、 translucent であれば false を指定します
            bool depthWrite = false;
            switch (modeType)
            {
                case render_state_modeType.opaque:
                case render_state_modeType.mask:
                    depthWrite = true;
                    break;
                case render_state_modeType.translucent:
                    depthWrite = false;
                    break;
                case render_state_modeType.custom:
                    return;
            }
            commandSet.Add(CreateEditCommand_render_state_depth_test_write(Targets, depthWrite, false));
        }

        private void SetCmdDepthFunc(render_state_modeType modeType, EditCommandSet commandSet)
        {
            // <render_state> mode が custom でなければ true を指定します
            if (modeType != render_state_modeType.custom)
            {
                commandSet.Add(CreateEditCommand_render_state_depth_test_func(Targets, depth_test_funcType.lequal, false));
            }
        }

        private render_state_blend_modeType SetCmdBlendMode(render_state_modeType modeType, EditCommandSet commandSet)
        {
            // <render_state> mode が opaque か mask であれば none を、 translucent であれば color を指定します
            var blendModeType = (render_state_blend_modeType)cmbBlendMode.SelectedItemData;
            switch (modeType)
            {
                case render_state_modeType.opaque:
                case render_state_modeType.mask:
                    blendModeType = render_state_blend_modeType.none;
                    break;
                case render_state_modeType.translucent:
                    blendModeType = render_state_blend_modeType.color;
                    break;
                case render_state_modeType.custom:
                    return blendModeType;
            }

            commandSet.Add(CreateEditCommand_render_state_blend_mode(Targets, blendModeType, false));
            return blendModeType;
        }

        private void SetCmdBlendLogicOp(render_state_modeType modeType, render_state_blend_modeType blendModeType, EditCommandSet commandSet)
        {
            // <render_state> mode が custom でなければ、copy を指定します
            //logical_blend_opType logicalblendOpTypeOld = (logical_blend_opType)cmbBlendLogicOp.SelectedItemData;
            //logical_blend_opType logicalblendOpType = logicalblendOpTypeOld;

            if (modeType == render_state_modeType.custom)
            {
                // モードがカスタムの場合、
                if (blendModeType == render_state_blend_modeType.logic)
                {
                    return;
                }
            }

            // ブレンドモードが論理ブレンドでない場合、ソースからーに変更する。
            var logicalblendOpType = logical_blend_opType.copy;

            commandSet.Add(CreateEditCommand_render_state_logical_blend_op(Targets, logicalblendOpType, false));
        }

        private void cmbRenderState_SelectedIndexChanged(object sender, EventArgs e)
        {
            var commandSet = new EditCommandSet();
            {
                var modeType = (render_state_modeType)cmbRenderState.SelectedItemData;

                SetCmdAlphaEnable(modeType, commandSet);
                SetCmdDepthEnable(modeType, commandSet);
                SetCmdDepthWrite(modeType, commandSet);
                SetCmdDepthFunc(modeType, commandSet);
                render_state_blend_modeType newBlendMode = SetCmdBlendMode(modeType, commandSet);
                SetCmdBlendLogicOp(modeType, newBlendMode, commandSet);

                commandSet.Add(CreateEditCommand_render_state_mode(Targets, modeType, false));
            }

            var materials = Targets.Objects.OfType<Material>().ToArray();
            commandSet.OnPostEdit += (s, args) =>
            {
                Viewer.SetMaterialRenderStateAllOrReload.Send(materials.Where(NeedToReload).ToArray());
            };
            TheApp.CommandManager.Execute(commandSet);
        }

        private GuiObjectGroup CustomModeGroup()
        {
            var editables =
                Targets.Objects.OfType<Material>()
                    .Where(x => x.Data.render_state.mode == render_state_modeType.custom);
            return new GuiObjectGroup(editables);
        }

        private GuiObjectGroup DepthEnabledGroup()
        {
            var editables =
                Targets.Objects.OfType<Material>()
                    .Where(x => x.Data.render_state.mode == render_state_modeType.custom)
                    .Where(x => x.Data.render_state.depth_test.enable);
            return new GuiObjectGroup(editables);
        }


        private void cbxDepthEnable_CheckedChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_depth_test_enable(CustomModeGroup(), mcbDepthEnable.Checked, true));
        }


        private void cbxDepthWrite_CheckedChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_depth_test_write(DepthEnabledGroup(), mcbDepthWrite.Checked, true));
        }

        private void cmbDepthFunc_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_depth_test_func(DepthEnabledGroup(), (depth_test_funcType)cmbDepthFunc.SelectedItemData, true));
        }

        private void cbxAlphaEnable_CheckedChanged(object sender, EventArgs e)
        {
            // アルファテストの有効/無効は、opaque、maskの時は触れないようにする。
            var editables =
                Targets.Objects.OfType<Material>()
                    .Where(
                        x =>
                            x.Data.render_state.mode != render_state_modeType.opaque &&
                            x.Data.render_state.mode != render_state_modeType.mask);
            var group = new GuiObjectGroup(editables);
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_alpha_test_enable(group, mcbAlphaEnable.Checked, true));
        }

        private GuiObjectGroup AlphaEditableGroup()
        {
            var editables =
                Targets.Objects.OfType<Material>()
                    .Where(x => x.Data.render_state.alpha_test.enable);
            return new GuiObjectGroup(editables);
        }

        private void cmbAlphaFunc_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_alpha_test_func(AlphaEditableGroup(), (alpha_test_funcType)cmbAlphaFunc.SelectedItemData, true));
        }


        private void ftbAlphaCompareValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {
                Viewer.SetMaterialAlphaRefValue.Send(AlphaEditableGroup(), ftbAlphaCompareValue.Value, 0xFFFFFFFF, 0xFFFFFFFF);
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_render_state_alpha_test_value(AlphaEditableGroup(), ftbAlphaCompareValue.Value));
            }
        }

        private void cmbBlendMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            var commandSet = new EditCommandSet();
            {
                var blendModeType = (render_state_blend_modeType)cmbBlendMode.SelectedItemData;
                commandSet.Add(CreateEditCommand_render_state_blend_mode(CustomModeGroup(), blendModeType, false));
                var modeType = (render_state_modeType)cmbRenderState.SelectedItemData;
                SetCmdBlendLogicOp(modeType, blendModeType, commandSet);
            }

            var materials = CustomModeGroup().Objects.OfType<Material>().ToArray();
            commandSet.OnPostEdit += (s, args) =>
            {
                Viewer.SetMaterialRenderStateAllOrReload.Send(materials.Where(NeedToReload).ToArray());
            };
            TheApp.CommandManager.Execute(commandSet);
        }

        private IEnumerable<Material> ColorBlendEditables()
        {
            var editables =
                Targets.Objects.OfType<Material>()
                    .Where(
                        x =>
                            x.Data.render_state.mode != render_state_modeType.opaque &&
                            x.Data.render_state.mode != render_state_modeType.mask)
                    .Where(x => x.Data.render_state.blend_mode == render_state_blend_modeType.color);
            return editables;
        }

        private void cmbColorBlendEquation_SelectedIndexChanged(object sender, EventArgs e)
        {
            var group = new GuiObjectGroup(ColorBlendEditables());

            TheApp.CommandManager.Execute(CreateEditCommand_render_state_color_blend_rgb_op(group, (color_blend_opType)cmbColorBlendEquation.SelectedItemData, true));
        }

        private GuiObjectGroup ColorBlendSrcDstEditableGroup()
        {
            var editables =
                ColorBlendEditables()
                    .Where(
                        x =>
                            x.Data.render_state.color_blend.rgb_op != color_blend_opType.min &&
                            x.Data.render_state.color_blend.rgb_op != color_blend_opType.max);
            var group = new GuiObjectGroup(editables);
            return group;
        }

        private void cmbColorBlendSrc_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_color_blend_rgb_src_func(ColorBlendSrcDstEditableGroup(), (color_blend_rgb_funcType)cmbColorBlendSrc.SelectedItemData, true));
        }

        private void cmbColorBlendDst_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_color_blend_rgb_dst_func(ColorBlendSrcDstEditableGroup(), (color_blend_rgb_funcType)cmbColorBlendDst.SelectedItemData, true));
        }

        private GuiObjectGroup AlphaBlendSrcDstEditableGroup()
        {
            var editables =
                ColorBlendEditables()
                    .Where(
                        x =>
                            x.Data.render_state.color_blend.alpha_op != color_blend_opType.min &&
                            x.Data.render_state.color_blend.alpha_op != color_blend_opType.max);
            var group = new GuiObjectGroup(editables);
            return group;
        }

        private void cmbAlphaBlendEquation_SelectedIndexChanged(object sender, EventArgs e)
        {
            var group = new GuiObjectGroup(ColorBlendEditables());

            TheApp.CommandManager.Execute(CreateEditCommand_render_state_blend_control_alpha_op(group, (color_blend_opType)cmbAlphaBlendEquation.SelectedItemData, true));
        }

        private void cmbAlphaBlendSrc_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_blend_control_alpha_src_func(AlphaBlendSrcDstEditableGroup(), (color_blend_alpha_funcType)cmbAlphaBlendSrc.SelectedItemData, true));
        }

        private void cmbAlphaBlendDst_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_render_state_blend_control_alpha_dst_func(AlphaBlendSrcDstEditableGroup(), (color_blend_alpha_funcType)cmbAlphaBlendDst.SelectedItemData, true));
        }

        private void cepBlendColor_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var group = new GuiObjectGroup(ColorBlendEditables());

            uint modifiedPropertyElements = e.ElementBits;	// RGBAのどの要素が変更になったか？
            if (e.Changing)
            {
                Viewer.SetMaterialConstantColor.Send(group, cepBlendColor.Color, 0xFFFFFFFF, modifiedPropertyElements);
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_render_state_color_blend_const_color(group, cepBlendColor.Color.ToArray(), modifiedPropertyElements));
            }
        }

        private void cmbBlendLogicOp_SelectedIndexChanged(object sender, EventArgs e)
        {
            var editables =
                Targets.Objects.OfType<Material>()
                    .Where(x => x.Data.render_state.mode == render_state_modeType.custom)
                    .Where(x => x.Data.render_state.blend_mode == render_state_blend_modeType.logic);
            var group = new GuiObjectGroup(editables);

            TheApp.CommandManager.Execute(CreateEditCommand_render_state_logical_blend_op(group, (logical_blend_opType)cmbBlendLogicOp.SelectedItemData, true));
        }

        private void cmbBlendPreset_SelectedIndexChanged(object sender, EventArgs e)
        {
            var presetType = (BlendPresetType)cmbBlendPreset.SelectedItemData;
            if (presetType != BlendPresetType.Custom)
            {
                Debug.Assert(blendPresets.ContainsKey(presetType));

                var preset = blendPresets[presetType];

                var editables =
                    Targets.Objects.OfType<Material>()
                        .Where(x =>
                                x.Data.render_state.mode == render_state_modeType.custom ||
                                x.Data.render_state.mode == render_state_modeType.translucent)
                        .Where(x => x.Data.render_state.blend_mode == render_state_blend_modeType.color);
                var group = new GuiObjectGroup(editables);

                var commandSet = new EditCommandSet();
                {
                    commandSet.Add(CreateEditCommand_render_state_color_blend_rgb_op(           group, preset.ColorBlendOp, false));
                    commandSet.Add(CreateEditCommand_render_state_color_blend_rgb_src_func(     group, preset.ColorSrc,     false));
                    commandSet.Add(CreateEditCommand_render_state_color_blend_rgb_dst_func(     group, preset.ColorDst,     false));
                    commandSet.Add(CreateEditCommand_render_state_blend_control_alpha_op(       group, preset.AlphaBlendOp, false));
                    commandSet.Add(CreateEditCommand_render_state_blend_control_alpha_src_func( group, preset.AlphaSrc,     false));
                    commandSet.Add(CreateEditCommand_render_state_blend_control_alpha_dst_func( group, preset.AlphaDst,     false));
                }

                //var materials = Targets.Objects.OfType<Material>().ToArray();
                commandSet.OnPostEdit += (s, args) =>
                {
                    Viewer.SetMaterialRenderStateAllOrReload.Send(editables.Where(NeedToReload).ToArray());
                };
                TheApp.CommandManager.Execute(commandSet);
            }
        }
    }
}
