﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class ObjectPropertyDialog : ModelessDialog
    {
        // ダイアログリスト
        private static readonly List<ObjectPropertyDialog> dialogs_ = new List<ObjectPropertyDialog>();

        // 親ダイアログ
        private ObjectPropertyDialog parent_;

        // アクティブパネル
        private ObjectPropertyPanel activePanel_;

        // ターゲット
        private GuiObjectGroup targets_ = new GuiObjectGroup();

        // パネル配列
        private readonly ObjectPropertyPanel[] propertyPanels_ = new ObjectPropertyPanel[Enum.GetValues(typeof(PanelID)).Length];

        // コピーオブジェクト
        private static object copyObject_ = null;
        // コピーオブジェクトＩＤ
        private static PropertyPageID copyObjectID_ = PropertyPageID.Null;
        // コピーオブジェクト情報
        private static object copyObjectInfo_ = null;

        // ダイアログのデフォルト値
        private const int defaultWidth		= 560;
        private const int defaultHeight		= 416;
        private const int defaultLeftWidth	= 178;

        private readonly EventHandler objSelectedChangedHandler;

        // プロパティで表示するヘルプページ
        private HelpUtility.PageKey helpPageKey;

        // パネルＩＤ
        public enum PanelID
        {
            Null,
            //
            Project,
            Model,
            Texture,
            Material,
            Shape,
            SkeletalAnimation,
            ShapeAnimation,
            MaterialAnimation,
            ShaderParameterAnimation,
            ColorAnimation,
            TextureSrtAnimation,
            MaterialVisibilityAnimation,
            BoneVisibilityAnimation,
            TexturePatternAnimation,
            SceneAnimation,
            CameraAnimation,
            LightAnimation,
            FogAnimation,
            ShaderDefinition,
            Bone,
        }

        /// <summary>
        /// プロパティダイアログ表示。
        /// </summary>
        public static void ShowPropertyDialog()
        {
            bool activated = false;
            // ターゲット固定されていないダイアログがあればそれをアクティブにするだけ
            foreach (var dialog in dialogs_)
            {
                if (Screen.AllScreens.All(x => !x.Bounds.IntersectsWith(dialog.Bounds)))
                {
                    dialog.Location = new Point(0, 0);
                }

                if (!dialog.TargetFixed)
                {
                    using (new ObjectPropertyDialog.InternallyChangingSize())
                    {
                        if (dialog.WindowState == FormWindowState.Minimized)
                        {
                            dialog.WindowState = dialog.LastWindowStateWithoutMinimized;
                        }
                        if (!activated)
                        {
                            activated = true;
                            dialog.Activate();
                        }
                    }
                }
            }

            if (activated)
            {
                return;
            }

            // 新規作成して表示
            using (var wait = new WaitCursor())
            {
                ObjectPropertyDialog dialog = new ObjectPropertyDialog(null);
                dialog.Show();
            }
        }

        /// <summary>
        // プロパティで表示するヘルプページ。
        /// </summary>
        public HelpUtility.PageKey HelpPageKey
        {
            get { return helpPageKey; }
            set
            {
                helpPageKey = value;
            }
        }

        /// <summary>
        /// ターゲット。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public GuiObjectGroup Targets
        {
            get { return targets_; }
            set
            {
                // コピーを作成して設定
                // 参照をそのまま設定しては駄目
                targets_ = new GuiObjectGroup(value);
                UpdateTargetBox(targets_);

                // フォーム更新
                if (Visible)
                {
                    if (TargetChanged != null)
                    {
                        TargetChanged();
                    }
                    UpdateForm(true);
                }
            }
        }

        public event Action TargetChanged;
        /// <summary>
        /// ターゲットの固定。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool TargetFixed
        {
            get { return tsiTargetFixed.Checked; }
            set
            {
                tsiTargetFixed.Checked = value;
                UpdateTargetFixedState();
            }
        }

        /// <summary>
        /// コピーボタンの有効状態。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool CopyButtonEnabled
        {
            get { return tsiCopy.Enabled; }
        }

        /// <summary>
        /// ペーストボタンの有効状態。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool PasteButtonEnabled
        {
            get { return tsiPaste.Enabled; }
        }

        // 強制的に編集中のものを確定させる
        public static void ForceFixEditting()
        {
            foreach (var dialog in dialogs_)
            {
                // フォーカスを持っているコントロールを探してくる
                var focusedControl = dialog.ActiveControl;
                {
                    while(true)
                    {
                        var nextFocusedControl = focusedControl as ContainerControl;
                        if (nextFocusedControl != null)
                        {
                            focusedControl = nextFocusedControl.ActiveControl;
                        }
                        else
                        {
                            break;
                        }
                    }
                }

                if (focusedControl != null)
                {
                    var parent = focusedControl.Parent as ContainerControl;
                    if (parent != null)
                    {
                        // エディット系のフォーカースが外れたイベントで編集を確定させている動作を擬似的に発生させる。
                        parent.ActiveControl = null;
                        parent.ActiveControl = focusedControl;
                    }
                }
            }
        }

        public ObjectPropertyDialog(ObjectPropertyDialog parent, GuiObjectGroup targets = null)
        {
            InitializeComponent();

            Owner = TheApp.MainFrame;

            // ターゲットボックス幅を初期化
            UpdateTargetBoxSize();

            // 親なし（新規表示用）
            if (parent == null)
            {
                if (targets != null)
                {
                    Targets = targets;
                }
                else
                {
                    Targets = App.AppContext.SelectedTarget;
                }

                // サイズ設定
                Size = new Size(
                    ConfigData.ApplicationConfig.Setting.PropertyEdit.PropertyPageSize.Width,
                    ConfigData.ApplicationConfig.Setting.PropertyEdit.PropertyPageSize.Height);

                // 位置設定
                if (dialogs_.Count == 0)
                {
                    var rcParent = TheApp.MainFrame.DesktopBounds;

                    DesktopLocation = ConfigData.ApplicationConfig.Setting.PropertyEdit.Location;

                    // デスクトップ外に出ていたら戻す
                    {
                        if (!InScreen(DesktopLocation, Size))
                        {
                            DesktopLocation = new Point(
                                rcParent.X + (rcParent.Width - Size.Width) / 2,
                                rcParent.Y + (rcParent.Height - Size.Height) / 2
                                );
                        }
                    }
                }
                else
                {
                    SetLocationNextTo(dialogs_[dialogs_.Count - 1]);
                }
            }
            // 親あり（複製表示用）
            else
            {
                if (targets != null)
                {
                    Targets = targets;
                }
                else
                {
                    Targets = parent.Targets;
                }

                Size			= parent.Size;
                TargetFixed	= parent.TargetFixed;

                // 位置設定
                SetLocationNextTo(parent);
            }

            // 親を保持するのはパネル登録で利用する為
            parent_ = parent;
            dialogs_.Add(this);

            var anim = Targets.Active as AnimationDocument;
            if (anim != null)
            {
                // マテリアルに関係するかして対象マテリアルのみ表示の場合
                if (anim.IsMaterialRelated && ApplicationConfig.Setting.CurveEditor.IsOnlyTargetShowNode)
                {
                    TargetFixed = true;
                }
            }

            objSelectedChangedHandler  = Event_GuiObjectManager_SelectedChanged;
            DocumentAddOrRemovedEventHandler AddedOrRemovedHandler = (o, a, r, s, reloadedDocuments) =>
            {
                if (Targets != null)
                {
                    if (s.Any() || r.Any() || reloadedDocuments.Any())
                    {
                        foreach (var form in OwnedForms)
                        {
                            form.DialogResult = System.Windows.Forms.DialogResult.Cancel;
                        }

                        var curTargets = TargetFixed ? Targets : new GuiObjectGroup(Targets);

                        {
                            foreach (var swap in s)
                            {
                                curTargets.Swap(swap.Key, swap.Value);
                            }

                            foreach (var reloaded in reloadedDocuments)
                            {
                                foreach (var target in curTargets.Objects.ToArray())
                                {
                                    if (target.OwnerDocument == reloaded)
                                    {
                                        var newContent =
                                            reloaded.ContentObjects.FirstOrDefault(
                                                x => x.ObjectID == target.ObjectID && x.Name == target.Name);
                                        if (newContent != null)
                                        {
                                            curTargets.Swap(target, newContent);
                                        }
                                        else
                                        {
                                            curTargets.Remove(target);
                                        }
                                    }
                                }
                            }

                            foreach (var removed in curTargets.Objects.Where(x => r.Contains(x.OwnerDocument)).ToArray())
                            {
                                curTargets.Remove(removed);
                            }

                            if (curTargets.IsEmpty)
                            {
                                Close();
                                return;
                            }
                        }
                    }
                }

                UpdateCopyPasteButton();
            };

            DocumentPropertyChangedEventHandler PropertyChangedHandler = (s, e) =>
                {
                    UpdateCopyPasteButton();
                    UpdateTargetBox(Targets);
                };

            App.AppContext.SelectedTargetChanged += objSelectedChangedHandler;
            App.AppContext.DocumentAddedOrRemoved += AddedOrRemovedHandler;
            App.AppContext.PropertyChanged += PropertyChangedHandler;
            // 削除（Disposedで行う）
            Disposed += delegate
            {
                App.AppContext.SelectedTargetChanged -= objSelectedChangedHandler;
                App.AppContext.DocumentAddedOrRemoved -= AddedOrRemovedHandler;
                App.AppContext.PropertyChanged -= PropertyChangedHandler;
            };

            FormClosed += (s, e) =>
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.Location = DesktopLocation;
            };

            // ヘルプの設定
            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_3deditor);

            // コンテクストメニューを追加
            tsiTargetBox.HostControl.ContextMenuStrip = cmiTargetBox;
            cmiTargetBox.Opening += (sender, args) =>
                { cmiTargetBoxCopyName.Enabled = tsiTargetBox.HostControl.Objects != null && !tsiTargetBox.HostControl.Objects.IsEmpty; };
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            // パネルにフォーカスを移す
            if (activePanel_ != null)
            {
                activePanel_.Focus();
            }
            base.OnGotFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnVisibleChanged(EventArgs e)
        {
            // フォーム更新
            if (Visible)
            {
                UpdateTargetBoxSize();
                UpdateForm(false);
            }
            base.OnVisibleChanged(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDeactivate(EventArgs e)
        {
            // ターゲットを変えた時に無効なコントロールにフォーカスが
            // 当たったままにならないように、アクティブを解除しておく
            ActiveControl = null;
            base.OnDeactivate(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClosing(CancelEventArgs e)
        {
            // 登録していたハンドラを外す
            App.AppContext.SelectedTargetChanged -= objSelectedChangedHandler;

            // 管理対象から外してオーナーに制御を移す
            dialogs_.Remove(this);
            Owner.Activate();
            base.OnClosing(e);
        }

        public FormWindowState LastWindowStateWithoutMinimized { get; private set; }
        private FormWindowState LastWindowState { get; set; }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            // ページのドッキングスタイルを更新
            if (activePanel_ != null)
            {
                activePanel_.UpdateActivePageDockStyle();
            }
            base.OnResize(e);
            if (WindowState != FormWindowState.Minimized)
            {
                LastWindowStateWithoutMinimized = WindowState;
            }

            if (WindowState == FormWindowState.Normal && Visible && !ObjectPropertyDialog.InternallyChangingSize.IsChanging)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.PropertyPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.PropertyPageSize.Height = Height;
            }

            if (LastWindowState != WindowState && LastWindowState == FormWindowState.Minimized)
            {
                if (activePanel_ != null)
                {
                    activePanel_.SetCategoryMinimumSize();
                }
            }

            LastWindowState = WindowState;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragEnter(DragEventArgs e)
        {
            base.OnDragEnter(e);

            // ファイルドラッグ時
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                // 非モーダル時
                if (!UIForm.IsModalState(TheApp.MainFrame) || DialogUtility.IsOpened)
                {
                    // インポート可能状態
                    if (tsiImport.Enabled)
                    {
                        // マテリアルテンプレートファイル（１ファイル）のみ
                        string[] files = (string[])e.Data.GetData(DataFormats.FileDrop);
                        if (files != null)
                        {
                            if (files.Length == 1 && string.Compare(Path.GetExtension(files[0]), "." + MaterialTemplate.ext, true) == 0)
                            {
                                e.Effect = DragDropEffects.Copy;
                                return;
                            }
                        }
                    }
                }
            }
            e.Effect = DragDropEffects.None;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragDrop(DragEventArgs e)
        {
            // アクティブ化しておく
            Activate();

            // ドロップ処理
            string[] files = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            if (files != null)
            {
                string filePath = files[0];
                if (DialogUtility.IsOpened)
                {
                    ImportMaterialTemplate(filePath);
                }
                else
                {
                    OneShotIdleProcess.Execute(() => ImportMaterialTemplate(filePath));
                }
            }
            base.OnDragDrop(e);
        }

        private void UpdateTargetFixedState()
        {
            if (TargetFixed)
            {
                tsiTargetFixed.Image = imgButtons.Images[1];
//				tsiTargetBox.BackColor = SystemColors.Control;
            }
            else
            {
                tsiTargetFixed.Image = imgButtons.Images[0];
//				tsiTargetBox.BackColor = SystemColors.Window;
            }
        }

        private void UpdateTargetBoxSize()
        {
            int left  = tsiTargetFixed.Bounds.Right + tsiTargetBox.Margin.Left;
            int right = tsiCopy.Bounds.Left - tsiTargetBox.Margin.Right;
            int width = right - left;

            tsiTargetBox.Width = width;
        }

        private void UpdateForm(bool targetOrPageChanged)
        {
            // プロパティパネル
            UpdatePropertyPanel(targetOrPageChanged);

            // コピー、ペーストボタン
            UpdateCopyPasteButton();

            // インポート、エクスポートボタン
            UpdateImportExportButton();
        }

        /// <summary>
        /// インポート、エクスポートボタン更新。
        /// </summary>
        private void UpdateImportExportButton()
        {
            // アクティブターゲットがマテリアルのみ
            if (!targets_.IsEmpty && targets_.Active is Material)
            {
                tsiImport.Enabled = true;
                if (!targets_.IsMulti)
                {
                    tsiExport.Enabled = true;
                }
                else
                {
                    tsiExport.Enabled = false;
                }
            }
            else
            {
                tsiImport.Enabled = false;
                tsiExport.Enabled = false;
            }
        }

        /// <summary>
        /// プロパティパネル更新。
        /// </summary>
        private void UpdatePropertyPanel(bool targetOrPageChanged)
        {
            DebugConsole.WriteLine("UpdatePropertyPanel");

            using (var block = new InternallyChangingSize())
            {
                // ターゲットに対応したパネルを取得
                ObjectPropertyPanel panel = null;
                if (targets_.IsEmpty)
                {
                    panel = GetPropertyPanel(PanelID.Null);
                }
                else
                {
                    switch (targets_.Active.ObjectID)
                    {
                        case GuiObjectID.Project: panel = GetPropertyPanel(PanelID.Project); break;
                        case GuiObjectID.Model: panel = GetPropertyPanel(PanelID.Model); break;
                        case GuiObjectID.Texture: panel = GetPropertyPanel(PanelID.Texture); break;
                        case GuiObjectID.Material: panel = GetPropertyPanel(PanelID.Material); break;
                        case GuiObjectID.Shape: panel = GetPropertyPanel(PanelID.Shape); break;
                        case GuiObjectID.SkeletalAnimation: panel = GetPropertyPanel(PanelID.SkeletalAnimation); break;
                        case GuiObjectID.ShapeAnimation: panel = GetPropertyPanel(PanelID.ShapeAnimation); break;
                        case GuiObjectID.MaterialAnimation: panel = GetPropertyPanel(PanelID.MaterialAnimation); break;
                        case GuiObjectID.ShaderParameterAnimation: panel = GetPropertyPanel(PanelID.ShaderParameterAnimation); break;
                        case GuiObjectID.ColorAnimation: panel = GetPropertyPanel(PanelID.ColorAnimation); break;
                        case GuiObjectID.TextureSrtAnimation: panel = GetPropertyPanel(PanelID.TextureSrtAnimation); break;
                        case GuiObjectID.MaterialVisibilityAnimation: panel = GetPropertyPanel(PanelID.MaterialVisibilityAnimation); break;
                        case GuiObjectID.BoneVisibilityAnimation: panel = GetPropertyPanel(PanelID.BoneVisibilityAnimation); break;
                        case GuiObjectID.TexturePatternAnimation: panel = GetPropertyPanel(PanelID.TexturePatternAnimation); break;
                        case GuiObjectID.SceneAnimation: panel = GetPropertyPanel(PanelID.SceneAnimation); break;
                        case GuiObjectID.CameraAnimation: panel = GetPropertyPanel(PanelID.CameraAnimation); break;
                        case GuiObjectID.LightAnimation: panel = GetPropertyPanel(PanelID.LightAnimation); break;
                        case GuiObjectID.FogAnimation: panel = GetPropertyPanel(PanelID.FogAnimation); break;
                        case GuiObjectID.ShaderDefinition: panel = GetPropertyPanel(PanelID.ShaderDefinition); break;
                        case GuiObjectID.Bone: panel = GetPropertyPanel(PanelID.Bone); break;
                        case GuiObjectID.AnimationSet: panel = GetPropertyPanel(PanelID.Null); break;
                        case GuiObjectID.SeparateMaterial: panel = GetPropertyPanel(PanelID.Material); break;
                        default: Debug.Assert(false); return;
                    }
                }

                // アクティブが未設定なら新規表示
                if (activePanel_ == null)
                {
                    activePanel_ = panel;
                    activePanel_.Visible = true;
                    activePanel_.UpdateForm(true, targetOrPageChanged);
                }
                // アクティブと異なる場合は表示切り替え
                else if (activePanel_ != panel)
                {
                    var old = activePanel_;
                    if (old != null && old.ActivePage != null)
                    {
                        old.ActivePage.BeforePageDeactivated();
                    }

                    activePanel_ = panel;
                    panel.Visible = true;

                    if (ApplicationConfig.Setting.PropertyEdit.OptimizePageSize)
                    {
                        panel.OptimizePageSize(panel.ActivePage);
                    }

                    old.Visible = false;

                    activePanel_.UpdateForm(true, targetOrPageChanged);

                    if (activePanel_.ActivePage != null)
                    {
                        activePanel_.ActivePage.AfterPageActiveted(old != null ? old.ActivePage : null);
                    }
                }
                // アクティブと同じならパネルを更新するだけ
                else
                {
                    activePanel_.UpdateForm(true, targetOrPageChanged);
                }

                if (activePanel_.ActivePage == null)
                {
                    activePanel_.SetDefaultCategory();
                }
            }
        }

        /// <summary>
        /// プログラム側の操作によりサイズ変更を行わている可能性を明示的に示すためのブロック
        /// </summary>
        public class InternallyChangingSize : IDisposable
        {
            private static int count = 0;
            public InternallyChangingSize()
            {
                count++;
            }

            public void Dispose()
            {
                count--;
            }

            public static bool IsChanging { get { return count > 0; } }
        }
        /// <summary>
        /// プロパティパネル取得。
        /// </summary>
        public ObjectPropertyPanel GetPropertyPanel(PanelID id)
        {
            // 未登録なら作成する
            if (propertyPanels_[(int)id] == null)
            {
                switch (id)
                {
                    case PanelID.Null:							RegisterPropertyPanel(id, new ObjectPropertyPanel());					break;
                    case PanelID.Project:						RegisterPropertyPanel(id, new ProjectPropertyPanel());					break;
                    case PanelID.Model:							RegisterPropertyPanel(id, new ModelPropertyPanel());					break;
                    case PanelID.Texture:						RegisterPropertyPanel(id, new TexturePropertyPanel());					break;
                    case PanelID.Material:						RegisterPropertyPanel(id, new MaterialPropertyPanel());					break;
                    case PanelID.Shape:			 				RegisterPropertyPanel(id, new ShapePropertyPanel());					break;
                    case PanelID.SkeletalAnimation:				RegisterPropertyPanel(id, new SkeletalAnimationPropertyPanel());		break;
                    case PanelID.ShapeAnimation:				RegisterPropertyPanel(id, new ShapeAnimationPropertyPanel());			break;
                    case PanelID.MaterialAnimation:             RegisterPropertyPanel(id, new MaterialAnimationPropertyPanel(id)); break;
                    case PanelID.ShaderParameterAnimation:		RegisterPropertyPanel(id, new ShaderParameterAnimationPropertyPanel(id));	break;
                    case PanelID.ColorAnimation:				RegisterPropertyPanel(id, new ShaderParameterAnimationPropertyPanel(id));	break;
                    case PanelID.TextureSrtAnimation:			RegisterPropertyPanel(id, new ShaderParameterAnimationPropertyPanel(id));	break;
                    case PanelID.MaterialVisibilityAnimation:	RegisterPropertyPanel(id, new MaterialVisibilityAnimationPanel());		break;
                    case PanelID.BoneVisibilityAnimation:		RegisterPropertyPanel(id, new BoneVisibilityAnimationPanel());			break;
                    case PanelID.TexturePatternAnimation:		RegisterPropertyPanel(id, new TexturePatternAnimationPanel());			break;
                    case PanelID.SceneAnimation:				RegisterPropertyPanel(id, new SceneAnimationPropertyPanel());			break;
                    case PanelID.CameraAnimation:				RegisterPropertyPanel(id, new CameraAnimationPanel());					break;
                    case PanelID.LightAnimation:				RegisterPropertyPanel(id, new LightAnimationPanel());					break;
                    case PanelID.FogAnimation:					RegisterPropertyPanel(id, new FogAnimationPanel());						break;
                    case PanelID.ShaderDefinition:							RegisterPropertyPanel(id, new ShaderDefinitionPropertyPanel());					break;
                    case PanelID.Bone:			  				RegisterPropertyPanel(id, new BonePropertyPanel());						break;
                    default: 									Debug.Assert(false);													break;
                }
            }

            switch(id)
            {
                case PanelID.Material:
                {
                    break;
                }

                case PanelID.Model:
                {
                    Debug.Assert(targets_.Active is Model);
                    Viewer.QueryModelLayoutMessage.Send((Model)targets_.Active);
                    break;
                }
            }

            return propertyPanels_[(int)id];
        }

        /// <summary>
        /// プロパティパネル登録。
        /// </summary>
        private void RegisterPropertyPanel(PanelID id, ObjectPropertyPanel panel)
        {
            Debug.Assert(propertyPanels_[(int)id] == null);

            propertyPanels_[(int)id] = panel;

            // パネル初期状態
            if (activePanel_ != null)
            {
                panel.LeftWidth = activePanel_.LeftWidth;
            }
            else if (parent_ != null && parent_.activePanel_ != null)
            {
                panel.SetLeftWidthForcibly(parent_.activePanel_.LeftWidth);
            }
            else
            {
                panel.LeftWidth = defaultLeftWidth;
            }

            // パネル基本設定
            panel.Owner   = this;
            panel.Parent  = pnlClient;
            panel.Dock	= DockStyle.Fill;
            panel.Visible = false;
            panel.InitializeForm();

            if (activePanel_ == null && parent_ != null && parent_.activePanel_ != null)
            {
                panel.SetInitialPage(parent_.activePanel_);
            }

            // パネルのヘルプ設定
            //BindHelp(panel, panel.HelpPath);
            DialogUtility.SetHelp(panel, panel.HelpKey);

            // もう親の情報は不要
            parent_ = null;
        }

        public void UpdateCopyPasteButton(ObjectPropertyPanel updatingPanel = null)
        {
            if (updatingPanel != null && updatingPanel != activePanel_)
            {
                // パネルがアクティブに設定される前に更新中
                return;
            }

            if (activePanel_ == null)
            {
                tsiCopy.Enabled  = false;
                tsiPaste.Enabled = false;
            }
            else
            {
                tsiCopy.Enabled  = activePanel_.CanCopy();
                tsiPaste.Enabled = copyObject_ != null && copyObjectID_ == activePanel_.ActivePageID && activePanel_.CanPaste(copyObjectInfo_, copyObject_);
            }
        }

        /// <summary>
        /// 指定ダイアログの隣接位置に設定。
        /// </summary>
        public void SetLocationNextTo(ObjectPropertyDialog nextTo)
        {
            // 右下位置取得
            Point location = new Point(
                nextTo.Left + TheApp.WindowCascadingMargin,
                nextTo.Top + TheApp.WindowCascadingMargin
            );

            // 画面外ならリセット
            if (!InScreen(location, Size))
            {
                location = new Point(0, 0);
            }
            // 設定
            Location = location;
        }

        public static bool InScreen(Point location, Size size)
        {
            // 少し余裕を持っておく
            var rect = new Rectangle(location, size);
            var right = Screen.AllScreens.Max(x => x.Bounds.Right);
            var bottom = Screen.AllScreens.Max(x => x.Bounds.Bottom);
            if (location.X > right - 300 ||
                location.Y > bottom - 200 ||
                Screen.AllScreens.All(x => !x.Bounds.Contains(rect.Location) && !x.Bounds.Contains(new Point(rect.X + 300, rect.Y))))
            {
                return false;
            }

            return true;
        }

        //---------------------------------------------------------------------------
        // 選択オブジェクト
        private void Event_GuiObjectManager_SelectedChanged(object sender, EventArgs e)
        {
            if (!TargetFixed)
            {
                Targets = App.AppContext.SelectedTarget;
            }
        }

        public static void ForEachActivePanel(Action<ObjectPropertyPage> action)
        {
            foreach (var dialog in dialogs_)
            {
                if (dialog.activePanel_ != null && dialog.activePanel_.ActivePage != null)
                {
                    action(dialog.activePanel_.ActivePage);
                }
            }
        }
        #region イベントハンドラ
        private void tspMain_Resize(object sender, EventArgs e)
        {
            // ターゲットボックス幅を調整
            UpdateTargetBoxSize();
        }

        private void tsiTargetFixed_Click(object sender, EventArgs e)
        {
            UpdateTargetFixedState();
        }

        private void tsiCopy_Click(object sender, EventArgs e)
        {
            Debug.Assert(
                activePanel_ != null   &&
                activePanel_.CanCopy() &&
                activePanel_.ActivePageID != PropertyPageID.Null
            );

            copyObject_ = activePanel_.Copy(ref copyObjectInfo_);
            copyObjectID_ = activePanel_.ActivePageID;

            // 他のダイアログで同種ターゲットの同一ページを開いている
            // 可能性があるので全ダイアログのボタンを更新する
            foreach (var dialog in dialogs_)
            {
                dialog.UpdateCopyPasteButton();
            }
        }

        private void tsiPaste_Click(object sender, EventArgs e)
        {
            Debug.Assert(
                activePanel_  != null &&
                copyObject_   != null &&
                copyObjectID_ == activePanel_.ActivePageID
            );

            using (var wait = new WaitCursor())
            {
                activePanel_.Paste(copyObject_);
            }
        }

        private void tsiImport_Click(object sender, EventArgs e)
        {
            //MessageBox.Show("未実装 - tsiImport_Click()");
            ImportMaterialTemplate(null);
        }

        private void ImportMaterialTemplate(string filePath)
        {
            var materialTemplate = new MaterialTemplate();
            if (filePath == null)
            {
                filePath = materialTemplate.GetReadPath();
            }

            if (string.IsNullOrWhiteSpace(filePath))
            {
                return;
            }

            using (var wait = new WaitCursor())
            {
                if (!materialTemplate.Read(filePath))
                {
                    return;
                }
            }

            var shaderAssign = materialTemplate.material.MaterialShaderAssign;
            if (shaderAssign.ShadingModel != null &&
                !shaderAssign.ShadingModel.IsMaterialShader())
            {
                if (!UIMessageBox.OkCancel(
                        string.Format(res.Strings.MaterialTemplate_NotMaterialShaderAssigned,
                            shaderAssign.ShaderName,
                            shaderAssign.ShaderDefinitionFileName)))
                {
                    return;
                }
            }

            using (var dialog = new MaterialTemplateDialog(materialTemplate, targets_.GetObjects(GuiObjectID.Material).OfType<Material>().ToList()))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    materialTemplate.Apply(dialog.CheckedMaterials.ToList(), dialog.SamplerImport);
                }
            }

        }

        private void tsiExport_Click(object sender, EventArgs e)
        {
            Debug.Assert(targets_.Active is Material);
            var material = (Material)targets_.Active;

            // samplerが参照するテクスチャにtgaが含まれる場合はexport出来ないようにする。
            var textureNames = new HashSet<string>(material.ResolvedSamplerTextureNames);
            var tgaTextures = DocumentManager.Textures.Where(x => textureNames.Contains(x.Name)).Where(x => x.IsTemporary).Select(x => x.Name).ToArray();
            if (tgaTextures.Any())
            {
                UIMessageBox.Error(Strings.ObjectPropertyDialog_tsiExport_Click_SamplerHasTgaCannotExport + "\n" + String.Join(", ", tgaTextures));
                return;
            }

            if (!ShaderAssignUtility.IsMaterialShader(material))
            {
                if (!UIMessageBox.OkCancel(
                        string.Format(res.Strings.MaterialTemplate_NotMaterialShaderAssignedExport,
                            material.Name,
                            material.MaterialShaderAssign.ShaderName,
                            material.MaterialShaderAssign.ShaderDefinitionFileName)))
                {
                    return;
                }
            }

            var template = new MaterialTemplate();
            if (!template.SetSourceMaterial((Material)targets_.Active))
            {
                return;
            }

            using (var dialog = new MaterialTemplateDialog(template, null))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    string filePath = template.GetWritePath();
                    if (filePath == null)
                    {
                        return;
                    }
                    using (WaitCursor wait = new WaitCursor())
                    {
                        template.Write(filePath, dialog.SamplerImport);
                    }
                }
            }
        }

        private void tsiDuplicate_Click(object sender, EventArgs e)
        {
            using (var wait = new WaitCursor())
            {
                ObjectPropertyDialog dialog = new ObjectPropertyDialog(this);
                dialog.TargetFixed = true;
                dialog.Show();
            }
        }

        private void tsiHelp_Click(object sender, EventArgs e)
        {
            DialogUtility.EmulatePushF1();
        }

        /// <summary>
        /// ヘルプの設定
        /// </summary>
        public void BindHelp(Control control, string helpPath)
        {
            if (helpPath != null)
            {
                DialogUtility.SetHelp(control, helpPath);
            }
            else
            {
                DebugConsole.WriteLine("No helpPath " + control.ToString());
            }
        }

        #endregion

        private void UpdateTargetBox(GuiObjectGroup targets)
        {
            tsiTargetBox.HostControl.Objects = targets;
        }

        private void cmiTargetBoxCopyName_Click(object sender, EventArgs e)
        {
            if (tsiTargetBox.HostControl.Objects != null && !tsiTargetBox.HostControl.Objects.IsEmpty)
            {
                MainFrameContextMenuItems.SetObjectsNameClipboard(tsiTargetBox.HostControl.Objects.Objects);
            }
        }
    }
}
