﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Linq;
using App.Controls;
using App.Data;
using App.Properties;
using App.Utility;
using App.ObjectView.List;
using App.PropertyEdit;
using App.res;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;

using App.ConfigData;

namespace App.ObjectView
{
    /// <summary>
    /// オブジェクトビューパネルクラス。
    /// </summary>
    public sealed partial class ObjectViewPanel : UIUserControl
    {
        // レイアウトインデクス
        private readonly int _layoutIndex;

        // ビューモード
        private AppConfig.ObjectViewMode _viewMode;

        /// <summary>ビューモード変更イベント。</summary>
        public event ObjectViewModeChangedEventHandler ViewModeChanged = null;

        const int EM_SETCUEBANNER = 0x1500 + 1;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ObjectViewPanel(int layoutIndex, AppConfig.ObjectViewMode viewMode)
        {
            InitializeComponent();

            _layoutIndex = layoutIndex;
            _viewMode    = viewMode;

            cbxFilter.Visible = false;
            SetupFilterItems();
            App.AppContext.SpecChanged += SetupFilterItems;
/*
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectViewPanel_All, PropertyPageID.Null.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_General, PropertyPageID.MaterialGeneral.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_Preview, PropertyPageID.MaterialPreview.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_RenderState, PropertyPageID.MaterialRenderState.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_Sampler, PropertyPageID.MaterialSampler.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_Shader, PropertyPageID.MaterialShader.ToString()));
*/

            cmiViewModeSchematic.Text		= Strings.ObjectViewePanel_ViewModeSchematic;
            cmiViewModeModelList.Text		= Strings.ObjectViewePanel_ViewModeModelList;
            cmiViewModeMaterialList.Text	= Strings.ObjectViewePanel_ViewModeMaterialList;
            cmiViewModeBoneList.Text		= Strings.ObjectViewePanel_ViewModeBoneList;
            cmiViewModeTextureList.Text		= Strings.ObjectViewePanel_ViewModeTextureList;
            cmiViewModeShapeList.Text		= Strings.ObjectViewePanel_ViewModeShapeList;
            cmiColumnSetting.Text = Strings.ObjectViewePanel_ColumnSetting;
            ttpHint.SetToolTip(lblDropDown, Strings.ObjectViewePanel_DropDownHint);

            // ビューモードメニュー項目
            cmiViewModeSchematic.Tag	= AppConfig.ObjectViewMode.SchematicView;
            cmiViewModeModelList.Tag	= AppConfig.ObjectViewMode.ModelList;
            cmiViewModeMaterialList.Tag	= AppConfig.ObjectViewMode.MaterialList;
            cmiViewModeBoneList.Tag		= AppConfig.ObjectViewMode.BoneList;
            cmiViewModeTextureList.Tag	= AppConfig.ObjectViewMode.TextureList;
            cmiViewModeShapeList.Tag	= AppConfig.ObjectViewMode.ShapeList;

            // 不要かもしれないが、念のためにイベントハンドラに登録しておく
            tbxSearch.HandleCreated += (s, e) => UpdateFilterWaterMark();
        }

        public void SetupFilterItems()
        {
            var lastSelect = cbxFilter.SelectedItemData as string;
            cbxFilter.Items.Clear();
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectViewPanel_All, PropertyPageID.Null.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_General, PropertyPageID.MaterialGeneral.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_Preview, PropertyPageID.MaterialPreview.ToString()));
            if ((ApplicationConfig.DefaultValue.RenderStateInfoVisible && App.AppContext.SelectedPlatformPreset.UseNw))
            {
                cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_RenderState, PropertyPageID.MaterialRenderState.ToString()));
            }
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_Sampler, PropertyPageID.MaterialSampler.ToString()));
            cbxFilter.AddItem(new UIListControlItem(res.Strings.ObjectPropertyPanel_Material_Shader, PropertyPageID.MaterialShader.ToString()));

            if (cbxFilter.Items.OfType<UIListControlItem>().Any(x => (string)(x.Data) == lastSelect))
            {
                cbxFilter.SelectedItemData = lastSelect;
            }
            else
            {
                cbxFilter.SetSelectedIndex(0);
            }
        }

        public void UpdateFilterWaterMark()
        {
            if (tbxSearch.IsHandleCreated)
            {
                string text = string.Empty;
                switch (ViewMode)
                {
                    case AppConfig.ObjectViewMode.Undefined:
                    case AppConfig.ObjectViewMode.SchematicView:
                    case AppConfig.ObjectViewMode.TreeView:
                        break;
                    case AppConfig.ObjectViewMode.ModelList:
                        text = res.Strings.ObjectView_SearchModel;
                        break;
                    case AppConfig.ObjectViewMode.MaterialList:
                        text = res.Strings.ObjectView_SearchMaterial;
                        break;
                    case AppConfig.ObjectViewMode.BoneList:
                        text = res.Strings.ObjectView_SearchBone;
                        break;
                    case AppConfig.ObjectViewMode.ShapeList:
                        text = res.Strings.ObjectView_SearchShape;
                        break;
                    case AppConfig.ObjectViewMode.TextureList:
                        text = res.Strings.ObjectView_SearchTexture;
                        break;
                    default:
                        break;
                }
                var ptr = Marshal.StringToBSTR(text);
                Win32.NativeMethods.SendMessage(tbxSearch.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
            }
        }

        public void UpdateFilterComboBox()
        {
            var materialListView = View as MaterialListView;
            if (materialListView != null)
            {
                cbxFilter.SelectedItemData = materialListView.FilterCategoryString;
            }
        }

        /// <summary>
        /// レイアウトインデクス。
        /// </summary>
        public int LayoutIndex
        {
            get { return _layoutIndex; }
        }

        /// <summary>
        /// ビューモード。
        /// </summary>
        public AppConfig.ObjectViewMode ViewMode
        {
            get { return _viewMode; }
        }

        /// <summary>
        /// ビュー。
        /// </summary>
        public IObjectView View
        {
            get { return pnlView.Controls.Count > 0 ? (IObjectView) pnlView.Controls[0] : null; }
            set
            {
                Debug.Assert(value != null);

                IObjectView prevView = View;
                if (value != prevView)
                {
                    using (var block = new UIControlEventSuppressBlock())
                    {
                        // 次のビューを設定
                        value.Control.Parent = pnlView;

                        // 前のビューを外す
                        if (prevView != null)
                        {
                            if (prevView is App.ObjectView.List.StandardObjectListView)
                            {
                                var listView = (App.ObjectView.List.StandardObjectListView)prevView;
                                listView.ColumnWidthChanged -= ColumnWidthChanged;
                            }
                            prevView.Control.Parent = null;
                        }

                        if (value is App.ObjectView.List.StandardObjectListView)
                        {
                            var listView = (App.ObjectView.List.StandardObjectListView)value;
                            listView.ColumnWidthChanged += ColumnWidthChanged;
                        }

                        // 状態更新
                        _viewMode = value.ViewMode;

                        pnlHeader.View = value;
                        pnlHeader.ViewMode = _viewMode;

                        UpdateFilterWaterMark();

                        if (value is StandardObjectListView)
                        {
                            tbxSearch.Text = ((StandardObjectListView)value).SearchText;
                            tbxSearch.Visible = true;
                            if (_viewMode == AppConfig.ObjectViewMode.MaterialList)
                            {
                                var materialListView = (MaterialListView)value;
                                cbxFilter.SelectedItemData = materialListView.FilterCategoryString;
                                cbxFilter.Visible = true;
                            }
                            else
                            {
                                cbxFilter.Visible = false;
                            }
                        }
                        else
                        {
                            tbxSearch.Text = string.Empty;
                            tbxSearch.Visible = false;
                            cbxFilter.Visible = false;
                        }
                    }
                }
            }
        }

        public void ColumnWidthChanged(object sender, ColumnWidthChangedEventArgs e)
        {
            pnlHeader.Invalidate();
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // ビューモードメニュー
        private void cspViewMode_Opening(object sender, CancelEventArgs e)
        {
            // 各項目の状態を更新
            foreach (ToolStripItem item in cspViewMode.Items)
            {
                if (item is ToolStripMenuItem)
                {
                    if (item.Tag != null)
                    {
                        (item as ToolStripMenuItem).Checked = item.Tag.Equals(_viewMode);
                    }
                }
            }

            switch (_viewMode)
            {
                case AppConfig.ObjectViewMode.Undefined:
                case AppConfig.ObjectViewMode.SchematicView:
                case AppConfig.ObjectViewMode.TreeView:
                    break;
                case AppConfig.ObjectViewMode.ModelList:
                    ColumnSettingDialog.InitialPageViewID = ViewID.Model;
                    break;
                case AppConfig.ObjectViewMode.MaterialList:
                    ColumnSettingDialog.InitialPageViewID = ViewID.Material;
                    break;
                case AppConfig.ObjectViewMode.BoneList:
                    ColumnSettingDialog.InitialPageViewID = ViewID.Bone;
                    break;
                case AppConfig.ObjectViewMode.ShapeList:
                    ColumnSettingDialog.InitialPageViewID = ViewID.Shape;
                    break;
                case AppConfig.ObjectViewMode.TextureList:
                    ColumnSettingDialog.InitialPageViewID = ViewID.Texture;
                    break;
                default:
                    break;
            }
        }

        private void cspViewMode_Closed(object sender, ToolStripDropDownClosedEventArgs e)
        {
            // ドロップダウン解除
            lblDropDown.DropDown = false;
        }

        //---------------------------------------------------------------------
        // ビューモードメニュー項目
        private void cmiViewMode_Click(object sender, EventArgs e)
        {
            AppConfig.ObjectViewMode viewMode = (AppConfig.ObjectViewMode)((ToolStripItem)sender).Tag;
            if (viewMode != _viewMode)
            {
                if (ViewModeChanged != null)
                {
                    ViewModeChanged(this, new ObjectViewModeChangedEventArgs(_layoutIndex, viewMode, this));
                }
            }
        }

        //---------------------------------------------------------------------
        // ドロップダウンラベル
        private void lblDropDown_Click(object sender, EventArgs e)
        {
            // ドロップダウン開始
            lblDropDown.DropDown = true;

            // コンテキストメニュー表示
            cspViewMode.Show(this, new Point(lblDropDown.Left, lblDropDown.Bottom + 1));
        }
        #endregion

        //--------------------------------------------------------------------
        // 内部クラス
        //--------------------------------------------------------------------
        #region HeaderPanel
        /// <summary>
        /// ヘッダパネルクラス。
        /// </summary>
        private sealed class HeaderPanel : UIPanel
        {
            // ビューモード
            private AppConfig.ObjectViewMode _viewMode;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public HeaderPanel()
            {
                base.ResizeRedraw   = true;
                base.DoubleBuffered = true;

                // イベント登録
                App.AppContext.DocumentAddedOrRemoved += (s,a,r, swaped, reloaded)=>Invalidate();
                App.AppContext.PropertyChanged += Document_PropertyChanged;
            }

            /// <summary>
            /// ビューモード。
            /// </summary>
            [Browsable(false)]
            [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
            public AppConfig.ObjectViewMode ViewMode
            {
                private get { return _viewMode; }
                set
                {
                    _viewMode = value;
                    Invalidate();
                }
            }

            [Browsable(false)]
            [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
            public IObjectView View;

            private static readonly Dictionary<AppConfig.ObjectViewMode, Image> images = new Dictionary<AppConfig.ObjectViewMode, Image>();
            private static Image GetImage(AppConfig.ObjectViewMode viewMode)
            {
                Image image;
                if (images.TryGetValue(viewMode, out image))
                {
                    return image;
                }

                image = (Image)Resources.ResourceManager.GetObject("ObjectViewMode_" + viewMode.ToString(), Resources.Culture);
                images.Add(viewMode, image);
                return image;
            }

            //---------------------------------------------------------------------------
            // ドキュメント内容
            private void Document_PropertyChanged(object sender, IEnumerable<DocumentPropertyChangedArgs> e)
            {
                if (e.Any(x => x is App.FileView.DocumentPropertyChangedShowInObjViewArgs))
                {
                    // 再描画でアイテム数を更新する。
                    Invalidate();
                }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void OnPaint(PaintEventArgs e)
            {
                // 背景
                Rectangle rcClient = ClientRectangle;

                e.Graphics.FillRectangle(SystemBrushes.ControlDark, rcClient);

                e.Graphics.DrawLine(SystemPens.ControlDarkDark, ClientRectangle.Left, ClientRectangle.Top, ClientRectangle.Right, ClientRectangle.Top);
                e.Graphics.DrawLine(SystemPens.ControlDarkDark, ClientRectangle.Left, ClientRectangle.Top, ClientRectangle.Left, ClientRectangle.Bottom);
                e.Graphics.DrawLine(SystemPens.ControlDarkDark, ClientRectangle.Right - 1, ClientRectangle.Top, ClientRectangle.Right - 1, ClientRectangle.Bottom);

                // タイトル
                if (!DesignMode)
                {
                    // アイコン
                    Rectangle rcIcon = rcClient;
                    rcIcon.X   += 4;
                    rcIcon.Y   += (rcIcon.Height - 16) / 2 + 1;
                    rcIcon.Size = new Size(16, 16);
                    e.Graphics.DrawImage(GetImage(ViewMode), rcIcon);

                    // テキスト
                    Rectangle rcText = rcClient;
                    rcText.Y += 1;
                    RectangleUtility.OffsetLeft(ref rcText, rcIcon.Right);
                    using (StringFormat sf = new StringFormat())
                    {
                        sf.Alignment     = StringAlignment.Near;
                        sf.LineAlignment = StringAlignment.Center;
                        sf.FormatFlags  |= StringFormatFlags.NoWrap;

                        string text = string.Empty;
                        int listCount = -1;
                        switch (_viewMode)
                        {
                            case AppConfig.ObjectViewMode.SchematicView:	text = App.res.Strings.ObjectView_Schematic;												break;
                            case AppConfig.ObjectViewMode.TreeView:			text = App.res.Strings.ObjectView_Tree;														break;
                            case AppConfig.ObjectViewMode.ModelList:		text = App.res.Strings.ObjectView_ModelList;	listCount = DocumentManager.Models.Where(x => x.IsShowInObjView).Count();   break;
                            case AppConfig.ObjectViewMode.MaterialList:     text = App.res.Strings.ObjectView_MaterialList; listCount = DocumentManager.Models.Where(x => x.IsShowInObjView).SelectMany(y => y.Materials).Count();  break;
                            case AppConfig.ObjectViewMode.BoneList:         text = App.res.Strings.ObjectView_BoneList;     listCount = DocumentManager.Models.Where(x => x.IsShowInObjView).SelectMany(y => y.Bones).Count();      break;
                            case AppConfig.ObjectViewMode.ShapeList:        text = App.res.Strings.ObjectView_ShapeList;    listCount = DocumentManager.Models.Where(x => x.IsShowInObjView).SelectMany(y => y.Shapes).Count();     break;
                            case AppConfig.ObjectViewMode.TextureList:		text = App.res.Strings.ObjectView_TextureList; listCount = DocumentManager.TextureCount;	break;
                            default:										Debug.Assert(false);																		break;
                        }

                        // 一覧表示時の項目数
                        if (listCount != -1)
                        {
                            text += string.Format(" ({0})", listCount);
                        }

                        string warningText = null;
                        if (View is App.ObjectView.List.StandardObjectListView)
                        {
                            SizeF size = e.Graphics.MeasureString(text, Font, rcText.Width, sf);
                            var listView = (App.ObjectView.List.StandardObjectListView)View;

                            if (listView.ColumnWidthSum > App.ObjectView.List.StandardObjectListView.ColumnWidhtSumUpperBound)
                            {
                                warningText = string.Format(res.Strings.ListView_Column_Waring,
                                    listView.ColumnWidthSum, App.ObjectView.List.StandardObjectListView.ColumnWidhtSumUpperBound);
                            }
                        }

                        if (warningText != null)
                        {
                            text += " " + warningText;
                        }

                        // 警告または null をツールチップに設定
                        ((ObjectViewPanel)Parent).ttpHint.SetToolTip(this, warningText);

                        e.Graphics.DrawString(text, Font, SystemBrushes.HighlightText, rcText, sf);

                    }
                }

                base.OnPaint(e);
            }
        }
        #endregion

        #region DropDownLabel
        /// <summary>
        /// ドロップダウンラベル。
        /// </summary>
        private sealed class DropDownLabel : UILabel
        {
            // マウスポイント中フラグ
            private bool _mouseIsOver = false;
            // ドロップダウン中フラグ
            private bool _dropDown = false;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public DropDownLabel()
            {
                base.DoubleBuffered = true;
            }

            /// <summary>
            /// ドロップダウン中フラグ。
            /// </summary>
            [Browsable(false)]
            [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
            public bool DropDown
            {
                set
                {
                    if (_dropDown != value)
                    {
                        _dropDown = value;
                        Invalidate();
                    }
                }
            }

            /// <summary>
            /// マウスポイント中フラグ。
            /// </summary>
            private bool MouseIsOver
            {
                set
                {
                    if (_mouseIsOver != value)
                    {
                        _mouseIsOver = value;
                        Invalidate();
                    }
                }
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void OnMouseEnter(EventArgs e)
            {
                MouseIsOver = true;
                base.OnMouseEnter(e);
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void OnMouseLeave(EventArgs e)
            {
                MouseIsOver = false;
                base.OnMouseLeave(e);
            }

            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void OnPaint(PaintEventArgs e)
            {
                // デザイン時
                if (DesignMode)
                {
                    Color color = BackColor.GetBrightness() < 0.5f ? ControlPaint.Light(BackColor) : ControlPaint.Dark(BackColor);
                    using (Pen pen = new Pen(color))
                    {
                        pen.DashStyle = DashStyle.Dash;
                        GraphicsUtility.DrawRectangle(e.Graphics, pen, ClientRectangle);
                    }
                }
                else
                {
                    // 通常時
                    Rectangle rcBounds = ClientRectangle;
                    if (_mouseIsOver || _dropDown)
                    {
                        Color backColor = Color.FromArgb(64, SystemColors.Highlight);
                        Color lineColor = SystemColors.Highlight;

                        using (Brush brush = new SolidBrush(backColor))
                        using (Pen pen = new Pen(lineColor))
                        {
                            e.Graphics.FillRectangle(brush, rcBounds);
                            GraphicsUtility.DrawRectangle(e.Graphics, pen, rcBounds);
                        }
                    }
                    else
                    {
                        GraphicsUtility.DrawRectangle(e.Graphics, SystemPens.ControlDark, ClientRectangle);
                    }
                }

                base.OnPaint(e);
            }
        }
        #endregion

        private void uiToolStripMenuItem1_Click(object sender, EventArgs e)
        {
            using (var dialog = new App.ObjectView.List.ColumnSettingDialog())
            {
                dialog.ShowDialog(this);
            }
        }

        private void cbxFilter_SelectedIndexChanged(object sender, EventArgs e)
        {
            var materialListView = View as MaterialListView;
            if (materialListView != null && materialListView.FilterCategoryString != (string)cbxFilter.SelectedItemData)
            {
                materialListView.FilterCategoryString = (string)cbxFilter.SelectedItemData;
                materialListView.ColumnFilterChanged();
                if (FilterChanged != null)
                {
                    FilterChanged();
                }
            }
        }

        public event Action FilterChanged;

        private void tbxSearch_TextChanged(object sender, EventArgs e)
        {
            var listView = View as StandardObjectListView;
            if (listView != null)
            {
                listView.SearchText = tbxSearch.Text;
            }
        }
    }
}
