﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;
using App.Controls;
using App.Data;
using App.Properties;
using App.PropertyEdit;
using App.Utility;
using App.ConfigData;
using ConfigCommon;

namespace App.ObjectView.List
{
    /// <summary>
    /// オブジェクトリストビュークラス。
    /// </summary>
    public abstract class ObjectListView : UIListView, IObjectView
    {
        // ビューＩＤ
        private readonly ViewID _viewId;
        // 列項目登録時ビューＩＤ（列項目登録中のみで使用する値です）
        protected static ViewID ColumnRegisterViewId = (ViewID)0;
        // 列情報テーブル（キー：ViewID、値：ColumnInfo）
        private static readonly Dictionary<ViewID, ColumnInfo> ColumnInfoTable = new Dictionary<ViewID, ColumnInfo>();
        // 選択変更中フラグ
        private bool _selectionChanging = false;

        /// <summary>列項目デフォルト幅（大）。</summary>
        protected const int ColumnDefaultWidthL = 160;
        /// <summary>列項目デフォルト幅（中）。</summary>
        protected const int ColumnDefaultWidthM = 120;
        /// <summary>列項目デフォルト幅（小）。</summary>
        protected const int ColumnDefaultWidthS = 80;
        /// <summary>列項目デフォルト幅（カラー３成分用）。</summary>
        protected const int ColumnDefaultWidthColor3 = 150; // とりあえず４成分用と同じにしています
        /// <summary>列項目デフォルト幅（カラー４成分用）。</summary>
        protected const int ColumnDefaultWidthColor4 = 150;
        /// <summary>列項目デフォルト幅（番号）。</summary>
        protected const int ColumnDefaultWidthNumber = 32;
        /// <summary>列項目デフォルト幅（状態）。</summary>
        protected const int ColumnDefaultWidthState = 32;
        /// <summary>アニメーションカラー（値は検討の末決めました）。</summary>
        protected readonly Color AnimationColor = Color.FromArgb(237,175,165);

        /// <summary>BOOL値。</summary>
        public enum BoolValueTextType
        {
            OnOff = 0,
            YesNo = 1,
            EnableDisable = 2,
            ExistNotexist = 3,
            UseNotuse = 4,
            VisibleNonvisible = 5,
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected ObjectListView()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected ObjectListView(ViewID viewId, AppConfig.ObjectViewMode viewMode, bool enableSorting)
        {
            //DebugConsole.WriteLine("create ObjectListView:[{0}]", this.GetType().Name);

            _viewId = viewId;
            ViewMode = viewMode;

            CustomDraw         = true;
            OwnerDraw          = true;
            AllowColumnReorder = true;
            GridLines          = true;
            HighlightSelection = true;
            MultiSelect        = true;
//			this.SmallImageList     = GuiObject.ObjectIconImageList;
            StripeDraw = true;

            // 選択変更間隔を長めにとる
            SelectionChangedInterval = 50;

            // 列ソート無効指定
            // 列項目作成前に行うこと！
            if (!enableSorting)
            {
                HeaderStyle = ColumnHeaderStyle.Nonclickable;
            }

            // 列項目作成
            RefreshColumn();

            // イベント登録
            ColumnInfo.SettingChanged += columnInfo_SettingChanged;
        }

        /// <summary>
        /// ビューＩＤ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ViewID ViewID
        {
            get { return _viewId; }
        }

        /// <summary>
        /// 列情報。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ColumnInfo ColumnInfo
        {
            get { return GetColumnInfo(_viewId); }
        }

        /// <summary>
        /// 自身で選択変更中か。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsSelectionChanging
        {
            get { return _selectionChanging; }
        }

        /// <summary>
        /// 列情報取得。
        /// </summary>
        public static ColumnInfo GetColumnInfo(ViewID viewId)
        {
            ColumnInfo columnInfo;

            if (ColumnInfoTable.TryGetValue(viewId, out columnInfo))
            {
                return columnInfo;
            }

            Initialize(viewId);
            return ColumnInfoTable[viewId];
        }

        public static void Initialize(ViewID id)
        {
            switch (id)
            {
                case ViewID.Model:
                    ModelListView.InitializeColumn();
                    break;
                case ViewID.Material:
                    MaterialListView.InitializeColumn();
                    break;
                case ViewID.Bone:
                    BoneListView.InitializeColumn();
                    break;
                case ViewID.Shape:
                    ShapeListView.InitializeColumn();
                    break;
                case ViewID.Texture:
                    TextureListView.InitializeColumn();
                    break;
            }
        }

        // 選択されているアイテムが見える位置までスクロールする
        public void EnsureSelectedItemVisible()
        {
            var selectedItem = (SelectedIndex != -1) ? Items[SelectedIndex] : null;
            if (selectedItem != null)
            {
                selectedItem.EnsureVisible();
            }
        }

        #region 列項目登録
        private static int _order;
        /// <summary>
        /// 列項目登録開始。
        /// </summary>
        protected static void BeginRegisterColumn(ViewID viewId, string profileSection)
        {
            // 作業用のビューＩＤを保存
            // 列項目登録終了時のカウントチェックの為です
            ColumnRegisterViewId = viewId;

            CurrentPropertyPageID = PropertyPageID.Null;

            _order = 0;
            //columns = new List<ConfigData.Column>();
            AppConfig.ListView config = ApplicationConfig.Setting.ObjectView.GetListView(ColumnRegisterViewId);
            foreach (var column in config.Columns)
            {
                column.SetColumnItem(null);
            }

            // 列情報作成
            if (ColumnInfoTable.ContainsKey(viewId))
            {
                ColumnInfoTable[viewId].RegisteredColumns.Clear();
            }
            else
            {
                ColumnInfoTable[viewId] = new ColumnInfo();
            }
        }

        protected static PropertyPageID CurrentPropertyPageID;
        protected static void RegisterColumn(ColumnItem column, string id = null, string configName = null, string type = null, bool defaultShow = true)
        {
            if (configName == null)
            {
                configName = column.Text;
            }

            // ビューの列情報を取得
            ColumnInfo columnInfo = GetColumnInfo(ColumnRegisterViewId);
            AppConfig.ListView config = ApplicationConfig.Setting.ObjectView.GetListView(ColumnRegisterViewId);
            if (config != null)
            {
                AppConfig.Column configColumn = config.Columns.FirstOrDefault(x => x.Name == configName && x.Id == id && x.Type == type);
                if (configColumn == null)
                {
                    configColumn = new AppConfig.Column
                        {
                        Name = configName,
                        Show = true,
                        Id = id,
                        Type = type,
                    };
                    config.Columns.Add(configColumn);
                }
                if (configColumn.Width != -1)
                {
                    column.Width = configColumn.Width;
                }
                configColumn.Text = column.Text;
                configColumn.SetColumnItem(column);
                configColumn.DefaultOrder = _order++;
                configColumn.DefaultShow = defaultShow;
                configColumn.Category = CurrentPropertyPageID.ToString();
            }
            column.PageID = CurrentPropertyPageID;
            columnInfo.RegisterColumn(column);
        }

        /// <summary>
        /// 列項目登録。
        /// </summary>
        protected static void RegisterColumn(
            object				columnId,
            string				signature,
            string				textStr,
            HorizontalAlignment	textAlign,
            int					width,
            bool				visible = true,
            Func<object, object> func = null,
            string id = null,
            string configName = null,
            Predicate<object> isValid = null
        )
        {
            ColumnInfo columnInfo = GetColumnInfo(ColumnRegisterViewId);
            Debug.Assert(columnInfo != null);

            // 順序（列項目ＩＤの妥当性を確認）
            //int order = columnInfo.RegisteredColumns.Count;
            //Debug.Assert((int)columnID == order);

            // 列項目作成
            var column = new ColumnItem(signature, width, visible)
                {
                    Text = textStr,
                    TextAlign = textAlign,
                    Width = width,
                    Visible = visible,
                    Tag = columnId,
                    GetParamObject = func,
                };

            if (isValid != null)
            {
                column.IsValid = isValid;
            }

            RegisterColumn(column, id, configName);
        }

        /// <summary>
        /// 列項目登録。
        /// </summary>
        protected static void RegisterCommentColorColumn(
            object columnId,
            string signature,
            string textStr,
            HorizontalAlignment textAlign,
            int width,
            bool visible = true,
            string id = null,
            string configName = null,
            Predicate<object> isValid = null
        )
        {
            ColumnInfo columnInfo = GetColumnInfo(ColumnRegisterViewId);
            Debug.Assert(columnInfo != null);

            // 列項目作成
            var item = new ColumnItem(signature, width, visible)
            {
                Text = textStr,
                TextAlign = textAlign,
                Width = width,
                Visible = visible,
                Tag = columnId,
                GetParamObject = x => x,
            };
            item.CustomDraw = (v, e, x) =>
            {
                var g = e.Graphics;
                var obj = x as GuiObject;
                if (g != null && obj != null)
                {
                    var color = obj.EditColor ?? new RgbaColor();
                    var comment = obj.Comment ?? string.Empty;
                    // カラー枠
                    var rcColor = e.Bounds;
                    rcColor.Inflate(-2, -2);
                    var h = rcColor.Height;
                    MathUtility.Clamp(ref h, 0, 12);
                    rcColor.X = e.Bounds.Right - h - 2;
                    if (rcColor.X <= e.Bounds.Left) rcColor.X = e.Bounds.Left+1;
                    rcColor.Y += (e.Bounds.Height - h - 2) / 2;
                    rcColor.Width = rcColor.Height = h;

                    using (var brush = new SolidBrush(color.ToColor()))
                    {
                        g.SetClip(e.Bounds);
                        g.FillRectangle(brush, rcColor);
                        GraphicsUtility.DrawRectangle(e.Graphics, (color.R == 1 && color.G == 1 && color.B == 1) ? Pens.DimGray : Pens.Black, rcColor);
                    }

                    // 処理しました
                    e.Handled = true;
                }
            };
            item.CustomComparer = (o1, o2) =>
            {
                var obj1 = (GuiObject)o1;
                var obj2 = (GuiObject)o2;

                if (obj1.EditColor == obj2.EditColor)
                {
                    return 0;
                }
                if (!obj1.EditColor.HasValue)
                {
                    return -1;
                }
                if (!obj2.EditColor.HasValue)
                {
                    return 1;
                }
                return CompareItem_Color(obj1.EditColor.Value, obj2.EditColor.Value, false);
            };

            if (isValid != null)
            {
                item.IsValid = isValid;
            }

            RegisterColumn(item, id, configName);
        }

        /// <summary>
        /// 列項目登録終了。
        /// </summary>
        protected static void EndRegisterColumn()
        {
        }
        #endregion

        #region 列項目更新
        /// <summary>
        /// 列項目更新。
        /// </summary>
        private void RefreshColumn()
        {
            //DebugConsole.WriteLine("refresh listview column:[{0}]", this.GetType().Name);
            //DebugConsole.WriteLine(" {0}", this.ColumnInfo.ToString());

            using (var ub = new UpdateBlock(this))
            {
                if (ColumnInfoTable.ContainsKey(_viewId) == false)
                {
                    Initialize(_viewId);
                }

                AppConfig.ListView config = ApplicationConfig.Setting.ObjectView.GetListView(_viewId);

                // 最初に既定順序で作成
                // 行が存在する状態だとColumnのAdd()が遅くなるので、一旦すべての項目と列を削除してしまう。
                Clear();
                AddColumns(config);
            }
        }
        #endregion

        protected virtual void AddColumns(AppConfig.ListView config)
        {
            foreach (ColumnItem column in config.Columns.Where(x => x.Show).Select(x => x.GetColumnItem()).Where(x => x != null && x.Visible))
            {
                Columns.Add(column);
            }
        }

        #region 項目登録
        protected void RegisterItem(GuiObject obj, GuiObjectID objectId, ViewID viewId, string tooltip = null)
        {
            ListViewItem item = null;
            var columnInfo = GetColumnInfo(viewId);
            for (var i = 0; i < columnInfo.RegisteredColumns.Count; i++)
            {
                if (i == 0)
                {
                    item = new ListViewItem(i+ ":dummy") {Tag = obj};//, (int)objectID);
                    item.ToolTipText = tooltip ?? "";
                }
                else
                {
                    var subitem = item.SubItems.Add(i+":dummy");
                }
            }

            Items.Add(item);
        }
        #endregion

        #region 項目更新
        /// <summary>
        /// 選択同期をとる。
        /// </summary>
        protected void SynchronizeSelectedState()
        {
            //DebugConsole.WriteLine("synchronize ObjectListView selected state:[{0}]", this.GetType().Name);

            using (var block = new UIControlEventSuppressBlock())
            {
                if (Items.Count > 0)
                {
                    // 選択状態
                    var selected = App.AppContext.SelectedTarget;
                    foreach (ListViewItem item in Items)
                    {
                        var obj = (GuiObject)item.Tag;
                        SetItemSelectedState(item, obj != null && selected.Contains(obj));
                    }

                    // フォーカス状態
                    if (selected.Active != null)
                    {
                        ListViewItem item = SearchItem(selected.Active);
                        if (item != null)
                        {
                            SetItemFocusedState(item, true);
                        }
                    }
                    if (Focused && FocusedItem == null)
                    {
                        Items[0].Focused = true;
                    }
                }
            }
        }

        /// <summary>
        /// 選択項目中から指定オブジェクトに対応する項目を検索。
        /// </summary>
        private ListViewItem SearchItem(GuiObject obj)
        {
            return SelectedItems.Cast<ListViewItem>().FirstOrDefault(item => item.Tag == obj);
        }

        #endregion

        #region 項目描画

        // 初期項目作成
        protected virtual void RefreshList()
        {
        }
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnCustomDrawItem(CustomDrawListViewItemEventArgs e)
        {
            object o = e.Item.Tag;
            var item = (ColumnItem)e.ColumnHeader;

            if (item.GetSpecificForeColor != null)
            {
                var color = item.GetSpecificForeColor(e, o);

                if (color != null)
                {
                    e.SpecificForeColor = (Color)color;
                }
            }

            if (item.GetSpecificImage != null)
            {
                var image = item.GetSpecificImage(e, o);

                if (image != null)
                {
                    e.SpecificImage = image;
                }
            }

            if (!item.IsValid(o))
            {
                e.SpecificText = string.Empty;
            }
            else if (item.CustomDraw != null)
            {
                item.CustomDraw(this, e, item.GetParamObject(o));
            }
            else
            {
                DefaultCustomDrawItem(e, item.GetParamObject(o));
            }

            base.OnCustomDrawItem(e);
        }

        /// <summary>
        /// 項目描画（デフォルト）。
        /// </summary>
        protected void DefaultCustomDrawItem(CustomDrawListViewItemEventArgs e, object target, BoolValueTextType boolType = BoolValueTextType.UseNotuse)
        {
            if (target is string)
            {
                e.SpecificText = (string)target;
                return;
            }
            if (target is int)
            {
                e.SpecificText = target.ToString();
                return;
            }
            if (target is short)
            {
                e.SpecificText = target.ToString();
                return;
            }
            if (target is bool)
            {
                switch (boolType)
                {
                    case BoolValueTextType.OnOff:
                        e.SpecificText = UIText.FlagOnOff((bool)target);
                        break;
                    case BoolValueTextType.YesNo:
                        e.SpecificText = UIText.FlagYesNo((bool)target);
                        break;
                    case BoolValueTextType.EnableDisable:
                        e.SpecificText = UIText.FlagEnableDisable((bool)target);
                        break;
                    case BoolValueTextType.ExistNotexist:
                        e.SpecificText = UIText.FlagExistOrNot((bool)target);
                        break;
                    case BoolValueTextType.UseNotuse:
                        e.SpecificText = UIText.FlagUseOrNot((bool)target);
                        break;
                    case BoolValueTextType.VisibleNonvisible:
                        e.SpecificText = UIText.FlagVisibleOrNonvisible((bool)target);
                        break;
                    default:
                        e.SpecificText = UIText.FlagUseOrNot((bool)target);
                        break;
                }
                return;
            }
            if (target is float[])
            {
                var targets = target as float[];
                switch (targets.Length)
                {
                    case 3:
                        e.SpecificText = UIText.Float3Label(targets, false);
                        return;
                    case 4:
                        e.SpecificText = UIText.Float4Label(targets, false);
                        return;
                }
                // 配列数が３、４以外の場合はアウト
            }
            else
            {
                e.SpecificText = target.ToString();
                return;
            }
            DrawItem_Invalid(e);
        }

        /// <summary>
        /// 項目描画（未実装部）。
        /// </summary>
        public void DrawItem_Invalid(CustomDrawListViewItemEventArgs e)
        {
            // すぐ気付くように分かりやすく描画
            e.SpecificBackColor = Color.Red;
        }

        /// <summary>
        /// 項目描画（条件付きパラメタ）。
        /// </summary>
        protected void DrawItem_Conditional(CustomDrawListViewItemEventArgs e, string label, bool enableState, bool hasAnimation)
        {
            e.SpecificText = label;

            // 無効前景色
            if (!enableState)
            {
                e.SpecificForeColor = SystemColors.GrayText;
            }

            // アニメーション背景色
            if (hasAnimation)
            {
                e.SpecificBackColor = AnimationColor;
            }
        }

        /// <summary>
        /// 項目描画（条件付きカラー）。
        /// </summary>
        protected void DrawItem_ConditionalColor(CustomDrawListViewItemEventArgs e, RgbaColor color, bool enableAlpha, bool enableState, bool hasAnimation)
        {
            DrawItem_Color(
                e,
                color,
                enableAlpha,
                enableState  ? Color.Empty : SystemColors.GrayText,
                hasAnimation ? AnimationColor : Color.Empty,
                null
            );
        }

        public void DrawItem_Image(CustomDrawListViewItemEventArgs e, Image image, string text)
        {
            // アイコン幅
            Rectangle rcImage = e.Bounds;
            rcImage.X += 2;
            rcImage.Y += 2;
            rcImage.Height = image.Height;
            rcImage.Width = image.Width;

            e.Graphics.DrawImage(image, rcImage);

            // 成分値
            Rectangle rcValue = new Rectangle(rcImage.Right + 2, e.Bounds.Y, e.Bounds.Width - (2 + rcImage.Width + 2), e.Bounds.Height);
            using (StringFormat sf = new StringFormat())
            {
                sf.Alignment = StringAlignment.Near;
                sf.LineAlignment = StringAlignment.Center;
                sf.Trimming = StringTrimming.None;
                sf.FormatFlags |= StringFormatFlags.NoWrap;

                // カラー
                Color textColor = SystemColors.WindowText;
                /*
                if (labelColor != Color.Empty)
                {
                    textColor = labelColor;
                }
                else
                {
                    if (e.IsSelected)
                    {
                        textColor = SystemColors.HighlightText;
                    }
                }*/
                if (e.IsSelected)
                {
                    textColor = SystemColors.HighlightText;
                }

                // テキスト
                using (Brush brush = new SolidBrush(textColor))
                {
                    e.Graphics.DrawString(text, Font, brush, rcValue, sf);
                }
            }

            // 処理しました
            e.Handled = true;



        }

        public void DrawItem_Color(CustomDrawListViewItemEventArgs e, float[] color, string auxiliary = null)
        {
            DrawItem_Color(e,
                           color.Length == 4
                               ? new RgbaColor(color[0], color[1], color[2], color[3])
                               : new RgbaColor(color[0], color[1], color[2], 1),
                               color.Length == 4,
                               auxiliary);
        }

        /// <summary>
        /// 項目描画（カラー）。
        /// </summary>
        protected void DrawItem_Color(CustomDrawListViewItemEventArgs e, RgbaColor color, bool enableAlpha, string auxiliary)
        {
            DrawItem_Color(e, color, enableAlpha, Color.Empty, Color.Empty, auxiliary);
        }

        /// <summary>
        /// 項目描画（カラー）。
        /// </summary>
        protected void DrawItem_Color(CustomDrawListViewItemEventArgs e, RgbaColor color, bool enableAlpha, Color labelColor, Color backColor, string auxiliary)
        {
            // カラー枠
            Rectangle rcColor = e.Bounds;
            rcColor.Inflate(-2, -2);
            rcColor.Width = rcColor.Height;

            using (var brush = new SolidBrush(enableAlpha ? color.ToColor() : color.ToColorNoAlpha()))
            {
                e.Graphics.FillRectangle(brush, rcColor);
                GraphicsUtility.DrawRectangle(e.Graphics, (color.R == 1 && color.G == 1 && color.B == 1)? Pens.DimGray: Pens.Black, rcColor);
            }

            // 成分値
            Rectangle rcValue = new Rectangle(rcColor.Right + 2, e.Bounds.Y, e.Bounds.Width - (2 + rcColor.Width + 2), e.Bounds.Height);
            using (StringFormat sf = new StringFormat())
            {
                sf.Alignment     = StringAlignment.Near;
                sf.LineAlignment = StringAlignment.Center;
                sf.Trimming      = StringTrimming.None;
                sf.FormatFlags  |= StringFormatFlags.NoWrap;

                // カラー
                Color textColor = SystemColors.WindowText;
                if (labelColor != Color.Empty)
                {
                    textColor = labelColor;
                }
                else
                {
                    if (e.IsSelected)
                    {
                        textColor = SystemColors.HighlightText;
                    }
                }

                // テキスト
                using (Brush brush = new SolidBrush(textColor))
                {
                    var text = enableAlpha ? string.Format("{0:F3}, {1:F3}, {2:F3}, {3:F3}", color.R, color.G, color.B, color.A) : string.Format("{0:F3}, {1:F3}, {2:F3}", color.R, color.G, color.B);

                    // 追加
                    if (auxiliary != null)
                    {
                        text += auxiliary;
                    }

                    e.Graphics.DrawString(text, Font, brush, rcValue, sf);
                }
            }

            // 処理しました
            e.Handled = true;
        }
        #endregion

        #region 項目比較
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override int OnCompareItem(CompareListViewItemEventArgs e)
        {
            var o1 = e.Item1.Tag;
            var o2 = e.Item2.Tag;

            var item = (ColumnItem)e.ColumnHeader;
            bool isValid1 = item.IsValid(o1);
            bool isValid2 = item.IsValid(o2);
            int validCompare = isValid1.CompareTo(isValid2);
            if (validCompare != 0)
            {
                return validCompare;
            }

            if (isValid1)
            {
                return item.CustomComparer != null ?
                    item.CustomComparer(item.GetParamObject(o1), item.GetParamObject(o2)) :
                    DefaultCompareItem(e, item.GetParamObject(o1), item.GetParamObject(o2));
            }
            else
            {
                return 0;
            }
        }

        /// <summary>
        /// 比較。
        /// </summary>
        protected static int DefaultCompareItem(CompareListViewItemEventArgs e, object target1, object target2)
        {
            if (target1 is IComparable)
            {
                return ((IComparable)target1).CompareTo(target2);
            }
            else if (target1 is System.Collections.IEnumerable)
            {
                var enum1 = ((System.Collections.IEnumerable)target1).GetEnumerator();
                var enum2 = ((System.Collections.IEnumerable)target2).GetEnumerator();
                while (enum1.MoveNext() && enum2.MoveNext())
                {
                    if (enum1.Current is IComparable)
                    {
                        int result = ((IComparable)enum1.Current).CompareTo(enum2.Current);
                        if (result != 0)
                        {
                            return result;
                        }
                    }
                }

                return 0;
            }
            else if (target1 is GuiObject)
            {
                return target1.ToString().CompareTo(target2.ToString());
            }
            Debug.Assert(false);
            return 0;
        }

        /// <summary>
        /// カラー比較。
        /// </summary>
        protected static int CompareItem_Color(RgbaColor color1, RgbaColor color2, bool enableAlpha)
        {
            float intensity1 = color1.R + color1.G + color1.B;
            float intensity2 = color2.R + color2.G + color2.B;
            if (enableAlpha)
            {
                intensity1 += color1.A;
                intensity2 += color2.A;
            }

            // 明度で比較
            if (intensity1 < intensity2) { return -1; }
            if (intensity1 > intensity2) { return  1; }

            // 成分で比較
            if (enableAlpha)
            {
                if (color1.A < color2.A)
                {
                    return -1;
                }
                if (color1.A > color2.A)
                {
                    return 1;
                }
            }
            if (color1.B < color2.B)
            {
                return -1;
            }
            if (color1.B > color2.B)
            {
                return 1;
            }
            if (color1.G < color2.G)
            {
                return -1;
            }
            if (color1.G > color2.G)
            {
                return 1;
            }
            if (color1.R < color2.R)
            {
                return -1;
            }
            if (color1.R > color2.R)
            {
                return 1;
            }
            //if      (component1 < component2) { return -1; }
            //else if (component1 > component2) { return  1; }
            return 0;
        }
        #endregion

        #region オーバーライド
        //---------------------------------------------------------------------
        // これ以上継承させないものは sealed をつけています
        //---------------------------------------------------------------------
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnSelectedIndexChanged(EventArgs e)
        {
            // SynchronizeSelectedState() でのイベント受け取らないようにする
            if (!UIControlEventSuppressBlock.Enabled)
            {
                base.OnSelectedIndexChanged(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnSelectionChanged(EventArgs e)
        {
            // 選択オブジェクトグループ作成
            var selected = new GuiObjectGroup();
            foreach (ListViewItem item in SelectedItems)
            {
                if (item.Tag != null)
                {
                    selected.Add((GuiObject)item.Tag, item.Focused);
                }
            }
            if (!selected.IsEmpty && selected.Active == null)
            {
                selected.Active = selected.Objects[0];
            }

            // 選択オブジェクト変更
            _selectionChanging = true;
            {
                App.AppContext.SelectedTarget.Set(selected.Objects, selected.Active, true);
            }
            _selectionChanging = false;

            base.OnSelectionChanged(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnKeyUp(KeyEventArgs e)
        {
            // コンテキストメニュー
            if (e.KeyCode == Keys.Apps)
            {
                // コンテキストメニュー
                // 項目位置が取れないのでカーソル位置に表示
                TheApp.MainFrame.ShowObjectMenu(Cursor.Position, this);
            }
            base.OnKeyUp(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                ListViewItem item = GetItemAt(e.X, e.Y);
                if (item != null && item.Tag != null)
                {
                    // コンテキストメニュー
                    TheApp.MainFrame.ShowObjectMenu(PointToScreen(new Point(e.X, e.Y)), this);
                }
            }
            base.OnMouseUp(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnMouseDoubleClick(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                ListViewItem item = GetItemAt(e.X, e.Y);
                if (item != null && item.Tag != null)
                {
                    // プロパティウィンドウ表示
                    PropertyEdit.ObjectPropertyDialog.ShowPropertyDialog();
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnColumnReordered(ColumnReorderedEventArgs e)
        {
            // 順序未変更時は何もしない
            if (e.OldDisplayIndex == e.NewDisplayIndex) { return; }

            // 列情報を更新
            var config = ConfigData.ApplicationConfig.Setting.ObjectView.GetListView(_viewId);
            // TODO:整理
            var visibleColumns = config.Columns.Where(x => x.Show && x.GetColumnItem() != null && x.GetColumnItem().Visible).ToArray();
            var oldPosItem = visibleColumns[e.OldDisplayIndex];
            var newPosItem = visibleColumns[e.NewDisplayIndex];
            var oldPosIndex = config.Columns.IndexOf(oldPosItem);
            var newPosIndex = config.Columns.IndexOf(newPosItem);
            DebugConsole.WriteLine("old: " + oldPosIndex + " new: " + newPosIndex);
            config.Columns.RemoveAt(oldPosIndex);
            config.Columns.Insert(newPosIndex, oldPosItem);
            DebugConsole.WriteLine("column reordered:{0}", string.Concat(config.Columns.Select(x => x.Name + ": ")));
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnColumnDividerDoubleClick(ColumnDividerDoubleClickEventArgs e)
        {
            // 列幅を既定値にする
            var column = (ColumnItem)e.Header;
            column.Width = column.DefaultWidth;

            // 既定処理させない
            e.Handled = true;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnColumnContextMenuPopup(ColumnMouseEventArgs e)
        {
            // 設定ダイアログの初期表示ページを設定
            ColumnSettingDialog.InitialPageViewID = _viewId;

            // コンテキストメニュー表示
            TheApp.MainFrame.ShowListViewColumnMenu(PointToScreen(e.Location));
        }
        #endregion

        public void ColumnFilterChanged()
        {
            // 列項目更新
            RefreshColumn();

            // 項目更新
            RefreshList();
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------------
        // 列項目設定変更
        private void columnInfo_SettingChanged(object sender, EventArgs e)
        {
            // 列項目更新
            RefreshColumn();

            // 項目更新
            RefreshList();
        }
        #endregion

        protected override void OnKeyDown(KeyEventArgs e)
        {
            if ((e.Modifiers == Keys.Control) && (e.KeyCode == Keys.A))
            {
                SelectAll();
            }
            else if (e.KeyCode == Keys.Enter)
            {
                // プロパティウィンドウ表示
                PropertyEdit.ObjectPropertyDialog.ShowPropertyDialog();
            }
            else
            {
                base.OnKeyDown(e);
            }
        }

        // 選択アイテムが変わらなくてもViewerに選択を通知するために
        // OnSelectedIndexChangedを呼ぶ。
        // クリックで選択アイテムが変わっても、UIListViewのOnSelectedIndexChangedで
        // バッファされるので、複数回呼び出しにはならない。
        protected override void OnMouseClick(MouseEventArgs e)
        {
            OnSelectedIndexChanged(EventArgs.Empty);
            base.OnMouseClick(e);
        }

        public virtual AppConfig.ObjectViewMode ViewMode
        {
            get;
            private set;
        }

        public virtual Control Control
        {
            get { return this; }
        }

        public virtual void FocusClient()
        {
            Focus();
        }

        public virtual void SelectAll()
        {
            // 選択オブジェクト変更
            _selectionChanging = true;
            {
                foreach(ListViewItem item in Items)
                {
                    item.Selected = true;
                }
            }
            _selectionChanging = false;
        }

        public virtual void ToggleSelection()
        {
            // 選択オブジェクト変更
            _selectionChanging = true;
            {
                foreach(ListViewItem item in Items)
                {
                    item.Selected = ! item.Selected;
                }
            }
            _selectionChanging = false;
        }

        public virtual bool EnableSelectAll
        {
            get { return true; }
        }

        public virtual bool EnableToggleSelection
        {
            get { return true; }
        }

        internal static void NotifyColumnSettingChanged()
        {
            var columnInfo = GetColumnInfo(ColumnRegisterViewId);
            if (columnInfo != null)
            {
                columnInfo.NotifySettingChanged();
            }
        }
    }

    #region ViewID
    /// <summary>
    /// ビューＩＤ。
    /// </summary>
    public enum ViewID
    {
        Model,
        Material,
        Shape,
        Texture,
        Bone,
    }
    #endregion

    #region ColumnInfo
    /// <summary>
    /// 列情報クラス。
    /// </summary>
    public sealed class ColumnInfo
    {
        // 列項目リスト
        private readonly List<ColumnItem> _registeredColumns = new List<ColumnItem>();

        /// <summary>設定変更イベント。</summary>
        public event EventHandler SettingChanged = null;

        /// <summary>
        /// 順序付き列項目リスト。
        /// </summary>
        public List<ColumnItem> RegisteredColumns
        {
            get { return _registeredColumns; }
        }

        /// <summary>
        /// 列項目登録。
        /// </summary>
        public void RegisterColumn(ColumnItem column)
        {
            // 参照はお互い同じ物を共有する事に注意
            //_defaultColumns.Add(column);
            //_initialColumns.Add(column);
            _registeredColumns.Add(column);
        }

        /// <summary>
        /// 設定変更をリストに通知する。
        /// </summary>
        public void NotifySettingChanged()
        {
            //Debug.Assert(_defaultColumns.Count == _orderedColumns.Count);

            if (SettingChanged != null)
            {
                SettingChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            string result = string.Empty;
            foreach (ColumnItem column in _registeredColumns)
            {
                if (column.Visible)
                {
                    result += string.Format("[{0}:{1}]", column.Signature, column.Width);
                }
                else
                {
                    result += string.Format("[({0}):{1}]", column.Signature, column.Width);
                }
            }
            return result;
        }
    }
    #endregion

    #region ColumnItem
    /// <summary>
    /// 列項目クラス。
    /// </summary>
    public sealed class ColumnItem : ColumnHeader, AppConfig.Column.IColumnItem
    {
        // シグネチャ（プロファイル用）
        private readonly string _signature = string.Empty;
        // デフォルト幅
        private readonly int _defaultWidth = 0;
        // デフォルト表示フラグ
        private readonly bool _defaultVisible = true;
        // 表示フラグ
        private bool _visible = true;

        public ColumnItem()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnItem(string signature, int defaultWidth, bool defaultVisible)
        {
            _signature      = signature;
            _defaultWidth   = defaultWidth;
            _defaultVisible = defaultVisible;
        }

        /// <summary>
        /// シグネチャ。
        /// </summary>
        public string Signature
        {
            get { return _signature; }
        }

        /// <summary>
        /// デフォルト幅。
        /// </summary>
        public int DefaultWidth
        {
            get { return _defaultWidth; }
        }

        /// <summary>
        /// デフォルト表示フラグ。
        /// </summary>
        public bool DefaultVisible
        {
            get { return _defaultVisible; }
        }

        /// <summary>
        /// 表示フラグ。
        /// </summary>
        public bool Visible
        {
            get { return _visible; }
            set { _visible = value; }
        }

        /// <summary>
        /// タイプ
        /// </summary>
        public string Type { get; set; }

        /// <summary>
        /// ページID
        /// </summary>
        public App.PropertyEdit.PropertyPageID PageID { get; set; }

        /// <summary>デリゲート用関数宣言</summary>
        //public delegate S GetParamObject( data);
        /// <summary>デリゲート関数インスタンス</summary>
        public Func<object, object> GetParamObject = null;

        public Comparison<object> CustomComparer = null;

        public Func<CustomDrawListViewItemEventArgs, object, Color?> GetSpecificForeColor = null;
        public Func<CustomDrawListViewItemEventArgs, object, Image> GetSpecificImage = null;
        public Func<CustomDrawListViewItemEventArgs, object, String> GetSpecificTooltip = null;

        /// <summary>
        /// object は getParamObject の第二引数型
        /// </summary>
        public Action<ObjectListView, CustomDrawListViewItemEventArgs, object> CustomDraw = null;

        public Predicate<object> IsValid = x => true;
    }
    #endregion
}
