﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Controls;
using App.Utility;
using App.PropertyEdit;
using App.res;
using App.Data;

namespace App.ObjectView.List
{
    using AppConfig;

    using IdLabelType = Tuple<string, string, string>;

    /// <summary>
    /// 列項目設定ページクラス。
    /// </summary>
    public sealed partial class ColumnSettingPanel : UIUserControl
    {
        // ビューＩＤ
        private readonly ViewID _viewID;
        // 列情報
        private readonly ColumnInfo _columnInfo;
        // リスト初期化フラグ
        private bool _listInitialized = false;
        // 編集フラグ
        private bool _modified = false;

        private AppConfig.ListView Config;

        public class IndexedListViewItem : ListViewItem
        {
            public bool Deleted;
        }

        internal List<IndexedListViewItem> ListViewItems = new List<IndexedListViewItem>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnSettingPanel(ViewID viewID)
        {
            InitializeComponent();

            // マテリアル以外はあまり項目がないのでツリービューは非表示にする
            if (viewID != ViewID.Material)
            {
                uiSplitContainer1.Panel1Collapsed = true;
                uiSplitContainer1.IsSplitterFixed = true;
            }

            this._viewID = viewID;
            _columnInfo = ObjectListView.GetColumnInfo(viewID);

            bool btnDeleteVisible = false;
            switch (viewID)
            {
                case ViewID.Material:
                case ViewID.Shape:
                    btnDeleteVisible = true;
                    break;
            }
            btnDeleteAll.Visible = btnDeleteVisible;
            btnDelete.Visible = btnDeleteVisible;
            btnDeleteAll.Enabled = false;
            btnDelete.Enabled = false;

            UpdateCategoryTreeView();

            // 項目作成
            this.InitializeListView();

            UpdateItemListView();

            // フォーム更新
            UpdateForm();
        }

        public void InitializeListView()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                this.ListViewItems.Clear();
                this.lvwItems.Items.Clear();
                this.Config = ApplicationConfig.Setting.ObjectView.GetListView(this.ViewID);
                Debug.Assert(this.Config != null);
                foreach (var column in this.Config.Columns)
                {
                    var item = new IndexedListViewItem
                                   {
                                       Text = column.Text ?? column.Name,
                                       Tag = column,
                                       Checked = column.Show,
                                   };

                    if (column.GetColumnItem() == null)
                    {
                        item.ForeColor = SystemColors.GrayText;
                        this.btnDeleteAll.Enabled = true;
                    }
                    this.ListViewItems.Add(item);
                    this.lvwItems.Items.Add(item);
                }

                Debug.Assert(this.lvwItems.Items.Count > 0);
                this.lvwItems.Items[0].Focused = true;
                this.lvwItems.Items[0].Selected = true;
            }
        }

        private class CategoryNode : TreeNode
        {
            public string Category;

            public CategoryNode()
            {
                Contains = DefaultContains;
            }

            public new int ImageIndex
            {
                set
                {
                    base.SelectedImageIndex = base.ImageIndex = value;
                }
            }

            public Predicate<IndexedListViewItem> Contains;

            public bool DefaultContains(IndexedListViewItem item)
            {
                return ((AppConfig.Column)(item.Tag)).Category == Category;
            }
        }

        private void UpdateCategoryTreeView()
        {
            switch (this.ViewID)
            {
                case ViewID.Material:
                    UpdateMaterialCategoryTreeView();
                    break;
                case ViewID.Model:
                    tvCategory.Nodes.Clear();
                    tvCategory.Nodes.Add(new CategoryNode()
                    {
                        Text = Strings.ObjectPropertyPanel_Model_Root,
                        Category = PropertyPageID.ModelRoot.ToString(),
                        ImageIndex = 0,
                        Contains = x => true,
                    });
                    break;
                case ViewID.Texture:
                    tvCategory.Nodes.Clear();
                    tvCategory.Nodes.Add(new CategoryNode()
                    {
                        Text = Strings.ObjectPropertyPanel_Texture_Root,
                        Category = PropertyPageID.TextureRoot.ToString(),
                        ImageIndex = 0,
                        Contains = x => true,
                    });
                    break;
                case ViewID.Shape:
                    tvCategory.Nodes.Clear();
                    tvCategory.Nodes.Add(new CategoryNode()
                    {
                        Text = Strings.ObjectPropertyPanel_Shape_Root,
                        Category = PropertyPageID.ShapeRoot.ToString(),
                        ImageIndex = 0,
                        Contains = x => true,
                    });
                    break;
                case ViewID.Bone:
                    tvCategory.Nodes.Clear();
                    tvCategory.Nodes.Add(new CategoryNode()
                    {
                        Text = Strings.ObjectPropertyPanel_Bone_Root,
                        Category = PropertyPageID.BoneRoot.ToString(),
                        ImageIndex = 0,
                        Contains = x => true,
                    });
                    break;
                default:
                    tvCategory.Nodes.Clear();
                    tvCategory.Nodes.Add(new CategoryNode()
                    {
                    });
                    break;
            }
        }

        private void UpdateMaterialCategoryTreeView()
        {
            tvCategory.Nodes.Clear();
            var root = new CategoryNode()
            {
                Text = Strings.ObjectPropertyPanel_Material_Root,
                Category = PropertyPageID.MaterialRoot.ToString(),
                ImageIndex = 0,
            };
            var general = new CategoryNode()
            {
                Text = Strings.ObjectPropertyPanel_Material_General,
                Category = PropertyPageID.MaterialGeneral.ToString(),
                ImageIndex = 1,
            };
            var preview = new CategoryNode()
            {
                Text = Strings.ObjectPropertyPanel_Material_Preview,
                Category = PropertyPageID.MaterialPreview.ToString(),
                ImageIndex = 1,
            };
            var renderState = new CategoryNode()
            {
                Text = Strings.ObjectPropertyPanel_Material_RenderState,
                Category = PropertyPageID.MaterialRenderState.ToString(),
                ImageIndex = 1,
            };
            var sampler = new CategoryNode()
            {
                Text = Strings.ObjectPropertyPanel_Material_Sampler,
                Category = PropertyPageID.MaterialSampler.ToString(),
                ImageIndex = 1,
            };
            var shader = new CategoryNode()
            {
                Text = Strings.ObjectPropertyPanel_Material_Shader,
                Category = PropertyPageID.MaterialShader.ToString(),
                ImageIndex = 1,
            };

            tvCategory.Nodes.Add(root);
            root.Nodes.Add(general);
            root.Nodes.Add(preview);
            if (ApplicationConfig.DefaultValue.RenderStateInfoVisible
                && App.AppContext.SelectedPlatformPreset.UseNw)
            {
                root.Nodes.Add(renderState);
            }
            root.Nodes.Add(sampler);
            root.Nodes.Add(shader);
            root.Expand();

            tvCategory.SelectedNode = root.Nodes.OfType<CategoryNode>()
                .FirstOrDefault(x => x.Category == ApplicationConfig.Setting.ObjectView.MaterialListView.Filter)
                ?? root;

            var materialGroups = (from model in DocumentManager.Models
                                 from material in model.Materials
                                 let shadingModel = material.MaterialShaderAssign.ShadingModel
                                 group material by shadingModel);

            var pages = (from materialGroup in materialGroups
                         let material = materialGroup.First()
                         where material.MaterialShaderAssign.PageMode
                         from g in material.MaterialShaderAssign.GetGroups(true)
                         let pageName = g.@group != null ? g.@group.page_name : null
                         let page = materialGroup.Key.Pages().FirstOrDefault(x => x.name == pageName) ?? MaterialPropertyPanel.defaultPage
                         orderby page.Order(), page.index
                         group new { g, material } by page.Label() ?? page.name);

            //var pageNodes = new HashSet<string, CategoryNode>();

            foreach (var page in pages)
            {
                var pageNode = new CategoryNode()
                {
                    Text = page.Key,
                    Category = PropertyPageID.MaterialShader.ToString(),
                    ImageIndex = 3,
                };

                pageNode.Contains = x => false;

                var nodes = new Dictionary<string, CategoryNode>();
                var items = new Dictionary<string, IdLabelType>();
                foreach (var group in page.GroupBy(x => x.g.Label))
                {
                    var groupNode = new CategoryNode()
                    {
                        Text = group.Key,
                        Category = PropertyPageID.MaterialShader.ToString(),
                        ImageIndex = 4,
                    };

                    if (group.Key != ShaderTypeUtility.RemainsGroupLabel())
                    {
                        pageNode.Nodes.Add(groupNode);
                    }

                    nodes.Add(group.Key, pageNode);

                    HashSet<IdLabelType> idLabels = new HashSet<Tuple<string, string, string>>();
                    foreach (var p in group)
                    {
                        var material = p.material;
                        var groupId = p.g.groupName;
                        var shadingModel = material.MaterialShaderAssign.ShadingModel;
                        foreach (var item in shadingModel.Attributes().Where(x => x.Group() == groupId))
                        {
                            idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.attrib_assign.ToString()));
                        }

                        foreach (var item in shadingModel.Options().Where(x => x.Group() == groupId))
                        {
                            idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.shader_option.ToString()));
                        }

                        foreach (var item in shadingModel.Samplers().Where(x => x.Group() == groupId))
                        {
                            idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.sampler_assign.ToString()));
                        }

                        foreach (var item in shadingModel.MaterialUniforms().Where(x => x.Group() == groupId))
                        {
                            idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.shader_param.ToString()));
                        }

                        foreach (var item in shadingModel.RenderInfoSlots().Where(x => x.Group() == groupId))
                        {
                            idLabels.Add(new IdLabelType(item.name, item.Label() ?? item.name, ColumnTypes.render_info.ToString()));
                        }
                    }

                    if (group.Key != ShaderTypeUtility.RemainsGroupLabel())
                    {
                        groupNode.Contains = x =>
                        {
                            var column = (AppConfig.Column)x.Tag;
                            return groupNode.DefaultContains(x) &&
                                idLabels.Contains(new IdLabelType(column.Id, column.Name, column.Type));
                        };
                    }
                    else
                    {
                        pageNode.Contains = x =>
                        {
                            var column = (AppConfig.Column)x.Tag;
                            return pageNode.DefaultContains(x) &&
                                idLabels.Contains(new IdLabelType(column.Id, column.Name, column.Type));
                        };
                    }
                }

                shader.Nodes.Add(pageNode);
            }

            var groups = from materialGroup in materialGroups
                         let material = materialGroup.First()
                         where !material.MaterialShaderAssign.PageMode
                         from g in material.MaterialShaderAssign.GetGroups(false)
                         group new { g, material } by g.Label;
            foreach (var group in groups)
            {
                var groupNode = new CategoryNode()
                {
                    Text = group.Key,
                    Category = PropertyPageID.MaterialShader.ToString(),
                    ImageIndex = 4,
                };

                shader.Nodes.Add(groupNode);

                HashSet<IdLabelType> idLabels = new HashSet<IdLabelType>();
                foreach (var p in group)
                {
                    var material = p.material;
                    var groupId = p.g.groupName;
                    var shadingModel = material.MaterialShaderAssign.ShadingModel;
                    foreach (var item in shadingModel.Attributes().Where(x => x.Group() == groupId))
                    {
                        idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.attrib_assign.ToString()));
                    }

                    foreach (var item in shadingModel.Options().Where(x => x.Group() == groupId))
                    {
                        idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.shader_option.ToString()));
                    }

                    foreach (var item in shadingModel.Samplers().Where(x => x.Group() == groupId))
                    {
                        idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.sampler_assign.ToString()));
                    }

                    foreach (var item in shadingModel.MaterialUniforms().Where(x => x.Group() == groupId))
                    {
                        idLabels.Add(new IdLabelType(item.id, item.Label() ?? item.id, ColumnTypes.shader_param.ToString()));
                    }

                    foreach (var item in shadingModel.RenderInfoSlots().Where(x => x.Group() == groupId))
                    {
                        idLabels.Add(new IdLabelType(item.name, item.Label() ?? item.name, ColumnTypes.render_info.ToString()));
                    }
                }

                groupNode.Contains = x =>
                {
                    var column = (AppConfig.Column)x.Tag;
                    return groupNode.DefaultContains(x) &&
                        idLabels.Contains(new IdLabelType(column.Id, column.Name, column.Type));
                };
            }

            shader.Expand();
        }

        public void UpdateItemListView()
        {
            var selectedNode = (CategoryNode)tvCategory.SelectedNode;
            var predicates = selectedNode != null ?
                CollectNodes(selectedNode).Select(x => x.Contains):
                Enumerable.Repeat((Predicate<IndexedListViewItem>)(x => true), 1);

            // 項目作成
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                lvwItems.BeginUpdate();
                lvwItems.Items.Clear();
                foreach (var item in ListViewItems.Where(x => !x.Deleted && predicates.Any(p => p(x))))
                {
                    lvwItems.Items.Add(item);
                }

                lvwItems.EndUpdate();
            }

            UpdateForm();
        }

        private IEnumerable<CategoryNode> CollectNodes(CategoryNode node)
        {
            return Enumerable.Repeat(node, 1).Concat(node.Nodes.OfType<CategoryNode>().SelectMany(x => CollectNodes(x)));
        }

        /// <summary>
        /// 無効な状態か。
        /// </summary>
        public bool IsInvalidateState
        {
            get { return !ListViewItems.Any(x => !x.Deleted && x.Checked); }
        }

        internal bool Modified
        {
            get
            {
                return this._modified;
            }
        }

        internal ViewID ViewID
        {
            get
            {
                return this._viewID;
            }
        }

        /// <summary>
        /// ＯＫ時の処理。
        /// ColumnSettingDialog から呼ばれます。
        /// </summary>
        public void OnResultOk()
        {
            // 編集された時のみ
            if (this.Modified)
            {
                List<AppConfig.Column> columns = new List<AppConfig.Column>();
                foreach (IndexedListViewItem item in ListViewItems)
                {
                    if (!item.Deleted)
                    {
                        var column = (AppConfig.Column)item.Tag;
                        column.Show = item.Checked;
                        columns.Add(column);
                    }
                }
                this.Config.Columns = columns;
                //ObjectListView.Initialize();
                ObjectListView.Initialize(this.ViewID);
                _columnInfo.NotifySettingChanged();
            }
        }

        /// <summary>
        /// フォーム更新。
        /// </summary>
        private void UpdateForm()
        {
            if (lvwItems.SelectedItems.Count > 0)
            {
                int selectedIndex = lvwItems.SelectedIndices[0];
                btnUp.Enabled   = selectedIndex > 0;
                btnDown.Enabled = selectedIndex < (lvwItems.Items.Count - 1);
                btnDelete.Enabled = ((AppConfig.Column)lvwItems.SelectedItems[0].Tag).GetColumnItem() == null;
            }
            else
            {
                btnUp.Enabled   = false;
                btnDown.Enabled = false;
                btnDelete.Enabled = false;
            }

            btnReset.Enabled = tvCategory.SelectedNode == null || tvCategory.Nodes[0] == tvCategory.SelectedNode;
        }

        /// <summary>
        /// 編集状態にする。
        /// </summary>
        internal void SetModified()
        {
            if (!this.Modified)
            {
                DebugConsole.WriteLine("modify setting");
                this._modified = true;
            }
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // 項目リスト
        private void lvwItems_VisibleChanged(object sender, EventArgs e)
        {
            // ItemChecked イベントが項目作成中に発生するため
            // 表示されるのをもって初期化済みとする...悔しい。
            if (lvwItems.Visible)
            {
                _listInitialized = true;
            }
        }

        private void lvwItems_SizeChanged(object sender, EventArgs e)
        {
            lvwItems.BeginInvokeOrExecute(() =>
                {
                    // 列幅を最大にする
                    clhItem.Width = lvwItems.ClientSize.Width;
                });
        }

        private void lvwItems_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (_listInitialized)
            {
                SetModified();
            }
        }

        private void lvwItems_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateForm();
        }

        //---------------------------------------------------------------------
        // 上下移動
        private void btnUp_Click(object sender, EventArgs e)
        {
            var index = lvwItems.SelectedIndex;
            var target = (IndexedListViewItem)lvwItems.Items[index - 1];
            var selected = ((IndexedListViewItem)lvwItems.SelectedItems[0]);
            var next = selected;
            int start = ListViewItems.IndexOf(target);
            int last = ListViewItems.IndexOf(selected);
            for (int i = start; i <= last; i++)
            {
                var tmp = ListViewItems[i];
                ListViewItems[i] = next;
                next = tmp;
            }

            lvwItems.MoveSelectedItemPrev();
            lvwItems.Focus();

            UpdateForm();
            SetModified();
        }

        private void btnDown_Click(object sender, EventArgs e)
        {
            var index = lvwItems.SelectedIndex;
            var target = (IndexedListViewItem)lvwItems.Items[index + 1];
            var selected = ((IndexedListViewItem)lvwItems.SelectedItems[0]);
            var next = selected;
            int start = ListViewItems.IndexOf(target);
            int last = ListViewItems.IndexOf(selected);
            for (int i = start; i >= last; i--)
            {
                var tmp = ListViewItems[i];
                ListViewItems[i] = next;
                next = tmp;
            }

            lvwItems.MoveSelectedItemNext();
            lvwItems.Focus();

            UpdateForm();
            SetModified();
        }

        //---------------------------------------------------------------------
        // リセット
        private void btnReset_Click(object sender, EventArgs e)
        {
            // 初期順序＆表示フラグのリストを作成
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                lvwItems.BeginUpdate();
                {
                    ListViewItems.Clear();
                    lvwItems.Items.Clear();
                    foreach (var column in this.Config.Columns.Where(x => x.GetColumnItem() != null).OrderBy(x => x.DefaultOrder))
                    {
                        var item = new IndexedListViewItem
                        {
                            Text = column.Text ?? column.Name,
                        };
                        ListViewItems.Add(item);
                        item.Tag = column;
                        item.Checked = column.DefaultShow;
                        item.ForeColor = column.GetColumnItem() != null ? SystemColors.ControlText : SystemColors.GrayText;
                        lvwItems.Items.Add(item);
                    }

                    Debug.Assert(lvwItems.Items.Count > 0);
                    lvwItems.Items[0].Focused = true;
                    lvwItems.Items[0].Selected = true;
                }
                lvwItems.EndUpdate();
            }

            lvwItems.Focus();
            btnDeleteAll.Enabled = false;
            UpdateForm();
            SetModified();
        }
        #endregion

        private void btnDelete_Click(object sender, EventArgs e)
        {
            // 初期順序＆表示フラグのリストを作成
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                lvwItems.BeginUpdate();
                {
                    int index = lvwItems.SelectedIndex;
                    ((IndexedListViewItem)lvwItems.Items[index]).Deleted = true;
                    lvwItems.Items.RemoveAt(lvwItems.SelectedIndex);
                    index = Math.Min(index, lvwItems.Items.Count - 1);

                    Debug.Assert(lvwItems.Items.Count > 0);
                    lvwItems.Items[index].Focused = true;
                    lvwItems.Items[index].Selected = true;
                }
                lvwItems.EndUpdate();
            }

            lvwItems.Focus();
            btnDeleteAll.Enabled = lvwItems.Items.Cast<ListViewItem>().Any(x => ((AppConfig.Column)x.Tag).GetColumnItem() == null);
            UpdateForm();
            SetModified();
        }

        private void btnDeleteAll_Click(object sender, EventArgs e)
        {
            // 初期順序＆表示フラグのリストを作成
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                lvwItems.BeginUpdate();
                {
                    foreach (IndexedListViewItem item in lvwItems.Items)
                    {
                        if (((AppConfig.Column)item.Tag).GetColumnItem() == null)
                        {
                            item.Deleted = true;
                        }
                    }
                    var items = lvwItems.Items.Cast<ListViewItem>().Where(x => ((AppConfig.Column)x.Tag).GetColumnItem() != null).ToArray();
                    lvwItems.Items.Clear();
                    lvwItems.Items.AddRange(items);

                    Debug.Assert(lvwItems.Items.Count > 0);
                    lvwItems.Items[0].Focused = true;
                    lvwItems.Items[0].Selected = true;
                }
                lvwItems.EndUpdate();
            }

            btnDeleteAll.Enabled = false;
            lvwItems.Focus();
            UpdateForm();
            SetModified();
        }

        private void btnCheckAll_Click(object sender, EventArgs e)
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                lvwItems.BeginUpdate();
                foreach (ListViewItem item in lvwItems.Items)
                {
                    item.Checked = true;
                }
                lvwItems.EndUpdate();
            }

            UpdateForm();
            SetModified();
        }

        private void btnClearAll_Click(object sender, EventArgs e)
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                lvwItems.BeginUpdate();
                foreach (ListViewItem item in lvwItems.Items)
                {
                    item.Checked = false;
                }
                lvwItems.EndUpdate();
            }

            UpdateForm();
            SetModified();
        }

        private void tvCategory_AfterSelect(object sender, TreeViewEventArgs e)
        {
            UpdateItemListView();
        }
    }
}
