﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;

using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.FileView;
using App.ObjectView;
using App.ObjectView.Schematic;
using App.PropertyEdit;
using App.res;
using App.Utility;

using AppConfig;

using ConfigCommon;

using nw.g3d.nw4f_3dif;

using TeamConfig;

using Viewer;

namespace App
{
    using nw.g3d.iflib;

    public partial class MainFrameContextMenuItems : UserControl
    {
        public MainFrameContextMenuItems()
        {
            InitializeComponent();
        }

        private void cmiCreateSceneAnimation2_MenuCommandHandler(MenuCommandArgs args)
        {
            AnimationSet owner = (App.AppContext.CurrentSelectedObject(args) as AnimationSet) ??
                DocumentManager.DefaultSceneAnimationSet;

            // 更新時
            if (args.RequireUpdate)
            {
                //				args.CommandUI.Enabled = owner != null;
            }
            // 実行時
            else
            {
                string animationName = null;
                using (var dialog = new SceneAnimationCreateDialog(GuiObjectID.SceneAnimation))
                {
                    if (dialog.ShowDialog(this) == DialogResult.OK)
                    {
                        animationName = dialog.AnimationName;
                    }
                }

                if (animationName != null)
                {
                    AnimationDocument document = null;
                    var data = new scene_animType()
                    {
                        scene_anim_info = new scene_anim_infoType()
                        {
                            frame_resolution = 1,
                            dcc_preset = "",
                            dcc_magnify = 1,
                            dcc_start_frame = 0,
                            dcc_end_frame = 60,
                            dcc_fps = (float)ApplicationConfig.Preview.Fps,
                            bake_all = false,
                            //
                            bake_tolerance_rotate = 0.1f,
                            bake_tolerance_translate = 0.01f,
                            bake_tolerance_color = 0.001f,
                            quantize_tolerance_rotate = 0.2f,
                            quantize_tolerance_translate = 0.01f,
                        }
                    };

                    var anim = new SceneAnimation(data, new List<G3dStream>())
                    {
                        Name = animationName,
                        FileDotExt = G3dPath.SceneAnimBinaryExtension,
                    };

                    document = anim;
                    document.UpdateSavedData();

                    // 作ったドキュメントをマネージャーに登録します
                    Debug.Assert(document != null);
                    //document.IsModifiedObject = true;
                    EditCommandSet commandSet = new EditCommandSet();
                    using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                    {
                        using (var vdsb = new ViewerDrawSuppressBlock())
                        {
                            commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(document, 1), true).Execute());

                            if (owner != null)
                            {
                                commandSet.Add(DocumentManager.CreateAnimationsEditCommand(owner, owner.Animations.Concat(Enumerable.Repeat(
                                    new AnimationSetItem(document.FileName, document.FileLocation), 1)).ToArray(), false).Execute());
                                commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(Enumerable.Repeat(document, 1)).Execute());
                            }
                        }
                    }
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                }
            }
        }

        private void cmiUnbindAllAnimationFromScene_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.DefaultSceneAnimationSet.Animations.Any() || DocumentManager.SceneAnimationSets.Any();
                return;
            }

            EditCommandSet commandSet = new EditCommandSet();
            commandSet.Add(DocumentManager.CreateAnimationsEditCommand(DocumentManager.DefaultSceneAnimationSet, new AnimationSetItem[] { }, false));
            commandSet.Add(DocumentManager.CreateSceneAnimationSetEditCommand(new List<AnimationSet>()));
            TheApp.CommandManager.Execute(commandSet);
        }

        private void cmiUnbindSceneAnim_MenuCommandHandler(MenuCommandArgs args)
        {
            AnimationDocument animation = App.AppContext.CurrentSelectedObject(args) as AnimationDocument;
            AnimationSet hasAnimation = (App.AppContext.SelectedFileViewObjectOwner as AnimationSet)
                ?? DocumentManager.DefaultSceneAnimationSet;

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = animation != null;
                return;
            }

            DocumentManager.UnbindSceneAnimation(animation, hasAnimation);
        }

        private void cmiFileOpenSceneAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            AnimationSet animationOwner = (App.AppContext.CurrentSelectedObject(args) as AnimationSet);

            if (animationOwner == null && !ApplicationConfig.Setting.MainFrame.IsAutoAnimationBindMode)
            {
                // アニメーションセットのメニューでなくかつ自動バインドが無効のときは
                // 既定アニメーションセットにバインドする
                animationOwner = DocumentManager.DefaultSceneAnimationSet;
            }

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = true;
                return;
            }

            string[] fileNames;
            if (DialogUtility.ExecuteOpenFileDialog(out fileNames, ObjectIDUtility.FileFilter(GuiObjectID.SceneAnimation), true))
            {
                DocumentManager.LoadFromFile(fileNames, animationOwner);
            }
        }

        private void cmiFileOpenModel_MenuCommandHandler(MenuCommandArgs args)
        {
            TheApp.MainFrame.FileOpen(args, GuiObjectID.Model);
        }

        private void cmiFileCloseAllModel_MenuCommandHandler(MenuCommandArgs args)
        {
            TheApp.MainFrame.FilesClose(args, DocumentManager.Models);
        }

        private void cmiFileCloseSelectionModel_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.Models.Any();
                return;
            }

            using (var dialog = new DocumentsCloseSelectionDialog(DocumentManager.Models, ApplicationConfig.Setting.IO.CloseModelReference))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    ApplicationConfig.Setting.IO.CloseModelReference = dialog.CloseReference;
                    DocumentManager.RemoveDocuments(dialog.CloseDocuments, null, dialog.CloseReference, false);
                }
            }
        }

        private void cmiFileCloseModel_MenuCommandHandler(MenuCommandArgs args)
        {
            var model = App.AppContext.CurrentSelectedObject(args) as Model;
            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = model != null;
                return;
            }

            List<Document> documents = null;

            // オブジェクトビュー側で削除された場合、ファイルビュー側の選択は無視する
            if (App.AppContext.FileTreeView.Focused)
            {
                documents = App.AppContext.SelectedFileViewObjects?.OfType<Document>().ToList();
            }
            else
            {
                documents = new List<Document>();
                documents.Add(model);
            }

            if (documents != null)
            {
                if (documents.Any())
                {
                    DocumentManager.RemoveDocuments(documents, null, false, false);
                }
            }
        }

        private void Command_FileClose(MenuCommandArgs args)
        {
            TheApp.MainFrame.Command_FileClose(args);
        }

        private void cmiFileSaveModelSet_MenuCommandHandler(MenuCommandArgs args)
        {
            Model model = App.AppContext.CurrentSelectedObject(args) as Model;
            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = model != null;
                return;
            }

            if (model != null)
            {
                (new DocumentSaver()).SaveWithChildren(model);
            }
        }

        private void cmiFileSaveAllModelSet_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.Models.Any();
                return;
            }

            (new DocumentSaver()).SaveWithChildren(DocumentManager.Models);
        }

        private void cmiCloseBindingAnimationSelected_MenuCommandHandler(MenuCommandArgs args)
        {
            var selected = App.AppContext.CurrentSelectedObject(args);
            AnimationSet animationOwner = selected is Model ? ((Model)selected).DefaultAnimationSet : (selected as AnimationSet);
            IEnumerable<Document> animations = animationOwner != null ?
                DocumentManager.GetAnimations(animationOwner.Animations)
                : Enumerable.Empty<Document>();

            using (var dialog = new DocumentsCloseSelectionDialog(animations, true))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    DocumentManager.RemoveDocuments(dialog.CloseDocuments, animationOwner, dialog.CloseReference, false);
                }
            }
        }

        private void mniCloseBindingAnimationAll_MenuCommandHandler(MenuCommandArgs args)
        {
            var selected = App.AppContext.CurrentSelectedObject(args);
            AnimationSet animationOwner = selected is Model ? ((Model)selected).DefaultAnimationSet : (selected as AnimationSet);
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = (animationOwner != null && animationOwner.Animations.Any());
                return;
            }

            if (animationOwner != null)
            {
                DocumentManager.RemoveDocuments(DocumentManager.GetAnimations(animationOwner.Animations).ToArray(), animationOwner, true, false);
            }
        }

        private void mniOpenAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            var selected = App.AppContext.CurrentSelectedObject(args);
            AnimationSet animationOwner = selected as AnimationSet;
            Model animationOwnerModel = selected as Model;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = animationOwner != null || animationOwnerModel != null;
                return;
            }

            string[] fileNames;
            if (DialogUtility.ExecuteOpenFileDialog(out fileNames, ObjectIDUtility.FileFilterModelAnimation(), true))
            {
                DocumentManager.LoadFromFile(fileNames, animationOwner, animationOwnerModel: animationOwnerModel);
            }
        }

        private void cmiCreateCameraAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                var owner = (SceneAnimation)App.AppContext.CurrentSelectedObject(args);
                using (var dialog = new CameraAnimationCreateDialog(owner))
                {
                    if (dialog.ShowDialog(this) == DialogResult.OK)
                    {
                        var camera_anim = new camera_animType
                        {
                            projection_mode = dialog.ProjectionMode,
                            rotate_mode = dialog.RotateMode,
                            camera_name = dialog.AnimationName,
                            frame_count = dialog.FrameCount,
                            loop = dialog.IsLoop,
                            camera_anim_target = Enum.GetValues(typeof(camera_anim_target_targetType)).OfType<camera_anim_target_targetType>()
                            .Select(x => new camera_anim_targetType()
                            {
                                target = x,
                                base_value = CameraAnimation.DefaultBases[x]
                            }
                            ).ToArray(),
                        };

                        var animation = new CameraAnimation(camera_anim, owner);
                        var animations = owner.CameraAnims.Concat(Enumerable.Repeat(animation, 1)).ToList();
                        TheApp.CommandManager.Execute(DocumentManager.CreateCameraAnimationsEditCommand(owner, animations));
                    }
                }
            }
        }

        private void cmiCreateLightAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                var owner = (SceneAnimation)App.AppContext.CurrentSelectedObject(args);
                using (var dialog = new LightAnimationCreateDialog(owner))
                {
                    if (dialog.ShowDialog(this) == DialogResult.OK)
                    {
                        var targetTypes = LightAnimation.GetTargetTypes(dialog.Type);
                        var defaultBase = LightAnimation.GetDefaultBases(dialog.Type);

                        var lightAnim = new light_animType
                        {
                            angle_attn_func =
                                ApplicationConfig.Preset.LightAngAttnPresets.Select(x => x.Function).FirstOrDefault() ?? string.Empty,
                            dist_attn_func =
                                ApplicationConfig.Preset.LightDistAttnPresets.Select(x => x.Function).FirstOrDefault() ?? string.Empty,
                            frame_count = dialog.FrameCount,
                            light_anim_target = targetTypes.Select(x => new light_anim_targetType()
                            {
                                target = x,
                                base_value = defaultBase[x]
                            }).ToArray(),
                            light_name = dialog.AnimationName,
                            loop = dialog.IsLoop,
                            type = dialog.Type,
                        };

                        var animation = new LightAnimation(lightAnim, owner);
                        var animations = owner.LightAnims.Concat(Enumerable.Repeat(animation, 1)).ToList();
                        TheApp.CommandManager.Execute(DocumentManager.CreateLightAnimationsEditCommand(owner, animations));
                    }
                }
            }
        }

        private void cmiCreateFogAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                var owner = (SceneAnimation)App.AppContext.CurrentSelectedObject(args);
                using (var dialog = new FogAnimationCreateDialog(owner))
                {
                    if (dialog.ShowDialog(this) == DialogResult.OK)
                    {
                        var targets = Enum.GetValues(typeof(fog_anim_target_targetType)).OfType<fog_anim_target_targetType>();

                        var fogAnim = new fog_animType
                        {

                            dist_attn_func =
                                ApplicationConfig.Preset.FogDistAttnPresets.Select(x => x.Function).FirstOrDefault() ?? string.Empty,
                            frame_count = dialog.FrameCount,
                            fog_anim_target = targets.Select(x => new fog_anim_targetType()
                            {
                                target = x,
                                base_value = x == fog_anim_target_targetType.dist_attn_end ? 100 : 0
                            }).ToArray(),
                            fog_name = dialog.AnimationName,
                            loop = dialog.IsLoop,
                        };

                        var animation = new FogAnimation(fogAnim, owner);
                        var animations = owner.FogAnims.Concat(Enumerable.Repeat(animation, 1)).ToList();
                        TheApp.CommandManager.Execute(DocumentManager.CreateFogAnimationsEditCommand(owner, animations));
                    }
                }
            }
        }

        private void cmiCreateAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            TheApp.MainFrame.cmiCreateAnimation_MenuCommandHandler(args);
        }

        private void cmiUnbindAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            var selectedNodes = App.AppContext.FileTreeView?.SelectedNodes ?? new List<TreeNode>();
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = selectedNodes.Any() && selectedNodes.All(x => x.Tag is AnimationDocument && x.Parent?.Tag != null && (x.Parent.Tag is AnimationSet || (FileTreeView.FileViewFolder)x.Parent.Tag == FileTreeView.FileViewFolder.Scene));
                return;
            }

            var animationSetGroups = selectedNodes.GroupBy(x => x.Parent.Tag as AnimationSet);
            var commandSet = new EditCommandSet();

            foreach (var group in animationSetGroups)
            {
                var animationSet = group.Key;
                if (animationSet != null)
                {
                    var unbindAnimations = group.Select(x => x.Tag as AnimationDocument);
                    var newAnimations = animationSet.Animations.Where(x => unbindAnimations.All(y => x != new AnimationSetItem(y.FileName, y.FileLocation))).ToArray();
                    commandSet.Add(DocumentManager.CreateAnimationsEditCommand(animationSet, newAnimations, false));
                }
            }

            commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(selectedNodes.Select(x => x.Tag).OfType<AnimationDocument>()));
            TheApp.CommandManager.Execute(commandSet);
        }

        private void cmiFileOpenAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                string[] fileNames;
                if (DialogUtility.ExecuteOpenFileDialog(out fileNames, ObjectIDUtility.FileFilterAnimation(), true))
                {
                    DocumentManager.LoadFromFile(fileNames);
                }
            }
        }

        private void cmiFileCloseAllAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.Animations.Any();
                return;
            }

            DocumentManager.RemoveDocuments(DocumentManager.Animations, null, false, false);
        }

        private void cmiCreateAnimationSet_MenuCommandHandler(MenuCommandArgs args)
        {
            var model = App.AppContext.CurrentSelectedObject(args) as Model;
            if (args.RequireUpdate)
            {
                //args.CommandUI.Enabled = model != null;
                return;
            }

            if (model != null)
            {
                var ngNames = model.AnimationSets.Select(x => x.Name).ToArray();
                var newAnimSetName = StringUtility.UniqueName(
                    Strings.MainFrameContextMenuItems_CreateNewAnimationSetName,
                    ngNames,
                    Enumerable.Range(1, int.MaxValue).Select(x => x.ToString()));

                using (var dialog = new AnimationSetCreateDialog(newAnimSetName, ngNames))
                {
                    if (dialog.ShowDialog() != DialogResult.OK)
                    {
                        return;
                    }
                    var animationSet = new AnimationSet(false);
                    animationSet.Name = dialog.AnimationSetName;
                    DocumentManager.AddAnimationSet(model, animationSet);
                }
            }
            else
            {
                var ngNames = DocumentManager.SceneAnimationSets.Select(x => x.Name).ToArray();
                var newAnimSetName = StringUtility.UniqueName(
                    Strings.MainFrameContextMenuItems_CreateNewAnimationSetName,
                    ngNames,
                    Enumerable.Range(1, int.MaxValue).Select(x => x.ToString()));

                using (var dialog = new AnimationSetCreateDialog(newAnimSetName, ngNames))
                {
                    if (dialog.ShowDialog() != DialogResult.OK)
                    {
                        return;
                    }
                    var animationSet = new AnimationSet(true);
                    animationSet.Name = dialog.AnimationSetName;
                    DocumentManager.AddSceneAnimationSet(animationSet);
                }
            }
        }

        private void cmiFileCloseUnboundAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            var boundAnimations = new HashSet<AnimationSetItem>(DocumentManager.Models.SelectMany(y => y.DefaultAnimationSet.Animations.Concat(y.AnimationSets.SelectMany(z => z.Animations))).Concat(
                DocumentManager.AllSceneAnimationSets.SelectMany(x => x.Animations)));
            var unboundAnimations = DocumentManager.Animations.Where(x => !boundAnimations.Contains(new AnimationSetItem(x.FileName, x.FileLocation)));

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = unboundAnimations.Any();
                return;
            }

            DocumentManager.RemoveDocuments(unboundAnimations, null, false, false);
        }

        private void cmiUnbindAnimationsFromAnimationSet_MenuCommandHandler(MenuCommandArgs args)
        {
            var animationSets = new List<AnimationSet>();
            if (args.CommandUI.IsObjectViewContextMenu)
            {
                var animationSet = App.AppContext.CurrentSelectedObject(args) as AnimationSet;
                if (animationSet != null)
                {
                    animationSets.Add(animationSet);
                }
            }
            else
            {
                var selectedSets = App.AppContext.SelectedFileViewObjects?.OfType<AnimationSet>();
                if (selectedSets != null)
                {
                    animationSets.AddRange(selectedSets);
                }
            }

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = animationSets.Any(x => x.Animations.Any());
                return;
            }

            var commandSet = new EditCommandSet();
            foreach (var animationSet in animationSets.Where(x => x.Animations.Any()))
            {
                commandSet.Add(DocumentManager.CreateAnimationsEditCommand(animationSet, new AnimationSetItem[] { }, false).Execute());
            }
            commandSet.Add(
                DocumentManager.CreateAnimationUpdateBindCommand(
                    DocumentManager.Animations.Where(
                        x =>
                        animationSets.SelectMany(y => y.Animations).Distinct()
                            .Contains(new AnimationSetItem(x.FileName, x.FileLocation)))).Execute());
            TheApp.CommandManager.Add(commandSet);
        }

        private void cmiUnbindAllAnimationFromModel_MenuCommandHandler(MenuCommandArgs args)
        {
            Model model = App.AppContext.CurrentSelectedObject(args) as Model;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = model != null && (model.DefaultAnimationSet.Animations.Any() || model.AnimationSets.Any());
                return;
            }

            var oldAnimations = model.AllAnimations.ToArray();
            EditCommandSet commandSet = new EditCommandSet();
            commandSet.Add(DocumentManager.CreateAnimationsEditCommand(model.DefaultAnimationSet, new AnimationSetItem[] { }, false));
            commandSet.Add(DocumentManager.CreateAnimationSetEditCommand(model, new AnimationSet[] { }));
            commandSet.Add(new LazyCommand(() => DocumentManager.CreateAnimationUpdateBindCommand(oldAnimations)));
            TheApp.CommandManager.Execute(commandSet);
        }

        private void cmiDeleteAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                GuiObject selected = App.AppContext.CurrentSelectedObject(args);
                switch (selected.ObjectID)
                {
                    case GuiObjectID.CameraAnimation:
                        {
                            var camera = (CameraAnimation)selected;
                            var animations = camera.Owner.CameraAnims.Except(Enumerable.Repeat(camera, 1)).ToList();
                            TheApp.CommandManager.Execute(DocumentManager.CreateCameraAnimationsEditCommand(camera.Owner, animations));
                        }
                        break;
                    case GuiObjectID.LightAnimation:
                        {
                            var light = (LightAnimation)selected;
                            var animations = light.Owner.LightAnims.Except(Enumerable.Repeat(light, 1)).ToList();
                            TheApp.CommandManager.Execute(DocumentManager.CreateLightAnimationsEditCommand(light.Owner, animations));
                        }
                        break;
                    case GuiObjectID.FogAnimation:
                        {
                            var fog = (FogAnimation)selected;
                            var animations = fog.Owner.FogAnims.Except(Enumerable.Repeat(fog, 1)).ToList();
                            TheApp.CommandManager.Execute(DocumentManager.CreateFogAnimationsEditCommand(fog.Owner, animations));
                        }
                        break;
                }
            }
        }

        private void cmiDeleteAnimationSet_MenuCommandHandler(MenuCommandArgs args)
        {
            var selectedNodes = App.AppContext.FileTreeView?.SelectedNodes ?? new List<TreeNode>();

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled =
                    selectedNodes.Select(x => x.Tag)
                        .All(x => x is AnimationSet && ((AnimationSet)x).IsDefaultAnimationSet == false);
                return;
            }

            var animSetGroups = selectedNodes.ToLookup(x => x.Parent?.Tag as Model, x => x.Tag as AnimationSet);
            var commandSet = new EditCommandSet();
            foreach (var group in animSetGroups)
            {
                var model = group.Key;
                if (model != null)
                {
                    commandSet.Add(
                        DocumentManager.CreateAnimationSetEditCommand(
                            model,
                            model.AnimationSets.Where(x => !group.Contains(x)).ToArray()));
                }
                else
                {
                    commandSet.Add(
                        DocumentManager.CreateSceneAnimationSetEditCommand(
                            DocumentManager.SceneAnimationSets.Where(x => !group.Contains(x)).ToList()));
                }
            }
            TheApp.CommandManager.Execute(commandSet);
        }

        private void cmiDeleteAnimationSetWithChildren_MenuCommandHandler(MenuCommandArgs args)
        {
            var selectedNodes = App.AppContext.FileTreeView?.SelectedNodes ?? new List<TreeNode>();

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled =
                    selectedNodes.Select(x => x.Tag)
                        .All(x => x is AnimationSet && ((AnimationSet)x).IsDefaultAnimationSet == false);
                return;
            }

            var animSetGroups = selectedNodes.ToLookup(x => x.Parent?.Tag as Model, x => x.Tag as AnimationSet);
            var animationSetItems = animSetGroups.SelectMany(x => x.SelectMany(y => y.Animations));
            var animations = DocumentManager.GetAnimations(animationSetItems).Distinct().ToArray();

            // モデル関係のアニメーション
            //if (model != null)
            {
                var commandSet = new EditCommandSet();
                {
                    // anims がほかで参照されていないかチェックする
                    // 参照されていれば実行するかを問い合わせる
                    var refAnimes = new List<AnimationSetItem>();
                    {
                        var animSets =
                            DocumentManager.Models.SelectMany(x => x.AnimationSetsWithDefault)
                                .Concat(DocumentManager.SceneAnimationSetsWithDefault)
                                .Where(x => !animSetGroups.Any(y => y.Contains(x)))
                                .ToArray();

                        foreach (var anim in animations)
                        {
                            var animFileName = anim.FileName;
                            var animFileLocation = anim.FileLocation;

                            foreach (var animSet in animSets)
                            {
                                refAnimes.AddRange(animSet.Animations.Where(x => x == new AnimationSetItem(animFileName, animFileLocation)));
                            }
                        }
                    }

                    refAnimes = refAnimes.Distinct().ToList();

                    if (refAnimes.Any()
                        && (UIMessageBox.YesNo(
                            Strings.Fileview_AnimationReference + "\n\n" + string.Join("\n", refAnimes.Select(x => x.Name))) == false))
                    {
                        // 他で使われている
                        return;
                    }

                    var targetDocs = new List<Document>();
                    {
                        // クローズ対象になるものを集める
                        foreach (var anim in animations)
                        {
                            targetDocs.Add(anim);
                        }

                        for (var i = 0; i < targetDocs.Count; i++)
                        {
                            targetDocs.AddRange(
                                targetDocs[i].ReferenceDependDocuments.Where(
                                    x =>
                                    !targetDocs.Contains(x)
                                    && DocumentManager.Documents.Except(targetDocs)
                                           .All(y => !y.ReferenceDocuments.Contains(x))
                                    && x.ObjectID != GuiObjectID.ShaderDefinition));
                        }
                    }

                    targetDocs = targetDocs.Distinct().ToList();

                    // アニメーションと、アニメーションが参照するファイルが未保存なら保存するかを聞く
                    if (DocumentManager.CheckAndSaveModified(targetDocs) == false)
                    {
                        // キャンセルボタン押下でなにもしない
                        return;
                    }

                    commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(targetDocs, false));
                    foreach (var animSetGroup in animSetGroups)
                    {
                        var model = animSetGroup.Key;
                        if (model != null)
                        {
                            commandSet.Add(DocumentManager.CreateAnimationSetEditCommand(model, model.AnimationSets.Where(x => !animSetGroup.Contains(x)).ToArray()));
                        }
                        else
                        {
                            commandSet.Add(DocumentManager.CreateSceneAnimationSetEditCommand(DocumentManager.SceneAnimationSets.Where(x => !animSetGroup.Contains(x)).ToList()));
                        }
                    }
                    commandSet.Add(new LazyCommand(DocumentManager.CreateRemoveTextureReferencePaths));
                    var paths = targetDocs.Select(x => Tuple.Create(x, x.FilePath)).Where(x => !string.IsNullOrEmpty(x.Item2)).ToArray();
                    if (paths.Any())
                    {
                        commandSet.canUndo += () => PrePostIO.ExecutePreOpenUndoRedo(paths, true);
                    }
                }
                TheApp.CommandManager.Execute(commandSet);
            }
        }

        public static bool AnimationSetPreviewState(Model model, AnimationSet animationSet, bool preview = true)
        {
            var enable = false;

            if (model != null)
            {
                enable = animationSet != null && model.PreviewAnimSet != animationSet;
            }
            else
            {
                // シーンアニメーションの場合
                enable = animationSet != null && DocumentManager.PreviewSceneAnimSet != animationSet;
            }
            return preview ? enable : !enable;
        }


        private void AnimationSetPreview(MenuCommandArgs args, bool preview)
        {
            var animationSet = App.AppContext.CurrentSelectedObject(args) as AnimationSet;
            var model = App.AppContext.SelectedFileViewObjectOwner as Model;

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = AnimationSetPreviewState(model, animationSet, preview);
                return;
                /*
                            var enable = false;

                            if (model != null)
                            {
                                enable = animationSet != null && model.PreviewAnimSet != animationSet;
                            }
                            else
                            {
                                // シーンアニメーションの場合
                                enable = animationSet != null && DocumentManager.PreviewSceneAnimSet != animationSet;
                            }
                            args.CommandUI.Enabled = preview? enable : !enable;
                            return;
            */
            }

            if (model != null)
            {
                // プレビュー対象のアニメーションセットを設定する。
                model.PreviewAnimSet = preview ? animationSet : null;

                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
            }
            else
            {
                // プレビュー対象のアニメーションセットを設定する。
                DocumentManager.PreviewSceneAnimSet = preview ? animationSet : null;

                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
            }
        }

        private void cmiPreviewAnimationSetOn_MenuCommandHandler(MenuCommandArgs args)
        {
            AnimationSetPreview(args, true);
        }

        private void cmiPreviewAnimationSetOff_MenuCommandHandler(MenuCommandArgs args)
        {
            AnimationSetPreview(args, false);
        }

        private void PauseAnimationSet(MenuCommandArgs args, bool pause)
        {
            var selectedNodes = App.AppContext.FileTreeView?.SelectedNodes ?? new List<TreeNode>();
            var animSetGroups = selectedNodes.ToLookup(x => x.Parent?.Tag as Model, x => x.Tag as AnimationSet);

            var animationGroups =
                animSetGroups.SelectMany(
                    animInfo =>
                    animInfo.SelectMany(
                        animSet =>
                        DocumentManager.GetAnimations(animSet.Animations)
                            .Where(
                                anim =>
                                pause != anim.PauseFrames.ContainsKey(new KeyValuePair<object, string>(animInfo.Key?.ModelId, animSet.Name)))
                            .Select(x => new Tuple<AnimationSet, AnimationDocument>(animSet, x)))
                        .Select(anim => new Tuple<Model, Tuple<AnimationSet, AnimationDocument>>(animInfo.Key, anim)))
                    .ToLookup(x => x.Item1, x => x.Item2);

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = animationGroups.SelectMany(x => x).Any() && animSetGroups.All(x => x.Key != null);
                //&& animSetGroups.All(x => x.All(y => y == x.Key.PreviewAnimSet));
            }
            else
            {
                foreach (var animInfo in animationGroups)
                {
                    foreach (var tuple in animInfo)
                    {
                        var animationSet = tuple.Item1;
                        var animation = tuple.Item2;
                        var model = animInfo.Key;
                        var invisible =
                            animation.Pause.InvisibleBinds.Contains(
                                new KeyValuePair<object, string>(model.ModelId, animationSet.Name));
                        animation.SetPause(pause, 0, model.ModelId, animationSet.Name, invisible);
                    }
                }

                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
            }
        }

        private void cmiClearPauseAnimationSet_MenuCommandHandler(MenuCommandArgs args)
        {
            PauseAnimationSet(args, false);
        }

        private void cmiPauseAnimationSet_MenuCommandHandler(MenuCommandArgs args)
        {
            PauseAnimationSet(args, true);
        }

        private void cmiPreviewAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            AnimationDocument animation = App.AppContext.CurrentSelectedObject(args) as AnimationDocument;
            Model model = App.AppContext.SelectedFileViewObjectOwner as Model;

            if (args.RequireUpdate)
            {
                if (model != null)
                {
                    args.CommandUI.Enabled = animation != null && model != null && model.PreviewAnimSet != model.DefaultAnimationSet;
                }
                else
                {
                    // シーンアニメーションの場合
                    args.CommandUI.Enabled = animation != null && DocumentManager.PreviewSceneAnimSet != DocumentManager.DefaultSceneAnimationSet;
                }

                return;
            }

            if (model != null)
            {
                // 野良アニメーションはすべてまとめてデフォルトアニメーションセットになるので、
                // nullを設定します。
                model.PreviewAnimSet = model.DefaultAnimationSet;

                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
            }
            else
            {
                // シーンアニメーションの場合
                // 野良アニメーションはすべてまとめてデフォルトアニメーションセットになるので、
                // nullを設定します。
                DocumentManager.PreviewSceneAnimSet = DocumentManager.DefaultSceneAnimationSet;

                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
            }

            //// ビューアへプレビューの設定を送る。
            //Viewer.ViewerUtility.SendImplicitAnimationPreview(model, animation);
        }

        private void cmiNotPreviewAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            Model model = App.AppContext.CurrentSelectedObject(args) as Model;

            if (args.RequireUpdate)
            {
                if (model != null)
                {
                    args.CommandUI.Enabled = model.PreviewAnimSet != null;
                }
                else
                {
                    // シーンアニメーションの場合
                    args.CommandUI.Enabled = DocumentManager.PreviewSceneAnimSet != null;
                }

                return;
            }

            if (model != null)
            {
                // プレビュー対象のアニメーションセットをクリアする。
                model.PreviewAnimSet = null;

                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
            }
            else
            {
                // プレビュー対象のアニメーションセットをクリアする。
                DocumentManager.PreviewSceneAnimSet = null;

                App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
            }

        }


        private void cmiFileOpenTexture_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                string[] fileNames;
                if (DialogUtility.ExecuteOpenFileDialog(out fileNames, GuiObjectID.Texture, true))
                {
                    DocumentManager.LoadFromFile(fileNames);
                }
            }
        }

        private void cmiFileCloseAllTexture_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.Textures.Any();
                return;
            }

            DocumentManager.RemoveDocuments(DocumentManager.Textures, null, false, false);
        }

        private void cmiFileTextureCloseUnreferenced_MenuCommandHandler(MenuCommandArgs args)
        {
            var textures = DocumentManager.Textures.Where(x => !DocumentManager.Documents.Any(y => y.ReferenceDocuments.Contains(x)));
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = textures.Any();
                return;
            }

            DocumentManager.RemoveDocuments(textures, null, false, false);
        }

        private void cmiFileOpenShaderDefinition_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                string[] fileNames;
                if (DialogUtility.ExecuteOpenFileDialog(out fileNames, GuiObjectID.ShaderDefinition, true))
                {
                    DocumentManager.LoadFromFile(fileNames);
                }
            }
        }

        private void cmiFileCloseShaderDefinitionAll_MenuCommandHandler(MenuCommandArgs args)
        {
            var shaderDefinitions = DocumentManager.ShaderDefinitions;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = shaderDefinitions.Any();
                return;
            }

            // ShaderDefinitionの場合、保存はしないが、glslファイルが変更されている場合、
            // 閉じるかどうか？をたずねる。
            var modifiedShaderDefinition = shaderDefinitions.Where(x => x.IsModifiedObject);
            if (modifiedShaderDefinition.Any())
            {
                string fileNames = string.Empty;
                foreach (var shaderDefinition in modifiedShaderDefinition)
                {
                    fileNames += "'" + shaderDefinition.FileName + "' ";
                }
                string msg = string.Format("{0}\n{1}", fileNames, Strings.IO_CloseModifiedShaderDifinitionFiles);
                using (UIMessageBox msgBox = new UIMessageBox(msg, UIMessageBoxButtons.OKCancel, MessageBoxIcon.Information, MessageBoxDefaultButton.Button1, false))
                {
                    if (msgBox.ShowDialog() == DialogResult.OK)
                    {
                        DocumentManager.RemoveDocuments(shaderDefinitions, null, false, false);
                    }
                    else
                    {
                        var notModifiedShaderDefinitions = shaderDefinitions.Where(x => x.IsModifiedObject == false);
                        DocumentManager.RemoveDocuments(notModifiedShaderDefinitions, null, false, false);
                    }
                }
            }
            else
            {
                DocumentManager.RemoveDocuments(shaderDefinitions, null, false, false);
            }
        }

        private void cmiFileOpenExplorer_MenuCommandHandler(MenuCommandArgs args)
        {
            Document document = App.AppContext.CurrentSelectedObject(args) as Document;

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = (document != null) && document.FileExsists;
            }
            else
            {
                Process.Start("EXPLORER.EXE", string.Format("/select,{0}", document.FilePath));
            }
        }

        private void mniViewProperty_MenuCommandHandler(MenuCommandArgs args)
        {
            TheApp.MainFrame.mniViewProperty_MenuCommandHandler(args);
        }

        private void cmiSelectAll_MenuCommandHandler(MenuCommandArgs args)
        {
            var viewPanel = args.CommandData as IObjectView;
            Debug.Assert(viewPanel != null);

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.EnableSelectAll;
            }
            else
            {
                viewPanel.SelectAll();
            }
        }

        private void cmiToggleSelection_MenuCommandHandler(MenuCommandArgs args)
        {
            var viewPanel = args.CommandData as IObjectView;
            Debug.Assert(viewPanel != null);

            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.EnableToggleSelection;
            }
            else
            {
                viewPanel.ToggleSelection();
            }
        }

        private void cmiSchematicMagnifyPlus_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableMagnifyPlus;
            }
            // 実行時
            else
            {
                viewPanel.OpBar.MagnifyPlus();
            }

        }

        private void cmiSchematicMagnifyMinus_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableMagnifyMinus;
            }
            // 実行時
            else
            {
                viewPanel.OpBar.MagnifyMinus();
            }
        }

        private void cmiSchematicCenteringWhole_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableCenteringWhole;
            }
            // 実行時
            else
            {
                viewPanel.View.CenteringWhole(false);
            }
        }

        private void cmiSchematicCenteringSelected_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableCenteringSelected;
            }
            // 実行時
            else
            {
                viewPanel.View.CenteringSelected();
            }
        }

        private void cmiSchematicAlign_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableAlign;
                args.CommandUI.Checked = viewPanel.View.CurrentAlignMode == ObjectSchematicView.AlignMode.Schematic;
            }
            // 実行時
            else
            {
                viewPanel.View.Align(ObjectSchematicView.AlignMode.Schematic);
                viewPanel.OpBar.UpdateAlignButton();
            }
        }

        private void cmiSchematicAlignVertical_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableAlign;
                args.CommandUI.Checked = viewPanel.View.CurrentAlignMode == ObjectSchematicView.AlignMode.Vertical;
            }
            // 実行時
            else
            {
                viewPanel.View.Align(ObjectSchematicView.AlignMode.Vertical);
                viewPanel.OpBar.UpdateAlignButton();
            }
        }

        private void cmiSchematicAlignHorizontal_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableAlign;
                args.CommandUI.Checked = viewPanel.View.CurrentAlignMode == ObjectSchematicView.AlignMode.Horizontal;
            }
            // 実行時
            else
            {
                viewPanel.View.Align(ObjectSchematicView.AlignMode.Horizontal);
                viewPanel.OpBar.UpdateAlignButton();
            }
        }

        private void cmiSchematicAlignSelected_MenuCommandHandler(MenuCommandArgs args)
        {
            ObjectSchematicViewPanel viewPanel = TheApp.MainFrame.ObjectViewClient.GetViewControl(ObjectViewMode.SchematicView)
                as ObjectSchematicViewPanel;
            Debug.Assert(viewPanel != null);

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = viewPanel.View.IsEnableAlignSelected;
            }
            // 実行時
            else
            {
                viewPanel.View.AlignSelected();
            }
        }

        private void cmiFileReload_MenuCommandHandler(MenuCommandArgs args)
        {
            var active = App.AppContext.CurrentSelectedObject(args) as Document;

            // オブジェクトビュー側で削除された場合、ファイルビュー側の選択は無視する
            List<Document> documents = null;
            if (App.AppContext.FileTreeView.Focused)
            {
                documents = App.AppContext.SelectedFileViewObjects?.OfType<Document>().ToList();
            }
            else
            {
                documents = new List<Document>();
                documents.Add(active);
            }

            if (documents == null)
            {
                return;
            }

            if (documents.Any() == false)
            {
                return;
            }

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = active != null && documents != null && documents.All(x => x.FileExsists);
            }
            // 実行時
            else
            {
                var commandSet = new EditCommandSet();
                foreach (var document in documents)
                {
                    var command = DocumentManager.Reload(document.FilePath, document, false, fromMenu: true, returnCommanSet: true);
                    if (command != null)
                    {
                        commandSet.Add(command);
                    }
                }
                if (commandSet.CommandCount > 0)
                {
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                }
            }
        }

        private void ChangeShowInObjView(MenuCommandArgs args, bool flag)
        {
            var models = new List<Model>();
            if (args.CommandUI.IsObjectViewContextMenu)
            {
                models.AddRange(App.AppContext.SelectedTarget.Objects.OfType<Model>());
            }
            else
            {
                models.AddRange(App.AppContext.SelectedFileViewObjects?.OfType<Model>());
            }

            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = models.Any();
                return;
            }
            // 実行時

            if (!models.Any() || models.All(x => x.IsShowInObjView == flag)) return;

            //            models.ForEach(x => x.PreviewInfo.ShowInObjView = flag);
            // プロジェクトの変更マーク更新用
            DocumentManager.ProjectDocument.SetMaybeModified();
            foreach (var model in models)
            {
                model.PreviewInfo.ShowInObjView = flag;
                var evtArgs = new DocumentPropertyChangedEventArgs(new DocumentPropertyChangedShowInObjViewArgs(model));
                App.AppContext.ExecutePropertyChangedEvent(this, evtArgs.GetArgs());
            }
        }
        private void cmiShowModelInObjView_MenuCommandHandler(MenuCommandArgs args)
        {
            ChangeShowInObjView(args, true);
        }

        private void cmiNotShowModelInObjView_MenuCommandHandler(MenuCommandArgs args)
        {
            ChangeShowInObjView(args, false);
        }

        private void cmiFileCloseSelectionAnimation_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.Animations.Any();
                return;
            }

            using (var dialog = new DocumentsCloseSelectionDialog(DocumentManager.Animations, false, false))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    DocumentManager.RemoveDocuments(dialog.CloseDocuments, null, false, false);
                }
            }
        }

        private void cmiFileCloseSelectionTexture_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.Textures.Any();
                return;
            }

            using (var dialog = new DocumentsCloseSelectionDialog(DocumentManager.Textures, false, false))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    DocumentManager.RemoveDocuments(dialog.CloseDocuments, null, false, false);
                }
            }
        }

        private void cmiFileCloseSelectionShader_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.ShaderDefinitions.Any();
                return;
            }

            using (var dialog = new DocumentsCloseSelectionDialog(DocumentManager.ShaderDefinitions, false, false))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    DocumentManager.RemoveDocuments(dialog.CloseDocuments, null, false, false);
                }
            }
        }

        private void cmiRetarget_MenuCommandHandler(MenuCommandArgs args)
        {
            var animationSet = App.AppContext.CurrentSelectedObject(args) as AnimationSet;
            var model = App.AppContext.SelectedFileViewObjectOwner as Model;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = model != null && animationSet != null && DocumentManager.GetAnimations(animationSet.Animations).OfType<SkeletalAnimation>().Any();
                return;
            }

            TheApp.CommandManager.Execute(SkeletalAnimationPreviewPage.CreateSetRetargetingHostModelCommand(DocumentManager.GetAnimations(animationSet.Animations).OfType<SkeletalAnimation>(), model.Name));

        }

        private void cmiCopyNames_MenuCommandHandler(MenuCommandArgs args)
        {
            var view = args.CommandData as IObjectView;
            Debug.Assert(view != null);

            var guiObjects = App.AppContext.SelectedTarget.Objects;
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = guiObjects.Any();
            }
            else
            {
                if (guiObjects.Any())
                {
                    SetObjectsNameClipboard(guiObjects);
                }
            }
        }

        internal static void SetObjectsNameClipboard(ReadOnlyList<GuiObject> guiObjects)
        {
            var groups = guiObjects.GroupBy(x => x.ObjectID).ToList();
            if (!groups.Any()) return;
            var names = groups.SelectMany(g => g.OrderBy(o => o.Name).Select(x => x.Name));
            App.Utility.ClipboardUtility.SetText(string.Join("\n", names));
        }

        private void cmiCreateSeparateMaterial_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.ShaderDefinitions.Any(x =>
                    (x.Data != null) &&
                    (x.Data.shading_model_array != null) &&
                    (x.Data.shading_model_array.shading_model != null) &&
                    x.Data.shading_model_array.shading_model.Any(y => y.IsMaterialShader()));
                return;
            }

            using (var dialog = new SeparateMaterialCreateDialog())
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    var materialName = dialog.MaterialName;
                    var shaderDefinition = dialog.ShaderDefinition;
                    var shadingModel = dialog.ShadingModel;

                    var matFile = new Nintendo.G3dTool.Entities.IntermediateFile(Nintendo.G3dTool.Entities.IntermediateFileKind.Material);
                    var mat = matFile.RootEntity as Nintendo.G3dTool.Entities.Material;
                    mat.Name = materialName;
                    mat.ShaderAssign.ShaderArchive = shaderDefinition.Name;
                    mat.ShaderAssign.ShadingModel = shadingModel.name;

                    nw4f_3difType file = matFile.CreateSerializableData();
                    var separateMaterial = new SeparateMaterial(file.RootElement as materialType, null)
                    {
                        Name = materialName,
                        FileDotExt = G3dPath.MaterialBinaryExtension
                    };

                    {
                        var materials = separateMaterial.Materials.ToArray();
                        var targets = new GuiObjectGroup(materials);

                        MaterialShaderPage.CreateEditCommand_ShaderPool(targets).Execute();

                        ShaderAssignUtility.CreateEditCommand_ShaderAssign(
                            targets,
                            materials.Select(x => MaterialShaderPage.GetDefaultShaderAssign(
                                x,
                                x.MaterialShaderAssign.ShaderDefinition.Name,
                                x.MaterialShaderAssign.ShadingModel.name)),
                            false).Execute();

                        foreach (var material in materials)
                        {
                            var copyData = MaterialShaderPage.CreateCopyData(
                                material.MaterialShaderAssignPool,
                                PropertyEdit.ShaderParamControls.ShaderParamControlGroup.CreatePageCopyData(
                                    material,
                                    pageMode: false,
                                    page: null,
                                    checkEditable: true,
                                    checkVisible: false));
                            var command = PropertyEdit.ShaderParamControls.ShaderParamControlGroup.CreateItemsPasteCommand(
                                new GuiObjectGroup(material),
                                copyData,
                                group_id: null,
                                pageMode: false,
                                page: null,
                                checkEditable: true,
                                ReloadModel: false,
                                sendViewer: false, checkVisible: false);
                            if (command != null)
                            {
                                command.Execute();
                            }
                        }
                        ShaderAssignUtility.CreateVtxBufferEditCommand(targets).Execute();

                        foreach (var material in materials)
                        {
                            material.MaterialShaderAssign.AddNotAssignedAttributes();
                            material.Data.shader_assign = material.MaterialShaderAssign.To_shader_assign(material, true);
                        }
                        if (materials.Where(x => !ShaderAssignUtility.IsConsistentWithDefinition(x)).Any())
                        {
                            ShaderAssignUtility.ExecuteFixParameters(
                            materials,
                            updateModified: false,
                            useHint: false,
                            reload: false).Execute();
                        }

                        DocumentManager.CreateAnimationUpdateBindCommand(materials.SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()).Execute();
                    }

                    // 作ったドキュメントをマネージャーに登録。
                    separateMaterial.UpdateSavedData();
                    EditCommandSet commandSet = new EditCommandSet();
                    using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                    {
                        using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                        {
                            commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(separateMaterial, 1), true).Execute());
                        }
                    }
                    commandSet.Reverse();
                    TheApp.CommandManager.Add(commandSet);
                }
            }
        }

        private void cmiFileOpenSeparateMaterial_MenuCommandHandler(MenuCommandArgs args)
        {
            TheApp.MainFrame.FileOpen(args, GuiObjectID.SeparateMaterial);
        }

        private void cmiFileCloseAllSeparateMaterial_MenuCommandHandler(MenuCommandArgs args)
        {
            TheApp.MainFrame.FilesClose(args, DocumentManager.SeparateMaterials);
        }

        private void cmiFileCloseSelectionSeparateMaterial_MenuCommandHandler(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = DocumentManager.SeparateMaterials.Any();
                return;
            }

            using (var dialog = new DocumentsCloseSelectionDialog(DocumentManager.SeparateMaterials, ApplicationConfig.Setting.IO.CloseReference, false))
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    ApplicationConfig.Setting.IO.CloseModelReference = dialog.CloseReference;
                    DocumentManager.RemoveDocuments(dialog.CloseDocuments, null, dialog.CloseReference, false);
                }
            }
        }

        private void cmiFileSeparateMaterialCloseUnreferenced_MenuCommandHandler(MenuCommandArgs args)
        {
            var separateMaterials = DocumentManager.SeparateMaterials.Where(x => x.Materials.Any(y => !y.Referrers.Any() && !y.ChildMaterials.Any()));
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = separateMaterials.Any();
                return;
            }

            DocumentManager.RemoveDocuments(separateMaterials, null, ApplicationConfig.Setting.IO.CloseReference, false);
        }
    }
}
