﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using System.Collections.Generic;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.FileView;
using App.Utility;
using App.res;
using ConfigCommon;
using System.Collections;

namespace App
{
    using nw.g3d.iflib;

    using TeamConfig;

    using Viewer;

    public partial class MainFrame
    {
        // メニューコマンドホスト
        private readonly MenuCommandHost menuCommandHost_ = new MenuCommandHost();

        /// <summary>
        /// メニューを取得する。
        /// </summary>
        private IEnumerable<ToolStrip> Menus
        {
            get
            {
                yield return stsMain;
                yield return tspFilter;
                yield return TreeViewMenu.Items;
            }
        }

        /// <summary>
        /// メニューコマンド初期化。
        /// </summary>
        private void InitializeMenuCommand()
        {
            RegisterCommands();

            menuCommandHost_.Setup(stsMain);
            menuCommandHost_.Setup(tspFilter);
            menuCommandHost_.Setup(TreeViewMenu.Items);
            menuCommandHost_.MessageHandler += delegate(string message) { StatusMessage = message; };

            Application.Idle += (sender, e) => menuCommandHost_.OnIdleUpdate();

#if DEBUG
            #region Debug
            {
                //nw.g3d.nw4f_3dif.G3dParallel.IsParallel = false;
                UIToolStripButton debugButton = new UIToolStripButton();
                debugButton.Text = "テスト";
                debugButton.Click += (o, e) => {
                    throw new Exception();
                    /*
                    ConnectToHio(false);
                    System.Threading.Thread.Sleep(1000);
                    ConnectToHio(true);
                    ConnectToHio(false);
                    System.Threading.Thread.Sleep(1000);
                    ConnectToHio(true);
                     */
                    /*
                    System.Threading.Thread.Sleep(100);
                    ConnectToHio(false);
                    System.Threading.Thread.Sleep(100);
                    ConnectToHio(true);*/
                    /*
                    // 送信スレッドで例外を起こす
                    Viewer.GeneralActionMessage.Send(() => {
                        throw new System.Exception();
                    });
                    DebugConsole.WriteLine("GC.GetTotalMemory():{0:n0}bytes", GC.GetTotalMemory(false));
                    DebugConsole.WriteLine("System.Environment.WorkingSet:{0:n0}bytes", System.Environment.WorkingSet);
        			var currentProcess = Process.GetCurrentProcess();
                    DebugConsole.WriteLine("Process.PrivateMemorySize64:{0:n0}bytes", currentProcess.PrivateMemorySize64);*/
                };

                tspFilter.Items.Add(debugButton);

                // コンソール出力制御
                var outputConsoleButton = new UIToolStripButton()
                {
                    Text = DebugConsole.Output ? "Console: Enabled" : "Console: Disabled",
                };
                outputConsoleButton.Click += (o, e) =>
                {
                    DebugConsole.Output = !DebugConsole.Output;
                    outputConsoleButton.Text = DebugConsole.Output ? "Console: Enabled" : "Console: Disabled";
                };
                tspFilter.Items.Add(outputConsoleButton);
            }
            #endregion

#endif
        }

        private void RegisterCommands()
        {
            menuCommandHost_.BindCommandItem(mniFileOpen, tsiFileOpen);
            menuCommandHost_.BindCommandItem(mniFileSave, tsiFileSave);
            menuCommandHost_.BindCommandItem(mniFileSaveAllTo, tsiFileSaveAllTo);
            menuCommandHost_.BindCommandItem(mniFileSaveAll, tsiFileSaveAll);
            menuCommandHost_.BindCommandItem(mniEditUndo, tsiEditUndo);
            menuCommandHost_.BindCommandItem(mniEditRedo, tsiEditRedo);
            menuCommandHost_.BindCommandItem(mniViewLayoutSingle, tsiViewLayoutSingle);
            menuCommandHost_.BindCommandItem(mniViewLayoutMulti2Stacked, tsiViewLayoutMulti2Stacked);
            menuCommandHost_.BindCommandItem(mniViewLayoutMulti2SideBySide, tsiViewLayoutMulti2SideBySide);
            menuCommandHost_.BindCommandItem(mniViewLayoutMulti3SplitBottom, tsiViewLayoutMulti3SplitBottom);
            menuCommandHost_.BindCommandItem(mniViewLayoutMulti3SplitTop, tsiViewLayoutMulti3SplitTop);
            menuCommandHost_.BindCommandItem(mniViewLayoutMulti3SplitRight, tsiViewLayoutMulti3SplitRight);
            menuCommandHost_.BindCommandItem(mniViewLayoutMulti3SplitLeft, tsiViewLayoutMulti3SplitLeft);
            menuCommandHost_.BindCommandItem(mniViewLayoutMulti4, tsiViewLayoutMulti4);
            menuCommandHost_.BindCommandItem(mniViewProperty, tsiViewProperty);
            menuCommandHost_.BindCommandItem(mniHelpHelp, tsiHelp);
            menuCommandHost_.BindCommandItem(mniHelpAbout, tsiAbout);
            menuCommandHost_.BindCommandItem(mniHioConnection_SwitchConnection, tsiHioConnection_SwitchConnection);
            menuCommandHost_.BindCommandItem(mniOptimizeShader, tsiOptimizeShader);
            menuCommandHost_.BindCommandItem(mniForceOptimizeShader, tsiForceOptimizeShader);
            menuCommandHost_.BindCommandItem(mniAutoAnimationBindMode, tsiAutoAnimationBindMode);

            mniCreateMaterialAnimation.Tag = TreeViewMenu.cmiCreateMaterialAnimation.Tag =
                new MaterialAnimationCreateTag() { id = GuiObjectID.MaterialAnimation, subType = MaterialAnimation.SubType.Material };
            mniCreateShaderParameterMaterialAnimation.Tag = TreeViewMenu.cmiCreateShaderParameterMaterialAnimation.Tag =
                new MaterialAnimationCreateTag() { id = GuiObjectID.MaterialAnimation, subType = MaterialAnimation.SubType.ShaderParameter };
            mniCreateColorMaterialAnimation.Tag = TreeViewMenu.cmiCreateColorMaterialAnimation.Tag =
                new MaterialAnimationCreateTag() { id = GuiObjectID.MaterialAnimation, subType = MaterialAnimation.SubType.Color };
            mniCreateTexSrtMaterialAnimation.Tag = TreeViewMenu.cmiCreateTexSrtMaterialAnimation.Tag =
                new MaterialAnimationCreateTag() { id = GuiObjectID.MaterialAnimation, subType = MaterialAnimation.SubType.TextureSRT };
            mniCreateTexPatternMaterialAnimation.Tag = TreeViewMenu.cmiCreateTexPatternMaterialAnimation.Tag =
                new MaterialAnimationCreateTag() { id = GuiObjectID.MaterialAnimation, subType = MaterialAnimation.SubType.TexturePattern };
            mniCreateVisibilityMaterialAnimation.Tag = TreeViewMenu.cmiCreateVisibilityMaterialAnimation.Tag =
                new MaterialAnimationCreateTag() { id = GuiObjectID.MaterialAnimation, subType = MaterialAnimation.SubType.MaterialVisibility };
            mniCreateShaderParameterAnimation.Tag = TreeViewMenu.cmiCreateShaderParameterAnimation.Tag =
                new AnimationCreateTag() { id = GuiObjectID.ShaderParameterAnimation };
            mniCreateColorAnimation.Tag = TreeViewMenu.cmiCreateColorAnimation.Tag =
                new AnimationCreateTag() { id = GuiObjectID.ColorAnimation };
            mniCreateTextureSrtAnimation.Tag = TreeViewMenu.cmiCreateTextureSrtAnimation.Tag =
                new AnimationCreateTag() { id = GuiObjectID.TextureSrtAnimation };
            mniCreateBoneVisibilityAnimation.Tag = TreeViewMenu.cmiCreateBoneVisibilityAnimation.Tag =
                new AnimationCreateTag() { id = GuiObjectID.BoneVisibilityAnimation };
            mniCreateMaterialVisibilityAnimation.Tag = TreeViewMenu.cmiCreateMaterialVisibilityAnimation.Tag =
                new AnimationCreateTag() { id = GuiObjectID.MaterialVisibilityAnimation };
            mniCreateTexturePatternAnimation.Tag = TreeViewMenu.cmiCreateTexturePatternAnimation.Tag =
                new AnimationCreateTag() { id = GuiObjectID.TexturePatternAnimation };
            mniCreateSceneAnimation.Tag = TreeViewMenu.cmiCreateSceneAnimation.Tag =
                new AnimationCreateTag() { id = GuiObjectID.SceneAnimation };

            var oldMenus = new[] {
                mniCreateShaderParameterAnimation, TreeViewMenu.cmiCreateShaderParameterAnimation,
                mniCreateColorAnimation, TreeViewMenu.cmiCreateColorAnimation,
                mniCreateTextureSrtAnimation, TreeViewMenu.cmiCreateTextureSrtAnimation,
                mniCreateTexturePatternAnimation, TreeViewMenu.cmiCreateTexturePatternAnimation,
                mniCreateMaterialVisibilityAnimation, TreeViewMenu.cmiCreateMaterialVisibilityAnimation,
            };
            var separateMenus = new[] {
                mniCreateShaderParameterMaterialAnimation, TreeViewMenu.cmiCreateShaderParameterMaterialAnimation,
                mniCreateColorMaterialAnimation, TreeViewMenu.cmiCreateColorMaterialAnimation,
                mniCreateTexSrtMaterialAnimation, TreeViewMenu.cmiCreateTexSrtMaterialAnimation,
                mniCreateTexPatternMaterialAnimation, TreeViewMenu.cmiCreateTexPatternMaterialAnimation,
                mniCreateVisibilityMaterialAnimation, TreeViewMenu.cmiCreateVisibilityMaterialAnimation,
            };
            var materialMenus = new[] {
                mniCreateMaterialAnimation, TreeViewMenu.cmiCreateMaterialAnimation,
            };


            if (ApplicationConfig.Preset.EnableMaterialAnimCreation)
            {
                foreach (var menu in oldMenus)
                {
                    menu.Visible = false;
                }

                if (ApplicationConfig.Preset.SeparateMaterialAnimCreationMenu)
                {
                    foreach (var menu in materialMenus)
                    {
                        menu.Visible = false;
                    }
                }
                else
                {
                    foreach (var menu in separateMenus)
                    {
                        menu.Visible = false;
                    }
                }
            }
            else
            {
                foreach (var menu in materialMenus)
                {
                    menu.Visible = false;
                }

                foreach (var menu in separateMenus)
                {
                    menu.Visible = false;
                }
            }
        }

        private class AnimationCreateTag
        {
            public GuiObjectID id;
        }

        private class MaterialAnimationCreateTag : AnimationCreateTag
        {
            public MaterialAnimation.SubType subType;
        }

        public UIContextMenuStrip CreateFileViewFolderMenu(FileTreeView.FileViewFolder folder)
        {
            UIContextMenuStrip menu = menuCommandHost_.CreateGeneralContextMenu();
            switch (folder)
            {
                case FileTreeView.FileViewFolder.Scene:
                    menu.Items.Add(TreeViewMenu.cmiFileOpenSceneAnimation);
                    menu.Items.Add(TreeViewMenu.cmiCreateSceneAnimation2);
                    menu.Items.Add(TreeViewMenu.cmiCreateAnimationSet);
                    menu.Items.Add(TreeViewMenu.cmiUnbindAllAnimationFromScene);
                    break;
                case FileTreeView.FileViewFolder.Model:
                    // 開く、閉じる
                    menu.Items.Add(TreeViewMenu.cmiFileOpenModel);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseAllModel);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseSelectionModel);

                    // 保存
                    menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                    menu.Items.Add(TreeViewMenu.cmiFileSaveAllModelSet);

                    break;
                case FileTreeView.FileViewFolder.Texture:
                    menu.Items.Add(TreeViewMenu.cmiFileOpenTexture);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseAllTexture);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseSelectionTexture);
                    menu.Items.Add(TreeViewMenu.cmiFileTextureCloseUnreferenced);
                    break;
                case FileTreeView.FileViewFolder.Animation:
                    // アニメーション新規作成
                    // UIToolStripMenuItem.CreateClone() では UIToolStripMenuItem.Available を考慮しないので CreateClone() 前にフィルタリング。
                    // ここでのフィルタに UIToolStripMenuItem.Visible を用いるのは間違い。
                    var create = TreeViewMenu.cmiCreateAnimation.CreateClone();
                    create.DropDownItems.AddRange(
                        TreeViewMenu.cmiCreateAnimation.DropDownItems
                        .OfType<UIToolStripMenuItem>()
                        .Where(x => x.Available)
                        .Select(x => x.CreateClone()).ToArray());
                    create.DropDownItems.Add(TreeViewMenu.cmiCreateSceneAnimation);

                    menu.Items.Add(create);
                    menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                    menu.Items.Add(TreeViewMenu.cmiFileOpenAnimation);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseAllAnimation);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseSelectionAnimation);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseUnboundAnimation);
                    break;
                case FileTreeView.FileViewFolder.ShaderDefinition:
                    menu.Items.Add(TreeViewMenu.cmiFileOpenShaderDefinition);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseShaderDefinitionAll);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseSelectionShader);
                    break;
                case FileTreeView.FileViewFolder.Material:
                    menu.Items.Add(TreeViewMenu.cmiCreateSeparateMaterial);
                    menu.Items.Add(TreeViewMenu.cmiFileOpenSeparateMaterial);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseAllSeparateMaterial);
                    menu.Items.Add(TreeViewMenu.cmiFileCloseSelectionSeparateMaterial);
                    menu.Items.Add(TreeViewMenu.cmiFileSeparateMaterialCloseUnreferenced);
                    break;
            }

            return menu;
        }

        public UIContextMenuStrip CreateMultiSelectMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();
            var selectedNode = App.AppContext.FileTreeView?.SelectedNode;
            if (selectedNode == null) { return menu; }

            var selectedNodes = App.AppContext.FileTreeView?.SelectedNodes;
            if (selectedNodes == null || !selectedNodes.Any()) { return menu; }

            var guiObjects = App.AppContext.SelectedFileViewObjects?.ToArray();
            if (guiObjects == null || !guiObjects.Any()) { return menu; }

            var documents = guiObjects.OfType<Document>().ToList();

            var active = guiObjects.First();
            var isModel = guiObjects.All(x => x.ObjectID == GuiObjectID.Model);
            if (isModel)
            {
                menu.Items.Add(
                    ((Model)active).IsShowInObjView
                    ? TreeViewMenu.cmiNotShowModelInObjView
                    : TreeViewMenu.cmiShowModelInObjView);
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                menu.Items.Add(TreeViewMenu.cmiFileCloseModelSet);
                menu.Items.Add(TreeViewMenu.cmiFileCloseModel);
            }
            else if (selectedNodes.All(x => x.Tag is AnimationSet))
            {
                menu.Items.Add(TreeViewMenu.cmiUnbindAnimationsFromAnimationSet);
                //if (selectedNodes.All(x => ((AnimationSet)x.Tag).IsDefaultAnimationSet == false))
                {
                    menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                    menu.Items.Add(TreeViewMenu.cmiDeleteAnimationSet);
                    menu.Items.Add(TreeViewMenu.cmiDeleteAnimationSetWithChildren);
                }
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                menu.Items.Add(TreeViewMenu.cmiPauseAnimationSet);
                menu.Items.Add(TreeViewMenu.cmiClearPauseAnimationSet);
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            }
            else if (selectedNodes.All(x => x.Tag is AnimationDocument && x.Parent.Tag is AnimationSet))
            {
                // アニメーションセット内のアニメーション
                menu.Items.Add(TreeViewMenu.cmiUnbindAnimation);
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                menu.Items.Add(mniFileClose.CreateClone());
            }
            else
            {
                menu.Items.Add(mniFileClose.CreateClone());
            }
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());

            // ユーザーコマンドメニューを追加
            AddUserCommandMenu(menu, documents);
            // ランタイムユーザースクリプトを追加
            if (isModel)
            {
                AddRuntimeUserScriptMenu(menu, documents);
            }
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            SetViewPropertyMenuItem(menu);
            return menu;
        }

        public UIContextMenuStrip CreateModelMenu(Model model, UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(TreeViewMenu.cmiOpenAnimation);
            menu.Items.Add(TreeViewMenu.cmiCreateAnimation);
            menu.Items.Add(TreeViewMenu.cmiCreateAnimationSet);
            menu.Items.Add(TreeViewMenu.cmiUnbindAllAnimationFromModel);
            if (model != null)
            {
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                menu.Items.Add(
                    model.IsShowInObjView
                    ? TreeViewMenu.cmiNotShowModelInObjView
                    : TreeViewMenu.cmiShowModelInObjView);
            }
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileCloseModelSet);
            menu.Items.Add(TreeViewMenu.cmiFileCloseModel);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(mniFileSaveAs.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileSaveModelSet);

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileMergeModel.CreateClone());
            menu.Items[menu.Items.Count - 1].Text = Strings.Menu_FileMergeModel;

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            AddUserCommandMenu(menu, document);
            AddRuntimeUserScriptMenu(menu, document);
            return menu;
        }

        public UIContextMenuStrip CreateObjectUnderModelMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            // ユーザーコマンドメニューを追加
            var sel = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu);
            if (sel != null)
            {
                var separator = menuCommandHost_.CreateGeneralSeparator();
                menu.Items.Add(separator);
                var add = AddUserCommandMenu(menu, sel.OwnerDocument, Strings.MainFrame_CreateObjectUnderModelMenu_UserCommandToModel);
                add = add || AddRuntimeUserScriptMenu(menu, sel.OwnerDocument);
                if (!add)
                {
                    // ユーザーコマンドメニューが追加されなかったときはセパレータを削除
                    menu.Items.Remove(separator);
                }
            }
            return menu;
        }

        public UIContextMenuStrip CreateSceneAnimationMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(mniFileSaveAs.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiCreateSceneAnimationItem);

            // TODO: マージ削除
            //menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());

            //menu.Items.Add(mniFileMergeAnimation.CreateClone());
            //menu.Items[menu.Items.Count - 1].Text = Strings.Menu_FileMergeAnimation;

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());

            menu.Items.Add(mniFileRename.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            AddUserCommandMenu(menu, document);

            return menu;
        }

        public UIContextMenuStrip CreateSceneAnimationMenuUnderScene(bool underAnimationSet, UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            if (!underAnimationSet)
            {
                menu.Items.Add(TreeViewMenu.cmiPreviewAnimation);
            }
            menu.Items.Add(TreeViewMenu.cmiUnbindSceneAnim);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(mniFileSaveAs.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiCreateSceneAnimationItem);

            // TODO: マージ削除
            //menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());

            //menu.Items.Add(mniFileMergeAnimation.CreateClone());
            //menu.Items[menu.Items.Count - 1].Text = Strings.Menu_FileMergeAnimation;

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());

            menu.Items.Add(mniFileRename.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            AddUserCommandMenu(menu, document);

            return menu;
        }

        public UIContextMenuStrip CreateSceneContentMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(TreeViewMenu.cmiDeleteAnimation);

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());

            return menu;
        }

        public UIContextMenuStrip CreateTextureMenu(UIContextMenuStrip target = null, bool isTemporaryFile = false)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(mniFileClose.CreateClone());

            if (isTemporaryFile == false)
            {
                menu.Items.Add(mniFileSave.CreateClone());
                menu.Items.Add(mniFileSaveAs.CreateClone());

                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                menu.Items.Add(mniFileRename.CreateClone());

                // TODO: マージ削除
                //menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());

                //menu.Items.Add(mniFileMergeTexture.CreateClone());
                //menu.Items[menu.Items.Count - 1].Text = Strings.Menu_FileMergeTexture;
            }

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var texture = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as Texture;
            if (texture != null && string.Compare(texture.FileDotExt, ".tga", true) != 0)
            {
                AddUserCommandMenu(menu, texture);
            }

            return menu;
        }

        public UIContextMenuStrip CreateShaderDefinitionMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            AddUserCommandMenu(menu, document);

            return menu;
        }

        public UIContextMenuStrip CreateSeparateMaterialMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            // fmt ファイルはファイル名がマテリアル名なので、別名保存や名前変更には未対応。
            // マテリアル名変更は課題化されている。
            // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-37520?src=confmacro

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(mniFileSave.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            if (document != null)
            {
                AddUserCommandMenu(menu, document);
            }

            return menu;
        }

        public UIContextMenuStrip CreateProjectDocumentMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(mniFileSaveAs.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            // TODO: プロジェクトは別？
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());
            return menu;
        }

        public UIContextMenuStrip CreateDocumentMenu(UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(mniFileSaveAs.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            if (document != null)
            {
                AddUserCommandMenu(menu, document);
            }

            return menu;
        }

        public UIContextMenuStrip CreateAnimationMenu(GuiObject guiObjectTarget, UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(mniFileSaveAs.CreateClone());

            // TODO: マージ削除
            /*
            if (CanMerge(guiObjectTarget))
            {
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                menu.Items.Add(mniFileMergeAnimation.CreateClone());
                menu.Items[menu.Items.Count - 1].Text = Strings.Menu_FileMergeAnimation;
            }
            */

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            if (document != null)
            {
                AddUserCommandMenu(menu, document);
            }

            return menu;
        }

        public UIContextMenuStrip CreateAnimationUnderModel(GuiObject guiObjectTarget, bool underAnimationSet, UIContextMenuStrip target = null)
        {
            var menu = target ?? menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(TreeViewMenu.cmiUnbindAnimation);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());

            if (!underAnimationSet)
            {
                menu.Items.Add(TreeViewMenu.cmiPreviewAnimation);
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            }

            menu.Items.Add(mniFileClose.CreateClone());
            menu.Items.Add(mniFileSave.CreateClone());
            menu.Items.Add(mniFileSaveAs.CreateClone());

            // TODO: マージ削除
            /*
            if (CanMerge(guiObjectTarget))
            {
                menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
                menu.Items.Add(mniFileMergeAnimation.CreateClone());
                menu.Items[menu.Items.Count - 1].Text = Strings.Menu_FileMergeAnimation;
            }
            */

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());

            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiFileReload.CreateClone());
            menu.Items.Add(TreeViewMenu.cmiFileOpenExplorer.CreateClone());

            // ユーザーコマンドメニューを追加
            var document = App.AppContext.CurrentSelectedObject(menu.IsObjectViewContextMenu) as IntermediateFileDocument;
            if (document != null)
            {
                AddUserCommandMenu(menu, document);
            }

            return menu;
        }

        // TODO: マージ削除

        /*
        // マージできるかどうか
        private static bool CanMerge(GuiObject target)
        {
            Debug.Assert(target is AnimationDocument);

            return (target as AnimationDocument).CanMerge;
        }
        */

        public UIContextMenuStrip CreateAnimationSetUnderModelMenu(Model model, AnimationSet animationSet)
        {
            var menu = menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(TreeViewMenu.cmiOpenAnimation);
            menu.Items.Add(TreeViewMenu.cmiCreateAnimation);
            menu.Items.Add(TreeViewMenu.cmiUnbindAnimationsFromAnimationSet);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiDeleteAnimationSet);
            menu.Items.Add(TreeViewMenu.cmiDeleteAnimationSetWithChildren);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(
                MainFrameContextMenuItems.AnimationSetPreviewState(model, animationSet)
                    ? TreeViewMenu.cmiPreviewAnimationSetOn
                    : TreeViewMenu.cmiPreviewAnimationSetOff);
            menu.Items.Add(TreeViewMenu.cmiPauseAnimationSet);
            menu.Items.Add(TreeViewMenu.cmiClearPauseAnimationSet);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiRetarget);

            return menu;
        }

        public UIContextMenuStrip CreateAnimationSetUnderSceneMenu(Model model, AnimationSet animationSet)
        {
            var menu = menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(TreeViewMenu.cmiFileOpenSceneAnimation);
            menu.Items.Add(TreeViewMenu.cmiCreateSceneAnimation2);
            menu.Items.Add(TreeViewMenu.cmiUnbindAnimationsFromAnimationSet);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(TreeViewMenu.cmiDeleteAnimationSet);
            menu.Items.Add(TreeViewMenu.cmiDeleteAnimationSetWithChildren);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(
                MainFrameContextMenuItems.AnimationSetPreviewState(model, animationSet)
                    ? TreeViewMenu.cmiPreviewAnimationSetOn
                    : TreeViewMenu.cmiPreviewAnimationSetOff);
            menu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            menu.Items.Add(mniFileRename.CreateClone());
            return menu;
        }

        public class UserCommandMenuNode
        {
            public string name;
            public TeamConfig.FileIo.UserCommand command;
            public string prog;
            public bool enabled;
            public List<UserCommandMenuNode> children = new List<UserCommandMenuNode>();
        }

        public void AddUserCommandMenu(UIToolStripMenuItem menu, UIContextMenuStrip contextMenu, IEnumerable<Document> documents, Predicate<string> validCommand, EventHandler clickEventHandler, bool skipNoTarget, bool anyDocument)
        {
            var root = new UserCommandMenuNode();
            foreach (var com in ApplicationConfig.FileIo.UserCommands.Where(x => x.Enable))
            {
                string basePath;
                string[] commands;
                try
                {
                    basePath = com.GetBaseDirectoryName();
                    commands = com.Commands.ToArray();
                }
                catch
                {
                    // 検索に失敗したら無視
                    continue;
                }

                if (commands.Length == 0)
                {
                    continue;
                }

                var exts = com.FilterExts;
                var enabled = true;
                // パスが未確定でTemporaryがFalseの場合は無効にする
                if (documents != null && documents.Any())
                {
                    if (!com.Temporary)
                    {
                        if (documents.Any(x => string.IsNullOrEmpty(x.FilePath)))
                        {
                            enabled = false;
                        }
                    }
                }

                if (exts.Any())
                {
                    if (!exts.Any(x => validCommand(x)))
                    {
                        if (skipNoTarget)
                        {
                            continue;
                        }
                        else
                        {
                            enabled = false;
                        }
                    }
                }
                else if (!anyDocument)
                {
                    enabled = false;
                }

                var menuTree = com.Name.Split(new []{'/'}, StringSplitOptions.RemoveEmptyEntries);

                var mni = root;
                foreach (var s in menuTree)
                {
                    var cmni = mni.children.Where(x => x.name == s && x.command == null).FirstOrDefault();
                    if (cmni == null)
                    {
                        cmni = new UserCommandMenuNode()
                        {
                            name = s,
                        };
                        mni.children.Add(cmni);
                    }
                    mni = cmni;
                }

                Debug.Assert(mni != null);

                foreach (var prog in commands)
                {
                    var dmni = mni;
                    try
                    {
                        var directory = Path.GetDirectoryName(prog);
                        var relativePath = PathUtility.MakeRelativePath(basePath + @"\", directory + @"\");
                        var nodes = relativePath.Split(new[] { '/' }, StringSplitOptions.RemoveEmptyEntries);
                        foreach (var s in nodes)
                        {
                            var cmni = dmni.children.Where(x => x.name == s && x.command == null).FirstOrDefault();
                            if (cmni == null)
                            {
                                cmni = new UserCommandMenuNode()
                                {
                                    name = s,
                                };
                                dmni.children.Add(cmni);
                            }
                            dmni = cmni;
                        }
                    }
                    catch
                    {
                        Debug.Assert(false);
                        continue;
                    }

                    {
                        var cmni = new UserCommandMenuNode()
                        {
                            name = Path.GetFileNameWithoutExtension(prog),
                            prog = prog,
                            enabled = enabled,
                            command = com,
                        };

                        dmni.children.Add(cmni);
                    }
                }
            }

            ConvertUserCommandNodeToMenuItem(menu, root, contextMenu, clickEventHandler);

        }

        public void ConvertUserCommandNodeToMenuItem(UIToolStripMenuItem mni, UserCommandMenuNode node,  UIContextMenuStrip contextMenu, EventHandler clickEventHandler)
        {
            if (node.name != null)
            {
                mni.Text = node.name;
            }

            bool promoted = false;
            // 子が一つならノードを省略
            if (node.children.Count == 1 && node.children[0].command != null && node.name != null)
            {
                promoted = true;
                node = node.children[0];
            }

            if (node.command != null)
            {
                mni.Tag = new UserCommandTag
                {
                    UserCommand = node.command,
                    Path = node.prog,
                    Parent = contextMenu,
                };
                mni.ToolTipText = node.prog;
                mni.Click += clickEventHandler;
                mni.Enabled = node.enabled;
                if (promoted && node.command.ShortCutKeyBind != Keys.None && contextMenu != null
                    && contextMenu.IsObjectViewContextMenu != node.command.FileTreeShortCut)
                {
                    mni.ShortcutKeyDisplayString = node.command.ShortCut;
                }
            }
            else
            {
                foreach (var child in node.children)
                {
                    var cmni = new UIToolStripMenuItem();
                    ConvertUserCommandNodeToMenuItem(cmni, child, contextMenu, clickEventHandler);
                    mni.DropDownItems.Add(cmni);
                }
            }
        }

        public bool AddUserCommandMenu(UIContextMenuStrip contextMenu, Document document, string menuText = null)
        {
            return AddUserCommandMenu(contextMenu, new[] { document }, menuText);
        }

        public bool AddUserCommandMenu(UIContextMenuStrip contextMenu, IEnumerable<Document> documents, string menuText = null)
        {
            // コマンドがない場合はユーザーコマンド自体を追加しない
            if (ApplicationConfig.FileIo.UserCommands.Count == 0)
            {
                return false;
            }

            var exts = documents.Select(x => x.FileExt);
            var menu = new UIToolStripMenuItem { Text = menuText ?? Strings.MainFrame_CreateUserCommandMenu_UserCommand };
            AddUserCommandMenu(
                menu,
                contextMenu,
                documents,
                x => exts.All(ext => string.Equals((x.StartsWith(".") ? x.Substring(1) : x), ext, StringComparison.OrdinalIgnoreCase)),
                UserCommandClickHandler,
                true,
                true);

            // コマンドが一つも追加されない場合はユーザーコマンドメニュー自体を追加しない
            if (menu.DropDownItems.Count <= 0)
            {
                return false;
            }
            contextMenu.Items.Add(menuCommandHost_.CreateGeneralSeparator());
            contextMenu.Items.Add(menu);
            return true;
        }

        public class RuntimeUserScriptMenuNode
        {
            public string name;
            public TeamConfig.FileIo.RuntimeUserScript script;
            public string prog;
            public bool enabled;
            public List<RuntimeUserScriptMenuNode> children = new List<RuntimeUserScriptMenuNode>();
        }

        public void AddRuntimeUserScriptMenu(UIToolStripMenuItem menu, UIContextMenuStrip contextMenu, IEnumerable<Document> documents, EventHandler clickEventHandler)
        {
            var root = new RuntimeUserScriptMenuNode();
            foreach (var com in ApplicationConfig.FileIo.RuntimeUserScripts.Where(x => x.Enable))
            {
                string basePath;
                string[] scripts;
                try
                {
                    basePath = com.GetBaseDirectoryName();
                    scripts = com.Scripts.ToArray();
                }
                catch
                {
                    // 検索に失敗したら無視
                    continue;
                }

                if (scripts.Length == 0)
                {
                    continue;
                }

                var enabled = Manager.Instance.IsConnected;
                var menuTree = com.Name.Split(new [] {'/'}, StringSplitOptions.RemoveEmptyEntries);

                var mni = root;
                foreach (var s in menuTree)
                {
                    var cmni = mni.children.FirstOrDefault(x => x.name == s && x.script == null);
                    if (cmni == null)
                    {
                        cmni = new RuntimeUserScriptMenuNode()
                        {
                            name = s,
                        };
                        mni.children.Add(cmni);
                    }
                    mni = cmni;
                }

                Debug.Assert(mni != null);

                foreach (var prog in scripts)
                {
                    var dmni = mni;
                    try
                    {
                        var directory = Path.GetDirectoryName(prog);
                        var relativePath = PathUtility.MakeRelativePath(basePath + @"\", directory + @"\");
                        var nodes = relativePath.Split(new[] { '/' }, StringSplitOptions.RemoveEmptyEntries);
                        foreach (var s in nodes)
                        {
                            var cmni = dmni.children.FirstOrDefault(x => x.name == s && x.script == null);
                            if (cmni == null)
                            {
                                cmni = new RuntimeUserScriptMenuNode()
                                {
                                    name = s,
                                };
                                dmni.children.Add(cmni);
                            }
                            dmni = cmni;
                        }
                    }
                    catch
                    {
                        Debug.Assert(false);
                        continue;
                    }

                    {
                        var cmni = new RuntimeUserScriptMenuNode()
                        {
                            name = Path.GetFileNameWithoutExtension(prog),
                            prog = prog,
                            enabled = enabled,
                            script = com,
                        };

                        dmni.children.Add(cmni);
                    }
                }
            }

            ConvertRuntimeUserScriptNodeToMenuItem(menu, root, contextMenu, clickEventHandler, documents);
        }

        public void ConvertRuntimeUserScriptNodeToMenuItem(UIToolStripMenuItem mni, RuntimeUserScriptMenuNode node, UIContextMenuStrip contextMenu, EventHandler clickEventHandler, IEnumerable<Document> documents)
        {
            if (node.name != null)
            {
                mni.Text = node.name;
            }

            // 子が一つならノードを省略
            if (node.children.Count == 1 && node.children[0].script != null && node.name != null)
            {
                node = node.children[0];
            }

            if (node.script != null)
            {
                mni.Tag = new RuntimeUserScriptTag
                {
                    RuntimeUserScript = node.script,
                    Path = node.prog,
                    Parent = contextMenu,
                    Models = documents.Cast<Model>().ToArray(),
                };
                mni.ToolTipText = node.prog;
                mni.Click += clickEventHandler;
                mni.Enabled = node.enabled;

                /*
                if (promoted && node.command.ShortCutKeyBind != Keys.None && contextMenu != null
                    && contextMenu.IsObjectViewContextMenu != node.command.FileTreeShortCut)
                {
                    mni.ShortcutKeyDisplayString = node.command.ShortCut;
                }
                */
            }
            else
            {
                foreach (var child in node.children)
                {
                    var cmni = new UIToolStripMenuItem();
                    ConvertRuntimeUserScriptNodeToMenuItem(cmni, child, contextMenu, clickEventHandler, documents);
                    mni.DropDownItems.Add(cmni);
                }
            }
        }

        public bool AddRuntimeUserScriptMenu(UIContextMenuStrip contextMenu, Document document)
        {
            return AddRuntimeUserScriptMenu(contextMenu, new[] { document });
            // スクリプトがない場合はユーザースクリプト自体を追加しない
        }

        private bool AddRuntimeUserScriptMenu(UIContextMenuStrip contextMenu, IEnumerable<Document> documents)
        {
            if (ApplicationConfig.FileIo.RuntimeUserScripts.Count == 0)
            {
                return false;
            }
            var menu = new UIToolStripMenuItem { Text = Strings.MainFrame_AddRuntimeUserScriptMenu_RuntimeUserScript };
            AddRuntimeUserScriptMenu(
                menu,
                contextMenu,
                documents,
                RuntimeUserScriptClickHandler);

            // スクリプトが一つも追加されない場合はユーザースクリプトメニュー自体を追加しない
            if (menu.DropDownItems.Count <= 0)
            {
                return false;
            }

            // ビューアが接続されていて、モデルのプレビューがONの時だけメニューを有効にする
            menu.Enabled = Viewer.Manager.Instance.IsConnected && documents.All(x => x is Model) &&
                           documents.Cast<Model>().All(x => x.IsVisible);
            contextMenu.Items.Add(menu);
            return true;
        }


        public void AddShaderParameterContextMenu(UIContextMenuStrip contextMenu)
        {
            contextMenu.Items.Add(TreeViewMenu.cmiCopyID);
            contextMenu.Items.Add(TreeViewMenu.cmiEditLabel);
            contextMenu.Items.Add(TreeViewMenu.cmiMaterialReferenceBehaviorValue);
            contextMenu.Items.Add(TreeViewMenu.cmiMaterialReferenceBehaviorRestriction);
        }

        public class UserCommandTag
        {
            // user command
            public FileIo.UserCommand UserCommand;
            public string Path;
            public UIContextMenuStrip Parent;
        }

        public class RuntimeUserScriptTag
        {
            // runtime script
            public FileIo.RuntimeUserScript RuntimeUserScript;
            public string Path;
            public UIContextMenuStrip Parent;
            public Model[] Models;
        }

        private static void UserCommandClickHandler(object sender, EventArgs e)
        {
            var command = (UIToolStripMenuItem)sender;
            var userCommandTag = (UserCommandTag)command.Tag;
            var documents = new List<IntermediateFileDocument>();
            if (userCommandTag.Parent.IsObjectViewContextMenu)
            {
                documents.Add(UserCommandUtil.GetSelectedDocument(false, userCommandTag.UserCommand.Name));
            }
            else
            {
                documents.AddRange(App.AppContext.SelectedFileViewObjects.OfType<IntermediateFileDocument>());
            }
            if (!documents.Any())
            {
                return;
            }

            UserCommandUtil.ExecuteUserCommand(
                documents,
                userCommandTag.Path,
                userCommandTag.UserCommand);
        }

        private static void RuntimeUserScriptClickHandler(object sender, EventArgs e)
        {
            var command = (UIToolStripMenuItem)sender;
            var scriptTag = (RuntimeUserScriptTag)command.Tag;
            if (scriptTag.Models == null || !scriptTag.Models.Any())
            {
                return;
            }

            try
            {
                var script = File.ReadAllText(scriptTag.Path);
                foreach (var model in scriptTag.Models)
                {
                    Viewer.ModelRuntimeUserScript.Send(model, script);
                }
            }
            catch (Exception)
            {
                UIMessageBox.Error(
                    Strings.MainFrame_RuntimeUserScriptClickHandler_ScriptLoadError,
                    Strings.ApplicationConfig_UserScript,
                    scriptTag.RuntimeUserScript.Name,
                    scriptTag.Path);
            }
        }

        /// <summary>
        /// リストビュー列項目メニュー表示。
        /// </summary>
        public void ShowListViewColumnMenu(Point point)
        {
            var menu = menuCommandHost_.CreateGeneralContextMenu();

            menu.Items.Add(mniViewListColumnSetting.CreateClone());
            menu.Show(this, PointToClient(point));
        }

        public void AddMostRecentlyUsedFile(bool file)
        {
            var menuItem = file ? mniMostRecentlyUsedFiles : mniMostRecentlyUsedProjects;
            menuItem.DropDownItems.Clear();
            var stack = file ? recentlyUsedFiles : recentlyUsedProjects;
            int index = 1;
            var items = new List<UIToolStripMenuItem>();
            stack.Capacity = file ?
                ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedFileCount :
                ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedProjectCount;
            foreach (var item in stack)
            {
                var text = string.Format("{0} {1}(&{2})", index, item, index < 10 ? index: 0);
                var mni = new UIToolStripMenuItem()
                {
                    Name = item,
                    Text = text,
                };
                mni.Enabled = DocumentManager.Documents.All(x => string.Compare(x.FilePath, item, true) != 0);
                mni.Click += (s, e) => DocumentManager.LoadFromFile(Enumerable.Repeat(item, 1));
                index++;
                items.Add(mni);
            }
            menuItem.DropDownItems.AddRange(items.ToArray());
            menuItem.Enabled = items.Any();
        }

        public FixedSizeUniqueStack<string> recentlyUsedFiles = new FixedSizeUniqueStack<string>();

        public FixedSizeUniqueStack<string> recentlyUsedProjects = new FixedSizeUniqueStack<string>();

        public void AddMostRecentlyUsedFile(string filePath, bool file, int maxCount)
        {
            var stack = file ? recentlyUsedFiles: recentlyUsedProjects;
            stack.Capacity = maxCount;
            stack.Push(filePath);
        }

        /// <summary>
        /// サイズ固定のスタック、重複する要素は削除される。
        /// </summary>
        public class FixedSizeUniqueStack<T> : IEnumerable<T>
        {
            private int max = 10;
            public int Capacity
            {
                get
                {
                    return max;
                }
                set
                {
                    if (max != value)
                    {
                        max = value;
                        Compactify();
                    }
                }
            }

            public Stack<T> stack = new Stack<T>();
            private void Compactify()
            {
                stack = new Stack<T>(stack.Distinct().Take(max).Reverse());
            }

            public IEnumerator<T> GetEnumerator()
            {
                return stack.Distinct().Take(max).GetEnumerator();
            }

            System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
            {
                return this.GetEnumerator();
            }

            public void Push(T v)
            {
                stack.Push(v);
                if (stack.Count > 2 * max)
                {
                    Compactify();
                }
            }
        }

        public void FileAdded(Document added)
        {
            if (string.IsNullOrEmpty(added.FilePath))
            {
                return;
            }
            switch (added.ObjectID)
            {
                case GuiObjectID.Project:
                    AddMostRecentlyUsedFile(
                        added.FilePath,
                        false,
                        ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedProjectCount);
                    break;
                default:
                    AddMostRecentlyUsedFile(
                        added.FilePath,
                        true,
                        ApplicationConfig.UserSetting.IO.MaximumRecentlyUsedFileCount);
                    break;
            }
        }
    }
}
