﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;

namespace App.FileView
{
    [ToolboxItem(true)]
    public partial class FileViewPanel : UserControl
    {
        public FileViewPanel()
        {
            InitializeComponent();

            ctlTitleBar.Text = Strings.FileViewPanel_TitleBar;

            tvwView.Initialize();
            App.AppContext.FileTreeView = tvwView;

            OneShotIdleProcess.Execute(UpdateForm);

            App.AppContext.PropertyChanged	+= (s, e) =>
            {
                UpdateDocumentList(e);
                UpdateForm();
            };

            tvwView.AfterSelect			+= (s, e) => UpdateForm();
        }

        public int SplitterDistance
        {
            get
            {
                return spcClient.SplitterDistance;
            }

            set
            {
                spcClient.SplitterDistance = value;
            }
        }

        public void UpdateDocumentList(IEnumerable<DocumentPropertyChangedArgs> args)
        {
            tvwView.UpdateDocumentList(args);
        }

        private void tvwView_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                if (tvwView.SelectedObject != null)
                {
                    // 選択オブジェクト変更
                    App.AppContext.SelectedTarget.Set(tvwView.SelectedObject);

                    // プロパティウィンドウ表示
                    App.PropertyEdit.ObjectPropertyDialog.ShowPropertyDialog();
                }
            }
        }

        private void UpdateForm()
        {
            var target = App.AppContext.SelectedFileViewObject;

            if (target != null)
            {
                FileData data = null;
                {
                    switch(target.ObjectID)
                    {
                        case GuiObjectID.Project:						data = new ProjectFileData(						target as ProjectDocument);				break;
                        case GuiObjectID.Model:							data = new ModelFileData(						target as Model);						break;
                        case GuiObjectID.Texture:						data = new TextureFileData(						target as Texture);						break;
                        case GuiObjectID.SkeletalAnimation:				data = new SkeletalAnimationFileData(			target as SkeletalAnimation);			break;
                        case GuiObjectID.ShapeAnimation:				data = new ShapeAnimationFileData(				target as ShapeAnimation);				break;
                        case GuiObjectID.MaterialAnimation:             data = new MaterialAnimationFileData(           target as MaterialAnimation); break;
                        case GuiObjectID.ShaderParameterAnimation:		data = new ShaderParameterAnimationFileData(	target as ShaderParameterAnimation);	break;
                        case GuiObjectID.ColorAnimation:				data = new ColorAnimationFileData(				target as ColorAnimation);				break;
                        case GuiObjectID.TextureSrtAnimation:			data = new TextureSrtAnimationFileData(			target as TextureSrtAnimation);			break;
                        case GuiObjectID.MaterialVisibilityAnimation:	data = new MaterialVisibilityAnimationFileData(	target as MaterialVisibilityAnimation);	break;
                        case GuiObjectID.BoneVisibilityAnimation:		data = new BoneVisibilityAnimationFileData(		target as BoneVisibilityAnimation);		break;
                        case GuiObjectID.TexturePatternAnimation:		data = new TexturePatternAnimationFileData(		target as TexturePatternAnimation);		break;
                        case GuiObjectID.ShaderDefinition:				data = new ShaderDefinitionFileData(			target as ShaderDefinition);			break;
                        case GuiObjectID.SeparateMaterial:				data = new SeparateMaterialFileData(target as SeparateMaterial); break;
                        default:
                        {
                            if (target is IntermediateFileDocument)
                            {
                                data = new IntermediateFileData(target as IntermediateFileDocument);
                            }
                            break;
                        }
                    }
                }

                InitializePgdInformation();

                pgdInformation.SelectedObject = data;
                pgdInformation.Enabled = true;
                pgdInformation.ViewForeColor = Color.FromArgb(0, 0, 1);
                pgdInformation.CategoryForeColor = SystemColors.ControlText;

                // 項目を展開しておく
                pgdInformation.ExpandAllGridItems();
            }
            else
            {
                InitializePgdInformation();

                pgdInformation.SelectedObject = new FileData();
                pgdInformation.Enabled = false;
                pgdInformation.ViewForeColor = SystemColors.WindowText;
                pgdInformation.CategoryForeColor = SystemColors.GrayText;
            }
        }

        private void pnlPanelFolderBase_Paint(object sender, PaintEventArgs e)
        {
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Top, ClientRectangle.Left, tvwView.Bottom - 1);
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Right - 1, ClientRectangle.Top, ClientRectangle.Right - 1, tvwView.Bottom - 1);
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, tvwView.Bottom, ClientRectangle.Right - 1, tvwView.Bottom);
        }

        private Controls.UIPropertyGrid pgdInformation;

        private void InitializePgdInformation()
        {
            if (pgdInformation != null)
            {
                return;
            }

            pgdInformation = new App.Controls.UIPropertyGrid();

            //
            // pgdInformation
            //
            pgdInformation.Dock = System.Windows.Forms.DockStyle.Fill;
            pgdInformation.HelpVisible = false;
            pgdInformation.Location = new System.Drawing.Point(0, 0);
            pgdInformation.Name = "pgdInformation";
            pgdInformation.PropertySort = System.Windows.Forms.PropertySort.Categorized;
            pgdInformation.Size = new System.Drawing.Size(293, 213);
            pgdInformation.TabIndex = 0;
            pgdInformation.ToolbarVisible = false;

            spcClient.Panel2.Controls.Add(pgdInformation);
        }

        private void spcClient_Panel2_Paint(object sender, PaintEventArgs e)
        {
            e.Graphics.DrawLine(SystemPens.ControlDark, spcClient.Panel2.ClientRectangle.Left, spcClient.Panel2.ClientRectangle.Top, spcClient.Panel2.ClientRectangle.Left, spcClient.Panel2.ClientRectangle.Bottom - 1);
            e.Graphics.DrawLine(SystemPens.ControlDark, spcClient.Panel2.ClientRectangle.Right - 1, spcClient.Panel2.ClientRectangle.Top, spcClient.Panel2.ClientRectangle.Right - 1, spcClient.Panel2.ClientRectangle.Bottom - 1);
            e.Graphics.DrawLine(SystemPens.ControlDark, spcClient.Panel2.ClientRectangle.Left, tvwView.Top, spcClient.Panel2.ClientRectangle.Right - 1, tvwView.Top);
            e.Graphics.DrawLine(SystemPens.ControlDark, spcClient.Panel2.ClientRectangle.Left, spcClient.Panel2.ClientRectangle.Bottom - 1, spcClient.Panel2.ClientRectangle.Right - 1, spcClient.Panel2.ClientRectangle.Bottom - 1);
        }
    }
}
