﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Properties;
using App.PropertyEdit;
using App.Utility;
using App.res;
using ConfigCommon;
using NintendoWare.G3d.Edit;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using EditCommand = App.Command.EditCommand;
using Timer = System.Windows.Forms.Timer;

namespace App.FileView
{
    using System.Runtime.CompilerServices;

    /// <summary>
    /// ファイルツリービュークラス。
    /// </summary>
    public sealed class FileTreeView : UITreeView
    {
        public GuiObject SelectedObject { get { return (SelectedNode != null) ? (SelectedNode.Tag as GuiObject) : null; } }

        private ViewItem project;
        private ViewItem sceneFolder;
        private ViewItem modelFolder;
        private ViewItem materialFolder;
        private ViewItem animationFolder;
        private ViewItem textureFolder;
        private ViewItem shaderDefinitionFolder;

        private enum IconName
        {
            // ReSharper disable UnusedMember.Local
            Folder_Scene,
            Folder_Model,
            Folder_Texture,
            Folder_Animation,
            Folder_ShaderDefinition,
            Folder_SeparateMaterial,

            Doc_Project,
            Doc_Model,
            Doc_AnimationSet,
            Doc_Texture,
            Doc_SkeletalAnimation,
            Doc_ShapeAnimation,
            Doc_MaterialAnimation,
            Doc_ShaderParameterAnimation,
            Doc_ColorAnimation,
            Doc_TextureSrtAnimation,
            Doc_MaterialVisibilityAnimation,
            Doc_BoneVisibilityAnimation,
            Doc_TexturePatternAnimation,
            Doc_SceneAnimation,
            Doc_CameraAnimation,
            Doc_LightAnimation,
            Doc_FogAnimation,
            Doc_ShaderDefinition,
            Doc_SeparateMaterial,

            //
            Doc_DefaultAnimationSet,
            // ReSharper restore UnusedMember.Local
        };

        private enum SmallIconName
        {
            Error,
            Attach,
            Preview,
            Startup
        }

        private const string TexturePrefix = "*TEXTURE_";

        private ImageList imageList_;

        /// <summary>
        /// 複数選択モード
        /// </summary>
        private enum MultiSelection
        {
            Disable,    // 無効状態
            Ready,      // 有効に遷移可能状態
            Enable,     // 有効状態
        }
        private MultiSelection multiSelection = MultiSelection.Disable;

        /// <summary>
        /// 直前のマウス座標
        /// </summary>
        private System.Drawing.Point? lastMouseLocation = null;

        // LastDropTargetにはドロップ先のアニメーションセットまたはモデルが入る
        public static GuiObject LastDropTarget { get; set; }

        // ToolTip
        private HintToolTip _ttpHint = null;
        private string oldHintText = string.Empty;
        private Timer _HintTimer = null;

        public FileTreeView()
        {
            App.AppContext.BeginRenameEvent += RenameMenuCommand;
            AfterLabelEdit += AfterLabelEditHandler;
            TreeViewNodeSorter = new ViewItemComparer();

            // プロジェクトの縮小をキャンセル
            SuppressDoubleClickExpand = true;

            DrawMode = TreeViewDrawMode.OwnerDrawText;
            App.AppContext.PauseChangedEvent += stateChanged =>
            {
                if (stateChanged)
                {
                    Invalidate();
                }
            };

            Model.IsSendAttachedChanged += (x, y) => InvokeInvalidate();
            Model.IsAttachedChanged += (x, y) => InvokeInvalidate();
            ShaderDefinition.IsAttachedChanged += (x, y) => InvokeInvalidate();

            // ツールチップ
            ControlRemoved += (sender, e) =>
            {
                if (_HintTimer != null)
                {
                    _HintTimer.Dispose();
                    _HintTimer = null;
                }
            };
        }

        public void InvokeInvalidate()
        {
            if (IsHandleCreated)
            {
                Action action = Invalidate;
                BeginInvoke(action, null);
            }
        }

        public void Initialize()
        {
            InitializeImageList();
            CreateFolderNodes();
            UpdateDocumentList();
        }

        protected override void OnBeforeCollapse(TreeViewCancelEventArgs e)
        {
            if (e.Node == project)
            {
                e.Cancel = true;
            }
            base.OnBeforeCollapse(e);
        }



        public class ViewItem : TreeNode
        {
            public bool IsPreview{ get; set; }
            public bool IsShowInObjView { get; set; }
            public int DocIconImageX;
            public int SmallIconImageX;

            public ViewItem(string text)
                : base(text)
            {
            }

            public ViewItem() { }

            #region ドラグアンドドロップ

            /// <summary>
            /// ノードがドラッグ可能かどうか
            /// </summary>
            //public bool Draggable { get; set; }
            public DragDropEffects DragEffects { get; set; }


            /// <summary>
            /// ノードにドロップ可能かどうかを判断するイベントハンドラ
            /// </summary>
            public event DragEventHandler DragOver = null;

            /// <summary>
            /// ドロップ時のイベントハンドラ
            /// </summary>
            public event DragEventHandler DragDrop = null;

            public void OnItemDrag(object sender, ItemDragEventArgs e)
            {
                if (DragEffects != DragDropEffects.None)
                {
                    if (e.Button == System.Windows.Forms.MouseButtons.Left)
                    {
                        var item = e.Item as ViewItem;
                        if (item != null)
                        {
                            TreeView.DoDragDrop(e.Item, DragEffects);
                        }
                    }
                }
            }

            public void OnDragOver(object sender, DragEventArgs e)
            {
                if (DragOver != null)
                {
                    DragOver(sender, e);
                }
            }

            public void OnDragDrop(object sender, DragEventArgs e)
            {
                if (DragDrop != null)
                {
                    //using (var block = new App.Data.DocumentBase.PropertyChangedSuppressBlock())
                    {
                        DragDrop(sender, e);
                    }
                }
            }
            #endregion
        }

        private class AnimNodeInfo
        {
            public AnimNodeInfo(TreeNode node, AnimationDocument animation, AnimationSet animationSet)
            {
                Node = node;
                Animation = animation;
                AnimationSet = animationSet;
            }

            public TreeNode Node;
            public AnimationDocument Animation;
            public AnimationSet AnimationSet;
        }

        private AnimNodeInfo[] GetSourceInfos()
        {
            return SelectedNodes.Where(x => x?.Tag is AnimationDocument).Select(
                x =>
                new AnimNodeInfo(
                    x,
                    x.Tag as AnimationDocument,
                    x.Parent == sceneFolder
                        ? DocumentManager.DefaultSceneAnimationSet
                        : x.Parent.Tag as AnimationSet)).ToArray();
        }

        public void AnimationDragOver(DragEventArgs e, AnimationSet hasAnimation, bool sceneAnimation, Model ownerModel)
        {
            var source = (ViewItem)e.Data.GetData(typeof(ViewItem));
            if (source == null) { return; }

            var sources = SelectedNodes.Where(x => x.Tag is AnimationDocument).ToArray();
            var sourceInfos = GetSourceInfos();


            var destination = GetNodeAt(PointToClient(new Point(e.X, e.Y))) as ViewItem;
            if (destination == null) { return; }

            var guiObjects = App.AppContext.SelectedFileViewObjects?.ToArray();
            if (guiObjects == null || !guiObjects.Any()) { return; }
            if (guiObjects.Any(x => !(x is AnimationDocument))) { return; }

            var animations = guiObjects.Cast<AnimationDocument>().ToArray();
            if (animations.Any(x => (x.ObjectID == GuiObjectID.SceneAnimation) != sceneAnimation))
            {
                return;
            }
            // 同名のアニメーションはドロップできない
            if (sourceInfos.Any(x => sourceInfos.Any(y => x != y && string.Equals(x.Animation.FileName, y.Animation.FileName, StringComparison.OrdinalIgnoreCase))))
            {
                return;
            }

            if (sceneAnimation)
            {
                if (hasAnimation.Animations.Any(x => animations.Any(y => string.Equals(x.Name, y.FileName, StringComparison.OrdinalIgnoreCase))))
                {
                    // アニメーションセット内でのドラッグドロップを移動として扱う
                    // 複数のアニメーションセットに属するアニメーションをどれかのアニメーションセットにドロップは出来ない
                    if (sources.Any(x => x.Parent == destination)
                        || sources.Any(x => !(x.Parent?.Tag is AnimationSet))
                        || sources.Any(x => (x.Parent?.Tag as AnimationSet) != hasAnimation))
                    {
                        return;
                    }

                    if (source.DragEffects != DragDropEffects.Move)
                    {
                        var effect = GetDragDropEffect();
                        if (effect == DragDropEffects.Move)
                        {
                            e.Effect = effect;
                        }
                    }
                    else
                    {
                        e.Effect = source.DragEffects;
                    }
                    return;
                }
            }
            else
            {
                if (animations.Any(data => ownerModel.AnimationSetsWithDefault
                    .SelectMany(x => x.Animations).Contains(new AnimationSetItem(data.FileName, data.FileLocation))))
                {
                    if (animations.Any(data => hasAnimation.Animations.Any(x => string.Equals(x.Name, data.FileName, StringComparison.OrdinalIgnoreCase))))
                    {
                        // アニメーションセット内でのドラッグドロップを移動として扱う
                        // 複数のアニメーションセットに属するアニメーションをどれかのアニメーションセットにドロップは出来ない
                        if (sources.Any(x => x.Parent == destination)
                            || sources.Any(x => !(x.Parent?.Tag is AnimationSet))
                            || sources.Any(x => (x.Parent?.Tag as AnimationSet) != hasAnimation))
                        {
                            return;
                        }

                        if (source.DragEffects != DragDropEffects.Move)
                        {
                            var effect = GetDragDropEffect();
                            if (effect == DragDropEffects.Move)
                            {
                                e.Effect = effect;
                            }
                        }
                        else
                        {
                            e.Effect = source.DragEffects;
                        }
                        return;
                    }
                }
                else
                {
                    if (animations.Any(data => !DocumentManager.CanAddAnimation(data, ownerModel)))
                    {
                        return;
                    }
                }
            }

            switch (source.DragEffects)
            {
                case DragDropEffects.Move:
                case DragDropEffects.Copy:
                    e.Effect = source.DragEffects;
                    break;
                default:
                    e.Effect = GetDragDropEffect();
                    break;
            }
        }

        public void AnimationSetDragOver(DragEventArgs e, Model target)
        {
            var source = (ViewItem)e.Data.GetData(typeof(ViewItem));
            if (source == null)
            {
                return;
            }

            var guiObjects = App.AppContext.SelectedFileViewObjects?.ToArray();
            if (guiObjects == null || !guiObjects.Any()) { return; }

            var animationSets = guiObjects.OfType<AnimationSet>().ToArray();
            if (!animationSets.Any())
            {
                return;
            }

            if (animationSets.Any(x => x.IsDefaultAnimationSet))
            {
                return;
            }

            if (animationSets.Any(x => x.IsSceneAnimationSet))
            {
                return;
            }

            DragDropEffects effect = DragDropEffects.None;
            switch (source.DragEffects)
            {
                case DragDropEffects.Move:
                case DragDropEffects.Copy:
                    effect = source.DragEffects;
                    break;
                default:
                    effect = GetDragDropEffect();
                    break;
            }

            if (effect != DragDropEffects.Copy && animationSets.Any(x => target.AnimationSets.Contains(x)))
            {
                return;
            }

            if (animationSets.Any(data =>
                                  target.AnimationSetsWithDefault.SelectMany(x => x.Animations)
                                      .Any(x => data.Animations.Any(y => x.Name == y.Name && x.Directory != y.Directory))))
            {
                return;
            }

            e.Effect = effect;
        }

        public void AnimationDragDrop(DragEventArgs e, AnimationSet target, bool sceneAnimation, Model model = null)
        {
            var source = (ViewItem)e.Data.GetData(typeof(ViewItem));
            if (source == null)
            {
                return;
            }

            var sources = SelectedNodes.Where(x => x.Tag is AnimationDocument);

            var sourceInfos = GetSourceInfos();

            var animations = sourceInfos.Select(x => x.Animation).ToArray();

            var destination = GetNodeAt(PointToClient(new Point(e.X, e.Y))) as ViewItem;
            if (destination == null)
            {
                return;
            }

            var commandSet = new EditCommandSet();
            // 同一アニメーションセット内でのドラッグドロップは優先度変更とする
            {
                //var parent = source.Parent.Tag as AnimationSet;

                if (sourceInfos.All(x => x.AnimationSet == target))
                {
                    if (destination == source)
                    {
                        return;
                    }
                    var destData = destination.Tag as AnimationDocument;
                    if (destData == null)
                    {
                        return;
                    }

                    var sourceItems = animations.Select(x => new AnimationSetItem(x.FileName, x.FileLocation)).ToArray();
                    var sourceIds = sourceItems.Select(x => target.Animations.IndexOf(x));

                    var destId =
                        target.Animations.IndexOf(new AnimationSetItem(destData.FileName, destData.FileLocation));

                    // 新しいアニメーション配列を作る
                    var newAnimations = target.Animations.Where((t, i) => !sourceIds.Contains(i)).ToList();
                    destId = destId > newAnimations.Count ? newAnimations.Count : destId;
                    newAnimations.InsertRange(destId, sourceItems);

                    commandSet.Add(DocumentManager.CreateAnimationsEditCommand(target, newAnimations.ToArray(), false));
                    commandSet.Add(new LazyCommand(() => DocumentManager.CreateAnimationUpdateBindCommand(animations)));
                    TheApp.CommandManager.Execute(commandSet);
                    return;
                }
            }

            // 移動の確認
            if (e.Effect == DragDropEffects.Move && Control.ModifierKeys == Keys.None)
            {
                string targetName = model != null ? model.FileName : ((GuiObject)target).Name;
                if (target == DocumentManager.DefaultSceneAnimationSet)
                {
                    targetName = res.Strings.FileViewFolderName_Scene;
                }

                if (!UIMessageBox.YesNo(
                        res.Strings.FileView_OkCancel_Move,
                        string.Join(", ", animations.Select(x => x.FileName)),
                        targetName))
                {
                    return;
                }
            }

            // スケルタルアニメーションのバインドの確認
            var owner = DocumentManager.Models.FirstOrDefault(x => x.AnimationSetsWithDefault.Contains(target));
            if (animations.Any(x => x is SkeletalAnimation))
            {
                if (owner != null)
                {
                    var errorInfos = new List<SkeletalAnimBindDialog.ErrorInfo>();
                    foreach (var skeletalAnim in animations.OfType<SkeletalAnimation>())
                    {
                        var messages = skeletalAnim.CheckBind(owner).ToArray();
                        if (messages.Any())
                        {
                            errorInfos.Add(
                                new SkeletalAnimBindDialog.ErrorInfo()
                                    {
                                        model = owner,
                                        animation = skeletalAnim,
                                        messages = messages,
                                    });
                        }
                    }
                    if (errorInfos.Any())
                    {
                        using (var dialog = new SkeletalAnimBindDialog(errorInfos.ToArray(), true))
                        {
                            if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.Cancel)
                            {
                                return;
                            }
                        }
                    }
                }
            }

/*
            AnimationSet sourceOwner;
            if (source.Parent == sceneFolder)
            {
                sourceOwner = DocumentManager.DefaultSceneAnimationSet;
            }
            else
            {
                sourceOwner = source.Parent.Tag as AnimationSet;
            }

            Model sourceOwnerModel = null;
            if (sourceOwner != null)
            {
                sourceOwnerModel = source.Parent.Parent.Tag as Model;
            }
*/

            // フレーム状態を引き継ぐ
            if (owner != null)
            {
                double frame = 0;
                foreach (var nodeInfo in sourceInfos)
                {
                    var sourceOwner = nodeInfo.Node.Parent == sceneFolder
                                          ? DocumentManager.DefaultSceneAnimationSet
                                          : nodeInfo.Node.Parent?.Tag as AnimationSet;
                    var sourceOwnerModel = (sourceOwner != null) ? nodeInfo.Node.Parent?.Parent?.Tag as Model : null;

                    var pause = sourceOwnerModel != null
                                && nodeInfo.Animation.PauseFrames.TryGetValue(
                                    new KeyValuePair<object, string>(sourceOwnerModel.ModelId, sourceOwner.Name),
                                    out frame);
                    var invisible = sourceOwnerModel != null
                                    && nodeInfo.Animation.Pause.InvisibleBinds.Contains(
                                        new KeyValuePair<object, string>(sourceOwnerModel.ModelId, sourceOwner.Name));
                    nodeInfo.Animation.SetPause(pause, frame, owner.ModelId, target.Name, invisible);
                }
            }

            // シーンアニメーションの表示・非表示を引き継ぐ。
            {
                foreach (var nodeInfo in sourceInfos)
                {
                    // 表示・非表示の切り替えは Undo/Redo の対象外のようだが、
                    // 引継ぎ後に状態を切り替えて Undo や手動で元の位置に戻すと、切り替え状態が破綻する。
                    // しかしこの挙動はすぐ上の「フレーム状態を引き継ぐ」を踏襲したものである。
                    var sceneAnim = nodeInfo.Animation as SceneAnimation;
                    if (sceneAnim != null)
                    {
                        if (sceneAnim.InvisibleBinds.Contains(nodeInfo.AnimationSet))
                        {
                            sceneAnim.InvisibleBinds.Add(target);
                        }
                        else
                        {
                            sceneAnim.InvisibleBinds.Remove(target);
                        }
                    }
                }
                Viewer.ViewerUtility.SendPreviewSceneAnimSet();
            }

            // バインド変更
            {
                var destData = destination.Tag as AnimationDocument;
                var newAnims = target.Animations.ToList();
                if (destData != null)
                {
                    var targetId = newAnims.IndexOf(new AnimationSetItem(destData.FileName, destData.FileLocation));
                    newAnims.InsertRange(
                        targetId,
                        animations.Select(x => new AnimationSetItem(x.FileName, x.FileLocation)));
                }
                else
                {
                    newAnims =
                        target.Animations.Concat(
                            animations.Select(x => new AnimationSetItem(x.FileName, x.FileLocation))).ToList();
                }

                switch (e.Effect)
                {
                    case DragDropEffects.Move:
                        {
                            // ドロップ先を記録しておく
                            LastDropTarget = target;

                            var sourceInfoGroups = sourceInfos.GroupBy(x => x.AnimationSet);
                            foreach (var sourceInfoGroup in sourceInfoGroups)
                            {
                                commandSet.Add(
                                    DocumentManager.CreateAnimationsEditCommand(
                                        sourceInfoGroup.Key,
                                        sourceInfoGroup.Key.Animations.Where(
                                            x =>
                                            sourceInfoGroup.All(
                                                y =>
                                                !(x.Name == y.Animation.FileName
                                                  && x.Directory.Equals(y.Animation.FileLocation, StringComparison.OrdinalIgnoreCase))))
                                            .ToArray(),
                                        false));
                            }
                        }
                        commandSet.Add(
                            DocumentManager.CreateAnimationsEditCommand(target, newAnims.ToArray(), false));
                        commandSet.Add(
                            new LazyCommand(() => DocumentManager.CreateAnimationUpdateBindCommand(animations)));
                        break;
                    case DragDropEffects.Copy:
                        {
                            // ドロップ先を記録しておく
                            LastDropTarget = target;

                            commandSet.Add(
                                DocumentManager.CreateAnimationsEditCommand(target, newAnims.ToArray(), false));
                            commandSet.Add(
                                new LazyCommand(() => DocumentManager.CreateAnimationUpdateBindCommand(animations)));
                        }
                        break;
                }
            }
            TheApp.CommandManager.Execute(commandSet);
        }

        public void AnimationSetDragDrop(DragEventArgs e, Model target)
        {
            var source = (ViewItem)e.Data.GetData(typeof(ViewItem));
            if (source == null)
            {
                return;
            }

            var guiObjects = App.AppContext.SelectedFileViewObjects?.ToArray();
            if (guiObjects == null || !guiObjects.Any()) { return; }

            var animationSets = guiObjects.OfType<AnimationSet>().ToArray();
            if (!animationSets.Any())
            {
                return;
            }

            var dragAnimations = animationSets.SelectMany(x => x.Animations).ToArray();

            // 移動の確認
            if (e.Effect == DragDropEffects.Move && Control.ModifierKeys == Keys.None)
            {
                if (!UIMessageBox.YesNo(res.Strings.FileView_OkCancel_Move, string.Join(", ", animationSets.Select(x => x.Name)), target.Name))
                {
                    return;
                }
            }

            // アニメーションのバインド変更チェック
            {
                var skeletalAnimations = DocumentManager.Animations.OfType<SkeletalAnimation>().ToArray();
                var errorInfos = new List<SkeletalAnimBindDialog.ErrorInfo>();
                foreach (var skeletalAnimation in skeletalAnimations)
                {
                    if (dragAnimations.Contains(new AnimationSetItem(skeletalAnimation.FileName, skeletalAnimation.FileLocation)))
                    {
                        var messages = skeletalAnimation.CheckBind(target).ToArray();
                        if (messages.Any())
                        {
                            errorInfos.Add(
                                new SkeletalAnimBindDialog.ErrorInfo()
                                {
                                    model = target,
                                    animation = skeletalAnimation,
                                    messages = messages,
                                });
                        }
                    }
                }
                if (errorInfos.Any())
                {
                    using (var dialog = new SkeletalAnimBindDialog(errorInfos.ToArray(), true))
                    {
                        if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.Cancel)
                        {
                            return;
                        }
                    }
                }
            }

            // バインドの変更
            if (e.Effect == DragDropEffects.Move)
            {
                // ドロップ先を記録しておく
                LastDropTarget = target;
                var commandSet = new EditCommandSet();
                var newAnimationSets = new List<AnimationSet>();
                var animations = new List<AnimationDocument>();
                var owners = new List<Model>();
                foreach (var animationSet in animationSets)
                {
                    var owner =
                        DocumentManager.Models.FirstOrDefault(
                            x => x.AnimationSetsWithDefault.Any(y => y == animationSet));
                    Debug.Assert(owner != null, "owner != null");

                    var newAnimationSet = new AnimationSet(false);
                    newAnimationSet.Name = StringUtility.UniqueName(animationSet.Name, target.AnimationSets.Select(x => x.Name));
                    newAnimationSet.Animations.AddRange(animationSet.Animations);

                    // 一時停止状態を初期化する
                    foreach (var animation in DocumentManager.GetAnimations(newAnimationSet.Animations))
                    {
                        double frame;
                        var key = new KeyValuePair<object, string>(owner.ModelId, animationSet.Name);
                        var pause = animation.PauseFrames.TryGetValue(key, out frame);
                        var invisible = animation.Pause.InvisibleBinds.Contains(key);
                        animation.SetPause(pause, frame, target.ModelId, newAnimationSet.Name, invisible);
                    }
                    owners.Add(owner);
                    newAnimationSets.Add(newAnimationSet);
                    animations.AddRange(DocumentManager.GetAnimations(newAnimationSet.Animations));
                }
                foreach (var owner in owners)
                {
                    commandSet.Add(DocumentManager.CreateAnimationSetEditCommand(owner, owner.AnimationSets.Where(x => !animationSets.Contains(x)).ToArray()));
                }
                commandSet.Add(DocumentManager.CreateAnimationSetEditCommand(target, target.AnimationSets.Concat(newAnimationSets).ToArray()));
                commandSet.Add(new LazyCommand(() => DocumentManager.CreateAnimationUpdateBindCommand(animations.Distinct())));
                TheApp.CommandManager.Execute(commandSet);

            }
            else if (e.Effect == DragDropEffects.Copy)
            {
                // ドロップ先を記録しておく
                LastDropTarget = target;
                var commandSet = new EditCommandSet();
                var newAnimationSets = new List<AnimationSet>();
                var animations = new List<AnimationDocument>();
                var owners = new List<Model>();

                foreach (var animationSet in animationSets)
                {
                    var owner =
                        DocumentManager.Models.FirstOrDefault(
                            x => x.AnimationSetsWithDefault.Any(y => y == animationSet));
                    Debug.Assert(owner != null, "owner != null");

                    var newAnimationSet = new AnimationSet(false);
                    newAnimationSet.Name = StringUtility.UniqueName(animationSet.Name, target.AnimationSets.Select(x => x.Name));
                    newAnimationSet.Animations.AddRange(animationSet.Animations);

                    // 一時停止状態を初期化する
                    foreach (var animation in DocumentManager.GetAnimations(newAnimationSet.Animations))
                    {
                        double frame;
                        var key = new KeyValuePair<object, string>(owner.ModelId, animationSet.Name);
                        bool pause = animation.PauseFrames.TryGetValue(key, out frame);
                        bool invisible = animation.Pause.InvisibleBinds.Contains(key);
                        animation.SetPause(pause, frame, target.ModelId, newAnimationSet.Name, invisible);
                    }

                    newAnimationSets.Add(newAnimationSet);
                    animations.AddRange(DocumentManager.GetAnimations(animationSet.Animations));
                }
                commandSet.Add(DocumentManager.CreateAnimationSetEditCommand(target, target.AnimationSets.Concat(newAnimationSets).ToArray()));
                commandSet.Add(new LazyCommand(() => DocumentManager.CreateAnimationUpdateBindCommand(animations)));
                TheApp.CommandManager.Execute(commandSet);
            }
        }

        public void AnimationDragOver(DragEventArgs e, SceneAnimation owner)
        {
            var source = (ViewItem)e.Data.GetData(typeof(ViewItem));
            if (source == null) { return; }

            var destination = GetNodeAt(PointToClient(new Point(e.X, e.Y))) as ViewItem;
            if (destination == null) { return; }

            var guiObjects = App.AppContext.SelectedFileViewObjects?.ToArray();
            if (guiObjects == null || !guiObjects.Any()) { return; }

            var srcAnim = source.Tag as ISceneAnimationObject;
            var dstAnim = destination.Tag as ISceneAnimationObject;
            if ((srcAnim?.Owner == owner) && (dstAnim?.Owner == owner) && (owner != null))
            {
                // 同ファイル内の同種間でのみ移動許可。
                if (guiObjects.GroupBy(x => x.ObjectID).Count() != 1)
                {
                    return;
                }
                bool movable = false;
                switch (guiObjects.First().ObjectID)
                {
                    case GuiObjectID.CameraAnimation:
                        {
                            movable =
                                (dstAnim as CameraAnimation != null) &&
                                (guiObjects.OfType<CameraAnimation>().Where(x => x.Owner == owner).Count() == guiObjects.Length);
                            break;
                        }
                    case GuiObjectID.LightAnimation:
                        {
                            movable =
                                (dstAnim as LightAnimation != null) &&
                                (guiObjects.OfType<LightAnimation>().Where(x => x.Owner == owner).Count() == guiObjects.Length);
                            break;
                        }
                    case GuiObjectID.FogAnimation:
                        {
                            movable =
                                (dstAnim as FogAnimation != null) &&
                                (guiObjects.OfType<FogAnimation>().Where(x => x.Owner == owner).Count() == guiObjects.Length);
                            break;
                        }
                }
                if ((source.DragEffects == DragDropEffects.Move) && movable)
                {
                    var effect = GetDragDropEffect();
                    if (effect == DragDropEffects.Move)
                    {
                        e.Effect = effect;
                    }
                }
            }
        }

        public void AnimationDragDrop(DragEventArgs e, SceneAnimation owner)
        {
            var source = (ViewItem)e.Data.GetData(typeof(ViewItem));
            if (source == null)
            {
                return;
            }

            var destination = GetNodeAt(PointToClient(new Point(e.X, e.Y))) as ViewItem;
            if (destination == null)
            {
                return;
            }

            var guiObjects = App.AppContext.SelectedFileViewObjects?.ToArray();
            if (guiObjects == null || !guiObjects.Any())
            {
                return;
            }

            var srcAnim = source.Tag as ISceneAnimationObject;
            var dstAnim = destination.Tag as ISceneAnimationObject;
            if ((srcAnim?.Owner == owner) && (dstAnim?.Owner == owner) && (owner != null))
            {
                // 同ファイル内の同種間でのみ移動許可。
                if (guiObjects.GroupBy(x => x.ObjectID).Count() != 1)
                {
                    return;
                }

                Func<List<ISceneAnimationObject>, List<ISceneAnimationObject>, List<ISceneAnimationObject>> createReorderedList = (srcAnims, targetAnims) =>
                {
                    // 元リストでの順序を維持しつつ並び替えるので、並び替え対象ノードをソートしておく。
                    var srcAnimIndexTable = srcAnims.Select((x, i) => Tuple.Create(x, i)).ToDictionary(x => x.Item1, x => x.Item2);
                    targetAnims = targetAnims.OrderBy(x => srcAnimIndexTable[x]).ToList();

                    // 並び替え前の状態での移動元と移動先。
                    var from = srcAnims.IndexOf(srcAnim);
                    var to = srcAnims.IndexOf(dstAnim);

                    // 並び替え前の移動先。
                    // 移動は基本的に挿入で行い、pos が挿入位置となる。
                    // 後方位置への移動の場合は、その次の位置に挿入する。
                    // 移動方向によって挙動が異なるが、同じ挙動であるモデルアニメーションに合わせている。
                    int srcPos = (to >= from) ? (to + 1) : to;

                    // 移動先の決定と並び替え。
                    int newPos = -1;
                    var newAnims = srcAnims.Except(targetAnims).ToList();
                    foreach (var anim in srcAnims.Skip(srcPos))
                    {
                        newPos = newAnims.IndexOf(anim);
                        if (newPos != -1)
                        {
                            break;
                        }
                    }
                    if (newPos != -1)
                    {
                        newAnims.InsertRange(newPos, targetAnims);
                    }
                    else
                    {
                        newAnims.AddRange(targetAnims);
                    }
                    return newAnims;
                };

                switch (guiObjects.First().ObjectID)
                {
                    case GuiObjectID.CameraAnimation:
                        {
                            var targetAnims = guiObjects.OfType<ISceneAnimationObject>().ToList();
                            if ((targetAnims.Count() == guiObjects.Length) && targetAnims.All(x => x.Owner == owner))
                            {
                                var srcAnims = owner.CameraAnims.Cast<ISceneAnimationObject>().ToList();
                                var newCameraAnims = createReorderedList(srcAnims, targetAnims).Cast<CameraAnimation>().ToList();
                                TheApp.CommandManager.Execute(DocumentManager.CreateCameraAnimationsEditCommand(owner, newCameraAnims));
                            }
                            break;
                        }
                    case GuiObjectID.LightAnimation:
                        {
                            var targetAnims = guiObjects.OfType<ISceneAnimationObject>().ToList();
                            if ((targetAnims.Count() == guiObjects.Length) && targetAnims.All(x => x.Owner == owner))
                            {
                                var srcAnims = owner.LightAnims.Cast<ISceneAnimationObject>().ToList();
                                var newLightAnims = createReorderedList(srcAnims, targetAnims).Cast<LightAnimation>().ToList();
                                TheApp.CommandManager.Execute(DocumentManager.CreateLightAnimationsEditCommand(owner, newLightAnims));
                            }
                            break;
                        }
                    case GuiObjectID.FogAnimation:
                        {
                            var targetAnims = guiObjects.OfType<ISceneAnimationObject>().ToList();
                            if ((targetAnims.Count() == guiObjects.Length) && targetAnims.All(x => x.Owner == owner))
                            {
                                var srcAnims = owner.FogAnims.Cast<ISceneAnimationObject>().ToList();
                                var newFogAnims = createReorderedList(srcAnims, targetAnims).Cast<FogAnimation>().ToList();
                                TheApp.CommandManager.Execute(DocumentManager.CreateFogAnimationsEditCommand(owner, newFogAnims));
                            }
                            break;
                        }
                }
            }
        }

        protected override bool CanMultiSelect(TreeNode node, TreeNode selectedNode = null)
        {
            selectedNode = selectedNode ?? SelectedNode;
            if ((selectedNode != null) && (node != null))
            {
                if (selectedNode.Tag is AnimationDocument && node.Tag is AnimationDocument)
                {
                    return true;
                }
                else if (selectedNode.Tag is Model && node.Tag is Model)
                {
                    return true;
                }
                else if (selectedNode.Tag is Texture && node.Tag is Texture)
                {
                    return true;
                }
                else if (selectedNode.Tag is AnimationSet && node.Tag is AnimationSet)
                {
                    return true;
                }
            }
            return false;
        }

        protected override bool KeepMultiSelectByClick(TreeNode treenode, MouseEventArgs e)
        {
            var node = treenode as ViewItem;
            if ((node == null) ||
                !CanMultiSelect(node) ||
                //(node.Tag.GetType() != SelectedNode.Tag.GetType()) ||
                !SelectedNodes.Contains(node))
            {
                return false;
            }

            var hitTestInfo = HitTest(e.Location);
            if (node.Tag is Model)
            {
                if (e.Location.X < _previewImageXOfModel) { return false; }
                if ((e.Location.X < _previewImageXOfModel + previewOn_.Width)) { return true; }
                if (e.Location.X > node.DocIconImageX && (e.Location.X < node.DocIconImageX + ShowInObjViewOn.Width)) { return true; }
            }
            else if (node.Tag is AnimationDocument)
            {
                if (node.Parent?.Tag is AnimationSet)
                {
                    if (_pauseImageXOfAnimationUnderModel <= e.Location.X && e.Location.X < _pauseImageXOfAnimationUnderModel + pauseMark_.Width) { return true; }
                    if (_previewImageXOfAnimationUnderModel <= e.Location.X && e.Location.X < _previewImageXOfAnimationUnderModel + previewOn_.Width) { return true; }
                }
            }
            else if (node.Tag is Texture)
            {
                // ファイルビューからテクスチャを複数同時にドラッグできる
                return true;
            }

            return false;
        }

        public DragDropEffects GetDragDropEffect()
        {
            switch (Control.ModifierKeys)
            {
                case Keys.None:
                    //ShowMoveDialog = true;
                    return DragDropEffects.Move;
                case Keys.Shift:
                    //ShowMoveDialog = false;
                    return DragDropEffects.Move;
                case Keys.Control:
                    return DragDropEffects.Copy;
                default:
                    return DragDropEffects.None;
            }
        }

        protected override void OnItemDrag(ItemDragEventArgs e)
        {
            base.OnItemDrag(e);

            ViewItem node = e.Item as ViewItem;
            if (node != null)
            {
                node.OnItemDrag(this, e);
            }
        }

        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);

            // キー押下中にフォーカスを失うと、キーが離されたことを検知できないので
            // フォーカスロスト時に複数選択モードを強制解除する。
            if (multiSelection != MultiSelection.Disable)
            {
                // 複数選択モードを解除して、カーソルをデフォルトに戻す。
                multiSelection = MultiSelection.Disable;
                Cursor = Cursors.Default;
            }

            // マウス座標を空に。
            lastMouseLocation = null;
        }

        protected override void OnKeyDown(KeyEventArgs e)
        {
            base.OnKeyDown(e);

            // Control か Shift が押下されていれば、複数選択モードに切り替える。
            if (e.Modifiers.HasFlag(Keys.Control) || e.Modifiers.HasFlag(Keys.Shift))
            {
                multiSelection = MultiSelection.Ready;

                // 以下理由により、ここでのカーソル変更が必要。
                // - キー押下直後にマウスが移動されるとは限らない
                // - Redo (Control+Y) の Y キー押下時は、ツリービューの構成がかわっている可能性がある
                //
                // ただし、Curosrs.Default 以外への変更はここでは行わない。
                // Cursors.No に変更することによってユーザーが Undo/Redo できないと勘違いすることを避けるため。
                var clientMousePos = PointToClient(MousePosition);
                var node = GetNodeAt(clientMousePos.X, clientMousePos.Y) as ViewItem;
                var cursor = ((node == null) || CanMultiSelect(node)) ? Cursors.Default : Cursors.No;
                if (cursor == Cursors.Default)
                {
                    Cursor = cursor;
                }
            }
        }

        protected override void OnKeyUp(KeyEventArgs e)
        {
            base.OnKeyUp(e);

            if (multiSelection != MultiSelection.Disable)
            {
                // Control か Shift が押下されていれば、複数選択モードを継続。
                if (e.Modifiers.HasFlag(Keys.Control) || e.Modifiers.HasFlag(Keys.Shift))
                {
                    // 以下理由により、ここでのカーソル変更が必要。
                    // - キー解除直後にマウスが移動されるとは限らない
                    // - Undo (Control+Z) の Z キー解除時は、ツリービューの構成が変わっている可能性がある
                    //
                    // MultiSelection.Enable ではない場合には、Curosrs.Default 以外への変更はここでは行わない。
                    // Cursors.No に変更することによってユーザーが Undo/Redo できないと勘違いすることを避けるため。
                    var clientMousePos = PointToClient(MousePosition);
                    var node = GetNodeAt(clientMousePos.X, clientMousePos.Y) as ViewItem;
                    var cursor = ((node == null) || CanMultiSelect(node)) ? Cursors.Default : Cursors.No;
                    if ((multiSelection == MultiSelection.Enable) || (cursor == Cursors.Default))
                    {
                        Cursor = cursor;
                    }
                }
                else
                {
                    // 複数選択モードが解除されたので、カーソルをデフォルトに戻す。
                    multiSelection = MultiSelection.Disable;
                    Cursor = Cursors.Default;
                }
            }
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            // マウス座標を空に。
            lastMouseLocation = null;
        }

        // ViewItemのドラッグ＆ドロップの実装
        //	ラベル右側も範囲に入れる
        //	ラベル横のアイコン等をオーナードローで書いているので、WinFormが参照するドラッグ開始位置とズレるためこのように実装しています
        private bool isDragging = false;
        private bool isFiredItemDrag = false;
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            isDragging = true;
            isFiredItemDrag = false;

            if (multiSelection != MultiSelection.Disable)
            {
                var clientMousePos = PointToClient(MousePosition);
                var node = GetNodeAt(clientMousePos.X, clientMousePos.Y) as ViewItem;
                Cursor = ((node == null) || CanMultiSelect(node)) ? Cursors.Default : Cursors.No;
            }
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            // マウスが移動したかどうか。
            // ツリービューのノードが再構築された場合になぜか OnMouseMove() が呼ばれるので
            // 本当にマウスが移動したかどうかをチェックする。
            var isMoved = !lastMouseLocation.HasValue || (lastMouseLocation.Value != e.Location);

            // lastMouseLocation への代入を IDisposable 継承クラスの Dispose() に記述し、
            // using で OnMouseMove() を抜けるときに更新したほうがよいが
            // 現状では以降の処理で lastMouseLocation を参照することはないので、とりあえずこのまま。
            lastMouseLocation = e.Location;

            if (isDragging && (isFiredItemDrag == false))
            {
                if (e.Button == MouseButtons.Left)
                {
                    var node = GetNodeAt(e.X, e.Y) as ViewItem;
                    if (node != null)
                    {
                        var info = HitTest(e.X, e.Y);
                        if (info.Location == TreeViewHitTestLocations.Label
                            || info.Location == TreeViewHitTestLocations.RightOfLabel
                            || info.Location == TreeViewHitTestLocations.Image)
                        {
                            isFiredItemDrag = true;
                            OnItemDrag(new ItemDragEventArgs(e.Button, node));
                        }
                    }
                }
            }
            else if (DesignMode == false && e.Button == MouseButtons.None)
            {
                var node = GetNodeAt(e.X, e.Y) as ViewItem;
                if (node != null)
                {
                    // 複数選択モードによってカーソルを変える。
                    switch (multiSelection)
                    {
                        case MultiSelection.Ready:
                            if (isMoved)
                            {
                                // 選択モードを有効にする。
                                multiSelection = MultiSelection.Enable;
                                goto case MultiSelection.Enable;
                            }
                            break;
                        case MultiSelection.Enable:
                            Cursor = CanMultiSelect(node) ? Cursors.Default : Cursors.No;
                            break;
                        default:
                            Cursor = Cursors.Default;
                            break;
                    }

                    var info = HitTest(e.X, e.Y);

                    var newMessage = "";
                    if (info.Location == TreeViewHitTestLocations.Label
                        || info.Location == TreeViewHitTestLocations.Image)
                    {
                        var lx = e.X;// - node.Bounds.Left;
                        var ly = e.Y - node.Bounds.Top;

                        // モデルファイル
                        if (node.Tag is Model)
                        {
                            if (((Model)node.Tag).IsSendAttached == false && lx >= _previewImageXOfModel && lx <= _previewImageXOfModel + previewOn_.Width)
                            {
                                // プレビュー対象
                                newMessage = Strings.FileTreeView_OnMouseMove_ToolTipPreviewModel;
                            }
                            else if (lx >= node.DocIconImageX && lx <= node.DocIconImageX + ShowInObjViewOn.Width)
                            {
                                // エディタ上で表示
                                newMessage = Strings.FileTreeView_OnMouseMove_ToolTipObjectView;
                            }

                        }
                        else if (node.Tag is AnimationSet) // アニメーションセット
                        {
                            if (info.Location == TreeViewHitTestLocations.Image)
                            {
                                // プレビュー対象
                                newMessage = Strings.FileTreeView_OnMouseMove_ToolTipPreviewAnimationSet;
                            }
                        }
                        else if (node.Tag is AnimationDocument) // アニメーション
                        {
                            var animSet = node.Parent.Tag as AnimationSet;
                            if (animSet != null)
                            {
                                var model = node.Parent.Parent.Tag as Model;
                                if (model != null)
                                {
                                    if (lx >= _previewImageXOfAnimationUnderModel && lx <= _previewImageXOfAnimationUnderModel + previewOn_.Width)
                                    {
                                        // プレビュー対象
                                        newMessage = Strings.FileTreeView_OnMouseMove_ToolTipPreviewAnimation;
                                    }
                                    else if (lx >= _pauseImageXOfAnimationUnderModel && lx <= _pauseImageXOfAnimationUnderModel + previewOn_.Width)
                                    {
                                        // プレビュー対象
                                        newMessage = Strings.FileTreeView_OnMouseMove_ToolTipPauseAnimation;
                                    }
                                }
                            }
                        }

                        // 変更済みマークなどのツールチップ
                        if (string.IsNullOrEmpty(newMessage)
                            && (node.DocIconImageX - node.SmallIconImageX > 2)
                            && (lx >= node.SmallIconImageX && lx <= node.DocIconImageX)
                            )
                        {
                            var guiObject = node.Tag as GuiObject;
                			var doc = node.Tag as Document;
                            if (guiObject != null)
                            {
                                if (ly < starMark_.Height)
                                {
                                    if (lx < node.SmallIconImageX + starMark_.Width)
                                    {
                                        if (guiObject.IsModifiedObject || (doc != null && doc.ContentsModified))
                                        {
                                            newMessage = Strings.FileTreeView_OnMouseMove_Modified;
                                        }
                                    }
                                }
                                else if (doc != null)
                                {
                                    var smallIcons = new List<SmallIconName>();

                                    // アタッチ
                                    if (doc is Model)
                                    {
                                        if (((Model)doc).IsSendAttached)
                                        {
                                            smallIcons.Add(SmallIconName.Attach);
                                        }
                                        else if (doc.IsAttached)
                                        {
                                            smallIcons.Add(SmallIconName.Preview);
                                        }
                                    }
                                    else if (doc is ShaderDefinition)
                                    {
                                        var shaderDefinition = (ShaderDefinition)doc;
                                        if (shaderDefinition.FailedToBinarize)
                                        {
                                            smallIcons.Add(SmallIconName.Error);
                                        }

                                        if (shaderDefinition.IsAttached)
                                        {
                                            smallIcons.Add(SmallIconName.Attach);
                                        }
                                    }

                                    if (doc.OpenedFromStartUp)
                                    {
                                        smallIcons.Add(SmallIconName.Startup);
                                    }
                                    var ix = node.SmallIconImageX;
                                    foreach (var icon in smallIcons)
                                    {
                                        if (lx >= ix && lx < ix + attachedMark_.Width)
                                        {
                                            switch (icon)
                                            {
                                                case SmallIconName.Attach:
                                                    newMessage = Strings.FileTreeView_OnMouseMove_Attached;
                                                    break;
                                                case SmallIconName.Preview:
                                                    newMessage = Strings.FileTreeView_OnMouseMove_Preview;
                                                    break;
                                                case SmallIconName.Startup:
                                                    newMessage = Strings.FileTreeView_OnMouseMove_Startup;
                                                    break;
                                                case SmallIconName.Error:
                                                    newMessage = Strings.FileTreeView_OnMouseMove_ConvertFail;
                                                    break;
                                            }
                                            break;
                                        }
                                        ix += attachedMark_.Width;
                                    }
                                }
                            }
                        }
                    }

                    if (string.IsNullOrEmpty(newMessage))
                    {
                        oldHintText = "";
                        if (_HintTimer != null)
                        {
                            _HintTimer.Dispose();
                            _HintTimer = null;
                        }
                        if (_ttpHint != null)
                        {
                            _ttpHint.Hide(this);
                            _ttpHint.RemoveAll();
                        }
                    }
                    else if (oldHintText != newMessage)
                    {
                        var lx = e.X - node.Bounds.Left;
                        oldHintText = newMessage;

                        if (_HintTimer != null)
                        {
                            _HintTimer.Dispose();
                            _HintTimer = null;
                        }

                        if (_HintTimer == null)
                        {
                            _HintTimer = new Timer()
                            {
                                Interval = 500
                            };
                        }

                        _HintTimer.Tick += (sender, te) =>
                        {
                            try
                            {
                                _HintTimer.Dispose();
                                _HintTimer = null;

                                if (_ttpHint == null)
                                {
                                    _ttpHint = new HintToolTip {ShowAlways = true, };
                                    _ttpHint.Popup += (s, ev) => DebugConsole.WriteLine("Tootip Popup");
                                }
                                else
                                {
                                    _ttpHint.Hide(this);
                                    _ttpHint.RemoveAll();
                                }

                                Point pos = PointToClient(System.Windows.Forms.Cursor.Position);
                                int x = pos.X + 0;
                                int y = pos.Y + 24;

                                _ttpHint.Show(oldHintText, this, x, y);
//                                DebugConsole.WriteLine("Show tooltip:{0}, {1}", oldHintText, _ttpHint.Active);
                            }
                            catch (Exception /*exception*/)
                            {
                                // 握りつぶす
                                // DebugConsole.WriteLine("{0}", exception.ToString());
                            }
                        };

                        _HintTimer.Start();
                    }
                }
            }
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            // ラベル変更の確定をマウスで行うと、SelectedNode に null が入る現象の回避策。
            // 詳細は UpdateDocumentList() を参照。
            if ((SelectedNode == null) && (SelectedNodes != null) && SelectedNodes.Any())
            {
                SelectedNode = SelectedNodes.Last();
            }

            base.OnMouseUp(e);

            isDragging = false;
            isFiredItemDrag = false;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragEnter(DragEventArgs e)
        {
            base.OnDragEnter(e);

            var effect = DragDropEffects.None;

            // ファイルであり、モーダルでないかファイルダイアログが開いている時はドラッグＯＫ
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                if (!UIForm.IsModalState(TheApp.MainFrame) || DialogUtility.IsOpened)
                {
                    effect = DragDropEffects.Copy;
                }
            }

            e.Effect = effect;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragDrop(DragEventArgs e)
        {
            // メッセージボックスを出すことを考慮してアクティブ化しておく?
            TheApp.MainFrame.Activate();

            // ドラッグドロップ処理
            string[] files = (string[])e.Data.GetData(DataFormats.FileDrop);
            if (files != null)
            {
                Point pt = PointToClient(new Point(e.X, e.Y));
                ViewItem destination = GetNodeAt(pt) as ViewItem;

                AnimationSet hasAnimation = null;
                Model animationOwnerModel = null;

                if (destination != null)
                {
                    if (destination == sceneFolder && !ApplicationConfig.Setting.MainFrame.IsAutoAnimationBindMode)
                    {
                        hasAnimation = DocumentManager.DefaultSceneAnimationSet;
                    }
                    else if (destination.Tag is Model)
                    {
                        animationOwnerModel = ((Model)destination.Tag);
                    }
                    else
                    {
                        hasAnimation = destination.Tag as AnimationSet;
                    }
                }
                // 描画更新
                Update();

                // ファイルを開く
                if (DialogUtility.IsOpened)
                {
                    DocumentManager.LoadFromFileOrDirectory(files, hasAnimation, animationOwnerModel);
                }
                else
                {
                    OneShotIdleProcess.Execute(() => DocumentManager.LoadFromFileOrDirectory(files, hasAnimation, animationOwnerModel));
                }
            }
            else
            {
                Point pt = PointToClient(new Point(e.X, e.Y));
                ViewItem destination = GetNodeAt(pt) as ViewItem;
                if (destination != null)
                {
                    destination.OnDragDrop(this, e);
                }
            }
            base.OnDragDrop(e);
        }

        protected override void OnDragOver(DragEventArgs drgevent)
        {
            base.OnDragOver(drgevent);

            if (drgevent.Data.GetDataPresent(DataFormats.FileDrop))
            {
                // 何もしない
            }
            else
            {
                drgevent.Effect = DragDropEffects.None;
                Point pt = PointToClient(new Point(drgevent.X, drgevent.Y));
                ViewItem destination = GetNodeAt(pt) as ViewItem;
                if (destination != null)
                {
                    destination.OnDragOver(this, drgevent);
                }
            }
        }

        protected override void OnContextMenuPopup(ContextMenuPopupEventArgs e)
        {
            UIContextMenuStrip menu = null;
            if (SelectedNode.Tag is FileViewFolder)
            {
                menu = TheApp.MainFrame.CreateFileViewFolderMenu((FileViewFolder)SelectedNode.Tag);
            }
            else if (SelectedNodes.Count >= 2)
            {
                // 複数選択の場合
                var active = (GuiObject)SelectedNode.Tag;
                menu = TheApp.MainFrame.CreateMultiSelectMenu();
            }
            else if (SelectedNode.Tag is GuiObject)
            {
                GuiObject guiObject = (GuiObject)SelectedNode.Tag;
                switch (guiObject.ObjectID)
                {
                    case GuiObjectID.Model:
                        menu = TheApp.MainFrame.CreateModelMenu(guiObject as Model);
                        break;
                    case GuiObjectID.Texture:
                        Debug.Assert(guiObject is Texture);
                        menu = TheApp.MainFrame.CreateTextureMenu(null, (guiObject as Document).IsTemporary);
                        break;
                    case GuiObjectID.ShaderDefinition:
                        menu = TheApp.MainFrame.CreateShaderDefinitionMenu();
                        break;
                    case GuiObjectID.Project:
                        menu = TheApp.MainFrame.CreateProjectDocumentMenu();
                        break;
                    case GuiObjectID.SceneAnimation:
                        if (SelectedNode.Parent == animationFolder)
                        {
                            menu = TheApp.MainFrame.CreateSceneAnimationMenu();
                        }
                        else
                        {
                            bool underAnimationSet = SelectedNode.Parent != null && SelectedNode.Parent.Tag is GuiObject && ((GuiObject)SelectedNode.Parent.Tag).ObjectID == GuiObjectID.AnimationSet;
                            menu = TheApp.MainFrame.CreateSceneAnimationMenuUnderScene(underAnimationSet);
                        }
                        break;
                    case GuiObjectID.AnimationSet:
                        if (SelectedNode.Parent != null && SelectedNode.Parent.Tag is Model)
                        {
                            menu = TheApp.MainFrame.CreateAnimationSetUnderModelMenu(SelectedNode.Parent.Tag as Model, guiObject as AnimationSet);
                        }
                        else
                        {
                            menu = TheApp.MainFrame.CreateAnimationSetUnderSceneMenu(SelectedNode.Parent.Tag as Model, guiObject as AnimationSet);
                        }
                        break;
                    case GuiObjectID.CameraAnimation:
                    case GuiObjectID.FogAnimation:
                    case GuiObjectID.LightAnimation:
                        menu = TheApp.MainFrame.CreateSceneContentMenu();
                        break;
                    case GuiObjectID.SeparateMaterial:
                        menu = TheApp.MainFrame.CreateSeparateMaterialMenu();
                        break;
                    default:
                        if (guiObject is AnimationDocument)
                        {
                            if (SelectedNode.Parent == animationFolder)
                            {
                                menu = TheApp.MainFrame.CreateAnimationMenu(guiObject);
                            }
                            else
                            {
                                menu = TheApp.MainFrame.CreateAnimationUnderModel(guiObject,
                                    SelectedNode.Parent != null && SelectedNode.Parent.Tag is GuiObject && ((GuiObject)SelectedNode.Parent.Tag).ObjectID == GuiObjectID.AnimationSet);
                            }
                        }
                        else
                        {
                            menu = TheApp.MainFrame.CreateDocumentMenu();
                        }
                        break;
                }
            }

            if (menu != null)
            {
                TheApp.MainFrame.ShowFileTreeViewContextMenu(menu, PointToScreen(e.Location));
            }
            //base.OnContextMenuPopup(e);
        }

        public void HideToolTip()
        {
            if (_ttpHint != null)
            {
                _ttpHint.Hide(this);
                _ttpHint.RemoveAll();
                _ttpHint = null;
            }
        }

        private void InitializeImageList()
        {
            // 16+1の意味 : 行間を開ける
            ImageList = new ImageList()
            {
                ImageSize			= new Size(16, 16+1),
                ColorDepth			= ColorDepth.Depth32Bit,
                TransparentColor	= Color.Transparent
            };

            imageList_ = new ImageList()
            {
                ImageSize			= new Size(16, 16),
                ColorDepth			= ColorDepth.Depth32Bit,
                TransparentColor	= Color.Transparent
            };

            foreach (IconName name in Enum.GetValues(typeof(IconName)))
            {
                imageList_.Images.Add(name.ToString(), (Image)Resources.ResourceManager.GetObject("TreeView_" + name.ToString(), Resources.Culture));
            }
        }

        public void UpdateDocumentList(IEnumerable<DocumentPropertyChangedArgs> args = null)
        {
            if (args == null)
            {
                args = new DocumentPropertyChangedArgs[0];
            }

            ForceExpand = true;
            using (var lw = new LockWindowUpdate(this))
            {
                var currentScrollY = ScrollY;
                var topNode = TopNode;
                var topNodeParent = TopNode.Parent;

                LabelEdit = false;
                var activeNode = SelectedNode ?? project;
                var selectedNodes = SelectedNodes;
                TreeNode parentNode = null;
                int indexInParentNode = 0;
                if (activeNode?.Parent != null)
                {
                    parentNode = activeNode.Parent;
                    indexInParentNode = parentNode.Nodes.IndexOf(activeNode);
                }
                // LastDropTargetにはドロップ先のアニメーションセットまたはモデルが入る
                // 他で誤作動を起こさないためにすぐにクリアしておく
                var target = LastDropTarget;
                LastDropTarget = null;


                UpdateProject();
                UpdateScene();
                UpdateModel();
                UpdateMaterial();
                UpdateAnimation();
                UpdateTexture(args);
                UpdateShaderDefinition();

                // SelectedNode の更新時に描画メッセージが発行されることがある。
                // FileTreeView.OnDrawNode() では SelectedNodes を参照しているため、
                // SelectedNode の変更は SelectedNodes の変更後に行わなければならない。
                // そこで、下記のローカル変数に一時コピーしておき、最後に変更を確定する。
                TreeNode newSelectedNode = SelectedNode;
                List<TreeNode> newSelectedNodes = SelectedNodes;

                if (activeNode != null && (activeNode != SelectedNode || target != null))
                {
                    newSelectedNode = null;
                    newSelectedNodes = null;

                    var activeObject = activeNode?.Tag as GuiObject;
                    var selectedObjects = selectedNodes?.Select(x => x.Tag as GuiObject)
                        .Where(x => x != null).ToArray();
                    var allNodeObjectPairs = Descendants(Nodes).Where(x => x.Tag is GuiObject).Select(x => new Tuple<TreeNode, GuiObject>(x, (GuiObject)x.Tag)).ToArray();

                    if (activeNode.TreeView == this && target == null)
                    {
                        newSelectedNode = activeNode;
                        newSelectedNodes = selectedNodes;
                    }
                    else if (target != null)
                    {
                        if (activeNode.Tag is AnimationDocument)
                        {
                            newSelectedNode =
                                allNodeObjectPairs.FirstOrDefault(
                                    x => x.Item2 == activeObject && target == x.Item1.Parent?.Tag as GuiObject)?.Item1;

                            newSelectedNodes =
                                selectedObjects?.Select(
                                    x =>
                                    allNodeObjectPairs.FirstOrDefault(
                                        y => y.Item2 == x && target == y.Item1.Parent?.Tag as GuiObject)?.Item1)
                                    .Where(x => x != null).Distinct().ToList();
                        }
                        else if (activeNode.Tag is AnimationSet)
                        {
                            // tag の名前を比較する
                            newSelectedNode =
                                allNodeObjectPairs.FirstOrDefault(
                                    x =>
                                    x.Item2.ObjectID == activeObject.ObjectID && x.Item2.Name == activeObject.Name
                                    && target == x.Item1.Parent?.Tag as GuiObject)?.Item1;
                            newSelectedNodes =
                                selectedObjects?.Select(
                                    obj =>
                                    allNodeObjectPairs.FirstOrDefault(
                                        x =>
                                        x.Item2.ObjectID == obj.ObjectID && x.Item2.Name == obj.Name
                                        && target == x.Item1.Parent?.Tag as GuiObject)?.Item1)
                                    .Where(x => x != null).Distinct().ToList();
                        }
                        else
                        {
                            Debug.Assert(false);
                        }
                    }
                    else
                    {
                        // 選択状態をなるべく元と同じにする
                        if (activeObject != null)
                        {
                            // ノードのtag自体を比較する
                            newSelectedNode =
                                allNodeObjectPairs
                                    .FirstOrDefault(
                                        x =>
                                        x.Item2 == activeObject && x.Item1.Parent == parentNode)?.Item1;

                            if (newSelectedNode == null)
                            {
                                // ドキュメントがリロードされた時のためファイルパスを比較する
                                newSelectedNode =
                                    allNodeObjectPairs.FirstOrDefault(
                                        x =>
                                        x.Item2.ObjectID == activeObject.ObjectID
                                        && x.Item2 is Document && activeObject is Document
                                        && ((Document)x.Item2).FilePath == ((Document)activeObject).FilePath
                                        && x.Item1.Parent == parentNode)?.Item1;
                            }
                        }

                        if (selectedObjects != null && selectedObjects.Any())
                        {
                            newSelectedNodes =
                                selectedObjects.Select(
                                    obj =>
                                    allNodeObjectPairs.FirstOrDefault(
                                        x => x.Item2 == obj && x.Item1.Parent == parentNode)?.Item1)
                                    .Where(x => x != null).Distinct().ToList();
                            if (!newSelectedNodes.Any())
                            {
                                // ドキュメントがリロードされた時のためファイルパスを比較する
                                newSelectedNodes =
                                    selectedObjects.Select(
                                        obj =>
                                        allNodeObjectPairs.FirstOrDefault(
                                            x =>
                                            x.Item2.ObjectID == obj.ObjectID && x.Item2 is Document && obj is Document
                                            && ((Document)x.Item2).FilePath == ((Document)obj).FilePath
                                            && x.Item1.Parent == parentNode)?.Item1)
                                        .Where(x => x != null).Distinct().ToList();
                            }
                        }

                        if (newSelectedNode == null)
                        {
                            //
                            if (parentNode != null && parentNode.TreeView == this)
                            {
                                if (indexInParentNode < parentNode.Nodes.Count)
                                {
                                    // 同じ位置にする
                                    newSelectedNode = parentNode.Nodes[indexInParentNode];
                                }
                                else
                                {
                                    // 親にする
                                    newSelectedNode = parentNode;
                                }
                            }
                        }
                    }
                }

                if (newSelectedNode == null)
                {
                    newSelectedNode = project;
                }

                if (newSelectedNodes == null)
                {
                    newSelectedNodes = new List<TreeNode>() { newSelectedNode };
                }
                else if (!newSelectedNodes.Contains(newSelectedNode))
                {
                    newSelectedNodes.Insert(0, newSelectedNode);
                }

                // SelectedNode と SelectedNodes の更新を確定。
                // SelectedNode の更新時に描画メッセージが発行されることがある。
                // FileTreeView.OnDrawNode() では SelectedNodes を参照しているため、
                // SelectedNode の変更は SelectedNodes の変更後に行わなければならない。
                //
                // ※ラベル変更の確定をマウスで行うと、AfterLabelEdit 後のマウスクリックが優先されるため
                // AfterLabelEditHandler での SelectedNode 変更が拒否される。
                // さらに、AfterLabelEditHandler ではラベル変更後のソートのためにノードの抜き差しを行っており、
                // ノードを取り除いたときに SelectedNode が null になるため、ここでの SelectedNodes と一致しなくなる。
                // そこで、ここでの null は許容し、OnMouseUp() 側で SelectedNode に SelectedNodes.Last() を代入することでこの問題を回避している。
                SelectedNodes = newSelectedNodes;
                SelectedNode = newSelectedNode;

                // スクロール位置を維持するためにTopNodeを更新
                TreeNode newTopNode = null;
                if (topNode != null)
                {
                    newTopNode = this.Descendants(this.Nodes).FirstOrDefault(x => x == topNode);
                    if (newTopNode == null)
                    {
                        var topObject = topNode.Tag as GuiObject;
                        if (topObject != null)
                        {
                            // 前のTopNodeが存在しない場合は同一ファイルを参照しているノードを探す
                            newTopNode = this.Descendants(this.Nodes).FirstOrDefault(
                                x =>
                                    {
                                        var tag = x.Tag as GuiObject;
                                        return tag != null && tag.ObjectID == topObject.ObjectID && tag is Document
                                               && topObject is Document
                                               && (tag as Document).FilePath == (topObject as Document).FilePath
                                               && x.Parent == topNodeParent;
                                    });
                        }
                    }
                }

                if (newTopNode != null)
                {
                    TopNode = newTopNode;
                }
                else
                {
                    // TopNodeが見つからない場合はスクロール位置を復元（旧方式。ずれる場合あり）
                    ScrollY = currentScrollY;
                }
            }
            ForceExpand = false;
        }

        public enum FileViewFolder
        {
            Scene,
            Model,
            Material,
            Animation,
            Texture,
            ShaderDefinition,
        }

        private void CreateFolderNodes()
        {
            ShowRootLines = false;
            project = new ViewItem(DocumentManager.ProjectDocument.Name)
            {
                ImageKey = IconName.Doc_Project.ToString(),
                SelectedImageKey = IconName.Doc_Project.ToString(),
                Tag = DocumentManager.ProjectDocument,
            };

            Nodes.Add(project);
            TopNode = project;

            sceneFolder = new ViewItem()
            {
                ImageKey = IconName.Folder_Scene.ToString(),
                SelectedImageKey = IconName.Folder_Scene.ToString(),
                Tag = FileViewFolder.Scene,
            };
            sceneFolder.DragOver += (s, e) => AnimationDragOver(e, DocumentManager.DefaultSceneAnimationSet, true, null);
            sceneFolder.DragDrop += (s, e) => AnimationDragDrop(e, DocumentManager.DefaultSceneAnimationSet, true);
            project.Nodes.Add(sceneFolder);

            modelFolder = new ViewItem()
            {
                ImageKey = IconName.Folder_Model.ToString(),
                SelectedImageKey = IconName.Folder_Model.ToString(),
                Tag = FileViewFolder.Model,
                //ToolTipText = IconName.Folder_Model.ToString()
            };
            project.Nodes.Add(modelFolder);

            materialFolder = new ViewItem()
            {
                ImageKey = IconName.Folder_SeparateMaterial.ToString(),
                SelectedImageKey = IconName.Folder_SeparateMaterial.ToString(),
                Tag = FileViewFolder.Material,
            };
            project.Nodes.Add(materialFolder);

            animationFolder = new ViewItem()
            {
                ImageKey = IconName.Folder_Animation.ToString(),
                SelectedImageKey = IconName.Folder_Animation.ToString(),
                Tag = FileViewFolder.Animation,
            };
            project.Nodes.Add(animationFolder);

            textureFolder = new ViewItem()
            {
                ImageKey = IconName.Folder_Texture.ToString(),
                SelectedImageKey = IconName.Folder_Texture.ToString(),
                Tag = FileViewFolder.Texture,
            };
            project.Nodes.Add(textureFolder);

            shaderDefinitionFolder = new ViewItem()
            {
                ImageKey = IconName.Folder_ShaderDefinition.ToString(),
                SelectedImageKey = IconName.Folder_ShaderDefinition.ToString(),
                Tag = FileViewFolder.ShaderDefinition,
            };
            project.Nodes.Add(shaderDefinitionFolder);

            project.ExpandAll();
        }

        private void UpdateProject()
        {
            project.Tag					= DocumentManager.ProjectDocument;
            project.UpdateText(((ProjectDocument)project.Tag).FileName);
            project.ImageKey			= IconName.Doc_Project.ToString();
            project.SelectedImageKey	= project.ImageKey;
        }

        private bool RemoveExcept(TreeNodeCollection nodes, IEnumerable<GuiObject> items)
        {
            bool removed = false;
            for (int i = nodes.Count - 1; i >= 0; i--)
            {
                if (!items.Contains(nodes[i].Tag))
                {
                    nodes.RemoveAt(i);
                    removed = true;
                }
            }
            return removed;
        }

        private ViewItem SearchNodeByTag(TreeNodeCollection nodes, GuiObject tag)
        {
            foreach (ViewItem node in nodes)
            {
                if (node.Tag == tag)
                {
                    return node;
                }
            }

            return null;
        }

        IEnumerable<TreeNode> Descendants(TreeNodeCollection nodes)
        {
            return from node in nodes.OfType<TreeNode>()
                   from descendant in Enumerable.Repeat<TreeNode>(node, 1).Concat(Descendants(node.Nodes))
                   select descendant;
        }

        private static string MakeAnimSetChildCountString(AnimationSet animationSet)
        {
            return string.Format("({0})", animationSet.Animations.Count());
        }

        private void UpdateScene()
        {
            sceneFolder.UpdateText(string.Format("{0}", res.Strings.FileViewFolderName_Scene));

            bool changed = RemoveExcept(sceneFolder.Nodes, DocumentManager.SceneAnimationSetsWithDefault);

            foreach (var animationSet in DocumentManager.SceneAnimationSetsWithDefault)
            {
                var childCountString = MakeAnimSetChildCountString(animationSet);

                var node = SearchNodeByTag(sceneFolder.Nodes, animationSet);
                if (node == null)
                {
                    node = new ViewItem(animationSet.Name + childCountString)
                    {
                        Tag = animationSet,
                    };
                    var animationSet1 = animationSet;
                    node.DragOver += (s, e) => AnimationDragOver(e, animationSet1, true, null);
                    node.DragDrop += (s, e) => AnimationDragDrop(e, animationSet1, true, null);
                    sceneFolder.Nodes.Add(node);
                    changed = true;
                }
                else if (node.Text != animationSet.Name + childCountString)
                {
                    sceneFolder.Nodes.Remove(node);
                    node.UpdateText(animationSet.Name + childCountString);
                    sceneFolder.Nodes.Add(node);
                }

                node.ImageKey			= "Doc_" + animationSet.ObjectID.ToString();
                node.IsPreview			= DocumentManager.PreviewSceneAnimSet == animationSet;
                node.SelectedImageKey	= node.ImageKey;

                changed |= UpdateSceneAnimationNodeUnderAnimationSet(node, animationSet);
            }

            if (changed)
            {
                sceneFolder.Expand();
            }
        }

        private bool UpdateSceneAnimationNodeUnderAnimationSet(ViewItem folder, AnimationSet animationSet)
        {
            var animations = DocumentManager.GetAnimations(animationSet.Animations).ToArray();
            bool changed = RemoveExcept(folder.Nodes, animations);

            // アニメーションの更新
            foreach (var animation in animations.OfType<SceneAnimation>())
            {
                bool nodeCreated = false;
                var node = SearchNodeByTag(folder.Nodes, animation);
                if (node == null)
                {
                    node = new ViewItem(animation.FileName)
                    {
                        Tag = animation,
                        DragEffects = DragDropEffects.Move | DragDropEffects.Copy,
                    };
                    folder.Nodes.Add(node);
                    node.DragOver += (s, e) => AnimationDragOver(e, animationSet, true, null);
                    node.DragDrop += (s, e) => AnimationDragDrop(e, animationSet, true);
                    changed = true;
                    nodeCreated = true;
                }
                else// if (node.Text != animation.FileName)
                {
                    folder.Nodes.Remove(node);
                    node.UpdateText(animation.FileName);
                    folder.Nodes.Add(node);
                }

                UpdateSceneAnimation(node, nodeCreated);

                node.ImageKey			= "Doc_" + animation.ObjectID.ToString();
                node.IsPreview          = !animation.InvisibleBinds.Contains(animationSet);
                node.SelectedImageKey	= node.ImageKey;
            }

            if (changed && !animationSet.SuppressOpenTreeNode)
            {
                folder.Expand();
            }
            animationSet.SuppressOpenTreeNode = false;

            return changed;
        }

        private void UpdateModel()
        {
            modelFolder.UpdateText(string.Format("{0}({1})", res.Strings.DocumentName_Model, DocumentManager.ModelCount));
            bool changed = RemoveExcept(modelFolder.Nodes, DocumentManager.Models);

            foreach (var model in DocumentManager.Models)
            {
                var node = (ViewItem)SearchNodeByTag(modelFolder.Nodes, model);
                if (node == null)
                {
                    node = new ViewItem(model.FileName)
                    {
                        Tag = model,
                    };
                    var model1 = model; // スコープを制限します。
                    node.DragOver += (s, e) => AnimationDragOver(e, model1.DefaultAnimationSet, false, model1);
                    node.DragOver += (s, e) => AnimationSetDragOver(e, model1);
                    node.DragDrop += (s, e) => AnimationDragDrop(e, model1.DefaultAnimationSet, false, model1);
                    node.DragDrop += (s, e) => AnimationSetDragDrop(e, model1);
                    modelFolder.Nodes.Add(node);
                    changed = true;
                }
                else if (node.Text != model.FileName)
                {
                    modelFolder.Nodes.Remove(node);
                    node.UpdateText(model.FileName);
                    modelFolder.Nodes.Add(node);
                }
                node.ImageKey = IconName.Doc_Model.ToString();
                node.SelectedImageKey = node.ImageKey;
                changed |= UpdateAnimationNodeUnderModel(node, model);
                node.IsPreview = model.IsVisible;
                node.IsShowInObjView = model.IsShowInObjView;
            }

            if (changed)
            {
                modelFolder.Expand();
            }
        }

        private void UpdateMaterial()
        {
            materialFolder.UpdateText(string.Format("{0}({1})", res.Strings.DocumentName_Material, DocumentManager.SeparateMaterialCount));
            bool changed = RemoveExcept(materialFolder.Nodes, DocumentManager.SeparateMaterials);

            foreach (var materialDoc in DocumentManager.SeparateMaterials)
            {
                var node = (ViewItem)SearchNodeByTag(materialFolder.Nodes, materialDoc);
                if (node == null)
                {
                    node = new ViewItem(materialDoc.FileName)
                    {
                        Tag = materialDoc,
                    };
                    materialFolder.Nodes.Add(node);
                    changed = true;
                }
                else if (node.Text != materialDoc.FileName)
                {
                    materialFolder.Nodes.Remove(node);
                    node.UpdateText(materialDoc.FileName);
                    materialFolder.Nodes.Add(node);
                }
                node.DragEffects = DragDropEffects.Copy | DragDropEffects.Move | DragDropEffects.Link;
                node.ForeColor = materialDoc.Materials.All(x => x.Referrers.Any() || x.ChildMaterials.Any()) ? SystemColors.ControlText : Color.Red;
                node.ImageKey = IconName.Doc_SeparateMaterial.ToString();
                node.SelectedImageKey = node.ImageKey;
            }

            if (changed)
            {
                materialFolder.Expand();
            }
        }

        private class TextureNodeInfo
        {
            public Texture Texture{ get; set; }
            public ViewItem Node{ get; set; }
            public string NodeName{ get; set; }
        }

        public class TextureImageChangedArgs : DocumentPropertyChangedArgs
        {
            public Texture Texture { get {	return (Texture)Target;	} }

            public TextureImageChangedArgs(GuiObject target, EditCommand command) : base(target, command){}
        }

        private void UpdateTexture(IEnumerable<DocumentPropertyChangedArgs> args)
        {
            // イメージリストをリフレッシュする
            RefreshImageList();

            // TextureImageChangedArgs なテクスチャを一旦削除する
            HashSet<string> changedKeys = new HashSet<string>(args.OfType<TextureImageChangedArgs>().Select(x => GetTextureKey(x.Texture)).Distinct());

            foreach(var dara in args.OfType<DocumentAddedOrRemovedArg>())
            {
                foreach(var addedDoc in dara.AddedDocuments)
                {
                    var addedTexture = addedDoc as Texture;
                    if (addedTexture != null)
                    {
                        imageList_.Images.RemoveByKey(GetTextureKey(addedTexture));
                    }
                }
            }

            bool isExpand = textureFolder.IsExpanded || textureFolder.Nodes.Count == 0;

            textureFolder.UpdateText(string.Format("{0}({1})", res.Strings.DocumentName_Texture, DocumentManager.TextureCount));
            RemoveExcept(textureFolder.Nodes, DocumentManager.Textures);

            var textureNodeInfos = new List<TextureNodeInfo>();

            HashSet<Texture> referencedTextures = new HashSet<Texture>();
            var refs = DocumentManager.DocumentsWithoutProject
                .Where(x => x.ObjectID == GuiObjectID.Model || x.ObjectID == GuiObjectID.SeparateMaterial || x.ObjectID == GuiObjectID.MaterialAnimation || x.ObjectID == GuiObjectID.TexturePatternAnimation)
                .SelectMany(x => x.ReferenceDocuments)
                .Where(x => x.ObjectID == GuiObjectID.Texture)
                .Select(x => (Texture)x);
            referencedTextures.UnionWith(refs);
            foreach(var texture in DocumentManager.Textures)
            {
                var displayName = texture.FileName;
                var node = (ViewItem)SearchNodeByTag(textureFolder.Nodes, texture);
                if (node == null)
                {
                    node = new ViewItem(displayName)
                    {
                        Tag = texture
                    };
                    textureFolder.Nodes.Add(node);
                }
                else if (node.Text != displayName)
                {
                    textureFolder.Nodes.Remove(node);
                    node.UpdateText(displayName);
                    textureFolder.Nodes.Add(node);
                }
                node.DragEffects = DragDropEffects.Copy | DragDropEffects.Move | DragDropEffects.Link;
                node.ForeColor = referencedTextures.Contains(texture) ? SystemColors.ControlText : Color.Red;

                string nodeName = GetTextureKey(texture);

                node.ImageKey			= imageList_.Images.ContainsKey(nodeName) ? nodeName : IconName.Doc_Texture.ToString();
                node.SelectedImageKey	= node.ImageKey;

                textureNodeInfos.Add(
                    new TextureNodeInfo()
                    {
                        Texture		= texture,
                        Node		= node,
                        NodeName	= nodeName
                    }
                );
            }

            // アイコン画像を更新する
            {
                // サイズでソートする
                var query = textureNodeInfos.OrderBy(x => x.Texture.Data.texture_info.size).ThenBy(x => x.Texture.Name).ToArray();

                foreach (var textureNodeInfo in query)
                {
                    UpdateImageList(textureNodeInfo.Texture, textureNodeInfo.Node, textureNodeInfo.NodeName, changedKeys);
                }
            }

            if (isExpand)
            {
                textureFolder.Expand();
            }
            else
            {
                textureFolder.Collapse();
            }
        }

        // 別スレッドで作成中のテクスチャ
        private readonly HashSet<string> creatingTextureNodeNames = new HashSet<string>();

        private void UpdateImageList(Texture texture, ViewItem node, string nodeName, HashSet<string> changedKeys)
        {
            lock (creatingTextureNodeNames)
            {
                // 登録されていないかつ、作っている最中でなければ
                if (((imageList_.Images.ContainsKey(nodeName) == false) &&
                    (creatingTextureNodeNames.Contains(nodeName) == false)) ||
                    changedKeys.Contains(nodeName))
                {
                    creatingTextureNodeNames.Add(nodeName);

                    BackgroundTaskManager.AddTask(
                        () =>
                        {
                            // テクスチャがロードされていなければスキップ
                            lock (creatingTextureNodeNames)
                            lock (texture.LoadedLock)
                            {
                                if (!texture.Loaded)
                                {
                                    creatingTextureNodeNames.Remove(nodeName);
                                    return;
                                }
                            }

                            var iconImage = CreateIconImage(texture);

                            const int timeoutCount = 10;

                            for (int i = 0; i != timeoutCount; ++i)
                            {
                                try
                                {
                                    BeginInvoke(
                                        new MethodInvoker(
                                            () =>
                                            {
                                                int index = imageList_.Images.IndexOfKey(nodeName);
                                                if (index >= 0)
                                                {
                                                    if (iconImage != null)
                                                    {
                                                        imageList_.Images[index] = iconImage;
                                                    }
                                                }
                                                else
                                                {
                                                    if (iconImage != null)
                                                    {
                                                        imageList_.Images.Add(nodeName, iconImage);
                                                    }
                                                }

                                                node.ImageKey = nodeName;
                                                node.SelectedImageKey = node.ImageKey;
                                                lock (creatingTextureNodeNames)
                                                {
                                                    creatingTextureNodeNames.Remove(nodeName);
                                                }
                                            }
                                        )
                                    ).AsyncWaitHandle.WaitOne();

                                    break;
                                }
                                catch (Exception)
                                {
                                    DebugConsole.WriteLine("RETRY -- FileTreeView.UpdateImageList() : {0}", i);
                                    Thread.Sleep(1 + i * 10);
                                }
                            }
                        }
                    );
                }
            }
        }

        private Bitmap CreateIconImage(Texture texture)
        {
            if (texture.Is1d)
            {
                var image = new Bitmap(16, 16);
                {
                    using (var g = Graphics.FromImage(image))
                    {
                        texture.DrawColorThumbnail(g, new Rectangle(0, 0, 16, 256)); // 高さを大きめに取らないと正しく描画してくれない
                    }
                }
                return image;
            }
            else
            {
                return new Bitmap(texture.GetColorThumbnailCopy(), 16, 16);
            }
        }

        // オーバーフローすることはないはず
        // 使いまわすように修正するのは難しくないけどやらない
        int totalTextureCount = 0;

        private Dictionary<Texture, string> textureToKey = new Dictionary<Texture, string>();
        private string GetTextureKey(Texture texture)
        {
            string key;
            if (!textureToKey.TryGetValue(texture, out key))
            {
                key = TexturePrefix + totalTextureCount;
                totalTextureCount++;
                textureToKey[texture] = key;
            }

            return key;
        }

        private void RefreshImageList()
        {
            foreach (var texture in textureToKey.Keys.Except(DocumentManager.Textures).ToArray())
            {
                textureToKey.Remove(texture);
            }

            // 削除する
            foreach (var key in imageList_.Images.Keys.OfType<string>()
                .Where(x => x.StartsWith(TexturePrefix)).Except(textureToKey.Values).ToArray())
            {
                imageList_.Images.RemoveByKey(key);
            }
        }

        private bool UpdateAnimationNodeUnderModel(ViewItem folder, Model model)
        {
            bool changed = RemoveExcept(folder.Nodes, model.AnimationSetsWithDefault);

            // モデルに含まれるアニメーションセットの更新
            foreach (var animationSet in model.AnimationSetsWithDefault)
            {
                var childCountString = MakeAnimSetChildCountString(animationSet);

                var node = SearchNodeByTag(folder.Nodes, animationSet);
                if (node == null)
                {
                    node = new ViewItem(animationSet.Name + childCountString)
                    {
                        Tag = animationSet,
                        DragEffects =
                            animationSet.IsDefaultAnimationSet ?			// 既定アニメーションセットはドラッグ
                                DragDropEffects.None :
                                DragDropEffects.Move | DragDropEffects.Copy,
                    };
                    var animationSet1 = animationSet;
                    node.DragOver += (s, e) => AnimationDragOver(e, animationSet1, false, model);
                    node.DragDrop += (s, e) => AnimationDragDrop(e, animationSet1, false);
                    folder.Nodes.Add(node);
                    changed = true;
                }
                else if (node.Text != animationSet.Name + childCountString)
                {
                    folder.Nodes.Remove(node);
                    node.UpdateText(animationSet.Name + childCountString);
                    folder.Nodes.Add(node);
                }

                node.ImageKey = "Doc_" + animationSet.ObjectID.ToString();
                node.IsPreview = model.PreviewAnimSet == animationSet;
                node.SelectedImageKey = node.ImageKey;
                changed |= UpdateAnimationNodeUnderAnimationSet(node, animationSet, model);
            }

            if (changed)
            {
                folder.Expand();
            }

            return changed;
        }

        private bool UpdateAnimationNodeUnderAnimationSet(ViewItem folder, AnimationSet animationSet, Model model)
        {
            bool changed = RemoveExcept(folder.Nodes, DocumentManager.GetAnimations(animationSet.Animations));

            foreach (var animation in DocumentManager.GetAnimations(animationSet.Animations))
            {
                var node = SearchNodeByTag(folder.Nodes, animation);
                if (node == null)
                {
                    node = new ViewItem(animation.FileName)
                    {
                        Tag = animation,
                        DragEffects = DragDropEffects.Move | DragDropEffects.Copy,
                    };
                    folder.Nodes.Add(node);
                    node.DragOver += (s, e) => AnimationDragOver(e, animationSet, false, model);
                    node.DragDrop += (s, e) => AnimationDragDrop(e, animationSet, false);
                    changed = true;
                }
                else// if (node.Text != animation.FileName)
                {
                    folder.Nodes.Remove(node);
                    node.UpdateText(animation.FileName);
                    folder.Nodes.Add(node);
                }

                node.ImageKey = "Doc_" + animation.ObjectID.ToString();
                node.SelectedImageKey = node.ImageKey;
                if (animation.ObjectID == GuiObjectID.TexturePatternAnimation)
                {
                    // テクスチャパターンアニメーションのテクスチャが参照切れのときに、ファイルビューのテクスチャパターンアニメーションを赤字で表示する
                    var tpAnim = animation as TexturePatternAnimation;
                    node.ForeColor = (tpAnim == null || !tpAnim.IsInvalidTextureReference()) ?
                        SystemColors.ControlText :
                        Color.Red;
                }
                else if (animation.ObjectID == GuiObjectID.MaterialAnimation)
                {
                    // テクスチャパターンアニメーションのテクスチャが参照切れのときに、ファイルビューのテクスチャパターンアニメーションを赤字で表示する
                    var matAnim = animation as MaterialAnimation;
                    node.ForeColor = (matAnim == null || !matAnim.IsInvalidTextureReference()) ?
                        SystemColors.ControlText :
                        Color.Red;
                }
            }

            // アニメーションセットを閉じた状態で作成したい場合を考慮
            if (changed && !animationSet.SuppressOpenTreeNode)
            {
                folder.Expand();
            }
            animationSet.SuppressOpenTreeNode = false;

            return changed;
        }

        /// <summary>
        /// アニメーションカテゴリフォルダの更新
        /// </summary>
        private void UpdateAnimation()
        {
            animationFolder.UpdateText(string.Format("{0}({1})", res.Strings.DocumentName_Animation, DocumentManager.AnimationCount));
            bool changed = RemoveExcept(animationFolder.Nodes, DocumentManager.Animations);

            foreach(var animation in DocumentManager.Animations)
            {
                bool nodeCreated = false;
                var node = SearchNodeByTag(animationFolder.Nodes, animation);
                if (node == null)
                {
                    node = new ViewItem(animation.FileName)
                    {
                        Tag = animation,
                    };
                    //if (animation.ObjectID != GuiObjectID.SceneAnimation)
                    {
                        node.DragEffects = DragDropEffects.Copy;
                    }
                    animationFolder.Nodes.Add(node);

                    nodeCreated = true;
                    changed = true;
                }
                else// if (node.Text != animation.FileName)
                {
                    animationFolder.Nodes.Remove(node);
                    node.UpdateText(animation.FileName);
                    animationFolder.Nodes.Add(node);
                }

                if (animation.ObjectID == GuiObjectID.SceneAnimation)
                {
                    UpdateSceneAnimation(node, nodeCreated);
                    node.ForeColor = DocumentManager.AllSceneAnimationSets.Any(x => x.Animations.Contains(new AnimationSetItem(animation.FileName, animation.FileLocation))) ?
                        SystemColors.ControlText :
                        Color.Red;
                }
                else if (animation.ObjectID == GuiObjectID.TexturePatternAnimation)
                {
                    // テクスチャパターンアニメーションのテクスチャが参照切れのときに、ファイルビューのテクスチャパターンアニメーションを赤字で表示する
                    var tpAnim = animation as TexturePatternAnimation;
                    node.ForeColor = (tpAnim == null || !tpAnim.IsInvalidTextureReference()) ?
                        SystemColors.ControlText :
                        Color.Red;
                }
                else if (animation.ObjectID == GuiObjectID.MaterialAnimation)
                {
                    // テクスチャパターンアニメーションのテクスチャが参照切れのときに、ファイルビューのテクスチャパターンアニメーションを赤字で表示する
                    var matAnim = animation as MaterialAnimation;
                    node.ForeColor = (matAnim == null || !matAnim.IsInvalidTextureReference()) ?
                        SystemColors.ControlText :
                        Color.Red;
                }
                else
                {
                    node.ForeColor = DocumentManager.HasModelAnimations.Any(x => x.Animations.Contains(new AnimationSetItem(animation.FileName, animation.FileLocation))) ?
                        SystemColors.ControlText :
                        Color.Red;
                }

                node.ImageKey			= "Doc_" + animation.ObjectID.ToString();
                node.SelectedImageKey	= node.ImageKey;
            }

            if (changed)
            {
                animationFolder.Expand();
            }
        }

        private void UpdateSceneAnimation(ViewItem sceneNode, bool nodeCreated)
        {
            SceneAnimation sceneAnimation = (SceneAnimation)sceneNode.Tag;
            bool changed = RemoveExcept(sceneNode.Nodes, sceneAnimation.ChildAnims);

            foreach (var animation in sceneAnimation.ChildAnims)
            {
                var node = SearchNodeByTag(sceneNode.Nodes, animation);
                if (node == null)
                {
                    node = new ViewItem(animation.Name)
                    {
                        Tag = animation,
                        DragEffects = DragDropEffects.Move
                    };
                    node.DragOver += (s, e) => AnimationDragOver(e, sceneAnimation);
                    node.DragDrop += (s, e) => AnimationDragDrop(e, sceneAnimation);
                    sceneNode.Nodes.Add(node);

                    if (!nodeCreated)
                    {
                        changed = true;
                    }
                }

                node.ImageKey			= "Doc_" + animation.ObjectID.ToString();
                node.UpdateText(animation.Name);
                node.SelectedImageKey	= node.ImageKey;
            }

            if (changed)
            {
                sceneNode.Expand();
            }
        }

        private void UpdateShaderDefinition()
        {
            var shaderDefinitions = DocumentManager.ShaderDefinitions;
            shaderDefinitionFolder.UpdateText(string.Format("{0}({1})", res.Strings.DocumentName_ShaderDefinition, shaderDefinitions.Count()));
            bool changed = RemoveExcept(shaderDefinitionFolder.Nodes, shaderDefinitions);

            HashSet<ShaderDefinition> referencedShaderDefinitions = new HashSet<ShaderDefinition>();
            var refs = DocumentManager.Models.Cast<Document>().Concat(DocumentManager.SeparateMaterials)
                .SelectMany(x => x.ReferenceDocuments)
                .Where(x => x.ObjectID == GuiObjectID.ShaderDefinition)
                .Select(x => (ShaderDefinition)x);
            referencedShaderDefinitions.UnionWith(refs);
            foreach (var shaderDefinition in shaderDefinitions)
            {
                var node = SearchNodeByTag(shaderDefinitionFolder.Nodes, shaderDefinition);
                if (node == null)
                {
                    node = new ViewItem(shaderDefinition.FileName)
                    {
                        Tag = shaderDefinition
                    };
                    shaderDefinitionFolder.Nodes.Add(node);

                    changed = true;
                }
                else if (node.Text != shaderDefinition.FileName)
                {
                    shaderDefinitionFolder.Nodes.Remove(node);
                    node.UpdateText(shaderDefinition.FileName);
                    shaderDefinitionFolder.Nodes.Add(node);
                }
                node.ForeColor			= shaderDefinition.IsAttached || referencedShaderDefinitions.Contains(shaderDefinition) ? SystemColors.ControlText : Color.Red;
                node.ImageKey			= "Doc_" + shaderDefinition.ObjectID.ToString();
                node.SelectedImageKey	= node.ImageKey;
            }

            if (changed)
            {
                shaderDefinitionFolder.Expand();
            }
        }

        /// <summary>
        /// ツリービューの並び順を定義します。
        /// </summary>
        public class ViewItemComparer : System.Collections.IComparer
        {
            int System.Collections.IComparer.Compare(object x, object y)
            {
                ViewItem itemx = x as ViewItem;
                ViewItem itemy = y as ViewItem;
                if (itemx == itemy)
                {
                    return 0;
                }
                if (itemx == null)
                {
                    return -1;
                }
                if (itemy == null)
                {
                    return 1;
                }

                return ViewItemCompare(itemx, itemy);
            }

            int ViewItemCompare(ViewItem x, ViewItem y)
            {
                if (x.Tag == null && y.Tag == null)
                {
                    return 0;
                }
                if (x.Tag == null)
                {
                    return -1;
                }
                if (y.Tag == null)
                {
                    return 1;
                }
                if (x.Tag is FileViewFolder && y.Tag is FileViewFolder)
                {
                    return ((FileViewFolder)x.Tag).CompareTo(y.Tag);
                }
                else if (x.Tag is FileViewFolder)
                {
                    return -1;
                }
                else if (y.Tag is FileViewFolder)
                {
                    return 1;
                }
                // 既定アニメーションセットは一番上
                else if ((x.Tag is AnimationSet) && (x.Tag as AnimationSet).IsDefaultAnimationSet)
                {
                    return -1;
                }
                else if ((y.Tag is AnimationSet) && (y.Tag as AnimationSet).IsDefaultAnimationSet)
                {
                    return 1;
                }

                // 1217: アニメーション再生順を制御できる機能を追加する
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1217

                // アニメーションセット内のアニメーションは名前文字列でソートを掛けない

                var xObj = (GuiObject)x.Tag;
                var yObj = (GuiObject)y.Tag;

                if ((x.Parent.Tag is AnimationSet) &&
                    GuiObject.IsAnimation(xObj.ObjectID) &&
                    GuiObject.IsAnimation(yObj.ObjectID))
                {
                    // ソートを掛けない
                    return 0;
                }
                else if ((x.Parent.Tag is SceneAnimation) &&
                    GuiObject.IsAnimation(xObj.ObjectID) &&
                    GuiObject.IsAnimation(yObj.ObjectID))
                {
                    // SceneAnimation.ChildAnims でソート。
                    var sceneAnim = (SceneAnimation)x.Parent.Tag;
                    var xindex = sceneAnim.ChildAnims.Select((a, i) => a.Equals(xObj) ? i : -1).Where(i => i != -1).FirstOrDefault();
                    var yindex = sceneAnim.ChildAnims.Select((a, i) => a.Equals(yObj) ? i : -1).Where(i => i != -1).FirstOrDefault();
                    return xindex.CompareTo(yindex);
                }
                else
                {
                    var result = xObj.ObjectID.CompareTo(yObj.ObjectID);
                    if (result != 0)
                    {
                        return result;
                    }

                    result = x.Text.CompareTo(y.Text);
                    if (result != 0)
                    {
                        return result;
                    }

                    if (xObj is AnimationDocument || xObj is Texture)
                    {
                        // インデックスでソート
                        int xindex, yindex;
                        DocumentManager.IndexInSameNames.TryGetValue(xObj, out xindex);
                        DocumentManager.IndexInSameNames.TryGetValue(yObj, out yindex);
                        return xindex.CompareTo(yindex);
                    }

                    return result;
                }
            }
        }

        private void SetSelectedNodeBySelectedObject(GuiObject obj)
        {
            ViewItem item = null;
            item = SearchNodeByTag(project.Nodes, obj);
            if (item != null)
            {
                SelectedNode = item;
                return;
            }
            item = SearchNodeByTag(modelFolder.Nodes, obj);
            if (item != null)
            {
                SelectedNode = item;
                return;
            }
            item = SearchNodeByTag(materialFolder.Nodes, obj);
            if (item != null)
            {
                SelectedNode = item;
                return;
            }
            item = SearchNodeByTag(animationFolder.Nodes, obj);
            if (item != null)
            {
                SelectedNode = item;
                return;
            }
            item = SearchNodeByTag(textureFolder.Nodes, obj);
            if (item != null)
            {
                SelectedNode = item;
                return;
            }
            item = SearchNodeByTag(shaderDefinitionFolder.Nodes, obj);
            if (item != null)
            {
                SelectedNode = item;
            }
        }

        /// <summary>
        /// 名前変更を開始します。
        /// </summary>
        private void RenameMenuCommand(MenuCommandArgs args)
        {
            if (LabelEdit)
            {
                return;
            }

            ViewItem item = SelectedNode as ViewItem;

            // ファイルツリービューの選択ノードとコンテキストメニューのターゲットが違う場合があるので、
            // その場合、コンテキストメニューのターゲットと同じものを
            // ファイルツリービューで選択してあげる。
            var document = App.AppContext.CurrentSelectedObject(args) as Document;

            var animSet = App.AppContext.CurrentSelectedObject(args) as AnimationSet;
            if (item.Tag != document && item.Tag != animSet)
            {
                if (document != null)
                {
                    SetSelectedNodeBySelectedObject(document);
                }
                else if (animSet != null)
                {
                    SetSelectedNodeBySelectedObject(animSet);
                }
                item = SelectedNode as ViewItem;
            }

            AnimationSet target = item == null? null: item.Tag as AnimationSet;
            if (args.RequireUpdate)
            {
                if (target != null)
                {
                    args.CommandUI.Enabled = true;
                }
                return;
            }

            // リネーム不可であれば何も行わない
            {
                var guiObject = item == null ? null : item.Tag as GuiObject;
                if ((guiObject != null) && (guiObject.Renamable == false))
                {
                    return;
                }
            }

            // アニメーションセットであれば子数部分を削除する
            if (item.Tag is AnimationSet)
            {
                var braceIndex = item.Text.IndexOf("(");
                Debug.Assert(braceIndex != -1);

                item.Text = item.Text.Substring(0, braceIndex);
            }

            LabelEdit = true;
            item.BeginEdit();
        }

        /// <summary>
        /// 名前変更を行います。
        /// </summary>
        private void AfterLabelEditHandler(object sender, NodeLabelEditEventArgs args)
        {
            if (LabelEdit == false)
            {
                return;
            }

            var item	= args.Node as ViewItem;

            if (item.Tag is AnimationSet)
            {
                AfterLabelEditHandler_AnimationSet(args, item, item.Tag as AnimationSet);
            }
            else if (item.Tag is Document)
            {
                AfterLabelEditHandler_Document(args, item, item.Tag as Document);
            }
            else if (item.Tag is ISceneAnimationObject)
            {
                AfterLabelEditHandler_SceneAnimationObject(args, item, item.Tag as GuiObject);
            }
            else
            {
                Debug.Assert(false);
            }
        }

        private void AfterLabelEditHandler_AnimationSet(NodeLabelEditEventArgs args, ViewItem item, AnimationSet target)
        {
            var name	= args.Label;

            // 変化なし
            if (string.IsNullOrWhiteSpace(name) ||
                (name == target.Name))
            {
                args.CancelEdit = true;
                LabelEdit = false;
                UpdateDocumentList();
                return;
            }

            // 空文字
            if (string.IsNullOrWhiteSpace(name))
            {
                UIMessageBox.Warning(res.Strings.AnimationSetCreateDialog_NameEmpty);
                args.CancelEdit = true;
                item.BeginEdit();
                return;
            }

            // 名前の重複チェック
            //Debug.Assert(item.Parent.Tag is Model);
            var parentModel = item.Parent.Tag as Model;
            List<AnimationSet> animationSets = parentModel != null ? parentModel.AnimationSets : DocumentManager.SceneAnimationSets;

            if (animationSets.Any(x => (x != target) && (x.Name == name)))
            {
                UIMessageBox.Warning(res.Strings.Rename_NameDuplicate, name);
                args.CancelEdit = true;
                item.BeginEdit();
                return;
            }

            args.CancelEdit = true;
            LabelEdit = false;
            if (parentModel != null)
            {
                foreach (var animation in DocumentManager.GetAnimations(target.Animations))
                {
                    double frame;
                    bool pause = animation.PauseFrames.TryGetValue(new KeyValuePair<object, string>(parentModel.ModelId, target.Name), out frame);
                    bool invisible = animation.Pause.InvisibleBinds.Contains(new KeyValuePair<object, string>(parentModel.ModelId, target.Name));
                    animation.SetPause(pause, frame, parentModel.ModelId, name, invisible);
                }
            }
            TheApp.CommandManager.Execute(DocumentManager.CreateAnimationSetRenameEditCommand(target, name));
        }

        private void AfterLabelEditHandler_Document(NodeLabelEditEventArgs args, ViewItem item, Document target)
        {
            // 入力文字列のチェック(パスに使用できる文字列かどうか？)
            if (!string.IsNullOrEmpty(args.Label) &&
                RegexMatch.Check(args.Label, "[0-9A-Za-z\\-\\._]+") == false)
            {
                UIMessageBox.Warning(string.Format(Strings.IO_Name_Invalid, args.Label));
                args.CancelEdit = true;
                LabelEdit = false;
                UpdateDocumentList();
                return;
            }

            var name	= Path.GetFileNameWithoutExtension(args.Label);
            var ext		= Path.GetExtension(args.Label);
            var srcExtName = target.FileDotExt;

            if (string.IsNullOrWhiteSpace(name) || // 空になった
                args.Label == target.FileName || // 何も変わらない
                (args.Label == target.Name && ext != srcExtName)) // 拡張子だけ消えかつ、末尾が現在の拡張子でない
            {
                args.CancelEdit = true;
                LabelEdit = false;
                UpdateDocumentList();
                return;
            }

            // たかだかアスキー/バイナリの変更
            if (srcExtName == ext ||
                (G3dPath.IsPath(ext) && G3dPath.IsPath(srcExtName) &&
                (PathUtility.MakeNoFormatIfName(srcExtName) == PathUtility.MakeNoFormatIfName(ext))))
            {
                ;
            }
            // 知らない拡張子の変更があった -> 名前分に含め、拡張子は元のまま
            else
            {
                name = name + ext;
                ext  = srcExtName;
            }

            // マテリアルアニメーションの接尾辞のチェック
            if (ApplicationConfig.Preset.SeparateMaterialAnimCreationMenu && target is MaterialAnimation)
            {
                var subType = MaterialAnimation.NameToType(name);
                var materialAnimation = (MaterialAnimation)target;
                if (subType != materialAnimation.MaterialAnimationSubType)
                {
                    var postFix = MaterialAnimation.PostFix(materialAnimation.MaterialAnimationSubType);
                    if (!string.IsNullOrEmpty(postFix))
                    {
                        UIMessageBox.Warning(
                            string.Format(
                            Strings.Name_PostFix,
                            MaterialAnimation.SubTypeString(materialAnimation.MaterialAnimationSubType),
                            postFix));
                    }
                    else
                    {
                        UIMessageBox.Warning(
                            string.Format(
                            Strings.Name_NotPostFix,
                            MaterialAnimation.SubTypeString(materialAnimation.MaterialAnimationSubType),
                            MaterialAnimation.PostFix(subType)));
                    }

                    args.CancelEdit = true;
                    LabelEdit = false;
                    UpdateDocumentList();
                    return;
                }
            }

            // 名前の重複チェック
            if (ObjectIDUtility.CanOpenSameNameFile(ext))
            {
                if (DocumentManager.Objects(target.ObjectID).OfType<Document>().Any(
                    x => x != target && x.FileName == name + ext && string.IsNullOrEmpty(x.FileLocation)))
                {
                    UIMessageBox.Warning(res.Strings.Rename_NameDuplicateEmptyPath, name + ext);
                    args.CancelEdit = true;
                    item.BeginEdit();
                    return;
                }

                // モデル以下で重複
                var duplication = (from model in DocumentManager.Models
                                   let fileName = DocumentSaver.DuplicatedName(model, target, name)
                                   where fileName != null
                                   select new { modelName = model.Name, fileName }).FirstOrDefault();
                if (duplication != null)
                {
                    UIMessageBox.Warning(res.Strings.Rename_NameDuplicateUnderModel, duplication.modelName, duplication.fileName);
                    args.CancelEdit = true;
                    item.BeginEdit();
                    return;
                }
            }
            else if (DocumentManager.Objects(target.ObjectID).Any(x => (x != target) && (x.Name == name)))
            {
                UIMessageBox.Warning(res.Strings.Rename_NameDuplicate, name);
                args.CancelEdit = true;
                item.BeginEdit();
                return;
            }

            var referenceDocuments = DocumentManager.DocumentsWithoutProject.Where(x => x.ReferenceDocuments.Contains(target)).ToArray();
            //  編集コマンド群を作る
            var commandSet = new EditCommandSet();
            {
                if (target is Model && !target.FileExsists)
                {
                    if (!UIMessageBox.YesNo(Strings.FileTreeView_AfterLabelEditHandler_WarnRenameNotExistModel, target.FileName))
                    {
                        args.CancelEdit = true;
                        LabelEdit = false;
                        UpdateDocumentList();
                        return;

                    }
                }
                // 参照されていたら、参照先も書き換えるかを問い合わせる
                if (referenceDocuments.Any())
                {
                    if (target is AnimationDocument)
                    {
                        // アニメーションは常に参照元（バインド元）を書き換える
                        foreach(var doc in referenceDocuments)
                        {
                            var command = doc.CreateReferenceObjectRenameCommand(target, target.Name, target.FileLocation, name, ext, string.Empty);
                            if (command != null)
                            {
                                commandSet.Add(command);
                            }
                        }
                    }
                    else
                    {
                        switch (UIMessageBox.YesNoCancel(Strings.IO_RenameReferencedFile, target.FileName))
                        {
                            case DialogResult.Yes:
                            {
                                if (target is Texture)
                                {
                                    foreach (var model in referenceDocuments.OfType<Model>())
                                    {
                                        // 名前を変更して重複する場合は禁止
                                        if (model.ReferenceTexturePaths.ContainsKey(name))
                                        {
                                            UIMessageBox.Error(
                                                Strings.FileTreeView_AfterLabelEditHandler_ModelReferSameNameTexture,
                                                model.Name,
                                                name
                                                );
                                            args.CancelEdit = true;
                                            item.BeginEdit();
                                            return;
                                        }

                                        // テクスチャのリネームは、予約サンプラで使われている場合はリネーム禁止
                                        foreach (var material in model.Materials)
                                        {
                                            foreach (var sampler in material.sampler_array.sampler)
                                            {
                                                if (sampler.tex_name == target.Name
                                                    && ApplicationConfig.Preset.FollowDccSamplerNameRule
                                                    && sampler.name.StartsWith("_")
                                                    && !string.IsNullOrEmpty(sampler.hint))
                                                {
                                                    UIMessageBox.Error(
                                                        Strings.FileTreeView_AfterLabelEditHandler_DccSamplerTextureCannotChange,
                                                        material.Name, sampler.name, sampler.tex_name
                                                        );
                                                    args.CancelEdit = true;
                                                    LabelEdit = false;
                                                    UpdateDocumentList();
                                                    return;
                                                }
                                            }
                                        }
                                    }

                                    foreach (var anim in referenceDocuments.OfType<TexturePatternAnimation>())
                                    {
                                        // 名前を変更して重複する場合は禁止
                                        if (anim.ReferenceTexturePaths.ContainsKey(name))
                                        {
                                            var animName = anim.Name;
                                            if (DocumentManager.Animations.OfType<TexturePatternAnimation>().Any(x => x.Name == anim.Name && x != anim))
                                            {
                                                var indexText = DocumentManager.GetSameNameIndexText(anim, false);
                                                if (!string.IsNullOrEmpty(indexText))
                                                {
                                                    animName += string.Format(" ({0} {1})", anim.FileName, indexText);
                                                }
                                                else
                                                {
                                                    animName += string.Format(" ({0})", anim.FileName);
                                                }
                                            }

                                            UIMessageBox.Error(
                                                Strings.FileTreeView_AfterLabelEditHandler_TexturePatternAnimReferSameNameTexture,
                                                animName,
                                                name
                                                );
                                            args.CancelEdit = true;
                                            item.BeginEdit();
                                            return;
                                        }
                                    }
                                }


                                // 参照元も書き換える
                                foreach(var doc in referenceDocuments)
                                {
                                    var command = doc.CreateReferenceObjectRenameCommand(target, target.Name, target.FileLocation, name, ext, string.Empty);
                                    if (command != null)
                                    {
                                        commandSet.Add(command);
                                        if (target is Texture)
                                        {
                                                if (doc is Model)
                                                {
                                                    commandSet.Add(MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(doc), GuiObjectID.Model, name, string.Empty));
                                                    commandSet.Add(new LazyCommand(() => MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(doc), GuiObjectID.Model)));
                                                }
                                                else if (doc is SeparateMaterial)
                                                {
                                                    commandSet.Add(MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(doc), GuiObjectID.SeparateMaterial, name, string.Empty));
                                                    commandSet.Add(new LazyCommand(() => MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(doc), GuiObjectID.SeparateMaterial)));
                                                }
                                                else if (doc is TexturePatternAnimation)
                                                {
                                                    commandSet.Add(TexturePatternAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(doc), name, string.Empty));
                                                    commandSet.Add(new LazyCommand(() => TexturePatternAnimationPatternPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(doc))));
                                                }
                                        }

                                    }
                                }

                                break;
                            }

                            case DialogResult.No:
                            {
                                // 参照先テクスチャの情報を消す
                                if (target is Texture)
                                {
                                    foreach (var doc in referenceDocuments)
                                    {
                                        if (doc is Model)
                                        {
                                            commandSet.Add(MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(doc), GuiObjectID.Model, target.Name, null));
                                        }
                                        else if (doc is SeparateMaterial)
                                        {
                                            commandSet.Add(MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(doc), GuiObjectID.SeparateMaterial, target.Name, null));
                                        }
                                        else if (doc is TexturePatternAnimation)
                                        {
                                            commandSet.Add(TexturePatternAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(doc), target.Name, null));
                                        }
                                    }
                                }
                                break;
                            }

                            case DialogResult.Cancel:
                            {
                                args.CancelEdit = true;
                                LabelEdit = false;
                                UpdateDocumentList();
                                return;
                            }

                            default:
                            {
                                Debug.Assert(false);
                                break;
                            }
                        }
                    }
                }

                // シーンのアニメーションセットを修正
                if (target.ObjectID == GuiObjectID.SceneAnimation)
                {
                    var animationSets = DocumentManager.SceneAnimationSets.Concat(Enumerable.Repeat(DocumentManager.DefaultSceneAnimationSet, 1));
                    foreach (var animationSet in animationSets)
                    {
                        int index = animationSet.Animations.FindIndex(x => x == new AnimationSetItem(target.FileName, target.FileLocation));
                        if (index != -1)
                        {
                            commandSet.Add(animationSet.CreateReferenceObjectRenameCommand(target, target.Name, target.FileLocation, name, ext, string.Empty));
                        }
                    }
                }

                args.CancelEdit = true;
                LabelEdit = false;

                // PrePostIO
                if (!string.IsNullOrEmpty(target.FilePath))
                {
                    var paths = new [] { Tuple.Create(target, target.FilePath) };
                    commandSet.canUndo += () => PrePostIO.ExecutePreOpenUndoRedo(paths, true);
                }

                commandSet.Add(DocumentManager.CreateDocumentRenameEditCommand(target, name, ext));
                commandSet.Add(DocumentManager.CreateProjectGuiObjectRenameEditCommand(target, name));
            }
            commandSet.OnPostEdit += (s, e) =>
            {
                foreach (var doc in referenceDocuments)
                {
                    if (doc is Model)
                    {
                        Viewer.LoadOrReloadModel.Send((Model)doc);
                    }
                    else if (doc is AnimationDocument)
                    {
                        ((AnimationDocument)doc).CheckAndDisConnect();
                        Viewer.LoadOrReloadAnimation.Send((AnimationDocument)doc);
                    }
                }
            };
            TheApp.CommandManager.Execute(commandSet);
        }

        private void AfterLabelEditHandler_SceneAnimationObject(NodeLabelEditEventArgs args, ViewItem item, GuiObject target)
        {
            Debug.Assert(target is ISceneAnimationObject);

            // 入力文字列のチェック(パスに使用できる文字列かどうか？)
            if (!string.IsNullOrEmpty(args.Label) &&
                RegexMatch.Check(args.Label, "[0-9A-Za-z\\-\\._]+") == false)
            {
                UIMessageBox.Warning(string.Format(Strings.IO_Name_Invalid, args.Label));
                args.CancelEdit = true;
                LabelEdit = false;
                UpdateDocumentList();
                return;
            }

            var name	= args.Label;

            // 変化なし
            if (string.IsNullOrWhiteSpace(name) ||
                (name == target.Name))
            {
                args.CancelEdit = true;
                LabelEdit = false;
                UpdateDocumentList();
                return;
            }

            // 名前の重複チェック
            if (DocumentManager.Objects(target.ObjectID).Any(x => (x != target) && (x.Name == name)))
            {
                UIMessageBox.Warning(res.Strings.Rename_NameDuplicate, name);
                args.CancelEdit = true;
                item.BeginEdit();
                return;
            }

            var owner = (target as ISceneAnimationObject).Owner;

            args.CancelEdit = true;
            LabelEdit = false;
            TheApp.CommandManager.Execute(DocumentManager.CreateSceneAnimationObjectRenameEditCommand(owner, target, name));
        }

        public GuiObject SelectedFileViewObject
        {
            get
            {
                if (SelectedNode != null)
                {
                    return SelectedNode.Tag as GuiObject;
                }

                return null;
            }
        }
        public IEnumerable<GuiObject> SelectedFileViewObjects
        {
            get
            {
                if (SelectedNodes != null && SelectedNodes.Any(x => x.Tag is GuiObject))
                {
                    return SelectedNodes.Select(x => x.Tag as GuiObject).Where(x => x != null);
                }
                else if ((SelectedNode?.Tag is GuiObject))
                {
                    return new[] { SelectedNode.Tag as GuiObject };
                }

                return null;
            }
        }


        public GuiObject SelectedFileViewObjectOwner
        {
            get
            {
                return SelectedNode != null && SelectedNode.Parent != null ?
                    (SelectedNode.Parent == sceneFolder ? DocumentManager.DefaultSceneAnimationSet: SelectedNode.Parent.Tag as GuiObject):
                    null;
            }
        }

        private static readonly Bitmap starMark_	= App.Properties.Resources.Control_Star;
        private static readonly Bitmap startUpMark_	= App.Properties.Resources.TreeView_StartUp;
        private static readonly Bitmap attachedMark_= App.Properties.Resources.TreeView_Attached;
        private static readonly Bitmap previewingMark_ = App.Properties.Resources.TreeView_Previewing;
        private static readonly Bitmap previewingErrorMark_ = App.Properties.Resources.TreeView_Previewing_Error;
        private static readonly Bitmap pauseMark_	= App.Properties.Resources.TreeView_AnimPause_On;
        private static readonly Bitmap playMark_ = App.Properties.Resources.TreeView_AnimPause_Off;
        private static readonly Bitmap pauseGrayMark_ = App.Properties.Resources.TreeView_AnimPause_On_Gray;
        private static readonly Bitmap playGrayMark_ = App.Properties.Resources.TreeView_AnimPause_Off_Gray;
        private static readonly Bitmap previewOn_ = App.Properties.Resources.TreeView_AnimPreviw_On;
        private static readonly Bitmap previewOff_	= App.Properties.Resources.TreeView_AnimPreviw_Off;
        private static readonly Bitmap AnimSetPreviewOn = App.Properties.Resources.TreeView_AnimSetPreview_On;
        private static readonly Bitmap AnimSetPreviewOff = App.Properties.Resources.TreeView_AnimSetPreview_Off;

        private static readonly Bitmap ShowInObjViewOn   = App.Properties.Resources.TreeView_ShowInObjView_On;
        private static readonly Bitmap ShowInObjViewOff  = App.Properties.Resources.TreeView_ShowInObjView_Off;

//        private int _docIconImageX = 0;
        private int _previewImageXOfModel = 0;
        private int _previewImageXOfSceneAnimationUnderAnimationSet = 0;
        private int _pauseImageXOfAnimationUnderModel = 0;
        private int _previewImageXOfAnimationUnderModel = 0;

        private static int BoundCentering(Rectangle bounds, Bitmap image)
        {
            return bounds.Top + (bounds.Height - image.Height) / 2;
        }

        protected override void OnDrawNode(DrawTreeNodeEventArgs e)
        {
            var node = e.Node as ViewItem;
            Debug.Assert(node != null);

            var bgBounds = node.Bounds;

            // この状態は描画しない
            if ((bgBounds.X == 0) && (bgBounds.Y == 0))
            {
                return;
            }

            var graphics = e.Graphics;
            var font     = node.NodeFont ?? Font;

            graphics.FillRectangle(SystemBrushes.Window, bgBounds);

            bgBounds.X -= ImageList.ImageSize.Width;

            var guiObject = node.Tag as GuiObject;
            if (guiObject != null)
            {
                var doc = guiObject as Document;

                if (node.Tag is Model)
                {
                    _previewImageXOfModel = bgBounds.X;

                    // モデルのプレビュー対象
                    if (!((Model)node.Tag).IsSendAttached)
                    {
                        var image = node.IsPreview ? previewOn_ : previewOff_;

                        graphics.DrawImage(image, new Point(bgBounds.Left, BoundCentering(bgBounds, image)));
                        bgBounds.X += image.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                    }
                }
                else if (node.Tag is AnimationSet)
                {
                    // アニメーションセットのプレビュー対象
                    var image = node.IsPreview ? AnimSetPreviewOn : AnimSetPreviewOff;

                    graphics.DrawImage(image, new Point(bgBounds.Left, BoundCentering(bgBounds, image)));
                    bgBounds.X += image.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                }
                else if (node.Tag is SceneAnimation)
                {
                    // バインド済みを意味する AnimationSet 下のノードがアイコン表示対象。
                    if (node.Parent?.Tag is AnimationSet)
                    {
                        _previewImageXOfSceneAnimationUnderAnimationSet = bgBounds.X;

                        // シーンアニメーションのプレビュー対象
                        var image = node.IsPreview ? previewOn_ : previewOff_;

                        graphics.DrawImage(image, new Point(bgBounds.Left, BoundCentering(bgBounds, image)));
                        bgBounds.X += image.Width + 2;  // +2 はテキストの選択ボックスとくっついてしまうので
                    }
                }
/*
                else if (node.Tag is Model && !((Model)node.Tag).IsSendAttached)
                {
                    var image = node.IsPreview ? previewOn_ : previewOff_;

                    graphics.DrawImage(image, new Point(bgBounds.Left, bgBounds.Top));
                    bgBounds.X += image.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                }
*/


                // アニメーションのプレビュー対象と一時停止
                {
                    var pause = node.Tag as AnimationDocument;
                    if (pause != null)
                    {
                        var animSet = node.Parent?.Tag as AnimationSet;
                        if (animSet != null)
                        {
                            var animSetNode = (node.Parent as ViewItem);
                            var isPreview = animSetNode != null && animSetNode.IsPreview;
                            var model = node.Parent.Parent?.Tag as Model;
                            if (model != null)
                            {
                                var key = new KeyValuePair<object, string>(model.ModelId, animSet.Name);

                                // プレビュー対象
                                _previewImageXOfAnimationUnderModel = bgBounds.X;
                                var invisible = pause.Pause.InvisibleBinds.Contains(key);
                                {
                                    var img = invisible ? previewOff_ : previewOn_;
                                    graphics.DrawImage(img, new Point(bgBounds.Left, BoundCentering(bgBounds, img)));
                                    bgBounds.X += previewOn_.Width + 2;
                                }

                                isPreview = isPreview && !invisible;

                                // 一時停止
                                _pauseImageXOfAnimationUnderModel = bgBounds.X;
                                var pausing = pause.PauseFrames.ContainsKey(key);

                                if (pausing)
                                {
                                    var img = isPreview ? pauseMark_ : pauseGrayMark_;
                                    graphics.DrawImage(img, new Point(bgBounds.Left, BoundCentering(bgBounds, img)));
                                    bgBounds.X += pauseMark_.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                                }
                                else
                                {
                                    var img = isPreview ? playMark_ : playGrayMark_;
                                    graphics.DrawImage(img, new Point(bgBounds.Left, BoundCentering(bgBounds, img)));
                                    bgBounds.X += pauseMark_.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                                }
                            }
                        }
                    }
                }

                // 星印・スタートアップ
                {
                    node.SmallIconImageX = bgBounds.X;

                    int upperOffset = 0;
                    int topPos = bgBounds.Top + (bgBounds.Height - starMark_.Height - attachedMark_.Height) / 2;
                    // 星印
                    if (guiObject.IsModifiedObject || ((doc != null) && doc.ContentsModified))
                    {
                        graphics.DrawImage(starMark_, new Point(bgBounds.Left, topPos));

                        upperOffset++;
                    }

                    int lowerOffset = 0;
                    // アタッチ
                    bool attached = false;
                    bool error = false;
                    if (doc is Model)
                    {
                        if (((Model)doc).IsSendAttached)
                        {
                            attached = true;
                        }
                    }
                    else if (doc is ShaderDefinition)
                    {
                        var shaderDefinition = (ShaderDefinition)doc;
                        if (shaderDefinition.FailedToBinarize)
                        {
                            error = true;
                        }

                        if (shaderDefinition.IsAttached)
                        {
                            attached = true;
                        }
                    }

                    if (error)
                    {
                        graphics.DrawImage(previewingErrorMark_, new Point(bgBounds.Left, topPos + starMark_.Height));
                        bgBounds.X += previewingErrorMark_.Width + 2;
                        lowerOffset++;
                    }

                    if (attached)
                    {
                        graphics.DrawImage(attachedMark_, new Point(bgBounds.Left, topPos + starMark_.Height));
                        bgBounds.X += attachedMark_.Width + 2;
                        lowerOffset++;
                    }
                    else
                    {
                        // モデルのプレビュー
                        if (doc is Model && doc.IsAttached)
                        {
                            graphics.DrawImage(previewingMark_, new Point(bgBounds.Left, topPos + starMark_.Height));
                            bgBounds.X += previewingMark_.Width + 2;
                            lowerOffset++;
                        }
                    }


                    // スタートアップ
                    if ((doc != null) && doc.OpenedFromStartUp)
                    {
                        graphics.DrawImage(startUpMark_, new Point(bgBounds.Left, topPos + starMark_.Height));
                        bgBounds.X += startUpMark_.Width + 2;
                        lowerOffset++;
                    }

                    if (lowerOffset == 0 && upperOffset > 0)
                    {
                        bgBounds.X += starMark_.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                    }
                }

            }

            // ドキュメントアイコン
            {
                var animSet = node.Tag as AnimationSet;
                node.DocIconImageX = bgBounds.X;

                if (animSet != null)
                {
                    var image = animSet.IsDefaultAnimationSet ? Resources.TreeView_Doc_DefaultAnimationSet : Resources.TreeView_Doc_AnimationSet;
                    if (image != null)
                    {
                        graphics.DrawImage(image, new Point(bgBounds.Left, BoundCentering(bgBounds, image)));
                        bgBounds.X += image.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                    }
                }
                else if (node.Tag is Model)
                {
                    var image = node.IsShowInObjView ? ShowInObjViewOn : ShowInObjViewOff;
                    graphics.DrawImage(image, new Point(bgBounds.Left, BoundCentering(bgBounds, image)));
                    bgBounds.X += image.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
//                    DebugConsole.WriteLine("docicon x:{0}", _docIconImageX);
                }
                else
                {
                    var image = imageList_.Images[node.ImageKey];
                    if (image != null)
                    {
                        graphics.DrawImage(image, new Point(bgBounds.Left, bgBounds.Top + (bgBounds.Height - image.Height)/2));
                        bgBounds.X += image.Width + 2;	// +2 はテキストの選択ボックスとくっついてしまうので
                    }
                }
            }

            // テキスト
            {
                var textFont = (guiObject is Model) ? TheApp.GuiFontBold : font;
                var textSize = graphics.MeasureString(node.Text, textFont);
                var textTop = bgBounds.Top + (bgBounds.Height - (int)textSize.Height) / 2;
                bgBounds.Width = (int)textSize.Width + 1;
                var textColor = node.ForeColor;
                Brush fillColor = null;
                if (node.Tag is Model && !node.IsShowInObjView)
                {
                    if (node.IsSelected)
                    {
                        textColor = SystemColors.HighlightText;
                        fillColor = SystemBrushes.ControlDarkDark;
                    }
                    else if (SelectedNodes.Contains(node))
                    {
                        textColor = SystemColors.HighlightText;
                        fillColor = SystemBrushes.ControlDark;
                    }
                    else
                    {
                        textColor = SystemColors.GrayText;
                    }
                }
                else if (node.IsSelected)
                {
                    textColor = SystemColors.HighlightText;
                    fillColor = SystemBrushes.HotTrack;
                }
                else if (SelectedNodes.Contains(node))
                {
                    textColor = SystemColors.HighlightText;
                    fillColor = SystemBrushes.Highlight;
                }


                if (fillColor != null)
                {
                    graphics.FillRectangle(fillColor, bgBounds);
                }
                TextRenderer.DrawText(graphics, node.Text, textFont, new Point(bgBounds.X, textTop), textColor);

                bgBounds.X += (int)textSize.Width;

                if (node.Tag is AnimationDocument)
                {
                    var document = (AnimationDocument)node.Tag;

                    var text = DocumentManager.GetSameNameIndexText(document, false);

                    if (!string.IsNullOrEmpty(text))
                    {
                        TextRenderer.DrawText(graphics, text, font, new Point(bgBounds.X, textTop), Color.Gray);
                    }
                }
                else if (node.Tag is Texture)
                {
                    var tex = (Texture)node.Tag;

                    // 重複
                    var text = DocumentManager.GetSameNameIndexText(tex, false);

                    // 編集不可
                    if (tex.IsTemporary)
                    {
                        if (!string.IsNullOrEmpty(text))
                        {
                            text += " ";
                        }
                        text += Strings.FileTreeView_CannotSave;
                    }

                    if (!string.IsNullOrEmpty(text))
                    {
                        TextRenderer.DrawText(graphics, text, font, new Point(bgBounds.X, textTop), Color.Gray);
                    }
                }
            }
        }

        protected override void OnNodeMouseClick(TreeNodeMouseClickEventArgs e)
        {
            base.OnNodeMouseClick(e);

            // ・アニメーションセット切り替えをシングルクリックにする。
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1478

            FlipPreview(e);
        }

        protected override void OnNodeMouseDoubleClick(TreeNodeMouseClickEventArgs e)
        {
            base.OnNodeMouseDoubleClick(e);

            // ・アニメーションセットをダブルクリックで切り替えた場合には、
            // 　停止中アニメーションを再生する。
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1478

            var isFlipped = FlipPreview(e);
            if (isFlipped)
            {
                if (e.Node.Tag is AnimationSet && e.Node.Nodes.Count > 0)
                {
                    TheApp.MainFrame.PlayAnimation();
                }
            }
            else
            {
                if (e.Button == MouseButtons.Left)
                {
                    var hitTestInfo = HitTest(e.Location);

                    //if ((hitTestInfo.Location & (TreeViewHitTestLocations.Image | TreeViewHitTestLocations.Label | TreeViewHitTestLocations.RightOfLabel)) != 0)
                    if ((!(e.Node.Tag is Model)||e.Location.X > _previewImageXOfModel + previewOn_.Width) &&
                        (hitTestInfo.Location & (TreeViewHitTestLocations.Image | TreeViewHitTestLocations.Label | TreeViewHitTestLocations.RightOfLabel)) != 0)
                    {
                        if (SelectedObject != null)
                        {
                            // 選択オブジェクト変更
                            if (SelectedObject is SeparateMaterial)
                            {
                                // fmt マテリアルは fmd マテリアルと同様に扱うのでマテリアルを選択オブジェクトとする。
                                App.AppContext.SelectedTarget.Set(((SeparateMaterial)SelectedObject).Materials.FirstOrDefault());
                            }
                            else
                            {
                                App.AppContext.SelectedTarget.Set(SelectedObject);
                            }

                            // プロパティウィンドウ表示
                            ObjectPropertyDialog.ShowPropertyDialog();
                        }
                    }
                }
            }
        }

        private bool FlipPreview(TreeNodeMouseClickEventArgs e)
        {
            if (e.Button != MouseButtons.Left) { return false; }
            var selnode = e.Node as ViewItem;
            Debug.Assert(selnode != null, "selnode != null");
            var treeview = selnode.TreeView as FileTreeView;
            Debug.Assert(treeview != null, "treeview != null");

            // 複数選択モード時には SelectedNode が null の場合がある。
            // クリックしたノードの取得には SelectedNode ではなく selnode を使うこと。

            // クリックされたノードが選択されなかった場合は何もしない
            if (!SelectedNodes.Contains(selnode)) { return false; }

            var hitTestInfo = HitTest(e.Location);

            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1316#24
            // アニメーションセットの目の時はプレビューを切り替える
            var animationSet = selnode.Tag as AnimationSet;
            if ((animationSet != null) && ((hitTestInfo.Location & TreeViewHitTestLocations.Image) != 0))
            {
                var model = App.AppContext.SelectedFileViewObjectOwner as Model;
                if ((model != null) && (model.PreviewAnimSet != animationSet))
                {
                    using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                    {
                        var oldPreviewAnimSet = model.PreviewAnimSet;
                        model.PreviewAnimSet = animationSet;

                        // 指定のアニメーションセットに変更できなければもとに戻す
                        if (model.PreviewAnimSet != animationSet)
                        {
                            model.PreviewAnimSet = oldPreviewAnimSet;

                            // 変更がなかったことにする
                            block.IsIgnore = true;

                            UIMessageBox.Warning(Strings.FileTreeView_EmptyDefaultAnimationSet);
                            return false;
                        }
                        App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
                    }
                }
                else if (animationSet.IsSceneAnimationSet)
                {
                    if (DocumentManager.PreviewSceneAnimSet != animationSet)
                    {
                        DocumentManager.PreviewSceneAnimSet = animationSet;
                        App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
                    }
                }
                return true;
            }
            else
            {
                // クリックで選択されたノードを最初にする
                var selectedNodes = SelectedNodes.Where(x => x != selnode).ToList();
                selectedNodes.Insert(0, selnode);

                if (selectedNodes.All(x => x.Tag is Model))
                {
                    if (e.Location.X < _previewImageXOfModel)
                    {
                        return false;
                    }

                    var active = (Model)selnode.Tag;
                    var models = treeview.SelectedNodes.Select(x => x.Tag).OfType<Model>();

                    if ((e.Location.X < _previewImageXOfModel + previewOn_.Width))
                    {
                        var visible = !active.PreviewInfo.Visible;
                        foreach (var model in models.Where(x => !x.IsSendAttached && x.PreviewInfo.Visible != visible))
                        {
                            model.PreviewInfo.Visible = visible;
                            if (!Viewer.Manager.Instance.IsConnected) continue;
                            if (model.IsVisible)
                            {
                                Viewer.LoadOrReloadModel.Send(model);

                                // バインド関係が更新されることがあるので再転送
                                model.SendEditBoneBind();
                                model.SendEditModelLayout(false, sendChildren: true);

                                // アニメーションをバインドする
                                Viewer.ViewerUtility.SendAnimationSet(model);

                                // model をバインドしてるモデル (binders) のボーンバインド情報を転送する。
                                var binders = DocumentManager.Models.Where(m => (m.PreviewInfo.BindModelName == model.Name));
                                foreach (var binder in binders)
                                {
                                    binder.SendEditBoneBind();
                                }
                            }
                            else
                            {
                                Viewer.Close.Send(model);
                            }
                        }

                        // プロジェクトの変更マーク更新用
                        DocumentManager.ProjectDocument.SetMaybeModified();
                        App.AppContext.ExecutePropertyChangedEvent(this, (new DocumentPropertyChangedEventArgs()).GetArgs());
                        return true;
                    }
                    else if (e.Location.X > selnode.DocIconImageX
                             && (e.Location.X < selnode.DocIconImageX + ShowInObjViewOn.Width)) // view in editor icon
                    {
                        var showInObjView = !active.PreviewInfo.ShowInObjView;
                        foreach (var model in models.Where(x => x.PreviewInfo.ShowInObjView != showInObjView))
                        {
                            model.PreviewInfo.ShowInObjView = showInObjView;
                            var args =
                                new DocumentPropertyChangedEventArgs(
                                    new DocumentPropertyChangedShowInObjViewArgs(model));
                            App.AppContext.ExecutePropertyChangedEvent(this, args.GetArgs());
                        }

                        // プロジェクトの変更マーク更新用
                        DocumentManager.ProjectDocument.SetMaybeModified();
                        return true;
                    }
                }
                else if (selectedNodes.All(x => x.Tag is SceneAnimation)
                         && selectedNodes.All(x => x?.Parent?.Tag is AnimationSet))
                {
                    if (e.Location.X < _previewImageXOfSceneAnimationUnderAnimationSet)
                    {
                        return false;
                    }

                    var active = Tuple.Create((SceneAnimation)selnode.Tag, (AnimationSet)selnode.Parent.Tag);
                    var sceneAnims = treeview.SelectedNodes.Select(x => Tuple.Create((SceneAnimation)x.Tag, (AnimationSet)x.Parent.Tag));

                    if ((e.Location.X < _previewImageXOfSceneAnimationUnderAnimationSet + previewOn_.Width))
                    {
                        var visible = active.Item1.InvisibleBinds.Contains(active.Item2);
                        foreach (var sceneAnim in sceneAnims)
                        {
                            if (visible)
                            {
                                sceneAnim.Item1.InvisibleBinds.Remove(sceneAnim.Item2);
                            }
                            else
                            {
                                sceneAnim.Item1.InvisibleBinds.Add(sceneAnim.Item2);
                            }
                        }

                        DocumentManager.PreviewSceneAnimSetUpdated();
                        App.AppContext.ExecutePropertyChangedEvent(
                            this,
                            (new DocumentPropertyChangedEventArgs()).GetArgs());
                        return true;
                    }
                }
                else if (selectedNodes.All(x => x.Tag is AnimationDocument)
                         && selectedNodes.All(x => x?.Parent?.Tag is AnimationSet))
                {
                    var doPause = false;
                    var doInvisible = false;
                    bool? newPause = null;
                    bool? newInvisible = null;
                    if (_pauseImageXOfAnimationUnderModel <= e.Location.X
                        && e.Location.X < _pauseImageXOfAnimationUnderModel + pauseMark_.Width)
                    {
                        doPause = true;
                    }
                    else if (_previewImageXOfAnimationUnderModel <= e.Location.X
                             && e.Location.X < _previewImageXOfAnimationUnderModel + previewOn_.Width)
                    {
                        doInvisible = true;
                    }
                    if (doPause || doInvisible)
                    {
                        var models = new List<Model>();
                        foreach (var node in selectedNodes)
                        {
                            var animation = node.Tag as AnimationDocument;
                            var animSetNode = (node.Parent as ViewItem);
                            var parentAnimationSet = animSetNode?.Tag as AnimationSet;
                            if (animation != null && animSetNode != null && parentAnimationSet != null)
                            {
                                var parentModel = animSetNode.Parent.Tag as Model;
                                if (parentModel != null)
                                {
                                    models.Add(parentModel);
                                    var key = new KeyValuePair<object, string>(
                                        parentModel.ModelId,
                                        parentAnimationSet.Name);
                                    var pausing = animation.PauseFrames.ContainsKey(key);
                                    var invisible = animation.Pause.InvisibleBinds.Contains(key);
                                    if (!newPause.HasValue)
                                    {
                                        newPause = !animation.PauseFrames.ContainsKey(key);
                                    }
                                    if (!newInvisible.HasValue)
                                    {
                                        newInvisible = !animation.Pause.InvisibleBinds.Contains(key);
                                    }
                                    animation.SetPause(
                                        doPause ? newPause.Value : pausing,
                                        App.AppContext.PreviewCurrentFrame,
                                        parentModel.ModelId,
                                        parentAnimationSet.Name,
                                        doInvisible ? newInvisible.Value : invisible);
                                }
                            }
                        }

                        foreach (var model in models.Distinct())
                        {
                            model.PreviewAnimSetUpdated();
                        }
                        App.AppContext.ExecutePropertyChangedEvent(
                            this,
                            (new DocumentPropertyChangedEventArgs()).GetArgs());
                        return true;
                    }
                }
            }

            return false;
        } // FlipPreview
    }
    public class DocumentPropertyChangedShowInObjViewArgs : DocumentContentsChangedArgs
    {
        public DocumentPropertyChangedShowInObjViewArgs(GuiObject target, EditCommand command = null) : base(target, command) { }
    }
}
