﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Runtime.InteropServices;
using App.Controls;

namespace App
{
    // テクスチャコンバーターマネージャー
    static public class TexcvtrManager
    {
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate UInt32 GetCvtrVersionDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool SetOptionsDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ClearDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ReadInputFileDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] paths,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ReadFtxDataDelegate(
            IntPtr pData,
            int dataSize,
            string path,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ReadBitmapDataDelegate(
            IntPtr pData,
            int width,
            int height,
            int depth,
            bool hasAlpha,
            bool isFloat,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] paths,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] originalPaths,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ConvertToFileDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ConvertToDataDelegate32(
            ref IntPtr ppData,
            ref UInt32 pDataSize
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ConvertToDataDelegate64(
            ref IntPtr ppData,
            ref UInt64 pDataSize
        );

        // 戻り値を string にして MarshalAs UnmanagedType.LPWStr 等にしたときは原因不明で動作が不安定になる。
        // http://msdn.microsoft.com/en-us/library/ms912059.aspx
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate IntPtr GetErrorStringPtrDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool SetCreateInfoDelegate(
            string createInfo
        );

        public static GetCvtrVersionDelegate	GetCvtrVersion;
        public static SetOptionsDelegate		SetOptions;
        public static ClearDelegate				Clear;
        public static ReadInputFileDelegate		ReadInputFile;
        public static ReadFtxDataDelegate		ReadFtxData;
        public static ReadBitmapDataDelegate	ReadBitmapData;
        public static ConvertToFileDelegate		ConvertToFile;
        public static GetErrorStringPtrDelegate GetErrorStringPtr;
        public static SetCreateInfoDelegate		SetCreateInfo;

        private static ConvertToDataDelegate32	ConvertToData32;
        private static ConvertToDataDelegate64	ConvertToData64;

        /// <summary>
        /// エラーの取得
        /// </summary>
        public static string GetError()
        {
            return System.Runtime.InteropServices.Marshal.PtrToStringUni(GetErrorStringPtr());
        }

        public static bool ConvertToData(ref IntPtr ppData, ref int pDataSize)
        {
            bool i;

            if (Environment.Is64BitProcess)
            {
                UInt64 data = 0;
                i = ConvertToData64(ref ppData, ref data);

                pDataSize = (int)data;
            }
            else
            {
                UInt32 data = 0;
                i = ConvertToData32(ref ppData, ref data);

                pDataSize = (int)data;
            }

            return i;
        }

        private static string DllFilepath
        {
            get
            {
                // SIGLO-61624 SIGLO-61638 対応。
                string path = string.Format(
                    @"{0}\..\3dTools\{1}",
                    Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                    "3dTextureConverter.dll");

                return path;
            }
        }

        public static string TextureConverterExeFilepath
        {
            get
            {
                // SIGLO-61624 SIGLO-61638 対応
                string path = string.Format(
                    @"{0}\..\3dTools\{1}",
                    Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                    "3dTextureConverter.exe");

                return path;
            }
        }

        private static IntPtr dllHandle_ = IntPtr.Zero;

        public static void Initialize()
        {
            // texUtils.dll が読み込めないようなので、一時的にカレントディレクトリを移す

            string currentDir = System.Environment.CurrentDirectory;
            try
            {
                System.IO.Directory.SetCurrentDirectory(Path.GetDirectoryName(DllFilepath));
                dllHandle_ = App.Win32.NativeMethods.LoadLibrary(DllFilepath);

                SetExportFunction();
            }
            catch(Exception e)
            {
                UIMessageBox.Error(e.ToString());
            }
            finally
            {
                System.Environment.CurrentDirectory = currentDir;
            }
        }

        public static void Destroy()
        {
            if (dllHandle_ != IntPtr.Zero)
            {
                App.Win32.NativeMethods.FreeLibrary(dllHandle_);
                dllHandle_ = IntPtr.Zero;
            }
        }

        private static void SetExportFunction()
        {
            GetCvtrVersion		= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "GetCvtrVersion"	), typeof(GetCvtrVersionDelegate)	) as GetCvtrVersionDelegate;
            SetOptions			= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "SetOptions"		), typeof(SetOptionsDelegate)		) as SetOptionsDelegate;
            Clear				= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "Clear"			), typeof(ClearDelegate)			) as ClearDelegate;
            ReadInputFile		= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "ReadInputFile"	), typeof(ReadInputFileDelegate)	) as ReadInputFileDelegate;
            ReadFtxData			= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "ReadFtxData"		), typeof(ReadFtxDataDelegate)		) as ReadFtxDataDelegate;
            ReadBitmapData		= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "ReadBitmapData"	), typeof(ReadBitmapDataDelegate)	) as ReadBitmapDataDelegate;
            ConvertToFile		= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "ConvertToFile"	), typeof(ConvertToFileDelegate)	) as ConvertToFileDelegate;
            GetErrorStringPtr	= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "GetErrorString"	), typeof(GetErrorStringPtrDelegate)) as GetErrorStringPtrDelegate;
            SetCreateInfo		= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "SetCreateInfo"	), typeof(SetCreateInfoDelegate)	) as SetCreateInfoDelegate;

            if (Environment.Is64BitProcess)
            {
                ConvertToData64	= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "ConvertToData"	), typeof(ConvertToDataDelegate64)	) as ConvertToDataDelegate64;
            }
            else
            {
                ConvertToData32	= Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "ConvertToData"	), typeof(ConvertToDataDelegate32)	) as ConvertToDataDelegate32;
            }
        }
    }
}
