﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;

namespace App
{
    public class NotificationStringFormat : INotificationHandlerStringFormat
    {
        const string NoConversion = "{0}";

        public string UnsavedFile
        {
            get { return NoConversion; }
        }

        public string UnresolvedTexture
        {
            get { return NoConversion; }
        }
    }

    public class NotificationHandler : INotificationHandler
    {
        NotificationStringFormat stringFormat = new NotificationStringFormat();
        public INotificationHandlerStringFormat StringFormat
        {
            get { return stringFormat; }
        }

        public virtual void WriteMessageLog(MessageLog.LogType type, string message, Action onDoubleClick = null)
        {
            MessageLog.Write(type, message, onDoubleClick);
        }

        /// <summary>
        /// 情報メッセージ。
        /// </summary>
        public virtual void MessageBoxInformation(string message)
        {
            UIMessageBox.Information(message);
        }
        public virtual void MessageBoxInformation(string format, params object[] args)
        {
            MessageBoxInformation(string.Format(format, args));
        }

        /// <summary>
        /// 警告メッセージ。
        /// </summary>
        public virtual void MessageBoxWarning(string message)
        {
            UIMessageBox.Warning(message);
        }

        /// <summary>
        /// 警告メッセージ。
        /// </summary>
        public virtual void MessageBoxWarning(string format, params object[] args)
        {
            MessageBoxWarning(string.Format(format, args));
        }

        /// <summary>
        /// エラーメッセージ。
        /// </summary>
        public virtual void MessageBoxError(string message)
        {
            UIMessageBox.Error(message);
        }

        /// <summary>
        /// エラーメッセージ。
        /// </summary>
        public virtual void MessageBoxError(string format, params object[] args)
        {
            MessageBoxError(string.Format(format, args));
        }


        /// <summary>
        /// はい、いいえメッセージ。
        /// </summary>
        public virtual bool MessageBoxYesNo(MessageContext context, string message)
        {
            return UIMessageBox.YesNo(message);
        }
        public virtual bool MessageBoxYesNo(MessageContext context, string format, params object[] args)
        {
            return MessageBoxYesNo(context, string.Format(format, args));
        }
        /// <summary>
        /// はい、いいえ、キャンセルメッセージ。
        /// </summary>
        public virtual DialogResult MessageBoxYesNoCancel(MessageContext context, string message)
        {
            return UIMessageBox.YesNoCancel(message);
        }
        public virtual DialogResult MessageBoxYesNoCancel(MessageContext context, string format, params object[] args)
        {
            return MessageBoxYesNoCancel(context, string.Format(format, args));
        }

        /// <summary>
        /// ＯＫ、キャンセルメッセージ。
        /// </summary>
        public bool MessageBoxOkCancel(MessageContext context, string message)
        {
            return UIMessageBox.OkCancel(message);
        }

        public virtual void OkListBoxDialog(MessageContext context, string text, string label, IEnumerable<string> lines)
        {
            using (var dialog = new OkListBoxDialog())
            {
                dialog.Text = text;
                dialog.lblDescription.Text = label;
                foreach (var line in lines)
                {
                    dialog.AddLine(line);
                }
                dialog.ShowDialog();
            }
        }

        public bool OkCancelListBoxDialog(MessageContext context, string text, string label, string format, IEnumerable<Document> documents)
        {
            using (var dialog = new OkCancelListBoxDialog())
            {
                dialog.Text = text;
                dialog.lblDescription.Text = label;

                foreach (var doc in documents)
                {
                    dialog.lbxList.AddItem(new UIListControlItem(string.Format(format, doc.FileName + DocumentManager.GetSameNameIndexText(doc, true)), doc, doc.ObjectIcon));
                }

                var result = dialog.ShowDialog();
                return result == DialogResult.OK;
            }
        }

        public bool OkCancelListBoxDialog(MessageContext context, string text, string label, string format, IEnumerable<Tuple<string, System.Drawing.Image>> items)
        {
            using (var dialog = new OkCancelListBoxDialog())
            {
                dialog.Text = text;
                dialog.lblDescription.Text = label;

                foreach (var item in items)
                {
                    dialog.lbxList.AddItem(new UIListControlItem(string.Format(format, item.Item1), item.Item1, item.Item2));
                }

                var result = dialog.ShowDialog();
                return result == DialogResult.OK;
            }
        }

        public bool OkCancelTextBoxDialog(MessageContext context, string text, string label, IEnumerable<string> lines)
        {
            using (var dialog = new OkCancelTextBoxDialog())
            {
                dialog.Text = text;
                dialog.lblDescription.Text = label;

                foreach (var line in lines)
                {
                    dialog.AddLine(line);
                }

                var result = dialog.ShowDialog();
                return result == DialogResult.OK;
            }
        }

        /// <summary>
        /// DocumentsCloseDialog
        /// </summary>
        /// <param name="modified"></param>
        /// <param name="saveRequired"></param>
        /// <returns>ダイアログがキャンセルされたらfalse</returns>
        public bool DocumentsCloseDialog(IEnumerable<Document> modified, out IEnumerable<Document> saveRequired)
        {
            saveRequired = null;
            using (var dialog = new DocumentsCloseDialog(modified))
            {
                switch (dialog.ShowDialog())
                {
                    case DialogResult.Cancel: return false;
                    case DialogResult.Yes:
                    {
                        saveRequired = dialog.CloseDocuments;
                        break;
                    }
                }
            }
            return true;
        }

        public virtual void SkeletalAnimBindDialog(SkeletalAnimBindDialog.ErrorInfo[] skeletalAnimError, bool canCancel)
        {
            using (var dialog = new SkeletalAnimBindDialog(skeletalAnimError, false))
            {
                // 警告だけ
                dialog.ShowDialog(TheApp.MainFrame);
            }
        }

        public IEnumerable<ShaderAssignUtility.MaterialAnimationError> FixShaderParamAnimDialog(ShaderAssignUtility.MaterialAnimationError[] errorMaterialAnims)
        {
            using (var dialog = new FixShaderParamAnimDialog(errorMaterialAnims))
            {
                var ok = dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK;
                if (ok && dialog.CheckedMaterials.Any())
                {
                    return dialog.CheckedMaterials.ToArray();
                }
            }
            return null;
        }

        public virtual bool SameFileReload(IEnumerable<Document> updateddocs)
        {
            using (var dialog = new SameFileReload(updateddocs.Select(x => x.FilePath)))
            {
                dialog.ShowDialog();
                if (dialog.DialogResult == DialogResult.OK)
                {
                    ApplicationConfig.UserSetting.IO.AutoReloadUpdatedOnLoad = dialog.AutoReload;
                    return true;
                }
            }
            return false;
        }

        public virtual void FileErrorDialog(MessageContext context, IEnumerable<Tuple<string, string, string>> errors, string message)
        {
            using (var dialog = new FileErrorDialog(errors.ToArray()))
            {
                dialog.lblDescription.Text = message;
                dialog.ShowDialog();
            }
        }

        public virtual Model ModelSelectDialog(IEnumerable<Model> models, AnimationDocument animation, string chkText, ref bool applyAll)
        {
            using (var dialog = new ModelSelectDialog(models, animation, chkText))
            {
            	if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK)
            	{
                    applyAll = dialog.Checked;
                    return (Model)dialog.Selected;
            	}
            }
            return null;
        }

        public virtual void FixLightAnimAssignDialog(LightAnimation[] errorLightAnims, out LightAnimation[] fixLightAnims, out LightAnimation[] notFixLightAnims)
        {
            using (var dialog = new FixLightAnimationAssignDialog(errorLightAnims))
            {
                var ok = dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK;

                fixLightAnims = ok ? dialog.CheckeLightAnims.ToArray() : new LightAnimation[0];
                notFixLightAnims = ok ? dialog.NotCheckedLightAnims.ToArray() : errorLightAnims;

                if (ok)
                {
                    ApplicationConfig.UserSetting.IO.AutoFixLightAnimation = dialog.IsAutoFix;
                    ApplicationConfig.SaveUserSetting();
                }
            }
        }

        public virtual void FixShaderAssignDialog(Material[] errorMaterials, out Material[] fixMaterials, out Material[] notFixMaterials)
        {
            using (var dialog = new FixShaderAssignDialog(errorMaterials))
            {
                var ok = dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK;

                fixMaterials = ok ? dialog.CheckedMaterials.ToArray() : new Material[0];
                notFixMaterials = ok ? dialog.NotCheckedMaterials.ToArray() : errorMaterials;

                if (ok)
                {
                    ApplicationConfig.UserSetting.IO.AutoFix = dialog.IsAutoFix;
                    ApplicationConfig.SaveUserSetting();
                }
            }
        }

        public virtual List<Tuple<Document, string>> SaveDirectoryDialog(List<Document> saveDocuments)
        {
            using (var dialog = new SaveDirectoryDialog(saveDocuments))
            {
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    return dialog.SaveDocuments;
                }
                return null;
            }
        }

        public string SaveFileDialog(string filter, string ext, string title, string fileName = "", string fileLocation = "", bool prompt = true)
        {
            using (var dialog = new SaveFileDialog())
            {
                dialog.Filter = filter;
                dialog.DefaultExt = ext;
                dialog.Title = title;
                dialog.FileName = fileName;
                if (!string.IsNullOrEmpty(fileLocation))
                {
                    dialog.InitialDirectory = fileLocation;
                }
                dialog.OverwritePrompt = prompt;
                while(dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK)
                {
                    // マテリアル名の変更課題 SIGLO-37520 対応時にここを削除する。
                    try
                    {
                        var renamable =
                            string.IsNullOrEmpty(fileName) ||
                            !string.Equals(ext, nw.g3d.nw4f_3dif.G3dPath.MaterialBinaryExtension.Substring(1), StringComparison.OrdinalIgnoreCase);
                        if (!renamable && !string.Equals(System.IO.Path.GetFileName(dialog.FileName), fileName, StringComparison.Ordinal))
                        {
                            throw new Exception(Strings.NotificationHandler_SaveFileDialog_RenamingFilesIsNotAllowed);
                        }
                    }
                    catch (Exception e)
                    {
                        MessageBoxError(e.Message);
                        dialog.FileName = fileName;
                        continue;
                    }

                    try
                    {
                        var path = dialog.FileName;
                        return path;
                    }
                    catch (Exception)
                    {
                        MessageBoxError(Strings.NotificationHandler_SaveFileDialog_TooLongFilePath);
                        dialog.FileName = fileName;
                    }
                }
            }
            return null;
        }
    }
}
