﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;
using App.Controls;

namespace App
{
    public static class MessageLog
    {
        public enum LogType
        {
            CriticalError,
            Error,
            Help,
            Information,
            Question,
            Warning,
        }

        public static void Initialize()
        {
            Debug.Assert(TheApp.MainFrame != null);

            TheApp.MainFrame.LogList.DoubleClick += DoubleClick;
        }

        public static void Terminate()
        {
            Debug.Assert(TheApp.MainFrame != null);

            TheApp.MainFrame.LogList.DoubleClick -= DoubleClick;
        }

        private static readonly object sync_ = new object();

        public static void Clear()
        {
            lock(sync_)
            {
                Action proc = () =>
                {
                    if (TheApp.MainFrame == null)
                    {
                        return;
                    }

                    var logList = TheApp.MainFrame.LogList;
                    using(var ub = new UpdateBlock(logList))
                    {
                        logList.Items.Clear();
                    }
                };

                SafeExecute(proc);
            }
        }

        public static void SelectAll()
        {
            lock(sync_)
            {
                Action proc = () =>
                {
                    if (TheApp.MainFrame == null)
                    {
                        return;
                    }

                    var logList = TheApp.MainFrame.LogList;
                    foreach (LogListViewItem item in logList.Items)
                    {
                        item.Selected = true;
                    }
                };

                SafeExecute(proc);
            }
        }

        /// <summary>
        /// 選択メッセージをクリップボードにコピー。
        /// </summary>
        public static void CopySelectedMessage()
        {
            lock(sync_)
            {
                Action proc = () =>
                {
                    if (TheApp.MainFrame == null)
                    {
                        return;
                    }

                    // コピーする項目をリストアップ
                    var logList = TheApp.MainFrame.LogList;
                    var copyItems = new List<LogListViewItem>();
                    copyItems.AddRange(logList.SelectedItems.Count == 0
                                           ? logList.Items.Cast<LogListViewItem>()
                                           : logList.SelectedItems.Cast<LogListViewItem>());

                    // テキストをクリップボードにコピー
                    var result = new StringBuilder();
                    foreach (LogListViewItem item in copyItems)
                    {
                        result.AppendFormat("{0}\r\n", item.Text);
                    }
                    App.Utility.ClipboardUtility.SetDataObject(result.ToString(), true);
                };

                SafeExecute(proc);
            }
        }

        public static void Write(LogType type, string message, Action onDoubleClick = null)
        {
            lock(sync_)
            {
                Action proc = () =>
                {
                    if (TheApp.MainFrame == null)
                    {
                        return;
                    }

                    var logList = TheApp.MainFrame.LogList;

                    using(var ub = new UpdateBlock(logList))
                    {
                        // メッセージクリア
                        if (logList.Items.Count >= 1000)
                        {
                            logList.Items.RemoveAt(0);
                        }

                        // 選択状態解除
                        foreach (LogListViewItem selectedItem in logList.SelectedItems)
                        {

                            selectedItem.Selected = false;
                            selectedItem.Focused = false;
                        }

                        var color = (type == LogType.CriticalError) ? Color.Red :
                                    (type == LogType.Error) ? Color.Red :
                                    (type == LogType.Warning) ? Color.OrangeRed :
                                                                SystemColors.WindowText;

                        // 項目追加
                        var now = DateTime.Now;
                        var item = new LogListViewItem
                        {
                            ImageKey = type.ToString(),
                            Text = string.Format("[{0}]  {1}", now.ToString("HH:mm:ss"), message),
                            ForeColor = color,
                            OnDoubleClick = onDoubleClick,
                        };

                        logList.Items.Add(item);

                        // 最後の項目を見えるようにする
                        item.Selected = true;
                        item.Focused = true;
                        item.EnsureVisible();
                    }
                };

                SafeExecute(proc);
            }
        }

        private class LogListViewItem : ListViewItem
        {
            public Action OnDoubleClick{ get; set; }
        }

        private static List<Action>	beforeMainFrameCreatedProcs_ = null;

        private static void SafeExecute(Action proc)
        {
            if ((TheApp.MainFrame.IsHandleCreated == false))
            {
                if (beforeMainFrameCreatedProcs_ == null)
                {
                    beforeMainFrameCreatedProcs_ = new List<Action>();

                    // メインフレームが出来上がった時に実行する
                    {
                        EventHandler onMainFrameHandleCreated = null;

                        onMainFrameHandleCreated = (s, e) =>
                        {
                            TheApp.MainFrame.HandleCreated -= onMainFrameHandleCreated;

                            foreach(var beforeMainFrameCreatedProc in beforeMainFrameCreatedProcs_)
                            {
                                beforeMainFrameCreatedProc();
                            }

                            beforeMainFrameCreatedProcs_.Clear();
                            beforeMainFrameCreatedProcs_ = null;
                        };

                        TheApp.MainFrame.HandleCreated += onMainFrameHandleCreated;
                    }
                }

                beforeMainFrameCreatedProcs_.Add(proc);
            }
            else
            {
                if (Thread.CurrentThread != TheApp.MainThread)
                {
                    TheApp.MainFrame.BeginInvoke(new MethodInvoker(proc));
                }
                else
                {
                    proc();
                }
            }
        }

        private static void DoubleClick(object sender, EventArgs e)
        {
            lock(sync_)
            {
                var logList = TheApp.MainFrame.LogList;
                if (logList.SelectedIndex != -1)
                {
                    Debug.Assert(logList.Items[logList.SelectedIndex] is LogListViewItem);

                    var selectedItem = logList.Items[logList.SelectedIndex] as LogListViewItem;

                    if (selectedItem.OnDoubleClick != null)
                    {
                        selectedItem.OnDoubleClick();
                    }
                }
            }
        }
    }
}
