﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Windows.Forms;
using App.Controls;

namespace App
{
    /// <summary>
    /// エラーログ出力クラス。
    /// </summary>
    public static class ErrorLog
    {
        public enum LogKind
        {
            Normal,
            BinaryConvert,
            ShaderConvert,
        }

        // 出力ファイル名
        private static string LogFilePath
        {
            get
            {
                return Path.Combine(TheApp.LocalAppDataDirectory, "ConvertError.log");
            }
        }

        public static void Initialize()
        {
            if (!App.ConfigData.ApplicationConfig.Setting.IO.ErrorLog)
            {
                return;
            }

            try
            {
                // ファイルストリーム
                var writer = new StreamWriter(LogFilePath, false);
                writer.Close();

                // メモリストリーム作成

            }
            catch (Exception exception)
            {
                UIMessageBox.Error(App.res.Strings.Error_Log_Open, LogFilePath, exception.Message);
            }
        }

        // 書き込みデリゲート
        private delegate void WriteDelegate(string message, LogKind kind = LogKind.Normal);
        // Invoke用書き込みメソッド
        private static void _WriteLog(string message, LogKind kind = LogKind.Normal) { WriteLog(message, kind); }
        /// <summary>
        /// 書き込み。
        /// </summary>
        public static void WriteLog(string message, LogKind kind = LogKind.Normal)
        {
            if (!App.ConfigData.ApplicationConfig.Setting.IO.ErrorLog)
            {
                return;
            }

            if (string.IsNullOrEmpty(message))
            {
                // 文字列が入っていない場合は、出力しない。
                return;
            }

            // 別スレッドからの呼び出し用
            if (System.Threading.Thread.CurrentThread != TheApp.MainThread)
            {
                TheApp.MainFrame.BeginInvoke(new WriteDelegate(_WriteLog), new object[] { message, kind });
                return;
            }

            try
            {
                // 種類ごとにメモリストリームに保持させる。
                SetMemoryStream(message, kind);

                using (StreamWriter w = File.AppendText(LogFilePath))
                {
                    Write(message, w);
                }
            }
            // ログファイル出力失敗時はもうどうしようもない
            catch (Exception) { }
        }

        /// <summary>
        /// 書き込み。
        /// </summary>
        private static void Write(string message, TextWriter w)
        {
            w.Write("\r\nLog Entry : ");
            w.WriteLine("{0} {1}", DateTime.Now.ToLongTimeString(),	DateTime.Now.ToLongDateString());
            w.Write(message);
            w.WriteLine("-------------------------------");
            // Update the underlying file.
            w.Flush();
        }

        public static void WriteBinaryErrorLog(List<string> errMsgs)
        {
            string msgSum = string.Empty;

            foreach (var msg in errMsgs)
            {
                msgSum += (msg + "\r\n");
            }

            // ログ出力
            ErrorLog.WriteLog(msgSum, ErrorLog.LogKind.BinaryConvert);
        }

        private static void SetMemoryStream(string message, LogKind kind)
        {
            // 現状未実装です。

            if (kind == LogKind.Normal)
            {

            }
            else if (kind == LogKind.BinaryConvert)
            {

            }
            else if (kind == LogKind.ShaderConvert)
            {

            }

        }
    }
}
