﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Threading;
using System.Windows.Forms;
using App.Controls;
using System.Text;

namespace App
{
    /// <summary>
    /// デバッグコンソールクラス。
    /// </summary>
    public sealed partial class DebugConsole : UIForm
    {
        // インスタンス
        private static DebugConsole _instance = null;
        // 出力カウンタ
        private static int _outputCounter = 0;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private DebugConsole()
        {
            InitializeComponent();
        }

        /// <summary>
        /// 初期化処理。
        /// </summary>
        //[Conditional("DEBUG")]
        public static void Initialize()
        {
            if (_instance == null)
            {
                _instance = new DebugConsole();
                _instance.Show();
                _instance.Update();
                Output = true;

                // テキスト追加用タイマー
                // テキストの追加漏れや遅れが出ないように適当なタイミングで呼び出す
                var writeTimer = new System.Windows.Forms.Timer { Interval = 100 };
                writeTimer.Tick += (s, a) =>
                {
                    if (_instance != null && _instance.Visible)
                    {
                        _instance.Invoke(new Action(_Write));
                    }
                };
                writeTimer.Start();
            }
        }

        public static bool IsInitialized
        {
            get
            {
                return _instance != null;
            }
        }

        /// <summary>
        /// 終了処理。
        /// </summary>
        //[Conditional("DEBUG")]
        public static void Terminate()
        {
            if (_instance != null)
            {
                if (!_instance.IsHandleCreated)
                {
                    _instance.Close();
                }
            }
        }

        protected override void OnClosed(EventArgs e)
        {
            base.OnClosed(e);
            _instance = null;
        }

        // 書き込みデリゲート
        private delegate void WriteDelegate();

        // コンソールに出力するか
        public static bool Output { get; set; }

        // 出力文字
        // バッファが溢れるときにキューの要素を削除するが、要素内の文字数が多いと想定以上に削除されてしまう。
        // これを避けるために、要素内の文字数調整も行うため string ではなく StringBuilder にしている。
        public static Queue<StringBuilder> Buffer = new Queue<StringBuilder>();
        public static int BufferedLength = 0;
        private static DateTime _lastWrite;

        // 書き込みメソッド
        private static void _Write()
        {
            lock (Messages)
            {
                // テキストの追加は重いため、ある程度時間が経つか指定行数分貯まってから追加する。
                if ((DateTime.Now - _lastWrite).TotalMilliseconds < 100 && Messages.Count < 50) return;

                if (Messages.Count > 0)
                {
                    if (!Output)
                    {
                        Messages.Clear();
                        return;
                    }
                    _lastWrite = DateTime.Now;
                    string message = "";
                    while (Messages.Count > 0)
                    {
                        message += string.Format("[{0:d3}]{1}", _outputCounter, Messages.Dequeue());
                        // 出力カウンタ更新
                        if (++_outputCounter >= 1000)
                        {
                            _outputCounter = 0;
                        }
                    }
                    // 通常の呼び出し用
                    TextBox tbx = _instance.tbxOutput;
                    string output = message;
                    Buffer.Enqueue(new StringBuilder(output));
                    BufferedLength += output.Length;

                    if (BufferedLength > tbx.MaxLength)
                    {
                        tbx.Clear();

                        // 2/3にする
                        // Buffer の要素毎に削除すると、要素サイズが大きい場合に
                        // 必要以上に削除され、バッファがクリアされることがある。
                        // これを回避するために、キュー要素の文字数も調整する。
                        var threshold = 2 * tbx.MaxLength / 3;
                        if (BufferedLength > threshold)
                        {
                            // まずは粒度の大きな Buffer の要素単位で削除する。
                            for (var length = BufferedLength - Buffer.Peek().Length; length >= threshold; length -= Buffer.Peek().Length)
                            {
                                Buffer.Dequeue();
                                BufferedLength = length;
                            }

                            // 要素単位で削除しきれなかったことは、閾値を跨いでいる要素が残っていることを意味する。
                            // その要素は粒度の小さい行単位で削除する。
                            if (BufferedLength > threshold)
                            {
                                // 閾値を跨いでいる要素。
                                var item = Buffer.Peek();

                                var length = BufferedLength;

                                while (length > threshold)
                                {
                                    var str = item.ToString();
                                    var idx = str.IndexOf(Environment.NewLine);
                                    if (idx != -1)
                                    {
                                        var n = idx + Environment.NewLine.Length;
                                        item.Remove(0, n);
                                        length -= n;
                                    }
                                    else
                                    {
                                        // 行毎には削除しきれなかったので、要素内の全文字列を削除する。
                                        var n = item.Length;
                                        item.Remove(0, n);
                                        length -= n;
                                        break;
                                    }
                                }

                                // 要素内の文字列が空になった場合は要素を削除。
                                // length は調整済み。
                                if (item.Length == 0)
                                {
                                    Buffer.Dequeue();
                                }

                                BufferedLength = length;
                            }
                        }

                        var builder = new StringBuilder();
                        foreach (var item in Buffer)
                        {
                            builder.Append(item.ToString());
                        }
                        tbx.AppendText(builder.ToString());
                        Debug.Assert(tbx.Text.Length == BufferedLength);
                    }
                    else
                    {
                        tbx.AppendText(output);
                    }

                }
            }
        }

        private static readonly Queue<string> Messages = new Queue<string>();
        /// <summary>
        /// 書き込み。
        /// </summary>
        //[Conditional("DEBUG")]
        public static void Write(string message)
        {
            if (_instance != null && _instance.IsHandleCreated)
            {
                lock (Messages)
                {
                    Messages.Enqueue(message);
                }

                // 別スレッドからの呼び出し用
                if (Thread.CurrentThread != TheApp.MainThread)
                {
                    _instance.BeginInvoke(new WriteDelegate(_Write), new object[] {});
                    return;
                }

                _Write();
            }
        }

        /// <summary>
        /// 書式指定付き書き込み。
        /// </summary>
        //[Conditional("DEBUG")]
        public static void Write(string format, params object[] args)
        {
            Write(string.Format(format, args));
        }

        /// <summary>
        /// 書き込み＋改行。
        /// </summary>
        //[Conditional("DEBUG")]
        public static void WriteLine(string message)
        {
            Write(message + "\r\n");
        }

        /// <summary>
        /// 書式指定付き書き込み＋改行。
        /// </summary>
        //[Conditional("DEBUG")]
        public static void WriteLine(string format, params object[] args)
        {
            Write(string.Format(format, args) + "\r\n");
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 位置、サイズをデスクトップ隅に設定
            Rectangle desktop = Screen.PrimaryScreen.WorkingArea;
            Size      size    = new Size(desktop.Width / 2, desktop.Height / 2);
            Location = new Point(desktop.X, desktop.Bottom - size.Height);
            Size     = size;

            // 起動時に表示する場合はアイコンが表示されないがデバッグ実行時だけの問題なので
            // とりあえず気にしない。
            if (TheApp.MainFrame != null && TheApp.MainFrame.Icon != null)
            {
                Icon = TheApp.MainFrame.Icon;
            }

            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // ESCキーでバッファのクリア
            if (e.KeyCode == Keys.Escape)
            {
                _instance.tbxOutput.Clear();
                _outputCounter = 0;
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // 既定の処理
            if (base.ProcessCmdKey(ref msg, keyData))
            {
                return true;
            }
            // メインフレームのショートカット処理
            return TheApp.MainFrame.ProcessCommandKey(ref msg, keyData);
        }
        #endregion
    }
}
