﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using App.Utility;
using nw.g3d.toollib;
using Nintendo.ToolFoundation.CommandLine;
using System.Runtime.InteropServices;

namespace App
{
    public static class CommandLineOptionManager
    {
        private static readonly List<string> files_ = new List<string>();
        private static List<string> _errfiles = new List<string>();
        private static bool _loadRecursive = false;

        // ■注意
        //		プロパティーはnullの場合、オプションでの指定がなかったことになる

        // 開くファイル
        public static ReadOnlyList<string> Files { get { return new ReadOnlyList<string>(files_); } }

        /// <summary>
        /// --version、--help などコンソール出力用のコマンドラインオプションが指定されたかどうかを取得します。
        /// </summary>
        public static bool IsConsoleOptionSpecified { get; private set; } = false;

        /// <summary>
        /// チーム設定のパスを取得します。
        /// </summary>
        public static string TeamSettingsFile { get; private set; } = null;

        /// <summary>
        /// コマンドラインオプション解析のエラーメッセージを取得します。
        /// </summary>
        public static string ErrorMessage { get; private set; } = string.Empty;

        [DllImport("kernel32.dll")]
        public static extern bool AttachConsole(uint dwProcessId);
        [DllImport("kernel32.dll")]
        public static extern bool FreeConsole();

        public static void Initialize(string[] origargs)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Null<Exception>(TeamSettingsFile);
            Nintendo.ToolFoundation.Contracts.Ensure.True<Exception>(files_.Count == 0);
            Nintendo.ToolFoundation.Contracts.Ensure.True<Exception>(_errfiles.Count == 0);

            var targets = new List<string>();
            G3dToolUtility.Run(delegate()
            {
                var parser = new DelegateBasedCommandLineParser();
                parser.AddHelp(WriteLineToConsole);
                parser.AddVersion(version =>
                {
                    WriteLineToConsole(version);
                });
                parser.AddArgsFileOption();
                parser.AddFlagOption('r', "recursive", () => _loadRecursive = true)
                    .SetDescription(App.res.Strings.CommandLine_Recursive);
                parser.AddValueOption("team-settings-file", path => TeamSettingsFile = path)
                    .SetValueName("path")
                    .SetDescription(App.res.Strings.CommandLine_TeamSettingsFile);
                parser.AddValue(path =>
                    {
                        string fullPath = Path.GetFullPath(path);
                        targets.Add(path);
                    }, -1)
                    .SetDescription(App.res.Strings.CommandLine_Value)
                    .SetValueName("path");

                parser.ParseArgs(origargs, errorMessage =>
                {
                    ErrorMessage += $"{errorMessage}\n";
                });
            });

            if (targets.Any())
            {
                foreach (var target in targets)
                {
                    if (Directory.Exists(target))
                    {
                        files_.AddRange(FilesInFolder(target));
                    }
                    else if (File.Exists(target))
                    {
                        files_.Add(target);
                    }
                    else
                    {
                        // bad path
                        _errfiles.Add(target);
                    }
                }
            }

            if (_errfiles.Any())
            {
                var mes = _errfiles.Aggregate(string.Empty, (current, file) => current + (file + "\n"));
                ErrorMessage += $"{App.res.Strings.IO_Load_Failed}\n{mes}";
            }
        }

        private static IEnumerable<string> FilesInFolder(string path)
        {
            var searchOpt = _loadRecursive ? SearchOption.AllDirectories : SearchOption.TopDirectoryOnly;
            ObjectIDUtility.UnsupportedType unsupportedType;
            var files =
                Directory.GetFiles(path, "*.*", searchOpt)
                    .Where(x => ObjectIDUtility.IsSupportExtension(Path.GetExtension(x), out unsupportedType));
            return files;
        }

        private static void WriteLineToConsole(string message)
        {
            if (!AttachConsole(uint.MaxValue))
            {
                return;
            }

            IsConsoleOptionSpecified = true;
            Console.WriteLine(message);

            FreeConsole();
        }
    }
}
