﻿using System;
using System.Drawing;
using System.Xml.Serialization;

namespace App.Data
{
    /// <summary>
    /// RGBA色 0.0-1.0
    /// </summary>
    [Serializable]
    public struct RgbaColor
    {
        public RgbaColor(float r, float g, float b, float a) : this() {
            Set(r,g,b,a);
        }

        public RgbaColor(Color color) : this() {

            Set((float)color.R / 255.0f, (float)color.G / 255.0f, (float)color.B / 255.0f, (float)color.A / 255.0f);
        }

        public RgbaColor(Color color, float hdrFactor) : this()
        {
            Set(color.R * hdrFactor / 255, color.G * hdrFactor / 255, color.B * hdrFactor / 255, color.A / 255.0f);
        }

        public override string ToString()
        {
            return string.Format("({0}, {1}, {2}, {3})", R, G, B, A);
        }

        public float[] ToArray(bool withA = true)
        {
            return
                withA ?
                    new[]{R, G, B, A} :
                    new[]{R, G, B};
        }

        /// <summary>
        /// 設定
        /// </summary>
        public void Set(float r, float g, float b, float a)
        {
            R = r;
            G = g;
            B = b;
            A = a;
        }

        /// <summary>
        /// 設定
        /// </summary>
        public void Set(RgbaColor rgb, float a)
        {
            Set(rgb.R, rgb.G, rgb.B, a);
        }

        /// <summary>
        /// 設定
        /// </summary>
        public void Set(RgbaColor rgba)
        {
            Set(rgba.R, rgba.G, rgba.B, rgba.A);
        }

        public static RgbaColor Round(RgbaColor rgba)
        {
            return new RgbaColor(
                Utility.ColorUtility.Round(rgba.R),
                Utility.ColorUtility.Round(rgba.G),
                Utility.ColorUtility.Round(rgba.B),
                Utility.ColorUtility.Round(rgba.A));
        }

        /// <summary>
        /// R 成分を取得または設定します。
        /// </summary>
        [XmlAttribute]
        public float R { get; set; }

        /// <summary>
        /// G 成分を取得または設定します。
        /// </summary>
        [XmlAttribute]
        public float G { get; set; }

        /// <summary>
        /// B 成分を取得または設定します。
        /// </summary>
        [XmlAttribute]
        public float B { get; set; }

        /// <summary>
        /// A 成分を取得または設定します。
        /// </summary>
        [XmlAttribute]
        public float A { get; set; }

        /// <summary>
        /// ToColorNoAlpha。
        /// </summary>
        public Color ToColorNoAlpha()
        {
            var max = Math.Max(Math.Max(R, G), B);

            if (max <= 1.0f)
            {
                int byteR = Math.Min(Math.Max((int)(255 * R), 0), 255);
                int byteG = Math.Min(Math.Max((int)(255 * G), 0), 255);
                int byteB = Math.Min(Math.Max((int)(255 * B), 0), 255);

                return Color.FromArgb(byteR, byteG, byteB);
            }
            else
            {
                int byteR = Math.Min(Math.Max((int)(255 * R / max), 0), 255);
                int byteG = Math.Min(Math.Max((int)(255 * G / max), 0), 255);
                int byteB = Math.Min(Math.Max((int)(255 * B / max), 0), 255);

                return Color.FromArgb(byteR, byteG, byteB);
            }
        }

        public ColorPicker.RGB ToRGBNoAlpha()
        {
            var max = Math.Max(Math.Max(R, G), B);

            if (max <= 1.0f)
            {
                float byteR = Math.Min(Math.Max(R, 0), 1);
                float byteG = Math.Min(Math.Max(G, 0), 1);
                float byteB = Math.Min(Math.Max(B, 0), 1);

                return new ColorPicker.RGB(byteR, byteG, byteB);
            }
            else
            {
                float byteR = Math.Min(Math.Max((R / max), 0), 1);
                float byteG = Math.Min(Math.Max((G / max), 0), 1);
                float byteB = Math.Min(Math.Max((B / max), 0), 1);

                return new ColorPicker.RGB(byteR, byteG, byteB);
            }
        }

        /// <summary>
        /// ToColor。
        /// </summary>
        public Color ToColor()
        {
            int byteA = Math.Min(Math.Max((int)(255 * A), 0), 255);

            var rgb = ToColorNoAlpha();

            return Color.FromArgb(byteA, rgb.R, rgb.G, rgb.B);
        }

        public ColorPicker.RGB ToRGB()
        {
            float a = Math.Min(Math.Max(A, 0), 1);
            var rgb = ToRGBNoAlpha();
            return new ColorPicker.RGB(a, rgb);
        }

        /// <summary>
        /// Alpha 表示用
        /// </summary>
        /// <remarks>Alpha 成分は反映されません。</remarks>
        public Color ToColorAlpha()
        {
            int byteA = Math.Min(Math.Max((int)(255 * A), 0), 255);

            return Color.FromArgb(byteA, byteA, byteA);
        }

        /// <summary>
        /// FromColor。
        /// </summary>
        public static RgbaColor FromColor(Color src)
        {
            var color = new RgbaColor();
            color.R = src.R / 255.0F;
            color.G = src.G / 255.0F;
            color.B = src.B / 255.0F;
            color.A = src.A / 255.0F;
            return color;
        }

        /// <summary>
        /// SelectElement。
        /// </summary>
        public static RgbaColor SelectElement(RgbaColor src, RgbaColor dst, uint dstElemBits)
        {
            if((dstElemBits & (1<<0)) != 0){	src.R = dst.R;	}
            if((dstElemBits & (1<<1)) != 0){	src.G = dst.G;	}
            if((dstElemBits & (1<<2)) != 0){	src.B = dst.B;	}
            if((dstElemBits & (1<<3)) != 0){	src.A = dst.A;	}

            return src;
        }

        /// <summary>
        /// 比較
        /// </summary>
        public static bool operator ==(RgbaColor lhs, RgbaColor rhs)
        {
            return lhs.R == rhs.R && lhs.G == rhs.G && lhs.B == rhs.B && lhs.A == rhs.A;
        }

        /// <summary>
        /// 比較
        /// </summary>
        public static bool operator !=(RgbaColor lhs, RgbaColor rhs)
        {
            return !(lhs == rhs);
        }

        /// <summary>
        /// ハッシュコード
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        /// <summary>
        /// 比較
        /// </summary>
        public override bool Equals(object obj)
        {
            return base.Equals(obj);
        }

        public RgbaColor ToSrgbFromLinear()
        {
            return new RgbaColor(
                R > 0 ? (float)Utility.ColorUtility.ToSrgbFromLinear(R) : R,
                G > 0 ? (float)Utility.ColorUtility.ToSrgbFromLinear(G) : G,
                B > 0 ? (float)Utility.ColorUtility.ToSrgbFromLinear(B) : B,
                A);
        }


        public Color ToSrgbColor(bool fromLinear)
        {
            if (fromLinear)
            {
                return ToRGBNoAlpha().ToSrgbFromLinear().GetColor();
            }
            else
            {
                return ToRGBNoAlpha().GetColor();
            }
        }
    }
}
