﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace App.Data
{
    public enum InterpolationType
    {
        Hermite,
        Linear,
        Step,
    }

    public enum PrimitiveTypeKind
    {
        Float,
        Int,
        Uint,
        Bool,
    }

    [Serializable]
    public class KeyFrame : NintendoWare.G3d.Edit.ICurveValue
    {
#if true
        public float Frame { get; set; }
        public float Value { get; set; }
        public float InSlope { get; set; }
        public float OutSlope { get; set; }
#else
        private float _frame;
        private float _value;
        private float _inSlope;
        private float _outSlope;

//		private const float maxNumber = float.MaxValue;
        private const float maxNumber = +1000000.0f;

        public float	Frame
        {
            get { return _frame; }
            set {
                if (Math.Abs(value) > maxNumber)
                {
                    value = (value > 0.0f) ? maxNumber : -maxNumber;
                }

                _frame = value;
            }
        }

        public float	Value
        {
            get { return _value; }
            set {
                if (Math.Abs(value) > maxNumber)
                {
                    value = (value > 0.0f) ? maxNumber : -maxNumber;
                }

                _value = value;
            }
        }

        public float	InSlope
        {
            get { return _inSlope; }
            set {
                if (Math.Abs(value) > maxNumber)
                {
                    value = (value > 0.0f) ? maxNumber : -maxNumber;
                }

                _inSlope = value;
            }
        }

        public float	OutSlope
        {
            get { return _outSlope; }
            set {
                if (Math.Abs(value) > maxNumber)
                {
                    value = (value > 0.0f) ? maxNumber : -maxNumber;
                }

                _outSlope = value;
            }
        }
#endif
        public bool		IsUnionSlope{			get; set;	}

        public bool		Selected{				get; set;	}
        public bool		InHandleSelected{		get; set;	}
        public bool		OutHandleSelected{		get; set;	}

        public bool		AreaSelected{			get; set;	}
        public bool		InHandleAreaSelected{	get; set;	}
        public bool		OutHandleAreaSelected{	get; set;	}

        public bool		AnySelected{			get{ return Selected     || InHandleSelected     || OutHandleSelected;     } }
        public bool		AnyAreaSelected{		get{ return AreaSelected || InHandleAreaSelected || OutHandleAreaSelected; } }

        // フレームが同じとき時用のソート用のキー
        public int		Index{					get; set;	}

        public int SubIndex { get; set; }

/*
        public override bool Equals(object obj)
        {
            // TODO: 削除
            System.Diagnostics.Debug.Assert(false);

            // If parameter is null return false.
            if (obj == null)
            {
                return false;
            }

            // If parameter cannot be cast to Point return false.
            KeyFrame p = obj as KeyFrame;
            if ((object)p == null)
            {
                return false;
            }

            // Return true if the fields match:
            return
                (Frame					== p.Frame) &&
                (Value					== p.Value) &&
                (InSlope				== p.InSlope) &&
                (OutSlope				== p.OutSlope) &&
                (IsUnionSlope			== p.IsUnionSlope) &&
                (Selected				== p.Selected) &&
                (InHandleSelected		== p.InHandleSelected) &&
                (OutHandleSelected		== p.OutHandleSelected) &&
                (AreaSelected			== p.AreaSelected) &&
                (InHandleAreaSelected	== p.InHandleAreaSelected) &&
                (OutHandleAreaSelected	== p.OutHandleAreaSelected) &&
                (AnySelected			== p.AnySelected) &&
                (AnyAreaSelected		== p.AnyAreaSelected) &&
                (Index					== p.Index) &&
                (SubIndex				== p.SubIndex);
        }

        public override int GetHashCode()
        {
            // TODO: 削除
            System.Diagnostics.Debug.Assert(false);
            return base.GetHashCode();
        }
*/

        public KeyFrame()
        {
            IsUnionSlope	= true;
        }

        public void ResetAreaSelected()
        {
            AreaSelected			= false;
            InHandleAreaSelected	= false;
            OutHandleAreaSelected	= false;
        }

        public void Set(KeyFrame src)
        {
            Frame					= src.Frame;
            Value					= src.Value;
            InSlope				= src.InSlope;
            OutSlope				= src.OutSlope;
            IsUnionSlope			= src.IsUnionSlope;

            Selected				= src.Selected;
            InHandleSelected		= src.InHandleSelected;
            OutHandleSelected		= src.OutHandleSelected;

            AreaSelected			= src.AreaSelected;
            InHandleAreaSelected	= src.InHandleAreaSelected;
            OutHandleAreaSelected	= src.OutHandleAreaSelected;

            Index					= src.Index;
            SubIndex = src.SubIndex;
        }

        public KeyFrame Clone()
        {
            var keyFrame = new KeyFrame();

            keyFrame.Set(this);

            return keyFrame;
        }
    }

    // 拡張メソッド
    public static class CurveExtensions
    {
        /// <summary>
        /// カーブがデフォルト値かつ定数かどうか取得します。
        /// </summary>
        /// <returns>カーブがデフォルト値かつ定数のとき true 。</returns>
        public static bool IsDefaultValue(this List<KeyFrame> src, InterpolationType interpolationType, float defaultValue)
        {
            float base_value;
            bool isConstantCurve = IsConstantCurve(src, interpolationType, out base_value);

            return (isConstantCurve && base_value == defaultValue);
        }

        public static bool IsConstantCurve(this List<KeyFrame> src, out float base_value)
        {
            return IsConstantCurve(src, InterpolationType.Step, out base_value);
        }

        public static bool IsConstantCurve(this List<KeyFrame> src, InterpolationType interpolationType, out float base_value)
        {
            // キー無しはそもそもこの関数では判断できない
            if (src.Any() == false)
            {
                base_value = 0;
                return false;
            }

            // キーが一つ
            if (src.Count       == 1)
            {
                base_value = src[0].Value;
                return true;
            }
            else
            // キーが複数で、全キーの値が同じで、全キーのスロープが 0
            // リニアとステップはスロープ 0
            {
                float value       = src[0].Value;
                bool  isZeroSlope = (interpolationType == InterpolationType.Linear) || (interpolationType == InterpolationType.Step);

                foreach (var k in src)
                {
                    float inSlope  = isZeroSlope ? 0.0f : k.InSlope;
                    float outSlope = isZeroSlope ? 0.0f : k.OutSlope;

                    if ((value    != k.Value) ||
                        (inSlope  != 0.0f   ) ||
                        (outSlope != 0.0f   ))
                    {
                        base_value = 0;
                        return false;
                    }
                }

                base_value = src[0].Value;
                return true;
            }
        }
    }
}
